<?php
/**
 * Discount Code Styles Handler
 *
 * @package Shibhu Custom Messages for WooCommerce
 */

defined( 'ABSPATH' ) || exit;

class SCMW_Discount_Styles {
    
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_discount_styles' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_discount_scripts' ) );
    }
    
    /**
     * Enqueue frontend discount styles
     */
    public function enqueue_discount_styles() {
        if ( is_product() ) {
            wp_enqueue_style(
                'scmw-discount-styles',
                SCMW_PLUGIN_URL . 'assets/css/discount-styles.css',
                array(),
                SCMW_VERSION
            );
            
            wp_enqueue_script(
                'scmw-discount-scripts',
                SCMW_PLUGIN_URL . 'assets/js/discount-scripts.js',
                array( 'jquery' ),
                SCMW_VERSION,
                true
            );
            
            wp_localize_script( 'scmw-discount-scripts', 'scmwDiscount', array(
                'ajaxurl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'scmw_discount_nonce' ),
                'strings' => array(
                    'copied'       => __( 'Copied!', 'shibhu-custom-messages-for-woocommerce' ),
                    'copyFailed'   => __( 'Copy failed', 'shibhu-custom-messages-for-woocommerce' ),
                    'showCode'     => __( 'SHOW PROMO CODE', 'shibhu-custom-messages-for-woocommerce' ),
                    'hideCode'     => __( 'HIDE CODE', 'shibhu-custom-messages-for-woocommerce' ),
                    'revealCode'   => __( 'Click to Reveal', 'shibhu-custom-messages-for-woocommerce' ),
                    'copyCode'     => __( 'Copy', 'shibhu-custom-messages-for-woocommerce' ),
                ),
            ) );
        }
    }
    
    /**
     * Enqueue admin discount scripts
     */
    public function enqueue_admin_discount_scripts( $hook ) {
        global $post_type;
        
        if ( ( 'post.php' === $hook || 'post-new.php' === $hook ) && 'scmw_message' === $post_type ) {
            wp_enqueue_style(
                'scmw-discount-admin-styles',
                SCMW_PLUGIN_URL . 'assets/css/discount-admin.css',
                array(),
                SCMW_VERSION
            );
            
            wp_enqueue_script(
                'scmw-discount-admin-scripts',
                SCMW_PLUGIN_URL . 'assets/js/discount-admin.js',
                array( 'jquery' ),
                SCMW_VERSION,
                true
            );
        }
    }
    
    /**
     * Get available discount styles
     */
    public static function get_discount_styles() {
        return array(
            'style_1' => array(
                'label'       => __( 'Style 1 - Default', 'shibhu-custom-messages-for-woocommerce' ),
                'description' => __( 'Simple inline code with copy button', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'style_2' => array(
                'label'       => __( 'Style 2 - Reveal on Click', 'shibhu-custom-messages-for-woocommerce' ),
                'description' => __( 'Hidden code revealed with button click', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'style_3' => array(
                'label'       => __( 'Style 3 - Percentage Badge', 'shibhu-custom-messages-for-woocommerce' ),
                'description' => __( 'Discount percentage with circular badge', 'shibhu-custom-messages-for-woocommerce' ),
            ),
        );
    }
    
    /**
     * Render discount code by style
     */
    public static function render_discount_code( $message_id, $product = null ) {
        $discount_code   = get_post_meta( $message_id, '_scmw_discount_code', true );
        $discount_desc   = get_post_meta( $message_id, '_scmw_discount_description', true );
        $discount_style  = get_post_meta( $message_id, '_scmw_discount_style', true );
        $discount_style  = $discount_style ? $discount_style : 'style_1';
        
        // Style 3 specific options
        $discount_percentage = get_post_meta( $message_id, '_scmw_discount_percentage', true );
        $discount_bg_color   = get_post_meta( $message_id, '_scmw_discount_bg_color', true );
        $discount_text_color = get_post_meta( $message_id, '_scmw_discount_text_color', true );
        
        if ( ! $discount_code ) {
            return;
        }
        
        // Sanitize all values
        $discount_code       = sanitize_text_field( $discount_code );
        $discount_desc       = sanitize_text_field( $discount_desc );
        $discount_percentage = sanitize_text_field( $discount_percentage );
        $discount_bg_color   = sanitize_hex_color( $discount_bg_color );
        $discount_text_color = sanitize_hex_color( $discount_text_color );
        
        switch ( $discount_style ) {
            case 'style_2':
                self::render_style_2( $discount_code, $discount_desc );
                break;
                
            case 'style_3':
                self::render_style_3( $discount_code, $discount_desc, $discount_percentage, $discount_bg_color, $discount_text_color );
                break;
                
            case 'style_1':
            default:
                self::render_style_1( $discount_code, $discount_desc );
                break;
        }
    }
    
    /**
     * Render Style 1 - Default inline code
     */
    private static function render_style_1( $code, $description ) {
        ?>
        <div class="scmw-discount-container scmw-discount-style-1">
            <?php if ( $description ) : ?>
                <div class="scmw-discount-description"><?php echo esc_html( $description ); ?></div>
            <?php endif; ?>
            <div class="scmw-discount-code">
                <span class="scmw-code-text"><?php echo esc_html( $code ); ?></span>
                <button type="button" class="scmw-copy-btn" data-code="<?php echo esc_attr( $code ); ?>" aria-label="<?php esc_attr_e( 'Copy discount code', 'shibhu-custom-messages-for-woocommerce' ); ?>">
                    <?php esc_html_e( 'Copy', 'shibhu-custom-messages-for-woocommerce' ); ?>
                </button>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Style 2 - Reveal on click
     */
    private static function render_style_2( $code, $description ) {
        ?>
        <div class="scmw-discount-container scmw-discount-style-2">
            <?php if ( $description ) : ?>
                <div class="scmw-discount-description"><?php echo esc_html( $description ); ?></div>
            <?php endif; ?>
            <div class="scmw-discount-reveal-wrapper">
                <button type="button" class="scmw-reveal-btn" data-code="<?php echo esc_attr( $code ); ?>" aria-label="<?php esc_attr_e( 'Show promo code', 'shibhu-custom-messages-for-woocommerce' ); ?>" aria-expanded="false">
                    <span class="scmw-reveal-text"><?php esc_html_e( 'SHOW PROMO CODE', 'shibhu-custom-messages-for-woocommerce' ); ?></span>
                    <span class="scmw-reveal-icon">
                        <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
                            <path d="M5 7.5L10 12.5L15 7.5" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        </svg>
                    </span>
                </button>
                <div class="scmw-discount-code-revealed" style="display: none;" aria-live="polite">
                    <span class="scmw-code-text"><?php echo esc_html( $code ); ?></span>
                    <button type="button" class="scmw-copy-btn" data-code="<?php echo esc_attr( $code ); ?>" aria-label="<?php esc_attr_e( 'Copy discount code', 'shibhu-custom-messages-for-woocommerce' ); ?>">
                        <?php esc_html_e( 'Copy', 'shibhu-custom-messages-for-woocommerce' ); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Style 3 - Percentage badge
     */
    private static function render_style_3( $code, $description, $percentage, $bg_color, $text_color ) {
        $percentage = $percentage ? $percentage : '10';
        $bg_color   = $bg_color ? $bg_color : '#2563eb';
        $text_color = $text_color ? $text_color : '#ffffff';
        
        $inline_styles = sprintf(
            'background: linear-gradient(135deg, %s 0%%, %s 100%%); color: %s;',
            esc_attr( $bg_color ),
            esc_attr( self::adjust_brightness( $bg_color, -20 ) ),
            esc_attr( $text_color )
        );
        ?>
        <div class="scmw-discount-container scmw-discount-style-3" style="<?php echo esc_attr( $inline_styles ); ?>">
            <div class="scmw-discount-badge-wrapper">
                <div class="scmw-discount-badge">
                    <div class="scmw-badge-percentage"><?php echo esc_html( $percentage ); ?>%</div>
                    <div class="scmw-badge-label"><?php esc_html_e( 'OFF', 'shibhu-custom-messages-for-woocommerce' ); ?></div>
                </div>
                <div class="scmw-discount-content">
                    <?php if ( $description ) : ?>
                        <div class="scmw-discount-description"><?php echo esc_html( $description ); ?></div>
                    <?php endif; ?>
                    <div class="scmw-discount-code-wrapper">
                        <div class="scmw-discount-code">
                            <span class="scmw-code-text"><?php echo esc_html( $code ); ?></span>
                            <button type="button" class="scmw-copy-btn" data-code="<?php echo esc_attr( $code ); ?>" aria-label="<?php esc_attr_e( 'Copy discount code', 'shibhu-custom-messages-for-woocommerce' ); ?>">
                                <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
                                    <path d="M5.5 5.5V3.5C5.5 2.67157 6.17157 2 7 2H13C13.8284 2 14.5 2.67157 14.5 3.5V9.5C14.5 10.3284 13.8284 11 13 11H11" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"/>
                                    <rect x="1.5" y="5" width="9" height="9" rx="1.5" stroke="currentColor" stroke-width="1.5"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Adjust color brightness
     */
    private static function adjust_brightness( $hex, $steps ) {
        // Remove # if present
        $hex = str_replace( '#', '', $hex );
        
        // Convert to RGB
        $r = hexdec( substr( $hex, 0, 2 ) );
        $g = hexdec( substr( $hex, 2, 2 ) );
        $b = hexdec( substr( $hex, 4, 2 ) );
        
        // Adjust brightness
        $r = max( 0, min( 255, $r + $steps ) );
        $g = max( 0, min( 255, $g + $steps ) );
        $b = max( 0, min( 255, $b + $steps ) );
        
        // Convert back to hex
        return '#' . str_pad( dechex( $r ), 2, '0', STR_PAD_LEFT ) 
                   . str_pad( dechex( $g ), 2, '0', STR_PAD_LEFT ) 
                   . str_pad( dechex( $b ), 2, '0', STR_PAD_LEFT );
    }
    
    /**
     * Add meta box fields for discount styles
     */
    public static function add_discount_style_fields( $post ) {
        $discount_style      = get_post_meta( $post->ID, '_scmw_discount_style', true );
        $discount_style      = $discount_style ? $discount_style : 'style_1';
        $discount_percentage = get_post_meta( $post->ID, '_scmw_discount_percentage', true );
        $discount_percentage = $discount_percentage ? $discount_percentage : '10';
        $discount_bg_color   = get_post_meta( $post->ID, '_scmw_discount_bg_color', true );
        $discount_bg_color   = $discount_bg_color ? $discount_bg_color : '#2563eb';
        $discount_text_color = get_post_meta( $post->ID, '_scmw_discount_text_color', true );
        $discount_text_color = $discount_text_color ? $discount_text_color : '#ffffff';
        
        $styles = self::get_discount_styles();
        ?>
        <div class="scmw-field">
            <label for="scmw_discount_style"><strong><?php esc_html_e( 'Discount Code Style', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
            <select id="scmw_discount_style" name="scmw_discount_style" class="widefat scmw-discount-style-select">
                <?php foreach ( $styles as $key => $style ) : ?>
                    <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $discount_style, $key ); ?>>
                        <?php echo esc_html( $style['label'] ); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <p class="description"><?php esc_html_e( 'Choose the visual style for displaying the discount code', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
            
            <!-- Style previews -->
            <div class="scmw-discount-style-previews" style="margin-top: 15px;">
                <?php foreach ( $styles as $key => $style ) : ?>
                    <div class="scmw-style-preview scmw-style-preview-<?php echo esc_attr( $key ); ?>" style="<?php echo $discount_style === $key ? '' : 'display:none;'; ?>">
                        <p class="description" style="margin: 5px 0; color: #666; font-style: italic;">
                            <?php echo esc_html( $style['description'] ); ?>
                        </p>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- Style 3 specific options -->
        <div class="scmw-discount-style-options scmw-discount-style-3-options" style="<?php echo $discount_style === 'style_3' ? '' : 'display:none;'; ?>">
            <div class="scmw-field">
                <label for="scmw_discount_percentage"><strong><?php esc_html_e( 'Discount Percentage', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <input type="number" id="scmw_discount_percentage" name="scmw_discount_percentage" value="<?php echo esc_attr( $discount_percentage ); ?>" min="1" max="100" class="small-text" />
                <span class="description"><?php esc_html_e( '% (for badge display only)', 'shibhu-custom-messages-for-woocommerce' ); ?></span>
            </div>
            
            <div class="scmw-field">
                <label for="scmw_discount_bg_color"><strong><?php esc_html_e( 'Background Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <input type="text" id="scmw_discount_bg_color" name="scmw_discount_bg_color" value="<?php echo esc_attr( $discount_bg_color ); ?>" class="scmw-color-picker" data-default-color="#2563eb" />
            </div>
            
            <div class="scmw-field">
                <label for="scmw_discount_text_color"><strong><?php esc_html_e( 'Text Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <input type="text" id="scmw_discount_text_color" name="scmw_discount_text_color" value="<?php echo esc_attr( $discount_text_color ); ?>" class="scmw-color-picker" data-default-color="#ffffff" />
            </div>
        </div>
        <?php
    }
    
    /**
     * Save discount style meta
     */
    public static function save_discount_style_meta( $post_id ) {
        // Verify nonce is handled in main meta box save function
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in parent function
        if ( isset( $_POST['scmw_discount_style'] ) ) {
            $discount_style = sanitize_text_field( wp_unslash( $_POST['scmw_discount_style'] ) );
            $valid_styles = array_keys( self::get_discount_styles() );
            
            if ( in_array( $discount_style, $valid_styles, true ) ) {
                update_post_meta( $post_id, '_scmw_discount_style', $discount_style );
            }
        }
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in parent function
        if ( isset( $_POST['scmw_discount_percentage'] ) ) {
            $percentage = absint( $_POST['scmw_discount_percentage'] );
            $percentage = max( 1, min( 100, $percentage ) );
            update_post_meta( $post_id, '_scmw_discount_percentage', $percentage );
        }
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in parent function
        if ( isset( $_POST['scmw_discount_bg_color'] ) ) {
            $color = sanitize_hex_color( wp_unslash( $_POST['scmw_discount_bg_color'] ) );
            if ( $color ) {
                update_post_meta( $post_id, '_scmw_discount_bg_color', $color );
            }
        }
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in parent function
        if ( isset( $_POST['scmw_discount_text_color'] ) ) {
            $color = sanitize_hex_color( wp_unslash( $_POST['scmw_discount_text_color'] ) );
            if ( $color ) {
                update_post_meta( $post_id, '_scmw_discount_text_color', $color );
            }
        }
    }
}

// Initialize
SCMW_Discount_Styles::get_instance();