<?php
/**
 * Conditional Logic Handler (Adapted from Elegant Labels)
 *
 * @package Shibhu Custom Messages for WooCommerce
 */

defined( 'ABSPATH' ) || exit;

class SCMW_Conditional_Logic {
    
    private static $instance = null;
    private $select_options = array();
    private $cached_categories = null;
    private $cached_tags = null;
    
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init_select_options();
    }
    
    /**
     * Initialize conditional logic options
     */
    private function init_select_options() {
        $this->select_options = array(
            'field_types' => array(
                'product-status' => __( 'Product Stock', 'shibhu-custom-messages-for-woocommerce' ),
                'category'       => __( 'Product Category', 'shibhu-custom-messages-for-woocommerce' ),
                'tags'           => __( 'Product Tags', 'shibhu-custom-messages-for-woocommerce' ),
                'price'          => __( 'Product Price', 'shibhu-custom-messages-for-woocommerce' ),
                'quantity'       => __( 'Product Quantity', 'shibhu-custom-messages-for-woocommerce' ),
                'title'          => __( 'Product Title', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'text_operators' => array(
                'equal'        => __( 'Equal To', 'shibhu-custom-messages-for-woocommerce' ),
                'not-equal'    => __( 'Not Equal To', 'shibhu-custom-messages-for-woocommerce' ),
                'contains'     => __( 'Contains', 'shibhu-custom-messages-for-woocommerce' ),
                'not-contains' => __( 'Does Not Contain', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'category_operators' => array(
                'equal'     => __( 'Is in Category', 'shibhu-custom-messages-for-woocommerce' ),
                'not-equal' => __( 'Is not in Category', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'tags_operators' => array(
                'equal'     => __( 'Has Tag', 'shibhu-custom-messages-for-woocommerce' ),
                'not-equal' => __( 'Does not have Tag', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'numeric_operators' => array(
                'equal'         => __( 'Equal To', 'shibhu-custom-messages-for-woocommerce' ),
                'not-equal'     => __( 'Not Equal To', 'shibhu-custom-messages-for-woocommerce' ),
                'greater'       => __( 'Greater Than', 'shibhu-custom-messages-for-woocommerce' ),
                'greater-equal' => __( 'Greater or Equal To', 'shibhu-custom-messages-for-woocommerce' ),
                'less'          => __( 'Less Than', 'shibhu-custom-messages-for-woocommerce' ),
                'less-equal'    => __( 'Less or Equal To', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'stock_status_operators' => array(
                'equal' => __( 'Is', 'shibhu-custom-messages-for-woocommerce' ),
            ),
            'stock_status' => array(
                'instock'     => __( 'In Stock', 'shibhu-custom-messages-for-woocommerce' ),
                'outofstock'  => __( 'Out of Stock', 'shibhu-custom-messages-for-woocommerce' ),
                'onbackorder' => __( 'On Backorder', 'shibhu-custom-messages-for-woocommerce' ),
            ),
        );
    }
    
    /**
     * Get select options
     */
    public function get_select_options() {
        return $this->select_options;
    }
    
    /**
     * Check if WooCommerce is active
     */
    private function is_woocommerce_active() {
        return class_exists( 'WooCommerce' ) && function_exists( 'wc_get_product' );
    }
    
    /**
     * Cache terms for better performance
     */
    private function cache_terms() {
        if ( ! $this->is_woocommerce_active() ) {
            return;
        }
        
        if ( $this->cached_categories === null ) {
            $this->cached_categories = $this->get_product_categories();
        }
        
        if ( $this->cached_tags === null ) {
            $this->cached_tags = $this->get_product_tags();
        }
    }
    
    /**
     * Get cached categories
     */
    public function get_cached_categories() {
        if ( $this->cached_categories === null ) {
            $this->cache_terms();
        }
        return $this->cached_categories;
    }
    
    /**
     * Get cached tags
     */
    public function get_cached_tags() {
        if ( $this->cached_tags === null ) {
            $this->cache_terms();
        }
        return $this->cached_tags;
    }
    
    /**
     * Get WooCommerce product categories
     */
    private function get_product_categories() {
        if ( ! $this->is_woocommerce_active() || ! taxonomy_exists( 'product_cat' ) ) {
            return array();
        }
        
        $categories = get_terms( array(
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'orderby'    => 'name',
            'order'      => 'ASC',
            'number'     => 500,
        ) );
        
        return is_wp_error( $categories ) ? array() : $categories;
    }
    
    /**
     * Get WooCommerce product tags
     */
    private function get_product_tags() {
        if ( ! $this->is_woocommerce_active() || ! taxonomy_exists( 'product_tag' ) ) {
            return array();
        }
        
        $tags = get_terms( array(
            'taxonomy'   => 'product_tag',
            'hide_empty' => false,
            'orderby'    => 'name',
            'order'      => 'ASC',
            'number'     => 500,
        ) );
        
        return is_wp_error( $tags ) ? array() : $tags;
    }
    
    /**
     * Format terms for JavaScript
     */
    public function format_terms_for_js( $terms ) {
        if ( ! $terms || is_wp_error( $terms ) || empty( $terms ) ) {
            return array();
        }
        
        $formatted = array();
        foreach ( $terms as $term ) {
            $formatted[] = array(
                'id'    => absint( $term->term_id ),
                'name'  => esc_html( $term->name ),
                'slug'  => esc_html( $term->slug ),
                'count' => absint( $term->count ),
            );
        }
        return $formatted;
    }
    
    /**
     * Get valid operators for a field type
     */
    public function get_operators_for_field( $field_type ) {
        switch ( $field_type ) {
            case 'price':
            case 'quantity':
                return $this->select_options['numeric_operators'];
            case 'product-status':
                return $this->select_options['stock_status_operators'];
            case 'category':
                return $this->select_options['category_operators'];
            case 'tags':
                return $this->select_options['tags_operators'];
            case 'title':
            default:
                return $this->select_options['text_operators'];
        }
    }
    
    /**
     * Sanitize and validate conditions
     */
    public function sanitize_and_validate_conditions( $conditions ) {
        if ( ! is_array( $conditions ) ) {
            return array();
        }
        
        $sanitized = array();
        foreach ( $conditions as $group_index => $group ) {
            if ( ! is_array( $group ) ) {
                continue;
            }
            
            $sanitized_group = array();
            foreach ( $group as $cond_index => $condition ) {
                if ( ! is_array( $condition ) ) {
                    continue;
                }
                
                $field_type = sanitize_text_field( $condition['field_type'] ?? '' );
                $operator   = sanitize_text_field( $condition['operator'] ?? '' );
                
                // Validate operator for field type
                if ( ! empty( $operator ) ) {
                    $valid_operators = $this->get_operators_for_field( $field_type );
                    if ( ! array_key_exists( $operator, $valid_operators ) ) {
                        $operator = array_key_first( $valid_operators );
                    }
                } else {
                    $valid_operators = $this->get_operators_for_field( $field_type );
                    $operator = array_key_first( $valid_operators );
                }
                
                $sanitized_condition = array(
                    'field_type'   => $field_type,
                    'operator'     => $operator,
                    'value'        => sanitize_text_field( $condition['value'] ?? '' ),
                    'stock_status' => sanitize_text_field( $condition['stock_status'] ?? '' ),
                );
                
                // Only add non-empty conditions
                if ( ! empty( $sanitized_condition['field_type'] ) ) {
                    $sanitized_group[] = $sanitized_condition;
                }
            }
            
            if ( ! empty( $sanitized_group ) ) {
                $sanitized[] = $sanitized_group;
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Evaluate conditions for a product (OR between groups, AND within group)
     */
    public function evaluate_conditions( $conditions, $product_id ) {
        if ( empty( $conditions ) || ! is_array( $conditions ) ) {
            return false;
        }
        
        // OR logic between groups
        foreach ( $conditions as $group ) {
            if ( $this->evaluate_condition_group( $group, $product_id ) ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Evaluate a condition group (AND logic)
     */
    private function evaluate_condition_group( $conditions, $product_id ) {
        if ( empty( $conditions ) || ! is_array( $conditions ) ) {
            return false;
        }
        
        // AND logic within group
        foreach ( $conditions as $condition ) {
            if ( ! $this->evaluate_single_condition( $condition, $product_id ) ) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Evaluate a single condition
     */
    private function evaluate_single_condition( $condition, $product_id ) {
        if ( ! $this->is_woocommerce_active() ) {
            return false;
        }
        
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return false;
        }
        
        $field_type   = $condition['field_type'] ?? '';
        $operator     = $condition['operator'] ?? '';
        $value        = $condition['value'] ?? '';
        $stock_status = $condition['stock_status'] ?? '';
        
        switch ( $field_type ) {
            case 'product-status':
                $product_stock_status = $product->get_stock_status();
                return $this->compare_values( $product_stock_status, $operator, $stock_status );
                
            case 'category':
                $categories = wp_get_post_terms( $product_id, 'product_cat', array( 'fields' => 'ids' ) );
                if ( is_wp_error( $categories ) ) {
                    return false;
                }
                return $this->compare_array_values( $categories, $operator, absint( $value ) );
                
            case 'tags':
                $tags = wp_get_post_terms( $product_id, 'product_tag', array( 'fields' => 'ids' ) );
                if ( is_wp_error( $tags ) ) {
                    return false;
                }
                return $this->compare_array_values( $tags, $operator, absint( $value ) );
                
            case 'price':
                $price = floatval( $product->get_price() );
                return $this->compare_numeric_values( $price, $operator, floatval( $value ) );
                
            case 'quantity':
                $quantity = $product->get_stock_quantity();
                $quantity = $quantity !== null ? absint( $quantity ) : 0;
                return $this->compare_numeric_values( $quantity, $operator, absint( $value ) );
                
            case 'title':
                $title = $product->get_name();
                return $this->compare_text_values( $title, $operator, $value );
        }
        
        return false;
    }
    
    /**
     * Compare two values
     */
    private function compare_values( $product_value, $operator, $condition_value ) {
        switch ( $operator ) {
            case 'equal':
                return $product_value === $condition_value;
            case 'not-equal':
                return $product_value !== $condition_value;
            default:
                return false;
        }
    }
    
    /**
     * Compare array values (categories/tags)
     */
    private function compare_array_values( $product_values, $operator, $condition_value ) {
        $has_value = in_array( $condition_value, $product_values, true );
        
        switch ( $operator ) {
            case 'equal':
                return $has_value;
            case 'not-equal':
                return ! $has_value;
            default:
                return false;
        }
    }
    
    /**
     * Compare numeric values
     */
    private function compare_numeric_values( $product_value, $operator, $condition_value ) {
        switch ( $operator ) {
            case 'equal':
                return abs( $product_value - $condition_value ) < 0.001;
            case 'not-equal':
                return abs( $product_value - $condition_value ) >= 0.001;
            case 'greater':
                return $product_value > $condition_value;
            case 'greater-equal':
                return $product_value >= $condition_value;
            case 'less':
                return $product_value < $condition_value;
            case 'less-equal':
                return $product_value <= $condition_value;
            default:
                return false;
        }
    }
    
    /**
     * Compare text values
     */
    private function compare_text_values( $product_value, $operator, $condition_value ) {
        $product_value   = strtolower( trim( $product_value ) );
        $condition_value = strtolower( trim( $condition_value ) );
        
        switch ( $operator ) {
            case 'equal':
                return $product_value === $condition_value;
            case 'not-equal':
                return $product_value !== $condition_value;
            case 'contains':
                return strpos( $product_value, $condition_value ) !== false;
            case 'not-contains':
                return strpos( $product_value, $condition_value ) === false;
            default:
                return false;
        }
    }
}
