/**
 * Frontend JavaScript for Advance Custom Messages Woo
 * 
 * Handles:
 * - Text ticker animation
 * - Stock counter animations
 * - Countdown timer
 * - Discount code copy functionality
 * - Social share interactions
 */

jQuery(document).ready(function($) {
    'use strict';
    
    console.log('SCMW Frontend JS loaded');
    
    /**
     * TEXT TICKER ANIMATION
     */
    function initTextTicker() {
        $('.scmw-ticker-container').each(function() {
            var container = $(this);
            var ticker = container.find('.scmw-ticker-text').first();
            var speed = ticker.data('speed') || 50;
            
            // Calculate animation duration based on content width and speed
            var contentWidth = ticker.outerWidth();
            var duration = (contentWidth / speed) + 's';
            
            // Apply animation
            ticker.css({
                'animation-duration': duration
            });
            
            // Clone for seamless loop if needed
            var clone = ticker.clone();
            clone.attr('aria-hidden', 'true');
            container.find('.scmw-ticker-wrapper').append(clone);
            
            console.log('Text ticker initialized with speed:', speed);
        });
    }
    
    /**
     * STOCK COUNTER ANIMATION
     */
    function initStockCounter() {
        $('.scmw-stock-count').each(function() {
            var element = $(this);
            var targetCount = parseInt(element.text()) || 0;
            
            // Animate count up
            animateValue(element, 0, targetCount, 1000);
        });
    }
    
    /**
     * Animate number from start to end
     */
    function animateValue(element, start, end, duration) {
        var range = end - start;
        var current = start;
        var increment = range / (duration / 16); // 60fps
        
        var timer = setInterval(function() {
            current += increment;
            if ((increment > 0 && current >= end) || (increment < 0 && current <= end)) {
                current = end;
                clearInterval(timer);
            }
            element.text(Math.floor(current));
        }, 16);
    }
    
    /**
     * COUNTDOWN TIMER
     */
    function initCountdown() {
        $('.scmw-countdown-container').each(function() {
            var container = $(this);
            var endTime = parseInt(container.data('end-time'));
            
            if (!endTime) {
                return;
            }
            
            // Update countdown every second
            var countdownInterval = setInterval(function() {
                updateCountdown(container, endTime, countdownInterval);
            }, 1000);
            
            // Initial update
            updateCountdown(container, endTime, countdownInterval);
            
            console.log('Countdown timer initialized');
        });
    }
    
    /**
     * Update countdown display
     */
    function updateCountdown(container, endTime, interval) {
        var now = Math.floor(Date.now() / 1000);
        var remaining = endTime - now;
        
        if (remaining <= 0) {
            clearInterval(interval);
            container.fadeOut(500, function() {
                $(this).remove();
            });
            return;
        }
        
        // Calculate time units
        var days = Math.floor(remaining / 86400);
        var hours = Math.floor((remaining % 86400) / 3600);
        var minutes = Math.floor((remaining % 3600) / 60);
        var seconds = remaining % 60;
        
        // Update display with leading zeros
        container.find('.scmw-days').text(padZero(days));
        container.find('.scmw-hours').text(padZero(hours));
        container.find('.scmw-minutes').text(padZero(minutes));
        container.find('.scmw-seconds').text(padZero(seconds));
    }
    
    /**
     * Add leading zero to single digit numbers
     */
    function padZero(num) {
        return num < 10 ? '0' + num : num.toString();
    }
    
    /**
     * DISCOUNT CODE COPY FUNCTIONALITY
     */
    $(document).on('click', '.scmw-copy-btn', function(e) {
        e.preventDefault();
        
        var button = $(this);
        var code = button.data('code');
        var originalText = button.text();
        
        // Copy to clipboard
        copyToClipboard(code).then(function() {
            // Success feedback
            button.addClass('copied').text(scmwFrontend.strings.copied);
            
            // Reset after 2 seconds
            setTimeout(function() {
                button.removeClass('copied').text(originalText);
            }, 2000);
            
            console.log('Discount code copied:', code);
        }).catch(function(err) {
            // Error feedback
            alert(scmwFrontend.strings.copyFailed);
            console.error('Copy failed:', err);
        });
    });
    
    /**
     * Copy text to clipboard
     */
    function copyToClipboard(text) {
        // Modern approach using Clipboard API
        if (navigator.clipboard && navigator.clipboard.writeText) {
            return navigator.clipboard.writeText(text);
        }
        
        // Fallback for older browsers
        return new Promise(function(resolve, reject) {
            var textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.top = '0';
            textarea.style.left = '0';
            textarea.style.width = '1px';
            textarea.style.height = '1px';
            textarea.style.padding = '0';
            textarea.style.border = 'none';
            textarea.style.outline = 'none';
            textarea.style.boxShadow = 'none';
            textarea.style.background = 'transparent';
            
            document.body.appendChild(textarea);
            textarea.focus();
            textarea.select();
            
            try {
                var successful = document.execCommand('copy');
                document.body.removeChild(textarea);
                
                if (successful) {
                    resolve();
                } else {
                    reject(new Error('Copy command failed'));
                }
            } catch (err) {
                document.body.removeChild(textarea);
                reject(err);
            }
        });
    }
    
    /**
     * SOCIAL SHARE BUTTONS
     */
    $(document).on('click', '.scmw-social-btn', function(e) {
        e.preventDefault();
        
        var url = $(this).attr('href');
        var platform = $(this).attr('class').match(/scmw-social-btn (\w+)/)[1];
        
        // Open in popup window
        var width = 600;
        var height = 400;
        var left = (screen.width - width) / 2;
        var top = (screen.height - height) / 2;
        
        var options = 'width=' + width + ',height=' + height + ',top=' + top + ',left=' + left;
        options += ',toolbar=0,location=0,menubar=0,scrollbars=1,resizable=1';
        
        window.open(url, 'share-' + platform, options);
        
        console.log('Social share opened:', platform);
        
        return false;
    });
    
    /**
     * ENHANCED ANIMATIONS ON SCROLL
     */
    function initScrollAnimations() {
        // Check if IntersectionObserver is supported
        if ('IntersectionObserver' in window) {
            var observer = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        $(entry.target).addClass('scmw-visible');
                        
                        // Trigger stock counter animation when visible
                        if ($(entry.target).hasClass('scmw-stock-counter')) {
                            animateStockCounterOnScroll($(entry.target));
                        }
                    }
                });
            }, {
                threshold: 0.1
            });
            
            // Observe all message containers
            $('.scmw-message-container').each(function() {
                observer.observe(this);
            });
        } else {
            // Fallback: add visible class immediately
            $('.scmw-message-container').addClass('scmw-visible');
        }
    }
    
    /**
     * Animate stock counter when scrolled into view
     */
    function animateStockCounterOnScroll(container) {
        var countElement = container.find('.scmw-stock-count');
        if (countElement.length > 0 && !countElement.data('animated')) {
            countElement.data('animated', true);
            var targetCount = parseInt(countElement.text()) || 0;
            animateValue(countElement, 0, targetCount, 1500);
        }
    }
    
    /**
     * TICKER PAUSE ON HOVER
     */
    $(document).on('mouseenter', '.scmw-ticker-container', function() {
        $(this).find('.scmw-ticker-text').css('animation-play-state', 'paused');
    }).on('mouseleave', '.scmw-ticker-container', function() {
        $(this).find('.scmw-ticker-text').css('animation-play-state', 'running');
    });
    
    /**
     * RESPONSIVE ADJUSTMENTS
     */
    function handleResponsive() {
        var windowWidth = $(window).width();
        
        // Adjust countdown timer size on mobile
        if (windowWidth < 768) {
            $('.scmw-countdown-value').css('font-size', '24px');
            $('.scmw-countdown-label').css('font-size', '10px');
        } else {
            $('.scmw-countdown-value').css('font-size', '32px');
            $('.scmw-countdown-label').css('font-size', '12px');
        }
        
        // Adjust ticker speed on mobile
        if (windowWidth < 768) {
            $('.scmw-ticker-text').each(function() {
                var speed = $(this).data('speed') || 50;
                var mobileSpeed = speed * 0.7; // Slower on mobile
                var contentWidth = $(this).outerWidth();
                var duration = (contentWidth / mobileSpeed) + 's';
                $(this).css('animation-duration', duration);
            });
        }
    }
    
    /**
     * ACCESSIBILITY ENHANCEMENTS
     */
    function initAccessibility() {
        // Add ARIA labels to social buttons
        $('.scmw-social-btn').each(function() {
            var platform = $(this).attr('class').match(/scmw-social-btn (\w+)/);
            if (platform) {
                var label = 'Share on ' + platform[1].charAt(0).toUpperCase() + platform[1].slice(1);
                $(this).attr('aria-label', label);
            }
        });
        
        // Add ARIA live region for countdown
        $('.scmw-countdown-timer').attr('aria-live', 'polite');
        
        // Add ARIA labels for discount code
        $('.scmw-copy-btn').attr('aria-label', 'Copy discount code');
    }
    
    /**
     * PERFORMANCE OPTIMIZATION
     */
    function optimizePerformance() {
        // Use requestAnimationFrame for smooth animations
        if (window.requestAnimationFrame) {
            window.requestAnimationFrame(function() {
                initTextTicker();
                initScrollAnimations();
            });
        } else {
            initTextTicker();
            initScrollAnimations();
        }
        
        // Lazy load images in custom image messages
        if ('loading' in HTMLImageElement.prototype) {
            $('.scmw-custom-image-container img').attr('loading', 'lazy');
        }
    }
    
    /**
     * INITIALIZE ALL FEATURES
     */
    function init() {
        optimizePerformance();
        initStockCounter();
        initCountdown();
        initAccessibility();
        handleResponsive();
        
        console.log('SCMW Frontend initialized successfully');
    }
    
    // Initialize on document ready
    init();
    
    // Re-run responsive adjustments on window resize (debounced)
    var resizeTimer;
    $(window).on('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            handleResponsive();
        }, 250);
    });
    
    /**
     * PUBLIC API for external integrations
     */
    window.SCMW = {
        version: '1.0.0',
        
        // Refresh countdown timers
        refreshCountdowns: function() {
            initCountdown();
        },
        
        // Refresh tickers
        refreshTickers: function() {
            initTextTicker();
        },
        
        // Manually trigger stock counter animation
        animateStockCounter: function(selector) {
            $(selector).find('.scmw-stock-count').each(function() {
                var element = $(this);
                var targetCount = parseInt(element.text()) || 0;
                animateValue(element, 0, targetCount, 1000);
            });
        },
        
        // Copy discount code programmatically
        copyDiscountCode: function(code) {
            return copyToClipboard(code);
        }
    };
    
    console.log('SCMW Public API available:', window.SCMW);
});