/**
 * Discount Code Scripts - Frontend
 * 
 * @package Advance Custom Messages Woo
 * @since 1.1.0
 */

jQuery(document).ready(function($) {
    'use strict';
    
    console.log('SCMW Discount Scripts loaded');
    
    /**
     * Copy discount code to clipboard
     */
    function copyToClipboard(text) {
        // Modern approach using Clipboard API
        if (navigator.clipboard && navigator.clipboard.writeText) {
            return navigator.clipboard.writeText(text);
        }
        
        // Fallback for older browsers
        return new Promise(function(resolve, reject) {
            var textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.top = '0';
            textarea.style.left = '0';
            textarea.style.width = '1px';
            textarea.style.height = '1px';
            textarea.style.padding = '0';
            textarea.style.border = 'none';
            textarea.style.outline = 'none';
            textarea.style.boxShadow = 'none';
            textarea.style.background = 'transparent';
            
            document.body.appendChild(textarea);
            textarea.focus();
            textarea.select();
            
            try {
                var successful = document.execCommand('copy');
                document.body.removeChild(textarea);
                
                if (successful) {
                    resolve();
                } else {
                    reject(new Error('Copy command failed'));
                }
            } catch (err) {
                document.body.removeChild(textarea);
                reject(err);
            }
        });
    }
    
    /**
     * Handle copy button clicks (all styles)
     */
    $(document).on('click', '.scmw-copy-btn', function(e) {
        e.preventDefault();
        
        var button = $(this);
        var code = button.data('code');
        var originalContent = button.html();
        
        if (!code) {
            return;
        }
        
        // Disable button during copy
        button.prop('disabled', true);
        
        // Copy to clipboard
        copyToClipboard(code).then(function() {
            // Success feedback
            button.addClass('copied');
            
            // Different feedback for different styles
            if (button.closest('.scmw-discount-style-3').length) {
                // Style 3 - just change color
                button.html('<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M3 8L6.5 11.5L13 5" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>');
            } else {
                // Style 1 & 2 - text feedback
                button.text(scmwDiscount.strings.copied);
            }
            
            // Reset after 2 seconds
            setTimeout(function() {
                button.removeClass('copied').html(originalContent).prop('disabled', false);
            }, 2000);
            
            console.log('Discount code copied:', code);
            
            // Track copy event (if analytics available)
            if (typeof gtag === 'function') {
                gtag('event', 'discount_code_copied', {
                    'event_category': 'Discount',
                    'event_label': code
                });
            }
        }).catch(function(err) {
            // Error feedback
            button.prop('disabled', false);
            alert(scmwDiscount.strings.copyFailed);
            console.error('Copy failed:', err);
        });
    });
    
    /**
     * Handle reveal button clicks (Style 2)
     */
    $(document).on('click', '.scmw-reveal-btn', function(e) {
        e.preventDefault();
        
        var button = $(this);
        var wrapper = button.closest('.scmw-discount-reveal-wrapper');
        var codeRevealed = wrapper.find('.scmw-discount-code-revealed');
        var isActive = button.hasClass('active');
        
        if (isActive) {
            // Hide code
            button.removeClass('active');
            button.find('.scmw-reveal-text').text(scmwDiscount.strings.showCode);
            button.attr('aria-expanded', 'false');
            
            codeRevealed.removeClass('active').slideUp(300);
            
            console.log('Discount code hidden');
        } else {
            // Show code
            button.addClass('active');
            button.find('.scmw-reveal-text').text(scmwDiscount.strings.hideCode);
            button.attr('aria-expanded', 'true');
            
            codeRevealed.addClass('active').slideDown(300);
            
            console.log('Discount code revealed');
            
            // Track reveal event (if analytics available)
            if (typeof gtag === 'function') {
                gtag('event', 'discount_code_revealed', {
                    'event_category': 'Discount',
                    'event_label': button.data('code')
                });
            }
        }
    });
    
    /**
     * Accessibility: Handle keyboard navigation
     */
    $(document).on('keydown', '.scmw-reveal-btn, .scmw-copy-btn', function(e) {
        // Enter or Space key
        if (e.which === 13 || e.which === 32) {
            e.preventDefault();
            $(this).trigger('click');
        }
    });
    
    /**
     * Initialize animations on scroll
     */
    function initScrollAnimations() {
        if ('IntersectionObserver' in window) {
            var observer = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        $(entry.target).addClass('scmw-visible');
                        
                        // Animate badge rotation for Style 3
                        if ($(entry.target).hasClass('scmw-discount-style-3')) {
                            animateBadge($(entry.target).find('.scmw-discount-badge'));
                        }
                    }
                });
            }, {
                threshold: 0.2
            });
            
            $('.scmw-discount-container').each(function() {
                observer.observe(this);
            });
        } else {
            // Fallback
            $('.scmw-discount-container').addClass('scmw-visible');
        }
    }
    
    /**
     * Animate badge entrance
     */
    function animateBadge(badge) {
        if (badge.length && !badge.data('animated')) {
            badge.data('animated', true);
            badge.css({
                opacity: 0,
                transform: 'scale(0.8) rotate(-180deg)'
            });
            
            setTimeout(function() {
                badge.css({
                    transition: 'all 0.6s cubic-bezier(0.34, 1.56, 0.64, 1)',
                    opacity: 1,
                    transform: 'scale(1) rotate(0deg)'
                });
            }, 100);
        }
    }
    
    /**
     * Prevent double clicks
     */
    var clickTimeout;
    $(document).on('click', '.scmw-copy-btn, .scmw-reveal-btn', function() {
        var button = $(this);
        clearTimeout(clickTimeout);
        button.prop('disabled', true);
        
        clickTimeout = setTimeout(function() {
            if (!button.hasClass('copied')) {
                button.prop('disabled', false);
            }
        }, 300);
    });
    
    /**
     * Handle touch events for mobile
     */
    var touchStartTime;
    
    $(document).on('touchstart', '.scmw-reveal-btn', function() {
        touchStartTime = Date.now();
    });
    
    $(document).on('touchend', '.scmw-reveal-btn', function(e) {
        var touchDuration = Date.now() - touchStartTime;
        
        // Only trigger if tap was quick (not a scroll)
        if (touchDuration < 200) {
            $(this).trigger('click');
        }
    });
    
    /**
     * Initialize on page load
     */
    function init() {
        initScrollAnimations();
        console.log('SCMW Discount Scripts initialized successfully');
    }
    
    // Run initialization
    init();
    
    /**
     * Public API for external integrations
     */
    window.SCMWDiscount = {
        version: '1.1.0',
        
        // Manually copy code
        copyCode: function(code) {
            return copyToClipboard(code);
        },
        
        // Reveal a specific discount code
        revealCode: function(selector) {
            $(selector).find('.scmw-reveal-btn').trigger('click');
        },
        
        // Reset all discount states
        reset: function() {
            $('.scmw-reveal-btn').removeClass('active');
            $('.scmw-discount-code-revealed').hide();
            $('.scmw-copy-btn').removeClass('copied').prop('disabled', false);
        }
    };
    
    console.log('SCMW Discount Public API available:', window.SCMWDiscount);
});