/**
 * Discount Code Scripts - Admin
 * 
 * @package Advance Custom Messages Woo
 * @since 1.1.0
 */

jQuery(document).ready(function($) {
    'use strict';
    
    console.log('SCMW Discount Admin Scripts loaded');
    
    /**
     * Initialize color pickers for discount styles
     */
    function initColorPickers() {
        if ($.fn.wpColorPicker) {
            $('.scmw-discount-style-options .scmw-color-picker').each(function() {
                var $this = $(this);
                
                // Only initialize if not already initialized
                if (!$this.hasClass('wp-color-picker')) {
                    $this.wpColorPicker({
                        change: function(event, ui) {
                            // Optional: trigger preview update
                            updateStylePreview();
                        }
                    });
                }
            });
        }
    }
    
    /**
     * Handle discount style changes
     */
    $(document).on('change', '.scmw-discount-style-select', function() {
        var selectedStyle = $(this).val();
        
        console.log('Discount style changed to:', selectedStyle);
        
        // Hide all style-specific options
        $('.scmw-discount-style-options').hide();
        
        // Show the selected style's options
        $('.scmw-discount-style-' + selectedStyle + '-options').show();
        
        // Update preview descriptions
        $('.scmw-style-preview').hide();
        $('.scmw-style-preview-' + selectedStyle).show();
        
        // Re-initialize color pickers for visible options
        setTimeout(function() {
            initColorPickers();
        }, 100);
    });
    
    /**
     * Update style preview (optional feature)
     */
    function updateStylePreview() {
        // This could be expanded to show a live preview of the discount code
        // For now, it's a placeholder for future enhancement
        console.log('Style preview updated');
    }
    
    /**
     * Validate discount code settings
     */
    function validateDiscountSettings() {
        var isValid = true;
        var errors = [];
        
        // Get message type
        var messageType = $('#scmw_message_type').val();
        
        // Only validate if message type is discount_code
        if (messageType !== 'discount_code') {
            return true;
        }
        
        // Validate discount code
        var discountCode = $('#scmw_discount_code').val();
        if (!discountCode || discountCode.trim() === '') {
            errors.push('Please enter a discount code');
            $('#scmw_discount_code').addClass('scmw-discount-error');
            isValid = false;
        } else {
            $('#scmw_discount_code').removeClass('scmw-discount-error');
        }
        
        // Validate style-specific options
        var discountStyle = $('#scmw_discount_style').val();
        
        if (discountStyle === 'style_3') {
            var percentage = $('#scmw_discount_percentage').val();
            
            if (!percentage || percentage < 1 || percentage > 100) {
                errors.push('Please enter a valid percentage (1-100)');
                $('#scmw_discount_percentage').addClass('scmw-discount-error');
                isValid = false;
            } else {
                $('#scmw_discount_percentage').removeClass('scmw-discount-error');
            }
        }
        
        // Show errors if any
        if (!isValid) {
            $('.scmw-discount-error-messages').remove();
            
            var errorHtml = '<div class="scmw-discount-error-messages notice notice-error" style="margin: 15px 0;"><p><strong>Discount Code Errors:</strong></p><ul>';
            errors.forEach(function(error) {
                errorHtml += '<li>' + error + '</li>';
            });
            errorHtml += '</ul></div>';
            
            $('.scmw-panel-discount_code').prepend(errorHtml);
            
            // Scroll to errors
            $('html, body').animate({
                scrollTop: $('.scmw-discount-error-messages').offset().top - 100
            }, 500);
        } else {
            $('.scmw-discount-error-messages').remove();
        }
        
        return isValid;
    }
    
    /**
     * Form validation on submit
     */
    $('form#post').on('submit', function(e) {
        // Clear previous error states
        $('.scmw-discount-error').removeClass('scmw-discount-error');
        $('.scmw-discount-error-messages').remove();
        
        // Validate discount settings
        var isValid = validateDiscountSettings();
        
        if (!isValid) {
            e.preventDefault();
            return false;
        }
        
        return true;
    });
    
    /**
     * Real-time validation as user types
     */
    $('#scmw_discount_code').on('input', function() {
        var value = $(this).val().trim();
        
        if (value === '') {
            $(this).addClass('scmw-discount-error');
        } else {
            $(this).removeClass('scmw-discount-error');
        }
    });
    
    $('#scmw_discount_percentage').on('input', function() {
        var value = parseInt($(this).val());
        
        if (!value || value < 1 || value > 100) {
            $(this).addClass('scmw-discount-error');
        } else {
            $(this).removeClass('scmw-discount-error');
        }
    });
    
    /**
     * Sanitize discount code input
     */
    $('#scmw_discount_code').on('blur', function() {
        var value = $(this).val();
        
        // Convert to uppercase and remove spaces
        value = value.toUpperCase().replace(/\s+/g, '');
        
        // Remove special characters except hyphens and underscores
        value = value.replace(/[^A-Z0-9\-_]/g, '');
        
        $(this).val(value);
    });
    
    /**
     * Add helpful hints
     */
    function addHelpfulHints() {
        // Add hint for discount code
        if ($('#scmw_discount_code').length && !$('#scmw_discount_code').next('.scmw-discount-help-text').length) {
            var codeHint = '<div class="scmw-discount-help-text" style="margin-top: 10px;">' +
                '<strong>Tips:</strong> Use uppercase letters, numbers, hyphens, or underscores. ' +
                'Example: SUMMER2024, SAVE20, WELCOME-10' +
                '</div>';
            
            $('#scmw_discount_code').after(codeHint);
        }
    }
    
    /**
     * Handle percentage slider (if you want to add one)
     */
    function initPercentageSlider() {
        var percentageInput = $('#scmw_discount_percentage');
        
        if (percentageInput.length && !percentageInput.next('.scmw-percentage-value').length) {
            percentageInput.after('<span class="scmw-percentage-value" style="margin-left: 10px; font-weight: 600; color: #2271b1;">' + percentageInput.val() + '%</span>');
        }
        
        percentageInput.on('input', function() {
            $(this).next('.scmw-percentage-value').text($(this).val() + '%');
        });
    }
    
    /**
     * Copy example code to clipboard (admin preview)
     */
    function initAdminPreview() {
        // This could be expanded to show a live preview in the admin
        // For now, it's a placeholder for future enhancement
        console.log('Admin preview initialized');
    }
    
    /**
     * Handle keyboard shortcuts
     */
    $(document).on('keydown', function(e) {
        // Ctrl/Cmd + S to save (already handled by WordPress, but we can add feedback)
        if ((e.ctrlKey || e.metaKey) && e.which === 83) {
            // Just let it save normally, but we could add visual feedback here
            console.log('Saving discount code settings...');
        }
    });
    
    /**
     * Auto-save draft periodically (optional)
     */
    var autoSaveTimer;
    var autoSaveDelay = 60000; // 1 minute
    
    function startAutoSave() {
        clearTimeout(autoSaveTimer);
        
        autoSaveTimer = setTimeout(function() {
            // Trigger WordPress autosave if available
            if (typeof wp !== 'undefined' && wp.autosave) {
                console.log('Auto-saving discount code settings...');
                wp.autosave.server.triggerSave();
            }
            
            startAutoSave(); // Restart timer
        }, autoSaveDelay);
    }
    
    /**
     * Initialize all features
     */
    function init() {
        initColorPickers();
        addHelpfulHints();
        initPercentageSlider();
        initAdminPreview();
        
        // Trigger initial style change to show correct options
        $('.scmw-discount-style-select').trigger('change');
        
        // Start auto-save timer
        startAutoSave();
        
        console.log('SCMW Discount Admin Scripts initialized successfully');
    }
    
    // Run initialization
    init();
    
    /**
     * Clean up on page unload
     */
    $(window).on('beforeunload', function() {
        clearTimeout(autoSaveTimer);
    });
    
    /**
     * Public API for extensions
     */
    window.SCMWDiscountAdmin = {
        version: '1.1.0',
        
        // Validate settings programmatically
        validate: function() {
            return validateDiscountSettings();
        },
        
        // Refresh color pickers
        refreshColorPickers: function() {
            initColorPickers();
        },
        
        // Get current style settings
        getSettings: function() {
            return {
                style: $('#scmw_discount_style').val(),
                code: $('#scmw_discount_code').val(),
                description: $('#scmw_discount_description').val(),
                percentage: $('#scmw_discount_percentage').val(),
                bgColor: $('#scmw_discount_bg_color').val(),
                textColor: $('#scmw_discount_text_color').val()
            };
        }
    };
    
    console.log('SCMW Discount Admin Public API available:', window.SCMWDiscountAdmin);
});