/**
 * Admin JavaScript for Advance Custom Messages Woo
 * 
 * Handles:
 * - Message type switching
 * - Image upload
 * - Color picker
 * - Conditional logic UI
 * - Dynamic condition creation
 */

jQuery(document).ready(function($) {
    'use strict';
    
    console.log('SCMW Admin JS loaded');
    
    /**
     * Initialize color pickers
     */
    if ($.fn.wpColorPicker) {
        $('.scmw-color-picker').wpColorPicker();
    }
    
    /**
     * Message type switching
     */
    $('.scmw-message-type-select').on('change', function() {
        var selectedType = $(this).val();
        $('.scmw-message-type-panel').hide();
        $('.scmw-panel-' + selectedType).show();
    }).trigger('change');
    
    /**
     * Image Upload Handler
     */
    var mediaUploader;
    
    $(document).on('click', '.scmw-upload-image-button', function(e) {
        e.preventDefault();
        
        var button = $(this);
        
        // If the uploader object already exists, reopen it
        if (mediaUploader) {
            mediaUploader.open();
            return;
        }
        
        // Create the media uploader
        mediaUploader = wp.media({
            title: scmwAdmin.strings.selectImage,
            button: {
                text: scmwAdmin.strings.useImage
            },
            multiple: false
        });
        
        // When an image is selected
        mediaUploader.on('select', function() {
            var attachment = mediaUploader.state().get('selection').first().toJSON();
            
            // Update the hidden input
            $('#scmw_custom_image_id').val(attachment.id);
            
            // Update the preview
            button.siblings('.scmw-image-preview').html(
                '<img src="' + attachment.url + '" style="max-width: 300px; height: auto;" />'
            );
            
            // Show remove button
            button.siblings('.scmw-remove-image-button').show();
        });
        
        // Open the uploader
        mediaUploader.open();
    });
    
    /**
     * Remove Image Handler
     */
    $(document).on('click', '.scmw-remove-image-button', function(e) {
        e.preventDefault();
        
        $('#scmw_custom_image_id').val('');
        $(this).siblings('.scmw-image-preview').html('');
        $(this).hide();
    });
    
    /**
     * CONDITIONAL LOGIC HANDLERS
     */
    
    // Cache terms data
    var cachedCategories = typeof scmwAdmin !== 'undefined' && scmwAdmin.cachedTerms ? scmwAdmin.cachedTerms.categories : [];
    var cachedTags = typeof scmwAdmin !== 'undefined' && scmwAdmin.cachedTerms ? scmwAdmin.cachedTerms.tags : [];
    
    console.log('Cached categories:', cachedCategories.length);
    console.log('Cached tags:', cachedTags.length);

    

    /**
     * TICKER COLOR PRESETS
     */
    $(document).on('click', '.scmw-ticker-preset', function(e) {
        e.preventDefault();
        
        var preset = $(this).data('preset');
        var presets = {
            blue: {
                label_bg: '#0d6efd',
                label_color: '#ffffff',
                border_color: '#0d6efd',
                text_bg: '#ffffff',
                text_color: '#333333'
            },
            red: {
                label_bg: '#dc3545',
                label_color: '#ffffff',
                border_color: '#dc3545',
                text_bg: '#ffffff',
                text_color: '#333333'
            },
            green: {
                label_bg: '#28a745',
                label_color: '#ffffff',
                border_color: '#28a745',
                text_bg: '#ffffff',
                text_color: '#333333'
            },
            dark: {
                label_bg: '#1a202c',
                label_color: '#ffffff',
                border_color: '#2d3748',
                text_bg: '#2d3748',
                text_color: '#ffffff'
            },
            orange: {
                label_bg: '#fd7e14',
                label_color: '#ffffff',
                border_color: '#fd7e14',
                text_bg: '#ffffff',
                text_color: '#333333'
            }
        };
        
        if (presets[preset]) {
            var colors = presets[preset];
            
            // Update color picker fields
            $('#scmw_ticker_label_bg').val(colors.label_bg).trigger('change');
            $('#scmw_ticker_label_color').val(colors.label_color).trigger('change');
            $('#scmw_ticker_border_color').val(colors.border_color).trigger('change');
            $('#scmw_ticker_text_bg').val(colors.text_bg).trigger('change');
            $('#scmw_ticker_text_color').val(colors.text_color).trigger('change');
            
            // Update WordPress color pickers if they exist
            if ($.fn.wpColorPicker) {
                $('#scmw_ticker_label_bg').wpColorPicker('color', colors.label_bg);
                $('#scmw_ticker_label_color').wpColorPicker('color', colors.label_color);
                $('#scmw_ticker_border_color').wpColorPicker('color', colors.border_color);
                $('#scmw_ticker_text_bg').wpColorPicker('color', colors.text_bg);
                $('#scmw_ticker_text_color').wpColorPicker('color', colors.text_color);
            }
            
            // Visual feedback
            $(this).addClass('button-primary');
            setTimeout(function() {
                $('.scmw-ticker-preset').removeClass('button-primary');
            }, 500);
            
            console.log('Applied ticker preset:', preset);
        }
    });
    
    /**
     * Add OR Group
     */
    $(document).on('click', '.scmw-add-group', function(e) {
        e.preventDefault();
        
        var wrapper = $('#scmw-conditions-wrapper');
        var groupCount = wrapper.find('.scmw-condition-group').length;
        var newGroupIndex = groupCount;
        
        var groupHtml = createConditionGroup(newGroupIndex);
        wrapper.append(groupHtml);
        
        // Initialize the first condition row
        var newGroup = wrapper.find('.scmw-condition-group').last();
        initializeConditionRow(newGroup.find('.scmw-condition-row').first());
        
        console.log('Added new OR group:', newGroupIndex);
    });
    
    /**
     * Remove OR Group
     */
    $(document).on('click', '.scmw-remove-group', function(e) {
        e.preventDefault();
        
        if (confirm(scmwAdmin.strings.confirmRemoveGroup)) {
            $(this).closest('.scmw-condition-group').remove();
            console.log('Removed OR group');
        }
    });
    
    /**
     * Add AND Condition
     */
    $(document).on('click', '.scmw-add-condition', function(e) {
        e.preventDefault();
        
        var group = $(this).closest('.scmw-condition-group');
        var groupIndex = group.data('group-index');
        var conditionsWrapper = group.find('.scmw-conditions-wrapper');
        var conditionCount = conditionsWrapper.find('.scmw-condition-row').length;
        var newConditionIndex = conditionCount;
        
        var conditionHtml = createConditionRow(groupIndex, newConditionIndex);
        conditionsWrapper.append(conditionHtml);
        
        // Initialize the new condition
        var newCondition = conditionsWrapper.find('.scmw-condition-row').last();
        initializeConditionRow(newCondition);
        
        console.log('Added new AND condition to group:', groupIndex);
    });
    
    /**
     * Remove Condition
     */
    $(document).on('click', '.scmw-remove-condition', function(e) {
        e.preventDefault();
        
        var conditionsWrapper = $(this).closest('.scmw-conditions-wrapper');
        
        // Don't allow removing the last condition in a group
        if (conditionsWrapper.find('.scmw-condition-row').length > 1) {
            if (confirm(scmwAdmin.strings.confirmRemoveCondition)) {
                $(this).closest('.scmw-condition-row').remove();
                console.log('Removed condition');
            }
        } else {
            alert('You must have at least one condition per group.');
        }
    });
    
    /**
     * Field Type Change Handler
     */
    $(document).on('change', '.scmw-field-type-select', function() {
        var fieldType = $(this).val();
        var conditionRow = $(this).closest('.scmw-condition-row');
        
        console.log('Field type changed to:', fieldType);
        
        // Show/hide appropriate operator selects
        showHideOperatorSelects(conditionRow, fieldType);
        
        // Show/hide appropriate value inputs
        showHideValueInputs(conditionRow, fieldType);
    });
    
    /**
     * Show/Hide Operator Selects
     */
    function showHideOperatorSelects(conditionRow, fieldType) {
        var operatorContainer = conditionRow.find('.scmw-condition-operator');
        
        // Hide all operator selects
        operatorContainer.find('.scmw-operator-select').hide().prop('disabled', true);
        
        // Show the correct operator select
        var targetSelect;
        switch (fieldType) {
            case 'product-status':
                targetSelect = operatorContainer.find('.scmw-operator-stock');
                break;
            case 'category':
                targetSelect = operatorContainer.find('.scmw-operator-category');
                break;
            case 'tags':
                targetSelect = operatorContainer.find('.scmw-operator-tags');
                break;
            case 'price':
            case 'quantity':
                targetSelect = operatorContainer.find('.scmw-operator-numeric');
                break;
            case 'title':
            default:
                targetSelect = operatorContainer.find('.scmw-operator-text');
                break;
        }
        
        if (targetSelect.length > 0) {
            targetSelect.show().prop('disabled', false);
        }
    }
    
    /**
     * Show/Hide Value Inputs
     */
    function showHideValueInputs(conditionRow, fieldType) {
        var valueContainer = conditionRow.find('.scmw-condition-value');
        
        // Hide all value inputs
        valueContainer.find('.scmw-value-input').hide().prop('disabled', true);
        
        // Show the appropriate input
        var targetInput;
        switch (fieldType) {
            case 'product-status':
                targetInput = valueContainer.find('.scmw-stock-status-select');
                break;
            case 'category':
                targetInput = valueContainer.find('.scmw-category-select');
                break;
            case 'tags':
                targetInput = valueContainer.find('.scmw-tags-select');
                break;
            case 'price':
            case 'quantity':
                targetInput = valueContainer.find('.scmw-numeric-input');
                break;
            case 'title':
            default:
                targetInput = valueContainer.find('.scmw-text-input');
                break;
        }
        
        if (targetInput.length > 0) {
            targetInput.show().prop('disabled', false);
        }
    }
    
    /**
     * Create Condition Group HTML
     */
    function createConditionGroup(groupIndex) {
        return `
            <div class="scmw-condition-group" data-group-index="${groupIndex}">
                <div class="scmw-group-header">
                    <h4>OR Group ${groupIndex + 1}</h4>
                    <button type="button" class="button-link scmw-remove-group" title="Remove Group">×</button>
                </div>
                <div class="scmw-conditions-wrapper">
                    ${createConditionRow(groupIndex, 0)}
                </div>
                <div class="scmw-group-actions">
                    <button type="button" class="button button-small scmw-add-condition">+ Add AND Condition</button>
                </div>
            </div>
        `;
    }
    
    /**
     * Create Condition Row HTML
     */
    function createConditionRow(groupIndex, conditionIndex) {
        var categoriesOptions = getCategoriesOptionsHTML();
        var tagsOptions = getTagsOptionsHTML();
        
        return `
            <div class="scmw-condition-row" data-condition-index="${conditionIndex}">
                <div class="scmw-condition-field">
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][field_type]" class="scmw-field-type-select">
                        <option value="product-status">Product Stock</option>
                        <option value="category">Product Category</option>
                        <option value="tags">Product Tags</option>
                        <option value="price">Product Price</option>
                        <option value="quantity">Product Quantity</option>
                        <option value="title">Product Title</option>
                    </select>
                </div>
                
                <div class="scmw-condition-operator">
                    <!-- Stock operators -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][operator]" 
                            class="scmw-operator-select scmw-operator-stock" 
                            data-field-type="product-status">
                        <option value="equal">Is</option>
                    </select>
                    
                    <!-- Text operators -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][operator]" 
                            class="scmw-operator-select scmw-operator-text" 
                            data-field-type="title" 
                            style="display:none;" disabled>
                        <option value="equal">Equal To</option>
                        <option value="not-equal">Not Equal To</option>
                        <option value="contains">Contains</option>
                        <option value="not-contains">Does Not Contain</option>
                    </select>
                    
                    <!-- Numeric operators -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][operator]" 
                            class="scmw-operator-select scmw-operator-numeric" 
                            data-field-type="price,quantity" 
                            style="display:none;" disabled>
                        <option value="equal">Equal To</option>
                        <option value="not-equal">Not Equal To</option>
                        <option value="greater">Greater Than</option>
                        <option value="greater-equal">Greater or Equal To</option>
                        <option value="less">Less Than</option>
                        <option value="less-equal">Less or Equal To</option>
                    </select>
                    
                    <!-- Category operators -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][operator]" 
                            class="scmw-operator-select scmw-operator-category" 
                            data-field-type="category" 
                            style="display:none;" disabled>
                        <option value="equal">Is in Category</option>
                        <option value="not-equal">Is not in Category</option>
                    </select>
                    
                    <!-- Tags operators -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][operator]" 
                            class="scmw-operator-select scmw-operator-tags" 
                            data-field-type="tags" 
                            style="display:none;" disabled>
                        <option value="equal">Has Tag</option>
                        <option value="not-equal">Does not have Tag</option>
                    </select>
                </div>
                
                <div class="scmw-condition-value">
                    <!-- Stock status select -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][stock_status]" 
                            class="scmw-value-input scmw-stock-status-select" 
                            data-field-type="product-status">
                        <option value="instock">In Stock</option>
                        <option value="outofstock">Out of Stock</option>
                        <option value="onbackorder">On Backorder</option>
                    </select>
                    
                    <!-- Category select -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][value]" 
                            class="scmw-value-input scmw-category-select" 
                            data-field-type="category" 
                            style="display:none;" disabled>
                        <option value="">Select Category...</option>
                        ${categoriesOptions}
                    </select>
                    
                    <!-- Tags select -->
                    <select name="scmw_conditions[${groupIndex}][${conditionIndex}][value]" 
                            class="scmw-value-input scmw-tags-select" 
                            data-field-type="tags" 
                            style="display:none;" disabled>
                        <option value="">Select Tag...</option>
                        ${tagsOptions}
                    </select>
                    
                    <!-- Numeric input -->
                    <input type="number" 
                           name="scmw_conditions[${groupIndex}][${conditionIndex}][value]" 
                           class="scmw-value-input scmw-numeric-input" 
                           data-field-type="price,quantity" 
                           min="0" 
                           step="0.01" 
                           placeholder="Enter value" 
                           style="display:none;" 
                           disabled />
                    
                    <!-- Text input -->
                    <input type="text" 
                           name="scmw_conditions[${groupIndex}][${conditionIndex}][value]" 
                           class="scmw-value-input scmw-text-input" 
                           data-field-type="title" 
                           placeholder="Enter text" 
                           style="display:none;" 
                           disabled />
                </div>
                
                <div class="scmw-condition-actions">
                    <button type="button" class="button-link scmw-remove-condition" title="Remove Condition">×</button>
                </div>
            </div>
        `;
    }
    
    /**
     * Get Categories Options HTML
     */
    function getCategoriesOptionsHTML() {
        var html = '';
        
        if (cachedCategories && cachedCategories.length > 0) {
            cachedCategories.forEach(function(category) {
                html += `<option value="${category.id}">${category.name} (${category.count})</option>`;
            });
        } else {
            // Fallback: copy from existing dropdown
            var existingSelect = $('.scmw-category-select').first();
            if (existingSelect.length > 0) {
                existingSelect.find('option').each(function() {
                    var val = $(this).val();
                    var text = $(this).text();
                    if (val && val !== '' && !$(this).is('[disabled]')) {
                        html += `<option value="${val}">${text}</option>`;
                    }
                });
            }
        }
        
        return html;
    }
    
    /**
     * Get Tags Options HTML
     */
    function getTagsOptionsHTML() {
        var html = '';
        
        if (cachedTags && cachedTags.length > 0) {
            cachedTags.forEach(function(tag) {
                html += `<option value="${tag.id}">${tag.name} (${tag.count})</option>`;
            });
        } else {
            // Fallback: copy from existing dropdown
            var existingSelect = $('.scmw-tags-select').first();
            if (existingSelect.length > 0) {
                existingSelect.find('option').each(function() {
                    var val = $(this).val();
                    var text = $(this).text();
                    if (val && val !== '' && !$(this).is('[disabled]')) {
                        html += `<option value="${val}">${text}</option>`;
                    }
                });
            }
        }
        
        return html;
    }
    
    /**
     * Initialize Condition Row
     */
    function initializeConditionRow(conditionRow) {
        var fieldType = conditionRow.find('.scmw-field-type-select').val();
        showHideOperatorSelects(conditionRow, fieldType);
        showHideValueInputs(conditionRow, fieldType);
    }
    
    /**
     * Initialize existing condition rows on page load
     */
    $('.scmw-condition-row').each(function() {
        initializeConditionRow($(this));
    });
    
    /**
     * Form Validation
     */
    $('form#post').on('submit', function(e) {
        var isValid = true;
        var errors = [];
        
        // Validate each condition group
        $('.scmw-condition-group').each(function() {
            var groupIndex = $(this).data('group-index');
            
            $(this).find('.scmw-condition-row').each(function() {
                var fieldType = $(this).find('.scmw-field-type-select').val();
                var valueContainer = $(this).find('.scmw-condition-value');
                
                // Validate based on field type
                switch (fieldType) {
                    case 'product-status':
                        var stockSelect = valueContainer.find('.scmw-stock-status-select:visible:enabled');
                        if (stockSelect.length > 0 && !stockSelect.val()) {
                            errors.push(`Group ${groupIndex + 1}: Please select a stock status`);
                            isValid = false;
                        }
                        break;
                        
                    case 'category':
                        var categorySelect = valueContainer.find('.scmw-category-select:visible:enabled');
                        if (categorySelect.length > 0 && !categorySelect.val()) {
                            errors.push(`Group ${groupIndex + 1}: Please select a category`);
                            isValid = false;
                        }
                        break;
                        
                    case 'tags':
                        var tagSelect = valueContainer.find('.scmw-tags-select:visible:enabled');
                        if (tagSelect.length > 0 && !tagSelect.val()) {
                            errors.push(`Group ${groupIndex + 1}: Please select a tag`);
                            isValid = false;
                        }
                        break;
                        
                    case 'price':
                    case 'quantity':
                        var numericInput = valueContainer.find('.scmw-numeric-input:visible:enabled');
                        if (numericInput.length > 0) {
                            var val = numericInput.val();
                            if (val === '' || val === null || isNaN(val) || parseFloat(val) < 0) {
                                errors.push(`Group ${groupIndex + 1}: Please enter a valid ${fieldType} value`);
                                isValid = false;
                            }
                        }
                        break;
                        
                    case 'title':
                        var textInput = valueContainer.find('.scmw-text-input:visible:enabled');
                        if (textInput.length > 0 && !textInput.val().trim()) {
                            errors.push(`Group ${groupIndex + 1}: Please enter a title value`);
                            isValid = false;
                        }
                        break;
                }
            });
        });
        
        // Show errors if validation failed
        if (!isValid) {
            e.preventDefault();
            alert('Please fix the following errors:\n\n' + errors.join('\n'));
            return false;
        }
        
        return true;
    });
    
    console.log('SCMW Admin JS initialized successfully');
});