document.addEventListener("DOMContentLoaded", function() {
    
    // IMMEDIATELY TELL CSS THAT JS IS WORKING
    // We add a class to the HTML element to enable "hidden" states for animations
    document.documentElement.classList.add('shps-js-enabled');

    // --- CONFIGURATION ---
    const delayStep = 0.2; // Delay step in seconds
    const triggerSelector = '.shps-animated'; 
    const groupSelector = '.shps-group'; 

    // Check for reduced motion preference
    const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;
    if (prefersReducedMotion) return; // Stop script if user requested no motion

    // --- FUNCTION 1: Calculate columns in a group ---
    function calculateColumnsInGroup(items) {
        if (items.length < 2) return 1;
        const firstItemTop = items[0].offsetTop;
        // Allow small margin of error (1px) for sub-pixel rendering issues
        for (let i = 1; i < items.length; i++) {
            if (items[i].offsetTop > firstItemTop + 1) { 
                return i;
            }
        }
        return items.length; // All items in one row
    }

    // --- FUNCTION 2: Apply delays ---
    function applyDelays() {
        const groups = document.querySelectorAll(groupSelector);

        groups.forEach(group => {
            const items = group.querySelectorAll(triggerSelector);
            if (items.length === 0) return;

            const columnsCount = calculateColumnsInGroup(items);

            items.forEach((item, index) => {
                if (item.className.includes('shps-delay-')) return;

                const delayIndex = index % columnsCount;
                item.style.transitionDelay = (delayIndex * delayStep) + 's';
            });
        });
    }

    // --- SCROLL OBSERVER ---
    const observerOptions = {
        root: null,
        rootMargin: '0px',
        threshold: 0.01 
    };

    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('is-visible');
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    // --- EXECUTION ---
    applyDelays();

    document.querySelectorAll(triggerSelector).forEach(el => {
        observer.observe(el);
    });

    // Event listeners for robustness
    window.addEventListener('load', applyDelays);
    
    let resizeTimer;
    window.addEventListener('resize', () => {
         clearTimeout(resizeTimer);
         resizeTimer = setTimeout(applyDelays, 200);
    });
});