<?php
/**
 * Plugin Name: Shailesh LeadCapture Pro
 * Description: Professional WhatsApp Lead CRM with Visual Analytics and secure CSV Export. Capture visitor details with zero security vulnerabilities.
 * Version:     1.0.9
 * Author:      Shailesh Panchal
 * Author URI:  https://shaileshpanchal.com/
 * Text Domain: shailesh-leadcapture-pro
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Stable tag:  1.0.9
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// 1. DATABASE SETUP
register_activation_hook( __FILE__, 'slcp_install_db' );
function slcp_install_db() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'slcp_leads';
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        time datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
        name varchar(100) NOT NULL,
        phone varchar(20) NOT NULL,
        email varchar(100) NOT NULL,
        department varchar(100) NOT NULL,
        source varchar(100) DEFAULT 'Direct',
        page_url varchar(255) DEFAULT '',
        PRIMARY KEY  (id)
    ) $charset_collate;";
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );
}

// 2. ADMIN MENU SETUP
add_action( 'admin_menu', 'slcp_setup_menu' );
function slcp_setup_menu() {
    add_menu_page( 'WA Chatbot', 'WA Chatbot', 'manage_options', 'slcp-leads', 'slcp_render_dashboard', 'dashicons-whatsapp', 90 );
    add_submenu_page( 'slcp-leads', 'Settings', 'Settings', 'manage_options', 'slcp-settings', 'slcp_render_settings' );
}

// 3. ASSETS & LOCALIZATION (Strictly Enqueued)
add_action( 'admin_enqueue_scripts', 'slcp_enqueue_admin_assets' );
function slcp_enqueue_admin_assets( $hook ) {
    if ( strpos( $hook, 'slcp-leads' ) === false && strpos( $hook, 'slcp-settings' ) === false ) return;

    wp_enqueue_script( 'slcp-chart-js', plugins_url( '/js/chart.js', __FILE__ ), array(), '4.5.1', true );
    wp_enqueue_script( 'slcp-admin-js', plugins_url( '/js/slcp-admin.js', __FILE__ ), array( 'slcp-chart-js' ), '1.0.9', true );

    global $wpdb;
    $table = $wpdb->prefix . 'slcp_leads';
    $sources = $wpdb->get_results( "SELECT source, COUNT(*) as count FROM {$table} GROUP BY source", ARRAY_A );
    
    $labels = []; $data = [];
    if ( $sources ) {
        foreach ( $sources as $s ) {
            $labels[] = $s['source'] ? $s['source'] : 'Direct';
            $data[]   = (int)$s['count'];
        }
    }

    wp_localize_script( 'slcp-admin-js', 'slcp_admin_vars', array(
        'labels' => $labels,
        'data'   => $data
    ));
}

add_action( 'wp_enqueue_scripts', 'slcp_enqueue_public_assets' );
function slcp_enqueue_public_assets() {
    wp_enqueue_style( 'slcp-style', plugins_url( '/css/slcp-style.css', __FILE__ ), array(), '1.0.9' );
    
    // Nayi file load karein jo humne Step 1 mein banayi
    wp_enqueue_script( 'slcp-public-js', plugins_url( '/js/slcp-public.js', __FILE__ ), array('jquery'), '1.0.9', true );

    wp_localize_script( 'slcp-public-js', 'slcp_vars', array(
        'ajax_url'  => admin_url( 'admin-ajax.php' ),
        'nonce'     => wp_create_nonce( 'slcp_nonce' ),
        'auto_open' => (int)get_option( 'slcp_auto_open', 5 ) * 1000
    ) );
}

// 4. CSV EXPORT (Fixed: gmdate & ob_end_clean)
add_action( 'admin_init', 'slcp_handle_csv_export' );
function slcp_handle_csv_export() {
    if ( isset( $_POST['slcp_export'] ) && check_admin_referer( 'slcp_export_nonce' ) ) {
        global $wpdb;
        $table = $wpdb->prefix . 'slcp_leads';
        $rows = $wpdb->get_results( "SELECT id, time, name, phone, email, department, source, page_url FROM {$table} ORDER BY id DESC", ARRAY_A );

        if ( $rows ) {
            if ( ob_get_length() ) ob_end_clean();
            header( 'Content-Type: text/csv; charset=utf-8' );
            header( 'Content-Disposition: attachment; filename="leads_export_' . gmdate( 'Y-m-d' ) . '.csv"' );
            $output = fopen( 'php://output', 'w' );
            fputcsv( $output, array( 'ID', 'Date', 'Name', 'Phone', 'Email', 'Dept', 'Source', 'URL' ) );
            foreach ( $rows as $r ) { fputcsv( $output, $r ); }
            exit;
        }
    }
}

// 5. DASHBOARD RENDER
function slcp_render_dashboard() {
    global $wpdb;
    $table = $wpdb->prefix . 'slcp_leads';

    if ( isset( $_POST['slcp_bulk_del'], $_POST['ids'] ) && check_admin_referer( 'slcp_bulk_action' ) ) {
        $ids = array_map( 'absint', (array)$_POST['ids'] );
        $placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );
        $wpdb->query( $wpdb->prepare( "DELETE FROM {$table} WHERE id IN ($placeholders)", $ids ) );
    }

    $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
    $leads = $wpdb->get_results( "SELECT id, time, name, phone, email, department, source FROM {$table} ORDER BY id DESC LIMIT 100" );

    ?>
    <div class="wrap">
        <h1>📊 LeadCapture CRM Dashboard</h1>
        <div style="display:flex; gap:20px; margin:20px 0; flex-wrap:wrap;">
            <div style="background:#fff; padding:20px; border-radius:10px; flex:1; text-align:center; box-shadow:0 2px 5px rgba(0,0,0,0.1);">
                <h3>Total Leads</h3>
                <h1 style="color:#008069; font-size:48px;"><?php echo absint( $total ); ?></h1>
            </div>
            <div style="background:#fff; padding:20px; border-radius:10px; flex:2; box-shadow:0 2px 5px rgba(0,0,0,0.1);">
                <canvas id="slcpPie" style="max-height:160px;"></canvas>
            </div>
        </div>

        <form method="post" style="margin-bottom:15px;">
            <?php wp_nonce_field( 'slcp_export_nonce' ); ?>
            <button type="submit" name="slcp_export" class="button button-primary">📥 Export CSV Database</button>
        </form>

        <form method="post">
            <?php wp_nonce_field( 'slcp_bulk_action' ); ?>
            <input type="submit" name="slcp_bulk_del" class="button action" value="🗑️ Delete Selected" onclick="return confirm('Delete permanently?');">
            <table class="wp-list-table widefat fixed striped" style="margin-top:10px;">
                <thead>
                    <tr><th class="check-column"><input type="checkbox" id="cb-select-all-1"></th><th>Date</th><th>Name</th><th>Phone</th><th>Email</th><th>Dept & Source</th></tr>
                </thead>
                <tbody>
                    <?php if ( $leads ) : foreach ( $leads as $l ) : ?>
                    <tr>
                        <th scope="row" class="check-column"><input type="checkbox" name="ids[]" value="<?php echo absint( $l->id ); ?>"></th>
                        <td><?php echo esc_html( $l->time ); ?></td>
                        <td><strong><?php echo esc_html( $l->name ); ?></strong></td>
                        <td><?php echo esc_html( $l->phone ); ?></td>
                        <td><?php echo esc_html( $l->email ); ?></td>
                        <td>
                            <mark style="background:#e7f5f2; color:#008069; padding:2px 6px; border-radius:4px; font-weight:bold;"><?php echo esc_html( $l->department ); ?></mark>
                            <br><small>Source: <?php echo esc_html( $l->source ); ?></small>
                        </td>
                    </tr>
                    <?php endforeach; else : ?>
                        <tr><td colspan="6">No leads found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </form>
    </div>
    <?php
}

// 6. SETTINGS VIEW
function slcp_render_settings() {
    ?>
    <div class="wrap">
        <h1>⚙️ Chatbot Settings</h1>
        <div style="background:#fff; padding:30px; border-radius:10px; border:1px solid #ddd; max-width:800px;">
            <form method="post" action="options.php">
                <?php settings_fields( 'slcp_options_grp' ); ?>
                <table class="form-table">
                    <tr><th>WhatsApp Number</th><td><input type="text" name="slcp_phone" value="<?php echo esc_attr( get_option( 'slcp_phone' ) ); ?>" class="regular-text"></td></tr>
                    <tr><th>Agent Name</th><td><input type="text" name="slcp_agent" value="<?php echo esc_attr( get_option( 'slcp_agent', 'Shailesh Panchal' ) ); ?>" class="regular-text"></td></tr>
                    <tr><th>Avatar URL</th><td><input type="text" name="slcp_avatar" value="<?php echo esc_attr( get_option( 'slcp_avatar' ) ); ?>" class="large-text"></td></tr>
                    <tr><th>Brand Color</th><td><input type="color" name="slcp_color" value="<?php echo esc_attr( get_option( 'slcp_color', '#008069' ) ); ?>"></td></tr>
                    <tr><th>Auto Open (Sec)</th><td><input type="number" name="slcp_auto_open" value="<?php echo absint( get_option( 'slcp_auto_open', 5 ) ); ?>"></td></tr>
                    <tr><th>Agents List</th><td><textarea name="slcp_agents_list" rows="5" class="large-text" placeholder="Sales|+91..."><?php echo esc_textarea( get_option( 'slcp_agents_list' ) ); ?></textarea></td></tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
    </div>
    <?php
}

// 7. SETTINGS & AJAX INIT
add_action( 'admin_init', 'slcp_init_settings' );
function slcp_init_settings() {
    register_setting( 'slcp_options_grp', 'slcp_phone', 'sanitize_text_field' );
    register_setting( 'slcp_options_grp', 'slcp_agent', 'sanitize_text_field' );
    register_setting( 'slcp_options_grp', 'slcp_avatar', 'esc_url_raw' );
    register_setting( 'slcp_options_grp', 'slcp_color', 'sanitize_hex_color' );
    register_setting( 'slcp_options_grp', 'slcp_auto_open', 'absint' );
    register_setting( 'slcp_options_grp', 'slcp_agents_list', 'sanitize_textarea_field' );
}

add_action( 'wp_ajax_slcp_save_lead', 'slcp_ajax_handler' );
add_action( 'wp_ajax_nopriv_slcp_save_lead', 'slcp_ajax_handler' );
function slcp_ajax_handler() {
    check_ajax_referer( 'slcp_nonce', 'nonce' );
    global $wpdb;
    $wpdb->insert( $wpdb->prefix . 'slcp_leads', array(
        'time' => current_time( 'mysql' ),
        'name' => sanitize_text_field( wp_unslash( $_POST['name'] ?? '' ) ),
        'phone' => sanitize_text_field( wp_unslash( $_POST['phone'] ?? '' ) ),
        'email' => sanitize_email( wp_unslash( $_POST['email'] ?? '' ) ),
        'department' => sanitize_text_field( wp_unslash( $_POST['dept'] ?? '' ) ),
        'source' => sanitize_text_field( wp_unslash( $_POST['source'] ?? 'Direct' ) ),
        'page_url' => esc_url_raw( wp_unslash( $_POST['url'] ?? '' ) )
    ) );
    wp_send_json_success();
}

// 8. FOOTER UI RENDER
add_action( 'wp_footer', 'slcp_render_chatbot_html' );
function slcp_render_chatbot_html() {
    $color  = get_option( 'slcp_color', '#008069' );
    $agent  = get_option( 'slcp_agent', 'Shailesh Panchal' );
    $avatar = get_option( 'slcp_avatar', '' );
    $list   = get_option( 'slcp_agents_list', '' );
    ?>
    <div id="slcp-ui">
        <div class="slcp-btn" id="slcp_toggle_btn" style="background:<?php echo esc_attr( $color ); ?>;">
            <svg width="32" height="32" viewBox="0 0 24 24" fill="white"><path d="M12.031 2C6.508 2 2.016 6.492 2.016 12.016c0 1.77.462 3.425 1.264 4.88L2 22l5.242-1.248a9.96 9.96 0 0 0 4.789 1.264h.005c5.522 0 10.014-4.492 10.014-10.016C22.045 6.494 17.551 2 12.031 2zm0 18.283h-.005a8.236 8.236 0 0 1-4.204-1.155l-.301-.179-3.116.743.83-3.033-.197-.314a8.256 8.256 0 0 1-1.264-4.329c0-4.566 3.714-8.281 8.281-8.281 4.565 0 8.279 3.715 8.279 8.281 0 4.567-3.715 8.267-8.303 8.267z"/></svg>
        </div>
        <div class="slcp-win" id="slcp_box">
            <div class="slcp-head" style="background:<?php echo esc_attr( $color ); ?>;">
                <?php if($avatar): ?><img src="<?php echo esc_url( $avatar ); ?>" alt="Agent"><?php endif; ?>
                <div><b><?php echo esc_html( $agent ); ?></b><br><span>Online | Usually replies instantly</span></div>
                <div class="slcp-close" id="slcp_close_btn">&times;</div>
            </div>
            <div class="slcp-body">
                <div class="slcp-field-group"><label>Full Name *</label><input type="text" id="sn" class="slcp-inp" placeholder="Enter your name"></div>
                <div class="slcp-field-group"><label>WhatsApp Number *</label><input type="tel" id="sp" class="slcp-inp" placeholder="e.g. 918454..."></div>
                <div class="slcp-field-group"><label>Email Address *</label><input type="email" id="se" class="slcp-inp" placeholder="your@email.com"></div>
                <div class="slcp-field-group">
                    <label>Department *</label>
                    <select id="sd" class="slcp-inp">
                        <option value="">-- Choose Option --</option>
                        <?php 
                        if( !empty( $list ) ){
                            foreach( preg_split( '/\r\n|\r|\n/', trim( $list ) ) as $l ){
                                $p = explode( '|', trim( $l ) );
                                if( count( $p ) >= 2 ) echo '<option value="'.esc_attr( trim($p[1]) ).'">'.esc_html( trim($p[0]) ).'</option>';
                            }
                        }
                        ?>
                    </select>
                </div>
                <button id="slcp_start_chat" class="slcp-submit" style="background:<?php echo esc_attr( $color ); ?>;">Start WhatsApp Chat</button>
            </div>
        </div>
    </div>
    <?php
}

add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function( $links ) {
    array_push( $links, '<a href="https://book.shaileshpanchal.com/wa-leadengine-pro/" target="_blank">Get Pro Licence Version</a>' );
    return $links;
} );