<?php
/**
 * Plugin Name: ShadowScan Security Link
 * Plugin URI: https://shadowscan.com.au/portal
 * Description: Connects a WordPress site to the ShadowScan portal, sends heartbeats, and executes security commands.
 * Version: 1.0.12
 * Author: ShadowScan
 * Author URI: https://shadowscan.com.au
 * Requires at least: 6.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: shadowscan-security-link
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

const SHADOWSCAN_SENTRY_DSN = '';

/**
 * @param string $filename
 * @return string|false
 */
function shadowscan_tempnam(string $filename) {
    if (!function_exists('wp_tempnam')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    if (function_exists('wp_tempnam')) {
        return wp_tempnam($filename);
    }
    return tempnam(sys_get_temp_dir(), 'shadowscan-');
}

function shadowscan_sentry_telemetry_enabled(): bool {
    return (bool) get_option('shadowscan_sentry_telemetry_enabled', false);
}

function shadowscan_get_sentry_dsn(): string {
    $dsn = getenv('SHADOWSCAN_SENTRY_DSN');
    if (is_string($dsn) && trim($dsn) !== '') {
        return trim($dsn);
    }
    return SHADOWSCAN_SENTRY_DSN;
}

function shadowscan_init_sentry(): void {
    if (!shadowscan_sentry_telemetry_enabled()) {
        return;
    }

    $dsn = shadowscan_get_sentry_dsn();
    $send_default_pii = getenv('SHADOWSCAN_SENTRY_SEND_DEFAULT_PII') === 'true';
    if (!$dsn) {
        return;
    }

    $autoload = __DIR__ . '/vendor/autoload.php';
    if (!file_exists($autoload)) {
        return;
    }

    require_once $autoload;

    if (!function_exists('\\Sentry\\init')) {
        return;
    }

    try {
        \Sentry\init([
            'dsn' => $dsn,
            // Setting this option to true will send default PII data to Sentry.
            // For example, automatic IP address collection on events
            'send_default_pii' => $send_default_pii,
            'traces_sample_rate' => 1.0,
        ]);
    } catch (\Throwable $error) {
        shadowscan_log_message('[ShadowScan] Sentry init failed: ' . $error->getMessage());
    }
}

shadowscan_init_sentry();

function shadowscan_get_filesystem(): ?WP_Filesystem_Base {
    if (!function_exists('WP_Filesystem')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    if (!WP_Filesystem()) {
        return null;
    }
    global $wp_filesystem;
    return $wp_filesystem instanceof WP_Filesystem_Base ? $wp_filesystem : null;
}

if (!defined('SHADOWSCAN_PLUGIN_VERSION')) {
    if (!function_exists('get_file_data')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    // Read the plugin header version so there's only one source of truth.
    $shadowscan_data = get_file_data(__FILE__, array('version' => 'Version'), 'plugin');
    define('SHADOWSCAN_PLUGIN_VERSION', $shadowscan_data['version'] ?? '0.0.0');
}
if (!defined('SHADOWSCAN_API_BASE')) {
    $shadowscan_api_base = getenv('SHADOWSCAN_API_BASE');
    if (!is_string($shadowscan_api_base) || trim($shadowscan_api_base) === '') {
        $shadowscan_api_base = 'https://foemwjtosslaiynduiyt.supabase.co/functions/v1/';
    }
    define('SHADOWSCAN_API_BASE', rtrim($shadowscan_api_base, '/'));
}
const SHADOWSCAN_PORTAL_URL = 'https://shadowscan.com.au/portal';
const SHADOWSCAN_PORTAL_BILLING_URL = 'https://shadowscan.com.au/portal/billing';
const SHADOWSCAN_DATA_HANDLING_URL = 'https://shadowscan.com.au/privacy';
const SHADOWSCAN_SUPPORT_URL = 'mailto:support@shadowscan.com.au';
const SHADOWSCAN_LOGO_URL = 'data:image/png;base64,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';
const SHADOWSCAN_MENU_ICON = 'dashicons-shield';

const SHADOWSCAN_OPTION_SITE_TOKEN = 'shadowscan_site_token';
const SHADOWSCAN_OPTION_SITE_ID = 'shadowscan_site_id';
const SHADOWSCAN_OPTION_LAST_PING = 'shadowscan_last_ping';
const SHADOWSCAN_OPTION_CAN_INSTALL = 'shadowscan_can_install_plugins';
const SHADOWSCAN_COMMAND_CONTEXT_KEY = 'shadowscan_command_context';
const SHADOWSCAN_OPTION_LOG = 'shadowscan_log_entries';
const SHADOWSCAN_OPTION_LAST_ERROR = 'shadowscan_last_error';
const SHADOWSCAN_OPTION_PENDING_HEARTBEAT = 'shadowscan_pending_heartbeat';
const SHADOWSCAN_OPTION_PENDING_RESULTS = 'shadowscan_pending_results';
const SHADOWSCAN_OPTION_LAST_SELF_CHECK = 'shadowscan_last_self_check';
const SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS = 'shadowscan_remote_diagnostics_enabled';
const SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES = 'shadowscan_remote_diagnostics_expires';
const SHADOWSCAN_OPTION_SENTRY_TELEMETRY = 'shadowscan_sentry_telemetry_enabled';
const SHADOWSCAN_OPTION_CAPABILITIES_SNAPSHOT = 'shadowscan_capabilities_snapshot';
const SHADOWSCAN_OPTION_LAST_CONTACT_ATTEMPT = 'shadowscan_last_contact_attempt';
const SHADOWSCAN_OPTION_LAST_CONTACT_ERROR = 'shadowscan_last_contact_error';
const SHADOWSCAN_OPTION_ENDPOINT_HEALTH = 'shadowscan_endpoint_health';
const SHADOWSCAN_OPTION_LAST_PORTAL_RECEIVED = 'shadowscan_last_portal_received';
const SHADOWSCAN_OPTION_COMMAND_BACKOFF_UNTIL = 'shadowscan_command_backoff_until';
const SHADOWSCAN_OPTION_COMMAND_BACKOFF_COUNT = 'shadowscan_command_backoff_count';
const SHADOWSCAN_OPTION_COMMAND_BACKOFF_LEVEL = 'shadowscan_command_backoff_level';
const SHADOWSCAN_OPTION_COMMAND_LAST_HTTP_STATUS = 'shadowscan_command_last_http_status';
const SHADOWSCAN_OPTION_COMMAND_TIMEOUT_COUNT = 'shadowscan_command_timeout_count';
const SHADOWSCAN_OPTION_UNKNOWN_COMMAND_TYPES = 'shadowscan_unknown_command_types';
const SHADOWSCAN_OPTION_SCHEDULER_LAST_REPAIR_AT = 'shadowscan_scheduler_last_repair_at';
const SHADOWSCAN_OPTION_SCHEDULER_REPAIRS_COUNT = 'shadowscan_scheduler_repairs_count';
const SHADOWSCAN_OPTION_UI_ADVANCED = 'shadowscan_ui_advanced';
const SHADOWSCAN_OPTION_UI_FOCUS_ANCHOR = 'shadowscan_ui_focus_anchor';
const SHADOWSCAN_OPTION_CONTROL_ACKS = 'control_acknowledgements';
const SHADOWSCAN_OPTION_LAST_REFRESH_AT = 'shadowscan_last_refresh_at';
const SHADOWSCAN_OPTION_LAST_SELF_CHECK_AT = 'shadowscan_last_self_check_at';
const SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE = 'shadowscan_subscription_override';
const SHADOWSCAN_OPTION_SUBSCRIPTION_STATE = 'shadowscan_subscription_state';
const SHADOWSCAN_OPTION_SUBSCRIPTION_STATE_UPDATED = 'shadowscan_subscription_state_updated';
const SHADOWSCAN_OPTION_SUBSCRIPTION_LKG = 'shadowscan_subscription_state_lkg';
const SHADOWSCAN_OPTION_SUBSCRIPTION_LKG_AT = 'shadowscan_subscription_state_lkg_at';
const SHADOWSCAN_OPTION_SUBSCRIPTION_PORTAL_UNREACHABLE = 'shadowscan_subscription_portal_unreachable';
const SHADOWSCAN_OPTION_AUTO_HTACCESS = 'shadowscan_auto_htaccess_apply';
if (!defined('SHADOWSCAN_FORCE_LOCAL_OVERRIDE')) {
    define('SHADOWSCAN_FORCE_LOCAL_OVERRIDE', false);
}
if (!defined('SHADOWSCAN_REQUIRE_POLICY_VERSION')) {
    define('SHADOWSCAN_REQUIRE_POLICY_VERSION', true);
}
if (!defined('SHADOWSCAN_DISCONNECT_ON_DEACTIVATE')) {
    define('SHADOWSCAN_DISCONNECT_ON_DEACTIVATE', false);
}
if (!defined('SHADOWSCAN_AUTH_RECOVERY_MAX_ATTEMPTS')) {
    define('SHADOWSCAN_AUTH_RECOVERY_MAX_ATTEMPTS', 1);
}
const SHADOWSCAN_REMOTE_LOG_MAX_BYTES = 200000;
const SHADOWSCAN_REMOTE_DIAGNOSTICS_TTL = 3600;
const SHADOWSCAN_AUTH_RECOVERY_COOLDOWN = 1800;
const SHADOWSCAN_REFRESH_COOLDOWN = 45;
const SHADOWSCAN_SELF_CHECK_COOLDOWN = 60;
const SHADOWSCAN_PAST_DUE_GRACE_DAYS = 7;
const SHADOWSCAN_STALE_DAYS_AFTER_SUSPEND = 0;
const SHADOWSCAN_OUTDATED_AFTER_DAYS = 30;
const SHADOWSCAN_PAUSED_AFTER_DAYS = 60;
const SHADOWSCAN_ACK_DEFAULT_DAYS = 7;
const SHADOWSCAN_ACK_MUTE_DAYS_30 = 30;
const SHADOWSCAN_ACK_MUTE_DAYS_90 = 90;
$shadowscan_autoload_paths = array(
    __DIR__ . '/vendor/autoload.php',
    __DIR__ . '/includes/vendor/autoload.php',
);
foreach ($shadowscan_autoload_paths as $shadowscan_autoload_path) {
    if (file_exists($shadowscan_autoload_path)) {
        require_once $shadowscan_autoload_path;
        break;
    }
}

require_once __DIR__ . '/includes/storage/class-storage.php';
require_once __DIR__ . '/includes/storage/class-event-buffer.php';
require_once __DIR__ . '/includes/storage/class-rate-limiter.php';
require_once __DIR__ . '/includes/api/class-api-client.php';
require_once __DIR__ . '/includes/api/class-api-endpoints.php';
require_once __DIR__ . '/includes/signals/class-signal-manager.php';
require_once __DIR__ . '/includes/signals/class-signal-auth.php';
require_once __DIR__ . '/includes/signals/class-signal-integrity.php';
require_once __DIR__ . '/includes/signals/class-signal-config.php';
require_once __DIR__ . '/includes/mfa/class-mfa.php';
require_once __DIR__ . '/includes/sync/class-sync-state.php';
require_once __DIR__ . '/includes/commands/class-commands.php';
require_once __DIR__ . '/includes/control-registry.php';
require_once __DIR__ . '/includes/controls/class-security-controls.php';
require_once __DIR__ . '/includes/controls/class-auto-updates.php';
require_once __DIR__ . '/includes/controls/class-password-policy.php';
require_once __DIR__ . '/includes/controls/class-geo-policy.php';
require_once __DIR__ . '/includes/controls/class-admin-geo-guard.php';
require_once __DIR__ . '/includes/subscription/class-subscription-state.php';
require_once __DIR__ . '/includes/policy/class-policy-state.php';
require_once __DIR__ . '/includes/policy/class-policy-resolver.php';
require_once __DIR__ . '/includes/guards/class-guard-manager.php';
require_once __DIR__ . '/includes/guards/class-guard-access-surface.php';
require_once __DIR__ . '/includes/guards/class-guard-auth.php';
require_once __DIR__ . '/includes/guards/class-guard-bruteforce.php';
require_once __DIR__ . '/includes/guards/class-guard-mfa.php';
require_once __DIR__ . '/includes/guards/class-guard-admin-integrity.php';
require_once __DIR__ . '/includes/guards/class-guard-integrity.php';
require_once __DIR__ . '/includes/guards/class-guard-integrity-targeted.php';
require_once __DIR__ . '/includes/guards/class-guard-config.php';
require_once __DIR__ . '/includes/guards/class-guard-emergency-mode.php';
require_once __DIR__ . '/includes/guards/class-guard-htaccess.php';
require_once __DIR__ . '/includes/guards/class-guard-enumeration.php';
require_once __DIR__ . '/includes/guards/class-guard-access-control.php';
require_once __DIR__ . '/includes/guards/class-guard-headers.php';
require_once __DIR__ . '/includes/owasp/status.php';
require_once __DIR__ . '/includes/ui/class-protection-registry.php';
if (defined('WP_CLI') && WP_CLI) {
    require_once __DIR__ . '/includes/cli/class-shadowscan-cli.php';
}

function shadowscan_guard_manager(): ShadowScan_Guard_Manager {
    static $manager = null;
    if ($manager === null) {
        $manager = new ShadowScan_Guard_Manager();
        $manager->init();
    }
    return $manager;
}

add_action('plugins_loaded', function () {
    shadowscan_guard_manager();
});

add_action('init', function () {
    if (class_exists('ShadowScan_Password_Policy')) {
        ShadowScan_Password_Policy::register();
    }
    if (class_exists('ShadowScan_Geo_Policy')) {
        ShadowScan_Geo_Policy::register();
    }
    if (class_exists('ShadowScan_Admin_Geo_Guard')) {
        ShadowScan_Admin_Geo_Guard::register();
    }

    add_filter('force_ssl_admin', function ($force) {
        if (shadowscan_guard_manager()->is_flag_enabled('force_ssl_admin', false)) {
            return true;
        }
        return $force;
    });

    add_filter('wp_debug_display', function ($display) {
        if (shadowscan_guard_manager()->is_flag_enabled('owasp_disable_debug_display', false)) {
            return false;
        }
        return $display;
    });

    add_filter('allow_minor_auto_core_updates', function ($enabled) {
        if (shadowscan_guard_manager()->is_flag_enabled('owasp_auto_update_minor', false)) {
            return true;
        }
        return $enabled;
    });

    // wp_debug_display filter covers disabling error display in supported contexts.
}, 1);

function shadowscan_run_security_control_checks(): void {
    if (class_exists('ShadowScan_AutoUpdates')) {
        ShadowScan_AutoUpdates::get_status();
        ShadowScan_AutoUpdates::get_core_status();
    }
    if (class_exists('ShadowScan_Password_Policy')) {
        ShadowScan_Password_Policy::get_status();
    }
    if (class_exists('ShadowScan_Geo_Policy')) {
        ShadowScan_Geo_Policy::get_status();
    }
    if (class_exists('ShadowScan_Admin_Geo_Guard') && class_exists('ShadowScan_Security_Controls')) {
        $status = ShadowScan_Admin_Geo_Guard::get_status();
        $raw_status = isset($status['status']) ? strtolower((string) $status['status']) : 'warn';
        if (!in_array($raw_status, array('ok', 'warn', 'critical', 'resolved'), true)) {
            $raw_status = 'warn';
        }
        $evidence = array(
            'policy' => isset($status['policy']) && is_array($status['policy']) ? $status['policy'] : array(),
            'confidence' => isset($status['confidence']) && is_array($status['confidence']) ? $status['confidence'] : array(),
            'countries' => isset($status['countries']) && is_array($status['countries']) ? $status['countries'] : array(),
            'last_country' => isset($status['last_country']) ? $status['last_country'] : null,
            'last_seen_at' => isset($status['last_seen_at']) ? $status['last_seen_at'] : null,
            'geo_source' => isset($status['geo_source']) ? $status['geo_source'] : null,
        );
        ShadowScan_Security_Controls::emit_status(
            'admin_geo_guard',
            $raw_status,
            false,
            'Review admin access location policy coverage and confidence.',
            $evidence,
            isset($status['checked_at']) ? (int) $status['checked_at'] : time()
        );
    }
}

function shadowscan_detect_third_party_security_plugins(): array {
    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    $plugins = get_plugins();
    if (!is_array($plugins)) {
        return array();
    }

    $keywords = array(
        'security',
        'firewall',
        'waf',
        'malware',
        'brute force',
        'bruteforce',
        'hardening',
        'intrusion',
        'threat',
        'antivirus',
    );

    $excluded_keywords = array(
        'backup',
        'seo',
        'spam',
        'cache',
        'analytics',
        'performance',
        'optimization',
        'marketing',
    );

    $excluded_names = array(
        'loginizer',
    );

    $detected = array();
    $self_plugin = plugin_basename(__FILE__);

    foreach ($plugins as $file => $data) {
        if ($file === $self_plugin) {
            continue;
        }

        $name = strtolower((string) ($data['Name'] ?? ''));
        $description = strtolower((string) ($data['Description'] ?? ''));
        $haystack = $name . ' ' . $description;

        $is_excluded = false;
        foreach ($excluded_keywords as $excluded) {
            if ($excluded !== '' && strpos($haystack, $excluded) !== false) {
                $is_excluded = true;
                break;
            }
        }
        if (!$is_excluded) {
            foreach ($excluded_names as $excluded) {
                if ($excluded !== '' && strpos($name, $excluded) !== false) {
                    $is_excluded = true;
                    break;
                }
            }
        }
        if ($is_excluded) {
            continue;
        }

        foreach ($keywords as $keyword) {
            if ($keyword !== '' && strpos($haystack, $keyword) !== false) {
                $label = trim((string) ($data['Name'] ?? $file));
                if ($label !== '') {
                    $detected[$label] = true;
                }
                break;
            }
        }
    }

    return array_values(array_keys($detected));
}

add_filter('cron_schedules', function ($schedules) {
    if (!isset($schedules['shadowscan_five_minutes'])) {
        $schedules['shadowscan_five_minutes'] = array(
            'interval' => 300,
            'display' => __('Every 5 Minutes', 'shadowscan-security-link'),
        );
    }
    if (!isset($schedules['shadowscan_ten_minutes'])) {
        $schedules['shadowscan_ten_minutes'] = array(
            'interval' => 600,
            'display' => __('Every 10 Minutes', 'shadowscan-security-link'),
        );
    }
    if (!isset($schedules['shadowscan_six_hours'])) {
        $schedules['shadowscan_six_hours'] = array(
            'interval' => 6 * HOUR_IN_SECONDS,
            'display' => __('Every 6 Hours', 'shadowscan-security-link'),
        );
    }
    return $schedules;
});

function shadowscan_activate(): void {
    shadowscan_ensure_cron_events(true);
    shadowscan_send_heartbeat(true);
}
register_activation_hook(__FILE__, 'shadowscan_activate');

function shadowscan_ensure_cron_events(bool $force = false): void {
    if (wp_installing()) {
        return;
    }

    // Keep runtime checks lightweight: only verify schedules periodically unless forced.
    if (!$force && !defined('WP_CLI')) {
        $lock_key = 'shadowscan_cron_self_heal_checked';
        if (get_transient($lock_key)) {
            return;
        }
        set_transient($lock_key, 1, 300);
    }

    $schedule_map = array(
        'shadowscan_heartbeat' => array('recurrence' => 'shadowscan_ten_minutes', 'offset' => 60),
        'shadowscan_poll_commands' => array('recurrence' => 'shadowscan_five_minutes', 'offset' => 120),
        'shadowscan_integrity_scan' => array('recurrence' => 'shadowscan_six_hours', 'offset' => 300),
        'shadowscan_integrity_targeted_scan' => array('recurrence' => 'shadowscan_six_hours', 'offset' => 360),
        'shadowscan_owasp_daily_check' => array('recurrence' => 'daily', 'offset' => 600),
    );

    foreach ($schedule_map as $hook => $config) {
        if (wp_next_scheduled($hook)) {
            continue;
        }
        wp_schedule_event(time() + (int) $config['offset'], (string) $config['recurrence'], $hook);
        update_option(SHADOWSCAN_OPTION_SCHEDULER_LAST_REPAIR_AT, time(), false);
        $repair_count = (int) get_option(SHADOWSCAN_OPTION_SCHEDULER_REPAIRS_COUNT, 0);
        update_option(SHADOWSCAN_OPTION_SCHEDULER_REPAIRS_COUNT, $repair_count + 1, false);
        shadowscan_log_message(sprintf('[ShadowScan] Schedule missing -> re-registered: %s', $hook));
    }
}

add_action('init', 'shadowscan_ensure_cron_events', 20);
add_action('admin_init', 'shadowscan_ensure_cron_events', 20);

function shadowscan_deactivate(): void {
    if (SHADOWSCAN_DISCONNECT_ON_DEACTIVATE) {
        shadowscan_notify_disconnect('plugin_deactivated');
    }
    $heartbeat = wp_next_scheduled('shadowscan_heartbeat');
    if ($heartbeat) {
        wp_unschedule_event($heartbeat, 'shadowscan_heartbeat');
    }
    $poller = wp_next_scheduled('shadowscan_poll_commands');
    if ($poller) {
        wp_unschedule_event($poller, 'shadowscan_poll_commands');
    }
    $integrity = wp_next_scheduled('shadowscan_integrity_scan');
    if ($integrity) {
        wp_unschedule_event($integrity, 'shadowscan_integrity_scan');
    }
    $targeted = wp_next_scheduled('shadowscan_integrity_targeted_scan');
    if ($targeted) {
        wp_unschedule_event($targeted, 'shadowscan_integrity_targeted_scan');
    }
    $emergency = wp_next_scheduled('shadowscan_emergency_heartbeat');
    if ($emergency) {
        wp_unschedule_event($emergency, 'shadowscan_emergency_heartbeat');
    }
    $owasp = wp_next_scheduled('shadowscan_owasp_daily_check');
    if ($owasp) {
        wp_unschedule_event($owasp, 'shadowscan_owasp_daily_check');
    }
}
register_deactivation_hook(__FILE__, 'shadowscan_deactivate');

function shadowscan_uninstall(): void {
    $token = shadowscan_get_site_token();
    if ($token !== '') {
        shadowscan_notify_disconnect('plugin_uninstalled');
    }
    shadowscan_clear_local_data();
    wp_clear_scheduled_hook('shadowscan_heartbeat');
    wp_clear_scheduled_hook('shadowscan_poll_commands');
    wp_clear_scheduled_hook('shadowscan_integrity_scan');
    wp_clear_scheduled_hook('shadowscan_integrity_targeted_scan');
    wp_clear_scheduled_hook('shadowscan_emergency_heartbeat');
    wp_clear_scheduled_hook('shadowscan_owasp_daily_check');
}
register_uninstall_hook(__FILE__, 'shadowscan_uninstall');

add_action('shadowscan_owasp_daily_check', function () {
    ShadowScan_Storage::set('cron_last_tick', time());
    shadowscan_owasp_refresh_status(true);
});

function shadowscan_notify_disconnect(string $reason): void {
    $token = shadowscan_get_site_token();
    if ($token === '') {
        return;
    }
    $response = shadowscan_send_api_request('site-disconnect', array('reason' => $reason), 'POST', true);
    if (is_wp_error($response)) {
        shadowscan_log_message('[ShadowScan] Disconnect notify failed: ' . $response->get_error_message());
    }
}

add_action('shadowscan_heartbeat', function () {
    ShadowScan_Storage::set('cron_last_tick', time());
    shadowscan_send_heartbeat();
});

add_action('shadowscan_poll_commands', function () {
    ShadowScan_Storage::set('cron_last_tick', time());
    shadowscan_poll_commands();
});

add_action('shadowscan_emergency_heartbeat', function () {
    ShadowScan_Storage::set('cron_last_tick', time());
    shadowscan_send_heartbeat(true);
});

add_action('init', function () {
    if (!shadowscan_is_connected()) {
        return;
    }
    $last_ping = (int) get_option(SHADOWSCAN_OPTION_LAST_PING, 0);
    if ($last_ping > 0 && (time() - $last_ping) < 600) {
        return;
    }
    if (get_transient('shadowscan_heartbeat_lock')) {
        return;
    }
    set_transient('shadowscan_heartbeat_lock', 1, 300);
    shadowscan_send_heartbeat();
});

function shadowscan_get_site_token(): string {
    if (is_multisite()) {
        return (string) get_site_option(SHADOWSCAN_OPTION_SITE_TOKEN, '');
    }
    return (string) get_option(SHADOWSCAN_OPTION_SITE_TOKEN, '');
}

function shadowscan_get_auth_token(): string {
    return shadowscan_get_site_token();
}

function shadowscan_set_site_token(string $token): void {
    if (is_multisite()) {
        update_site_option(SHADOWSCAN_OPTION_SITE_TOKEN, $token);
    } else {
        update_option(SHADOWSCAN_OPTION_SITE_TOKEN, $token, false);
    }
    if (function_exists('shadowscan_sync_set_auth_invalid')) {
        shadowscan_sync_set_auth_invalid(false);
    }
}

function shadowscan_get_site_id(): string {
    if (shadowscan_get_site_token() === '') {
        return '';
    }
    if (function_exists('shadowscan_sync_get_identity')) {
        $identity = shadowscan_sync_get_identity();
        if (!empty($identity['site_id'])) {
            return (string) $identity['site_id'];
        }
    }
    return '';
}

function shadowscan_set_site_id(string $site_id): void {
    if (function_exists('shadowscan_sync_set_identity')) {
        shadowscan_sync_set_identity($site_id, '');
    } elseif (function_exists('shadowscan_sync_set_site_id')) {
        shadowscan_sync_set_site_id($site_id);
    }
    if (is_multisite()) {
        update_site_option(SHADOWSCAN_OPTION_SITE_ID, $site_id);
    } else {
        update_option(SHADOWSCAN_OPTION_SITE_ID, $site_id, false);
    }
}

function shadowscan_set_domain_id(string $domain_id): void {
    if (function_exists('shadowscan_sync_set_identity')) {
        shadowscan_sync_set_identity('', $domain_id);
    } elseif (function_exists('shadowscan_sync_set_domain_id')) {
        shadowscan_sync_set_domain_id($domain_id);
    }
}

function shadowscan_persist_identity_from_response(array $response): void {
    if (!empty($response['site_id'])) {
        shadowscan_set_site_id((string) $response['site_id']);
    }
    if (!empty($response['domain_id'])) {
        shadowscan_set_domain_id((string) $response['domain_id']);
    }
}

function shadowscan_is_connected(): bool {
    return shadowscan_get_site_token() !== '';
}

function shadowscan_is_paired(): bool {
    $token = trim(shadowscan_get_site_token());
    $site_id = trim(shadowscan_get_site_id());
    if ($token === '' || $site_id === '') {
        return false;
    }
    if (in_array(strtolower($token), array('pending', 'placeholder', 'demo', 'test'), true)) {
        return false;
    }
    return true;
}

function shadowscan_format_protection_state_label(string $state): string {
    $map = array(
        'live' => __('Live', 'shadowscan-security-link'),
        'stale' => __('Live (no updates)', 'shadowscan-security-link'),
        'outdated' => __('Outdated', 'shadowscan-security-link'),
        'paused' => __('Paused', 'shadowscan-security-link'),
    );
    return $map[$state] ?? __('Paused', 'shadowscan-security-link');
}

function shadowscan_clear_local_data(): void {
    $options = array(
        SHADOWSCAN_OPTION_SITE_TOKEN,
        SHADOWSCAN_OPTION_SITE_ID,
        SHADOWSCAN_OPTION_LAST_PING,
        SHADOWSCAN_OPTION_CAN_INSTALL,
        SHADOWSCAN_OPTION_LOG,
        SHADOWSCAN_OPTION_LAST_ERROR,
        SHADOWSCAN_OPTION_PENDING_HEARTBEAT,
        SHADOWSCAN_OPTION_PENDING_RESULTS,
        SHADOWSCAN_OPTION_LAST_SELF_CHECK,
        SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS,
        SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES,
        SHADOWSCAN_OPTION_SENTRY_TELEMETRY,
        SHADOWSCAN_OPTION_UI_ADVANCED,
        SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE,
        SHADOWSCAN_OPTION_SUBSCRIPTION_STATE,
        SHADOWSCAN_OPTION_SUBSCRIPTION_STATE_UPDATED,
        SHADOWSCAN_OPTION_SUBSCRIPTION_LKG,
        SHADOWSCAN_OPTION_SUBSCRIPTION_LKG_AT,
        SHADOWSCAN_OPTION_SUBSCRIPTION_PORTAL_UNREACHABLE,
        SHADOWSCAN_OPTION_ENDPOINT_HEALTH,
    );
    foreach ($options as $option) {
        delete_option($option);
    }
    if (defined('SHADOWSCAN_OPTION_SYNC_STATE')) {
        if (is_multisite()) {
            delete_site_option(SHADOWSCAN_OPTION_SYNC_STATE);
        } else {
            delete_option(SHADOWSCAN_OPTION_SYNC_STATE);
        }
    }
    delete_transient('shadowscan_pair_notice');
    delete_transient('shadowscan_pair_notice_type');
}

function shadowscan_build_user_agent(): string {
    $wp_version = get_bloginfo('version');
    $php_version = PHP_VERSION;
    return 'ShadowScan/' . SHADOWSCAN_PLUGIN_VERSION . ' WordPress/' . $wp_version . ' PHP/' . $php_version;
}

function shadowscan_api_endpoints(): ?ShadowScan_Api_Endpoints {
    $token = shadowscan_get_auth_token();
    if ($token === '') {
        return null;
    }

    static $cache = array('token' => '', 'endpoints' => null);
    if (!isset($cache['token']) || $cache['token'] !== $token || $cache['endpoints'] === null) {
        $client = new ShadowScan_Api_Client(SHADOWSCAN_API_BASE, $token, 15, 5, shadowscan_build_user_agent());
        $cache = array(
            'token' => $token,
            'endpoints' => new ShadowScan_Api_Endpoints($client),
        );
    }

    return $cache['endpoints'];
}

function shadowscan_flush_event_queue(): void {
    $site_id = shadowscan_get_site_id();
    if ($site_id === '') {
        return;
    }

    $api = shadowscan_api_endpoints();
    if (!$api) {
        return;
    }

    ShadowScan_Event_Buffer::flush($api, $site_id);
}

function shadowscan_log_rate_limited(string $key, string $message, int $ttl): void {
    $state = ShadowScan_Storage::get_json('log_rate_limit', array());
    if (!is_array($state)) {
        $state = array();
    }
    $now = time();
    $last = (int) ($state[$key] ?? 0);
    if ($last > 0 && ($now - $last) < $ttl) {
        return;
    }
    $state[$key] = $now;
    ShadowScan_Storage::set_json('log_rate_limit', $state);
    shadowscan_log_message($message);
}

function shadowscan_record_contact_attempt(string $context): void {
    update_option(
        SHADOWSCAN_OPTION_LAST_CONTACT_ATTEMPT,
        array(
            'context' => $context,
            'at' => gmdate('Y-m-d H:i:s'),
        ),
        false
    );
}

function shadowscan_record_contact_error(string $context, string $message): void {
    update_option(
        SHADOWSCAN_OPTION_LAST_CONTACT_ERROR,
        array(
            'context' => $context,
            'message' => $message,
            'at' => gmdate('Y-m-d H:i:s'),
        ),
        false
    );
}

function shadowscan_clear_contact_error(): void {
    delete_option(SHADOWSCAN_OPTION_LAST_CONTACT_ERROR);
}

function shadowscan_endpoint_health_key(string $path): string {
    if ($path === 'heartbeat') {
        return 'heartbeat';
    }
    if ($path === 'site-policy' || strpos($path, 'site-policy?') === 0) {
        return 'site_policy';
    }
    if ($path === 'commands' || strpos($path, 'commands?') === 0) {
        return 'commands';
    }
    if ($path === 'events' || strpos($path, 'events?') === 0) {
        return 'events';
    }
    return 'other';
}

function shadowscan_is_core_portal_path(string $path): bool {
    $key = shadowscan_endpoint_health_key($path);
    return $key === 'heartbeat' || $key === 'site_policy';
}

function shadowscan_record_endpoint_health(string $path, bool $success, string $message = ''): void {
    $key = shadowscan_endpoint_health_key($path);
    $health = get_option(SHADOWSCAN_OPTION_ENDPOINT_HEALTH, array());
    if (!is_array($health)) {
        $health = array();
    }
    $entry = isset($health[$key]) && is_array($health[$key]) ? $health[$key] : array();
    $entry['path'] = $path;
    if ($success) {
        $entry['last_success_at'] = time();
        $entry['last_error_at'] = 0;
        $entry['last_error_message'] = '';
    } else {
        $entry['last_error_at'] = time();
        $entry['last_error_message'] = shadowscan_sanitize_error_message($message);
    }
    $health[$key] = $entry;
    update_option(SHADOWSCAN_OPTION_ENDPOINT_HEALTH, $health, false);
}

function shadowscan_get_endpoint_health_snapshot(): array {
    $health = get_option(SHADOWSCAN_OPTION_ENDPOINT_HEALTH, array());
    if (!is_array($health)) {
        $health = array();
    }
    return $health;
}

function shadowscan_auth_recovery_cooldown_active(): bool {
    if (!function_exists('shadowscan_sync_is_auth_invalid') || !shadowscan_sync_is_auth_invalid()) {
        return false;
    }
    $last_attempt_at = function_exists('shadowscan_sync_get_last_recovery_attempt_at')
        ? shadowscan_sync_get_last_recovery_attempt_at()
        : 0;
    return $last_attempt_at > 0 && (time() - $last_attempt_at) < SHADOWSCAN_AUTH_RECOVERY_COOLDOWN;
}

function shadowscan_auth_recovery_attempt_available(): bool {
    if (!function_exists('shadowscan_sync_is_auth_invalid') || !shadowscan_sync_is_auth_invalid()) {
        return false;
    }
    $attempt_count = function_exists('shadowscan_sync_get_recovery_attempt_count')
        ? shadowscan_sync_get_recovery_attempt_count()
        : 0;
    if ($attempt_count >= (int) SHADOWSCAN_AUTH_RECOVERY_MAX_ATTEMPTS) {
        return false;
    }
    return !shadowscan_auth_recovery_cooldown_active();
}

function shadowscan_should_pause_remote_calls(string $path = ''): bool {
    if (!function_exists('shadowscan_sync_is_auth_invalid') || !shadowscan_sync_is_auth_invalid()) {
        return false;
    }
    if ($path === 'heartbeat') {
        return false;
    }
    return shadowscan_auth_recovery_cooldown_active() || !shadowscan_auth_recovery_attempt_available();
}

function shadowscan_attempt_auth_recovery(string $trigger_path): void {
    static $recovery_running = false;
    if ($recovery_running) {
        return;
    }
    if (!shadowscan_auth_recovery_attempt_available()) {
        return;
    }
    $recovery_running = true;
    if (function_exists('shadowscan_sync_mark_recovery_attempt')) {
        shadowscan_sync_mark_recovery_attempt();
    }

    shadowscan_log_rate_limited(
        'auth_recovery_attempt',
        '[ShadowScan] Auth recovery attempt triggered by ' . $trigger_path . '.',
        300
    );
    $result = shadowscan_send_heartbeat(true);
    if (!is_wp_error($result) && function_exists('shadowscan_sync_set_auth_invalid')) {
        shadowscan_sync_set_auth_invalid(false);
    }
    if (is_wp_error($result)) {
        shadowscan_log_rate_limited(
            'auth_recovery_failed',
            '[ShadowScan] Auth recovery attempt failed: ' . $result->get_error_message(),
            900
        );
    }
    $recovery_running = false;
}

function shadowscan_get_auth_header_mode(): string {
    $raw = '';
    if (defined('SHADOWSCAN_AUTH_HEADER_MODE') && is_string(SHADOWSCAN_AUTH_HEADER_MODE)) {
        $raw = SHADOWSCAN_AUTH_HEADER_MODE;
    } else {
        $raw = (string) getenv('SHADOWSCAN_AUTH_HEADER_MODE');
    }
    $mode = strtolower(trim($raw));
    if ($mode === 'v2_only' || $mode === 'legacy_only' || $mode === 'dual') {
        return $mode;
    }
    return 'dual';
}

function shadowscan_send_api_request(string $path, array $payload = array(), string $method = 'POST', bool $auth = true, bool $idempotent_override = false) {
    if ($auth && shadowscan_should_pause_remote_calls($path)) {
        return new WP_Error(
            'auth_invalid_cooldown',
            __('Authentication rejected by API. Recovery cooldown is active; reconnect required if it persists.', 'shadowscan-security-link')
        );
    }

    $url = rtrim(SHADOWSCAN_API_BASE, '/') . '/' . ltrim($path, '/');
    $headers = array('Content-Type' => 'application/json');
    $api_key = (string) (defined('SHADOWSCAN_API_KEY') ? SHADOWSCAN_API_KEY : getenv('SHADOWSCAN_API_KEY'));
    if ($api_key !== '') {
        $headers['apikey'] = $api_key;
    }

    if (shadowscan_guard_manager()->is_flag_enabled('owasp_outbound_allowlist', false)
        && !shadowscan_is_outbound_url_allowed($url)) {
        return new WP_Error('outbound_blocked', __('Outbound request blocked by policy.', 'shadowscan-security-link'));
    }

    $user_agent = shadowscan_build_user_agent();
    $args = array(
        'timeout' => 15,
        'connect_timeout' => 5,
        'headers' => $headers,
    );
    if ($user_agent !== '') {
        $args['user-agent'] = $user_agent;
    }

    $method = strtoupper($method);
    $is_get = $method === 'GET';
    $has_idempotency_key = isset($payload['idempotency_key']) && is_string($payload['idempotency_key']) && $payload['idempotency_key'] !== '';
    $idempotent = $idempotent_override || $is_get || $has_idempotency_key;

    $body_string = '';
    if (!$is_get) {
        $body_string = wp_json_encode($payload);
        $args['body'] = $body_string;
    }

    if ($auth) {
        $token = shadowscan_get_auth_token();
        if ($token === '') {
            return new WP_Error('missing_token', __('Site token not configured', 'shadowscan-security-link'));
        }
        $headers['X-SS-Auth'] = 'site';

        $timestamp = (string) time();
        try {
            $nonce = rtrim(strtr(base64_encode(random_bytes(16)), '+/', '-_'), '=');
        } catch (Exception $e) {
            $nonce = wp_generate_uuid4();
        }
        $signed_path = shadowscan_canonical_path_with_query($url);
        $body_hash = hash('sha256', $body_string);
        $canonical = $method . "\n" . $signed_path . "\n" . $timestamp . "\n" . $nonce . "\n" . $body_hash;
        $signature = shadowscan_base64url_encode(hash_hmac('sha256', $canonical, $token, true));

        $auth_mode = shadowscan_get_auth_header_mode();
        $legacy_auth_value = sprintf(
            'Bearer %s, signature="%s", timestamp="%s", nonce="%s", body-sha256="%s", signed-path="%s"',
            $token,
            $signature,
            $timestamp,
            $nonce,
            $body_hash,
            $signed_path
        );

        if ($auth_mode === 'legacy_only') {
            $headers['Authorization'] = $legacy_auth_value;
            $headers['X-ShadowScan-Signature'] = $signature;
            $headers['X-ShadowScan-Timestamp'] = $timestamp;
            $headers['X-ShadowScan-Nonce'] = $nonce;
            $headers['X-ShadowScan-Body-Sha256'] = $body_hash;
            $headers['X-ShadowScan-Signed-Path'] = $signed_path;
        } else {
            if ($auth_mode === 'dual') {
                $headers['Authorization'] = $legacy_auth_value;
            } else {
                $headers['Authorization'] = 'Bearer ' . $token;
            }
            $headers['X-SS-Signature'] = $signature;
            $headers['X-SS-Timestamp'] = $timestamp;
            $headers['X-SS-Nonce'] = $nonce;
            $headers['X-SS-Body-Sha256'] = $body_hash;
            $headers['X-SS-Signed-Path'] = $signed_path;
            $headers['X-SS-Auth-Version'] = '2';
            if ($auth_mode === 'dual') {
                $headers['X-ShadowScan-Signature'] = $signature;
                $headers['X-ShadowScan-Timestamp'] = $timestamp;
                $headers['X-ShadowScan-Nonce'] = $nonce;
                $headers['X-ShadowScan-Body-Sha256'] = $body_hash;
                $headers['X-ShadowScan-Signed-Path'] = $signed_path;
            }
        }
        $args['headers'] = $headers;

        if (defined('WP_DEBUG') && WP_DEBUG) {
            shadowscan_log_message(sprintf(
                '[ShadowScan HMAC] %s %s | sig=%s | ts=%s | nonce=%s | body_hash=%s',
                $method,
                $path,
                isset($signature) ? substr($signature, 0, 12) . '...' : 'NONE',
                $timestamp ?? 'NONE',
                isset($nonce) ? substr($nonce, 0, 8) . '...' : 'NONE',
                isset($body_hash) ? substr($body_hash, 0, 12) . '...' : 'NONE'
            ));
        }
    }

    shadowscan_record_contact_attempt($path);

    $attempts = 0;
    $max_attempts = $idempotent ? 3 : 1;
    $response = null;
    while ($attempts < $max_attempts) {
        $attempts++;
        $response = wp_remote_request($url, array_merge($args, array('method' => $method)));
        if (!is_wp_error($response)) {
            break;
        }
        if ($attempts < $max_attempts) {
            usleep(200000 * $attempts);
        }
    }

    if (is_wp_error($response)) {
        shadowscan_record_error($path, $response->get_error_message());
        shadowscan_record_contact_error($path, $response->get_error_message());
        shadowscan_record_endpoint_health($path, false, $response->get_error_message());
        if (shadowscan_is_core_portal_path($path)) {
            ShadowScan_Subscription_State::mark_portal_unreachable();
        }
        return $response;
    }

    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $decoded = json_decode($body, true);

    if ($code < 200 || $code >= 300) {
        $correlation_id = '';
        foreach (array('x-request-id', 'x-correlation-id', 'x-trace-id', 'cf-ray') as $header_name) {
            $header_value = wp_remote_retrieve_header($response, $header_name);
            if (is_array($header_value)) {
                $header_value = implode(',', $header_value);
            }
            $header_value = is_string($header_value) ? trim($header_value) : '';
            if ($header_value !== '') {
                $correlation_id = $header_value;
                break;
            }
        }
        $snippet = '';
        if (is_string($body) && $body !== '') {
            $snippet = wp_strip_all_tags($body);
            $snippet = trim(preg_replace('/\s+/', ' ', $snippet));
            if (strlen($snippet) > 200) {
                $snippet = substr($snippet, 0, 200) . '...';
            }
        }
        $error_message = sprintf(
            'ShadowScan API error (%s %d%s%s)',
            $path,
            $code,
            $correlation_id !== '' ? ', id=' . $correlation_id : '',
            $snippet !== '' ? ', body=' . $snippet : ''
        );
        if ($code === 401 || $code === 403) {
            $was_auth_invalid = function_exists('shadowscan_sync_is_auth_invalid') ? shadowscan_sync_is_auth_invalid() : false;
            if (function_exists('shadowscan_sync_set_auth_invalid')) {
                shadowscan_sync_set_auth_invalid(true, 'auth_failed', $code);
            }
            if (!$was_auth_invalid) {
                if ($path === 'heartbeat' && function_exists('shadowscan_sync_mark_recovery_attempt')) {
                    shadowscan_sync_mark_recovery_attempt();
                } elseif ($path !== 'heartbeat') {
                    shadowscan_attempt_auth_recovery($path);
                }
            }
        }
        shadowscan_record_error($path, $error_message, $code);
        shadowscan_record_contact_error($path, $error_message);
        shadowscan_record_endpoint_health($path, false, $error_message);
        if (shadowscan_is_core_portal_path($path)) {
            ShadowScan_Subscription_State::mark_portal_unreachable();
        }
        return new WP_Error('api_error', $error_message, array('status' => $code, 'body' => $body));
    }

    shadowscan_record_endpoint_health($path, true);
    if (shadowscan_is_core_portal_path($path)) {
        ShadowScan_Subscription_State::clear_portal_unreachable();
    }
    shadowscan_clear_contact_error();
    shadowscan_clear_error($path);
    if (function_exists('shadowscan_sync_set_auth_invalid')) {
        shadowscan_sync_set_auth_invalid(false);
    }
    $received_header = wp_remote_retrieve_header($response, 'date');
    $received_at = $received_header ? strtotime((string) $received_header) : false;
    if (!$received_at) {
        $received_at = time();
    }
    update_option(
        SHADOWSCAN_OPTION_LAST_PORTAL_RECEIVED,
        array(
            'path' => $path,
            'received_at' => gmdate('c', (int) $received_at),
            'status' => $code,
        ),
        false
    );
    return $decoded ?? array();
}

function shadowscan_canonical_path_with_query(string $url): string {
    $path = (string) wp_parse_url($url, PHP_URL_PATH);
    $query = (string) wp_parse_url($url, PHP_URL_QUERY);
    if ($query === '') {
        return $path;
    }
    $pairs = array();
    foreach (explode('&', $query) as $part) {
        if ($part === '') {
            continue;
        }
        $kv = explode('=', $part, 2);
        $key = urldecode($kv[0]);
        $value = urldecode($kv[1] ?? '');
        $pairs[] = array($key, $value);
    }
    usort($pairs, static function ($a, $b) {
        if ($a[0] === $b[0]) {
            return strcmp($a[1], $b[1]);
        }
        return strcmp($a[0], $b[0]);
    });
    $encoded = array();
    foreach ($pairs as $pair) {
        $encoded[] = rawurlencode($pair[0]) . '=' . rawurlencode($pair[1]);
    }
    return $path . '?' . implode('&', $encoded);
}

function shadowscan_base64url_encode(string $binary): string {
    return rtrim(strtr(base64_encode($binary), '+/', '-_'), '=');
}

function shadowscan_is_outbound_url_allowed(string $url): bool {
    $host = wp_parse_url($url, PHP_URL_HOST);
    if (!is_string($host) || $host === '') {
        return false;
    }

    $allowed = array();
    $api_host = wp_parse_url(SHADOWSCAN_API_BASE, PHP_URL_HOST);
    if (is_string($api_host) && $api_host !== '') {
        $allowed[] = $api_host;
    }
    $portal_host = wp_parse_url(SHADOWSCAN_PORTAL_URL, PHP_URL_HOST);
    if (is_string($portal_host) && $portal_host !== '') {
        $allowed[] = $portal_host;
    }
    $site_host = wp_parse_url(home_url('/'), PHP_URL_HOST);
    if (is_string($site_host) && $site_host !== '') {
        $allowed[] = $site_host;
    }

    foreach ($allowed as $allowed_host) {
        if ($host === $allowed_host) {
            return true;
        }
        if (substr($host, -strlen('.' . $allowed_host)) === '.' . $allowed_host) {
            return true;
        }
    }
    return false;
}

function shadowscan_is_logging_degraded(): bool {
    $backoff_until = (int) ShadowScan_Storage::get('event_backoff_until', 0);
    if ($backoff_until > time()) {
        return true;
    }
    $queue = ShadowScan_Event_Buffer::count();
    return $queue >= 25;
}

function shadowscan_verify_command_signature(array $command): bool {
    $signature = isset($command['signature']) ? (string) $command['signature'] : '';
    if ($signature === '') {
        if (shadowscan_command_signature_debug_enabled()) {
            shadowscan_log_message('[ShadowScan Command Signature] verify failed: signature missing');
        }
        return false;
    }

    $payload = $command;
    unset($payload['signature']);
    $payload_variants = array(
        array(
            'label' => 'raw',
            'payload' => $payload,
        ),
    );
    $command_type = isset($payload['type']) ? (string) $payload['type'] : '';
    if ($command_type === 'refresh_status' || $command_type === 'force_heartbeat') {
        $legacy_payload = $payload;
        $legacy_payload['type'] = 'run_inventory';
        $payload_variants[] = array(
            'label' => 'alias_run_inventory',
            'payload' => $legacy_payload,
        );
    }

    $canonical_variants = array();
    foreach ($payload_variants as $variant) {
        $variant_payload = isset($variant['payload']) && is_array($variant['payload']) ? $variant['payload'] : array();
        $variant_canonical = shadowscan_command_signature_canonicalize($variant_payload);
        if ($variant_canonical !== '') {
            $canonical_variants[] = array(
                'label' => (string) ($variant['label'] ?? 'raw'),
                'canonical' => $variant_canonical,
                'sha' => substr(hash('sha256', $variant_canonical), 0, 16),
            );
        }
    }
    if (empty($canonical_variants)) {
        if (shadowscan_command_signature_debug_enabled()) {
            shadowscan_log_message('[ShadowScan Command Signature] verify failed: canonical payload empty');
        }
        return false;
    }

    $keys = array();
    $configured_key = defined('SHADOWSCAN_COMMAND_SIGNING_KEY') ? (string) SHADOWSCAN_COMMAND_SIGNING_KEY : '';
    if ($configured_key !== '') {
        $keys[] = array('label' => 'configured', 'key' => $configured_key);
    }
    $site_token = shadowscan_get_site_token();
    if ($site_token !== '' && $site_token !== $configured_key) {
        $keys[] = array('label' => 'site_token', 'key' => $site_token);
    }
    if (empty($keys)) {
        if (shadowscan_command_signature_debug_enabled()) {
            shadowscan_log_message('[ShadowScan Command Signature] verify failed: no signing key available');
        }
        return false;
    }

    foreach ($keys as $candidate) {
        $candidate_label = (string) $candidate['label'];
        $candidate_key = (string) $candidate['key'];
        foreach ($canonical_variants as $variant) {
            if (shadowscan_command_signature_matches_key((string) $variant['canonical'], $signature, $candidate_key)) {
                if (shadowscan_command_signature_debug_enabled()) {
                    $canary = substr(base64_encode(hash_hmac('sha256', 'probe', $candidate_key, true)), 0, 8);
                    shadowscan_log_message(sprintf(
                        '[ShadowScan Command Signature] verify ok: key=%s variant=%s canonical_sha256=%s sig=%s key_canary=%s',
                        $candidate_label,
                        (string) $variant['label'],
                        (string) $variant['sha'],
                        substr($signature, 0, 12) . '...',
                        $canary
                    ));
                }
                return true;
            }
        }
    }

    if (shadowscan_command_signature_debug_enabled()) {
        $variant_summaries = array();
        foreach ($canonical_variants as $variant) {
            $variant_summaries[] = (string) $variant['label'] . ':' . (string) $variant['sha'];
        }
        $key_summaries = array();
        foreach ($keys as $candidate) {
            $candidate_key = (string) $candidate['key'];
            $key_summaries[] = (string) $candidate['label'] . ':' . substr(base64_encode(hash_hmac('sha256', 'probe', $candidate_key, true)), 0, 8);
        }
        shadowscan_log_message(sprintf(
            '[ShadowScan Command Signature] verify failed: variants=%s sig=%s len=%d key_canaries=%s',
            implode(',', $variant_summaries),
            substr($signature, 0, 12) . '...',
            strlen($signature),
            implode(',', $key_summaries)
        ));
    }
    return false;
}

function shadowscan_command_signature_matches_key(string $canonical, string $signature, string $key): bool {
    if ($canonical === '' || $signature === '' || $key === '') {
        return false;
    }
    $expected_raw = hash_hmac('sha256', $canonical, $key, true);
    $expected_hex = hash_hmac('sha256', $canonical, $key);
    // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
    $decoded = base64_decode($signature, true);
    if ($decoded !== false && hash_equals($expected_raw, $decoded)) {
        return true;
    }

    $base64url = strtr($signature, '-_', '+/');
    $padding = strlen($base64url) % 4;
    if ($padding > 0) {
        $base64url .= str_repeat('=', 4 - $padding);
    }
    // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
    $decoded_urlsafe = base64_decode($base64url, true);
    if ($decoded_urlsafe !== false && hash_equals($expected_raw, $decoded_urlsafe)) {
        return true;
    }

    return hash_equals($expected_hex, $signature);
}

function shadowscan_command_signature_canonicalize(array $payload): string {
    $sorted = shadowscan_command_signature_sort_value($payload);
    $json = wp_json_encode($sorted, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    if (!is_string($json)) {
        return '';
    }
    return $json;
}

function shadowscan_command_signature_sort_value($value) {
    if (!is_array($value)) {
        return $value;
    }
    $sorted = array();
    foreach ($value as $key => $item) {
        $sorted[$key] = shadowscan_command_signature_sort_value($item);
    }
    if (array_keys($sorted) !== range(0, count($sorted) - 1)) {
        ksort($sorted);
    }
    return $sorted;
}

function shadowscan_command_signature_debug_enabled(): bool {
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        return false;
    }
    $explicit = defined('SHADOWSCAN_DEBUG_COMMAND_SIGNATURE') && (bool) SHADOWSCAN_DEBUG_COMMAND_SIGNATURE;
    return $explicit && shadowscan_is_dev_env();
}
function shadowscan_encrypt_secret(string $value): string {
    if (!function_exists('openssl_encrypt')) {
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
        return base64_encode($value);
    }

    $key = hash('sha256', AUTH_KEY, true);
    $iv = random_bytes(12);
    $tag = '';
    $cipher = openssl_encrypt($value, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
    if ($cipher === false) {
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
        return base64_encode($value);
    }
    // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
    return base64_encode($iv . $tag . $cipher);
}

function shadowscan_guard_safe_mode_monitor(): void {
    $error = error_get_last();
    if (!$error) {
        return;
    }

    $fatal_types = array(E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR);
    if (!in_array($error['type'], $fatal_types, true)) {
        return;
    }

    if (!class_exists('ShadowScan_Storage')) {
        return;
    }

    $now = time();
    $recent = ShadowScan_Storage::get_json('guard_fatal_errors', array());
    if (!is_array($recent)) {
        $recent = array();
    }
    $recent[] = $now;
    $cutoff = $now - (10 * MINUTE_IN_SECONDS);
    $recent = array_values(array_filter($recent, function ($timestamp) use ($cutoff) {
        return (int) $timestamp >= $cutoff;
    }));
    ShadowScan_Storage::set_json('guard_fatal_errors', $recent);

    if (count($recent) >= 3) {
        shadowscan_guard_manager()->set_safe_mode(30 * MINUTE_IN_SECONDS);
        if (class_exists('ShadowScan_Signal_Manager')) {
            ShadowScan_Signal_Manager::emit(
                'GUARD_SAFE_MODE_ENABLED',
                'warning',
                'Guard actions disabled after repeated errors',
                array('window_minutes' => 10)
            );
        }
    }
}

register_shutdown_function('shadowscan_guard_safe_mode_monitor');

function shadowscan_maybe_emit_mfa_coverage_snapshot(bool $force = false): void {
    if (!class_exists('ShadowScan_MFA') || !class_exists('ShadowScan_Signal_Manager')) {
        return;
    }

    $snapshot = ShadowScan_MFA::compute_admin_coverage();
    $previous = ShadowScan_Storage::get_json('mfa_coverage_snapshot', array());
    $last_sent = (int) ShadowScan_Storage::get('mfa_coverage_snapshot_at', 0);

    $changed = true;
    if (is_array($previous) && isset($previous['admin_total'], $previous['admin_enrolled'])) {
        $changed = $previous['admin_total'] !== $snapshot['admin_total']
            || $previous['admin_enrolled'] !== $snapshot['admin_enrolled']
            || ($previous['provider'] ?? '') !== ($snapshot['provider'] ?? '');
    }

    if (!$force && !$changed && (time() - $last_sent) < DAY_IN_SECONDS) {
        return;
    }

    $total = (int) $snapshot['admin_total'];
    $enrolled = (int) $snapshot['admin_enrolled'];
    $percent = (float) $snapshot['coverage_percent'];
    $provider = (string) ($snapshot['provider'] ?? '');

    $status = 'ok';
    $severity = 'info';
    if ($total === 0 || $provider === '') {
        $status = 'warn';
        $severity = 'warning';
    } elseif ($percent < 50) {
        $status = 'critical';
        $severity = 'high';
    } elseif ($percent < 90) {
        $status = 'warn';
        $severity = 'warning';
    }

    $details = array(
        'category' => 'security_control',
        'control_key' => 'mfa_coverage',
        'status' => $status,
        'admin_total' => $total,
        'admin_enrolled' => $enrolled,
        'coverage_percent' => $percent,
        'provider' => $provider,
        'recommended_action' => 'Enroll remaining administrators in MFA.',
        'evidence' => array(
            'admin_total' => $total,
            'admin_enrolled' => $enrolled,
            'coverage_percent' => $percent,
            'provider' => $provider,
        ),
        'last_checked_at' => gmdate('c', (int) $snapshot['computed_at']),
    );

    ShadowScan_Signal_Manager::emit(
        'mfa_coverage_snapshot',
        $severity,
        'MFA coverage snapshot',
        $details
    );

    ShadowScan_Storage::set_json('mfa_coverage_snapshot', $snapshot);
    ShadowScan_Storage::set('mfa_coverage_snapshot_at', time());
}

function shadowscan_send_heartbeat(bool $force = false) {
    if (!shadowscan_is_connected()) {
        return new WP_Error('not_connected', __('Site is not connected.', 'shadowscan-security-link'));
    }

    shadowscan_flush_pending_results();
    shadowscan_maybe_emit_mfa_coverage_snapshot();
    shadowscan_run_security_control_checks();
    shadowscan_flush_event_queue();

    $pending = get_option(SHADOWSCAN_OPTION_PENDING_HEARTBEAT, null);
    if (is_array($pending)) {
        $pending_response = shadowscan_send_api_request('heartbeat', $pending, 'POST', true);
        if (!is_wp_error($pending_response)) {
            if (is_array($pending_response)) {
                shadowscan_persist_identity_from_response($pending_response);
                ShadowScan_Subscription_State::ingest_from_response($pending_response);
                shadowscan_maybe_auto_apply_recommended();
                shadowscan_maybe_auto_apply_htaccess();
                if (class_exists('ShadowScan_Policy_State')
                    && (isset($pending_response['policy_version']) || isset($pending_response['policy']) || isset($pending_response['commands']))) {
                    $policy_payload = array(
                        'site_id' => isset($pending_response['site_id']) ? (string) $pending_response['site_id'] : shadowscan_get_site_id(),
                        'policy' => isset($pending_response['policy']) && is_array($pending_response['policy']) ? $pending_response['policy'] : array(),
                        'supportDebug' => isset($pending_response['supportDebug']) && is_array($pending_response['supportDebug']) ? $pending_response['supportDebug'] : array(),
                        'policy_version' => $pending_response['policy_version'] ?? null,
                        'commands' => isset($pending_response['commands']) && is_array($pending_response['commands']) ? $pending_response['commands'] : array(),
                    );
                    ShadowScan_Policy_State::ingest_policy_payload($policy_payload);
                }
                if (class_exists('ShadowScan_Policy_State')) {
                    ShadowScan_Policy_State::refresh_policy(false);
                }
                if (class_exists('ShadowScan_AutoUpdates')) {
                    ShadowScan_AutoUpdates::reconcile_applied_with_desired('heartbeat');
                }
            }
            delete_option(SHADOWSCAN_OPTION_PENDING_HEARTBEAT);
        }
    }

    $last_ping = (int) get_option(SHADOWSCAN_OPTION_LAST_PING, 0);
    if (!$force && (time() - $last_ping) < 600) {
        return null;
    }

    $report_data = shadowscan_get_connector_report_data();
    $capabilities = $report_data['capabilities'] ?? array();

    $sync_state = shadowscan_get_sync_state();
    $controls_dedupe = ShadowScan_Storage::get_json('security_control_dedupe', array());
    $controls_count_sent = is_array($controls_dedupe) ? count($controls_dedupe) : 0;
    $last_portal_received = get_option(SHADOWSCAN_OPTION_LAST_PORTAL_RECEIVED, array());
    $sync_state['heartbeat_sent_at'] = gmdate('c');
    $sync_state['portal_received_at'] = is_array($last_portal_received) ? ($last_portal_received['received_at'] ?? null) : null;
    $sync_state['controls_count_sent'] = $controls_count_sent;
    if (!isset($sync_state['controls_count_accepted'])) {
        $sync_state['controls_count_accepted'] = null;
    }
    if (!isset($sync_state['controls_rejected'])) {
        $sync_state['controls_rejected'] = array();
    }

    $payload = array(
        'site_url' => home_url(),
        'wp_home' => home_url(),
        'wp_admin_url' => admin_url(),
        'connector_report' => shadowscan_render_connector_report($report_data),
        'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
        'wp_version' => get_bloginfo('version'),
        'php_version' => PHP_VERSION,
        'can_install_plugins' => $capabilities['can_install_plugins'] ?? null,
        'can_activate_plugins' => $capabilities['can_activate_plugins'] ?? null,
        'portal_reachable' => $report_data['portal_reachable'] ?? null,
        'subscription_status' => $report_data['subscription_status'] ?? null,
        'subscription_plan' => $report_data['subscription_plan'] ?? null,
        'third_party_security_detected' => !empty($report_data['third_party_plugins']),
        'third_party_security_plugins' => $report_data['third_party_plugins'] ?? array(),
        'guard_status' => $report_data['guard_status'] ?? shadowscan_guard_manager()->get_guard_status(),
        'sync' => $sync_state,
    );

    $response = shadowscan_send_api_request('heartbeat', $payload, 'POST', true);
    update_option(SHADOWSCAN_OPTION_LAST_PING, time(), false);

    if (is_wp_error($response)) {
        update_option(SHADOWSCAN_OPTION_PENDING_HEARTBEAT, $payload, false);
        ShadowScan_Subscription_State::mark_portal_unreachable();
        shadowscan_log_message('[ShadowScan] Heartbeat failed: ' . $response->get_error_message());
        return $response;
    } elseif (is_array($response)) {
        shadowscan_persist_identity_from_response($response);
        ShadowScan_Subscription_State::ingest_from_response($response);
        ShadowScan_Subscription_State::clear_portal_unreachable();
        shadowscan_maybe_auto_apply_recommended();
        shadowscan_maybe_auto_apply_htaccess();
        if (class_exists('ShadowScan_Policy_State')
            && (isset($response['policy_version']) || isset($response['policy']) || isset($response['commands']))) {
            $policy_payload = array(
                'site_id' => isset($response['site_id']) ? (string) $response['site_id'] : shadowscan_get_site_id(),
                'policy' => isset($response['policy']) && is_array($response['policy']) ? $response['policy'] : array(),
                'supportDebug' => isset($response['supportDebug']) && is_array($response['supportDebug']) ? $response['supportDebug'] : array(),
                'policy_version' => $response['policy_version'] ?? null,
                'commands' => isset($response['commands']) && is_array($response['commands']) ? $response['commands'] : array(),
            );
            ShadowScan_Policy_State::ingest_policy_payload($policy_payload);
        }
        if (class_exists('ShadowScan_Policy_State')) {
            ShadowScan_Policy_State::refresh_policy(false);
        }
        if (class_exists('ShadowScan_AutoUpdates')) {
            ShadowScan_AutoUpdates::reconcile_applied_with_desired('heartbeat');
        }
    }
    return null;
}

function shadowscan_maybe_auto_apply_recommended(?array $subscription_state = null): void {
    if (!shadowscan_is_connected()) {
        return;
    }

    $state = $subscription_state ?: ShadowScan_Subscription_State::get_current_state();
    $status = $state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL;
    if ($status !== ShadowScan_Subscription_State::STATUS_ACTIVE) {
        return;
    }

    if (!ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_GUARD_PROFILE, $state)) {
        return;
    }
    if (!ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_HEADERS, $state)) {
        return;
    }

    if (ShadowScan_Subscription_State::get_protection_state($state) !== 'live') {
        return;
    }

    if (!shadowscan_guard_manager()->guard_actions_enabled()) {
        return;
    }

    $marker = ShadowScan_Storage::get_json('auto_recommended_applied', array());
    if (is_array($marker) && !empty($marker['applied'])) {
        return;
    }

    shadowscan_guard_manager()->apply_profile('baseline');
    shadowscan_guard_manager()->toggle_header_x_frame(true);

    ShadowScan_Storage::set_json('auto_recommended_applied', array(
        'applied' => true,
        'applied_at' => time(),
        'status' => $status,
    ));

    ShadowScan_Subscription_State::emit_event(
        'AUTO_RECOMMENDED_APPLIED',
        'info',
        'Auto-applied recommended baseline',
        array(
            'profile' => 'baseline',
            'headers' => array('x_frame' => true),
        )
    );
}

function shadowscan_maybe_auto_apply_htaccess(?array $subscription_state = null): void {
    if (!shadowscan_is_connected()) {
        return;
    }

    $enabled = (bool) get_option(SHADOWSCAN_OPTION_AUTO_HTACCESS, false);
    if (!$enabled) {
        return;
    }

    $state = $subscription_state ?: ShadowScan_Subscription_State::get_current_state();
    $status = $state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL;
    if ($status !== ShadowScan_Subscription_State::STATUS_ACTIVE) {
        return;
    }

    if (!ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_HTACCESS, $state)) {
        return;
    }

    if (ShadowScan_Subscription_State::get_protection_state($state) !== 'live') {
        return;
    }

    if (!shadowscan_guard_manager()->guard_actions_enabled()) {
        return;
    }

    $state_cache = shadowscan_guard_manager()->get_htaccess_state();
    if (($state_cache['last_result'] ?? '') === 'applied') {
        return;
    }

    $last_attempt = (int) ($state_cache['auto_attempted_at'] ?? 0);
    if ($last_attempt > 0 && (time() - $last_attempt) < DAY_IN_SECONDS) {
        return;
    }

    $ok = shadowscan_guard_manager()->apply_htaccess_baseline();
    $state_cache['auto_attempted_at'] = time();
    $state_cache['auto_attempt_result'] = $ok ? 'applied' : 'failed';
    ShadowScan_Storage::set_json('htaccess_state', $state_cache);
}

function shadowscan_command_known_types(): array {
    return array(
        'run_inventory',
        'apply_guard_profile',
        'toggle_guard',
        'enable_emergency_mode',
        'disable_emergency_mode',
        'apply_htaccess_baseline',
        'rollback_htaccess_shadowscan',
        'enable_enumeration_strict',
        'disable_enumeration_strict',
        'enable_csp_report_only',
        'disable_csp_report_only',
        'toggle_header_x_frame_options',
        'rotate_token',
        'debug_install_mu_helper',
        'debug_remove_mu_helper',
    );
}

function shadowscan_command_type_aliases(): array {
    return array(
        'set_guard_profile' => 'apply_guard_profile',
        'set_guard_flag' => 'toggle_guard',
        'toggle_guard_flag' => 'toggle_guard',
        'force_heartbeat' => 'run_inventory',
        'refresh_status' => 'run_inventory',
        'apply_htaccess' => 'apply_htaccess_baseline',
        'rollback_htaccess' => 'rollback_htaccess_shadowscan',
        'rotate_site_token' => 'rotate_token',
        'enable_emergency' => 'enable_emergency_mode',
        'disable_emergency' => 'disable_emergency_mode',
    );
}

function shadowscan_normalize_command($command, array &$unknown_types, array &$invalid_reasons): ?array {
    if (!is_array($command)) {
        $invalid_reasons['not_array'] = ($invalid_reasons['not_array'] ?? 0) + 1;
        return null;
    }

    $required = array('id', 'type', 'issued_at', 'expires_at', 'payload');
    foreach ($required as $field) {
        if (!array_key_exists($field, $command)) {
            $invalid_reasons['missing_' . $field] = ($invalid_reasons['missing_' . $field] ?? 0) + 1;
            return null;
        }
    }

    $type_raw = sanitize_text_field((string) $command['type']);
    if ($type_raw === '') {
        $invalid_reasons['missing_type'] = ($invalid_reasons['missing_type'] ?? 0) + 1;
        return null;
    }

    $aliases = shadowscan_command_type_aliases();
    $type = $aliases[$type_raw] ?? $type_raw;

    if (!in_array($type, shadowscan_command_known_types(), true)) {
        $unknown_types[$type] = ($unknown_types[$type] ?? 0) + 1;
        return null;
    }

    $issued_at = shadowscan_parse_command_time($command['issued_at']);
    if ($issued_at === 0) {
        $invalid_reasons['invalid_issued_at'] = ($invalid_reasons['invalid_issued_at'] ?? 0) + 1;
        return null;
    }

    $expires_at = shadowscan_parse_command_time($command['expires_at']);
    if ($expires_at === 0) {
        $invalid_reasons['invalid_expires_at'] = ($invalid_reasons['invalid_expires_at'] ?? 0) + 1;
        return null;
    }
    if ($expires_at < time()) {
        $invalid_reasons['expired'] = ($invalid_reasons['expired'] ?? 0) + 1;
        return null;
    }

    $payload = $command['payload'];
    if (!is_array($payload)) {
        $invalid_reasons['invalid_payload'] = ($invalid_reasons['invalid_payload'] ?? 0) + 1;
        return null;
    }

    $normalized = $command;
    $normalized['type'] = $type;
    $normalized['issued_at'] = $issued_at;
    $normalized['expires_at'] = $expires_at;
    if (!isset($normalized['schema_version'])) {
        $normalized['schema_version'] = '1';
    }

    return $normalized;
}

function shadowscan_parse_command_time($value): int {
    if (is_int($value) || is_float($value)) {
        return (int) $value;
    }
    if (is_string($value)) {
        $parsed = strtotime($value);
        if ($parsed !== false) {
            return $parsed;
        }
        if (ctype_digit($value)) {
            return (int) $value;
        }
    }
    return 0;
}

function shadowscan_report_unknown_command_types(array $unknown_types): void {
    if (empty($unknown_types)) {
        return;
    }

    $state = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_UNKNOWN_COMMAND_TYPES, array());
    if (!is_array($state)) {
        $state = array();
    }

    $now = time();
    $types_to_report = array();
    $total_count = 0;
    $first_seen = $now;

    foreach ($unknown_types as $type => $count) {
        if (!isset($state[$type])) {
            $state[$type] = array(
                'count' => 0,
                'first_seen_at' => $now,
                'last_seen_at' => 0,
                'last_reported_at' => 0,
            );
        }
        $state[$type]['count'] += (int) $count;
        $state[$type]['last_seen_at'] = $now;

        $last_reported = (int) ($state[$type]['last_reported_at'] ?? 0);
        if ($last_reported === 0 || ($now - $last_reported) >= HOUR_IN_SECONDS) {
            $types_to_report[] = $type;
            $state[$type]['last_reported_at'] = $now;
            $total_count += $state[$type]['count'];
            $first_seen = min($first_seen, (int) $state[$type]['first_seen_at']);
        }
    }

    ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_UNKNOWN_COMMAND_TYPES, $state);

    if (empty($types_to_report) || !class_exists('ShadowScan_Signal_Manager')) {
        return;
    }

    ShadowScan_Signal_Manager::emit(
        'COMMAND_UNKNOWN_TYPE',
        'warning',
        'Unknown command types received',
        array(
            'types' => array_values($types_to_report),
            'count' => $total_count,
            'first_seen_at' => gmdate('c', $first_seen),
        )
    );
}

function shadowscan_command_poll_is_backed_off(): bool {
    $until = (int) get_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_UNTIL, 0);
    return $until > 0 && time() < $until;
}

function shadowscan_command_poll_register_failure(string $message): void {
    $count = (int) get_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_COUNT, 0);
    $count++;
    update_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_COUNT, $count, false);

    $levels = array(60, 300, 900);
    $level = min(count($levels) - 1, (int) get_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_LEVEL, 0));
    if ($count >= 3) {
        $level = min($level + 1, count($levels) - 1);
        update_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_LEVEL, $level, false);
    }

    $delay = $levels[$level] ?? 300;
    update_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_UNTIL, time() + $delay, false);
    if (strpos(strtolower($message), 'timed out') !== false || strpos(strtolower($message), 'timeout') !== false) {
        $timeout_count = (int) get_option(SHADOWSCAN_OPTION_COMMAND_TIMEOUT_COUNT, 0);
        update_option(SHADOWSCAN_OPTION_COMMAND_TIMEOUT_COUNT, $timeout_count + 1, false);
    }
    shadowscan_record_contact_error('commands', $message);
}

function shadowscan_command_poll_register_success(): void {
    update_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_UNTIL, 0, false);
    update_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_COUNT, 0, false);
    update_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_LEVEL, 0, false);
    update_option(SHADOWSCAN_OPTION_COMMAND_LAST_HTTP_STATUS, 200, false);
    shadowscan_clear_contact_error();
}

function shadowscan_poll_commands(): void {
    if (!shadowscan_is_connected()) {
        return;
    }
    if (shadowscan_should_pause_remote_calls('commands')) {
        shadowscan_log_rate_limited(
            'command_poll_auth_cooldown',
            '[ShadowScan] Command poll paused during auth recovery cooldown.',
            900
        );
        return;
    }

    shadowscan_flush_pending_results();
    shadowscan_flush_event_queue();

    if (shadowscan_command_poll_is_backed_off()) {
        return;
    }

    $response = shadowscan_send_api_request('commands', array(), 'GET', true, true);
    if (is_wp_error($response)) {
        $error_data = $response->get_error_data();
        $status = 0;
        if (is_array($error_data) && isset($error_data['status'])) {
            $status = (int) $error_data['status'];
        }
        if ($status > 0) {
            update_option(SHADOWSCAN_OPTION_COMMAND_LAST_HTTP_STATUS, $status, false);
        }
        shadowscan_command_poll_register_failure($response->get_error_message());
        shadowscan_log_rate_limited('command_poll_failed', '[ShadowScan] Command poll failed: ' . $response->get_error_message(), 900);
        return;
    }
    shadowscan_command_poll_register_success();

    $commands = $response['commands'] ?? array();
    if (!is_array($commands) || empty($commands)) {
        return;
    }

    $unknown_types = array();
    $invalid_reasons = array();

    foreach ($commands as $command) {
        $raw_command = $command;
        $normalized = shadowscan_normalize_command($raw_command, $unknown_types, $invalid_reasons);
        if ($normalized === null) {
            continue;
        }
        $command = $normalized;

        // Verify signatures against the raw payload as received, then execute the normalized command.
        $result = shadowscan_execute_command($command, is_array($raw_command) ? $raw_command : null);
        shadowscan_send_command_result(
            $command['id'],
            $result['status'],
            $result['error'] ?? null,
            $result['payload'] ?? null
        );

        if ($command['type'] === 'rotate_token' && $result['status'] === 'succeeded') {
            delete_option(SHADOWSCAN_OPTION_SITE_TOKEN);
            delete_option(SHADOWSCAN_OPTION_SITE_ID);
            if (function_exists('shadowscan_sync_clear_identity')) {
                shadowscan_sync_clear_identity();
            }
        }
    }

    if (!empty($unknown_types)) {
        shadowscan_report_unknown_command_types($unknown_types);
    }
    if (!empty($invalid_reasons)) {
        $summary = array();
        foreach ($invalid_reasons as $reason => $count) {
            $summary[] = $reason . '=' . $count;
        }
        shadowscan_log_rate_limited(
            'command_schema_invalid',
            '[ShadowScan] Command schema invalid: ' . implode(', ', $summary),
            3600
        );
    }
}

function shadowscan_set_command_context(bool $active): void {
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
    $GLOBALS[SHADOWSCAN_COMMAND_CONTEXT_KEY] = $active;
}

function shadowscan_execute_command(array $command, ?array $signature_payload = null): array {
    shadowscan_set_command_context(true);
    $type = $command['type'];

    if (!in_array($type, shadowscan_command_known_types(), true)) {
        shadowscan_set_command_context(false);
        return array('status' => 'ignored', 'error' => __('Command not allowed.', 'shadowscan-security-link'));
    }

    if (shadowscan_guard_manager()->is_flag_enabled('owasp_enforce_command_signing', false)) {
        $signature_source = is_array($signature_payload) ? $signature_payload : $command;
        if (!shadowscan_verify_command_signature($signature_source)) {
            if (class_exists('ShadowScan_Signal_Manager')) {
                ShadowScan_Signal_Manager::emit(
                    'OWASP_COMMAND_SIGNATURE_INVALID',
                    'warning',
                    'Command signature verification failed',
                    array(
                        'control_id' => 'OWASP-A08',
                        'owasp_id' => 'A08',
                        'category' => 'owasp',
                        'control_key' => 'a08_enforce_signature_validation',
                        'status' => 'warn',
                        'recommended_action' => 'Enable command signing in the portal before enforcing signatures.',
                    )
                );
            }
            shadowscan_set_command_context(false);
            return array('status' => 'failed', 'error' => __('Command signature verification failed.', 'shadowscan-security-link'));
        }
    }

    $gated_features = array(
        'apply_guard_profile' => ShadowScan_Subscription_State::FEATURE_GUARD_PROFILE,
        'toggle_guard' => ShadowScan_Subscription_State::FEATURE_GUARD_PROFILE,
        'apply_htaccess_baseline' => ShadowScan_Subscription_State::FEATURE_HTACCESS,
        'rollback_htaccess_shadowscan' => ShadowScan_Subscription_State::FEATURE_HTACCESS,
        'enable_enumeration_strict' => ShadowScan_Subscription_State::FEATURE_HARDENING,
        'disable_enumeration_strict' => ShadowScan_Subscription_State::FEATURE_HARDENING,
        'enable_csp_report_only' => ShadowScan_Subscription_State::FEATURE_HEADERS,
        'disable_csp_report_only' => ShadowScan_Subscription_State::FEATURE_HEADERS,
        'toggle_header_x_frame_options' => ShadowScan_Subscription_State::FEATURE_HEADERS,
    );

    if (isset($gated_features[$type])) {
        $subscription_state = ShadowScan_Subscription_State::get_current_state();
        $status = $subscription_state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL;
        $policy_state = class_exists('ShadowScan_Policy_State') ? ShadowScan_Policy_State::get_current_policy() : array();
        $billing_active = $status === ShadowScan_Subscription_State::STATUS_ACTIVE;
        $enforcement_commands = array(
            'apply_guard_profile' => 'hardened',
            'toggle_guard' => 'hardened',
            'apply_htaccess_baseline' => 'managed',
            'rollback_htaccess_shadowscan' => 'managed',
            'enable_enumeration_strict' => 'hardened',
            'disable_enumeration_strict' => 'hardened',
            'enable_csp_report_only' => 'hardened',
            'disable_csp_report_only' => 'hardened',
            'toggle_header_x_frame_options' => 'hardened',
        );
        ShadowScan_Subscription_State::emit_event(
            'GATED_ACTION_ATTEMPTED',
            'info',
            'Gated command attempted',
            array(
                'action' => $type,
                'status' => $status,
            )
        );

        if (!ShadowScan_Subscription_State::is_feature_allowed($gated_features[$type], $subscription_state)) {
            ShadowScan_Subscription_State::emit_event(
                'GATED_ACTION_DENIED',
                'warning',
                'Gated command denied',
                array(
                    'action' => $type,
                    'status' => $status,
                )
            );
            shadowscan_set_command_context(false);
            return array('status' => 'failed', 'error' => __('Subscription inactive for this action.', 'shadowscan-security-link'));
        }
        if (isset($enforcement_commands[$type])) {
            if (!$billing_active && !shadowscan_is_dev_env()) {
                shadowscan_set_command_context(false);
                return array('status' => 'failed', 'error' => __('Billing inactive for this enforcement action.', 'shadowscan-security-link'));
            }
            if (!shadowscan_can_apply_enforcement($subscription_state, $policy_state, $enforcement_commands[$type])) {
                shadowscan_set_command_context(false);
                return array('status' => 'failed', 'error' => __('Policy tier does not allow this enforcement action.', 'shadowscan-security-link'));
            }
        }
    }

    $live_required_commands = array(
        'apply_guard_profile',
        'apply_htaccess_baseline',
        'rollback_htaccess_shadowscan',
        'enable_enumeration_strict',
        'disable_enumeration_strict',
        'enable_csp_report_only',
        'disable_csp_report_only',
        'toggle_header_x_frame_options',
        'toggle_guard',
        'enable_emergency_mode',
        'disable_emergency_mode',
    );
    if (shadowscan_guard_manager()->is_flag_enabled('owasp_fail_closed_logging', false)
        && in_array($type, $live_required_commands, true)
        && shadowscan_is_logging_degraded()) {
        shadowscan_set_command_context(false);
        return array('status' => 'failed', 'error' => __('Logging is degraded; risky actions are paused.', 'shadowscan-security-link'));
    }
    if (in_array($type, $live_required_commands, true)) {
        $subscription_state = ShadowScan_Subscription_State::get_current_state();
        $protection_state = ShadowScan_Subscription_State::get_protection_state($subscription_state);
        if ($protection_state !== 'live') {
            ShadowScan_Subscription_State::emit_event(
                'PROTECTION_STATE_DENIED',
                'warning',
                'Command blocked due to protection state',
                array(
                    'action' => $type,
                    'protection_state' => $protection_state,
                )
            );
            shadowscan_set_command_context(false);
            return array('status' => 'failed', 'error' => __('Protection state does not allow updates.', 'shadowscan-security-link'));
        }
    }

    if ($type === 'run_inventory') {
        shadowscan_send_heartbeat(true);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded');
    }

    if ($type === 'debug_install_mu_helper') {
        $result = shadowscan_install_mu_helper();
        shadowscan_set_command_context(false);
        return array(
            'status' => $result['ok'] ? 'succeeded' : 'failed',
            'error' => $result['error'] ?? null,
            'payload' => $result['payload'] ?? null,
        );
    }

    if ($type === 'debug_remove_mu_helper') {
        $result = shadowscan_remove_mu_helper();
        shadowscan_set_command_context(false);
        return array(
            'status' => $result['ok'] ? 'succeeded' : 'failed',
            'error' => $result['error'] ?? null,
            'payload' => $result['payload'] ?? null,
        );
    }

    if ($type === 'apply_guard_profile') {
        $payload = isset($command['payload']) && is_array($command['payload']) ? $command['payload'] : array();
        $profile = isset($payload['profile']) ? sanitize_text_field((string) $payload['profile']) : 'baseline';
        shadowscan_guard_manager()->apply_profile($profile);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded', 'payload' => array('profile' => $profile));
    }

    if ($type === 'enable_emergency_mode') {
        $payload = isset($command['payload']) && is_array($command['payload']) ? $command['payload'] : array();
        $reason = isset($payload['reason']) ? sanitize_text_field((string) $payload['reason']) : '';
        shadowscan_guard_manager()->enable_emergency_mode($reason);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded', 'payload' => array('enabled' => true));
    }

    if ($type === 'disable_emergency_mode') {
        shadowscan_guard_manager()->disable_emergency_mode();
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded', 'payload' => array('enabled' => false));
    }

    if ($type === 'apply_htaccess_baseline') {
        $ok = shadowscan_guard_manager()->apply_htaccess_baseline();
        shadowscan_set_command_context(false);
        return array('status' => $ok ? 'succeeded' : 'failed');
    }

    if ($type === 'rollback_htaccess_shadowscan') {
        $ok = shadowscan_guard_manager()->rollback_htaccess();
        shadowscan_set_command_context(false);
        return array('status' => $ok ? 'succeeded' : 'failed');
    }

    if ($type === 'enable_enumeration_strict') {
        shadowscan_guard_manager()->set_flag('disable_author_archives', true);
        shadowscan_guard_manager()->set_flag('restrict_rest_users', true);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded');
    }

    if ($type === 'disable_enumeration_strict') {
        shadowscan_guard_manager()->set_flag('disable_author_archives', false);
        shadowscan_guard_manager()->set_flag('restrict_rest_users', false);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded');
    }

    if ($type === 'enable_csp_report_only') {
        shadowscan_guard_manager()->enable_csp_report_only();
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded');
    }

    if ($type === 'disable_csp_report_only') {
        shadowscan_guard_manager()->set_flag('csp_report_only', false);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded');
    }

    if ($type === 'toggle_header_x_frame_options') {
        $payload = isset($command['payload']) && is_array($command['payload']) ? $command['payload'] : array();
        $enabled_raw = $payload['enabled'] ?? null;
        $enabled = filter_var($enabled_raw, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        if ($enabled === null) {
            shadowscan_set_command_context(false);
            return array('status' => 'failed', 'error' => __('Header toggle payload missing enabled flag.', 'shadowscan-security-link'));
        }
        shadowscan_guard_manager()->toggle_header_x_frame($enabled);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded', 'payload' => array('enabled' => $enabled));
    }

    if ($type === 'toggle_guard') {
        $payload = isset($command['payload']) && is_array($command['payload']) ? $command['payload'] : array();
        $key = isset($payload['guard']) ? (string) $payload['guard'] : (isset($payload['flag']) ? (string) $payload['flag'] : '');
        $key = sanitize_text_field($key);
        $enabled_raw = $payload['enabled'] ?? null;
        $enabled = filter_var($enabled_raw, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        if ($key === '' || $enabled === null) {
            shadowscan_set_command_context(false);
            return array('status' => 'failed', 'error' => __('Guard toggle payload missing guard or enabled flag.', 'shadowscan-security-link'));
        }
        shadowscan_guard_manager()->set_flag($key, $enabled);
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded', 'payload' => array('guard' => $key, 'enabled' => $enabled));
    }

    if ($type === 'rotate_token') {
        shadowscan_set_command_context(false);
        return array('status' => 'succeeded');
    }

    shadowscan_set_command_context(false);
    return array('status' => 'failed', 'error' => __('Unknown command type', 'shadowscan-security-link'));
}

function shadowscan_is_command_context(): bool {
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
    return !empty($GLOBALS[SHADOWSCAN_COMMAND_CONTEXT_KEY]);
}

function shadowscan_can_install_plugins(): bool {
    if (shadowscan_is_command_context()) {
        return true;
    }
    $snapshot = shadowscan_get_capabilities_snapshot();
    $value = $snapshot['values']['install_plugins'] ?? null;
    return $value === null ? false : (bool) $value;
}

function shadowscan_can_activate_plugins(): bool {
    if (shadowscan_is_command_context()) {
        return true;
    }
    $snapshot = shadowscan_get_capabilities_snapshot();
    $value = $snapshot['values']['activate_plugins'] ?? null;
    return $value === null ? false : (bool) $value;
}

function shadowscan_is_dev_env(): bool {
    if (defined('SHADOWSCAN_ENV')) {
        $env = strtolower((string) SHADOWSCAN_ENV);
        if (in_array($env, array('development', 'staging', 'local', 'docker'), true)) {
            return true;
        }
    }
    if (defined('WP_ENV')) {
        $env = strtolower((string) WP_ENV);
        if (in_array($env, array('development', 'staging', 'local', 'docker'), true)) {
            return true;
        }
    }
    $docker_flag = getenv('SHADOWSCAN_DOCKER');
    if (is_string($docker_flag) && strtolower($docker_flag) === 'true') {
        return true;
    }
    if (defined('SHADOWSCAN_ALLOW_SUBSCRIPTION_OVERRIDE') && SHADOWSCAN_ALLOW_SUBSCRIPTION_OVERRIDE) {
        return true;
    }
    $site_url = strtolower((string) site_url());
    $host = (string) wp_parse_url($site_url, PHP_URL_HOST);
    if ($host === '' && !empty($_SERVER['HTTP_HOST'])) {
        $host = sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']));
    }
    $host = strtolower($host);
    $is_local_host = $host !== '' && (
        $host === 'localhost'
        || $host === '127.0.0.1'
        || $host === 'wp-dev.pike.homes'
        || $host === 'staging.shadowscan.cloud'
        || strpos($host, '.local') !== false
        || strpos($host, '.test') !== false
        || strpos($host, '.dev') !== false
        || strpos($host, 'docker') !== false
    );
    if ($is_local_host) {
        return true;
    }
    if (defined('WP_DEBUG') && WP_DEBUG) {
        if (strpos($site_url, 'localhost') !== false
            || strpos($site_url, '127.0.0.1') !== false
            || strpos($site_url, 'wp-dev.pike.homes') !== false
            || strpos($site_url, 'docker') !== false
            || strpos($site_url, '.local') !== false
            || strpos($site_url, '.test') !== false
            || strpos($site_url, '.dev') !== false) {
            return true;
        }
    }
    return false;
}

function shadowscan_notifications_email_enabled(): bool {
    if (defined('SHADOWSCAN_NOTIFICATIONS_EMAIL_ENABLED')) {
        return (bool) SHADOWSCAN_NOTIFICATIONS_EMAIL_ENABLED;
    }
    if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env()) {
        return false;
    }
    return true;
}

function shadowscan_maybe_disable_notifications($pre, array $atts) {
    if (!shadowscan_notifications_email_enabled()) {
        shadowscan_log_rate_limited('notifications_suppressed', '[ShadowScan] Notification email suppressed (dev env).', 3600);
        return true;
    }
    return $pre;
}

function shadowscan_log_notification_failure($error): void {
    if ($error instanceof WP_Error) {
        shadowscan_log_rate_limited('notifications_failed', '[ShadowScan] Notification failed: ' . $error->get_error_message(), 3600);
    }
}

add_filter('pre_wp_mail', 'shadowscan_maybe_disable_notifications', 10, 2);
add_action('wp_mail_failed', 'shadowscan_log_notification_failure');

function shadowscan_is_admin_geo_verification_enabled(): bool {
    if (class_exists('ShadowScan_Policy_State')) {
        $support = ShadowScan_Policy_State::get_support_debug();
        if (!empty($support['enabled'])) {
            return true;
        }
    }
    if (defined('WP_DEBUG') && WP_DEBUG) {
        return true;
    }
    if (defined('SS_ENV') && strtolower((string) SS_ENV) === 'staging') {
        return true;
    }
    if (defined('SHADOWSCAN_ENV') && strtolower((string) SHADOWSCAN_ENV) === 'staging') {
        return true;
    }
    $user = wp_get_current_user();
    if ($user && !empty($user->user_email)) {
        $email = strtolower((string) $user->user_email);
        if (substr($email, -strlen('@shadowscan.com.au')) === '@shadowscan.com.au') {
            return true;
        }
    }
    return false;
}

function shadowscan_send_command_result(string $command_id, string $status, ?string $error_message = null, ?array $result_payload = null): void {
    $payload = array(
        'command_id' => $command_id,
        'status' => $status,
        'error' => $error_message,
    );

    if (!empty($result_payload)) {
        $payload['payload'] = $result_payload;
    }

    $response = shadowscan_send_api_request('commands-result', $payload, 'POST', true);
    if (is_wp_error($response)) {
        shadowscan_queue_pending_result($payload);
        shadowscan_log_message('[ShadowScan] Command result failed: ' . $response->get_error_message());
    }
}

function shadowscan_self_check(): array {
    $fs_available = false;
    if (!function_exists('WP_Filesystem')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    if (function_exists('WP_Filesystem')) {
        $fs_available = (bool) WP_Filesystem();
    }

    $temp_dir = function_exists('get_temp_dir') ? get_temp_dir() : sys_get_temp_dir();
    $fs = shadowscan_get_filesystem();
    $temp_writable = $fs && is_string($temp_dir) && $temp_dir !== '' ? $fs->is_writable($temp_dir) : false;
    $plugins_writable = $fs ? $fs->is_writable(WP_PLUGIN_DIR) : false;

    $payload = array(
        'wp_version' => get_bloginfo('version'),
        'php_version' => PHP_VERSION,
        'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
        'can_install_plugins' => shadowscan_can_install_plugins(),
        'can_activate_plugins' => shadowscan_can_activate_plugins(),
        'filesystem_available' => $fs_available,
        'plugins_dir_writable' => $plugins_writable,
        'temp_dir' => $temp_dir,
        'temp_dir_writable' => $temp_writable,
        'remote_diagnostics_enabled' => shadowscan_remote_diagnostics_enabled(),
        'sentry_telemetry_enabled' => shadowscan_sentry_telemetry_enabled(),
    );

    $ok = $plugins_writable && $fs_available && $temp_writable;
    if (!$ok) {
        return array(
            'success' => false,
            'message' => __('Self-check failed. Verify filesystem access and writable directories.', 'shadowscan-security-link'),
            'payload' => $payload,
        );
    }

    return array('success' => true, 'payload' => $payload);
}

function shadowscan_complete_pairing(string $code): array {
    $response = shadowscan_send_api_request('pair-complete', array('code' => $code), 'POST', false);
    if (is_wp_error($response)) {
        $guidance = shadowscan_connection_guidance(
            array(
                'context' => 'pair-complete',
                'message' => $response->get_error_message(),
                'status' => (int) ($response->get_error_data()['status'] ?? 0),
            )
        );
        $message = sprintf(
            '%s %s',
            $guidance['summary'],
            __('Check the pairing code and retry, or contact support if this persists.', 'shadowscan-security-link')
        );
        return array('success' => false, 'message' => $message);
    }

    if (empty($response['site_token'])) {
        shadowscan_record_error('pair-complete', __('Pairing failed: missing site token.', 'shadowscan-security-link'));
        return array('success' => false, 'message' => __('Pairing failed: missing site token.', 'shadowscan-security-link'));
    }

    shadowscan_set_site_token($response['site_token']);
    shadowscan_persist_identity_from_response($response);

    shadowscan_send_heartbeat(true);

    return array('success' => true, 'message' => __('Site connected to ShadowScan.', 'shadowscan-security-link'));
}

add_action('admin_menu', function () {
    add_menu_page(
        __('ShadowScan', 'shadowscan-security-link'),
        __('ShadowScan', 'shadowscan-security-link'),
        'manage_options',
        'shadowscan',
        'shadowscan_render_dashboard',
        SHADOWSCAN_MENU_ICON,
        81
    );
});

add_action('admin_init', function () {
    if (!is_admin() || !current_user_can('manage_options')) {
        return;
    }
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
    if ($page === 'shadowscan-dashboard') {
        shadowscan_redirect_legacy_dashboard();
    }
    if ($page === 'shadowscan-owasp') {
        shadowscan_redirect_legacy_owasp();
    }
});

function shadowscan_log_ui_redirect_once(string $from, string $to): void {
    $key = 'shadowscan_ui_redirect_logged';
    if (get_option($key)) {
        return;
    }
    update_option($key, 1, false);
    $message = sprintf('[ShadowScan] ui_route_redirect %s -> %s', $from, $to);
    shadowscan_log_message($message);
    if (class_exists('ShadowScan_Signal_Manager')) {
        ShadowScan_Signal_Manager::emit(
            'ui_route_redirect',
            'info',
            'UI route redirected',
            array('from' => $from, 'to' => $to)
        );
    }
}

function shadowscan_redirect_legacy_dashboard(): void {
    $target = admin_url('admin.php?page=shadowscan&tab=overview');
    shadowscan_log_ui_redirect_once('shadowscan-dashboard', 'shadowscan');
    wp_safe_redirect($target);
    exit;
}

function shadowscan_redirect_legacy_owasp(): void {
    $target = admin_url('admin.php?page=shadowscan&tab=coverage');
    shadowscan_log_ui_redirect_once('shadowscan-owasp', 'shadowscan');
    wp_safe_redirect($target);
    exit;
}

function shadowscan_log_message(string $message): void {
    $logs = get_option(SHADOWSCAN_OPTION_LOG, array());
    if (!is_array($logs)) {
        $logs = array();
    }
    $timestamp = gmdate('Y-m-d H:i:s');
    $logs[] = '[' . $timestamp . '] ' . $message;
    if (count($logs) > 200) {
        $logs = array_slice($logs, -200);
    }
    update_option(SHADOWSCAN_OPTION_LOG, $logs, false);
    do_action('shadowscan_log', $message);
}

function shadowscan_protection_status_mfa(array $entry = array()): array {
    $coverage = class_exists('ShadowScan_MFA')
        ? ShadowScan_MFA::compute_admin_coverage()
        : array('admin_total' => 0, 'admin_enrolled' => 0, 'coverage_percent' => 0, 'provider' => '');
    $total = (int) ($coverage['admin_total'] ?? 0);
    $percent = (int) ($coverage['coverage_percent'] ?? 0);
    $status = 'not_implemented';
    if ($total > 0) {
        if ($percent < 50) {
            $status = 'critical';
        } elseif ($percent < 90) {
            $status = 'warn';
        } else {
            $status = 'ok';
        }
    }
    $enforced = class_exists('ShadowScan_MFA') ? ShadowScan_MFA::is_enforcement_enabled() : false;
    return array(
        'status' => $status,
        'enforced' => $enforced,
        'last_checked_at' => time(),
        'message' => $total > 0
            /* translators: 1: enrolled admin count, 2: total admin count. */
            ? sprintf(__('Admins with MFA: %1$d/%2$d', 'shadowscan-security-link'), (int) ($coverage['admin_enrolled'] ?? 0), $total)
            : __('MFA coverage not yet reported.', 'shadowscan-security-link'),
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => $coverage,
    );
}

function shadowscan_protection_status_autoupdate(array $entry = array()): array {
    $status = class_exists('ShadowScan_AutoUpdates') ? ShadowScan_AutoUpdates::get_status() : array();
    $enabled = !empty($status['enabled']);
    return array(
        'status' => (string) ($status['status'] ?? ($enabled ? 'ok' : 'warn')),
        'enforced' => $enabled,
        'last_checked_at' => (int) ($status['checked_at'] ?? time()),
        'message' => $enabled ? __('Plugin auto-updates enabled.', 'shadowscan-security-link') : __('Plugin auto-updates disabled.', 'shadowscan-security-link'),
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => $status,
    );
}

function shadowscan_protection_status_password_policy(array $entry = array()): array {
    $status = class_exists('ShadowScan_Password_Policy') ? ShadowScan_Password_Policy::get_status() : array();
    $enforced = !empty($status['enforce_min_length']) || !empty($status['enforce_breached']);
    return array(
        'status' => (string) ($status['status'] ?? ($enforced ? 'ok' : 'warn')),
        'enforced' => $enforced,
        'last_checked_at' => (int) ($status['checked_at'] ?? time()),
        'message' => $enforced
            ? __('Password policy enforced on change/reset.', 'shadowscan-security-link')
            : __('Password policy not enforced on change/reset.', 'shadowscan-security-link'),
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => $status,
    );
}

function shadowscan_protection_status_geo_policy(array $entry = array()): array {
    $status = class_exists('ShadowScan_Geo_Policy') ? ShadowScan_Geo_Policy::get_status() : array();
    $mode = (string) ($status['mode'] ?? 'off');
    $enforced = $mode === 'enforce_wp_admin';
    return array(
        'status' => (string) ($status['status'] ?? ($mode === 'off' ? 'warn' : 'ok')),
        'enforced' => $enforced,
        'last_checked_at' => (int) ($status['checked_at'] ?? time()),
        'message' => $mode === 'off'
            ? __('Geo restrictions disabled.', 'shadowscan-security-link')
            /* translators: %s: geo policy mode string. */
            : sprintf(__('Geo mode: %s', 'shadowscan-security-link'), $mode),
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => $status,
    );
}

function shadowscan_protection_status_admin_geo_guard(array $entry = array()): array {
    $status = class_exists('ShadowScan_Admin_Geo_Guard') ? ShadowScan_Admin_Geo_Guard::get_status() : array();
    $message = (string) ($status['message'] ?? __('Tracking admin access locations.', 'shadowscan-security-link'));
    return array(
        'status' => (string) ($status['status'] ?? 'warn'),
        'enforced' => false,
        'last_checked_at' => (int) ($status['checked_at'] ?? time()),
        'message' => $message,
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => $status,
    );
}

function shadowscan_protection_status_delivery(array $entry = array()): array {
    $queue_count = class_exists('ShadowScan_Event_Buffer') ? ShadowScan_Event_Buffer::count() : 0;
    $cron_last_tick = (int) ShadowScan_Storage::get('cron_last_tick', 0);
    $cron_stale = $cron_last_tick > 0 && (time() - $cron_last_tick) > 1800;
    $cron_critical = $cron_last_tick > 0 && (time() - $cron_last_tick) > 7200;
    $logging_degraded = function_exists('shadowscan_is_logging_degraded') ? shadowscan_is_logging_degraded() : false;
    $status = 'ok';
    if ($cron_critical || $logging_degraded) {
        $status = 'critical';
    } elseif ($cron_stale || $queue_count > 0) {
        $status = 'warn';
    }
    return array(
        'status' => $status,
        'enforced' => false,
        'last_checked_at' => time(),
        'message' => $status === 'ok'
            ? __('Event delivery is healthy.', 'shadowscan-security-link')
            : __('Event delivery needs attention.', 'shadowscan-security-link'),
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => array(
            'queue_count' => $queue_count,
            'cron_last_tick' => $cron_last_tick,
            'logging_degraded' => $logging_degraded,
        ),
    );
}

function shadowscan_protection_status_evidence(array $entry = array()): array {
    $connected = function_exists('shadowscan_is_connected') ? shadowscan_is_connected() : false;
    return array(
        'status' => $connected ? 'ok' : 'warn',
        'enforced' => false,
        'last_checked_at' => time(),
        'message' => $connected ? __('Evidence exports available.', 'shadowscan-security-link') : __('Connect this site to enable evidence exports.', 'shadowscan-security-link'),
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => array('connected' => $connected),
    );
}

function shadowscan_protection_get_control_state(array $entry): array {
    $state = array(
        'status' => 'not_implemented',
        'enforced' => false,
        'last_checked_at' => 0,
        'message' => '',
        'recommended_action' => (string) ($entry['recommended_action'] ?? ''),
        'evidence' => array(),
        'fingerprint' => '',
    );
    $callback = $entry['status_callback'] ?? '';
    if ($callback && is_callable($callback)) {
        $custom = call_user_func($callback, $entry);
        if (is_array($custom)) {
            return array_merge($state, $custom);
        }
    }
    $control_keys = isset($entry['control_keys']) && is_array($entry['control_keys']) ? $entry['control_keys'] : array();
    foreach ($control_keys as $control_key) {
        $control = function_exists('shadowscan_owasp_get_control_status')
            ? shadowscan_owasp_get_control_status((string) $control_key)
            : array();
        if (!empty($control)) {
            return array_merge($state, array(
                'status' => (string) ($control['status'] ?? 'not_implemented'),
                'enforced' => !empty($control['enforced']),
                'last_checked_at' => (int) ($control['last_checked_at'] ?? 0),
                'message' => (string) ($control['message'] ?? ''),
                'recommended_action' => (string) ($control['recommended_action'] ?? $state['recommended_action']),
                'evidence' => $control['evidence'] ?? array(),
                'fingerprint' => (string) ($control['fingerprint'] ?? ''),
            ));
        }
    }
    return $state;
}

function shadowscan_policy_allows_enforcement(array $policy_state, string $required_tier = 'hardened'): bool {
    $enforcement_enabled = !empty($policy_state['enforcement_enabled']);
    if ($enforcement_enabled) {
        return true;
    }
    $tier = strtolower((string) ($policy_state['tier'] ?? 'monitor'));
    $rank = array(
        'monitor' => 0,
        'hardened' => 1,
        'managed' => 2,
    );
    $current = $rank[$tier] ?? 0;
    $required = $rank[strtolower($required_tier)] ?? 0;
    return $current >= $required;
}

function shadowscan_can_apply_enforcement(array $subscription_state, array $policy_state, string $required_tier = 'hardened'): bool {
    if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env()) {
        return true;
    }
    $status = strtolower((string) ($subscription_state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL));
    $billing_active = $status === ShadowScan_Subscription_State::STATUS_ACTIVE;
    return $billing_active || shadowscan_policy_allows_enforcement($policy_state, $required_tier);
}

function shadowscan_get_billing_capabilities(array $subscription_state, array $policy_state, string $protection_state): array {
    $is_dev = function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env();
    $status = strtolower((string) ($subscription_state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL));
    $billing_active = $status === ShadowScan_Subscription_State::STATUS_ACTIVE || $is_dev;
    $can_enforce_hardened = $billing_active
        && $protection_state === 'live'
        && shadowscan_can_apply_enforcement($subscription_state, $policy_state, 'hardened');
    $can_enforce_managed = $billing_active
        && $protection_state === 'live'
        && shadowscan_can_apply_enforcement($subscription_state, $policy_state, 'managed');

    return array(
        'billing_active' => $billing_active,
        'can_enforce_hardened' => $can_enforce_hardened,
        'can_enforce_managed' => $can_enforce_managed,
        'can_enforce_mfa' => $can_enforce_hardened,
        'can_configure_autoupdate' => $can_enforce_hardened,
        'can_configure_password' => $can_enforce_hardened,
        'can_configure_geo' => true,
        'can_enforce_geo' => $can_enforce_hardened,
        'can_configure_hardening' => $can_enforce_hardened,
        'can_configure_headers' => $can_enforce_hardened,
        'can_configure_server' => $can_enforce_managed,
        'can_configure_remote' => $can_enforce_managed,
        'can_use_safe_mode' => true,
        'can_use_emergency' => true,
    );
}

function shadowscan_coverage_ack_key(array $entry): string {
    $keys = isset($entry['control_keys']) && is_array($entry['control_keys']) ? $entry['control_keys'] : array();
    $control_key = $keys[0] ?? '';
    if ($control_key !== '') {
        return $control_key;
    }
    return (string) ($entry['id'] ?? '');
}

function shadowscan_coverage_ack_blocked(string $ack_key, string $entry_id = ''): bool {
    $blocked = array(
        'a01_privilege_escalation',
        'a08_plugin_file_integrity_changed',
        'a09_audit_log_disabled',
        'event_delivery_health',
    );
    if (in_array($ack_key, $blocked, true)) {
        return true;
    }
    if ($entry_id !== '' && in_array($entry_id, array('visibility.delivery'), true)) {
        return true;
    }
    return false;
}

function shadowscan_coverage_get_acknowledgements(): array {
    $acks = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_CONTROL_ACKS, array());
    return is_array($acks) ? $acks : array();
}

function shadowscan_coverage_save_acknowledgements(array $acks): void {
    ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_CONTROL_ACKS, $acks);
}

function shadowscan_coverage_control_fingerprint(array $entry, array $state): string {
    $fingerprint = (string) ($state['fingerprint'] ?? '');
    if ($fingerprint !== '') {
        return $fingerprint;
    }
    $payload = array(
        'id' => (string) ($entry['id'] ?? ''),
        'status' => (string) ($state['status'] ?? ''),
        'enforced' => !empty($state['enforced']),
        'message' => (string) ($state['message'] ?? ''),
        'recommended_action' => (string) ($state['recommended_action'] ?? ''),
    );
    return hash('sha256', wp_json_encode($payload));
}

function shadowscan_coverage_ack_is_active(array $ack, string $fingerprint): bool {
    $expires_at = (int) ($ack['expires_at'] ?? 0);
    if ($expires_at <= 0 || time() > $expires_at) {
        return false;
    }
    return isset($ack['fingerprint']) && hash_equals((string) $ack['fingerprint'], $fingerprint);
}

function shadowscan_coverage_label_managed(): string {
    return __('Managed by ShadowScan', 'shadowscan-security-link');
}

function shadowscan_coverage_label_external(): string {
    return __('External action', 'shadowscan-security-link');
}

function shadowscan_protection_coverage_label(string $status, bool $enforcement_supported, bool $enforced): array {
    $normalized = strtolower($status);
    if ($normalized === 'critical') {
        return array('label' => __('Action required', 'shadowscan-security-link'), 'tone' => 'is-bad');
    }
    if ($normalized === 'warn') {
        return array('label' => shadowscan_coverage_label_external(), 'tone' => 'is-warn');
    }
    if ($normalized === 'ok' || $normalized === 'resolved') {
        if ($enforcement_supported && !$enforced) {
            return array('label' => shadowscan_coverage_label_managed(), 'tone' => 'is-warn');
        }
        return array('label' => __('Covered', 'shadowscan-security-link'), 'tone' => 'is-ok');
    }
    return array('label' => __('Monitoring planned', 'shadowscan-security-link'), 'tone' => '');
}

function shadowscan_handle_coverage_actions(string &$notice, string &$notice_type, array $subscription_state, string $protection_state): void {
    if (!current_user_can('manage_options')) {
        return;
    }

    $request_method = isset($_SERVER['REQUEST_METHOD'])
        ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD']))
        : '';
    if ($request_method !== 'POST' || empty($_POST['shadowscan_coverage_action'])) {
        return;
    }

    check_admin_referer('shadowscan-coverage');
    $action = sanitize_text_field(wp_unslash($_POST['shadowscan_coverage_action']));
    $policy_state = class_exists('ShadowScan_Policy_State') ? ShadowScan_Policy_State::get_current_policy() : array();
    $registry = function_exists('shadowscan_owasp_get_registry') ? shadowscan_owasp_get_registry() : array();
    $controls_by_key = array();
    $categories_by_key = array();
    foreach ($registry as $category) {
        $category_id = (string) ($category['category_id'] ?? '');
        foreach ($category['controls'] ?? array() as $control) {
            if (!empty($control['control_key'])) {
                $controls_by_key[$control['control_key']] = $control;
                if ($category_id !== '') {
                    $categories_by_key[$control['control_key']] = $category_id;
                }
            }
        }
    }

    if ($action === 'toggle_control') {
        $control_key = sanitize_text_field(wp_unslash($_POST['control_key'] ?? ''));
        $enabled = isset($_POST['control_enabled']) && $_POST['control_enabled'] === '1';
        $control = $controls_by_key[$control_key] ?? null;
        $category_id = $categories_by_key[$control_key] ?? '';
        $feature_allowed = ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_HARDENING, $subscription_state);
        if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env()) {
            $feature_allowed = true;
        }
        $can_toggle = $feature_allowed
            && $protection_state === 'live'
            && shadowscan_can_apply_enforcement($subscription_state, $policy_state, 'hardened');

        if (!$control || empty($control['enforcement_callback'])) {
            $notice = esc_html__('Control could not be updated.', 'shadowscan-security-link');
            $notice_type = 'error';
            return;
        }
        if (!$can_toggle) {
            $notice = esc_html__('Changes are paused until billing is active or policy allows enforcement.', 'shadowscan-security-link');
            $notice_type = 'warning';
            return;
        }
        $previous_enabled = !empty($control['toggle_option_key'])
            ? shadowscan_guard_manager()->is_flag_enabled((string) $control['toggle_option_key'], false)
            : false;
        call_user_func($control['enforcement_callback'], $enabled);
        $notice = $enabled
            ? esc_html__('Control enforcement enabled.', 'shadowscan-security-link')
            : esc_html__('Control enforcement disabled.', 'shadowscan-security-link');
        $notice_type = 'success';
        if (class_exists('ShadowScan_Signal_Manager')) {
            $status_value = $enabled ? 'enforced' : 'unenforced';
            $fingerprint = hash('sha256', $control_key . '|' . $status_value);
            ShadowScan_Signal_Manager::emit(
                'owasp_enforcement_changed',
                'info',
                'Control enforcement toggled',
                array(
                    'control_id' => $category_id !== '' ? 'OWASP-' . $category_id : 'OWASP-A01',
                    'owasp_id' => $category_id !== '' ? $category_id : 'A01',
                    'category' => 'owasp',
                    'control_key' => $control_key,
                    'status_from' => $previous_enabled ? 'enforced' : 'unenforced',
                    'status_to' => $status_value,
                    'status' => $status_value,
                    'fingerprint' => $fingerprint,
                    'recommended_action' => function_exists('shadowscan_owasp_recommended_action')
                        ? shadowscan_owasp_recommended_action($control_key)
                        : '',
                    'change_type' => 'enforcement_toggle',
                    'owasp' => array(
                        'version' => '2025',
                        'id' => $category_id !== '' ? $category_id : 'A01',
                        'name' => function_exists('shadowscan_owasp_category_name')
                            ? shadowscan_owasp_category_name($category_id !== '' ? $category_id : 'A01')
                            : '',
                    ),
                )
            );
        }
        shadowscan_owasp_refresh_status(true);
        return;
    }

    if ($action === 'ack_control') {
        $ack_key = sanitize_text_field(wp_unslash($_POST['ack_key'] ?? ''));
        $ack_action = sanitize_text_field(wp_unslash($_POST['ack_action'] ?? ''));
        if ($ack_key === '' || $ack_action === '') {
            $notice = esc_html__('Acknowledgement request is invalid.', 'shadowscan-security-link');
            $notice_type = 'error';
            return;
        }

        $coverage_registry = class_exists('ShadowScan_Protection_Registry')
            ? ShadowScan_Protection_Registry::get_registry()
            : array();
        $entry = null;
        foreach ($coverage_registry as $candidate) {
            $candidate_key = shadowscan_coverage_ack_key($candidate);
            if ($candidate_key === $ack_key || ($candidate['id'] ?? '') === $ack_key) {
                $entry = $candidate;
                break;
            }
        }

        if (!$entry) {
            $notice = esc_html__('Control could not be found.', 'shadowscan-security-link');
            $notice_type = 'error';
            return;
        }

        $state = shadowscan_protection_get_control_state($entry);
        $label_meta = shadowscan_protection_coverage_label(
            (string) ($state['status'] ?? 'not_implemented'),
            !empty($entry['enforcement_supported']),
            !empty($state['enforced'])
        );
        $raw_label = $label_meta['label'];
        $eligible = in_array(
            $raw_label,
            array(__('Action required', 'shadowscan-security-link'), shadowscan_coverage_label_managed(), shadowscan_coverage_label_external()),
            true
        );
        if ($ack_action !== 'clear' && (!$eligible || shadowscan_coverage_ack_blocked($ack_key, (string) ($entry['id'] ?? '')))) {
            $notice = esc_html__('Acknowledgements are not available for this control.', 'shadowscan-security-link');
            $notice_type = 'warning';
            return;
        }

        $acks = shadowscan_coverage_get_acknowledgements();
        $fingerprint = shadowscan_coverage_control_fingerprint($entry, $state);
        $now = time();
        $ack_type = 'acknowledged';
        $expires_at = 0;

        if ($ack_action === 'clear') {
            if (!isset($acks[$ack_key])) {
                $notice = esc_html__('This control has no active acknowledgement.', 'shadowscan-security-link');
                $notice_type = 'warning';
                return;
            }
            unset($acks[$ack_key]);
            shadowscan_coverage_save_acknowledgements($acks);
            if (class_exists('ShadowScan_Signal_Manager')) {
                ShadowScan_Signal_Manager::emit(
                    'security_control_acknowledgement_cleared',
                    'info',
                    'Control acknowledgement cleared',
                    array(
                        'category' => 'security_control',
                        'control_key' => $ack_key,
                        'status' => 'cleared',
                        'fingerprint' => hash('sha256', $ack_key . '|cleared|' . $now),
                        'recommended_action' => (string) ($state['recommended_action'] ?? ''),
                        'evidence' => array(
                            'action' => 'cleared',
                            'cleared_at' => gmdate('c', $now),
                        ),
                        'last_checked_at' => gmdate('c', $now),
                        'owasp' => !empty($entry['owasp_ids']) ? array(
                            'version' => '2025',
                            'id' => (string) $entry['owasp_ids'][0],
                            'name' => function_exists('shadowscan_owasp_category_name')
                                ? shadowscan_owasp_category_name((string) $entry['owasp_ids'][0])
                                : (string) $entry['owasp_ids'][0],
                        ) : array(),
                    )
                );
            }
            $notice = esc_html__('Acknowledgement cleared.', 'shadowscan-security-link');
            $notice_type = 'success';
            return;
        }

        if ($ack_action === 'mute_30') {
            $ack_type = 'muted';
            $expires_at = $now + (DAY_IN_SECONDS * SHADOWSCAN_ACK_MUTE_DAYS_30);
        } elseif ($ack_action === 'mute_90') {
            $ack_type = 'muted';
            $expires_at = $now + (DAY_IN_SECONDS * SHADOWSCAN_ACK_MUTE_DAYS_90);
        } elseif ($ack_action === 'ack') {
            $ack_type = 'acknowledged';
            $expires_at = $now + (DAY_IN_SECONDS * SHADOWSCAN_ACK_DEFAULT_DAYS);
        } else {
            $notice = esc_html__('Acknowledgement request is invalid.', 'shadowscan-security-link');
            $notice_type = 'error';
            return;
        }

        $acks[$ack_key] = array(
            'type' => $ack_type,
            'expires_at' => $expires_at,
            'fingerprint' => $fingerprint,
            'set_at' => $now,
            'set_by' => get_current_user_id(),
        );
        shadowscan_coverage_save_acknowledgements($acks);

        if (class_exists('ShadowScan_Signal_Manager')) {
            ShadowScan_Signal_Manager::emit(
                'security_control_acknowledged',
                'info',
                'Control acknowledgement applied',
                array(
                    'category' => 'security_control',
                    'control_key' => $ack_key,
                    'status' => $ack_type,
                    'fingerprint' => hash('sha256', $ack_key . '|' . $ack_type . '|' . $expires_at),
                    'recommended_action' => (string) ($state['recommended_action'] ?? ''),
                    'evidence' => array(
                        'ack_type' => $ack_type,
                        'expires_at' => gmdate('c', $expires_at),
                        'set_by' => get_current_user_id(),
                        'set_at' => gmdate('c', $now),
                    ),
                    'last_checked_at' => gmdate('c', $now),
                    'owasp' => !empty($entry['owasp_ids']) ? array(
                        'version' => '2025',
                        'id' => (string) $entry['owasp_ids'][0],
                        'name' => function_exists('shadowscan_owasp_category_name')
                            ? shadowscan_owasp_category_name((string) $entry['owasp_ids'][0])
                            : (string) $entry['owasp_ids'][0],
                    ) : array(),
                )
            );
        }

        $notice = $ack_type === 'muted'
            ? esc_html__('Control muted temporarily.', 'shadowscan-security-link')
            : esc_html__('Control acknowledged.', 'shadowscan-security-link');
        $notice_type = 'success';
        return;
    }

    if ($action === 'coverage_test_alert') {
        if (class_exists('ShadowScan_Signal_Manager')) {
            ShadowScan_Signal_Manager::emit(
                'owasp_test_event',
                'info',
                'Coverage control test alert',
                array(
                    'control_id' => 'OWASP-A01',
                    'owasp_id' => 'A01',
                    'category' => 'owasp',
                    'control_key' => 'a01_test_alert',
                    'risk_level' => 'low',
                    'recommended_action' => 'Verify alert routing and dashboards.',
                    'owasp' => array(
                        'version' => '2025',
                        'id' => 'A01',
                        'name' => function_exists('shadowscan_owasp_category_name')
                            ? shadowscan_owasp_category_name('A01')
                            : '',
                    ),
                )
            );
        }
        $notice = esc_html__('Test alert sent.', 'shadowscan-security-link');
        $notice_type = 'success';
    }
}

add_action('shadowscan_log', function () {
    // Intentionally no-op. Hook is available for integrators who want to capture logs.
}, 10, 0);

function shadowscan_audit_event(string $type, string $summary, array $details = array(), string $severity = 'info'): void {
    $details = array_merge(
        array(
            'user_id' => get_current_user_id(),
            'user_role' => shadowscan_get_current_user_role(),
        ),
        $details
    );
    ShadowScan_Subscription_State::emit_event($type, $severity, $summary, $details);
}

function shadowscan_get_current_user_role(): string {
    $user = wp_get_current_user();
    if (!$user || empty($user->roles)) {
        return '';
    }
    return (string) $user->roles[0];
}

function shadowscan_billing_cta_url(): string {
    return SHADOWSCAN_PORTAL_BILLING_URL;
}

function shadowscan_remote_diagnostics_enabled(): bool {
    if (!shadowscan_sentry_telemetry_enabled()) {
        update_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS, false, false);
        delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES);
        if (shadowscan_mu_helper_installed()) {
            shadowscan_remove_mu_helper();
        }
        return false;
    }

    $enabled = (bool) get_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS, false);
    if (!$enabled) {
        return false;
    }
    $expires = (int) get_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES, 0);
    if ($expires > 0 && time() > $expires) {
        update_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS, false, false);
        delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES);
        return false;
    }
    return true;
}

function shadowscan_redact_log_line(string $line): string {
    $redacted = preg_replace('/Bearer\\s+[A-Za-z0-9\\-_.]+/', 'Bearer [redacted]', $line);
    $redacted = preg_replace('/pairing_code=([A-Za-z0-9\\-_.]+)/', 'pairing_code=[redacted]', $redacted ?? '');
    $redacted = preg_replace('/site_token[^A-Za-z0-9]*[A-Za-z0-9\\-_.]+/i', 'site_token=[redacted]', $redacted ?? '');
    $redacted = preg_replace('/api_key[^A-Za-z0-9]*[A-Za-z0-9\\-_.]+/i', 'api_key=[redacted]', $redacted ?? '');
    return $redacted ?? $line;
}

function shadowscan_read_log_tail(string $path, int $max_bytes): string {
    $fs = shadowscan_get_filesystem();
    if (!$fs || !$fs->exists($path) || !$fs->is_readable($path)) {
        return '';
    }
    $data = $fs->get_contents($path);
    if (!is_string($data)) {
        return '';
    }
    $size = strlen($data);
    if ($size <= 0) {
        return '';
    }
    if ($size > $max_bytes) {
        $data = substr($data, -$max_bytes);
    }
    $lines = preg_split('/\\r?\\n/', $data);
    $lines = array_map('shadowscan_redact_log_line', $lines);
    return implode("\n", $lines);
}

function shadowscan_collect_debug_logs(): array {
    $debug_log_path = WP_CONTENT_DIR . '/debug.log';
    $php_error_log = ini_get('error_log');
    $php_error_log = is_string($php_error_log) ? $php_error_log : '';

    $payload = array(
        'debug_log_path' => $debug_log_path,
        'php_error_log_path' => $php_error_log,
        'debug_log_tail' => $debug_log_path ? shadowscan_read_log_tail($debug_log_path, SHADOWSCAN_REMOTE_LOG_MAX_BYTES) : '',
        'php_error_log_tail' => $php_error_log ? shadowscan_read_log_tail($php_error_log, SHADOWSCAN_REMOTE_LOG_MAX_BYTES) : '',
    );

    return $payload;
}

function shadowscan_build_mu_helper_contents(): string {
    if (!shadowscan_sentry_telemetry_enabled()) {
        return '';
    }

    $dsn = shadowscan_get_sentry_dsn();
    if ($dsn === '') {
        return '';
    }

    $lines = array(
        '<?php',
        '/**',
        ' * Plugin Name: ShadowScan Sentry MU Helper',
        ' * Description: Captures early WordPress errors for ShadowScan diagnostics.',
        ' * Version: 0.1.0',
        ' */',
        '',
        "if (!defined('ABSPATH')) {",
        '    exit;',
        '}',
        '',
        "\$autoload = WP_CONTENT_DIR . '/plugins/shadowscan-security-link/vendor/autoload.php';",
        'if (file_exists($autoload)) {',
        '    require_once $autoload;',
        '}',
        '',
        "if (function_exists('\\\\Sentry\\\\init')) {",
        '    try {',
        '        $httpFactory = null;',
        "        if (class_exists('\\\\Nyholm\\\\Psr7\\\\Factory\\\\Psr17Factory')) {",
        '            $httpFactory = new \\\\Nyholm\\\\Psr7\\\\Factory\\\\Psr17Factory();',
        "        } elseif (class_exists('\\\\GuzzleHttp\\\\Psr7\\\\HttpFactory')) {",
        '            $httpFactory = new \\\\GuzzleHttp\\\\Psr7\\\\HttpFactory();',
        "        } elseif (class_exists('\\\\Slim\\\\Psr7\\\\Factory\\\\StreamFactory')) {",
        '            $httpFactory = new \\\\Slim\\\\Psr7\\\\Factory\\\\StreamFactory();',
        '        }',
        '',
        '        if ($httpFactory) {',
        '            \\\\Sentry\\\\init([',
        "                'dsn' => '__SENTRY_DSN__',",
        "                'send_default_pii' => true,",
        "                'traces_sample_rate' => 1.0,",
        "                'http_factory' => \$httpFactory,",
        '            ]);',
        '        } else {',
        "            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {",
        "                error_log('ShadowScan MU Sentry: no PSR-17 HTTP factory available.');",
        '            }',
        '        }',
        '    } catch (\\\\Throwable $e) {',
        "        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {",
        "            error_log('ShadowScan MU Sentry init failed: ' . \$e->getMessage());",
        '        }',
        '    }',
        '',
        '    register_shutdown_function(function () {',
        '        $error = error_get_last();',
        '        if ($error && in_array($error["type"], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {',
        '            $message = sprintf("Fatal error: %s in %s:%d", $error["message"], $error["file"], $error["line"]);',
        '            \\\\Sentry\\\\captureMessage($message, \\\\Sentry\\\\Severity::fatal());',
        '            \\\\Sentry\\\\flush(2);',
        '        }',
        '    });',
        '}',
    );

    $code = implode("\n", $lines);
    return str_replace('__SENTRY_DSN__', addslashes($dsn), $code);
}

function shadowscan_mu_helper_filename(): string {
    return 'shadowscan-sentry-mu-helper.php';
}

function shadowscan_mu_helper_path(): string {
    return WP_CONTENT_DIR . '/mu-plugins/' . shadowscan_mu_helper_filename();
}

function shadowscan_mu_helper_installed(): bool {
    return file_exists(shadowscan_mu_helper_path());
}

function shadowscan_install_mu_helper(): array {
    if (!shadowscan_remote_diagnostics_enabled()) {
        return array(
            'ok' => false,
            'error' => __('Remote diagnostics are not enabled.', 'shadowscan-security-link'),
        );
    }
    if (!shadowscan_sentry_telemetry_enabled()) {
        return array(
            'ok' => false,
            'error' => __('Sentry telemetry is disabled.', 'shadowscan-security-link'),
        );
    }

    if (!function_exists('WP_Filesystem')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }

    $fs = shadowscan_get_filesystem();
    if (!$fs) {
        return array(
            'ok' => false,
            'error' => __('Filesystem access is not available.', 'shadowscan-security-link'),
        );
    }

    $dir = WP_CONTENT_DIR . '/mu-plugins';
    if (!is_dir($dir)) {
        if (!function_exists('wp_mkdir_p') || !wp_mkdir_p($dir)) {
            if (!$fs->mkdir($dir)) {
                return array(
                    'ok' => false,
                    'error' => __('Unable to create mu-plugins directory.', 'shadowscan-security-link'),
                );
            }
        }
    }

    $path = shadowscan_mu_helper_path();
    $contents = shadowscan_build_mu_helper_contents();
    if ($contents === '') {
        return array(
            'ok' => false,
            'error' => __('MU helper could not be prepared because Sentry telemetry is disabled.', 'shadowscan-security-link'),
        );
    }
    $written = $fs->put_contents($path, $contents, FS_CHMOD_FILE);
    if (!$written) {
        return array(
            'ok' => false,
            'error' => __('Failed to write MU helper file.', 'shadowscan-security-link'),
        );
    }

    return array(
        'ok' => true,
        'payload' => array(
            'path' => $path,
            'installed' => true,
        ),
    );
}

function shadowscan_remove_mu_helper(): array {
    if (!function_exists('WP_Filesystem')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }

    $fs = shadowscan_get_filesystem();
    if (!$fs) {
        return array(
            'ok' => false,
            'error' => __('Filesystem access is not available.', 'shadowscan-security-link'),
        );
    }

    $path = shadowscan_mu_helper_path();
    $was_present = $fs->exists($path);
    if ($was_present) {
        if (!$fs->delete($path)) {
            return array(
                'ok' => false,
                'error' => __('Failed to remove MU helper file.', 'shadowscan-security-link'),
            );
        }
    }

    return array(
        'ok' => true,
        'payload' => array(
            'path' => $path,
            'installed' => false,
            'was_present' => $was_present,
        ),
    );
}

function shadowscan_redact_log_entry(string $entry): string {
    $redacted = $entry;
    $redacted = preg_replace('/Bearer\\s+[A-Za-z0-9\\-_.]+/', 'Bearer [redacted]', $redacted);
    $redacted = preg_replace('/pairing_code=([A-Za-z0-9\\-_.]+)/', 'pairing_code=[redacted]', $redacted);
    $redacted = preg_replace('/site_token[^A-Za-z0-9]*[A-Za-z0-9\\-_.]+/i', 'site_token=[redacted]', $redacted);
    $redacted = preg_replace('/api_key[^A-Za-z0-9]*[A-Za-z0-9\\-_.]+/i', 'api_key=[redacted]', $redacted);
    $redacted = preg_replace('/\"authorization\"\\s*:\\s*\"Bearer\\s+[A-Za-z0-9\\-_.]+\"/i', '"authorization":"Bearer [redacted]"', $redacted);
    $redacted = preg_replace('/\"site_token\"\\s*:\\s*\"[A-Za-z0-9\\-_.]+\"/i', '"site_token":"[redacted]"', $redacted);
    return $redacted;
}

function shadowscan_connection_guidance(array $last_error): array {
    $message = strtolower((string) ($last_error['message'] ?? ''));
    $status = isset($last_error['status']) ? (int) $last_error['status'] : 0;
    $context = (string) ($last_error['context'] ?? '');
    $title = __('Connection issue detected', 'shadowscan-security-link');
    $summary = __('We could not reach the ShadowScan portal.', 'shadowscan-security-link');
    $causes = array();

    if (strpos($message, 'curl error 6') !== false || strpos($message, 'could not resolve host') !== false) {
        $summary = __('Portal unreachable (DNS).', 'shadowscan-security-link');
        $causes[] = __('DNS or outbound network blocked by host.', 'shadowscan-security-link');
    } elseif (strpos($message, 'curl error 60') !== false || strpos($message, 'ssl') !== false || strpos($message, 'certificate') !== false) {
        $summary = __('TLS/Certificate issue while contacting the portal.', 'shadowscan-security-link');
        $causes[] = __('Hosting environment blocking or intercepting TLS.', 'shadowscan-security-link');
    } elseif (strpos($message, 'timed out') !== false || strpos($message, 'timeout') !== false) {
        $summary = __('Connection timed out.', 'shadowscan-security-link');
        $causes[] = __('Outbound requests blocked or slow network path.', 'shadowscan-security-link');
    } elseif (in_array($status, array(400, 404, 422), true) && $context === 'pair-complete') {
        $summary = __('Invalid pairing code.', 'shadowscan-security-link');
        $causes[] = __('The pairing code may be expired or mistyped.', 'shadowscan-security-link');
    } elseif (in_array($status, array(401, 403), true)) {
        $summary = __('Authentication rejected by the portal.', 'shadowscan-security-link');
        $causes[] = __('Clock skew or token mismatch.', 'shadowscan-security-link');
    } elseif ($context === 'pair-complete') {
        $summary = __('Unable to complete pairing.', 'shadowscan-security-link');
        $causes[] = __('The portal did not accept the pairing request.', 'shadowscan-security-link');
    }

    if (empty($causes)) {
        $causes[] = __('Portal unreachable or blocked by the host.', 'shadowscan-security-link');
    }

    return array(
        'title' => $title,
        'summary' => $summary,
        'causes' => $causes,
    );
}

function shadowscan_get_reference_admin_user_id(): int {
    $users = get_users(array(
        'role' => 'administrator',
        'fields' => array('ID'),
        'number' => 1,
    ));
    if (!empty($users) && isset($users[0]->ID)) {
        return (int) $users[0]->ID;
    }
    return 0;
}

function shadowscan_get_capabilities_snapshot(bool $force = false): array {
    $snapshot = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_CAPABILITIES_SNAPSHOT, array());
    $updated_at = (int) ($snapshot['updated_at'] ?? 0);
    if (!$force && $updated_at > 0 && (time() - $updated_at) < DAY_IN_SECONDS) {
        return $snapshot;
    }

    $caps = array('install_plugins', 'activate_plugins', 'update_plugins');
    $values = array();
    $source = 'unknown';

    if (is_user_logged_in()) {
        $source = 'current_user';
        foreach ($caps as $cap) {
            $values[$cap] = current_user_can($cap);
        }
    } else {
        $admin_id = shadowscan_get_reference_admin_user_id();
        if ($admin_id > 0) {
            $source = 'admin_user';
            foreach ($caps as $cap) {
                $values[$cap] = user_can($admin_id, $cap);
            }
        } else {
            foreach ($caps as $cap) {
                $values[$cap] = null;
            }
        }
    }

    $snapshot = array(
        'updated_at' => time(),
        'source' => $source,
        'values' => $values,
    );
    ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_CAPABILITIES_SNAPSHOT, $snapshot);

    return $snapshot;
}

function shadowscan_format_tristate($value): string {
    if ($value === true) {
        return esc_html__('Yes', 'shadowscan-security-link');
    }
    if ($value === false) {
        return esc_html__('No', 'shadowscan-security-link');
    }
    return esc_html__('Unknown', 'shadowscan-security-link');
}

function shadowscan_get_connector_report_data(): array {
    $connected = shadowscan_is_connected();
    $last_ping = (int) get_option(SHADOWSCAN_OPTION_LAST_PING, 0);
    $pending_heartbeat = get_option(SHADOWSCAN_OPTION_PENDING_HEARTBEAT, null);
    $pending_results = get_option(SHADOWSCAN_OPTION_PENDING_RESULTS, array());
    $last_error = get_option(SHADOWSCAN_OPTION_LAST_ERROR, array());
    $last_contact_attempt = get_option(SHADOWSCAN_OPTION_LAST_CONTACT_ATTEMPT, array());
    $last_contact_error = get_option(SHADOWSCAN_OPTION_LAST_CONTACT_ERROR, array());
    $third_party_plugins = shadowscan_detect_third_party_security_plugins();
    $next_heartbeat = wp_next_scheduled('shadowscan_heartbeat');
    $next_poll = wp_next_scheduled('shadowscan_poll_commands');
    $now = time();
    if ($next_heartbeat && $next_heartbeat < ($now - 60)) {
        $next_heartbeat = 0;
    }
    if ($next_poll && $next_poll < ($now - 60)) {
        $next_poll = 0;
    }
    $guard_status = shadowscan_guard_manager()->get_guard_status();
    $bruteforce_stats = shadowscan_guard_manager()->get_bruteforce_stats(false);
    $emergency_status = shadowscan_guard_manager()->get_emergency_status();
    $enumeration_state = shadowscan_guard_manager()->get_enumeration_state();
    $headers_state = shadowscan_guard_manager()->get_headers_state();
    $mfa_enforced = !empty($guard_status['enforce_mfa']);
    $enumeration_strict = !empty($enumeration_state['disable_author_archives']);
    $csp_report_only = !empty($headers_state['csp_report_only']);
    $x_frame_enabled = empty($headers_state['x_frame_disabled']);
    $event_queue_count = ShadowScan_Event_Buffer::count();
    $subscription_state = ShadowScan_Subscription_State::get_current_state();
    $subscription_status = strtoupper($subscription_state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL);
    $subscription_updated = !empty($subscription_state['updated_at'])
        ? gmdate('Y-m-d H:i', (int) $subscription_state['updated_at'])
        : esc_html__('Unknown', 'shadowscan-security-link');
    $subscription_plan = $subscription_state['plan'] ?? '';
    $subscription_override = !empty($subscription_state['override']);
    $subscription_test_mode = !empty($subscription_state['test_mode']);
    $subscription_emulated = !empty($subscription_state['emulated']);
    $subscription_emulated_status = $subscription_state['emulated_status'] ?? '';
    $subscription_source = isset($subscription_state['source'])
        ? sanitize_text_field((string) $subscription_state['source'])
        : 'unknown';
    if ($subscription_source === '') {
        $subscription_source = 'unknown';
    }
    $subscription_emulated_by = $subscription_state['emulated_by_admin_id'] ?? '';
    $subscription_suspended_since = !empty($subscription_state['suspended_since'])
        ? gmdate('Y-m-d H:i', (int) $subscription_state['suspended_since'])
        : esc_html__('Not set', 'shadowscan-security-link');
    $protection_state = ShadowScan_Subscription_State::get_protection_state($subscription_state);
    $protection_state_label = shadowscan_format_protection_state_label($protection_state);
    $dev_env = function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env();
    $override_set_by = !empty($subscription_state['override_set_by']) ? (int) $subscription_state['override_set_by'] : 0;
    $endpoint_health = shadowscan_get_endpoint_health_snapshot();
    $build_endpoint_status = static function (string $key, array $health): array {
        $entry = isset($health[$key]) && is_array($health[$key]) ? $health[$key] : array();
        $last_success_at = (int) ($entry['last_success_at'] ?? 0);
        $last_error_at = (int) ($entry['last_error_at'] ?? 0);
        $last_error_message = isset($entry['last_error_message']) ? (string) $entry['last_error_message'] : '';
        $reachable = $last_success_at > 0 && $last_success_at >= $last_error_at;
        return array(
            'reachable' => $reachable,
            'last_success_at' => $last_success_at,
            'last_error_at' => $last_error_at,
            'last_error_message' => $last_error_message,
        );
    };
    $heartbeat_health = $build_endpoint_status('heartbeat', $endpoint_health);
    $site_policy_health = $build_endpoint_status('site_policy', $endpoint_health);
    $commands_health = $build_endpoint_status('commands', $endpoint_health);
    $events_health = $build_endpoint_status('events', $endpoint_health);
    $portal_reachable = $heartbeat_health['reachable'] && $site_policy_health['reachable'];
    if (!$portal_reachable && empty($site_policy_health['last_error_message']) && !empty($last_contact_error['message'])) {
        $site_policy_health['last_error_message'] = (string) $last_contact_error['message'];
    }
    $command_backoff_until = (int) get_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_UNTIL, 0);
    $command_backoff_level = (int) get_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_LEVEL, 0);
    $command_failures = (int) get_option(SHADOWSCAN_OPTION_COMMAND_BACKOFF_COUNT, 0);
    $command_last_http_status = (int) get_option(SHADOWSCAN_OPTION_COMMAND_LAST_HTTP_STATUS, 0);
    $command_timeout_count = (int) get_option(SHADOWSCAN_OPTION_COMMAND_TIMEOUT_COUNT, 0);
    $scheduler_last_repair_at = (int) get_option(SHADOWSCAN_OPTION_SCHEDULER_LAST_REPAIR_AT, 0);
    $scheduler_repairs_count = (int) get_option(SHADOWSCAN_OPTION_SCHEDULER_REPAIRS_COUNT, 0);
    $sync_state = function_exists('shadowscan_get_sync_state') ? shadowscan_get_sync_state() : array();
    $last_apply_result = isset($sync_state['last_apply_result']) && is_array($sync_state['last_apply_result'])
        ? $sync_state['last_apply_result']
        : array();
    $last_policy_fetch_at = (int) ($sync_state['last_policy_fetch_at'] ?? 0);
    $last_policy_fetch_result = (string) ($sync_state['last_policy_fetch_result'] ?? '');
    $last_policy_fetch_error_code = (string) ($sync_state['last_policy_fetch_error_code'] ?? '');
    $last_policy_fetch_error_message = (string) ($sync_state['last_policy_fetch_error_message'] ?? '');
    $last_policy_apply_at = (int) ($last_apply_result['applied_at'] ?? 0);
    $last_policy_apply_result = (string) ($last_apply_result['status'] ?? '');
    $last_policy_apply_reason = (string) ($last_apply_result['reason'] ?? ($sync_state['last_policy_apply_reason'] ?? ''));
    $last_policy_error_code = (string) ($last_apply_result['error_code'] ?? '');
    $last_policy_error_message = (string) ($last_apply_result['error_message'] ?? '');
    $last_applied_policy_version = (string) ($sync_state['last_applied_policy_version'] ?? '');
    $last_seen_policy_version = (string) ($sync_state['last_seen_policy_version'] ?? '');
    $core_minor_effective = isset($sync_state['core_minor_effective']) && is_array($sync_state['core_minor_effective'])
        ? $sync_state['core_minor_effective']
        : array();
    $plugin_allowlist_effective = isset($sync_state['plugin_allowlist_effective']) && is_array($sync_state['plugin_allowlist_effective'])
        ? $sync_state['plugin_allowlist_effective']
        : array();
    $policy_version_migrated_from_legacy = !empty($sync_state['policy_version_migrated_from_legacy']);
    $policy_version_migrated_at = (int) ($sync_state['policy_version_migrated_at'] ?? 0);
    $policy_version_migrated_from = (string) ($sync_state['policy_version_migrated_from'] ?? '');
    $policy_version_migrated_to = (string) ($sync_state['policy_version_migrated_to'] ?? '');
    $auth_state = function_exists('shadowscan_sync_get_auth_invalid_state')
        ? shadowscan_sync_get_auth_invalid_state()
        : array();
    $auth_invalid = function_exists('shadowscan_sync_is_auth_invalid') && shadowscan_sync_is_auth_invalid();
    $auth_invalid_since = (int) ($auth_state['auth_invalid_first_at'] ?? 0);
    $last_auth_http_status = (int) ($auth_state['last_auth_http_status'] ?? 0);
    $last_recovery_attempt_at = (int) ($auth_state['last_recovery_attempt_at'] ?? 0);
    $recovery_attempt_count = (int) ($auth_state['recovery_attempt_count'] ?? 0);
    $auth_recovery_cooldown_active = shadowscan_auth_recovery_cooldown_active();
    $cap_snapshot = shadowscan_get_capabilities_snapshot();
    $cap_values = $cap_snapshot['values'] ?? array();
    $event_flush_stats = class_exists('ShadowScan_Storage') ? ShadowScan_Storage::get_json('event_flush_stats', array()) : array();
    if (!is_array($event_flush_stats)) {
        $event_flush_stats = array();
    }
    $event_last_flush_at = (int) ($event_flush_stats['last_success_at'] ?? 0);
    $event_queue_after_flush = (int) ($event_flush_stats['last_queue_after'] ?? 0);
    $event_queue_growth_since_flush = max(0, $event_queue_count - max(0, $event_queue_after_flush));

    return array(
        'connected' => $connected,
        'site_url' => home_url(),
        'last_ping' => $last_ping,
        'next_heartbeat' => $next_heartbeat,
        'next_poll' => $next_poll,
        'pending_heartbeat' => $pending_heartbeat,
        'pending_results_count' => is_array($pending_results) ? count($pending_results) : 0,
        'wp_version' => get_bloginfo('version'),
        'php_version' => PHP_VERSION,
        'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
        'guard_status' => $guard_status,
        'bruteforce_stats' => $bruteforce_stats,
        'emergency_status' => $emergency_status,
        'enumeration_strict' => $enumeration_strict,
        'mfa_enforced' => $mfa_enforced,
        'csp_report_only' => $csp_report_only,
        'x_frame_enabled' => $x_frame_enabled,
        'event_queue_count' => $event_queue_count,
        'third_party_plugins' => $third_party_plugins,
        'capabilities' => array(
            'can_install_plugins' => $cap_values['install_plugins'] ?? null,
            'can_activate_plugins' => $cap_values['activate_plugins'] ?? null,
            'can_update_plugins' => $cap_values['update_plugins'] ?? null,
            'source' => $cap_snapshot['source'] ?? 'unknown',
            'updated_at' => (int) ($cap_snapshot['updated_at'] ?? 0),
        ),
        'subscription_state' => $subscription_state,
        'subscription_status' => $subscription_status,
        'subscription_plan' => $subscription_plan,
        'subscription_updated' => $subscription_updated,
        'subscription_override' => $subscription_override,
        'subscription_test_mode' => $subscription_test_mode,
        'subscription_emulated' => $subscription_emulated,
        'subscription_emulated_status' => $subscription_emulated_status,
        'subscription_emulated_by' => $subscription_emulated_by,
        'subscription_source' => $subscription_source,
        'subscription_suspended_since' => $subscription_suspended_since,
        'protection_state_label' => $protection_state_label,
        'dev_env' => $dev_env,
        'override_set_by' => $override_set_by,
        'portal_reachable' => $portal_reachable,
        'core_policy_reachable' => (bool) $site_policy_health['reachable'],
        'heartbeat_reachable' => (bool) $heartbeat_health['reachable'],
        'commands_reachable' => (bool) $commands_health['reachable'],
        'events_reachable' => (bool) $events_health['reachable'],
        'core_policy_last_success_at' => (int) $site_policy_health['last_success_at'],
        'heartbeat_last_success_at' => (int) $heartbeat_health['last_success_at'],
        'commands_last_success_at' => (int) $commands_health['last_success_at'],
        'events_last_success_at' => (int) $events_health['last_success_at'],
        'last_error' => $last_error,
        'last_contact_attempt' => $last_contact_attempt,
        'last_contact_error' => $last_contact_error,
        'command_backoff_until' => $command_backoff_until,
        'command_backoff_level' => $command_backoff_level,
        'command_failures' => $command_failures,
        'command_last_http_status' => $command_last_http_status,
        'command_timeout_count' => $command_timeout_count,
        'scheduler_last_repair_at' => $scheduler_last_repair_at,
        'scheduler_repairs_count' => $scheduler_repairs_count,
        'auth_invalid' => $auth_invalid,
        'auth_invalid_since' => $auth_invalid_since,
        'last_auth_http_status' => $last_auth_http_status,
        'last_recovery_attempt_at' => $last_recovery_attempt_at,
        'recovery_attempt_count' => $recovery_attempt_count,
        'auth_recovery_cooldown_active' => $auth_recovery_cooldown_active,
        'last_policy_fetch_at' => $last_policy_fetch_at,
        'last_policy_fetch_result' => $last_policy_fetch_result,
        'last_policy_fetch_error_code' => $last_policy_fetch_error_code,
        'last_policy_fetch_error_message' => $last_policy_fetch_error_message,
        'last_policy_apply_at' => $last_policy_apply_at,
        'last_policy_apply_result' => $last_policy_apply_result,
        'last_policy_apply_reason' => $last_policy_apply_reason,
        'last_policy_error_code' => $last_policy_error_code,
        'last_policy_error_message' => $last_policy_error_message,
        'last_applied_policy_version' => $last_applied_policy_version,
        'last_seen_policy_version' => $last_seen_policy_version,
        'core_minor_effective' => $core_minor_effective,
        'plugin_allowlist_effective' => $plugin_allowlist_effective,
        'policy_version_migrated_from_legacy' => $policy_version_migrated_from_legacy,
        'policy_version_migrated_at' => $policy_version_migrated_at,
        'policy_version_migrated_from' => $policy_version_migrated_from,
        'policy_version_migrated_to' => $policy_version_migrated_to,
        'event_last_flush_at' => $event_last_flush_at,
        'event_queue_growth_since_flush' => $event_queue_growth_since_flush,
    );
}

function shadowscan_render_connector_report(array $data): string {
    $capabilities = $data['capabilities'] ?? array();
    $core_minor_effective = isset($data['core_minor_effective']) && is_array($data['core_minor_effective'])
        ? $data['core_minor_effective']
        : array();
    $plugin_allowlist_effective = isset($data['plugin_allowlist_effective']) && is_array($data['plugin_allowlist_effective'])
        ? $data['plugin_allowlist_effective']
        : array();
    $portal_line = $data['portal_reachable'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link');
    if (!$data['portal_reachable']) {
        $portal_error = !empty($data['last_contact_error']['message']) ? (string) $data['last_contact_error']['message'] : '';
        if ($portal_error !== '') {
            $portal_line .= ' (' . $portal_error . ')';
        }
    }

    $report_lines = array(
        esc_html__('ShadowScan Connector Report', 'shadowscan-security-link'),
        /* translators: %s: site URL. */
        sprintf(esc_html__('Site: %s', 'shadowscan-security-link'), $data['site_url']),
        /* translators: %s: connection status. */
        sprintf(esc_html__('Connected: %s', 'shadowscan-security-link'), $data['connected'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        sprintf(
            /* translators: %s: last heartbeat time. */
            esc_html__('Last heartbeat: %s', 'shadowscan-security-link'),
            $data['last_ping'] ? gmdate('Y-m-d H:i', $data['last_ping']) . ' UTC' : esc_html__('Never', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: next heartbeat time. */
            esc_html__('Next heartbeat: %s', 'shadowscan-security-link'),
            $data['next_heartbeat'] ? gmdate('Y-m-d H:i', $data['next_heartbeat']) . ' UTC' : esc_html__('Not scheduled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: next poll time. */
            esc_html__('Next poll: %s', 'shadowscan-security-link'),
            $data['next_poll'] ? gmdate('Y-m-d H:i', $data['next_poll']) . ' UTC' : esc_html__('Not scheduled', 'shadowscan-security-link')
        ),
        /* translators: %s: pending heartbeat status. */
        sprintf(esc_html__('Pending heartbeat: %s', 'shadowscan-security-link'), $data['pending_heartbeat'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        /* translators: %d: pending results count. */
        sprintf(esc_html__('Pending results: %d', 'shadowscan-security-link'), (int) $data['pending_results_count']),
        /* translators: %s: WordPress version. */
        sprintf(esc_html__('WP version: %s', 'shadowscan-security-link'), $data['wp_version']),
        /* translators: %s: PHP version. */
        sprintf(esc_html__('PHP version: %s', 'shadowscan-security-link'), $data['php_version']),
        /* translators: %s: plugin version. */
        sprintf(esc_html__('Connector: %s', 'shadowscan-security-link'), $data['plugin_version']),
        /* translators: %s: guard profile name. */
        sprintf(esc_html__('Guard profile: %s', 'shadowscan-security-link'), $data['guard_status']['profile'] ?? 'baseline'),
        sprintf(
            /* translators: %s: last integrity scan time. */
            esc_html__('Last integrity scan: %s', 'shadowscan-security-link'),
            !empty($data['guard_status']['last_integrity_scan']) ? gmdate('Y-m-d H:i', (int) $data['guard_status']['last_integrity_scan']) : esc_html__('Never', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last auth anomaly time. */
            esc_html__('Last auth anomaly: %s', 'shadowscan-security-link'),
            !empty($data['guard_status']['last_auth_anomaly']) ? gmdate('Y-m-d H:i', (int) $data['guard_status']['last_auth_anomaly']) : esc_html__('Never', 'shadowscan-security-link')
        ),
        /* translators: %d: queued events count. */
        sprintf(esc_html__('Events queued: %d', 'shadowscan-security-link'), (int) $data['event_queue_count']),
        sprintf(
            /* translators: %s: safe mode status. */
            esc_html__('Login abuse safe mode: %s', 'shadowscan-security-link'),
            !empty($data['bruteforce_stats']['safe_mode']) ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %d: login failures count. */
            esc_html__('Login failures (last window): %d', 'shadowscan-security-link'),
            (int) ($data['bruteforce_stats']['failed_last_window'] ?? 0)
        ),
        sprintf(
            /* translators: %s: emergency mode status. */
            esc_html__('Emergency mode: %s', 'shadowscan-security-link'),
            !empty($data['emergency_status']['enabled']) ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: MFA enforcement status. */
            esc_html__('MFA enforcement: %s', 'shadowscan-security-link'),
            $data['mfa_enforced'] ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: enumeration strict status. */
            esc_html__('Enumeration strict: %s', 'shadowscan-security-link'),
            $data['enumeration_strict'] ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: X-Frame-Options status. */
            esc_html__('X-Frame-Options: %s', 'shadowscan-security-link'),
            $data['x_frame_enabled'] ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: CSP report-only status. */
            esc_html__('CSP report-only: %s', 'shadowscan-security-link'),
            $data['csp_report_only'] ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: safe mode timestamp or status. */
            esc_html__('Safe mode: %s', 'shadowscan-security-link'),
            !empty($data['guard_status']['safe_mode_until']) && (int) $data['guard_status']['safe_mode_until'] > time()
                ? gmdate('Y-m-d H:i', (int) $data['guard_status']['safe_mode_until'])
                : esc_html__('Not active', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: detected third-party tool list. */
            esc_html__('Third-party security tools detected: %s', 'shadowscan-security-link'),
            $data['third_party_plugins'] ? esc_html(implode(', ', $data['third_party_plugins'])) : esc_html__('None', 'shadowscan-security-link')
        ),
        /* translators: %s: plugin install capability. */
        sprintf(esc_html__('Can install plugins: %s', 'shadowscan-security-link'), shadowscan_format_tristate($capabilities['can_install_plugins'] ?? null)),
        /* translators: %s: plugin activation capability. */
        sprintf(esc_html__('Can activate plugins: %s', 'shadowscan-security-link'), shadowscan_format_tristate($capabilities['can_activate_plugins'] ?? null)),
        /* translators: %s: remote diagnostics status. */
        sprintf(esc_html__('Remote diagnostics: %s', 'shadowscan-security-link'), shadowscan_remote_diagnostics_enabled() ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link')),
        /* translators: %s: subscription status. */
        sprintf(esc_html__('Subscription status: %s', 'shadowscan-security-link'), $data['subscription_status']),
        /* translators: %s: subscription plan name. */
        sprintf(esc_html__('Subscription plan: %s', 'shadowscan-security-link'), $data['subscription_plan'] !== '' ? esc_html($data['subscription_plan']) : esc_html__('Unknown', 'shadowscan-security-link')),
        /* translators: %s: subscription updated timestamp. */
        sprintf(esc_html__('Subscription updated: %s', 'shadowscan-security-link'), $data['subscription_updated']),
        sprintf(
            /* translators: %s: subscription grace end timestamp. */
            esc_html__('Subscription grace end: %s', 'shadowscan-security-link'),
            !empty($data['subscription_state']['grace_end']) ? gmdate('Y-m-d H:i', (int) $data['subscription_state']['grace_end']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        /* translators: %s: portal reachable status. */
        sprintf(esc_html__('Portal reachable: %s', 'shadowscan-security-link'), esc_html($portal_line)),
        sprintf(esc_html__('Core policy reachable: %s', 'shadowscan-security-link'), !empty($data['core_policy_reachable']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        sprintf(esc_html__('Heartbeat reachable: %s', 'shadowscan-security-link'), !empty($data['heartbeat_reachable']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        sprintf(esc_html__('Commands reachable: %s', 'shadowscan-security-link'), !empty($data['commands_reachable']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        sprintf(esc_html__('Events reachable: %s', 'shadowscan-security-link'), !empty($data['events_reachable']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        /* translators: %s: command backoff timestamp. */
        sprintf(esc_html__('Command poll backoff until: %s', 'shadowscan-security-link'), $data['command_backoff_until'] ? gmdate('Y-m-d H:i', $data['command_backoff_until']) : esc_html__('Not set', 'shadowscan-security-link')),
        /* translators: %d: command poll failure count. */
        sprintf(esc_html__('Command poll failures: %d', 'shadowscan-security-link'), (int) $data['command_failures']),
        sprintf(
            /* translators: %s: last command HTTP status code. */
            esc_html__('Last command HTTP status: %s', 'shadowscan-security-link'),
            !empty($data['command_last_http_status']) ? (string) (int) $data['command_last_http_status'] : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %d: command timeout count. */
            esc_html__('Command poll timeouts: %d', 'shadowscan-security-link'),
            (int) ($data['command_timeout_count'] ?? 0)
        ),
        sprintf(
            /* translators: %s: command backoff remaining seconds or status. */
            esc_html__('Command backoff remaining: %s', 'shadowscan-security-link'),
            !empty($data['command_backoff_until']) && (int) $data['command_backoff_until'] > time()
                ? sprintf(
                    /* translators: %d: seconds remaining in backoff. */
                    esc_html__('%d seconds', 'shadowscan-security-link'),
                    (int) $data['command_backoff_until'] - time()
                )
                : esc_html__('Not active', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: scheduler last repair timestamp. */
            esc_html__('Scheduler last repair: %s', 'shadowscan-security-link'),
            !empty($data['scheduler_last_repair_at']) ? gmdate('Y-m-d H:i', (int) $data['scheduler_last_repair_at']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %d: scheduler repair count. */
            esc_html__('Scheduler repairs: %d', 'shadowscan-security-link'),
            (int) ($data['scheduler_repairs_count'] ?? 0)
        ),
        sprintf(
            /* translators: %s: last policy fetch timestamp. */
            esc_html__('Last policy fetch: %s', 'shadowscan-security-link'),
            !empty($data['last_policy_fetch_at']) ? gmdate('Y-m-d H:i', (int) $data['last_policy_fetch_at']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last policy fetch result. */
            esc_html__('Last policy fetch result: %s', 'shadowscan-security-link'),
            $data['last_policy_fetch_result'] !== '' ? esc_html($data['last_policy_fetch_result']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last policy apply timestamp. */
            esc_html__('Last policy apply: %s', 'shadowscan-security-link'),
            !empty($data['last_policy_apply_at']) ? gmdate('Y-m-d H:i', (int) $data['last_policy_apply_at']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last policy apply result. */
            esc_html__('Last policy apply result: %s', 'shadowscan-security-link'),
            $data['last_policy_apply_result'] !== '' ? esc_html($data['last_policy_apply_result']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last policy apply reason. */
            esc_html__('Last policy apply reason: %s', 'shadowscan-security-link'),
            $data['last_policy_apply_reason'] !== '' ? esc_html($data['last_policy_apply_reason']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: core minor gate status. */
            esc_html__('Core minor gate status: %s', 'shadowscan-security-link'),
            isset($core_minor_effective['gate_status']) ? esc_html((string) $core_minor_effective['gate_status']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: core minor gate reason. */
            esc_html__('Core minor gate reason: %s', 'shadowscan-security-link'),
            isset($core_minor_effective['gate_reason']) ? esc_html((string) $core_minor_effective['gate_reason']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: 1: mode, 2: pause status, 3: approval mode. */
            esc_html__('Core minor policy mode: %1$s | Global pause: %2$s | Approval: %3$s', 'shadowscan-security-link'),
            isset($core_minor_effective['mode']) ? esc_html((string) $core_minor_effective['mode']) : esc_html__('Not set', 'shadowscan-security-link'),
            !empty($core_minor_effective['global_pause']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'),
            isset($core_minor_effective['approval_required']) ? esc_html((string) $core_minor_effective['approval_required']) : esc_html__('none', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: 1: rollout percentage, 2: site percentage, 3: delay hours. */
            esc_html__('Core minor rollout: %1$d%% | Site bucket: %2$d | Delay: %3$dh', 'shadowscan-security-link'),
            (int) ($core_minor_effective['rollout_percent'] ?? 0),
            (int) ($core_minor_effective['site_percent'] ?? 0),
            (int) ($core_minor_effective['delay_hours'] ?? 0)
        ),
        sprintf(
            /* translators: %s: plugin allowlist gate status. */
            esc_html__('Plugin allowlist gate status: %s', 'shadowscan-security-link'),
            isset($plugin_allowlist_effective['gate_status']) ? esc_html((string) $plugin_allowlist_effective['gate_status']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: plugin allowlist gate reason. */
            esc_html__('Plugin allowlist gate reason: %s', 'shadowscan-security-link'),
            isset($plugin_allowlist_effective['gate_reason']) ? esc_html((string) $plugin_allowlist_effective['gate_reason']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: 1: mode, 2: pause status, 3: approval mode. */
            esc_html__('Plugin allowlist policy mode: %1$s | Global pause: %2$s | Approval: %3$s', 'shadowscan-security-link'),
            isset($plugin_allowlist_effective['mode']) ? esc_html((string) $plugin_allowlist_effective['mode']) : esc_html__('Not set', 'shadowscan-security-link'),
            !empty($plugin_allowlist_effective['global_pause']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'),
            isset($plugin_allowlist_effective['approval_required']) ? esc_html((string) $plugin_allowlist_effective['approval_required']) : esc_html__('none', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: 1: daily cap, 2: site slot, 3: delay hours. */
            esc_html__('Plugin allowlist rollout cap/day: %1$s | Site slot: %2$s | Delay: %3$dh', 'shadowscan-security-link'),
            isset($plugin_allowlist_effective['max_sites_per_day']) && $plugin_allowlist_effective['max_sites_per_day'] !== null
                ? (string) (int) $plugin_allowlist_effective['max_sites_per_day']
                : esc_html__('Not set', 'shadowscan-security-link'),
            isset($plugin_allowlist_effective['daily_slot']) && $plugin_allowlist_effective['daily_slot'] !== null
                ? (string) (int) $plugin_allowlist_effective['daily_slot']
                : esc_html__('Not set', 'shadowscan-security-link'),
            (int) ($plugin_allowlist_effective['delay_hours'] ?? 0)
        ),
        sprintf(
            /* translators: %s: last applied policy version. */
            esc_html__('Last applied policy version: %s', 'shadowscan-security-link'),
            $data['last_applied_policy_version'] !== '' ? esc_html($data['last_applied_policy_version']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last seen policy version. */
            esc_html__('Last seen policy version: %s', 'shadowscan-security-link'),
            $data['last_seen_policy_version'] !== '' ? esc_html($data['last_seen_policy_version']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: policy version migration status. */
            esc_html__('Policy version migrated from legacy: %s', 'shadowscan-security-link'),
            !empty($data['policy_version_migrated_from_legacy']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: policy version migration timestamp. */
            esc_html__('Policy version migrated at: %s', 'shadowscan-security-link'),
            !empty($data['policy_version_migrated_at']) ? gmdate('Y-m-d H:i', (int) $data['policy_version_migrated_at']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: 1: from policy version, 2: to policy version. */
            esc_html__('Policy version migration: %1$s -> %2$s', 'shadowscan-security-link'),
            $data['policy_version_migrated_from'] !== '' ? esc_html($data['policy_version_migrated_from']) : esc_html__('Not set', 'shadowscan-security-link'),
            $data['policy_version_migrated_to'] !== '' ? esc_html($data['policy_version_migrated_to']) : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: last successful queue flush timestamp. */
            esc_html__('Last successful queue flush: %s', 'shadowscan-security-link'),
            !empty($data['event_last_flush_at']) ? gmdate('Y-m-d H:i', (int) $data['event_last_flush_at']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %d: queued events added since last successful flush. */
            esc_html__('Queue growth since last flush: %d', 'shadowscan-security-link'),
            (int) ($data['event_queue_growth_since_flush'] ?? 0)
        ),
        /* translators: %s: auth invalid status. */
        sprintf(esc_html__('Auth invalid: %s', 'shadowscan-security-link'), !empty($data['auth_invalid']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        sprintf(
            /* translators: %s: auth invalid timestamp. */
            esc_html__('Auth invalid since: %s', 'shadowscan-security-link'),
            !empty($data['auth_invalid_since']) ? gmdate('Y-m-d H:i', (int) $data['auth_invalid_since']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: auth HTTP status code. */
            esc_html__('Last auth HTTP status: %s', 'shadowscan-security-link'),
            !empty($data['last_auth_http_status']) ? (string) (int) $data['last_auth_http_status'] : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %d: auth recovery attempts. */
            esc_html__('Recovery attempts: %d', 'shadowscan-security-link'),
            (int) ($data['recovery_attempt_count'] ?? 0)
        ),
        sprintf(
            /* translators: %s: last recovery attempt time. */
            esc_html__('Last recovery attempt: %s', 'shadowscan-security-link'),
            !empty($data['last_recovery_attempt_at']) ? gmdate('Y-m-d H:i', (int) $data['last_recovery_attempt_at']) . ' UTC' : esc_html__('Not set', 'shadowscan-security-link')
        ),
        sprintf(
            /* translators: %s: recovery cooldown status. */
            esc_html__('Auth recovery cooldown: %s', 'shadowscan-security-link'),
            !empty($data['auth_recovery_cooldown_active']) ? esc_html__('Active', 'shadowscan-security-link') : esc_html__('Inactive', 'shadowscan-security-link')
        ),
        /* translators: %s: subscription override status. */
        sprintf(esc_html__('Subscription override: %s', 'shadowscan-security-link'), $data['subscription_override'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        /* translators: %s: override user ID or empty. */
        sprintf(esc_html__('Override set by user ID: %s', 'shadowscan-security-link'), $data['override_set_by'] ? (string) $data['override_set_by'] : esc_html__('Not set', 'shadowscan-security-link')),
        /* translators: %s: subscription test mode status. */
        sprintf(esc_html__('Subscription test mode: %s', 'shadowscan-security-link'), $data['subscription_test_mode'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        /* translators: %s: subscription emulation status. */
        sprintf(esc_html__('Subscription emulated: %s', 'shadowscan-security-link'), $data['subscription_emulated'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
        /* translators: %s: emulated subscription status. */
        sprintf(esc_html__('Emulated status: %s', 'shadowscan-security-link'), $data['subscription_emulated_status'] !== '' ? esc_html($data['subscription_emulated_status']) : esc_html__('Not set', 'shadowscan-security-link')),
        /* translators: %s: admin ID who set emulation. */
        sprintf(esc_html__('Emulated by admin ID: %s', 'shadowscan-security-link'), $data['subscription_emulated_by'] !== '' ? esc_html($data['subscription_emulated_by']) : esc_html__('Not set', 'shadowscan-security-link')),
        sprintf(
            /* translators: %s: subscription source. */
            esc_html__('Subscription source: %s', 'shadowscan-security-link'),
            $data['subscription_source'] !== 'unknown' ? esc_html($data['subscription_source']) : esc_html__('Unknown', 'shadowscan-security-link')
        ),
        /* translators: %s: suspension timestamp. */
        sprintf(esc_html__('Suspended since: %s', 'shadowscan-security-link'), $data['subscription_suspended_since']),
        /* translators: %s: protection status label. */
        sprintf(esc_html__('Protection status: %s', 'shadowscan-security-link'), $data['protection_state_label']),
        /* translators: %s: dev environment flag. */
        sprintf(esc_html__('Dev environment: %s', 'shadowscan-security-link'), $data['dev_env'] ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link')),
    );

    if (is_array($data['last_contact_attempt']) && !empty($data['last_contact_attempt']['context'])) {
        $report_lines[] = sprintf(
            /* translators: 1: request context, 2: timestamp. */
            esc_html__('Last contact attempt (%1$s): %2$s', 'shadowscan-security-link'),
            $data['last_contact_attempt']['context'],
            $data['last_contact_attempt']['at'] ?? ''
        );
    }

    if (is_array($data['last_error']) && !empty($data['last_error']['message'])) {
        $report_lines[] = sprintf(
            /* translators: 1: request context, 2: error message. */
            esc_html__('Last error (%1$s): %2$s', 'shadowscan-security-link'),
            $data['last_error']['context'] ?? esc_html__('unknown', 'shadowscan-security-link'),
            $data['last_error']['message']
        );
    }
    if (($data['last_policy_fetch_result'] ?? '') === 'error') {
        $report_lines[] = sprintf(
            /* translators: 1: fetch error code, 2: fetch error message. */
            esc_html__('Last policy fetch error: code=%1$s message=%2$s', 'shadowscan-security-link'),
            !empty($data['last_policy_fetch_error_code']) ? $data['last_policy_fetch_error_code'] : esc_html__('unknown', 'shadowscan-security-link'),
            !empty($data['last_policy_fetch_error_message']) ? $data['last_policy_fetch_error_message'] : esc_html__('none', 'shadowscan-security-link')
        );
    }
    if (!empty($data['last_policy_error_code']) || !empty($data['last_policy_error_message'])) {
        $report_lines[] = sprintf(
            /* translators: 1: policy error code, 2: policy error message. */
            esc_html__('Last policy error: code=%1$s message=%2$s', 'shadowscan-security-link'),
            !empty($data['last_policy_error_code']) ? $data['last_policy_error_code'] : esc_html__('unknown', 'shadowscan-security-link'),
            !empty($data['last_policy_error_message']) ? $data['last_policy_error_message'] : esc_html__('none', 'shadowscan-security-link')
        );
    }

    $logs = get_option(SHADOWSCAN_OPTION_LOG, array());
    if (is_array($logs) && !empty($logs)) {
        $report_lines[] = esc_html__('Recent logs:', 'shadowscan-security-link');
        $recent_logs = array_slice($logs, -10);
        foreach ($recent_logs as $entry) {
            $report_lines[] = '  ' . shadowscan_redact_log_entry($entry ?? '');
        }
    }

    return implode("\n", $report_lines);
}

function shadowscan_build_connector_report(): string {
    return shadowscan_render_connector_report(shadowscan_get_connector_report_data());
}

function shadowscan_dashboard_notice_transient_key(int $user_id): string {
    return 'shadowscan_dashboard_notice_' . $user_id;
}

function shadowscan_dashboard_set_notice(string $message, string $type = 'success'): void {
    $user_id = get_current_user_id();
    if ($user_id <= 0) {
        return;
    }
    set_transient(
        shadowscan_dashboard_notice_transient_key($user_id),
        array(
            'message' => sanitize_text_field($message),
            'type' => $type === 'error' || $type === 'warning' ? $type : 'success',
        ),
        120
    );
}

function shadowscan_dashboard_get_notice(): array {
    $user_id = get_current_user_id();
    if ($user_id <= 0) {
        return array();
    }
    $key = shadowscan_dashboard_notice_transient_key($user_id);
    $notice = get_transient($key);
    if (is_array($notice) && !empty($notice['message'])) {
        delete_transient($key);
        return $notice;
    }
    return array();
}

function shadowscan_dashboard_redirect_url(string $default_tab = 'diagnostics'): string {
    $tab = isset($_REQUEST['return_tab']) ? sanitize_text_field(wp_unslash((string) $_REQUEST['return_tab'])) : '';
    if ($tab === '') {
        $tab = isset($_REQUEST['tab']) ? sanitize_text_field(wp_unslash((string) $_REQUEST['tab'])) : '';
    }
    if ($tab === '') {
        $tab = $default_tab;
    }
    return add_query_arg(array('page' => 'shadowscan', 'tab' => $tab), admin_url('admin.php'));
}

function shadowscan_build_owasp_evidence_payload(): array {
    $snapshot = shadowscan_owasp_refresh_status();
    $payload = array(
        'generated_at' => gmdate('c'),
        'site_id' => shadowscan_get_site_id(),
        'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
        'owasp_snapshot' => $snapshot,
        'enforcement_flags' => ShadowScan_Storage::get_json('feature_flags', array()),
    );
    if (class_exists('ShadowScan_Signal_Manager')) {
        ShadowScan_Signal_Manager::emit(
            'owasp_evidence_exported',
            'info',
            'Evidence exported',
            array(
                'category' => 'owasp',
                'owasp' => array(
                    'version' => '2025',
                    'id' => 'A01',
                    'name' => shadowscan_owasp_category_name('A01'),
                ),
            )
        );
    }
    return $payload;
}

function shadowscan_handle_download_logs_admin_post(): void {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'shadowscan-security-link'));
    }
    check_admin_referer('shadowscan-dashboard');

    $logs = get_option(SHADOWSCAN_OPTION_LOG, array());
    if (!is_array($logs)) {
        $logs = array();
    }
    $sanitized = array();
    foreach ($logs as $entry) {
        $sanitized[] = shadowscan_redact_log_entry((string) $entry);
    }
    $contents = implode("\n", $sanitized);

    if (headers_sent()) {
        shadowscan_dashboard_set_notice(
            __('Unable to download logs because the response has already started. Refresh the page and try again.', 'shadowscan-security-link'),
            'error'
        );
        wp_safe_redirect(shadowscan_dashboard_redirect_url('diagnostics'));
        exit;
    }

    nocache_headers();
    header('Content-Type: text/plain');
    header('Content-Disposition: attachment; filename="shadowscan-connector-log.txt"');
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo $contents;
    exit;
}
add_action('admin_post_shadowscan_download_logs', 'shadowscan_handle_download_logs_admin_post');

function shadowscan_handle_export_owasp_evidence_admin_post(): void {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'shadowscan-security-link'));
    }
    check_admin_referer('shadowscan-dashboard');

    $payload = shadowscan_build_owasp_evidence_payload();
    if (headers_sent()) {
        shadowscan_dashboard_set_notice(
            __('Unable to export evidence because the response has already started. Refresh the page and try again.', 'shadowscan-security-link'),
            'error'
        );
        wp_safe_redirect(shadowscan_dashboard_redirect_url('diagnostics'));
        exit;
    }

    nocache_headers();
    header('Content-Type: application/json; charset=utf-8');
    header('Content-Disposition: attachment; filename="shadowscan-protection-evidence.json"');
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo wp_json_encode($payload);
    exit;
}
add_action('admin_post_shadowscan_export_owasp_evidence', 'shadowscan_handle_export_owasp_evidence_admin_post');

function shadowscan_record_error(string $context, string $message, ?int $status = null): void {
    update_option(
        SHADOWSCAN_OPTION_LAST_ERROR,
        array(
            'context' => $context,
            'message' => $message,
            'status' => $status,
            'at' => gmdate('Y-m-d H:i:s'),
        ),
        false
    );
    shadowscan_log_rate_limited('error_' . $context, '[ShadowScan] ' . $context . ' error: ' . $message, 900);
}

function shadowscan_clear_error(?string $context = null): void {
    if ($context === null || $context === '') {
        delete_option(SHADOWSCAN_OPTION_LAST_ERROR);
        return;
    }

    $last_error = get_option(SHADOWSCAN_OPTION_LAST_ERROR, array());
    if (!is_array($last_error)) {
        return;
    }
    $last_context = isset($last_error['context']) ? (string) $last_error['context'] : '';
    if ($last_context === $context) {
        delete_option(SHADOWSCAN_OPTION_LAST_ERROR);
    }
}

function shadowscan_queue_pending_result(array $payload): void {
    $queue = get_option(SHADOWSCAN_OPTION_PENDING_RESULTS, array());
    if (!is_array($queue)) {
        $queue = array();
    }
    $queue[] = $payload;
    if (count($queue) > 20) {
        $queue = array_slice($queue, -20);
    }
    update_option(SHADOWSCAN_OPTION_PENDING_RESULTS, $queue, false);
}

function shadowscan_flush_pending_results(): void {
    if (shadowscan_should_pause_remote_calls('commands-result')) {
        shadowscan_log_rate_limited(
            'pending_results_auth_cooldown',
            '[ShadowScan] Pending command results delivery paused during auth recovery cooldown.',
            900
        );
        return;
    }
    $queue = get_option(SHADOWSCAN_OPTION_PENDING_RESULTS, array());
    if (!is_array($queue) || empty($queue)) {
        return;
    }
    $remaining = array();
    foreach ($queue as $payload) {
        $response = shadowscan_send_api_request('commands-result', $payload, 'POST', true);
        if (is_wp_error($response)) {
            $remaining[] = $payload;
        }
    }
    update_option(SHADOWSCAN_OPTION_PENDING_RESULTS, $remaining, false);
}

add_action('admin_enqueue_scripts', function (string $hook_suffix) {
    if ($hook_suffix !== 'toplevel_page_shadowscan') {
        return;
    }
    wp_enqueue_style(
        'shadowscan-admin',
        plugin_dir_url(__FILE__) . 'assets/css/admin.css',
        array(),
        SHADOWSCAN_PLUGIN_VERSION
    );
    wp_enqueue_script(
        'shadowscan-admin',
        plugin_dir_url(__FILE__) . 'assets/js/admin.js',
        array('jquery'),
        SHADOWSCAN_PLUGIN_VERSION,
        true
    );
    wp_localize_script('shadowscan-admin', 'shadowscanAdmin', array(
        'copiedLabel'         => __('Copied!', 'shadowscan-security-link'),
        'isDevEnv'            => shadowscan_is_dev_env(),
        'coverageFocusAnchor' => (string) get_option(SHADOWSCAN_OPTION_UI_FOCUS_ANCHOR, ''),
    ));
});

function shadowscan_render_dashboard(): void {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'shadowscan-security-link'));
    }
    static $rendered = false;
    if ($rendered) {
        return;
    }
    $rendered = true;

    update_option(SHADOWSCAN_OPTION_CAN_INSTALL, current_user_can('install_plugins'), false);

    $notice = '';
    $notice_type = 'success';

    $pairing_code = '';
    $connected = shadowscan_is_connected();
    $request_method = isset($_SERVER['REQUEST_METHOD'])
        ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD']))
        : '';
    $subscription_state = ShadowScan_Subscription_State::get_current_state();
    $protection_state = ShadowScan_Subscription_State::get_protection_state($subscription_state);
    $policy_state = class_exists('ShadowScan_Policy_State') ? ShadowScan_Policy_State::get_current_policy() : array();
    $support_debug = class_exists('ShadowScan_Policy_State') ? ShadowScan_Policy_State::get_support_debug() : array();

    $action = '';
    if ($request_method === 'POST' && isset($_POST['shadowscan_action'])) {
        check_admin_referer('shadowscan-dashboard');
        $action = sanitize_text_field(wp_unslash($_POST['shadowscan_action']));
    }

    $is_dev_env = shadowscan_is_dev_env();
    if (in_array($action, array('subscription_override_set', 'subscription_override_clear'), true) && !$is_dev_env) {
        $action = '';
    }

    if ($action !== '') {
        if ($request_method !== 'POST' && shadowscan_guard_manager()->is_flag_enabled('owasp_enforce_sensitive_actions', false)) {
            status_header(403);
            $notice = esc_html__('Sensitive actions require a verified form submission.', 'shadowscan-security-link');
            $notice_type = 'warning';
            $action = '';
        }
    }

    if ($action !== '') {
        $status = $subscription_state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL;
        $deny_action = function (string $message) use (&$notice, &$notice_type, $status) {
            status_header(402);
            $notice = $message;
            $notice_type = in_array($status, array(ShadowScan_Subscription_State::STATUS_SUSPENDED, ShadowScan_Subscription_State::STATUS_CANCELLED), true)
                ? 'error'
                : 'warning';
        };

        $gated_actions = array(
            'save_mfa_settings' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'save_autoupdate_settings' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'save_password_policy' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'save_geo_policy' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'save_hardening' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'apply_htaccess_baseline' => ShadowScan_Subscription_State::FEATURE_HTACCESS,
            'rollback_htaccess' => ShadowScan_Subscription_State::FEATURE_HTACCESS,
            'toggle_auto_htaccess' => ShadowScan_Subscription_State::FEATURE_HTACCESS,
            'toggle_remote_diagnostics' => ShadowScan_Subscription_State::FEATURE_REMOTE_DIAGNOSTICS,
            'toggle_enum_strict' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'toggle_enforce_mfa' => ShadowScan_Subscription_State::FEATURE_HARDENING,
            'toggle_csp_report_only' => ShadowScan_Subscription_State::FEATURE_HEADERS,
            'toggle_x_frame_options' => ShadowScan_Subscription_State::FEATURE_HEADERS,
        );
        $enforcement_actions = array(
            'save_mfa_settings' => 'hardened',
            'save_autoupdate_settings' => 'hardened',
            'save_password_policy' => 'hardened',
            'save_geo_policy' => 'hardened',
            'save_hardening' => 'hardened',
            'toggle_enum_strict' => 'hardened',
            'toggle_enforce_mfa' => 'hardened',
            'toggle_csp_report_only' => 'hardened',
            'toggle_x_frame_options' => 'hardened',
            'apply_htaccess_baseline' => 'managed',
            'rollback_htaccess' => 'managed',
            'toggle_auto_htaccess' => 'managed',
        );

        if (isset($gated_actions[$action])) {
            shadowscan_audit_event(
                'GATED_ACTION_ATTEMPTED',
                'Gated action attempted',
                array(
                    'action' => $action,
                    'status' => $status,
                )
            );

            $allow_action = ShadowScan_Subscription_State::is_feature_allowed($gated_actions[$action], $subscription_state);
            if ($action === 'toggle_emergency_mode') {
                $enable = isset($_POST['emergency_state']) && $_POST['emergency_state'] === '1';
                if (!$enable) {
                    $allow_action = true;
                }
            }
            if ($action === 'toggle_remote_diagnostics') {
                $enable = isset($_POST['remote_diagnostics']) && $_POST['remote_diagnostics'] === '1';
                if (!$enable) {
                    $allow_action = true;
                }
            }

            if (!$allow_action) {
                shadowscan_audit_event(
                    'GATED_ACTION_DENIED',
                    'Gated action denied',
                    array(
                        'action' => $action,
                        'status' => $status,
                    ),
                    'warning'
                );
                $deny_action(__('Subscription inactive. Resolve billing to re-enable changes.', 'shadowscan-security-link'));
                $action = '';
            }
        }
        if ($action !== '' && isset($enforcement_actions[$action])) {
            $billing_active = $status === ShadowScan_Subscription_State::STATUS_ACTIVE;
            if (!$billing_active && !shadowscan_is_dev_env()) {
                $deny_action(__('Changes are paused until billing is active.', 'shadowscan-security-link'));
                $action = '';
            } elseif (!shadowscan_can_apply_enforcement($subscription_state, $policy_state, $enforcement_actions[$action])) {
                $deny_action(__('Enforcement is locked by policy tier.', 'shadowscan-security-link'));
                $action = '';
            }
        }

        $live_required_actions = array(
            'save_mfa_settings',
            'save_autoupdate_settings',
            'save_password_policy',
            'save_geo_policy',
            'save_hardening',
            'apply_htaccess_baseline',
            'rollback_htaccess',
            'toggle_auto_htaccess',
            'toggle_remote_diagnostics',
            'toggle_enum_strict',
            'toggle_enforce_mfa',
            'toggle_csp_report_only',
            'toggle_x_frame_options',
        );
        if ($action !== '' && in_array($action, $live_required_actions, true)
            && shadowscan_guard_manager()->is_flag_enabled('owasp_fail_closed_logging', false)
            && shadowscan_is_logging_degraded()) {
            status_header(423);
            $notice = esc_html__('Logging is degraded; configuration changes are paused.', 'shadowscan-security-link');
            $notice_type = 'warning';
            $action = '';
        }
        if ($action !== '' && in_array($action, $live_required_actions, true) && $protection_state !== 'live') {
            status_header(423);
            $notice = esc_html__('Protection is not in a live state. Updates are paused until billing resumes.', 'shadowscan-security-link');
            $notice_type = 'warning';
            $action = '';
        }
    }

    if ($action !== '') {
        if ($action === 'complete_pairing') {
            $code = sanitize_text_field(wp_unslash($_POST['pairing_code'] ?? ''));
            if ($code) {
                $result = shadowscan_complete_pairing($code);
                $notice = $result['message'];
                $notice_type = $result['success'] ? 'success' : 'error';
            } else {
                $notice = esc_html__('Please enter a pairing code.', 'shadowscan-security-link');
                $notice_type = 'error';
            }
        } elseif ($action === 'run_self_check') {
            $last_check_at = (int) get_option(SHADOWSCAN_OPTION_LAST_SELF_CHECK_AT, 0);
            $cooldown = SHADOWSCAN_SELF_CHECK_COOLDOWN - (time() - $last_check_at);
            if ($cooldown > 0) {
                /* translators: %d: seconds remaining. */
                $notice = esc_html(sprintf(__('Please wait %d seconds before running another self-check.', 'shadowscan-security-link'), $cooldown));
                $notice_type = 'warning';
            } else {
                update_option(SHADOWSCAN_OPTION_LAST_SELF_CHECK_AT, time(), false);
                $result = shadowscan_self_check();
                update_option(SHADOWSCAN_OPTION_LAST_SELF_CHECK, $result, false);
                $notice = $result['success'] ? esc_html__('Self-check passed.', 'shadowscan-security-link') : $result['message'];
                $notice_type = $result['success'] ? 'success' : 'error';
            }
        } elseif ($action === 'refresh_status') {
            $last_refresh_at = (int) get_option(SHADOWSCAN_OPTION_LAST_REFRESH_AT, 0);
            $cooldown = SHADOWSCAN_REFRESH_COOLDOWN - (time() - $last_refresh_at);
            if ($cooldown > 0) {
                /* translators: %d: seconds remaining. */
                $notice = esc_html(sprintf(__('Please wait %d seconds before checking status again.', 'shadowscan-security-link'), $cooldown));
                $notice_type = 'warning';
            } else {
                update_option(SHADOWSCAN_OPTION_LAST_REFRESH_AT, time(), false);
                $error = shadowscan_send_heartbeat(true);
                if (is_wp_error($error)) {
                    $guidance = shadowscan_connection_guidance(
                        array(
                            'context' => 'heartbeat',
                            'message' => $error->get_error_message(),
                            'status' => (int) ($error->get_error_data()['status'] ?? 0),
                        )
                    );
                    $notice = $guidance['summary'];
                    $notice_type = 'error';
                } else {
                    $notice = esc_html__('Status refreshed.', 'shadowscan-security-link');
                }
            }
        } elseif ($action === 'toggle_ui_advanced') {
            $enabled = isset($_POST['ui_advanced']) && $_POST['ui_advanced'] === '1';
            $previous = (bool) get_option(SHADOWSCAN_OPTION_UI_ADVANCED, false);
            $acknowledged = isset($_POST['ui_advanced_ack']) && $_POST['ui_advanced_ack'] === '1';
            $can_persist = $is_dev_env || !in_array($status, array(ShadowScan_Subscription_State::STATUS_SUSPENDED, ShadowScan_Subscription_State::STATUS_CANCELLED), true);
            $focus_target = isset($_POST['coverage_target']) ? sanitize_text_field(wp_unslash($_POST['coverage_target'])) : '';
            if ($focus_target && $focus_target[0] === '#') {
                $focus_target = substr($focus_target, 1);
            }

            if ($enabled && !$acknowledged) {
                $notice = esc_html__('Please acknowledge before enabling advanced controls.', 'shadowscan-security-link');
                $notice_type = 'error';
            } elseif ($enabled && !$can_persist) {
                $notice = esc_html__('Advanced controls are available only with an active subscription.', 'shadowscan-security-link');
                $notice_type = 'warning';
            } else {
                update_option(SHADOWSCAN_OPTION_UI_ADVANCED, $enabled, false);
                $notice = $enabled
                    ? esc_html__('Advanced controls are now visible.', 'shadowscan-security-link')
                    : esc_html__('Advanced controls are now hidden.', 'shadowscan-security-link');
                $notice_type = 'success';
                if ($enabled && $focus_target !== '') {
                    update_option(SHADOWSCAN_OPTION_UI_FOCUS_ANCHOR, $focus_target, false);
                }
                if ($previous !== $enabled) {
                    shadowscan_audit_event(
                        'UI_MODE_CHANGED',
                        'UI mode changed',
                        array(
                            'from' => $previous ? 'advanced' : 'hands_off',
                            'to' => $enabled ? 'advanced' : 'hands_off',
                        )
                    );
                }
            }
        } elseif ($action === 'subscription_override_set' && $is_dev_env) {
            $status = sanitize_text_field(wp_unslash($_POST['subscription_override_status'] ?? ''));
            $allowed = array(
                'none',
                ShadowScan_Subscription_State::STATUS_TRIAL,
                ShadowScan_Subscription_State::STATUS_ACTIVE,
                ShadowScan_Subscription_State::STATUS_PAST_DUE,
                ShadowScan_Subscription_State::STATUS_SUSPENDED,
                ShadowScan_Subscription_State::STATUS_CANCELLED,
            );
            if (!in_array($status, $allowed, true)) {
                $status = 'none';
            }
            if ($status === 'none') {
                delete_option(SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE);
                shadowscan_audit_event(
                    'SUBSCRIPTION_OVERRIDE_CLEARED',
                    'Subscription override cleared',
                    array(
                        'site_url' => home_url(),
                        'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
                    )
                );
                $notice = esc_html__('Subscription override cleared.', 'shadowscan-security-link');
                $notice_type = 'success';
            } else {
                $grace_raw = sanitize_text_field(wp_unslash($_POST['subscription_override_grace_end'] ?? ''));
                $suspended_raw = sanitize_text_field(wp_unslash($_POST['subscription_override_suspended_since'] ?? ''));
                $grace_end = $grace_raw ? (int) strtotime($grace_raw) : 0;
                $suspended_since = $suspended_raw ? (int) strtotime($suspended_raw) : 0;
                $override = array(
                    'status' => $status,
                    'grace_end' => $grace_end,
                    'suspended_since' => $suspended_since,
                    'updated_at' => time(),
                    'set_by_user_id' => get_current_user_id(),
                );
                update_option(SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE, $override, false);
                if (in_array($status, array(ShadowScan_Subscription_State::STATUS_SUSPENDED, ShadowScan_Subscription_State::STATUS_CANCELLED), true)) {
                    delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS);
                    delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES);
                }
                shadowscan_audit_event(
                    'SUBSCRIPTION_OVERRIDE_SET',
                    'Subscription override set',
                    array(
                        'status' => $status,
                        'site_url' => home_url(),
                        'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
                    )
                );
                $notice = esc_html__('Subscription override applied.', 'shadowscan-security-link');
                $notice_type = 'success';
            }
        } elseif ($action === 'subscription_override_clear' && $is_dev_env) {
            delete_option(SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE);
            shadowscan_audit_event(
                'SUBSCRIPTION_OVERRIDE_CLEARED',
                'Subscription override cleared',
                array(
                    'site_url' => home_url(),
                    'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
                )
            );
            $notice = esc_html__('Subscription override cleared.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'save_mfa_settings') {
            $nudge_enabled = isset($_POST['mfa_nudge_enabled']) && $_POST['mfa_nudge_enabled'] === '1';
            $enforce_enabled = isset($_POST['mfa_enforce_admins']) && $_POST['mfa_enforce_admins'] === '1';
            $grace_hours = isset($_POST['mfa_grace_hours']) ? (int) $_POST['mfa_grace_hours'] : 0;

            $previous_nudge = class_exists('ShadowScan_MFA') ? ShadowScan_MFA::is_nudge_enabled() : true;
            $previous_enforce = class_exists('ShadowScan_MFA') ? ShadowScan_MFA::is_enforcement_enabled() : false;

            if (class_exists('ShadowScan_MFA')) {
                ShadowScan_MFA::set_nudge_enabled($nudge_enabled);
                ShadowScan_MFA::set_enforcement_enabled($enforce_enabled);
                ShadowScan_MFA::set_grace_period_hours($grace_hours);
            } else {
                shadowscan_guard_manager()->set_flag('enforce_mfa', $enforce_enabled);
            }

            if ($previous_nudge !== $nudge_enabled || $previous_enforce !== $enforce_enabled) {
                ShadowScan_Signal_Manager::emit(
                    'mfa_policy_changed',
                    'info',
                    'MFA policy updated',
                    array(
                        'category' => 'security_control',
                        'control_key' => 'mfa_enforce_admins',
                        'enforced' => $enforce_enabled,
                        'previous_enforced' => $previous_enforce,
                        'nudge_enabled' => $nudge_enabled,
                    )
                );
            }

            $notice = esc_html__('MFA settings saved.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'save_autoupdate_settings' && class_exists('ShadowScan_AutoUpdates')) {
            $enabled = isset($_POST['autoupdate_plugins_enabled']) && $_POST['autoupdate_plugins_enabled'] === '1';
            $mode = isset($_POST['autoupdate_plugins_mode']) ? sanitize_text_field(wp_unslash($_POST['autoupdate_plugins_mode'])) : 'allowlist';
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized below.
            $allowlist_raw = isset($_POST['autoupdate_plugins_allowlist']) ? (array) wp_unslash($_POST['autoupdate_plugins_allowlist']) : array();
            $allowlist = array_map('sanitize_text_field', $allowlist_raw);
            $reason = 'manual';

            ShadowScan_AutoUpdates::sync_settings($enabled, $mode, $allowlist, $reason);
            $core_enabled = isset($_POST['autoupdate_core_enabled']) && $_POST['autoupdate_core_enabled'] === '1';
            ShadowScan_AutoUpdates::set_core_enabled($core_enabled);
            $status = ShadowScan_AutoUpdates::get_status();

            ShadowScan_Signal_Manager::emit(
                'autoupdate_setting_changed',
                'info',
                'Plugin auto-update settings updated',
                array(
                    'category' => 'security_control',
                    'control_key' => 'autoupdate_plugins_enforce',
                    'status' => $enabled ? 'ok' : 'warn',
                    'enforced' => $enabled,
                    'evidence' => array(
                        'enabled' => $enabled,
                        'mode' => $status['mode'] ?? 'allowlist',
                        'allowlist_count' => count($status['allowlist'] ?? array()),
                    ),
                    'last_checked_at' => gmdate('c'),
                )
            );

            $notice = esc_html__('Auto-update settings saved.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'save_password_policy' && class_exists('ShadowScan_Password_Policy')) {
            $previous_enforced = ShadowScan_Password_Policy::is_min_length_enforced() || ShadowScan_Password_Policy::is_breach_enforced();
            $min_length = isset($_POST['password_min_length']) ? (int) $_POST['password_min_length'] : 12;
            $enforce_min = isset($_POST['password_enforce_min_length']) && $_POST['password_enforce_min_length'] === '1';
            $enforce_breach = isset($_POST['password_enforce_breached']) && $_POST['password_enforce_breached'] === '1';
            $provider = isset($_POST['password_breach_provider']) ? sanitize_text_field(wp_unslash($_POST['password_breach_provider'])) : 'hibp';
            $email_notice_enabled = isset($_POST['password_policy_email_notice']) && $_POST['password_policy_email_notice'] === '1';
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized below.
            $roles = isset($_POST['password_enforce_roles']) ? (array) wp_unslash($_POST['password_enforce_roles']) : array();
            $roles = array_map('sanitize_text_field', $roles);
            $editable_roles = array_keys(get_editable_roles());
            $roles = array_values(array_intersect($roles, $editable_roles));
            if (empty($roles)) {
                $roles = array('administrator');
            }

            ShadowScan_Password_Policy::set_min_length($min_length);
            ShadowScan_Password_Policy::set_enforce_min_length($enforce_min);
            ShadowScan_Password_Policy::set_enforce_breached($enforce_breach);
            ShadowScan_Password_Policy::set_breach_provider($provider);
            ShadowScan_Password_Policy::set_enforced_roles($roles);
            ShadowScan_Password_Policy::set_policy_email_enabled($email_notice_enabled);
            $now_enforced = $enforce_min || $enforce_breach;
            $policy_enabled_at = ShadowScan_Password_Policy::get_policy_enabled_at();
            if ($now_enforced && (!$previous_enforced || $policy_enabled_at <= 0)) {
                ShadowScan_Password_Policy::set_policy_enabled_at(time());
            } elseif (!$now_enforced) {
                ShadowScan_Password_Policy::set_policy_enabled_at(0);
                ShadowScan_Password_Policy::set_policy_notice_sent_at(0);
            }
            if ($now_enforced) {
                $notice_sent_at = ShadowScan_Password_Policy::get_policy_notice_sent_at();
                if ($notice_sent_at <= 0 && $email_notice_enabled) {
                    $grace_days = 14;
                    $deadline = ShadowScan_Password_Policy::get_policy_grace_deadline();
                    $deadline_text = $deadline > 0 ? gmdate('Y-m-d H:i', $deadline) . ' UTC' : '';
                    $reset_url = wp_lostpassword_url();
                    // translators: %d: Number of days in grace period
                    $subject = sprintf(__('ShadowScan password policy enabled (%d-day grace period)', 'shadowscan-security-link'), $grace_days);
                    // translators: %1$d: Number of days in grace period, %2$s: Password reset URL, %3$s: Grace period end date
                    $message = sprintf(
                        __("ShadowScan has enabled a password policy for this site.\n\nYou must update your password within %1\$d days to continue accessing wp-admin.\n\nReset link: %2\$s\n\nGrace period ends: %3\$s", 'shadowscan-security-link'),
                        $grace_days,
                        $reset_url,
                        $deadline_text
                    );
                    $emails = array();
                    $users = get_users(array(
                        'role__in' => $roles,
                        'fields' => array('user_email'),
                    ));
                    foreach ($users as $user) {
                        if (!empty($user->user_email)) {
                            $emails[] = $user->user_email;
                        }
                    }
                    $emails = array_values(array_unique($emails));
                    if (!empty($emails)) {
                        wp_mail($emails, $subject, $message);
                        ShadowScan_Password_Policy::set_policy_notice_sent_at(time());
                    }
                }
            }

            ShadowScan_Signal_Manager::emit(
                'password_policy_changed',
                'info',
                'Password policy settings updated',
                array(
                    'category' => 'security_control',
                    'control_key' => 'password_policy_enforce_min_length',
                    'status' => ($enforce_min || $enforce_breach) ? 'ok' : 'warn',
                    'enforced' => $enforce_min || $enforce_breach,
                    'evidence' => array(
                        'min_length' => ShadowScan_Password_Policy::get_min_length(),
                        'enforce_min_length' => $enforce_min,
                        'enforce_breached' => $enforce_breach,
                        'provider' => ShadowScan_Password_Policy::get_breach_provider(),
                        'roles' => $roles,
                        'email_notice_enabled' => $email_notice_enabled,
                    ),
                    'last_checked_at' => gmdate('c'),
                )
            );

            $notice = esc_html__('Password policy settings saved.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'snooze_password_grace' && class_exists('ShadowScan_Password_Policy')) {
            $current_user = wp_get_current_user();
            if ($current_user instanceof WP_User) {
                $snooze_days = ShadowScan_Password_Policy::get_snooze_days();
                ShadowScan_Password_Policy::set_user_snooze_until($current_user, time() + ($snooze_days * DAY_IN_SECONDS));
                $notice = esc_html__('Password reminder snoozed.', 'shadowscan-security-link');
                $notice_type = 'success';
            }
        } elseif ($action === 'save_geo_policy' && class_exists('ShadowScan_Geo_Policy')) {
            $mode = isset($_POST['geo_block_mode']) ? sanitize_text_field(wp_unslash($_POST['geo_block_mode'])) : 'off';
            $countries_raw = isset($_POST['geo_blocked_countries']) ? sanitize_textarea_field(wp_unslash($_POST['geo_blocked_countries'])) : '';
            $countries = array_filter(array_map('trim', preg_split('/[\s,]+/', $countries_raw)));
            $allow_ips_raw = isset($_POST['geo_allow_admin_ips']) ? sanitize_textarea_field(wp_unslash($_POST['geo_allow_admin_ips'])) : '';
            $allow_ips = array_filter(array_map('trim', preg_split('/[\s,]+/', $allow_ips_raw)));

            ShadowScan_Geo_Policy::set_mode($mode);
            ShadowScan_Geo_Policy::set_countries($countries);
            ShadowScan_Geo_Policy::set_allow_ips($allow_ips);

            ShadowScan_Signal_Manager::emit(
                'geo_policy_changed',
                'info',
                'Geo policy updated',
                array(
                    'category' => 'security_control',
                    'control_key' => 'geo_policy_enforce_wp_admin',
                    'status' => $mode === 'off' ? 'warn' : 'ok',
                    'enforced' => $mode === 'enforce_wp_admin',
                    'evidence' => array(
                        'mode' => $mode,
                        'blocked_countries' => $countries,
                        'blocked_count' => count($countries),
                    ),
                    'last_checked_at' => gmdate('c'),
                )
            );

            $notice = esc_html__('Geo policy settings saved.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'save_hardening') {
            $enum_enabled = isset($_POST['enum_strict']) && $_POST['enum_strict'] === '1';
            $csp_enabled = isset($_POST['csp_report_only']) && $_POST['csp_report_only'] === '1';
            $x_frame_enabled = isset($_POST['x_frame_options']) && $_POST['x_frame_options'] === '1';

            shadowscan_guard_manager()->set_flag('disable_author_archives', $enum_enabled);
            shadowscan_guard_manager()->set_flag('restrict_rest_users', $enum_enabled);
            shadowscan_guard_manager()->toggle_header_x_frame($x_frame_enabled);
            if ($csp_enabled) {
                shadowscan_guard_manager()->enable_csp_report_only();
            } else {
                shadowscan_guard_manager()->set_flag('csp_report_only', false);
            }

            $notice = esc_html__('Protection settings saved.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'toggle_remote_diagnostics') {
            $enabled = isset($_POST['remote_diagnostics']) && $_POST['remote_diagnostics'] === '1';
            if ($enabled && !shadowscan_sentry_telemetry_enabled()) {
                $notice = esc_html__('Enable Sentry telemetry first, then retry remote diagnostics.', 'shadowscan-security-link');
                $notice_type = 'warning';
                $enabled = false;
            }
            update_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS, $enabled, false);
            if ($enabled) {
                update_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES, time() + SHADOWSCAN_REMOTE_DIAGNOSTICS_TTL, false);
            } else {
                delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES);
                shadowscan_remove_mu_helper();
            }
            if ($notice === '') {
                $notice = $enabled
                    ? esc_html__('Remote diagnostics enabled for 60 minutes.', 'shadowscan-security-link')
                    : esc_html__('Remote diagnostics disabled.', 'shadowscan-security-link');
                $notice_type = 'success';
            }
        } elseif ($action === 'toggle_sentry_telemetry') {
            $enabled = isset($_POST['sentry_telemetry']) && $_POST['sentry_telemetry'] === '1';
            update_option(SHADOWSCAN_OPTION_SENTRY_TELEMETRY, $enabled, false);
            if (!$enabled) {
                update_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS, false, false);
                delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES);
                shadowscan_remove_mu_helper();
            }
            $notice = $enabled
                ? esc_html__('Sentry telemetry enabled. You can now enable remote diagnostics when needed.', 'shadowscan-security-link')
                : esc_html__('Sentry telemetry disabled. Remote diagnostics and MU helper were turned off.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'toggle_bf_safe_mode') {
            $enabled = isset($_POST['bf_safe_mode']) && $_POST['bf_safe_mode'] === '1';
            ShadowScan_Storage::set('bf_safe_mode', $enabled);
            $notice = $enabled
                ? esc_html__('Login abuse detection safe mode enabled.', 'shadowscan-security-link')
                : esc_html__('Login abuse detection safe mode disabled.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'toggle_enforce_mfa') {
            $enabled = isset($_POST['enforce_mfa']) && $_POST['enforce_mfa'] === '1';
            if (class_exists('ShadowScan_MFA')) {
                ShadowScan_MFA::set_enforcement_enabled($enabled);
            } else {
                shadowscan_guard_manager()->set_flag('enforce_mfa', $enabled);
            }
            $notice = $enabled
                ? esc_html__('MFA enforcement enabled.', 'shadowscan-security-link')
                : esc_html__('MFA enforcement disabled.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'toggle_emergency_mode') {
            $enable = isset($_POST['emergency_state']) && $_POST['emergency_state'] === '1';
            $reason = sanitize_text_field(wp_unslash($_POST['emergency_reason'] ?? ''));
            if ($enable) {
                $confirmed = (isset($_POST['emergency_confirm']) && $_POST['emergency_confirm'] === '1')
                    || (isset($_POST['emergency_phrase']) && strtoupper(trim(sanitize_text_field(wp_unslash($_POST['emergency_phrase'])))) === 'ENABLE');
                if (!$confirmed) {
                    $notice = esc_html__('Please confirm before enabling emergency containment.', 'shadowscan-security-link');
                    $notice_type = 'error';
                } else {
                    shadowscan_guard_manager()->enable_emergency_mode($reason);
                    $notice = esc_html__('Emergency containment enabled.', 'shadowscan-security-link');
                    $notice_type = 'success';
                }
            } else {
                shadowscan_guard_manager()->disable_emergency_mode();
                $notice = esc_html__('Emergency containment disabled.', 'shadowscan-security-link');
                $notice_type = 'success';
            }
        } elseif ($action === 'toggle_auto_htaccess') {
            $enabled = isset($_POST['auto_htaccess']) && $_POST['auto_htaccess'] === '1';
            update_option(SHADOWSCAN_OPTION_AUTO_HTACCESS, $enabled, false);
            if ($enabled) {
                shadowscan_maybe_auto_apply_htaccess($subscription_state);
            }
            $notice = $enabled
                ? esc_html__('Automatic htaccess baseline enabled.', 'shadowscan-security-link')
                : esc_html__('Automatic htaccess baseline disabled.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'apply_htaccess_baseline') {
            $ok = shadowscan_guard_manager()->apply_htaccess_baseline();
            $notice = $ok
                ? esc_html__('htaccess baseline applied.', 'shadowscan-security-link')
                : esc_html__('htaccess baseline could not be applied.', 'shadowscan-security-link');
            $notice_type = $ok ? 'success' : 'error';
        } elseif ($action === 'rollback_htaccess') {
            $ok = shadowscan_guard_manager()->rollback_htaccess();
            $notice = $ok
                ? esc_html__('htaccess rollback completed.', 'shadowscan-security-link')
                : esc_html__('htaccess rollback not available.', 'shadowscan-security-link');
            $notice_type = $ok ? 'success' : 'error';
        } elseif ($action === 'toggle_enum_strict') {
            $enable = isset($_POST['enum_strict']) && $_POST['enum_strict'] === '1';
            shadowscan_guard_manager()->set_flag('disable_author_archives', $enable);
            shadowscan_guard_manager()->set_flag('restrict_rest_users', $enable);
            $notice = $enable
                ? esc_html__('Enumeration strict mode enabled.', 'shadowscan-security-link')
                : esc_html__('Enumeration strict mode disabled.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'toggle_csp_report_only') {
            $enable = isset($_POST['csp_report_only']) && $_POST['csp_report_only'] === '1';
            if ($enable) {
                shadowscan_guard_manager()->enable_csp_report_only();
                $notice = esc_html__('CSP report-only enabled.', 'shadowscan-security-link');
            } else {
                shadowscan_guard_manager()->set_flag('csp_report_only', false);
                $notice = esc_html__('CSP report-only disabled.', 'shadowscan-security-link');
            }
            $notice_type = 'success';
        } elseif ($action === 'toggle_x_frame_options') {
            $enable = isset($_POST['x_frame_options']) && $_POST['x_frame_options'] === '1';
            shadowscan_guard_manager()->toggle_header_x_frame($enable);
            $notice = $enable
                ? esc_html__('X-Frame-Options enabled.', 'shadowscan-security-link')
                : esc_html__('X-Frame-Options disabled.', 'shadowscan-security-link');
            $notice_type = 'success';
        } elseif ($action === 'download_logs') {
            shadowscan_dashboard_set_notice(
                __('Use the diagnostics download button again. Downloads now run via a dedicated endpoint.', 'shadowscan-security-link'),
                'warning'
            );
            wp_safe_redirect(shadowscan_dashboard_redirect_url('diagnostics'));
            exit;
        } elseif ($action === 'export_owasp_evidence') {
            shadowscan_dashboard_set_notice(
                __('Use the diagnostics export button again. Exports now run via a dedicated endpoint.', 'shadowscan-security-link'),
                'warning'
            );
            wp_safe_redirect(shadowscan_dashboard_redirect_url('diagnostics'));
            exit;
        } elseif ($action === 'disconnect_site') {
            $confirmed = isset($_POST['disconnect_confirm']) && $_POST['disconnect_confirm'] === '1';
            $phrase = strtoupper(trim(sanitize_text_field(wp_unslash($_POST['disconnect_phrase'] ?? ''))));
            if (!$confirmed || $phrase !== 'DISCONNECT') {
                $notice = esc_html__('Please confirm before disconnecting this site.', 'shadowscan-security-link');
                $notice_type = 'error';
            } else {
                shadowscan_audit_event('SITE_DISCONNECT_REQUESTED', 'Disconnect requested', array());
                shadowscan_notify_disconnect('manual_disconnect');
                delete_option(SHADOWSCAN_OPTION_SITE_TOKEN);
                delete_option(SHADOWSCAN_OPTION_SITE_ID);
                if (function_exists('shadowscan_sync_clear_identity')) {
                    shadowscan_sync_clear_identity();
                }
                $notice = esc_html__('Site disconnected.', 'shadowscan-security-link');
                $notice_type = 'success';
                shadowscan_audit_event('SITE_DISCONNECTED', 'Site disconnected', array());
            }
        } elseif ($action === 'remove_local_data') {
            $confirmed = isset($_POST['remove_confirm']) && $_POST['remove_confirm'] === '1';
            $phrase = strtoupper(trim(sanitize_text_field(wp_unslash($_POST['remove_phrase'] ?? ''))));
            if (!$confirmed || $phrase !== 'REMOVE') {
                $notice = esc_html__('Please confirm before removing local data.', 'shadowscan-security-link');
                $notice_type = 'error';
            } else {
                shadowscan_clear_local_data();
                $notice = esc_html__('Local ShadowScan data removed.', 'shadowscan-security-link');
                $notice_type = 'success';
                shadowscan_audit_event('LOCAL_DATA_REMOVED', 'Local data removed', array());
            }
        } elseif ($action === 'submit_contact') {
            $subject = sanitize_text_field(wp_unslash($_POST['contact_subject'] ?? ''));
            $message = sanitize_textarea_field(wp_unslash($_POST['contact_message'] ?? ''));
            $email = sanitize_email(wp_unslash($_POST['contact_email'] ?? ''));
            if ($subject === '' || $message === '' || $email === '') {
                $notice = esc_html__('Please complete all contact fields.', 'shadowscan-security-link');
                $notice_type = 'error';
            } elseif (!is_email($email)) {
                $notice = esc_html__('Please enter a valid email address.', 'shadowscan-security-link');
                $notice_type = 'error';
            } else {
                $payload = array(
                    'subject' => $subject,
                    'message' => $message,
                    'email' => $email,
                    'site_url' => home_url(),
                    'wp_version' => get_bloginfo('version'),
                    'php_version' => PHP_VERSION,
                    'plugin_version' => SHADOWSCAN_PLUGIN_VERSION,
                );
                $response = shadowscan_send_api_request('submit-plugin-contact', $payload, 'POST', true);
                if (is_wp_error($response)) {
                    $notice = esc_html__('Unable to send your message right now. Please try again.', 'shadowscan-security-link');
                    $notice_type = 'error';
                } else {
                    $notice = esc_html__('Message sent. Our team will follow up shortly.', 'shadowscan-security-link');
                    $notice_type = 'success';
                    shadowscan_audit_event('CONTACT_SUBMITTED', 'Contact form submitted', array());
                }
            }
        } elseif ($action === 'billing_cta') {
            shadowscan_audit_event('BILLING_CTA_CLICKED', 'Billing CTA clicked', array());
            wp_safe_redirect(SHADOWSCAN_PORTAL_BILLING_URL);
            exit;
        }
    }

    $third_party_plugins = shadowscan_detect_third_party_security_plugins();
    $last_ping = (int) get_option(SHADOWSCAN_OPTION_LAST_PING, 0);
    $can_install_plugins = current_user_can('install_plugins');
    $last_error = get_option(SHADOWSCAN_OPTION_LAST_ERROR, array());
    $pending_heartbeat = get_option(SHADOWSCAN_OPTION_PENDING_HEARTBEAT, null);
    $pending_results = get_option(SHADOWSCAN_OPTION_PENDING_RESULTS, array());
    $last_self_check = get_option(SHADOWSCAN_OPTION_LAST_SELF_CHECK, array());
    $last_refresh_at = (int) get_option(SHADOWSCAN_OPTION_LAST_REFRESH_AT, 0);
    $last_self_check_at = (int) get_option(SHADOWSCAN_OPTION_LAST_SELF_CHECK_AT, 0);
    $refresh_cooldown_remaining = max(0, SHADOWSCAN_REFRESH_COOLDOWN - (time() - $last_refresh_at));
    $self_check_cooldown_remaining = max(0, SHADOWSCAN_SELF_CHECK_COOLDOWN - (time() - $last_self_check_at));
    $next_heartbeat = wp_next_scheduled('shadowscan_heartbeat');
    $next_poll = wp_next_scheduled('shadowscan_poll_commands');
    $support_report = shadowscan_build_connector_report();
    $remote_diagnostics_enabled = shadowscan_remote_diagnostics_enabled();
    $remote_diagnostics_expires = (int) get_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES, 0);
    $sentry_telemetry_enabled = shadowscan_sentry_telemetry_enabled();
    $auto_htaccess_enabled = (bool) get_option(SHADOWSCAN_OPTION_AUTO_HTACCESS, false);
    $guard_status = shadowscan_guard_manager()->get_guard_status();
    $bruteforce_stats = shadowscan_guard_manager()->get_bruteforce_stats(true);
    $emergency_status = shadowscan_guard_manager()->get_emergency_status();
    $htaccess_state = shadowscan_guard_manager()->get_htaccess_state();
    $enumeration_state = shadowscan_guard_manager()->get_enumeration_state();
    $headers_state = shadowscan_guard_manager()->get_headers_state();
    $access_control_state = shadowscan_guard_manager()->get_access_control_state();
    $integrity_last_event_at = (int) ShadowScan_Storage::get('integrity_last_high_event', 0);
    $integrity_last_event_summary = (string) ShadowScan_Storage::get('integrity_last_high_summary', '');
    $mfa_enforced = class_exists('ShadowScan_MFA')
        ? ShadowScan_MFA::is_enforcement_enabled()
        : shadowscan_guard_manager()->is_flag_enabled('enforce_mfa', false);
    $mfa_nudge_enabled = class_exists('ShadowScan_MFA') ? ShadowScan_MFA::is_nudge_enabled() : true;
    $mfa_grace_hours = class_exists('ShadowScan_MFA') ? ShadowScan_MFA::get_grace_period_hours() : 0;
    $mfa_coverage = class_exists('ShadowScan_MFA')
        ? ShadowScan_MFA::compute_admin_coverage()
        : array('admin_total' => 0, 'admin_enrolled' => 0, 'coverage_percent' => 0, 'provider' => '');
    $autoupdate_status = class_exists('ShadowScan_AutoUpdates') ? ShadowScan_AutoUpdates::get_status() : array();
    $password_policy_status = class_exists('ShadowScan_Password_Policy') ? ShadowScan_Password_Policy::get_status() : array();
    $geo_policy_status = class_exists('ShadowScan_Geo_Policy') ? ShadowScan_Geo_Policy::get_status() : array();
    $event_queue_count = ShadowScan_Event_Buffer::count();
    $cron_last_tick = (int) ShadowScan_Storage::get('cron_last_tick', 0);
    $cron_stale = $cron_last_tick > 0 && (time() - $cron_last_tick) > 1800;
    $cron_critical = $cron_last_tick > 0 && (time() - $cron_last_tick) > 7200;
    $logging_degraded = shadowscan_is_logging_degraded();

    if ($cron_critical) {
        $last_alert = (int) ShadowScan_Storage::get('cron_stall_alert_at', 0);
        if ((time() - $last_alert) > DAY_IN_SECONDS) {
            ShadowScan_Storage::set('cron_stall_alert_at', time());
            if (class_exists('ShadowScan_Signal_Manager')) {
                ShadowScan_Signal_Manager::emit(
                    'monitoring_cron_stalled',
                    'warning',
                    'Scheduled monitoring appears stalled',
                    array('last_tick' => $cron_last_tick)
                );
            }
        }
    }

    if ($logging_degraded) {
        $last_alert = (int) ShadowScan_Storage::get('logging_degraded_alert_at', 0);
        if ((time() - $last_alert) > DAY_IN_SECONDS) {
            ShadowScan_Storage::set('logging_degraded_alert_at', time());
            if (class_exists('ShadowScan_Signal_Manager')) {
                ShadowScan_Signal_Manager::emit(
                    'monitoring_queue_stalled',
                    'warning',
                    'Event delivery backlog detected',
                    array('queue_count' => $event_queue_count)
                );
            }
        }
    }
    $safe_mode_until = (int) ($guard_status['safe_mode_until'] ?? 0);
    $safe_mode_active = $safe_mode_until > time();
    $mfa_enforced = class_exists('ShadowScan_MFA')
        ? ShadowScan_MFA::is_enforcement_enabled()
        : shadowscan_guard_manager()->is_flag_enabled('enforce_mfa', false);
    $php_version = PHP_VERSION;
    $last_error_summary = esc_html__('None', 'shadowscan-security-link');
    if (is_array($last_error) && !empty($last_error['message'])) {
        $last_error_summary = sprintf(
            '%s: %s',
            $last_error['context'] ?? esc_html__('unknown', 'shadowscan-security-link'),
            $last_error['message']
        );
    }
    $connection_guidance = array();
    if (is_array($last_error) && !empty($last_error['message'])) {
        $context = (string) ($last_error['context'] ?? '');
        if (in_array($context, array('pair-complete', 'heartbeat'), true)) {
            $connection_guidance = shadowscan_connection_guidance($last_error);
        }
    }
    $pending_notice = get_transient('shadowscan_pair_notice');
    $pending_notice_type = get_transient('shadowscan_pair_notice_type');
    if ($pending_notice) {
        $notice = $pending_notice;
        $notice_type = $pending_notice_type === 'error' ? 'error' : 'success';
        delete_transient('shadowscan_pair_notice');
        delete_transient('shadowscan_pair_notice_type');
    }
    $dashboard_notice = shadowscan_dashboard_get_notice();
    if (!empty($dashboard_notice['message'])) {
        $notice = (string) $dashboard_notice['message'];
        $notice_type = (string) ($dashboard_notice['type'] ?? 'success');
    }
    $auth_invalid = function_exists('shadowscan_sync_is_auth_invalid') ? shadowscan_sync_is_auth_invalid() : false;
    if ($auth_invalid && $notice === '') {
        $notice = esc_html__('Pairing remains configured but authentication was rejected by the API. Reconnect required.', 'shadowscan-security-link');
        $notice_type = 'warning';
    }

    $subscription_status = $subscription_state['status'] ?? ShadowScan_Subscription_State::STATUS_TRIAL;
    $subscription_plan = $subscription_state['plan'] ?? '';
    $subscription_grace_end = (int) ($subscription_state['grace_end'] ?? 0);
    $subscription_portal_unreachable = !empty($subscription_state['portal_unreachable']);
    $subscription_test_mode = !empty($subscription_state['test_mode']);
    $subscription_emulated = !empty($subscription_state['emulated']);
    $subscription_emulated_status = $subscription_state['emulated_status'] ?? '';
    $subscription_source = isset($subscription_state['source'])
        ? sanitize_text_field((string) $subscription_state['source'])
        : 'unknown';
    if ($subscription_source === '') {
        $subscription_source = 'unknown';
    }
    $policy_state = class_exists('ShadowScan_Policy_State') ? ShadowScan_Policy_State::get_current_policy() : array();
    $capabilities = shadowscan_get_billing_capabilities($subscription_state, $policy_state, $protection_state);
    $advanced_readonly = !$capabilities['billing_active'];
    if (!isset($readonly_note_message)) {
        $readonly_note_message = __('Changes require an active subscription. Status remains visible.', 'shadowscan-security-link');
    }
    $advanced_locked = in_array($subscription_state['status'] ?? '', array('suspended', 'cancelled'), true);
    $can_persist_advanced = !$advanced_locked;
    $ui_advanced = (bool) get_option(SHADOWSCAN_OPTION_UI_ADVANCED, false);
    if ($advanced_locked) {
        $ui_advanced = false;
    } elseif ($ui_advanced && !$can_persist_advanced) {
        update_option(SHADOWSCAN_OPTION_UI_ADVANCED, false, false);
        $ui_advanced = false;
        if ($notice === '') {
            $notice = esc_html__('Advanced controls are available only with an active subscription.', 'shadowscan-security-link');
            $notice_type = 'warning';
        }
    }
    $tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'overview';
    if (in_array($tab, array('owasp', 'protection'), true)) {
        shadowscan_redirect_legacy_owasp();
    }
    $tabs = array(
        'overview' => esc_html__('Overview', 'shadowscan-security-link'),
        'coverage' => esc_html__('Protection Coverage', 'shadowscan-security-link'),
        'diagnostics' => esc_html__('Diagnostics', 'shadowscan-security-link'),
        'emergency' => esc_html__('Emergency', 'shadowscan-security-link'),
        'contact' => esc_html__('Contact', 'shadowscan-security-link'),
    );
    if (!array_key_exists($tab, $tabs)) {
        $tab = 'overview';
    }
    $subscription_status_label_map = array(
        ShadowScan_Subscription_State::STATUS_ACTIVE => __('Active', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_TRIAL => __('Trial', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_PAST_DUE => __('Past due', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_SUSPENDED => __('Suspended', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_CANCELLED => __('Cancelled', 'shadowscan-security-link'),
    );
    $subscription_status_label = $subscription_status_label_map[$subscription_status] ?? __('Trial', 'shadowscan-security-link');
    $subscription_desc_map = array(
        ShadowScan_Subscription_State::STATUS_ACTIVE => __('Changes enabled', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_TRIAL => __('Activate to unlock changes', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_PAST_DUE => __('Changes paused', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_SUSPENDED => __('Changes locked', 'shadowscan-security-link'),
        ShadowScan_Subscription_State::STATUS_CANCELLED => __('Changes locked', 'shadowscan-security-link'),
    );
    $subscription_desc = $subscription_desc_map[$subscription_status] ?? $subscription_desc_map[ShadowScan_Subscription_State::STATUS_TRIAL];
    $billing_url = shadowscan_billing_cta_url();
    $override_state = $is_dev_env ? get_option(SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE, array()) : array();
    if (!is_array($override_state)) {
        $override_state = array();
    }
    $protection_state_label_map = array(
        'live' => __('Live', 'shadowscan-security-link'),
        'stale' => __('Live (no updates)', 'shadowscan-security-link'),
        'outdated' => __('Outdated', 'shadowscan-security-link'),
        'paused' => __('Paused', 'shadowscan-security-link'),
    );
    $protection_state_label = $protection_state_label_map[$protection_state] ?? __('Paused', 'shadowscan-security-link');
    $protection_tooltip_map = array(
        'live' => array(
            'title' => __('Protection: Live', 'shadowscan-security-link'),
            'body' => __('Protection updates and monitoring are active.', 'shadowscan-security-link'),
        ),
        'stale' => array(
            'title' => __('Protection: Live (no updates)', 'shadowscan-security-link'),
            'body' => __('Protection continues at the last applied level. New updates and configuration changes are paused until billing resumes.', 'shadowscan-security-link'),
        ),
        'outdated' => array(
            'title' => __('Protection: Outdated', 'shadowscan-security-link'),
            'body' => __('This site is not receiving updates and protection may be outdated. Reactivate to restore live updates.', 'shadowscan-security-link'),
        ),
        'paused' => array(
            'title' => __('Protection: Paused', 'shadowscan-security-link'),
            'body' => __('Protection is paused due to subscription status. Reactivate to restore protection.', 'shadowscan-security-link'),
        ),
    );

    $coverage_output_buffer = false;
    if ($is_dev_env && $tab === 'coverage') {
        $coverage_output_buffer = true;
        ob_start();
    }

    $owasp_registry = shadowscan_owasp_get_registry();
    $owasp_cache = shadowscan_owasp_refresh_status();
    $owasp_categories = is_array($owasp_cache['categories'] ?? null) ? $owasp_cache['categories'] : array();
    $owasp_total_categories = count($owasp_registry);
    $owasp_planned_count = 0;
    $owasp_critical = array();
    $owasp_warn = array();
    $owasp_last_checked = 0;
    foreach ($owasp_registry as $category) {
        $category_id = (string) ($category['category_id'] ?? '');
        if ($category_id === '') {
            continue;
        }
        $category_status = $owasp_categories[$category_id] ?? array();
        $status_key = (string) ($category_status['status'] ?? 'not_implemented');
        $last_checked_at = (int) ($category_status['last_checked_at'] ?? 0);
        $owasp_last_checked = max($owasp_last_checked, $last_checked_at);
        $label = $category_id . ' ' . (string) ($category['title'] ?? $category_id);
        if ($status_key === 'critical') {
            $owasp_critical[] = array('label' => $label, 'status' => __('High Risk', 'shadowscan-security-link'), 'last_checked_at' => $last_checked_at);
        } elseif ($status_key === 'warn') {
            $owasp_warn[] = array('label' => $label, 'status' => __('Degraded', 'shadowscan-security-link'), 'last_checked_at' => $last_checked_at);
        } elseif ($status_key === 'not_implemented') {
            $owasp_planned_count++;
        }
    }

    $owasp_overall = __('Healthy', 'shadowscan-security-link');
    if (!empty($owasp_critical)) {
        $owasp_overall = __('High risk', 'shadowscan-security-link');
    } elseif (!empty($owasp_warn)) {
        $owasp_overall = __('Attention required', 'shadowscan-security-link');
    } elseif ($owasp_total_categories > 0 && $owasp_planned_count >= (int) ceil($owasp_total_categories * 0.6)) {
        $owasp_overall = __('Monitoring planned', 'shadowscan-security-link');
    }

    $owasp_top_items = array_merge($owasp_critical, $owasp_warn);
    $owasp_top_items = array_slice($owasp_top_items, 0, 3);

    $format_time = function (?int $timestamp): string {
        if (!$timestamp || $timestamp < 946684800) {
            return '—';
        }
        return date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $timestamp);
    };

    $coverage_registry = class_exists('ShadowScan_Protection_Registry')
        ? ShadowScan_Protection_Registry::get_registry()
        : array();
    $coverage_groups = class_exists('ShadowScan_Protection_Registry')
        ? ShadowScan_Protection_Registry::get_groups()
        : array();
    $coverage_states = array();
    $coverage_group_summary = array();
    $coverage_last_checked = 0;
    $coverage_acknowledgements = shadowscan_coverage_get_acknowledgements();
    $coverage_ack_dirty = false;
    $coverage_ack_invalidations = array();
    foreach ($coverage_registry as $entry) {
        $group = (string) ($entry['group'] ?? '');
        if ($group === '') {
            continue;
        }
        $state = shadowscan_protection_get_control_state($entry);
        $state['fingerprint'] = shadowscan_coverage_control_fingerprint($entry, $state);
        $label_meta = shadowscan_protection_coverage_label(
            (string) ($state['status'] ?? 'not_implemented'),
            !empty($entry['enforcement_supported']),
            !empty($state['enforced'])
        );
        $raw_label = $label_meta['label'];
        $raw_tone = $label_meta['tone'];
        $ack_key = shadowscan_coverage_ack_key($entry);
        $ack_entry = $ack_key !== '' && isset($coverage_acknowledgements[$ack_key])
            ? $coverage_acknowledgements[$ack_key]
            : null;
        $ack_active = false;
        $ack_reason = '';
        if (is_array($ack_entry)) {
            $expires_at = (int) ($ack_entry['expires_at'] ?? 0);
            if ($expires_at > 0 && time() > $expires_at) {
                $ack_reason = 'expired';
            } elseif (!hash_equals((string) ($ack_entry['fingerprint'] ?? ''), (string) $state['fingerprint'])) {
                $ack_reason = 'fingerprint_changed';
            } else {
                $ack_active = true;
            }
            if (!$ack_active) {
                unset($coverage_acknowledgements[$ack_key]);
                $coverage_ack_dirty = true;
                $coverage_ack_invalidations[] = array(
                    'ack_key' => $ack_key,
                    'entry' => $entry,
                    'state' => $state,
                    'reason' => $ack_reason !== '' ? $ack_reason : 'cleared',
                );
            }
        }
        $ack_eligible = in_array(
            $raw_label,
            array(__('Action required', 'shadowscan-security-link'), shadowscan_coverage_label_managed(), shadowscan_coverage_label_external()),
            true
        ) && !shadowscan_coverage_ack_blocked($ack_key, (string) ($entry['id'] ?? ''));
        if ($ack_active) {
            $label_meta = array(
                'label' => __('Covered', 'shadowscan-security-link'),
                'tone' => 'is-ok',
            );
        }
        $coverage_states[$entry['id']] = array_merge($state, array(
            'label' => $label_meta['label'],
            'tone' => $label_meta['tone'],
            'raw_label' => $raw_label,
            'raw_tone' => $raw_tone,
            'acknowledged' => $ack_active,
            'ack_type' => (string) ($ack_entry['type'] ?? ''),
            'ack_expires_at' => (int) ($ack_entry['expires_at'] ?? 0),
            'ack_key' => $ack_key,
            'ack_eligible' => $ack_eligible,
            'entry' => $entry,
        ));
        $coverage_last_checked = max($coverage_last_checked, (int) ($state['last_checked_at'] ?? 0));

        if (!isset($coverage_group_summary[$group])) {
            $coverage_group_summary[$group] = array(
                'label' => __('Covered', 'shadowscan-security-link'),
                'tone' => 'is-ok',
                'monitoring' => 0,
                'enforcement_available' => 0,
                'enforcement_active' => 0,
                'last_checked_at' => 0,
            );
        }
        $coverage_group_summary[$group]['monitoring']++;
        if (!empty($entry['enforcement_supported'])) {
            $coverage_group_summary[$group]['enforcement_available']++;
        }
        if (!empty($state['enforced'])) {
            $coverage_group_summary[$group]['enforcement_active']++;
        }
        $coverage_group_summary[$group]['last_checked_at'] = max(
            $coverage_group_summary[$group]['last_checked_at'],
            (int) ($state['last_checked_at'] ?? 0)
        );
        $priority = array(
            __('Action required', 'shadowscan-security-link') => 4,
            shadowscan_coverage_label_external() => 3,
            shadowscan_coverage_label_managed() => 2,
            __('Monitoring planned', 'shadowscan-security-link') => 1,
            __('Covered', 'shadowscan-security-link') => 0,
        );
        $current_rank = $priority[$coverage_group_summary[$group]['label']] ?? 0;
        $incoming_rank = $priority[$label_meta['label']] ?? 0;
        if ($incoming_rank > $current_rank) {
            $coverage_group_summary[$group]['label'] = $label_meta['label'];
            $coverage_group_summary[$group]['tone'] = $label_meta['tone'];
        }
    }
    if ($coverage_ack_dirty) {
        shadowscan_coverage_save_acknowledgements($coverage_acknowledgements);
    }
    if (!empty($coverage_ack_invalidations) && class_exists('ShadowScan_Signal_Manager')) {
        foreach ($coverage_ack_invalidations as $invalid) {
            $invalid_entry = $invalid['entry'] ?? array();
            $invalid_state = $invalid['state'] ?? array();
            $invalid_key = (string) ($invalid['ack_key'] ?? '');
            $invalid_reason = (string) ($invalid['reason'] ?? 'cleared');
            if ($invalid_key === '') {
                continue;
            }
            ShadowScan_Signal_Manager::emit(
                'security_control_acknowledgement_cleared',
                'info',
                'Control acknowledgement cleared',
                array(
                    'category' => 'security_control',
                    'control_key' => $invalid_key,
                    'status' => 'cleared',
                    'fingerprint' => hash('sha256', $invalid_key . '|cleared|' . $invalid_reason),
                    'recommended_action' => (string) ($invalid_state['recommended_action'] ?? ''),
                    'evidence' => array(
                        'reason' => $invalid_reason,
                        'cleared_at' => gmdate('c', time()),
                    ),
                    'last_checked_at' => gmdate('c', time()),
                    'owasp' => !empty($invalid_entry['owasp_ids']) ? array(
                        'version' => '2025',
                        'id' => (string) $invalid_entry['owasp_ids'][0],
                        'name' => function_exists('shadowscan_owasp_category_name')
                            ? shadowscan_owasp_category_name((string) $invalid_entry['owasp_ids'][0])
                            : (string) $invalid_entry['owasp_ids'][0],
                    ) : array(),
                )
            );
        }
    }

    $coverage_overall = __('Covered', 'shadowscan-security-link');
    $coverage_overall_tone = 'is-ok';
    $coverage_action_items = array();
    $coverage_managed_items = array();
    $coverage_external_items = array();
    foreach ($coverage_states as $state) {
        $entry = $state['entry'];
        $label = (string) $state['label'];
        $item = array(
            'label' => (string) ($entry['title'] ?? $entry['id']),
            'status' => $label,
            'last_checked_at' => (int) ($state['last_checked_at'] ?? 0),
        );
        if ($label === __('Action required', 'shadowscan-security-link')) {
            $coverage_action_items[] = $item;
        } elseif ($label === shadowscan_coverage_label_managed()) {
            $coverage_managed_items[] = $item;
        } elseif ($label === shadowscan_coverage_label_external()) {
            $coverage_external_items[] = $item;
        }
    }
    if (!empty($coverage_action_items)) {
        $coverage_overall = __('Action required', 'shadowscan-security-link');
        $coverage_overall_tone = 'is-bad';
    } elseif (!empty($coverage_external_items)) {
        $coverage_overall = shadowscan_coverage_label_external();
        $coverage_overall_tone = 'is-warn';
    } elseif (!empty($coverage_managed_items)) {
        $coverage_overall = shadowscan_coverage_label_managed();
        $coverage_overall_tone = 'is-warn';
    } elseif (empty($coverage_states)) {
        $coverage_overall = __('Monitoring planned', 'shadowscan-security-link');
        $coverage_overall_tone = '';
    }
    $coverage_top_items = array_merge($coverage_action_items, $coverage_external_items, $coverage_managed_items);
    $coverage_top_items = array_slice($coverage_top_items, 0, 3);

    $render_status_row = function (string $icon, string $label, string $status, string $tone = 'is-ok', ?string $link = null, ?string $link_label = null, ?array $tooltip = null) {
        ?>
        <div class="shadowscan-status-row <?php echo esc_attr($tone); ?>">
            <span class="dashicons <?php echo esc_attr($icon); ?>" aria-hidden="true"></span>
            <div>
                <div class="shadowscan-status-text">
                    <?php echo esc_html($label); ?>
                    <?php if (!empty($tooltip['title']) && !empty($tooltip['body'])): ?>
                        <details class="shadowscan-inline-tooltip" style="display:inline-block;margin-left:6px;">
                            <summary aria-label="<?php echo esc_attr__('Protection status help', 'shadowscan-security-link'); ?>" style="display:inline-flex;align-items:center;cursor:pointer;">
                                <span class="dashicons dashicons-info-outline" aria-hidden="true"></span>
                            </summary>
                            <div class="shadowscan-tooltip-card" style="margin-top:6px;max-width:260px;">
                                <div style="font-weight:600;"><?php echo esc_html($tooltip['title']); ?></div>
                                <div class="shadowscan-muted" style="margin-top:4px;"><?php echo esc_html($tooltip['body']); ?></div>
                            </div>
                        </details>
                    <?php endif; ?>
                </div>
                <div class="shadowscan-muted"><?php echo esc_html($status); ?></div>
            </div>
            <?php if ($link): ?>
                <a class="button button-small" href="<?php echo esc_url($link); ?>">
                    <?php echo esc_html($link_label ?: __('View details', 'shadowscan-security-link')); ?>
                </a>
            <?php endif; ?>
        </div>
        <?php
    };

    $render_section = function (string $title, string $description, callable $body) {
        ?>
        <section class="shadowscan-section">
            <h2><?php echo esc_html($title); ?></h2>
            <p class="shadowscan-note"><?php echo esc_html($description); ?></p>
            <div class="shadowscan-divider"></div>
            <?php $body(); ?>
        </section>
        <?php
    };

    $render_access_control_section = function () use ($render_section, $render_status_row, $access_control_state) {
        $admin_state = $access_control_state['admin_capabilities'] ?? array();
        $rest_state = $access_control_state['rest_routes'] ?? array();
        $xmlrpc_state = $access_control_state['xmlrpc'] ?? array();
        $file_state = $access_control_state['file_mods'] ?? array();
        $abuse_state = $access_control_state['admin_endpoint_abuse'] ?? array();
        $events = $access_control_state['subcontrol_events'] ?? array();

        $admin_count = (int) ($admin_state['admin_count'] ?? 0);
        $admin_username_flag = !empty($admin_state['admin_username_flag']);
        $high_roles = isset($admin_state['high_risk_roles']) && is_array($admin_state['high_risk_roles']) ? $admin_state['high_risk_roles'] : array();
        $insecure_rest = (int) ($rest_state['insecure_count'] ?? 0);
        $xmlrpc_enabled = !empty($xmlrpc_state['enabled']);
        $xmlrpc_multicall = !empty($xmlrpc_state['multicall']);
        $file_edit_allowed = !empty($file_state['file_edit_allowed']);
        $file_mods_allowed = !empty($file_state['file_mods_allowed']);
        $abuse_count = (int) ($abuse_state['count'] ?? 0);

        $external_label = shadowscan_coverage_label_external();
        $admin_status = 'Covered';
        $admin_tone = 'is-ok';
        if ($admin_count > 5 || $admin_username_flag) {
            $admin_status = 'Action required';
            $admin_tone = 'is-bad';
        } elseif ($admin_count > 2 || !empty($high_roles)) {
            $admin_status = $external_label;
            $admin_tone = 'is-warn';
        }

        $priv_event = $events['privilege_escalation'] ?? array();
        $priv_status = 'Covered';
        $priv_tone = 'is-ok';
        if (!empty($priv_event['last_event_at'])) {
            $priv_status = 'Action required';
            $priv_tone = 'is-bad';
        }

        $rest_status = $insecure_rest > 0 ? $external_label : 'Covered';
        $rest_tone = $insecure_rest > 0 ? 'is-warn' : 'is-ok';

        $xmlrpc_status = $xmlrpc_enabled ? ($xmlrpc_multicall ? 'Action required' : $external_label) : 'Covered';
        $xmlrpc_tone = $xmlrpc_enabled ? ($xmlrpc_multicall ? 'is-bad' : 'is-warn') : 'is-ok';

        $abuse_status = $abuse_count >= 20 ? 'Action required' : ($abuse_count >= 10 ? $external_label : 'Covered');
        $abuse_tone = $abuse_count >= 20 ? 'is-bad' : ($abuse_count >= 10 ? 'is-warn' : 'is-ok');

        $file_status = ($file_edit_allowed || $file_mods_allowed) ? $external_label : 'Covered';
        $file_tone = ($file_edit_allowed || $file_mods_allowed) ? 'is-warn' : 'is-ok';

        $overall = 'Covered';
        if (in_array('Action required', array($admin_status, $priv_status, $xmlrpc_status, $abuse_status), true)) {
            $overall = 'Action required';
        } elseif (in_array($external_label, array($admin_status, $rest_status, $xmlrpc_status, $abuse_status, $file_status), true)) {
            $overall = $external_label;
        }

        $render_section(
            __('Access protection signals', 'shadowscan-security-link'),
            __('Scope: WordPress users, roles, REST, XML-RPC, admin endpoints, and file editing. This does not cover hosting panels, SSH/FTP, CDNs, or email accounts.', 'shadowscan-security-link'),
            function () use ($render_status_row, $overall, $admin_status, $admin_tone, $priv_status, $priv_tone, $rest_status, $rest_tone, $xmlrpc_status, $xmlrpc_tone, $abuse_status, $abuse_tone, $file_status, $file_tone) {
                ?>
                <p class="shadowscan-note"><strong><?php echo esc_html__('Overall state:', 'shadowscan-security-link'); ?></strong> <?php echo esc_html($overall); ?></p>
                <div class="shadowscan-status-list" style="margin-top: 10px;">
                    <?php
                    $render_status_row('dashicons-admin-users', __('Admin capability hardening', 'shadowscan-security-link'), $admin_status, $admin_tone);
                    $render_status_row('dashicons-shield-alt', __('Privilege escalation detection', 'shadowscan-security-link'), $priv_status, $priv_tone);
                    $render_status_row('dashicons-rest-api', __('REST API guardrails', 'shadowscan-security-link'), $rest_status, $rest_tone);
                    $render_status_row('dashicons-networking', __('XML-RPC access control', 'shadowscan-security-link'), $xmlrpc_status, $xmlrpc_tone);
                    $render_status_row('dashicons-admin-network', __('Admin endpoint signals', 'shadowscan-security-link'), $abuse_status, $abuse_tone);
                    $render_status_row('dashicons-admin-tools', __('File editor & install controls', 'shadowscan-security-link'), $file_status, $file_tone);
                    ?>
                </div>
                <div class="shadowscan-divider" style="margin-top: 12px;"></div>
                <p class="shadowscan-note"><strong><?php echo esc_html__('What this means:', 'shadowscan-security-link'); ?></strong> <?php echo esc_html__('ShadowScan watches for risky access-control conditions and alerts when they appear.', 'shadowscan-security-link'); ?></p>
                <p class="shadowscan-note"><strong><?php echo esc_html__('What ShadowScan does:', 'shadowscan-security-link'); ?></strong> <?php echo esc_html__('Detects and reports access-control risks. Some protections can be enabled manually.', 'shadowscan-security-link'); ?></p>
                <p class="shadowscan-note"><strong><?php echo esc_html__('What the site owner controls:', 'shadowscan-security-link'); ?></strong> <?php echo esc_html__('Admin roles, REST permissions, XML-RPC usage, and file editing settings.', 'shadowscan-security-link'); ?></p>
                <?php
            }
        );
    };

    $render_accordion = function (string $title, callable $body, bool $open = false) {
        ?>
        <details class="shadowscan-accordion" <?php echo $open ? 'open' : ''; ?> aria-label="<?php echo esc_attr($title); ?>">
            <summary><?php echo esc_html($title); ?></summary>
            <div style="margin-top: 12px;">
                <?php $body(); ?>
            </div>
        </details>
        <?php
    };

    $render_lock_note = function (string $message, bool $show_billing_cta = true) use ($billing_url) {
        ?>
        <p class="shadowscan-note shadowscan-lock">
            <span class="dashicons dashicons-lock" aria-hidden="true"></span>
            <span><?php echo esc_html($message); ?></span>
            <?php if ($show_billing_cta): ?>
                <a class="shadowscan-muted-link" href="<?php echo esc_url($billing_url); ?>" target="_blank" rel="noreferrer">
                    <?php echo esc_html__('Fix billing', 'shadowscan-security-link'); ?>
                </a>
            <?php endif; ?>
        </p>
        <?php
    };

    $advanced_readonly = false;
    $readonly_note_message = __('Changes require an active subscription. Status remains visible.', 'shadowscan-security-link');

    $lock_hardening = !ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_HARDENING, $subscription_state);
    $lock_headers = !ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_HEADERS, $subscription_state);
    $lock_safe_mode = !ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_SAFE_MODE, $subscription_state);
    $lock_htaccess = !ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_HTACCESS, $subscription_state);
    $lock_remote = !ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_REMOTE_DIAGNOSTICS, $subscription_state);
    $lock_emergency = !ShadowScan_Subscription_State::is_feature_allowed(ShadowScan_Subscription_State::FEATURE_EMERGENCY, $subscription_state);
    $lock_live = $protection_state !== 'live';
    $lock_note_message = __('Managed enforcement is paused due to billing. Status remains visible.', 'shadowscan-security-link');
    $readonly_note_message = __('Changes require an active subscription. Status remains visible.', 'shadowscan-security-link');
    $lock_hardening_ui = !$capabilities['can_configure_hardening'];
    $lock_headers_ui = !$capabilities['can_configure_headers'];
    $lock_safe_mode_ui = !$capabilities['can_use_safe_mode'];
    $lock_htaccess_ui = !$capabilities['can_configure_server'];
    $lock_remote_ui = !$capabilities['can_configure_remote'];
    $lock_emergency_ui = !$capabilities['can_use_emergency'];

    $render_connection_body = function () use ($connected, $format_time, $last_ping, $pairing_code, $refresh_cooldown_remaining, $connection_guidance) {
        ?>
        <p class="shadowscan-note">
            <?php echo esc_html($connected ? __('Connected', 'shadowscan-security-link') : __('Not connected yet', 'shadowscan-security-link')); ?>
            <?php if ($last_ping): ?>
                · <?php /* translators: %s: relative heartbeat time. */ ?>
                <?php echo esc_html(sprintf(__('Last heartbeat %s', 'shadowscan-security-link'), $format_time($last_ping))); ?>
            <?php endif; ?>
        </p>
        <?php if (!empty($connection_guidance)): ?>
            <div class="shadowscan-warning" style="margin-top: 12px;">
                <p style="margin:0 0 6px;font-weight:600;"><?php echo esc_html($connection_guidance['title']); ?></p>
                <p style="margin:0 0 6px;"><?php echo esc_html($connection_guidance['summary']); ?></p>
                <ul style="margin:0 0 8px 18px;">
                    <?php foreach ($connection_guidance['causes'] as $cause): ?>
                        <li><?php echo esc_html($cause); ?></li>
                    <?php endforeach; ?>
                </ul>
                <div class="shadowscan-actions">
                    <form method="post" style="margin:0;">
                        <?php wp_nonce_field('shadowscan-dashboard'); ?>
                        <input type="hidden" name="shadowscan_action" value="refresh_status" />
                        <button class="button" <?php echo $refresh_cooldown_remaining > 0 ? 'disabled' : ''; ?>>
                            <?php echo esc_html__('Retry connection', 'shadowscan-security-link'); ?>
                        </button>
                    </form>
                    <button type="button" class="button shadowscan-copy-report" data-copy-target="#shadowscan-status-report" data-copy-default="<?php echo esc_attr__('Copy support bundle', 'shadowscan-security-link'); ?>" data-copy-copied="<?php echo esc_attr__('Copied!', 'shadowscan-security-link'); ?>">
                        <?php echo esc_html__('Copy support bundle', 'shadowscan-security-link'); ?>
                    </button>
                    <a class="button" href="<?php echo esc_url(SHADOWSCAN_SUPPORT_URL); ?>"><?php echo esc_html__('Contact support', 'shadowscan-security-link'); ?></a>
                </div>
            </div>
        <?php endif; ?>
        <form method="post" style="margin-top: 12px;">
            <?php wp_nonce_field('shadowscan-dashboard'); ?>
            <input type="hidden" name="shadowscan_action" value="complete_pairing" />
            <label for="pairing_code"><?php echo esc_html__('Pairing code', 'shadowscan-security-link'); ?></label>
            <input type="text" id="pairing_code" name="pairing_code" value="<?php echo esc_attr($pairing_code); ?>" placeholder="XXXXXXXX" />
            <p class="shadowscan-note" style="margin-top: 6px;"><?php echo esc_html__('Copy the pairing code from the ShadowScan portal and paste it here.', 'shadowscan-security-link'); ?></p>
            <div class="shadowscan-actions">
                <button class="button button-primary"><?php echo esc_html__('Connect', 'shadowscan-security-link'); ?></button>
                <a href="<?php echo esc_url(SHADOWSCAN_PORTAL_URL); ?>" target="_blank" rel="noreferrer" class="button">
                    <?php echo esc_html__('Open portal', 'shadowscan-security-link'); ?>
                </a>
            </div>
        </form>
        <form method="post" style="margin-top: 12px;">
            <?php wp_nonce_field('shadowscan-dashboard'); ?>
            <input type="hidden" name="shadowscan_action" value="refresh_status" />
            <button class="button" <?php echo $refresh_cooldown_remaining > 0 ? 'disabled' : ''; ?>>
                <?php echo esc_html__('Check status', 'shadowscan-security-link'); ?>
            </button>
            <?php if ($refresh_cooldown_remaining > 0): ?>
                <p class="shadowscan-note" style="margin-top: 8px;">
                    <?php /* translators: %d: seconds remaining. */ ?>
                    <?php echo esc_html(sprintf(__('Please wait %d seconds before checking again.', 'shadowscan-security-link'), $refresh_cooldown_remaining)); ?>
                </p>
            <?php endif; ?>
        </form>
        <?php
    };

    $render_connection_section = function () use ($render_section, $render_connection_body) {
        $render_section(
            __('Onboarding & Connection', 'shadowscan-security-link'),
            __('Pair this site with ShadowScan to start monitoring and receive status updates.', 'shadowscan-security-link'),
            $render_connection_body
        );
    };

    $render_mfa_body = function () use ($mfa_enforced, $mfa_nudge_enabled, $mfa_grace_hours, $mfa_coverage, $render_lock_note, $lock_note_message, $capabilities) {
        $admin_total = (int) ($mfa_coverage['admin_total'] ?? 0);
        $admin_enrolled = (int) ($mfa_coverage['admin_enrolled'] ?? 0);
        $coverage_percent = (float) ($mfa_coverage['coverage_percent'] ?? 0);
        $provider = (string) ($mfa_coverage['provider'] ?? '');
        $coverage_label = $admin_total > 0
            ? sprintf('%d / %d (%.1f%%)', $admin_enrolled, $admin_total, $coverage_percent)
            : esc_html__('Unknown', 'shadowscan-security-link');
        $setup_url = class_exists('ShadowScan_MFA') ? ShadowScan_MFA::get_setup_url() : admin_url('profile.php');
        ?>
        <p class="shadowscan-note">
            <?php echo esc_html__('Provider:', 'shadowscan-security-link'); ?>
            <?php echo esc_html($provider !== '' ? $provider : esc_html__('Unknown', 'shadowscan-security-link')); ?>
            · <?php echo esc_html__('Admins with MFA:', 'shadowscan-security-link'); ?>
            <?php echo esc_html($coverage_label); ?>
        </p>
        <p class="shadowscan-note">
            <?php echo esc_html__('Enforcement will restrict wp-admin access until MFA is set up.', 'shadowscan-security-link'); ?>
        </p>
        <?php if ($setup_url !== ''): ?>
            <p class="shadowscan-note">
                <a class="button" href="<?php echo esc_url($setup_url); ?>">
                    <?php echo esc_html__('Set up MFA', 'shadowscan-security-link'); ?>
                </a>
            </p>
        <?php endif; ?>
        <form method="post" style="margin-top: 12px;">
            <?php wp_nonce_field('shadowscan-dashboard'); ?>
            <input type="hidden" name="shadowscan_action" value="save_mfa_settings" />
            <div class="shadowscan-toggle">
                <input type="checkbox" id="mfa_nudge_enabled" name="mfa_nudge_enabled" value="1" <?php checked($mfa_nudge_enabled); ?> />
                <label for="mfa_nudge_enabled"><?php echo esc_html__('Show MFA setup prompt after login', 'shadowscan-security-link'); ?></label>
            </div>
            <?php if (!$capabilities['can_enforce_mfa']): ?>
                <?php $render_lock_note($lock_note_message, false); ?>
                <p class="shadowscan-note"><?php echo esc_html__('Not available in current profile.', 'shadowscan-security-link'); ?></p>
            <?php endif; ?>
            <fieldset <?php echo !$capabilities['can_enforce_mfa'] ? 'disabled' : ''; ?>>
                <div class="shadowscan-toggle" style="margin-top: 10px;">
                    <input type="checkbox" id="mfa_enforce_admins" name="mfa_enforce_admins" value="1" <?php checked($mfa_enforced); ?> />
                    <label for="mfa_enforce_admins"><?php echo esc_html__('Enforce MFA for administrators', 'shadowscan-security-link'); ?></label>
                </div>
                <?php if ($mfa_enforced): ?>
                    <div style="margin-top: 10px;">
                        <label for="mfa_grace_hours"><?php echo esc_html__('Grace period (hours)', 'shadowscan-security-link'); ?></label>
                        <input type="number" min="0" id="mfa_grace_hours" name="mfa_grace_hours" value="<?php echo esc_attr((string) $mfa_grace_hours); ?>" />
                        <p class="shadowscan-note" style="margin-top: 6px;">
                            <?php echo esc_html__('Admins can access wp-admin during the grace period before enforcement begins.', 'shadowscan-security-link'); ?>
                        </p>
                    </div>
                <?php endif; ?>
            </fieldset>
            <div class="shadowscan-actions">
                <button class="button button-primary"><?php echo esc_html__('Save settings', 'shadowscan-security-link'); ?></button>
            </div>
        </form>
        <?php
    };

    $render_mfa_section = function () use ($render_section, $render_mfa_body) {
        $render_section(
            __('MFA', 'shadowscan-security-link'),
            __('Encourage MFA enrollment and enforce admin protection when ready.', 'shadowscan-security-link'),
            $render_mfa_body
        );
    };

    $render_autoupdate_body = function () use ($autoupdate_status, $render_lock_note, $lock_note_message, $capabilities) {
        $core_status = class_exists('ShadowScan_AutoUpdates') ? ShadowScan_AutoUpdates::get_core_status() : array();
        $enabled = !empty($autoupdate_status['enabled']);
        $mode = $autoupdate_status['mode'] ?? 'allowlist';
        $allowlist = is_array($autoupdate_status['allowlist'] ?? null) ? $autoupdate_status['allowlist'] : array();
        $core_enabled = !empty($core_status['desired_enabled']);
        $core_configurable = !empty($core_status['configurable']);

        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        ?>
        <div style="background: #f0f9ff; border-left: 4px solid #0284c7; padding: 12px 16px; margin-bottom: 16px; border-radius: 4px;">
            <p style="margin: 0 0 8px 0; font-weight: 600; color: #0c4a6e;">
                <?php echo esc_html__('Settings managed from ShadowScan Portal', 'shadowscan-security-link'); ?>
            </p>
            <p style="margin: 0; font-size: 13px; color: #475569;">
                <?php
                $site_id = shadowscan_get_site_id();
                if ($site_id) {
                    $portal_settings_url = SHADOWSCAN_PORTAL_URL . '/domains/' . urlencode($site_id) . '/settings';
                    echo wp_kses_post(
                        sprintf(
                            // translators: %s: Portal settings URL
                            __('Auto-update settings are configured in the <a href="%s" target="_blank" rel="noopener" style="color: #0284c7; text-decoration: underline;">ShadowScan Portal</a>. Changes sync to your site automatically within 10 minutes.', 'shadowscan-security-link'),
                            esc_url($portal_settings_url)
                        )
                    );
                } else {
                    echo esc_html__('Auto-update settings are configured in the ShadowScan Portal. Changes sync to your site automatically.', 'shadowscan-security-link');
                }
                ?>
            </p>
        </div>
        <p class="shadowscan-note"><?php echo esc_html__('Auto-updates can cause compatibility issues. Prefer allowlist mode.', 'shadowscan-security-link'); ?></p>
        <?php if (!$capabilities['can_configure_autoupdate']): ?>
            <?php $render_lock_note($lock_note_message); ?>
            <p class="shadowscan-note"><?php echo esc_html__('Not available in current profile.', 'shadowscan-security-link'); ?></p>
        <?php endif; ?>
        <form method="post">
            <?php wp_nonce_field('shadowscan-dashboard'); ?>
            <input type="hidden" name="shadowscan_action" value="save_autoupdate_settings" />
            <fieldset disabled>
            <div class="shadowscan-toggle">
                <input type="checkbox" id="autoupdate_plugins_enabled" name="autoupdate_plugins_enabled" value="1" <?php checked($enabled); ?> />
                <label for="autoupdate_plugins_enabled"><?php echo esc_html__('Enable plugin auto-updates (themes excluded)', 'shadowscan-security-link'); ?></label>
            </div>
            <div style="margin-top: 10px;">
                <label for="autoupdate_plugins_mode"><?php echo esc_html__('Mode', 'shadowscan-security-link'); ?></label>
                <select id="autoupdate_plugins_mode" name="autoupdate_plugins_mode">
                    <option value="allowlist" <?php selected($mode, 'allowlist'); ?>><?php echo esc_html__('Allowlist (recommended)', 'shadowscan-security-link'); ?></option>
                    <option value="all" <?php selected($mode, 'all'); ?>><?php echo esc_html__('All plugins', 'shadowscan-security-link'); ?></option>
                </select>
            </div>
            <div style="margin-top: 12px;">
                <p class="shadowscan-note"><?php echo esc_html__('Allowlisted plugins', 'shadowscan-security-link'); ?></p>
                <div style="max-height: 220px; overflow:auto; border:1px solid rgba(148,163,184,0.2); padding:10px; border-radius:8px;">
                    <?php if (!empty($plugins) && is_array($plugins)): ?>
                        <?php foreach ($plugins as $plugin_file => $plugin_data): ?>
                            <label style="display:flex; gap:8px; align-items:center; margin-bottom:6px;">
                                <input type="checkbox" name="autoupdate_plugins_allowlist[]" value="<?php echo esc_attr($plugin_file); ?>" <?php checked(in_array($plugin_file, $allowlist, true)); ?> />
                                <span><?php echo esc_html($plugin_data['Name'] ?? $plugin_file); ?></span>
                            </label>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p class="shadowscan-note"><?php echo esc_html__('No plugins detected.', 'shadowscan-security-link'); ?></p>
                    <?php endif; ?>
                </div>
            </div>
            <div class="shadowscan-toggle" style="margin-top: 12px;">
                <input type="checkbox" id="autoupdate_core_enabled" name="autoupdate_core_enabled" value="1" <?php checked($core_enabled); ?> <?php echo $core_configurable ? '' : 'disabled'; ?> />
                <label for="autoupdate_core_enabled"><?php echo esc_html__('Enable WordPress core auto-updates (recommended for security)', 'shadowscan-security-link'); ?></label>
            </div>
            <p class="shadowscan-note" style="margin-top: 6px;">
                <?php echo esc_html__('Core updates can occasionally cause compatibility issues. If your host already manages core updates, leave this off.', 'shadowscan-security-link'); ?>
            </p>
            <?php if (!$core_configurable): ?>
                <p class="shadowscan-note" style="margin-top: 6px;">
                    <?php echo esc_html__('Host-managed/locked. ShadowScan cannot enforce here.', 'shadowscan-security-link'); ?>
                </p>
            <?php endif; ?>
            </fieldset>
        </form>
        <?php
    };

    $render_autoupdate_section = function () use ($render_section, $render_autoupdate_body) {
        $render_section(
            __('Auto-Updates', 'shadowscan-security-link'),
            __('Enable plugin auto-updates while keeping themes excluded. Auto-updates apply to active plugins only.', 'shadowscan-security-link'),
            $render_autoupdate_body
        );
    };

    $render_password_policy_body = function () use ($password_policy_status, $render_lock_note, $lock_note_message, $capabilities) {
        $min_length = (int) ($password_policy_status['min_length'] ?? 12);
        $enforce_min = !empty($password_policy_status['enforce_min_length']);
        $enforce_breach = !empty($password_policy_status['enforce_breached']);
        $roles = $password_policy_status['roles'] ?? array('administrator');
        $provider = $password_policy_status['breach_provider'] ?? 'hibp';
        $email_notice_enabled = !empty($password_policy_status['email_notice_enabled']);
        $editable_roles = get_editable_roles();
        $role_labels = array(
            'administrator' => __('Administrators', 'shadowscan-security-link'),
            'editor' => __('Editors', 'shadowscan-security-link'),
            'shop_manager' => __('Shop Managers', 'shadowscan-security-link'),
        );
        ?>
        <div style="background: #f0f9ff; border-left: 4px solid #0284c7; padding: 12px 16px; margin-bottom: 16px; border-radius: 4px;">
            <p style="margin: 0 0 8px 0; font-weight: 600; color: #0c4a6e;">
                <?php echo esc_html__('Settings managed from ShadowScan Portal', 'shadowscan-security-link'); ?>
            </p>
            <p style="margin: 0; font-size: 13px; color: #475569;">
                <?php
                $site_id = shadowscan_get_site_id();
                if ($site_id) {
                    $portal_settings_url = SHADOWSCAN_PORTAL_URL . '/domains/' . urlencode($site_id) . '/settings';
                    echo wp_kses_post(
                        sprintf(
                            // translators: %s: Portal settings URL
                            __('Password policy settings are configured in the <a href="%s" target="_blank" rel="noopener" style="color: #0284c7; text-decoration: underline;">ShadowScan Portal</a>. Changes sync to your site automatically within 10 minutes.', 'shadowscan-security-link'),
                            esc_url($portal_settings_url)
                        )
                    );
                } else {
                    echo esc_html__('Password policy settings are configured in the ShadowScan Portal. Changes sync to your site automatically.', 'shadowscan-security-link');
                }
                ?>
            </p>
        </div>
        <p class="shadowscan-note"><?php echo esc_html__('This control applies when a user changes or resets a password. When enabled, in-scope users must update their password within 14 days of activation (enforced at next login).', 'shadowscan-security-link'); ?></p>
        <?php if ($enforce_breach): ?>
            <p class="shadowscan-note"><?php echo esc_html__('Breached-password blocking checks the chosen password against a breach database when enabled.', 'shadowscan-security-link'); ?></p>
        <?php endif; ?>
        <?php if (!$capabilities['can_configure_password']): ?>
            <?php $render_lock_note($lock_note_message); ?>
            <p class="shadowscan-note"><?php echo esc_html__('Not available in current profile.', 'shadowscan-security-link'); ?></p>
        <?php endif; ?>
        <form method="post">
            <?php wp_nonce_field('shadowscan-dashboard'); ?>
            <input type="hidden" name="shadowscan_action" value="save_password_policy" />
            <fieldset disabled>
            <div style="margin-top: 6px;">
                <label for="password_min_length"><?php echo esc_html__('Minimum length', 'shadowscan-security-link'); ?></label>
                <input type="number" min="8" id="password_min_length" name="password_min_length" value="<?php echo esc_attr((string) $min_length); ?>" />
            </div>
            <div class="shadowscan-toggle" style="margin-top: 10px;">
                <input type="checkbox" id="password_enforce_min_length" name="password_enforce_min_length" value="1" <?php checked($enforce_min); ?> />
                <label for="password_enforce_min_length"><?php echo esc_html__('Enforce minimum length', 'shadowscan-security-link'); ?></label>
            </div>
            <div class="shadowscan-toggle" style="margin-top: 10px;">
                <input type="checkbox" id="password_enforce_breached" name="password_enforce_breached" value="1" <?php checked($enforce_breach); ?> />
                <label for="password_enforce_breached"><?php echo esc_html__('Block breached passwords', 'shadowscan-security-link'); ?></label>
            </div>
            <div style="margin-top: 12px;">
                <label><?php echo esc_html__('Roles in scope', 'shadowscan-security-link'); ?></label>
                <div class="shadowscan-chip-row">
                    <?php foreach ($role_labels as $role_key => $role_label): ?>
                        <?php if (!isset($editable_roles[$role_key])) { continue; } ?>
                        <label style="display:flex;gap:6px;align-items:center;">
                            <input type="checkbox" name="password_enforce_roles[]" value="<?php echo esc_attr($role_key); ?>" <?php checked(in_array($role_key, $roles, true)); ?> />
                            <span><?php echo esc_html($role_label); ?></span>
                        </label>
                    <?php endforeach; ?>
                </div>
            </div>
            <div style="margin-top: 10px;">
                <label for="password_breach_provider"><?php echo esc_html__('Breach provider', 'shadowscan-security-link'); ?></label>
                <select id="password_breach_provider" name="password_breach_provider">
                    <option value="hibp" <?php selected($provider, 'hibp'); ?>><?php echo esc_html__('HIBP (k-anonymity)', 'shadowscan-security-link'); ?></option>
                    <option value="none" <?php selected($provider, 'none'); ?>><?php echo esc_html__('None', 'shadowscan-security-link'); ?></option>
                </select>
            </div>
            <div class="shadowscan-toggle" style="margin-top: 10px;">
                <input type="checkbox" id="password_policy_email_notice" name="password_policy_email_notice" value="1" <?php checked($email_notice_enabled); ?> />
                <label for="password_policy_email_notice"><?php echo esc_html__('Email in-scope admins when policy is enabled', 'shadowscan-security-link'); ?></label>
            </div>
            </fieldset>
        </form>
        <?php
    };

    $render_password_policy_section = function () use ($render_section, $render_password_policy_body) {
        $render_section(
            __('Password policy (change/reset)', 'shadowscan-security-link'),
            __('Enforces minimum password length when passwords are changed or reset. Optional breached-password blocking.', 'shadowscan-security-link'),
            $render_password_policy_body
        );
    };

    $render_geo_policy_body = function () use ($geo_policy_status, $render_lock_note, $lock_note_message, $capabilities) {
        $mode = $geo_policy_status['mode'] ?? 'off';
        $countries = $geo_policy_status['countries'] ?? array();
        $countries_text = is_array($countries) ? implode(', ', $countries) : '';
        ?>
        <div style="background: #f0f9ff; border-left: 4px solid #0284c7; padding: 12px 16px; margin-bottom: 16px; border-radius: 4px;">
            <p style="margin: 0 0 8px 0; font-weight: 600; color: #0c4a6e;">
                <?php echo esc_html__('Settings managed from ShadowScan Portal', 'shadowscan-security-link'); ?>
            </p>
            <p style="margin: 0; font-size: 13px; color: #475569;">
                <?php
                $site_id = shadowscan_get_site_id();
                if ($site_id) {
                    $portal_access_policy_url = SHADOWSCAN_PORTAL_URL . '/domains/' . urlencode($site_id) . '/access-policy';
                    echo wp_kses_post(
                        sprintf(
                            // translators: %s: Portal access policy URL
                            __('Admin access location rules are configured in the <a href="%s" target="_blank" rel="noopener" style="color: #0284c7; text-decoration: underline;">ShadowScan Portal</a>. Changes sync to your site automatically within 10 minutes.', 'shadowscan-security-link'),
                            esc_url($portal_access_policy_url)
                        )
                    );
                } else {
                    echo esc_html__('Admin access location rules are configured in the ShadowScan Portal. Changes sync to your site automatically.', 'shadowscan-security-link');
                }
                ?>
            </p>
        </div>
        <p class="shadowscan-note"><?php echo esc_html__('Edge enforcement is recommended. WP admin-only enforcement is a fallback.', 'shadowscan-security-link'); ?></p>
        <?php if (!$capabilities['can_enforce_geo']): ?>
            <?php $render_lock_note($lock_note_message); ?>
            <p class="shadowscan-note"><?php echo esc_html__('Not available in current profile.', 'shadowscan-security-link'); ?></p>
        <?php endif; ?>
        <form method="post" data-geo-config-form="1">
            <?php wp_nonce_field('shadowscan-dashboard'); ?>
            <input type="hidden" name="shadowscan_action" value="save_geo_policy" />
            <fieldset disabled>
            <div class="shadowscan-config-field">
                <label><?php echo esc_html__('Mode', 'shadowscan-security-link'); ?></label>
                <div class="shadowscan-muted"><?php echo esc_html__('Controls whether geo restrictions are off, monitored, or enforced on wp-admin.', 'shadowscan-security-link'); ?></div>
                <div class="shadowscan-radio-group" role="group" aria-label="<?php echo esc_attr__('Geo mode', 'shadowscan-security-link'); ?>">
                    <label class="<?php echo $mode === 'off' ? 'is-selected' : ''; ?>">
                        <input type="radio" name="geo_block_mode" value="off" <?php checked($mode, 'off'); ?> />
                        <?php echo esc_html__('Off', 'shadowscan-security-link'); ?>
                    </label>
                    <label class="<?php echo $mode === 'monitor' ? 'is-selected' : ''; ?>">
                        <input type="radio" name="geo_block_mode" value="monitor" <?php checked($mode, 'monitor'); ?> />
                        <?php echo esc_html__('Observe', 'shadowscan-security-link'); ?>
                    </label>
                    <label class="<?php echo $mode === 'enforce_wp_admin' ? 'is-selected' : ''; ?>">
                        <input type="radio" name="geo_block_mode" value="enforce_wp_admin" <?php checked($mode, 'enforce_wp_admin'); ?><?php echo !$capabilities['can_enforce_geo'] ? ' disabled' : ''; ?> />
                        <?php echo esc_html__('Enforce', 'shadowscan-security-link'); ?>
                    </label>
                </div>
                <div class="shadowscan-muted" style="margin-top:8px;">
                    <?php echo esc_html__('Enforce applies only to wp-admin/wp-login and may be limited by hosting.', 'shadowscan-security-link'); ?>
                </div>
            </div>
            <div class="shadowscan-config-field">
                <label for="geo_blocked_countries"><?php echo esc_html__('Blocked countries (ISO 3166-1 alpha-2)', 'shadowscan-security-link'); ?></label>
                <div class="shadowscan-muted"><?php echo esc_html__('Comma or space separated (e.g., US, CN, RU).', 'shadowscan-security-link'); ?></div>
                <textarea id="geo_blocked_countries" name="geo_blocked_countries" class="shadowscan-code-textarea" rows="3" placeholder="US, CN, RU" data-geo-chips="countries"><?php echo esc_textarea($countries_text); ?></textarea>
                <div class="shadowscan-chip-row" data-geo-chips-target="countries"></div>
            </div>
            <div class="shadowscan-config-field">
                <label for="geo_allow_admin_ips"><?php echo esc_html__('Allowlisted admin IPs (optional)', 'shadowscan-security-link'); ?></label>
                <div class="shadowscan-muted"><?php echo esc_html__('Comma or space separated IPv4/IPv6. CIDR only if already used on this site.', 'shadowscan-security-link'); ?></div>
                <?php
                $allow_ips = ShadowScan_Geo_Policy::get_allow_ips();
                $allow_text = implode(', ', $allow_ips);
                $cidr_supported = false;
                foreach ($allow_ips as $ip_entry) {
                    if (strpos((string) $ip_entry, '/') !== false) {
                        $cidr_supported = true;
                        break;
                    }
                }
                ?>
                <textarea id="geo_allow_admin_ips" name="geo_allow_admin_ips" class="shadowscan-code-textarea" rows="2" placeholder="203.0.113.10, 203.0.113.11" data-geo-chips="ips" data-cidr-supported="<?php echo $cidr_supported ? '1' : '0'; ?>"><?php echo esc_textarea($allow_text); ?></textarea>
                <div class="shadowscan-chip-row" data-geo-chips-target="ips"></div>
            </div>
            </fieldset>
        </form>
        <?php
    };

    $render_geo_policy_section = function () use ($render_section, $render_geo_policy_body) {
        $render_section(
            __('Geo restrictions', 'shadowscan-security-link'),
            __('Best enforced at the edge. Plugin enforcement is limited to wp-admin and wp-login.', 'shadowscan-security-link'),
            function () use ($render_geo_policy_body) {
                $render_geo_policy_body();
            }
        );
    };

    $render_hardening_section = function () use ($render_section, $enumeration_state, $headers_state, $render_accordion, $ui_advanced, $lock_hardening_ui, $lock_headers_ui, $render_lock_note, $lock_note_message, $capabilities) {
        $render_section(
            __('Hardening', 'shadowscan-security-link'),
            __('Keep protections focused and reversible. Changes apply immediately.', 'shadowscan-security-link'),
            function () use ($enumeration_state, $headers_state, $render_accordion, $ui_advanced, $lock_hardening_ui, $lock_headers_ui, $render_lock_note, $lock_note_message, $capabilities) {
                ?>
                <?php if (!$capabilities['can_configure_hardening']): ?>
                    <?php $render_lock_note($lock_note_message); ?>
                <?php endif; ?>
                <form method="post">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="save_hardening" />
                    <fieldset <?php echo !$capabilities['can_configure_hardening'] ? 'disabled' : ''; ?>>
                    <div class="shadowscan-toggle">
                        <input type="checkbox" id="enum_strict" name="enum_strict" value="1" <?php checked(!empty($enumeration_state['disable_author_archives'])); ?> />
                        <label for="enum_strict"><?php echo esc_html__('Enable strict enumeration protection', 'shadowscan-security-link'); ?></label>
                    </div>
                    <?php if ($ui_advanced): ?>
                        <div style="margin-top: 14px;">
                            <?php $render_accordion(__('Security headers (advanced)', 'shadowscan-security-link'), function () use ($headers_state, $lock_headers_ui, $render_lock_note, $lock_note_message, $capabilities) { ?>
                                <?php if (!$capabilities['can_configure_headers']): ?>
                                    <?php $render_lock_note($lock_note_message); ?>
                                <?php endif; ?>
                                <fieldset <?php echo (!$capabilities['can_configure_headers'] || $lock_headers_ui) ? 'disabled' : ''; ?>>
                                    <div class="shadowscan-toggle">
                                        <input type="checkbox" id="x_frame_options" name="x_frame_options" value="1" <?php checked(empty($headers_state['x_frame_disabled'])); ?> />
                                        <label for="x_frame_options"><?php echo esc_html__('Enable X-Frame-Options', 'shadowscan-security-link'); ?></label>
                                    </div>
                                    <div class="shadowscan-toggle" style="margin-top: 10px;">
                                        <input type="checkbox" id="csp_report_only" name="csp_report_only" value="1" <?php checked(!empty($headers_state['csp_report_only'])); ?> />
                                        <label for="csp_report_only"><?php echo esc_html__('Enable CSP report-only', 'shadowscan-security-link'); ?></label>
                                    </div>
                                    <p class="shadowscan-note" style="margin-top: 10px;">
                                        <?php echo esc_html__('Some protections depend on hosting configuration. ShadowScan applies safe controls where supported.', 'shadowscan-security-link'); ?>
                                    </p>
                                </fieldset>
                            <?php }); ?>
                        </div>
                    <?php else: ?>
                        <p class="shadowscan-note" style="margin-top: 10px;">
                            <?php echo esc_html__('Advanced header controls are hidden.', 'shadowscan-security-link'); ?>
                        </p>
                    <?php endif; ?>
                    <div class="shadowscan-actions">
                        <button class="button button-primary"><?php echo esc_html__('Save settings', 'shadowscan-security-link'); ?></button>
                    </div>
                    </fieldset>
                </form>
                <?php
            }
        );
    };

    $render_guard_section = function () use ($render_section, $bruteforce_stats, $render_accordion, $ui_advanced) {
        $render_section(
            __('Guard Layer & Safe Mode', 'shadowscan-security-link'),
            __('Containment responds to repeated login abuse. Safe mode pauses enforcement temporarily.', 'shadowscan-security-link'),
            function () use ($bruteforce_stats, $render_accordion, $ui_advanced) {
                ?>
                <p class="shadowscan-note">
                    <?php echo esc_html__('Failed logins (last 10 minutes):', 'shadowscan-security-link'); ?>
                    <?php echo esc_html((string) ($bruteforce_stats['failed_last_window'] ?? 0)); ?>
                    · <?php echo esc_html__('Active locks:', 'shadowscan-security-link'); ?>
                    <?php echo esc_html((string) ($bruteforce_stats['active_user_locks'] ?? 0)); ?>
                    · <?php echo esc_html__('Active IP blocks:', 'shadowscan-security-link'); ?>
                    <?php echo esc_html((string) ($bruteforce_stats['active_ip_blocks'] ?? 0)); ?>
                </p>
                <form method="post" style="margin-top: 12px;">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="toggle_bf_safe_mode" />
                    <fieldset>
                    <div class="shadowscan-toggle">
                        <input type="checkbox" id="bf_safe_mode" name="bf_safe_mode" value="1" <?php checked(!empty($bruteforce_stats['safe_mode'])); ?> />
                        <label for="bf_safe_mode"><?php echo esc_html__('Enable safe mode (disable enforcement)', 'shadowscan-security-link'); ?></label>
                    </div>
                    <div class="shadowscan-actions">
                        <button class="button"><?php echo esc_html__('Save settings', 'shadowscan-security-link'); ?></button>
                    </div>
                    </fieldset>
                </form>
                <?php if ($ui_advanced): ?>
                    <div style="margin-top: 12px;">
                        <?php $render_accordion(__('Break-glass notes', 'shadowscan-security-link'), function () { ?>
                            <p class="shadowscan-note"><?php echo esc_html__('Define SHADOWSCAN_SAFE_MODE in wp-config.php to disable enforcement.', 'shadowscan-security-link'); ?></p>
                        <?php }); ?>
                    </div>
                <?php endif; ?>
                <?php
            }
        );
    };

    $render_server_section = function () use ($render_section, $htaccess_state, $render_accordion, $ui_advanced, $format_time, $lock_htaccess_ui, $render_lock_note, $lock_note_message, $auto_htaccess_enabled, $capabilities) {
        $render_section(
            __('Server compatibility', 'shadowscan-security-link'),
            __('Advanced server controls for compatibility baselines.', 'shadowscan-security-link'),
            function () use ($htaccess_state, $render_accordion, $ui_advanced, $format_time, $lock_htaccess_ui, $render_lock_note, $lock_note_message, $auto_htaccess_enabled, $capabilities) {
                ?>
                <?php if ($ui_advanced): ?>
                    <?php if (!$capabilities['can_configure_server']): ?>
                        <p class="shadowscan-note"><?php echo esc_html__('Available on active plans.', 'shadowscan-security-link'); ?></p>
                        <?php return; ?>
                    <?php endif; ?>
                    <?php $render_accordion(__('Server compatibility (advanced)', 'shadowscan-security-link'), function () use ($htaccess_state, $format_time, $lock_htaccess_ui, $render_lock_note, $lock_note_message, $auto_htaccess_enabled) { ?>
                        <?php if ($lock_htaccess_ui): ?>
                            <?php $render_lock_note($lock_note_message); ?>
                            <?php return; ?>
                        <?php endif; ?>
                        <p class="shadowscan-note">
                            <?php echo esc_html__('.htaccess supported:', 'shadowscan-security-link'); ?>
                            <?php echo !empty($htaccess_state['last_result']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('Unknown', 'shadowscan-security-link'); ?>
                        </p>
                        <p class="shadowscan-note">
                            <?php echo esc_html__('Last .htaccess apply:', 'shadowscan-security-link'); ?>
                            <?php echo !empty($htaccess_state['last_apply']) ? esc_html($format_time((int) $htaccess_state['last_apply'])) : esc_html__('Never', 'shadowscan-security-link'); ?>
                        </p>
                        <form method="post" style="margin-top: 12px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="toggle_auto_htaccess" />
                            <div class="shadowscan-toggle">
                                <input type="checkbox" id="auto_htaccess" name="auto_htaccess" value="1" <?php checked($auto_htaccess_enabled); ?> />
                                <label for="auto_htaccess"><?php echo esc_html__('Auto-apply htaccess baseline when eligible', 'shadowscan-security-link'); ?></label>
                            </div>
                            <p class="shadowscan-note" style="margin-top: 8px;">
                                <?php echo esc_html__('Applies the baseline once when the subscription is active and protection is live. Recommended for Apache hosts.', 'shadowscan-security-link'); ?>
                            </p>
                            <div class="shadowscan-actions">
                                <button class="button"><?php echo esc_html__('Save settings', 'shadowscan-security-link'); ?></button>
                            </div>
                        </form>
                        <form method="post" style="margin-top: 12px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="apply_htaccess_baseline" />
                            <button class="button"><?php echo esc_html__('Apply htaccess baseline', 'shadowscan-security-link'); ?></button>
                        </form>
                        <form method="post" style="margin-top: 8px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="rollback_htaccess" />
                            <button class="button"><?php echo esc_html__('Rollback htaccess changes', 'shadowscan-security-link'); ?></button>
                        </form>
                    <?php }); ?>
                <?php else: ?>
                    <p class="shadowscan-note"><?php echo esc_html__('Advanced server actions are hidden.', 'shadowscan-security-link'); ?></p>
                <?php endif; ?>
                <?php
            }
        );
    };

    $render_disconnect_section = function () {
        $billing_url = shadowscan_billing_cta_url();
        ?>
        <details class="shadowscan-accordion" aria-label="<?php echo esc_attr__('Offboarding & removal', 'shadowscan-security-link'); ?>">
            <summary><?php echo esc_html__('Offboarding & removal', 'shadowscan-security-link'); ?></summary>
            <div style="margin-top: 12px;">
                <p class="shadowscan-note" style="margin-top:0;">
                    <?php echo esc_html__('Disconnecting stops check-ins and portal control for this site. Billing and cancellation are handled in the ShadowScan portal.', 'shadowscan-security-link'); ?>
                </p>
                <div class="shadowscan-actions" style="margin-top: 10px;">
                    <a class="button" href="<?php echo esc_url($billing_url); ?>" target="_blank" rel="noreferrer">
                        <?php echo esc_html__('Open billing (cancel subscription)', 'shadowscan-security-link'); ?>
                    </a>
                </div>
                <p class="shadowscan-note" style="margin-top: 8px;">
                    <?php echo esc_html__('This plugin never cancels subscriptions or triggers cancellation fees.', 'shadowscan-security-link'); ?>
                </p>

                <details class="shadowscan-accordion" style="margin-top: 14px;">
                    <summary><?php echo esc_html__('Disconnect this site', 'shadowscan-security-link'); ?></summary>
                    <div style="margin-top: 12px;">
                        <p class="shadowscan-note" style="margin-top:0;">
                            <?php echo esc_html__('Disconnecting will stop status updates and monitoring sync. You can reconnect later using a new pairing code.', 'shadowscan-security-link'); ?>
                        </p>
                        <form method="post" style="margin-top: 12px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="disconnect_site" />
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="checkbox" name="disconnect_confirm" value="1" />
                                <span><?php echo esc_html__('I understand this will disconnect the site from ShadowScan.', 'shadowscan-security-link'); ?></span>
                            </label>
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="text" name="disconnect_phrase" placeholder="<?php echo esc_attr__('Type DISCONNECT to confirm', 'shadowscan-security-link'); ?>" />
                            </label>
                            <button class="button button-danger">
                                <?php echo esc_html__('Confirm disconnect', 'shadowscan-security-link'); ?>
                            </button>
                        </form>
                    </div>
                </details>

                <details class="shadowscan-accordion" style="margin-top: 12px;">
                    <summary><?php echo esc_html__('Remove local ShadowScan data', 'shadowscan-security-link'); ?></summary>
                    <div style="margin-top: 12px;">
                        <p class="shadowscan-note" style="margin-top:0;">
                            <?php echo esc_html__('Remove local ShadowScan data clears pairing tokens, cached subscription state, diagnostic logs, and UI preferences stored by the plugin.', 'shadowscan-security-link'); ?>
                            <a class="shadowscan-muted-link" href="<?php echo esc_url(SHADOWSCAN_DATA_HANDLING_URL); ?>" target="_blank" rel="noreferrer" style="margin-left:6px;">
                                <?php echo esc_html__('View data handling', 'shadowscan-security-link'); ?>
                            </a>
                        </p>
                        <form method="post" style="margin-top: 12px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="remove_local_data" />
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="checkbox" name="remove_confirm" value="1" />
                                <span><?php echo esc_html__('I understand this will clear local ShadowScan data on this site.', 'shadowscan-security-link'); ?></span>
                            </label>
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="text" name="remove_phrase" placeholder="<?php echo esc_attr__('Type REMOVE to confirm', 'shadowscan-security-link'); ?>" />
                            </label>
                            <button class="button">
                                <?php echo esc_html__('Confirm data removal', 'shadowscan-security-link'); ?>
                            </button>
                        </form>
                    </div>
                </details>
            </div>
        </details>
        <?php
    };

    $render_connection_actions_section = function () use ($render_section, $connected, $pairing_code) {
        $render_section(
            __('Connection actions', 'shadowscan-security-link'),
            __('Reconnect this site with a new pairing code, or explicitly disconnect it from ShadowScan.', 'shadowscan-security-link'),
            function () use ($connected, $pairing_code) {
                ?>
                <p class="shadowscan-note">
                    <?php echo esc_html__('Current state:', 'shadowscan-security-link'); ?>
                    <?php echo $connected ? esc_html__('Connected', 'shadowscan-security-link') : esc_html__('Not connected', 'shadowscan-security-link'); ?>
                </p>
                <form method="post" style="margin-top: 10px;">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="complete_pairing" />
                    <label for="shadowscan_reconnect_pairing_code"><?php echo esc_html__('Reconnect with pairing code', 'shadowscan-security-link'); ?></label>
                    <input type="text" id="shadowscan_reconnect_pairing_code" name="pairing_code" value="<?php echo esc_attr($pairing_code); ?>" placeholder="XXXXXXXX" />
                    <p class="shadowscan-note" style="margin-top: 6px;"><?php echo esc_html__('Copy the pairing code from the ShadowScan portal and paste it here.', 'shadowscan-security-link'); ?></p>
                    <div class="shadowscan-actions">
                        <button class="button button-primary"><?php echo esc_html__('Reconnect now', 'shadowscan-security-link'); ?></button>
                    </div>
                </form>
                <details class="shadowscan-accordion" style="margin-top: 14px;">
                    <summary><?php echo esc_html__('Disconnect this site', 'shadowscan-security-link'); ?></summary>
                    <div style="margin-top: 12px;">
                        <p class="shadowscan-note" style="margin-top:0;">
                            <?php echo esc_html__('Disconnecting stops status updates and portal control for this site. You can reconnect later with a new pairing code.', 'shadowscan-security-link'); ?>
                        </p>
                        <form method="post" style="margin-top: 12px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="disconnect_site" />
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="checkbox" name="disconnect_confirm" value="1" />
                                <span><?php echo esc_html__('I understand this will disconnect the site from ShadowScan.', 'shadowscan-security-link'); ?></span>
                            </label>
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="text" name="disconnect_phrase" placeholder="<?php echo esc_attr__('Type DISCONNECT to confirm', 'shadowscan-security-link'); ?>" />
                            </label>
                            <button class="button button-danger">
                                <?php echo esc_html__('Confirm disconnect', 'shadowscan-security-link'); ?>
                            </button>
                        </form>
                    </div>
                </details>
                <?php
            }
        );
    };

    $render_remote_diagnostics_section = function () use ($render_section, $remote_diagnostics_enabled, $remote_diagnostics_expires, $sentry_telemetry_enabled, $format_time, $capabilities, $render_lock_note, $readonly_note_message) {
        $render_section(
            __('Remote diagnostics', 'shadowscan-security-link'),
            __('Enable only while troubleshooting. Access auto-disables after 60 minutes.', 'shadowscan-security-link'),
            function () use ($remote_diagnostics_enabled, $remote_diagnostics_expires, $sentry_telemetry_enabled, $format_time, $capabilities, $render_lock_note, $readonly_note_message) {
                $remote_locked = !$capabilities['can_configure_remote'];
                if ($remote_locked) {
                    $render_lock_note($readonly_note_message);
                }
                ?>
                <form method="post" style="margin-top: 12px;">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="toggle_sentry_telemetry" />
                    <label style="display:flex;gap:10px;align-items:center;">
                        <input type="checkbox" name="sentry_telemetry" value="1" <?php checked($sentry_telemetry_enabled); ?> />
                        <span><?php echo esc_html__('Enable Sentry telemetry (disabled by default)', 'shadowscan-security-link'); ?></span>
                    </label>
                    <p class="shadowscan-note" style="margin-top: 6px;">
                        <?php echo esc_html__('Sends error metadata only after explicit opt-in. Disable any time to stop future telemetry.', 'shadowscan-security-link'); ?>
                    </p>
                    <div class="shadowscan-actions">
                        <button class="button"><?php echo esc_html__('Save telemetry setting', 'shadowscan-security-link'); ?></button>
                    </div>
                </form>
                <?php if ($remote_diagnostics_enabled && $remote_diagnostics_expires): ?>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Auto-disables at:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($format_time($remote_diagnostics_expires)); ?>
                    </p>
                <?php endif; ?>
                <form method="post" style="margin-top: 12px;">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="toggle_remote_diagnostics" />
                    <?php if (($remote_locked && !$remote_diagnostics_enabled) || !$sentry_telemetry_enabled): ?>
                        <fieldset disabled>
                    <?php endif; ?>
                    <label style="display:flex;gap:10px;align-items:center;">
                        <input type="checkbox" name="remote_diagnostics" value="1" <?php checked($remote_diagnostics_enabled); ?> />
                        <span><?php echo esc_html__('Enable remote diagnostics', 'shadowscan-security-link'); ?></span>
                    </label>
                    <div class="shadowscan-actions">
                        <button class="button"><?php echo esc_html__('Save settings', 'shadowscan-security-link'); ?></button>
                    </div>
                    <?php if (($remote_locked && !$remote_diagnostics_enabled) || !$sentry_telemetry_enabled): ?>
                        </fieldset>
                    <?php endif; ?>
                </form>
                <?php if (!$sentry_telemetry_enabled): ?>
                    <p class="shadowscan-note"><?php echo esc_html__('Enable Sentry telemetry first to allow remote diagnostics.', 'shadowscan-security-link'); ?></p>
                <?php endif; ?>
                <?php
            }
        );
    };

    $render_scope_section = function () use ($render_accordion) {
        $render_accordion(__('Scope & limits', 'shadowscan-security-link'), function () {
            ?>
            <ul class="shadowscan-note" style="list-style:disc;padding-left:18px;">
                <li><?php echo esc_html__('ShadowScan reduces risk by monitoring and applying protections through the portal.', 'shadowscan-security-link'); ?></li>
                <li><?php echo esc_html__('Security updates and configuration changes may pause when billing is not active.', 'shadowscan-security-link'); ?></li>
                <li><?php echo esc_html__('ShadowScan does not automatically delete site files.', 'shadowscan-security-link'); ?></li>
                <li><?php echo esc_html__('No system can guarantee prevention of all compromises.', 'shadowscan-security-link'); ?></li>
            </ul>
            <?php
        });
    };

    ?>
    <div class="shadowscan-app">
        <div class="wrap shadowscan-wrap">
        <h1>
            <img src="<?php echo esc_url(SHADOWSCAN_LOGO_URL, array('data')); ?>" alt="ShadowScan" style="height:40px;width:40px;border-radius:8px;background:#0b1221;padding:4px;border:1px solid #12233f;" />
            <span><?php echo esc_html__('ShadowScan Admin', 'shadowscan-security-link'); ?></span>
            <span class="shadowscan-badge"><?php echo esc_html__('Heartbeat connector', 'shadowscan-security-link'); ?></span>
        </h1>
        <?php if ($notice): ?>
            <?php
            $notice_class = 'notice-success';
            if ($notice_type === 'error') {
                $notice_class = 'notice-error';
            } elseif ($notice_type === 'warning') {
                $notice_class = 'notice-warning';
            }
            ?>
            <div class="notice <?php echo esc_attr($notice_class); ?> is-dismissible">
                <p><?php echo esc_html($notice); ?></p>
            </div>
        <?php endif; ?>
        <?php if (class_exists('ShadowScan_Password_Policy')): ?>
            <?php
            $current_user = wp_get_current_user();
            $show_password_grace = $current_user instanceof WP_User
                && ShadowScan_Password_Policy::is_within_grace()
                && ShadowScan_Password_Policy::user_needs_password_update($current_user)
                && !ShadowScan_Password_Policy::is_user_snoozed($current_user);
            $grace_deadline = ShadowScan_Password_Policy::get_policy_grace_deadline();
            $snooze_days = ShadowScan_Password_Policy::get_snooze_days();
            ?>
            <?php if ($show_password_grace): ?>
                <div class="notice notice-warning is-dismissible">
                    <p>
                        <?php
                        $deadline_text = $grace_deadline > 0
                            ? gmdate('Y-m-d H:i', $grace_deadline) . ' UTC'
                            : esc_html__('soon', 'shadowscan-security-link');
                        // translators: %s: Deadline date/time for password update
                        echo esc_html(sprintf(__('Password policy enabled. Please update your password by %s.', 'shadowscan-security-link'), $deadline_text));
                        ?>
                        <a href="<?php echo esc_url(wp_lostpassword_url()); ?>">
                            <?php echo esc_html__('Reset password', 'shadowscan-security-link'); ?>
                        </a>
                        <form method="post" class="shadowscan-inline-form" style="display:inline;margin-left:8px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="snooze_password_grace" />
                            <button class="button-link" type="submit">
                                <?php
                                // translators: %d: Number of days to snooze password policy reminder
                                echo esc_html(sprintf(__('Remind me in %d days', 'shadowscan-security-link'), $snooze_days));
                                ?>
                            </button>
                        </form>
                    </p>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        <div class="shadowscan-hero">
            <div><?php echo esc_html__('Connect your site to ShadowScan for clear status and quick access to help.', 'shadowscan-security-link'); ?></div>
            <div><span><?php echo esc_html__('Portal:', 'shadowscan-security-link'); ?></span> <?php echo esc_html__('See live status and incident notes.', 'shadowscan-security-link'); ?></div>
        </div>
        <?php if ($subscription_portal_unreachable): ?>
            <div class="shadowscan-banner is-warning">
                <?php echo esc_html__('Portal unreachable; using last known subscription state.', 'shadowscan-security-link'); ?>
            </div>
        <?php endif; ?>
        <?php if ($subscription_status === ShadowScan_Subscription_State::STATUS_PAST_DUE): ?>
            <div class="shadowscan-banner is-warning">
                <?php echo esc_html__('Billing issue detected. Protection continues, but updates and configuration changes are paused until payment resumes.', 'shadowscan-security-link'); ?>
                <?php if ($subscription_grace_end): ?>
                    <?php /* translators: %s: grace period end time. */ ?>
                    <?php echo esc_html(sprintf(__(' Grace ends %s.', 'shadowscan-security-link'), $format_time($subscription_grace_end))); ?>
                <?php endif; ?>
                <a class="shadowscan-muted-link" href="<?php echo esc_url($billing_url); ?>" target="_blank" rel="noreferrer" style="margin-left:8px;">
                    <?php echo esc_html__('Fix billing', 'shadowscan-security-link'); ?>
                </a>
            </div>
        <?php endif; ?>
        <?php if ($subscription_status === ShadowScan_Subscription_State::STATUS_SUSPENDED && $protection_state !== 'paused'): ?>
            <div class="shadowscan-banner is-warning">
                <?php echo esc_html__('Subscription suspended. This site is not receiving security updates. Reactivate to restore live protection.', 'shadowscan-security-link'); ?>
                <a class="shadowscan-muted-link" href="<?php echo esc_url($billing_url); ?>" target="_blank" rel="noreferrer" style="margin-left:8px;">
                    <?php echo esc_html__('Fix billing', 'shadowscan-security-link'); ?>
                </a>
            </div>
        <?php endif; ?>
        <?php if ($protection_state === 'paused'): ?>
            <div class="shadowscan-banner is-critical">
                <?php echo esc_html__('Protection is paused due to billing status. Reactivate to restore protection.', 'shadowscan-security-link'); ?>
                <a class="shadowscan-muted-link" href="<?php echo esc_url($billing_url); ?>" target="_blank" rel="noreferrer" style="margin-left:8px;">
                    <?php echo esc_html__('Fix billing', 'shadowscan-security-link'); ?>
                </a>
            </div>
        <?php endif; ?>
        <?php if ($subscription_source === 'test_override' || (!empty($subscription_test_mode) && !empty($subscription_emulated))): ?>
            <div class="shadowscan-banner">
                <?php /* translators: %s: emulated subscription status. */ ?>
                <?php echo esc_html(sprintf(__('TEST MODE — Subscription status is being emulated (%s).', 'shadowscan-security-link'), strtoupper((string) $subscription_emulated_status))); ?>
                <a class="shadowscan-muted-link" href="<?php echo esc_url(SHADOWSCAN_PORTAL_URL); ?>" style="margin-left:8px;">
                    <?php echo esc_html__('View in Portal', 'shadowscan-security-link'); ?>
                </a>
            </div>
        <?php endif; ?>

        <?php $base_url = menu_page_url('shadowscan', false); ?>
        <div class="shadowscan-tabs" role="tablist" aria-label="<?php echo esc_attr__('ShadowScan modes', 'shadowscan-security-link'); ?>">
            <?php foreach ($tabs as $key => $label): ?>
                <?php $url = add_query_arg('tab', $key, $base_url); ?>
                <a class="shadowscan-tab <?php echo $tab === $key ? 'is-active' : ''; ?>" href="<?php echo esc_url($url); ?>">
                    <?php echo esc_html($label); ?>
                </a>
            <?php endforeach; ?>
        </div>

        <?php
        $support_debug_enabled = !empty($support_debug['enabled']);
    $support_debug_until_raw = isset($support_debug['enabledUntil']) ? (string) $support_debug['enabledUntil'] : '';
    $support_debug_until = $support_debug_until_raw !== '' ? strtotime($support_debug_until_raw) : 0;
    ?>
    <?php if ($support_debug_enabled): ?>
        <div class="shadowscan-banner">
                <?php
                $until_text = $support_debug_until ? $format_time($support_debug_until) : esc_html__('Unknown', 'shadowscan-security-link');
                /* translators: %s: debug expiry time. */
                echo esc_html(sprintf(__('Support Debug active until %s. This is read-only visibility.', 'shadowscan-security-link'), $until_text));
                ?>
            </div>
        <?php endif; ?>

        <?php
        if (!isset($advanced_readonly)) {
            $advanced_readonly = false;
        }
        if (!isset($readonly_note_message)) {
            $readonly_note_message = __('Changes require an active subscription. Status remains visible.', 'shadowscan-security-link');
        }
        ?>

        <div class="shadowscan-advanced-toggle">
            <?php if ($advanced_locked): ?>
                <p class="shadowscan-note">
                    <?php echo esc_html__('Advanced controls are locked while the subscription is suspended.', 'shadowscan-security-link'); ?>
                </p>
            <?php else: ?>
                <?php if ($advanced_readonly): ?>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Advanced controls are read-only while billing is inactive.', 'shadowscan-security-link'); ?>
                    </p>
                <?php endif; ?>
                <?php if ($ui_advanced): ?>
                <div class="shadowscan-actions">
                    <button
                        class="button"
                        type="button"
                        data-shadowscan-advanced-toggle
                        data-label-show="<?php echo esc_attr__('Show advanced controls', 'shadowscan-security-link'); ?>"
                        data-label-hide="<?php echo esc_attr__('Hide advanced controls', 'shadowscan-security-link'); ?>"
                        data-default-visible="<?php echo $ui_advanced ? '1' : '0'; ?>"
                        aria-pressed="false"
                    >
                        <?php echo esc_html__('Show advanced controls', 'shadowscan-security-link'); ?>
                    </button>
                    <form method="post" class="shadowscan-inline-form">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="toggle_ui_advanced" />
                    <input type="hidden" name="ui_advanced" value="0" />
                    <button class="button-link" type="submit"><?php echo esc_html__('Disable advanced mode', 'shadowscan-security-link'); ?></button>
                    </form>
                </div>
                <?php else: ?>
                <button class="button button-primary" data-shadowscan-open-advanced <?php echo $can_persist_advanced ? '' : 'disabled'; ?>>
                    <?php echo esc_html__('Show advanced controls', 'shadowscan-security-link'); ?>
                </button>
                <?php if (!$can_persist_advanced): ?>
                    <p class="shadowscan-note" style="margin-top: 6px;">
                        <?php echo esc_html__('Advanced controls are available only with an active subscription.', 'shadowscan-security-link'); ?>
                    </p>
                <?php endif; ?>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <div class="shadowscan-modal" id="shadowscan-advanced-modal" aria-hidden="true">
            <div class="shadowscan-modal-backdrop" data-shadowscan-modal-close></div>
            <div class="shadowscan-modal-card" role="dialog" aria-modal="true" aria-labelledby="shadowscan-advanced-title">
                <h2 id="shadowscan-advanced-title"><?php echo esc_html__('Enable advanced controls?', 'shadowscan-security-link'); ?></h2>
                <p class="shadowscan-note"><?php echo esc_html__('Advanced controls are intended for administrators. Some settings can affect site access.', 'shadowscan-security-link'); ?></p>
                <form method="post" id="shadowscan-advanced-form">
                    <?php wp_nonce_field('shadowscan-dashboard'); ?>
                    <input type="hidden" name="shadowscan_action" value="toggle_ui_advanced" />
                    <input type="hidden" name="ui_advanced" value="1" />
                    <input type="hidden" name="coverage_target" id="shadowscan-coverage-target" value="" />
                    <label style="display:flex;gap:10px;align-items:center;margin-top:12px;">
                        <input type="checkbox" id="shadowscan-advanced-ack" name="ui_advanced_ack" value="1" />
                        <span><?php echo esc_html__('I understand', 'shadowscan-security-link'); ?></span>
                    </label>
                    <div class="shadowscan-actions" style="margin-top: 16px;">
                        <button type="button" class="button" data-shadowscan-modal-close><?php echo esc_html__('Cancel', 'shadowscan-security-link'); ?></button>
                        <button type="submit" class="button button-primary" id="shadowscan-advanced-submit" disabled><?php echo esc_html__('Enable', 'shadowscan-security-link'); ?></button>
                    </div>
                </form>
            </div>
        </div>

        <?php if ($tab === 'overview'): ?>
            <?php
            $connection_status = $connected ? esc_html__('Connected', 'shadowscan-security-link') : esc_html__('Not connected', 'shadowscan-security-link');
            $connection_detail = $connected
                /* translators: %s: relative heartbeat time. */
                ? sprintf(esc_html__('Last heartbeat %s', 'shadowscan-security-link'), $format_time($last_ping))
                : esc_html__('Pairing required to start monitoring.', 'shadowscan-security-link');
            $connection_link = $ui_advanced ? add_query_arg('tab', 'overview', $base_url) : add_query_arg('tab', 'coverage', $base_url);
            $issues_required = false;
            if (!$connected) {
                $issues_required = true;
            } elseif (is_array($last_self_check) && !empty($last_self_check) && empty($last_self_check['success'])) {
                $issues_required = true;
            } elseif (is_array($last_error) && !empty($last_error['message'])) {
                $issues_required = true;
            } elseif (!empty($integrity_last_event_summary)) {
                $issues_required = true;
            }
            $issues_label = $issues_required
                ? esc_html__('Action recommended', 'shadowscan-security-link')
                : esc_html__('No active issues detected.', 'shadowscan-security-link');
            $subscription_summary = $subscription_status_label;
            if ($subscription_plan) {
                $subscription_summary .= sprintf(' · %s', $subscription_plan);
            }
            if ($subscription_desc) {
                $subscription_summary .= sprintf(' · %s', $subscription_desc);
            }
            ?>
            <?php $render_section(
                __('Security Health', 'shadowscan-security-link'),
                __('A quick roll-up of protection coverage and current signals.', 'shadowscan-security-link'),
                function () use ($coverage_overall, $coverage_top_items, $coverage_last_checked, $format_time, $base_url) {
                    ?>
                    <p class="shadowscan-note">
                        <strong><?php echo esc_html__('Overall:', 'shadowscan-security-link'); ?></strong>
                        <?php echo esc_html($coverage_overall); ?>
                        · <?php echo esc_html__('Last coverage check', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($format_time($coverage_last_checked)); ?>
                    </p>
                    <?php if (!empty($coverage_top_items)): ?>
                        <div class="shadowscan-status-list" style="margin-top: 10px;">
                            <?php foreach ($coverage_top_items as $item): ?>
                                <div class="shadowscan-status-row <?php echo $item['status'] === __('Action required', 'shadowscan-security-link') ? 'is-bad' : 'is-warn'; ?>">
                                    <span class="dashicons dashicons-warning" aria-hidden="true"></span>
                                    <div class="shadowscan-status-text">
                                        <div style="font-weight:600;"><?php echo esc_html($item['label']); ?></div>
                                        <div class="shadowscan-muted">
                                            <?php echo esc_html($item['status']); ?>
                                            · <?php echo esc_html__('Last checked', 'shadowscan-security-link'); ?>
                                            <?php echo esc_html($format_time((int) $item['last_checked_at'])); ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p class="shadowscan-note"><?php echo esc_html__('No active coverage gaps detected.', 'shadowscan-security-link'); ?></p>
                    <?php endif; ?>
                    <div class="shadowscan-actions" style="margin-top: 10px;">
                        <a class="button" href="<?php echo esc_url(add_query_arg('tab', 'coverage', $base_url)); ?>">
                            <?php echo esc_html__('View coverage details', 'shadowscan-security-link'); ?>
                        </a>
                    </div>
                    <?php
                }
            ); ?>
            <?php $render_section(
                __('Status summary', 'shadowscan-security-link'),
                __('A quick view of what matters most right now.', 'shadowscan-security-link'),
                function () use ($render_status_row, $connection_status, $connection_detail, $connection_link, $protection_state_label, $protection_state, $issues_label, $issues_required, $base_url, $ui_advanced, $protection_tooltip_map) {
                    $protection_details_link = $ui_advanced ? add_query_arg('tab', 'diagnostics', $base_url) : add_query_arg('tab', 'coverage', $base_url);
                    $issues_details_link = add_query_arg('tab', 'diagnostics', $base_url);
                    ?>
                    <div class="shadowscan-status-list">
                        <?php
                        $render_status_row(
                            'dashicons-admin-site-alt3',
                            __('Connection', 'shadowscan-security-link'),
                            $connection_status . ' · ' . $connection_detail,
                            $connection_status === __('Connected', 'shadowscan-security-link') ? 'is-ok' : 'is-bad',
                            $connection_link
                        );
                        $render_status_row(
                            'dashicons-shield-alt',
                            __('Protection status', 'shadowscan-security-link'),
                            $protection_state_label,
                            $protection_state === 'live' ? 'is-ok' : ($protection_state === 'stale' ? 'is-warn' : 'is-bad'),
                            $protection_details_link,
                            null,
                            $protection_tooltip_map[$protection_state] ?? null
                        );
                        $render_status_row(
                            'dashicons-warning',
                            __('Issues requiring action', 'shadowscan-security-link'),
                            $issues_label,
                            $issues_required ? 'is-warn' : 'is-ok',
                            $issues_details_link
                        );
                        ?>
                    </div>
                    <?php
                }
            ); ?>
            <?php $render_section(
                __('Quick actions', 'shadowscan-security-link'),
                __('Keep the essentials within reach.', 'shadowscan-security-link'),
                function () use ($connected, $billing_url, $subscription_status, $base_url) {
                    $primary_label = __('Open portal', 'shadowscan-security-link');
                    $primary_url = SHADOWSCAN_PORTAL_URL;
                    $primary_attrs = '';
                    if (!$connected) {
                        $primary_label = __('Connect this site', 'shadowscan-security-link');
                        $primary_url = add_query_arg('tab', 'coverage', $base_url);
                    } elseif ($subscription_status !== ShadowScan_Subscription_State::STATUS_ACTIVE) {
                        $primary_label = __('Fix billing', 'shadowscan-security-link');
                        $primary_url = $billing_url;
                        $primary_attrs = ' target="_blank" rel="noreferrer"';
                    }
                    ?>
                    <div class="shadowscan-actions">
                        <a class="button button-primary" href="<?php echo esc_url($primary_url); ?>"<?php // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?><?php echo $primary_attrs; ?>>
                            <?php echo esc_html($primary_label); ?>
                        </a>
                        <?php if ($primary_url !== SHADOWSCAN_PORTAL_URL): ?>
                            <a class="button" href="<?php echo esc_url(SHADOWSCAN_PORTAL_URL); ?>" target="_blank" rel="noreferrer">
                                <?php echo esc_html__('Open portal', 'shadowscan-security-link'); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                    <?php
                }
            ); ?>
            <?php $render_connection_actions_section(); ?>
        <?php elseif ($tab === 'coverage'): ?>
            <?php shadowscan_handle_coverage_actions($notice, $notice_type, $subscription_state, $protection_state); ?>
            <?php
            $policy_state = class_exists('ShadowScan_Policy_State') ? ShadowScan_Policy_State::get_current_policy() : array();
            $policy_tier = (string) ($policy_state['tier'] ?? 'monitor');
            $policy_updated_at = (int) ($policy_state['updated_at'] ?? 0);
            $policy_stale = !empty($policy_state['stale']);
            $monitoring_active = !empty($coverage_states);
            $enforcement_available = 0;
            $enforcement_active = 0;
            foreach ($coverage_states as $state) {
                $entry = $state['entry'] ?? array();
                if (!empty($entry['enforcement_supported'])) {
                    $enforcement_available++;
                    if (!empty($state['enforced'])) {
                        $enforcement_active++;
                    }
                }
            }
            $sync_state = function_exists('shadowscan_get_sync_state') ? shadowscan_get_sync_state() : array();
            $last_policy_fetch_at = (int) ($sync_state['last_policy_fetch_at'] ?? 0);
            if ($policy_stale && $last_policy_fetch_at > 0 && (time() - $last_policy_fetch_at) <= 900) {
                $policy_stale = false;
            }
            $last_applied_policy_version = (string) ($sync_state['last_applied_policy_version'] ?? '');
            $last_applied_policy = $last_applied_policy_version !== '' ? $last_applied_policy_version : '—';
            $last_seen_policy = (string) ($sync_state['last_seen_policy_version'] ?? '');
            $pending_policy_drift = $last_seen_policy !== '' && $last_seen_policy !== $last_applied_policy_version;
            $last_apply_result = isset($sync_state['last_apply_result']) && is_array($sync_state['last_apply_result'])
                ? $sync_state['last_apply_result']
                : array();
            $last_apply_at = (int) ($last_apply_result['applied_at'] ?? 0);
            $last_apply_age = $last_apply_at > 0 ? human_time_diff($last_apply_at, time()) : '';
            $heartbeat_ok = $last_ping > 0 && (time() - $last_ping) < 900;
            $heartbeat_age = $last_ping > 0 ? human_time_diff($last_ping, time()) : '';
                    $coverage_config_map = array(
                        'access.mfa' => 'coverage-config-mfa',
                        'updates.auto_updates' => 'coverage-config-autoupdate',
                        'access.password_policy' => 'coverage-config-password',
                        'access.admin_geo_guard' => 'coverage-config-geo',
                    );
                    $coverage_inline_config_map = array(
                        'access.mfa' => true,
                        'access.password_policy' => true,
                        'updates.auto_updates' => true,
                        'access.admin_geo_guard' => true,
                    );
                    $coverage_advanced_map = array(
                        'access.mfa' => $render_mfa_body,
                        'updates.auto_updates' => $render_autoupdate_body,
                        'access.password_policy' => $render_password_policy_body,
                        'access.admin_geo_guard' => $render_geo_policy_body,
                    );
            $coverage_focus_anchor = $ui_advanced ? (string) get_option(SHADOWSCAN_OPTION_UI_FOCUS_ANCHOR, '') : '';
            if ($coverage_focus_anchor !== '') {
                delete_option(SHADOWSCAN_OPTION_UI_FOCUS_ANCHOR);
            }
            $next_actions = array();
            if ($subscription_status !== ShadowScan_Subscription_State::STATUS_ACTIVE) {
                $next_actions[] = array(
                    'label' => __('Resolve billing to restore live updates.', 'shadowscan-security-link'),
                    'href' => $billing_url,
                    'external' => true,
                );
            }
            foreach ($coverage_states as $state) {
                $label = (string) ($state['label'] ?? '');
                if (!in_array($label, array(__('Action required', 'shadowscan-security-link'), shadowscan_coverage_label_managed(), shadowscan_coverage_label_external()), true)) {
                    continue;
                }
                $entry = $state['entry'] ?? array();
                $action = (string) ($state['recommended_action'] ?? '');
                if ($action === '') {
                    continue;
                }
                $entry_id = (string) ($entry['id'] ?? '');
                $entry_title = (string) ($entry['title'] ?? $entry_id);
                $next_actions[] = array(
                    'title' => $entry_title,
                    'label' => $action,
                    'href' => $entry_id !== '' ? '#control-' . $entry_id : '',
                    'external' => false,
                );
            }
            ?>
            <div class="shadowscan-coverage-container<?php echo $ui_advanced ? '' : ' shadowscan-hide-advanced'; ?>">
                <div class="shadowscan-grid" style="grid-template-columns: minmax(0, 3.5fr) minmax(0, 1fr); gap: 16px;">
                    <div class="shadowscan-card" style="margin:0;">
                        <?php if (!$connected): ?>
                            <div class="shadowscan-section" style="margin-bottom:16px;">
                                <?php $render_connection_section(); ?>
                            </div>
                        <?php endif; ?>
                        <?php
                        $total_controls_count = count($coverage_states);
                        $covered_count = 0;
                        $action_required_count = 0;
                        $managed_count = 0;
                        $external_count = 0;
                        foreach ($coverage_states as $cstate) {
                            $clabel = (string) ($cstate['label'] ?? '');
                            if ($clabel === __('Covered', 'shadowscan-security-link')) {
                                $covered_count++;
                            } elseif ($clabel === __('Action required', 'shadowscan-security-link')) {
                                $action_required_count++;
                            } elseif ($clabel === shadowscan_coverage_label_managed()) {
                                $managed_count++;
                            } elseif ($clabel === shadowscan_coverage_label_external()) {
                                $external_count++;
                            }
                        }
                        $coverage_percent = $total_controls_count > 0 ? round(($covered_count / $total_controls_count) * 100) : 0;
                        $progress_color = $coverage_percent >= 80 ? '#10b981' : ($coverage_percent >= 50 ? '#f59e0b' : '#ef4444');
                        ?>
                        <div class="shadowscan-health-overview" style="margin-bottom:16px;">
                            <div class="shadowscan-health-top">
                                <div>
                                    <div class="shadowscan-score-ring" style="--ss-progress: <?php echo (int) $coverage_percent; ?>; --ss-progress-color: <?php echo esc_attr($progress_color); ?>;">
                                        <span class="shadowscan-score-number"><?php echo (int) $coverage_percent; ?>%</span>
                                    </div>
                                    <div class="shadowscan-score-label">
                                        <?php
                                        /* translators: 1: covered count, 2: total count */
                                        echo esc_html(sprintf(__('%1$d of %2$d covered', 'shadowscan-security-link'), $covered_count, $total_controls_count));
                                        ?>
                                    </div>
                                </div>
                                <div class="shadowscan-health-stats">
                                    <div class="shadowscan-stat-item">
                                        <span class="dashicons dashicons-admin-site-alt3 <?php echo $heartbeat_ok ? 'is-ok' : 'is-bad'; ?>"></span>
                                        <span><?php echo esc_html($heartbeat_ok ? __('Connected', 'shadowscan-security-link') : __('Disconnected', 'shadowscan-security-link')); ?></span>
                                    </div>
                                    <div class="shadowscan-stat-item">
                                        <span class="dashicons dashicons-clock <?php echo $heartbeat_ok ? 'is-ok' : 'is-warn'; ?>"></span>
                                        <span>
                                            <?php
                                            // translators: %s: Time elapsed since last heartbeat (e.g., "5 minutes")
                                            echo esc_html($last_ping > 0 ? sprintf(__('Heartbeat %s ago', 'shadowscan-security-link'), $heartbeat_age) : __('No heartbeat', 'shadowscan-security-link'));
                                            ?>
                                        </span>
                                    </div>
                                    <div class="shadowscan-stat-item">
                                        <span class="dashicons dashicons-media-text"></span>
                                        <span><?php echo esc_html__('Policy:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_applied_policy === '—' ? __('Not set', 'shadowscan-security-link') : $last_applied_policy); ?></span>
                                    </div>
                                    <div class="shadowscan-stat-item">
                                        <span class="dashicons dashicons-update <?php echo $pending_policy_drift ? 'is-warn' : 'is-ok'; ?>"></span>
                                        <span>
                                            <?php
                                            echo esc_html($last_apply_at > 0
                                                ? sprintf(
                                                    /* translators: %s: elapsed time since last successful policy apply. */
                                                    __('Last apply %s ago', 'shadowscan-security-link'),
                                                    $last_apply_age
                                                )
                                                : __('Policy apply pending', 'shadowscan-security-link'));
                                            ?>
                                        </span>
                                    </div>
                                    <div class="shadowscan-stat-item">
                                        <span class="dashicons dashicons-shield-alt <?php echo $enforcement_active > 0 ? 'is-ok' : ''; ?>"></span>
                                        <span>
                                            <?php
                                            /* translators: %d: number of enforced controls. */
                                            echo esc_html(sprintf(__('%d enforced', 'shadowscan-security-link'), $enforcement_active));
                                            ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                            <?php if ($action_required_count > 0): ?>
                                <div class="shadowscan-health-action-strip">
                                    <span class="dashicons dashicons-warning"></span>
                                    <span>
                                        <?php
                                        /* translators: %d: number of controls needing attention */
                                        echo esc_html(sprintf(__('%d controls need attention', 'shadowscan-security-link'), $action_required_count));
                                        ?>
                                    </span>
                                    <a href="#shadowscan-coverage-all" class="button button-primary button-small"><?php echo esc_html__('Review now', 'shadowscan-security-link'); ?></a>
                                </div>
                            <?php endif; ?>
                            <div style="margin-top:16px;">
                                <div class="shadowscan-progress-header">
                                    <span class="shadowscan-control-label" style="margin-bottom:0;"><?php echo esc_html__('Overall Coverage', 'shadowscan-security-link'); ?></span>
                                    <span class="shadowscan-progress-value" style="color:<?php echo esc_attr($progress_color); ?>;"><?php echo (int) $coverage_percent; ?>%</span>
                                </div>
                                <div class="shadowscan-progress-bar">
                                    <div class="shadowscan-progress-fill" style="width:<?php echo (int) $coverage_percent; ?>%; background:<?php echo esc_attr($progress_color); ?>;"></div>
                                </div>
                            </div>
                        </div>
                        <?php if (!$heartbeat_ok && $connected): ?>
                        <div class="shadowscan-alert is-bad" style="margin-bottom:16px;">
                            <span class="dashicons dashicons-warning"></span>
                            <div class="shadowscan-alert-content">
                                <strong><?php echo esc_html__('Connection may be stale', 'shadowscan-security-link'); ?></strong>
                                <p class="shadowscan-muted" style="margin:4px 0 0;"><?php echo esc_html__('No heartbeat received recently. Check that WordPress cron is running and the site can reach the ShadowScan API.', 'shadowscan-security-link'); ?></p>
                            </div>
                            <?php
                            $heartbeat_nonce_url = wp_nonce_url(
                                add_query_arg(array('tab' => 'coverage', 'ss_force_heartbeat' => '1'), admin_url('admin.php?page=shadowscan')),
                                'shadowscan_force_heartbeat'
                            );
                            ?>
                            <a href="<?php echo esc_url($heartbeat_nonce_url); ?>" class="button button-small">
                                <?php echo esc_html__('Send heartbeat', 'shadowscan-security-link'); ?>
                            </a>
                        </div>
                        <?php elseif ($policy_stale): ?>
                        <div class="shadowscan-alert is-warn" style="margin-bottom:16px;">
                            <span class="dashicons dashicons-warning"></span>
                            <div class="shadowscan-alert-content">
                                <strong><?php echo esc_html__('Policy data may be stale', 'shadowscan-security-link'); ?></strong>
                                <p class="shadowscan-muted" style="margin:4px 0 0;"><?php echo esc_html__('The cached policy data may be outdated. Trigger a sync to refresh and get the latest status.', 'shadowscan-security-link'); ?></p>
                            </div>
                            <?php
                            $sync_nonce_url = wp_nonce_url(
                                add_query_arg(array('tab' => 'coverage', 'ss_force_heartbeat' => '1'), admin_url('admin.php?page=shadowscan')),
                                'shadowscan_force_heartbeat'
                            );
                            ?>
                            <a href="<?php echo esc_url($sync_nonce_url); ?>" class="button button-small">
                                <?php echo esc_html__('Sync now', 'shadowscan-security-link'); ?>
                            </a>
                        </div>
                        <?php elseif ($pending_policy_drift): ?>
                        <div class="shadowscan-alert is-warn" style="margin-bottom:16px;">
                            <span class="dashicons dashicons-warning"></span>
                            <div class="shadowscan-alert-content">
                                <strong><?php echo esc_html__('Policy changes are pending apply', 'shadowscan-security-link'); ?></strong>
                                <p class="shadowscan-muted" style="margin:4px 0 0;">
                                    <?php
                                    echo esc_html(
                                        sprintf(
                                            /* translators: 1: last seen policy version, 2: last applied policy version. */
                                            __('Seen version %1$s is newer than applied version %2$s. Trigger a sync and check connector diagnostics.', 'shadowscan-security-link'),
                                            $last_seen_policy,
                                            $last_applied_policy
                                        )
                                    );
                                    ?>
                                </p>
                            </div>
                            <?php
                            $sync_nonce_url = wp_nonce_url(
                                add_query_arg(array('tab' => 'coverage', 'ss_force_heartbeat' => '1'), admin_url('admin.php?page=shadowscan')),
                                'shadowscan_force_heartbeat'
                            );
                            ?>
                            <a href="<?php echo esc_url($sync_nonce_url); ?>" class="button button-small">
                                <?php echo esc_html__('Sync now', 'shadowscan-security-link'); ?>
                            </a>
                        </div>
                        <?php endif; ?>
                        <p class="shadowscan-note" style="margin-bottom: 12px;">
                            <?php echo esc_html__('These controls are security best-practice guidance and monitoring/enforcement status. This is not a compliance certification.', 'shadowscan-security-link'); ?>
                        </p>
                        <?php
                        $truncate_words = function (string $text, int $limit = 10): string {
                            $words = preg_split('/\s+/', trim($text));
                            if (!$words || count($words) <= $limit) {
                                return $text;
                            }
                            return implode(' ', array_slice($words, 0, $limit)) . '…';
                        };
                        $coverage_rank = function (array $entry, array $state): int {
                            $label = (string) ($state['label'] ?? __('Monitoring planned', 'shadowscan-security-link'));
                            if ($label === __('Action required', 'shadowscan-security-link')) {
                                return 0;
                            }
                            $enforcement_supported = !empty($entry['enforcement_supported']);
                            $enforced = !empty($state['enforced']);
                            if ($label === shadowscan_coverage_label_external() || $label === shadowscan_coverage_label_managed() || ($enforcement_supported && !$enforced)) {
                                return 1;
                            }
                            if ($label === __('Monitoring planned', 'shadowscan-security-link')) {
                                return 2;
                            }
                            if ($label === __('Covered', 'shadowscan-security-link')) {
                                return 3;
                            }
                            return 4;
                        };
                        $render_advanced_card = function (array $item) {
                            $title = (string) ($item['title'] ?? '');
                            $summary = (string) ($item['summary'] ?? '');
                            $render = $item['render'] ?? null;
                            $search_text = strtolower(trim($title . ' ' . $summary));
                            ?>
                            <div class="shadowscan-coverage-card" data-status="covered" data-shadowscan-advanced="1" data-search="<?php echo esc_attr($search_text); ?>">
                                <div class="shadowscan-row">
                                    <div class="shadowscan-row-main">
                                        <div class="shadowscan-control-title"><?php echo esc_html($title); ?></div>
                                        <div class="shadowscan-muted"><?php echo esc_html($summary); ?></div>
                                    </div>
                                    <div class="shadowscan-row-meta">
                                        <span class="shadowscan-coverage-pill is-muted"><?php echo esc_html__('Advanced', 'shadowscan-security-link'); ?></span>
                                    </div>
                                </div>
                                <?php if (is_callable($render)): ?>
                                    <details class="shadowscan-card-details" open>
                                        <summary class="button button-small" aria-expanded="false">
                                            <?php echo esc_html__('View details', 'shadowscan-security-link'); ?>
                                        </summary>
                                        <div class="shadowscan-control-details" data-shadowscan-advanced="1">
                                            <div class="ss-details-layout">
                                                <?php $render(); ?>
                                            </div>
                                        </div>
                                    </details>
                                <?php endif; ?>
                            </div>
                            <?php
                        };
                        $render_control_card = function (array $entry) use ($coverage_states, $format_time, $coverage_config_map, $coverage_inline_config_map, $coverage_advanced_map, $truncate_words, $ui_advanced, $advanced_locked, $subscription_state, $policy_state, $protection_state) {
                            $state = $coverage_states[$entry['id']] ?? array();
                            $label = (string) ($state['label'] ?? __('Monitoring planned', 'shadowscan-security-link'));
                            $tone = (string) ($state['tone'] ?? '');
                            $message = trim((string) ($state['message'] ?? ''));
                            $recommended = trim((string) ($state['recommended_action'] ?? ''));
                            $last_checked = (int) ($state['last_checked_at'] ?? 0);
                            $enforcement_supported = !empty($entry['enforcement_supported']);
                            $enforced = !empty($state['enforced']);
                            $has_inline_configuration = !empty($coverage_inline_config_map[$entry['id']] ?? false);
                            $config_anchor = $has_inline_configuration ? ($coverage_config_map[$entry['id']] ?? '') : '';
                            $advanced_body = $coverage_advanced_map[$entry['id']] ?? null;
                            $is_advanced = $ui_advanced && is_callable($advanced_body);
                            $control_key = $entry['control_keys'][0] ?? '';
                            $portal_link = (string) ($entry['portal_link'] ?? '');
                            $details_id = 'coverage-details-' . preg_replace('/[^a-zA-Z0-9_-]/', '-', (string) ($entry['id'] ?? 'control'));
                            $tech_details_id = $details_id . '-tech';
                            $ack_key = (string) ($state['ack_key'] ?? '');
                            $acknowledged = !empty($state['acknowledged']);
                            $ack_type = (string) ($state['ack_type'] ?? '');
                            $ack_expires_at = (int) ($state['ack_expires_at'] ?? 0);
                            $ack_eligible = !empty($state['ack_eligible']);
                            $billing_active = ($subscription_state['status'] ?? '') === ShadowScan_Subscription_State::STATUS_ACTIVE;
                            $enforcement_allowed = shadowscan_can_apply_enforcement($subscription_state, $policy_state, 'hardened');
                            $enforcement_locked = (!$billing_active && !shadowscan_is_dev_env())
                                || !$enforcement_allowed
                                || $protection_state !== 'live';
                            $summary = $message !== '' ? $truncate_words($message, 10) : __('Monitoring is active for this control.', 'shadowscan-security-link');
                            if ($acknowledged && $ack_expires_at > 0) {
                                if ($ack_type === 'muted') {
                                    /* translators: %s: acknowledgement expiry time. */
                                    $summary_template = __('Muted until %s', 'shadowscan-security-link');
                                } else {
                                    /* translators: %s: acknowledgement expiry time. */
                                    $summary_template = __('Acknowledged until %s', 'shadowscan-security-link');
                                }
                                $summary = sprintf($summary_template, $format_time($ack_expires_at));
                            }
                            $meaning = $message !== '' ? $message : (string) ($entry['description'] ?? '');
                            $shadow_scan_action = $enforcement_supported
                                ? __('Tracks this control and reports enforcement readiness.', 'shadowscan-security-link')
                                : __('Tracks this control and reports changes.', 'shadowscan-security-link');
                            $status_key = 'covered';
                            if ($label === __('Action required', 'shadowscan-security-link')) {
                                $status_key = 'action_required';
                            } elseif ($label === shadowscan_coverage_label_managed()) {
                                $status_key = 'managed';
                            } elseif ($label === shadowscan_coverage_label_external()) {
                                $status_key = 'needs_host';
                            }
                            $last_seen_label = $last_checked > 0 ? $format_time($last_checked) : '—';
                            $reason_bucket = __('Misconfiguration', 'shadowscan-security-link');
                            if ($last_checked <= 0) {
                                $reason_bucket = __('Telemetry stale', 'shadowscan-security-link');
                            } elseif ($label === __('Action required', 'shadowscan-security-link') && $enforcement_supported && !$enforced) {
                                $reason_bucket = __('Needs approval', 'shadowscan-security-link');
                            } elseif (strpos($control_key, 'abuse') !== false || strpos($control_key, 'failed_login') !== false || strpos($control_key, 'rate_limit') !== false) {
                                $reason_bucket = __('Detected abuse', 'shadowscan-security-link');
                            }
                            $trigger_text = __('Evaluated during heartbeat and scheduled checks.', 'shadowscan-security-link');
                            $observe_text = $enforcement_supported
                                ? __('Observe: monitors status and signals. Enforce: applies only when enabled in policy.', 'shadowscan-security-link')
                                : __('Observe only (monitoring and reporting).', 'shadowscan-security-link');
                            $limit_text = __('Results can be influenced by hosting, caching, and WordPress constraints.', 'shadowscan-security-link');
                            if (($entry['id'] ?? '') === 'access.mfa') {
                                $trigger_text = __('Evaluated when MFA coverage snapshot is reported and during heartbeat.', 'shadowscan-security-link');
                                $limit_text = __('Enforcement applies to wp-admin access only and depends on user enrollment.', 'shadowscan-security-link');
                            } elseif (($entry['id'] ?? '') === 'updates.auto_updates') {
                                $trigger_text = __('Evaluated during heartbeat and when update policy is synced.', 'shadowscan-security-link');
                                $limit_text = __('Auto-updates rely on WordPress update systems and host policies.', 'shadowscan-security-link');
                            } elseif (($entry['id'] ?? '') === 'access.password_policy') {
                                $trigger_text = __('Evaluated on password change/reset and at login after the 14-day grace period.', 'shadowscan-security-link');
                                $limit_text = __('In-scope users must update their password within 14 days of activation.', 'shadowscan-security-link');
                            } elseif (($entry['id'] ?? '') === 'access.admin_geo_guard') {
                                $trigger_text = __('Evaluated on wp-admin/wp-login requests and heartbeat summaries.', 'shadowscan-security-link');
                                $limit_text = __('Accuracy depends on request headers; enforcement is limited to wp-admin/wp-login.', 'shadowscan-security-link');
                            } elseif (strpos((string) ($entry['id'] ?? ''), 'owasp.') === 0) {
                                $trigger_text = __('Evaluated during heartbeat and when related security events are observed.', 'shadowscan-security-link');
                                $limit_text = __('Mapped to OWASP categories; mapping is informational only.', 'shadowscan-security-link');
                            }
                            $search_text = strtolower(trim((string) ($entry['title'] ?? '') . ' ' . (string) ($entry['id'] ?? '') . ' ' . $summary . ' ' . $recommended . ' ' . $label));
                            ?>
                            <div class="shadowscan-coverage-card <?php echo esc_attr($tone); ?>" id="control-<?php echo esc_attr($entry['id'] ?? 'unknown'); ?>" data-status="<?php echo esc_attr($status_key); ?>" data-search="<?php echo esc_attr($search_text); ?>">
                                <div class="shadowscan-row">
                                    <div class="shadowscan-row-main">
                                        <div class="shadowscan-control-title"><?php echo esc_html($entry['title'] ?? $entry['id']); ?></div>
                                        <div class="shadowscan-muted"><?php echo esc_html($summary); ?></div>
                                        <?php if ($recommended !== '' && $status_key === 'action_required'): ?>
                                            <div class="shadowscan-action-callout">
                                                <span class="dashicons dashicons-arrow-right-alt" aria-hidden="true"></span>
                                                <span><?php echo esc_html($recommended); ?></span>
                                            </div>
                                        <?php elseif ($recommended !== ''): ?>
                                            <div class="shadowscan-muted"><?php echo esc_html__('Next action:', 'shadowscan-security-link'); ?> <?php echo esc_html($recommended); ?></div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="shadowscan-row-meta">
                                        <span class="shadowscan-coverage-pill <?php echo esc_attr($tone); ?>">
                                            <?php if ($status_key === 'action_required'): ?>
                                                <span class="dashicons dashicons-warning" aria-hidden="true"></span>
                                            <?php elseif ($status_key === 'managed' || $status_key === 'needs_host'): ?>
                                                <span class="dashicons dashicons-info-outline" aria-hidden="true"></span>
                                            <?php else: ?>
                                                <span class="dashicons dashicons-yes-alt" aria-hidden="true"></span>
                                            <?php endif; ?>
                                            <?php echo esc_html($label); ?>
                                        </span>
                                        <?php if ($is_advanced): ?>
                                            <span class="shadowscan-coverage-pill is-muted"><?php echo esc_html__('Advanced', 'shadowscan-security-link'); ?></span>
                                        <?php endif; ?>
                                        <?php if ($status_key === 'action_required'): ?>
                                            <span class="shadowscan-coverage-pill is-muted"><?php echo esc_html($reason_bucket); ?></span>
                                        <?php endif; ?>
                                        <span class="shadowscan-muted"><?php echo esc_html__('Last seen', 'shadowscan-security-link'); ?>: <?php echo esc_html($last_seen_label); ?></span>
                                    </div>
                                </div>
                                <?php if ($portal_link): ?>
                                    <a class="button button-small" href="<?php echo esc_url($portal_link); ?>" target="_blank" rel="noreferrer">
                                        <?php echo esc_html__('Open portal settings', 'shadowscan-security-link'); ?>
                                    </a>
                                <?php endif; ?>
                                <details class="shadowscan-card-details" id="<?php echo esc_attr($details_id); ?>">
                                    <summary class="button button-small" aria-expanded="false">
                                        <?php echo esc_html__('View details', 'shadowscan-security-link'); ?>
                                    </summary>
                                    <div class="shadowscan-control-details">
                                    <div class="ss-details-layout">
                                    <?php $has_side = (is_callable($advanced_body) && $config_anchor !== '') || $acknowledged || $ack_eligible; ?>
                                    <div class="shadowscan-control-columns">
                                    <div>
                                    <details class="shadowscan-control-section shadowscan-control-collapse" open data-shadowscan-collapse="1">
                                        <summary class="shadowscan-control-label">
                                            <span class="dashicons dashicons-info-outline"></span>
                                            <?php echo esc_html__('What this means', 'shadowscan-security-link'); ?>
                                        </summary>
                                        <p class="shadowscan-note"><?php echo esc_html($meaning !== '' ? $meaning : __('Monitoring is active for this control.', 'shadowscan-security-link')); ?></p>
                                    </details>
                                    <details class="shadowscan-control-section shadowscan-control-collapse" open data-shadowscan-collapse="1">
                                        <summary class="shadowscan-control-label">
                                            <span class="dashicons dashicons-shield"></span>
                                            <?php echo esc_html__('What ShadowScan is doing', 'shadowscan-security-link'); ?>
                                        </summary>
                                        <p class="shadowscan-note"><?php echo esc_html($shadow_scan_action); ?></p>
                                        <?php if (($entry['id'] ?? '') === 'access.admin_geo_guard'): ?>
                                            <?php
                                            $last_country = (string) ($state['evidence']['last_country'] ?? '');
                                            $last_seen_at = (int) ($state['evidence']['last_seen_at'] ?? 0);
                                            $confidence = $state['evidence']['confidence'] ?? array();
                                            $policy = is_array($state['evidence']['policy'] ?? null) ? $state['evidence']['policy'] : array();
                                            $policy_mode = (string) ($policy['mode'] ?? 'observe');
                                            $allow = isset($policy['allow_countries']) && is_array($policy['allow_countries'])
                                                ? $policy['allow_countries']
                                                : array();
                                            $verification_enabled = function_exists('shadowscan_is_admin_geo_verification_enabled')
                                                && shadowscan_is_admin_geo_verification_enabled();
                                            if ($last_seen_at > 0):
                                                $country_label = $last_country !== '' ? $last_country : __('unknown', 'shadowscan-security-link');
                                                ?>
                                                <p class="shadowscan-note">
                                                    <?php /* translators: 1: country label, 2: last seen time. */ ?>
                                                    <?php echo esc_html(sprintf(__('Last observed: %1$s · %2$s', 'shadowscan-security-link'), $country_label, $format_time($last_seen_at))); ?>
                                                </p>
                                            <?php endif; ?>
                                            <p class="shadowscan-note">
                                                <?php /* translators: 1: policy mode, 2: allowlist count. */ ?>
                                                <?php echo esc_html(sprintf(__('Policy mode: %1$s · Allowlist: %2$d', 'shadowscan-security-link'), $policy_mode, count($allow))); ?>
                                            </p>
                                                <?php if (ShadowScan_Admin_Geo_Guard::bypass_cookie_active()): ?>
                                                    <?php
                                                    $remaining = 0;
                                                    $raw_cookie = isset($_COOKIE['ss_geo_bypass']) ? sanitize_text_field(wp_unslash($_COOKIE['ss_geo_bypass'])) : '';
                                                    if (is_numeric($raw_cookie)) {
                                                        $remaining = max(0, (int) $raw_cookie - time());
                                                    }
                                                    ?>
                                                    <p class="shadowscan-note">
                                                        <?php
                                                        if ($remaining > 0) {
                                                            /* translators: %d: minutes remaining. */
                                                            $bypass_message = sprintf(__('Bypass active: %d minutes remaining', 'shadowscan-security-link'), (int) ceil($remaining / 60));
                                                        } else {
                                                            $bypass_message = __('Bypass active', 'shadowscan-security-link');
                                                        }
                                                        echo esc_html($bypass_message);
                                                        ?>
                                                    </p>
                                                    <a class="button button-small" href="<?php echo esc_url(wp_nonce_url(add_query_arg('ss_bypass_clear', '1'), 'shadowscan_geo_bypass_clear')); ?>">
                                                        <?php echo esc_html__('Remove bypass', 'shadowscan-security-link'); ?>
                                                    </a>
                                                <?php endif; ?>
                                                <?php if (is_array($confidence) && !empty($confidence['level'])): ?>
                                                    <?php
                                                    $level = strtoupper((string) ($confidence['level'] ?? ''));
                                                    $known_ratio = isset($confidence['known_ratio']) ? (float) $confidence['known_ratio'] : 0.0;
                                                    $known_pct = (int) round($known_ratio * 100);
                                                    $reason = (string) ($confidence['reason'] ?? '');
                                                    $confidence_label = '';
                                                if ($level === 'LOW' && $reason === 'insufficient_data') {
                                                    $confidence_label = __('Low (insufficient data)', 'shadowscan-security-link');
                                                } elseif ($level === 'LOW' && $reason === 'missing_headers') {
                                                    $confidence_label = __('Low — headers often missing', 'shadowscan-security-link');
                                                } elseif ($level === 'HIGH') {
                                                    $confidence_label = __('High', 'shadowscan-security-link');
                                                } elseif ($level === 'MEDIUM') {
                                                    $confidence_label = __('Medium', 'shadowscan-security-link');
                                                } else {
                                                    $confidence_label = __('Low', 'shadowscan-security-link');
                                                }
                                                ?>
                                                <p class="shadowscan-note">
                                                    <?php /* translators: 1: confidence label, 2: percentage known. */ ?>
                                                    <?php echo esc_html(sprintf(__('Location confidence: %1$s (%2$d%% known)', 'shadowscan-security-link'), $confidence_label, $known_pct)); ?>
                                                </p>
                                                <?php if ($policy_mode === 'enforce' && $level === 'LOW'): ?>
                                                    <p class="shadowscan-note">
                                                        <?php echo esc_html__('Enforcement is enabled, but location confidence is LOW. Many requests lack headers, so rules may not behave predictably.', 'shadowscan-security-link'); ?>
                                                    </p>
                                                <?php endif; ?>
                                            <?php endif; ?>
                                                <?php if ($verification_enabled && class_exists('ShadowScan_Admin_Geo_Guard')): ?>
                                                    <?php
                                                    $geo_source = ShadowScan_Admin_Geo_Guard::get_geo_source();
                                                    $current_country = ShadowScan_Admin_Geo_Guard::get_current_country();
                                                    $confidence_total = isset($confidence['total']) ? (int) $confidence['total'] : 0;
                                                    $known_ratio = isset($confidence['known_ratio']) ? (float) $confidence['known_ratio'] : 0.0;
                                                    $known_pct = (int) round($known_ratio * 100);
                                                    $allow_list = $allow;
                                                    $allow_preview = implode(', ', array_slice($allow_list, 0, 5));
                                                    $bypass_active = ShadowScan_Admin_Geo_Guard::bypass_cookie_active();
                                                    $remaining = 0;
                                                    if ($bypass_active && !empty($_COOKIE['ss_geo_bypass'])) {
                                                        $bypass_value = sanitize_text_field(wp_unslash($_COOKIE['ss_geo_bypass']));
                                                        if (is_numeric($bypass_value)) {
                                                            $remaining = max(0, (int) $bypass_value - time());
                                                        }
                                                    }
                                                    $sim_result = null;
                                                    if (!empty($_POST['shadowscan_geo_simulate']) && check_admin_referer('shadowscan_geo_simulate')) {
                                                        $selected = sanitize_text_field(wp_unslash($_POST['geo_sim_country'] ?? ''));
                                                        $custom = sanitize_text_field(wp_unslash($_POST['geo_sim_custom'] ?? ''));
                                                    if ($custom !== '') {
                                                        $selected = $custom;
                                                    }
                                                    if ($selected === 'current') {
                                                        $selected = $current_country ?: '';
                                                    }
                                                    $selected = strtoupper($selected);
                                                    if ($selected === 'UNKNOWN' || $selected === '') {
                                                        $selected = '';
                                                    }
                                                    $sim_result = ShadowScan_Admin_Geo_Guard::simulate_decision($selected !== '' ? $selected : null);
                                                }
                                                ?>
                                                    <div class="shadowscan-control-section">
                                                        <div class="shadowscan-control-label"><?php echo esc_html__('Admin Access Guard — Verification', 'shadowscan-security-link'); ?></div>
                                                        <div class="shadowscan-note">
                                                            <?php /* translators: %s: geo header source label. */ ?>
                                                            <?php echo esc_html(sprintf(__('Geo header source: %s', 'shadowscan-security-link'), $geo_source)); ?>
                                                        </div>
                                                        <div class="shadowscan-note">
                                                            <?php /* translators: 1: allowlist count, 2: allowlist preview. */ ?>
                                                            <?php echo esc_html(sprintf(__('Allowlist: %1$d (%2$s)', 'shadowscan-security-link'), count($allow_list), $allow_preview !== '' ? $allow_preview : __('none', 'shadowscan-security-link'))); ?>
                                                        </div>
                                                        <div class="shadowscan-note">
                                                            <?php /* translators: %s: current country label. */ ?>
                                                            <?php echo esc_html(sprintf(__('Current country: %s', 'shadowscan-security-link'), $current_country ?: __('unknown', 'shadowscan-security-link'))); ?>
                                                        </div>
                                                        <div class="shadowscan-note">
                                                            <?php /* translators: 1: known percentage, 2: sample count. */ ?>
                                                            <?php echo esc_html(sprintf(__('Location confidence: %1$d%% known (%2$d samples)', 'shadowscan-security-link'), $known_pct, $confidence_total)); ?>
                                                        </div>
                                                        <div class="shadowscan-note">
                                                            <?php
                                                            if ($bypass_active) {
                                                                if ($remaining > 0) {
                                                                    /* translators: %d: minutes remaining. */
                                                                    $bypass_status = sprintf(__('Bypass status: active (%d minutes remaining)', 'shadowscan-security-link'), (int) ceil($remaining / 60));
                                                                } else {
                                                                    $bypass_status = __('Bypass status: active', 'shadowscan-security-link');
                                                                }
                                                            } else {
                                                                $bypass_status = __('Bypass status: inactive', 'shadowscan-security-link');
                                                            }
                                                            echo esc_html($bypass_status);
                                                            ?>
                                                        </div>
                                                    <form method="post" class="shadowscan-geo-sim-form">
                                                        <?php wp_nonce_field('shadowscan_geo_simulate'); ?>
                                                        <div class="shadowscan-note"><?php echo esc_html__('Simulate block check (no block)', 'shadowscan-security-link'); ?></div>
                                                        <div class="shadowscan-inline-fields">
                                                            <select name="geo_sim_country">
                                                                <option value="current"><?php echo esc_html__('Current request', 'shadowscan-security-link'); ?></option>
                                                                <?php foreach ($allow_list as $code): ?>
                                                                    <option value="<?php echo esc_attr($code); ?>"><?php echo esc_html($code); ?></option>
                                                                <?php endforeach; ?>
                                                                <option value="UNKNOWN"><?php echo esc_html__('Unknown', 'shadowscan-security-link'); ?></option>
                                                            </select>
                                                            <input type="text" name="geo_sim_custom" placeholder="<?php echo esc_attr__('Custom (e.g., US)', 'shadowscan-security-link'); ?>" />
                                                            <button type="submit" class="button button-small" name="shadowscan_geo_simulate" value="1">
                                                                <?php echo esc_html__('Run simulation', 'shadowscan-security-link'); ?>
                                                            </button>
                                                        </div>
                                                        <?php if (is_array($sim_result)): ?>
                                                            <div class="shadowscan-note">
                                                                <?php /* translators: 1: simulation decision, 2: simulation reason. */ ?>
                                                                <?php echo esc_html(sprintf(__('Simulation: %1$s (%2$s)', 'shadowscan-security-link'), strtoupper($sim_result['decision'] ?? 'allow'), $sim_result['reason'] ?? '')); ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </form>
                                                </div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    <?php if ($recommended || (is_callable($advanced_body) && $config_anchor) || $portal_link): ?>
                                    <details class="shadowscan-control-section shadowscan-control-collapse" open data-shadowscan-collapse="1">
                                        <summary class="shadowscan-control-label">
                                            <span class="dashicons dashicons-arrow-right-alt"></span>
                                            <?php echo esc_html__('What you can do', 'shadowscan-security-link'); ?>
                                        </summary>
                                        <?php if ($recommended): ?>
                                            <p class="shadowscan-note"><?php echo esc_html($recommended); ?></p>
                                        <?php endif; ?>
                                            <?php if ($config_anchor && is_callable($advanced_body)): ?>
                                                <a class="button button-small" href="<?php echo esc_url('#' . $config_anchor); ?>">
                                                    <?php echo esc_html__('Open configuration', 'shadowscan-security-link'); ?>
                                                </a>
                                            <?php elseif ($portal_link): ?>
                                                <a class="button button-small" href="<?php echo esc_url($portal_link); ?>" target="_blank" rel="noreferrer">
                                                    <?php echo esc_html__('Open portal settings', 'shadowscan-security-link'); ?>
                                                </a>
                                            <?php else: ?>
                                                <button type="button" class="button button-small" data-shadowscan-open-details data-shadowscan-details="<?php echo esc_attr($tech_details_id); ?>">
                                                    <?php echo esc_html__('View technical details', 'shadowscan-security-link'); ?>
                                                </button>
                                            <?php endif; ?>
                                    </details>
                                    <?php endif; ?>
                                    <?php if ($enforcement_supported && empty($entry['status_callback']) && $control_key !== ''): ?>
                                        <div class="shadowscan-control-section">
                                            <div class="shadowscan-control-label"><?php echo esc_html__('Enforcement', 'shadowscan-security-link'); ?></div>
                                            <p class="shadowscan-note"><?php echo esc_html__('Optional enforcement remains off until enabled.', 'shadowscan-security-link'); ?></p>
                                            <?php if ($enforcement_locked): ?>
                                                <p class="shadowscan-note"><?php echo esc_html__('Enforcement changes are locked until billing is active or policy allows enforcement.', 'shadowscan-security-link'); ?></p>
                                            <?php endif; ?>
                                            <form method="post">
                                                <?php wp_nonce_field('shadowscan-coverage'); ?>
                                                <input type="hidden" name="shadowscan_coverage_action" value="toggle_control" />
                                                <input type="hidden" name="control_key" value="<?php echo esc_attr($control_key); ?>" />
                                                <?php if ($enforcement_locked): ?>
                                                    <fieldset disabled>
                                                <?php endif; ?>
                                                <label style="display:flex;gap:10px;align-items:center;">
                                                    <input type="checkbox" name="control_enabled" value="1" <?php checked($enforced); ?> />
                                                    <span><?php echo esc_html__('Enable enforcement', 'shadowscan-security-link'); ?></span>
                                                </label>
                                                <div class="shadowscan-actions" style="margin-top:8px;">
                                                    <button class="button"><?php echo esc_html__('Save setting', 'shadowscan-security-link'); ?></button>
                                                </div>
                                                <?php if ($enforcement_locked): ?>
                                                    </fieldset>
                                                <?php endif; ?>
                                            </form>
                                        </div>
                                    <?php endif; ?>
                                    </div>
                                    <?php if ($has_side): ?>
                                        <div>
                                            <?php if (is_callable($advanced_body) && $config_anchor !== ''): ?>
                                                <details class="shadowscan-config-details" id="<?php echo esc_attr($config_anchor); ?>">
                                                    <summary class="shadowscan-config-toggle">
                                                        <span class="dashicons dashicons-admin-generic"></span>
                                                        <?php echo esc_html__('Configuration', 'shadowscan-security-link'); ?>
                                                    </summary>
                                                    <div class="shadowscan-config-card">
                                                        <?php $advanced_body(); ?>
                                                    </div>
                                                </details>
                                            <?php endif; ?>
                                            <?php if ($acknowledged || $ack_eligible): ?>
                                                <div class="shadowscan-control-section">
                                                    <div class="shadowscan-control-label"><?php echo esc_html__('Acknowledgement', 'shadowscan-security-link'); ?></div>
                                                    <?php if ($acknowledged && $ack_expires_at > 0): ?>
                                                        <p class="shadowscan-note">
                                                            <?php echo esc_html($ack_type === 'muted' ? __('Muted until', 'shadowscan-security-link') : __('Acknowledged until', 'shadowscan-security-link')); ?>
                                                            <?php echo esc_html($format_time($ack_expires_at)); ?>
                                                        </p>
                                                    <?php else: ?>
                                                        <p class="shadowscan-note"><?php echo esc_html__('Acknowledge this finding to reduce its visibility without disabling monitoring.', 'shadowscan-security-link'); ?></p>
                                                    <?php endif; ?>
                                                    <form method="post">
                                                        <?php wp_nonce_field('shadowscan-coverage'); ?>
                                                        <input type="hidden" name="shadowscan_coverage_action" value="ack_control" />
                                                        <input type="hidden" name="ack_key" value="<?php echo esc_attr($ack_key); ?>" />
                                                        <div class="shadowscan-actions">
                                                            <?php if ($acknowledged): ?>
                                                                <button class="button" name="ack_action" value="clear"><?php echo esc_html__('Clear acknowledgement', 'shadowscan-security-link'); ?></button>
                                                            <?php elseif ($ack_eligible): ?>
                                                                <button class="button" name="ack_action" value="ack"><?php echo esc_html__('Acknowledge (7 days)', 'shadowscan-security-link'); ?></button>
                                                                <button class="button" name="ack_action" value="mute_30"><?php echo esc_html__('Mute 30 days', 'shadowscan-security-link'); ?></button>
                                                                <button class="button" name="ack_action" value="mute_90"><?php echo esc_html__('Mute 90 days', 'shadowscan-security-link'); ?></button>
                                                            <?php endif; ?>
                                                        </div>
                                                    </form>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                    </div>
                                    </div>
                                    <details class="shadowscan-tech-details" id="<?php echo esc_attr($tech_details_id); ?>">
                                        <summary>
                                            <span class="dashicons dashicons-info-outline" aria-hidden="true"></span>
                                            <?php echo esc_html__('Technical details', 'shadowscan-security-link'); ?>
                                        </summary>
                                        <div class="shadowscan-tech-body">
                                            <div class="shadowscan-note">
                                            <?php echo esc_html__('Aligned frameworks: OWASP Top 10 (mapping only).', 'shadowscan-security-link'); ?>
                                            </div>
                                            <?php if (!empty($entry['owasp_ids'])): ?>
                                                <div class="shadowscan-note">
                                                    <?php echo esc_html__('Mapped categories:', 'shadowscan-security-link'); ?>
                                                    <?php echo esc_html(implode(', ', (array) $entry['owasp_ids'])); ?>
                                                </div>
                                            <?php endif; ?>
                                            <?php if (!empty($entry['control_keys'])): ?>
                                                <div class="shadowscan-note">
                                                    <?php echo esc_html__('Internal control keys:', 'shadowscan-security-link'); ?>
                                                    <?php echo esc_html(implode(', ', (array) $entry['control_keys'])); ?>
                                                </div>
                                            <?php endif; ?>
                                            <div class="shadowscan-note">
                                                <?php echo esc_html__('Last checked:', 'shadowscan-security-link'); ?>
                                                <?php echo esc_html($format_time($last_checked)); ?>
                                            </div>
                                            <div class="shadowscan-note">
                                                <?php echo esc_html__('Last raw status:', 'shadowscan-security-link'); ?>
                                                <?php echo esc_html((string) ($state['status'] ?? 'unknown')); ?>
                                            </div>
                                            <details class="shadowscan-tech-details">
                                                <summary>
                                                    <span class="dashicons dashicons-editor-help" aria-hidden="true"></span>
                                                    <?php echo esc_html__('Trigger & Scope', 'shadowscan-security-link'); ?>
                                                </summary>
                                                <div class="shadowscan-tech-body">
                                                    <div class="shadowscan-note">
                                                        <?php echo esc_html__('Trigger:', 'shadowscan-security-link'); ?>
                                                        <?php echo esc_html($trigger_text); ?>
                                                    </div>
                                                    <div class="shadowscan-note">
                                                        <?php echo esc_html__('Observe vs enforce:', 'shadowscan-security-link'); ?>
                                                        <?php echo esc_html($observe_text); ?>
                                                    </div>
                                                    <div class="shadowscan-note">
                                                        <?php echo esc_html__('Limitations:', 'shadowscan-security-link'); ?>
                                                        <?php echo esc_html($limit_text); ?>
                                                    </div>
                                                </div>
                                            </details>
                                        </div>
                                    </details>
                                </div>
                                </details>
                            </div>
                            <?php
                        };

                        $coverage_action_count = count($coverage_action_items);
                        $coverage_covered_count = 0;
                        foreach ($coverage_states as $state) {
                            $label = (string) ($state['label'] ?? '');
                            if ($label === __('Covered', 'shadowscan-security-link')) {
                                $coverage_covered_count++;
                            }
                        }
                        $coverage_tabs = array();
                        $group_action_counts = array();
                        foreach ($coverage_states as $cid => $cstate) {
                            $cgroup = $cstate['entry']['group'] ?? '';
                            if (!isset($group_action_counts[$cgroup])) {
                                $group_action_counts[$cgroup] = 0;
                            }
                            if (($cstate['label'] ?? '') === __('Action required', 'shadowscan-security-link')) {
                                $group_action_counts[$cgroup]++;
                            }
                        }
                        foreach ($coverage_groups as $group_id => $group_meta) {
                            $coverage_tabs[] = array(
                                'id' => $group_id,
                                'label' => (string) ($group_meta['title'] ?? $group_id),
                                'action_count' => $group_action_counts[$group_id] ?? 0,
                            );
                        }
                        $advanced_sections_by_group = $ui_advanced ? array(
                            'account_access' => array(
                                array(
                                    'id' => 'advanced.guard',
                                    'title' => __('Guard Layer & Safe Mode', 'shadowscan-security-link'),
                                    'summary' => __('Containment responds to repeated login abuse. Safe mode pauses enforcement temporarily.', 'shadowscan-security-link'),
                                    'render' => $render_guard_section,
                                ),
                            ),
                            'application_hardening' => array(
                                array(
                                    'id' => 'advanced.hardening',
                                    'title' => __('Hardening', 'shadowscan-security-link'),
                                    'summary' => __('Keep protections focused and reversible. Changes apply immediately.', 'shadowscan-security-link'),
                                    'render' => $render_hardening_section,
                                ),
                                array(
                                    'id' => 'advanced.server',
                                    'title' => __('Server compatibility', 'shadowscan-security-link'),
                                    'summary' => __('Baseline server hardening and htaccess automation.', 'shadowscan-security-link'),
                                    'render' => $render_server_section,
                                ),
                            ),
                            'visibility' => array(
                                array(
                                    'id' => 'advanced.remote',
                                    'title' => __('Remote diagnostics', 'shadowscan-security-link'),
                                    'summary' => __('Enable only while troubleshooting. Access auto-disables after 60 minutes.', 'shadowscan-security-link'),
                                    'render' => $render_remote_diagnostics_section,
                                ),
                                array(
                                    'id' => 'advanced.disconnect',
                                    'title' => __('Offboarding & removal', 'shadowscan-security-link'),
                                    'summary' => __('Disconnect this site or access billing actions.', 'shadowscan-security-link'),
                                    'render' => $render_disconnect_section,
                                ),
                            ),
                        ) : array();
                        ?>
                        <div id="shadowscan-coverage-all">
                        <div class="shadowscan-coverage-header">
                            <div>
                                <div class="shadowscan-control-title"><?php echo esc_html__('Protection Coverage', 'shadowscan-security-link'); ?></div>
                                <div class="shadowscan-muted"><?php echo esc_html__('A live snapshot of posture and control status.', 'shadowscan-security-link'); ?></div>
                            </div>
                            <div class="shadowscan-chip-row">
                                <?php /* translators: %d: item count. */ ?>
                                <span class="shadowscan-coverage-pill is-bad"><span class="dashicons dashicons-warning" aria-hidden="true"></span><?php echo esc_html(sprintf(__('Action required (%d)', 'shadowscan-security-link'), $coverage_action_count)); ?></span>
                                <?php /* translators: %d: item count. */ ?>
                                <span class="shadowscan-coverage-pill is-warn"><span class="dashicons dashicons-info-outline" aria-hidden="true"></span><?php echo esc_html(sprintf(__('Managed by ShadowScan (%d)', 'shadowscan-security-link'), $managed_count)); ?></span>
                                <?php /* translators: %d: item count. */ ?>
                                <span class="shadowscan-coverage-pill is-warn"><span class="dashicons dashicons-info-outline" aria-hidden="true"></span><?php echo esc_html(sprintf(__('External action (%d)', 'shadowscan-security-link'), $external_count)); ?></span>
                                <?php /* translators: %d: item count. */ ?>
                                <span class="shadowscan-coverage-pill is-ok"><span class="dashicons dashicons-yes-alt" aria-hidden="true"></span><?php echo esc_html(sprintf(__('Covered (%d)', 'shadowscan-security-link'), $coverage_covered_count)); ?></span>
                                <details class="shadowscan-inline-tooltip" style="display:inline-block;margin-left:8px;">
                                    <summary aria-label="<?php echo esc_attr__('Coverage label meanings', 'shadowscan-security-link'); ?>" style="display:inline-flex;align-items:center;gap:4px;cursor:pointer;">
                                        <span class="dashicons dashicons-info-outline" aria-hidden="true"></span>
                                        <span><?php echo esc_html__('What do these mean?', 'shadowscan-security-link'); ?></span>
                                    </summary>
                                    <div class="shadowscan-tooltip-card" style="margin-top:6px;max-width:320px;">
                                        <div class="shadowscan-muted"><?php echo esc_html__('Managed by ShadowScan: ShadowScan can complete this change in-platform.', 'shadowscan-security-link'); ?></div>
                                        <div class="shadowscan-muted" style="margin-top:6px;"><?php echo esc_html__('External action: Requires changes by your host or developer (outside ShadowScan).', 'shadowscan-security-link'); ?></div>
                                    </div>
                                </details>
                            </div>
                        </div>
                        <div class="shadowscan-subtabs">
                            <button type="button" class="shadowscan-subtab is-active" data-group="all">
                                <?php echo esc_html__('All', 'shadowscan-security-link'); ?>
                                <?php if ((int) $action_required_count > 0): ?>
                                    <span class="shadowscan-tab-badge"><?php echo (int) $action_required_count; ?></span>
                                <?php endif; ?>
                            </button>
                            <?php foreach ($coverage_tabs as $index => $tab): ?>
                                <button type="button" class="shadowscan-subtab" data-group="<?php echo esc_attr($tab['id']); ?>">
                                    <?php echo esc_html($tab['label']); ?>
                                    <?php if (($tab['action_count'] ?? 0) > 0): ?>
                                        <span class="shadowscan-tab-badge"><?php echo (int) $tab['action_count']; ?></span>
                                    <?php endif; ?>
                                </button>
                            <?php endforeach; ?>
                        </div>
                        <div class="shadowscan-filter-bar" role="search">
                            <div class="shadowscan-search" id="shadowscan-coverage-search-wrap">
                                <label class="shadowscan-sr-only" for="shadowscan-coverage-search"><?php echo esc_html__('Search modules', 'shadowscan-security-link'); ?></label>
                                <input type="search" id="shadowscan-coverage-search" placeholder="<?php echo esc_attr__('Search modules…', 'shadowscan-security-link'); ?>" />
                                <button type="button" class="button shadowscan-search-clear" aria-label="<?php echo esc_attr__('Clear search', 'shadowscan-security-link'); ?>">
                                    <?php echo esc_html__('Clear', 'shadowscan-security-link'); ?>
                                </button>
                            </div>
                            <div class="shadowscan-filter-row">
                                <button type="button" class="shadowscan-filter is-active" data-status="all"><?php echo esc_html__('All', 'shadowscan-security-link'); ?> <span class="shadowscan-filter-count"><?php echo (int) $total_controls_count; ?></span></button>
                                <button type="button" class="shadowscan-filter" data-status="action_required"><?php echo esc_html__('Action required', 'shadowscan-security-link'); ?> <span class="shadowscan-filter-count"><?php echo (int) $action_required_count; ?></span></button>
                                <button type="button" class="shadowscan-filter" data-status="managed"><?php echo esc_html__('Managed by ShadowScan', 'shadowscan-security-link'); ?> <span class="shadowscan-filter-count"><?php echo (int) $managed_count; ?></span></button>
                                <button type="button" class="shadowscan-filter" data-status="needs_host"><?php echo esc_html__('External action', 'shadowscan-security-link'); ?> <span class="shadowscan-filter-count"><?php echo (int) $external_count; ?></span></button>
                                <button type="button" class="shadowscan-filter" data-status="covered"><?php echo esc_html__('Covered', 'shadowscan-security-link'); ?> <span class="shadowscan-filter-count"><?php echo (int) $covered_count; ?></span></button>
                            </div>
                            <div class="shadowscan-search-count" id="shadowscan-coverage-count" aria-live="polite" aria-atomic="true"></div>
                        </div>
                        <?php
                        foreach ($coverage_groups as $group_id => $group_meta) {
                            $group_entries = array_values(array_filter($coverage_registry, function ($entry) use ($group_id) {
                                return ($entry['group'] ?? '') === $group_id;
                            }));
                            if (empty($group_entries)) {
                                continue;
                            }
                            $summary = $coverage_group_summary[$group_id] ?? array();
                            $group_items = array();
                            foreach ($group_entries as $entry) {
                                $state = $coverage_states[$entry['id']] ?? array();
                                $label = (string) ($state['label'] ?? __('Monitoring planned', 'shadowscan-security-link'));
                                $group_items[] = array(
                                    'entry' => $entry,
                                    'label' => $label,
                                    'rank' => $coverage_rank($entry, $state),
                                    'title' => strtolower((string) ($entry['title'] ?? '')),
                                );
                            }
                            if (!empty($advanced_sections_by_group[$group_id])) {
                                foreach ($advanced_sections_by_group[$group_id] as $advanced_item) {
                                    $group_items[] = array(
                                        'advanced' => true,
                                        'entry' => $advanced_item,
                                        'label' => __('Advanced', 'shadowscan-security-link'),
                                        'rank' => 4,
                                        'title' => strtolower((string) ($advanced_item['title'] ?? '')),
                                    );
                                }
                            }
                            usort($group_items, function ($a, $b) {
                                if ($a['rank'] === $b['rank']) {
                                    return strcmp($a['title'], $b['title']);
                                }
                                return $a['rank'] <=> $b['rank'];
                            });
                            ?>
                            <div class="shadowscan-coverage-group" data-group="<?php echo esc_attr($group_id); ?>">
                                <div class="shadowscan-group-header">
                                    <div>
                                        <div class="shadowscan-control-title"><?php echo esc_html($group_meta['title'] ?? $group_id); ?></div>
                                        <div class="shadowscan-muted"><?php echo esc_html($group_meta['description'] ?? ''); ?></div>
                                    </div>
                                    <span class="shadowscan-coverage-pill <?php echo esc_attr($summary['tone'] ?? ''); ?>">
                                        <span class="shadowscan-pill-dot" aria-hidden="true"></span>
                                        <?php echo esc_html($summary['label'] ?? __('Monitoring planned', 'shadowscan-security-link')); ?>
                                    </span>
                                </div>
                                <div class="shadowscan-card-grid">
                                    <?php foreach ($group_items as $item): ?>
                                        <?php
                                        if (!empty($item['advanced'])) {
                                            $render_advanced_card($item['entry']);
                                        } else {
                                            $render_control_card($item['entry']);
                                        }
                                        ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <?php
                        }
                        ?>
                        </div>

                    <div style="margin-top: 20px;">
                        <?php if (!$ui_advanced): ?>
                            <?php $render_section(
                                __('Advanced controls', 'shadowscan-security-link'),
                                __('Enable advanced controls to adjust hardening, server settings, and offboarding actions.', 'shadowscan-security-link'),
                                function () use ($can_persist_advanced) {
                                    ?>
                                    <div class="shadowscan-actions">
                                        <button class="button button-primary" data-shadowscan-open-advanced <?php echo $can_persist_advanced ? '' : 'disabled'; ?>>
                                            <?php echo esc_html__('Show advanced controls', 'shadowscan-security-link'); ?>
                                        </button>
                                    </div>
                                    <?php if (!$can_persist_advanced): ?>
                                        <p class="shadowscan-note" style="margin-top: 8px;">
                                            <?php echo esc_html__('Advanced controls are available only with an active subscription.', 'shadowscan-security-link'); ?>
                                        </p>
                                    <?php endif; ?>
                                    <?php
                                }
                            ); ?>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="shadowscan-card" style="margin:0;">
                    <?php $render_section(
                        __('Priority Actions', 'shadowscan-security-link'),
                        __('Focus on the highest-impact gaps first.', 'shadowscan-security-link'),
                        function () use ($next_actions) {
                            if (empty($next_actions)) {
                                echo '<p class="shadowscan-note">' . esc_html__('No immediate actions required.', 'shadowscan-security-link') . '</p>';
                                return;
                            }
                            $max_visible = 5;
                            $visible = array_slice($next_actions, 0, $max_visible);
                            $hidden = array_slice($next_actions, $max_visible);
                            echo '<ul class="shadowscan-priority-actions">';
                            foreach ($visible as $action) {
                                $title = $action['title'] ?? '';
                                $label = $action['label'] ?? '';
                                $href = $action['href'] ?? '';
                                $target_attr = !empty($action['external']) ? ' target="_blank" rel="noreferrer"' : '';
                                echo '<li class="shadowscan-priority-action">';
                                if ($title !== '') {
                                    echo '<strong class="shadowscan-priority-title">' . esc_html($title) . '</strong>';
                                }
                                echo '<span class="shadowscan-priority-detail">' . esc_html($label) . '</span>';
                                if ($href !== '') {
                                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped URL and static attributes.
                                    echo '<a class="shadowscan-matter-link" href="' . esc_url($href) . '"' . $target_attr . '>' . esc_html__('View details', 'shadowscan-security-link') . ' &rarr;</a>';
                                }
                                echo '</li>';
                            }
                            foreach ($hidden as $action) {
                                $title = $action['title'] ?? '';
                                $label = $action['label'] ?? '';
                                $href = $action['href'] ?? '';
                                $target_attr = !empty($action['external']) ? ' target="_blank" rel="noreferrer"' : '';
                                echo '<li class="shadowscan-priority-action" data-hidden="1" style="display:none;">';
                                if ($title !== '') {
                                    echo '<strong class="shadowscan-priority-title">' . esc_html($title) . '</strong>';
                                }
                                echo '<span class="shadowscan-priority-detail">' . esc_html($label) . '</span>';
                                if ($href !== '') {
                                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped URL and static attributes.
                                    echo '<a class="shadowscan-matter-link" href="' . esc_url($href) . '"' . $target_attr . '>' . esc_html__('View details', 'shadowscan-security-link') . ' &rarr;</a>';
                                }
                                echo '</li>';
                            }
                            echo '</ul>';
                            if (!empty($hidden)) {
                                /* translators: %d: total priority actions count. */
                                $expand_label = sprintf(__('View all (%d)', 'shadowscan-security-link'), count($next_actions));
                                $collapse_label = __('View less', 'shadowscan-security-link');
                                echo '<button type="button" class="button button-small" data-shadowscan-toggle-actions data-expanded="0" data-expand-label="' . esc_attr($expand_label) . '" data-collapse-label="' . esc_attr($collapse_label) . '">' . esc_html($expand_label) . '</button>';
                            }
                        }
                    ); ?>

                    <?php $render_section(
                        __('What ShadowScan is doing', 'shadowscan-security-link'),
                        __('A live snapshot of monitoring, policy, and enforcement.', 'shadowscan-security-link'),
                        function () use ($monitoring_active, $policy_tier, $policy_updated_at, $format_time, $enforcement_active, $enforcement_available, $policy_stale) {
                            ?>
                            <dl class="shadowscan-kv-list">
                                <dt><?php echo esc_html__('Monitoring', 'shadowscan-security-link'); ?></dt>
                                <dd><?php echo esc_html($monitoring_active ? __('Active', 'shadowscan-security-link') : __('Inactive', 'shadowscan-security-link')); ?></dd>
                                <dt><?php echo esc_html__('Policy tier', 'shadowscan-security-link'); ?></dt>
                                <dd><?php echo esc_html(ucfirst($policy_tier)); ?></dd>
                                <dt><?php echo esc_html__('Enforcement', 'shadowscan-security-link'); ?></dt>
                                <dd><?php echo esc_html($enforcement_active . '/' . $enforcement_available); ?></dd>
                                <dt><?php echo esc_html__('Last policy', 'shadowscan-security-link'); ?></dt>
                                <dd>
                                    <?php echo esc_html($format_time($policy_updated_at)); ?>
                                    <?php if ($policy_stale): ?>
                                        · <?php echo esc_html__('Stale cache', 'shadowscan-security-link'); ?>
                                    <?php endif; ?>
                                </dd>
                            </dl>
                            <?php
                        }
                    ); ?>

                    <?php $render_scope_section(); ?>

                    <?php $render_section(
                        __('Open portal', 'shadowscan-security-link'),
                        __('Use the portal for billing, evidence exports, and incident triage.', 'shadowscan-security-link'),
                        function () {
                            ?>
                            <div class="shadowscan-actions">
                                <a class="button button-primary" href="<?php echo esc_url(SHADOWSCAN_PORTAL_URL); ?>" target="_blank" rel="noreferrer">
                                    <?php echo esc_html__('Open portal', 'shadowscan-security-link'); ?>
                                </a>
                            </div>
                            <?php
                        }
                    ); ?>
                </div>
            </div>
        <?php elseif ($tab === 'diagnostics'): ?>
            <?php $render_section(
                __('Connection health', 'shadowscan-security-link'),
                __('Detailed status from the connector and scheduling.', 'shadowscan-security-link'),
                function () use ($format_time, $last_ping, $next_heartbeat, $next_poll, $pending_heartbeat, $pending_results, $last_error_summary, $event_queue_count, $cron_last_tick, $cron_stale, $logging_degraded) {
                    ?>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Last heartbeat:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($format_time($last_ping)); ?>
                        · <?php echo esc_html__('Next heartbeat:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($next_heartbeat ? $format_time($next_heartbeat) : esc_html__('Not scheduled', 'shadowscan-security-link')); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Next poll:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($next_poll ? $format_time($next_poll) : esc_html__('Not scheduled', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Pending heartbeat:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($pending_heartbeat ? __('Yes', 'shadowscan-security-link') : __('No', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Pending results:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html(is_array($pending_results) ? count($pending_results) : 0); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Queued events:', 'shadowscan-security-link'); ?> <?php echo esc_html($event_queue_count); ?>
                        · <?php echo esc_html__('Last error:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_error_summary); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Last cron tick:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html($format_time($cron_last_tick)); ?>
                        <?php if ($cron_stale): ?>
                            · <?php echo esc_html__('Cron appears stale', 'shadowscan-security-link'); ?>
                        <?php endif; ?>
                        <?php if ($logging_degraded): ?>
                            · <?php echo esc_html__('Event delivery degraded', 'shadowscan-security-link'); ?>
                        <?php endif; ?>
                    </p>
                    <?php
                }
            ); ?>

            <?php $render_section(
                __('Connection status', 'shadowscan-security-link'),
                __('Live pairing and delivery diagnostics for support.', 'shadowscan-security-link'),
                function () use ($format_time, $event_queue_count) {
                    $paired = function_exists('shadowscan_is_paired') ? shadowscan_is_paired() : false;
                    $auth_invalid = function_exists('shadowscan_sync_is_auth_invalid') ? shadowscan_sync_is_auth_invalid() : false;
                    $auth_state = function_exists('shadowscan_sync_get_auth_invalid_state') ? shadowscan_sync_get_auth_invalid_state() : array();
                    $auth_invalid_since = (int) ($auth_state['auth_invalid_first_at'] ?? 0);
                    $last_auth_http_status = (int) ($auth_state['last_auth_http_status'] ?? 0);
                    $last_recovery_attempt_at = (int) ($auth_state['last_recovery_attempt_at'] ?? 0);
                    $recovery_attempt_count = (int) ($auth_state['recovery_attempt_count'] ?? 0);
                    $auth_recovery_cooldown_active = shadowscan_auth_recovery_cooldown_active();
                    $site_id = function_exists('shadowscan_get_site_id') ? shadowscan_get_site_id() : '';
                    $domain_id = function_exists('shadowscan_sync_get_domain_id') ? shadowscan_sync_get_domain_id() : '';
                    $last_ping_at = (int) get_option(SHADOWSCAN_OPTION_LAST_PING, 0);
                    $delivery = class_exists('ShadowScan_Storage') ? ShadowScan_Storage::get_json('event_delivery_last_attempt', array()) : array();
                    $last_delivery_status = is_array($delivery) ? (int) ($delivery['http_status'] ?? 0) : 0;
                    $last_delivery_error = is_array($delivery) ? (string) ($delivery['error'] ?? '') : '';
                    $last_event_id = class_exists('ShadowScan_Storage') ? (string) ShadowScan_Storage::get('last_event_id', '') : '';
                    $last_portal_ack_id = class_exists('ShadowScan_Storage') ? (string) ShadowScan_Storage::get('last_portal_ack_id', '') : '';
                    $connector_report = shadowscan_get_connector_report_data();
                    $core_minor_effective = isset($connector_report['core_minor_effective']) && is_array($connector_report['core_minor_effective'])
                        ? $connector_report['core_minor_effective']
                        : array();
                    $plugin_allowlist_effective = isset($connector_report['plugin_allowlist_effective']) && is_array($connector_report['plugin_allowlist_effective'])
                        ? $connector_report['plugin_allowlist_effective']
                        : array();
                    $unknown_keys = class_exists('ShadowScan_Security_Controls')
                        ? ShadowScan_Security_Controls::get_unknown_control_keys(true)
                        : (class_exists('ShadowScan_Storage') ? ShadowScan_Storage::get_json('unknown_control_keys', array()) : array());
                    $unknown_count = is_array($unknown_keys) ? count($unknown_keys) : 0;
                    $payload = array(
                        'paired' => $paired,
                        'auth_invalid' => $auth_invalid,
                        'auth_invalid_since' => $auth_invalid_since > 0 ? gmdate('c', $auth_invalid_since) : null,
                        'last_auth_http_status' => $last_auth_http_status > 0 ? $last_auth_http_status : null,
                        'last_recovery_attempt_at' => $last_recovery_attempt_at > 0 ? gmdate('c', $last_recovery_attempt_at) : null,
                        'recovery_attempt_count' => $recovery_attempt_count,
                        'auth_recovery_cooldown_active' => $auth_recovery_cooldown_active,
                        'site_id' => $site_id !== '' ? $site_id : null,
                        'domain_id' => $domain_id !== '' ? $domain_id : null,
                        'last_heartbeat_at' => $last_ping_at ? gmdate('c', $last_ping_at) : null,
                        'last_delivery_status' => $last_delivery_status > 0 ? $last_delivery_status : null,
                        'last_delivery_error' => $last_delivery_error !== '' ? $last_delivery_error : null,
                        'last_event_id' => $last_event_id !== '' ? $last_event_id : null,
                        'last_portal_ack_id' => $last_portal_ack_id !== '' ? $last_portal_ack_id : null,
                        'portal_reachable' => !empty($connector_report['portal_reachable']),
                        'heartbeat_reachable' => !empty($connector_report['heartbeat_reachable']),
                        'core_policy_reachable' => !empty($connector_report['core_policy_reachable']),
                        'commands_reachable' => !empty($connector_report['commands_reachable']),
                        'events_reachable' => !empty($connector_report['events_reachable']),
                        'last_policy_apply_result' => $connector_report['last_policy_apply_result'] ?? null,
                        'last_policy_apply_reason' => $connector_report['last_policy_apply_reason'] ?? null,
                        'last_applied_policy_version' => $connector_report['last_applied_policy_version'] ?? null,
                        'last_seen_policy_version' => $connector_report['last_seen_policy_version'] ?? null,
                        'core_minor_effective' => $connector_report['core_minor_effective'] ?? null,
                        'plugin_allowlist_effective' => $connector_report['plugin_allowlist_effective'] ?? null,
                        'unknown_control_keys_count' => $unknown_count,
                        'queue_depth' => $event_queue_count,
                        'portal_base_url' => SHADOWSCAN_API_BASE,
                    );
                    ?>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Paired:', 'shadowscan-security-link'); ?> <?php echo $paired ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?>
                        · <?php echo esc_html__('Auth invalid:', 'shadowscan-security-link'); ?> <?php echo $auth_invalid ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Auth invalid since:', 'shadowscan-security-link'); ?> <?php echo esc_html($auth_invalid_since > 0 ? $format_time($auth_invalid_since) : __('Not set', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Last auth status:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_auth_http_status > 0 ? (string) $last_auth_http_status : __('Unknown', 'shadowscan-security-link')); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Recovery attempts:', 'shadowscan-security-link'); ?> <?php echo esc_html((string) $recovery_attempt_count); ?>
                        · <?php echo esc_html__('Last recovery attempt:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_recovery_attempt_at > 0 ? $format_time($last_recovery_attempt_at) : __('Not set', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Cooldown:', 'shadowscan-security-link'); ?> <?php echo $auth_recovery_cooldown_active ? esc_html__('Active', 'shadowscan-security-link') : esc_html__('Inactive', 'shadowscan-security-link'); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Site ID:', 'shadowscan-security-link'); ?> <?php echo esc_html($site_id !== '' ? $site_id : __('Unknown', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Domain ID:', 'shadowscan-security-link'); ?> <?php echo esc_html($domain_id !== '' ? $domain_id : __('Unknown', 'shadowscan-security-link')); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Last heartbeat:', 'shadowscan-security-link'); ?> <?php echo esc_html($format_time($last_ping_at)); ?>
                        · <?php echo esc_html__('Last delivery status:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_delivery_status > 0 ? (string) $last_delivery_status : __('Unknown', 'shadowscan-security-link')); ?>
                    </p>
                    <?php if ($last_delivery_error !== ''): ?>
                        <p class="shadowscan-note"><?php echo esc_html__('Last delivery error:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_delivery_error); ?></p>
                    <?php endif; ?>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Last event ID:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_event_id !== '' ? $last_event_id : __('Unknown', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Last portal ack ID:', 'shadowscan-security-link'); ?> <?php echo esc_html($last_portal_ack_id !== '' ? $last_portal_ack_id : __('Unknown', 'shadowscan-security-link')); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Core minor gate:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html(isset($core_minor_effective['gate_status']) ? (string) $core_minor_effective['gate_status'] : __('Not set', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Reason:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html(isset($core_minor_effective['gate_reason']) ? (string) $core_minor_effective['gate_reason'] : __('Not set', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Global pause:', 'shadowscan-security-link'); ?>
                        <?php echo !empty($core_minor_effective['global_pause']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Plugin allowlist gate:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html(isset($plugin_allowlist_effective['gate_status']) ? (string) $plugin_allowlist_effective['gate_status'] : __('Not set', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Reason:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html(isset($plugin_allowlist_effective['gate_reason']) ? (string) $plugin_allowlist_effective['gate_reason'] : __('Not set', 'shadowscan-security-link')); ?>
                        · <?php echo esc_html__('Daily cap:', 'shadowscan-security-link'); ?>
                        <?php echo esc_html(isset($plugin_allowlist_effective['max_sites_per_day']) ? (string) $plugin_allowlist_effective['max_sites_per_day'] : __('Not set', 'shadowscan-security-link')); ?>
                    </p>
                    <textarea id="shadowscan-connection-report" class="shadowscan-report" readonly><?php echo esc_textarea(wp_json_encode($payload, JSON_PRETTY_PRINT)); ?></textarea>
                    <div class="shadowscan-actions">
                        <button class="button shadowscan-copy-report" type="button" data-copy-target="#shadowscan-connection-report" data-copy-default="<?php echo esc_attr__('Copy diagnostics', 'shadowscan-security-link'); ?>" data-copy-copied="<?php echo esc_attr__('Copied!', 'shadowscan-security-link'); ?>">
                            <?php echo esc_html__('Copy diagnostics', 'shadowscan-security-link'); ?>
                        </button>
                    </div>
                    <?php
                }
            ); ?>

            <?php $render_section(
                __('MU helper', 'shadowscan-security-link'),
                __('Must-use helper used for early error capture during remote diagnostics.', 'shadowscan-security-link'),
                function () {
                    $remote_enabled = shadowscan_remote_diagnostics_enabled();
                    $sentry_enabled = shadowscan_sentry_telemetry_enabled();
                    $installed = shadowscan_mu_helper_installed();
                    $path = shadowscan_mu_helper_path();
                    ?>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Sentry telemetry:', 'shadowscan-security-link'); ?>
                        <?php echo $sentry_enabled ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link'); ?>
                        ·
                        <?php echo esc_html__('Remote diagnostics:', 'shadowscan-security-link'); ?>
                        <?php echo $remote_enabled ? esc_html__('Enabled', 'shadowscan-security-link') : esc_html__('Disabled', 'shadowscan-security-link'); ?>
                        · <?php echo esc_html__('MU helper installed:', 'shadowscan-security-link'); ?>
                        <?php echo $installed ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?>
                    </p>
                    <p class="shadowscan-note">
                        <?php echo esc_html__('Path:', 'shadowscan-security-link'); ?> <?php echo esc_html($path); ?>
                    </p>
                    <?php if (!$sentry_enabled): ?>
                        <p class="shadowscan-note"><?php echo esc_html__('Enable Sentry telemetry to allow MU helper diagnostics.', 'shadowscan-security-link'); ?></p>
                    <?php elseif (!$remote_enabled): ?>
                        <p class="shadowscan-note"><?php echo esc_html__('Enable remote diagnostics to install the MU helper from the portal.', 'shadowscan-security-link'); ?></p>
                    <?php endif; ?>
                    <?php
                }
            ); ?>

            <?php if ($is_dev_env): ?>
                <?php $render_section(
                    __('Control registry (Dev Only)', 'shadowscan-security-link'),
                    __('Unknown control keys seen during status emission.', 'shadowscan-security-link'),
                    function () use ($format_time) {
                        $unknown = class_exists('ShadowScan_Security_Controls')
                            ? ShadowScan_Security_Controls::get_unknown_control_keys(true)
                            : (class_exists('ShadowScan_Storage') ? ShadowScan_Storage::get_json('unknown_control_keys', array()) : array());
                        if (!is_array($unknown)) {
                            $unknown = array();
                        }
                        uasort($unknown, static function ($a, $b) {
                            return (int) ($b['last_seen'] ?? 0) <=> (int) ($a['last_seen'] ?? 0);
                        });
                        $unknown_keys = array_keys($unknown);
                        $unknown_count = count($unknown_keys);
                        $unknown_keys = array_slice($unknown_keys, 0, 10);
                        ?>
                        <p class="shadowscan-note">
                            <?php echo esc_html__('unknown_control_keys_count:', 'shadowscan-security-link'); ?>
                            <?php echo esc_html((string) $unknown_count); ?>
                        </p>
                        <?php if ($unknown_count === 0): ?>
                            <p class="shadowscan-note"><?php echo esc_html__('No unknown control keys recorded.', 'shadowscan-security-link'); ?></p>
                        <?php else: ?>
                            <p class="shadowscan-note"><?php echo esc_html__('Last 10 unknown keys (most recent first):', 'shadowscan-security-link'); ?></p>
                            <ul class="shadowscan-summary-list">
                                <?php foreach ($unknown_keys as $key): ?>
                                    <?php $entry = $unknown[$key] ?? array(); ?>
                                    <li>
                                        <?php echo esc_html($key); ?>
                                        · <?php echo esc_html__('count', 'shadowscan-security-link'); ?>: <?php echo esc_html((string) ($entry['count'] ?? 0)); ?>
                                        · <?php echo esc_html__('last seen', 'shadowscan-security-link'); ?>: <?php echo esc_html($format_time((int) ($entry['last_seen'] ?? 0))); ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                        <?php
                    }
                ); ?>
            <?php endif; ?>

            <?php $render_section(
                __('Support & logs', 'shadowscan-security-link'),
                __('Download logs or share the connector report with support.', 'shadowscan-security-link'),
                function () use ($support_report) {
                    ?>
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top: 8px;">
                        <?php wp_nonce_field('shadowscan-dashboard'); ?>
                        <input type="hidden" name="action" value="shadowscan_download_logs" />
                        <input type="hidden" name="return_tab" value="diagnostics" />
                        <button class="button"><?php echo esc_html__('Download connector logs', 'shadowscan-security-link'); ?></button>
                    </form>
                    <div style="margin-top: 12px;">
                        <label class="shadowscan-note"><?php echo esc_html__('Support bundle', 'shadowscan-security-link'); ?></label>
                        <textarea id="shadowscan-status-report" class="shadowscan-report" readonly><?php echo esc_textarea($support_report); ?></textarea>
                        <button class="button shadowscan-copy-report" type="button" data-copy-target="#shadowscan-status-report" data-copy-default="<?php echo esc_attr__('Copy support bundle', 'shadowscan-security-link'); ?>" data-copy-copied="<?php echo esc_attr__('Copied!', 'shadowscan-security-link'); ?>">
                            <?php echo esc_html__('Copy support bundle', 'shadowscan-security-link'); ?>
                        </button>
                        <p class="shadowscan-note" style="margin-top: 8px;">
                            <?php echo esc_html__('Diagnostics may include technical details.', 'shadowscan-security-link'); ?>
                        </p>
                    </div>
                    <?php
                }
            ); ?>

            <?php $render_section(
                __('Evidence export', 'shadowscan-security-link'),
                __('Export a JSON snapshot of control coverage for support or compliance review.', 'shadowscan-security-link'),
                function () {
                    ?>
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top: 8px;">
                        <?php wp_nonce_field('shadowscan-dashboard'); ?>
                        <input type="hidden" name="action" value="shadowscan_export_owasp_evidence" />
                        <input type="hidden" name="return_tab" value="diagnostics" />
                        <button class="button"><?php echo esc_html__('Export evidence (JSON)', 'shadowscan-security-link'); ?></button>
                    </form>
                    <?php
                }
            ); ?>

            <?php $render_section(
                __('Self-check output', 'shadowscan-security-link'),
                __('Full results of the latest self-check.', 'shadowscan-security-link'),
                function () use ($last_self_check, $self_check_cooldown_remaining) {
                    ?>
                    <?php if (is_array($last_self_check) && !empty($last_self_check['payload'])): ?>
                        <p class="shadowscan-note">
                            <?php echo esc_html__('Result:', 'shadowscan-security-link'); ?>
                            <?php echo !empty($last_self_check['success']) ? esc_html__('Passed', 'shadowscan-security-link') : esc_html__('Failed', 'shadowscan-security-link'); ?>
                        </p>
                        <p class="shadowscan-note">
                            <?php echo esc_html__('Filesystem:', 'shadowscan-security-link'); ?>
                            <?php echo !empty($last_self_check['payload']['filesystem_available']) ? esc_html__('OK', 'shadowscan-security-link') : esc_html__('Unavailable', 'shadowscan-security-link'); ?>
                            · <?php echo esc_html__('Plugins writable:', 'shadowscan-security-link'); ?>
                            <?php echo !empty($last_self_check['payload']['plugins_dir_writable']) ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?>
                        </p>
                        <label class="shadowscan-note"><?php echo esc_html__('Raw self-check data', 'shadowscan-security-link'); ?></label>
                        <textarea class="shadowscan-report" readonly><?php echo esc_textarea(wp_json_encode($last_self_check, JSON_PRETTY_PRINT)); ?></textarea>
                    <?php else: ?>
                        <p class="shadowscan-note"><?php echo esc_html__('No self-check data available yet.', 'shadowscan-security-link'); ?></p>
                    <?php endif; ?>
                    <form method="post" style="margin-top: 12px;">
                        <?php wp_nonce_field('shadowscan-dashboard'); ?>
                        <input type="hidden" name="shadowscan_action" value="run_self_check" />
                        <button class="button" <?php echo $self_check_cooldown_remaining > 0 ? 'disabled' : ''; ?>><?php echo esc_html__('Run self-check', 'shadowscan-security-link'); ?></button>
                        <?php if ($self_check_cooldown_remaining > 0): ?>
                            <p class="shadowscan-note" style="margin-top: 8px;">
                                <?php /* translators: %d: seconds remaining. */ ?>
                                <?php echo esc_html(sprintf(__('Please wait %d seconds before running another self-check.', 'shadowscan-security-link'), $self_check_cooldown_remaining)); ?>
                            </p>
                        <?php endif; ?>
                    </form>
                    <?php
                }
            ); ?>

            <?php $render_section(
                __('Environment snapshot', 'shadowscan-security-link'),
                __('Reference information for support and troubleshooting.', 'shadowscan-security-link'),
                function () use ($third_party_plugins, $can_install_plugins) {
                    ?>
                    <p class="shadowscan-note"><?php echo esc_html__('WordPress:', 'shadowscan-security-link'); ?> <?php echo esc_html(get_bloginfo('version')); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('PHP:', 'shadowscan-security-link'); ?> <?php echo esc_html(PHP_VERSION); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('Connector:', 'shadowscan-security-link'); ?> <?php echo esc_html(SHADOWSCAN_PLUGIN_VERSION); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('Third-party security tools detected:', 'shadowscan-security-link'); ?> <?php echo esc_html($third_party_plugins ? implode(', ', $third_party_plugins) : __('None', 'shadowscan-security-link')); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('Can install plugins:', 'shadowscan-security-link'); ?> <?php echo $can_install_plugins ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('Can activate plugins:', 'shadowscan-security-link'); ?> <?php echo shadowscan_can_activate_plugins() ? esc_html__('Yes', 'shadowscan-security-link') : esc_html__('No', 'shadowscan-security-link'); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('Protection runs externally. No file-level firewall is installed.', 'shadowscan-security-link'); ?></p>
                    <p class="shadowscan-note"><?php echo esc_html__('ShadowScan never auto-deletes files.', 'shadowscan-security-link'); ?></p>
                    <?php
                }
            ); ?>

            <?php if ($is_dev_env): ?>
                <?php $render_section(
                    __('Subscription Override (Dev Only)', 'shadowscan-security-link'),
                    __('Use this to test billing states in development or staging. Overrides do not apply in production.', 'shadowscan-security-link'),
                    function () use ($override_state) {
                        $current_status = $override_state['status'] ?? 'none';
                        $grace_value = !empty($override_state['grace_end']) ? gmdate('Y-m-d H:i', (int) $override_state['grace_end']) : '';
                        $suspended_value = !empty($override_state['suspended_since']) ? gmdate('Y-m-d H:i', (int) $override_state['suspended_since']) : '';
                        ?>
                        <form method="post">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="subscription_override_set" />
                            <label for="subscription_override_status"><?php echo esc_html__('Override status', 'shadowscan-security-link'); ?></label>
                            <select name="subscription_override_status" id="subscription_override_status">
                                <option value="none" <?php selected($current_status, 'none'); ?>><?php echo esc_html__('None (use portal)', 'shadowscan-security-link'); ?></option>
                                <option value="trial" <?php selected($current_status, 'trial'); ?>><?php echo esc_html__('TRIAL', 'shadowscan-security-link'); ?></option>
                                <option value="active" <?php selected($current_status, 'active'); ?>><?php echo esc_html__('ACTIVE', 'shadowscan-security-link'); ?></option>
                                <option value="past_due" <?php selected($current_status, 'past_due'); ?>><?php echo esc_html__('PAST_DUE', 'shadowscan-security-link'); ?></option>
                                <option value="suspended" <?php selected($current_status, 'suspended'); ?>><?php echo esc_html__('SUSPENDED', 'shadowscan-security-link'); ?></option>
                                <option value="cancelled" <?php selected($current_status, 'cancelled'); ?>><?php echo esc_html__('CANCELLED', 'shadowscan-security-link'); ?></option>
                            </select>
                            <div style="margin-top: 10px;">
                                <label for="subscription_override_grace_end"><?php echo esc_html__('Grace end (for past due)', 'shadowscan-security-link'); ?></label>
                                <input type="text" id="subscription_override_grace_end" name="subscription_override_grace_end" placeholder="YYYY-MM-DD HH:MM" value="<?php echo esc_attr($grace_value); ?>" />
                            </div>
                            <div style="margin-top: 10px;">
                                <label for="subscription_override_suspended_since"><?php echo esc_html__('Suspended since (for lifecycle testing)', 'shadowscan-security-link'); ?></label>
                                <input type="text" id="subscription_override_suspended_since" name="subscription_override_suspended_since" placeholder="YYYY-MM-DD HH:MM" value="<?php echo esc_attr($suspended_value); ?>" />
                            </div>
                            <div class="shadowscan-actions">
                                <button class="button button-primary"><?php echo esc_html__('Apply override', 'shadowscan-security-link'); ?></button>
                            </div>
                        </form>
                        <form method="post" style="margin-top: 10px;">
                            <?php wp_nonce_field('shadowscan-dashboard'); ?>
                            <input type="hidden" name="shadowscan_action" value="subscription_override_clear" />
                            <button class="button"><?php echo esc_html__('Clear override', 'shadowscan-security-link'); ?></button>
                        </form>
                        <?php
                    }
                ); ?>
            <?php endif; ?>
        <?php elseif ($tab === 'emergency'): ?>
            <?php $render_section(
                __('Emergency containment', 'shadowscan-security-link'),
                __('Use only when active containment is required. This mode is temporary and reversible.', 'shadowscan-security-link'),
                function () use ($emergency_status, $format_time) {
                    ?>
                    <div class="shadowscan-warning">
                        <p style="margin:0 0 8px;">
                            <?php echo !empty($emergency_status['enabled']) ? esc_html__('Emergency containment is currently ON.', 'shadowscan-security-link') : esc_html__('Emergency containment is currently OFF.', 'shadowscan-security-link'); ?>
                        </p>
                        <?php if (!empty($emergency_status['enabled_at'])): ?>
                            <p style="margin:0 0 8px;">
                                <?php echo esc_html__('Enabled at:', 'shadowscan-security-link'); ?>
                                <?php echo esc_html($format_time((int) $emergency_status['enabled_at'])); ?>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($emergency_status['reason'])): ?>
                            <p style="margin:0;">
                                <?php echo esc_html__('Reason:', 'shadowscan-security-link'); ?>
                                <?php echo esc_html($emergency_status['reason']); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                    <form method="post" style="margin-top: 16px;">
                        <?php wp_nonce_field('shadowscan-dashboard'); ?>
                        <input type="hidden" name="shadowscan_action" value="toggle_emergency_mode" />
                        <input type="hidden" name="emergency_state" value="<?php echo !empty($emergency_status['enabled']) ? '0' : '1'; ?>" />
                        <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                            <input type="text" name="emergency_reason" placeholder="<?php echo esc_attr__('Reason (optional)', 'shadowscan-security-link'); ?>" />
                        </label>
                        <?php if (empty($emergency_status['enabled'])): ?>
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="checkbox" name="emergency_confirm" value="1" />
                                <span><?php echo esc_html__('I understand this will temporarily disable normal enforcement.', 'shadowscan-security-link'); ?></span>
                            </label>
                            <label style="display:flex;gap:10px;align-items:center;margin-bottom:10px;">
                                <input type="text" name="emergency_phrase" placeholder="<?php echo esc_attr__('Type ENABLE to confirm', 'shadowscan-security-link'); ?>" />
                            </label>
                        <?php endif; ?>
                        <button class="button button-danger">
                            <?php echo !empty($emergency_status['enabled']) ? esc_html__('Disable emergency mode', 'shadowscan-security-link') : esc_html__('Enable emergency mode', 'shadowscan-security-link'); ?>
                        </button>
                    </form>
                    <?php
                }
            ); ?>
        <?php elseif ($tab === 'contact'): ?>
            <?php
            $current_user = wp_get_current_user();
            $contact_email = $current_user && $current_user->user_email ? $current_user->user_email : get_bloginfo('admin_email');
            ?>
            <?php $render_section(
                __('Contact', 'shadowscan-security-link'),
                __('Send a message to the ShadowScan team. Include any relevant context so we can help faster.', 'shadowscan-security-link'),
                function () use ($contact_email, $connected) {
                    ?>
                    <?php if (!$connected): ?>
                        <p class="shadowscan-note" style="margin-top:0;">
                            <?php echo esc_html__('This site is not connected, so we cannot send a portal-backed request.', 'shadowscan-security-link'); ?>
                            <a class="shadowscan-muted-link" href="<?php echo esc_url(SHADOWSCAN_SUPPORT_URL); ?>" style="margin-left:6px;">
                                <?php echo esc_html__('Email support', 'shadowscan-security-link'); ?>
                            </a>
                        </p>
                    <?php endif; ?>
                    <form method="post">
                        <?php wp_nonce_field('shadowscan-dashboard'); ?>
                        <input type="hidden" name="shadowscan_action" value="submit_contact" />
                        <div class="shadowscan-grid" style="grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap: 12px;">
                            <label style="display:flex;flex-direction:column;gap:6px;">
                                <span class="shadowscan-muted"><?php echo esc_html__('Your email', 'shadowscan-security-link'); ?></span>
                                <input type="email" name="contact_email" value="<?php echo esc_attr($contact_email); ?>" required <?php echo $connected ? '' : 'disabled'; ?> />
                            </label>
                            <label style="display:flex;flex-direction:column;gap:6px;">
                                <span class="shadowscan-muted"><?php echo esc_html__('Subject', 'shadowscan-security-link'); ?></span>
                                <input type="text" name="contact_subject" placeholder="<?php echo esc_attr__('Billing, access, or technical question', 'shadowscan-security-link'); ?>" required <?php echo $connected ? '' : 'disabled'; ?> />
                            </label>
                        </div>
                        <label style="display:flex;flex-direction:column;gap:6px;margin-top:12px;">
                            <span class="shadowscan-muted"><?php echo esc_html__('Message', 'shadowscan-security-link'); ?></span>
                            <textarea name="contact_message" rows="6" placeholder="<?php echo esc_attr__('Describe the issue or request.', 'shadowscan-security-link'); ?>" required <?php echo $connected ? '' : 'disabled'; ?>></textarea>
                        </label>
                        <p class="shadowscan-note" style="margin-top:8px;">
                            <?php echo esc_html__('Your site details and connector version will be included to help us troubleshoot.', 'shadowscan-security-link'); ?>
                        </p>
                        <div class="shadowscan-actions" style="margin-top:12px;">
                            <button class="button button-primary" type="submit" <?php echo $connected ? '' : 'disabled'; ?>>
                                <?php echo esc_html__('Send message', 'shadowscan-security-link'); ?>
                            </button>
                        </div>
                    </form>
                    <?php
                }
            ); ?>
        <?php endif; ?>
        <div style="margin-top: 24px;">
            <?php $render_scope_section(); ?>
        </div>
        <?php if ($tab !== 'diagnostics'): ?>
            <textarea id="shadowscan-status-report" class="shadowscan-report" style="position:absolute;left:-9999px;top:-9999px;" readonly><?php echo esc_textarea($support_report); ?></textarea>
        <?php endif; ?>
        </div>
    </div>

    <?php
    if ($coverage_output_buffer) {
        $output = ob_get_clean();
        $control_ids = array();
        if (preg_match_all('/id="control-([^"]+)"/', $output, $matches) && !empty($matches[1])) {
            $control_ids = array_values($matches[1]);
        }
        if (!empty($control_ids)) {
            $counts = array_count_values($control_ids);
            $duplicates = array_keys(array_filter($counts, static function ($count) {
                return (int) $count > 1;
            }));
            if (!empty($duplicates)) {
                shadowscan_log_rate_limited(
                    'coverage_duplicate_control_ids',
                    '[ShadowScan Debug] Coverage output contains duplicate control IDs: ' . implode(', ', $duplicates),
                    900
                );
            }
        }
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Output is from WordPress core rendering functions which are already escaped
        echo $output;
    }
}

function shadowscan_render_owasp_controls(): void {
    $target = admin_url('admin.php?page=shadowscan&tab=coverage');
    shadowscan_log_ui_redirect_once('shadowscan-owasp', 'shadowscan');
    wp_safe_redirect($target);
    exit;
}
