<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Protection_Registry {
    public static function get_groups(): array {
        return array(
            'account_access' => array(
                'title' => __('Identity & Access', 'shadowscan-security-link'),
                'description' => __('Controls that reduce account takeover and privileged access abuse.', 'shadowscan-security-link'),
                'why' => __('Identity controls reduce the impact of compromised credentials.', 'shadowscan-security-link'),
            ),
            'updates_integrity' => array(
                'title' => __('Updates & Integrity Protection', 'shadowscan-security-link'),
                'description' => __('Controls that keep software current and detect unexpected change.', 'shadowscan-security-link'),
                'why' => __('Unpatched components are a frequent source of compromise.', 'shadowscan-security-link'),
            ),
            'application_hardening' => array(
                'title' => __('Application Hardening', 'shadowscan-security-link'),
                'description' => __('Controls that reduce unsafe configurations and risky exposure.', 'shadowscan-security-link'),
                'why' => __('Configuration drift can expose sensitive admin areas.', 'shadowscan-security-link'),
            ),
            'network_request' => array(
                'title' => __('Network & Request Protection', 'shadowscan-security-link'),
                'description' => __('Controls that limit risky requests and outbound activity.', 'shadowscan-security-link'),
                'why' => __('Request guardrails reduce abuse and unexpected outbound calls.', 'shadowscan-security-link'),
            ),
            'visibility' => array(
                'title' => __('Visibility & Evidence', 'shadowscan-security-link'),
                'description' => __('Signals that confirm monitoring, logs, and evidence exports.', 'shadowscan-security-link'),
                'why' => __('Clear visibility keeps response times predictable.', 'shadowscan-security-link'),
            ),
        );
    }

    public static function get_registry(): array {
        $entries = array();
        if (function_exists('shadowscan_owasp_get_registry')) {
            $owasp_registry = shadowscan_owasp_get_registry();
            foreach ($owasp_registry as $category) {
                $category_id = (string) ($category['category_id'] ?? '');
                $controls = $category['controls'] ?? array();
                foreach ($controls as $control) {
                    $control_key = (string) ($control['control_key'] ?? '');
                    if ($control_key === '') {
                        continue;
                    }
                    $entries[] = array(
                        'id' => 'owasp.' . $control_key,
                        'title' => (string) ($control['name'] ?? $control_key),
                        'description' => (string) ($control['description'] ?? ''),
                        'group' => self::group_for_owasp($category_id),
                        'signals' => array($control_key),
                        'enforcement_supported' => !empty($control['toggle_option_key']),
                        'enforcement_toggle_key' => $control['toggle_option_key'] ?? '',
                        'recommended_action' => function_exists('shadowscan_owasp_recommended_action')
                            ? shadowscan_owasp_recommended_action($control_key)
                            : '',
                        'technical_mapping' => 'OWASP mapping only',
                        'owasp_ids' => $category_id ? array($category_id) : array(),
                        'control_keys' => array($control_key),
                        'notes' => '',
                    );
                }
            }
        }

        $entries[] = array(
            'id' => 'access.mfa',
            'title' => __('Multi-factor authentication', 'shadowscan-security-link'),
            'description' => __('Tracks admin MFA coverage and enforcement readiness.', 'shadowscan-security-link'),
            'group' => 'account_access',
            'signals' => array('mfa_coverage_snapshot'),
            'enforcement_supported' => true,
            'enforcement_toggle_key' => 'mfa_enforce_admins',
            'recommended_action' => __('Enroll remaining administrators in MFA.', 'shadowscan-security-link'),
            'technical_mapping' => 'MFA coverage snapshot',
            'owasp_ids' => array('A07'),
            'control_keys' => array('mfa_coverage_snapshot'),
            'notes' => '',
            'status_callback' => 'shadowscan_protection_status_mfa',
        );

        $entries[] = array(
            'id' => 'updates.auto_updates',
            'title' => __('Plugin auto-updates', 'shadowscan-security-link'),
            'description' => __('Controls automated plugin updates (themes excluded).', 'shadowscan-security-link'),
            'group' => 'updates_integrity',
            'signals' => array('autoupdate_plugins_detect'),
            'enforcement_supported' => true,
            'enforcement_toggle_key' => 'autoupdate_plugins_enabled',
            'recommended_action' => __('Enable auto-updates for critical plugins or use an allowlist.', 'shadowscan-security-link'),
            'technical_mapping' => 'security_control.autoupdate',
            'owasp_ids' => array('A06'),
            'control_keys' => array('autoupdate_plugins_detect'),
            'notes' => '',
            'status_callback' => 'shadowscan_protection_status_autoupdate',
        );

        $entries[] = array(
            'id' => 'access.password_policy',
            'title' => __('Password policy (change/reset)', 'shadowscan-security-link'),
            'description' => __('Enforces minimum password length when passwords are changed or reset. Optional breached-password blocking.', 'shadowscan-security-link'),
            'group' => 'account_access',
            'signals' => array('password_policy_enforced'),
            'enforcement_supported' => true,
            'enforcement_toggle_key' => 'password_enforce_min_length',
            'recommended_action' => __('Require 12+ character passphrases for admin accounts.', 'shadowscan-security-link'),
            'technical_mapping' => 'security_control.password_policy',
            'owasp_ids' => array('A07'),
            'control_keys' => array('password_policy_enforced'),
            'notes' => '',
            'status_callback' => 'shadowscan_protection_status_password_policy',
        );

        $entries[] = array(
            'id' => 'access.admin_geo_guard',
            'title' => __('Admin Access Guard', 'shadowscan-security-link'),
            'description' => __('Monitor where wp-admin and wp-login are accessed from.', 'shadowscan-security-link'),
            'group' => 'account_access',
            'signals' => array('admin_geo_observed'),
            'enforcement_supported' => false,
            'enforcement_toggle_key' => '',
            'recommended_action' => __('Manage allowed locations in the ShadowScan portal.', 'shadowscan-security-link'),
            'technical_mapping' => 'security_control.admin_geo_guard',
            'owasp_ids' => array('A01'),
            'control_keys' => array('admin_geo_guard'),
            'notes' => '',
            'status_callback' => 'shadowscan_protection_status_admin_geo_guard',
            'portal_link' => SHADOWSCAN_PORTAL_URL,
        );

        $entries[] = array(
            'id' => 'visibility.delivery',
            'title' => __('Event delivery health', 'shadowscan-security-link'),
            'description' => __('Confirms domain logs and evidence delivery are flowing.', 'shadowscan-security-link'),
            'group' => 'visibility',
            'signals' => array('event_delivery_health'),
            'enforcement_supported' => false,
            'enforcement_toggle_key' => '',
            'recommended_action' => __('Resolve queue backlogs and keep cron running.', 'shadowscan-security-link'),
            'technical_mapping' => 'monitoring.queue',
            'owasp_ids' => array('A09'),
            'control_keys' => array(),
            'notes' => '',
            'status_callback' => 'shadowscan_protection_status_delivery',
        );

        $entries[] = array(
            'id' => 'visibility.evidence',
            'title' => __('Evidence export readiness', 'shadowscan-security-link'),
            'description' => __('Confirms evidence exports and snapshots are available.', 'shadowscan-security-link'),
            'group' => 'visibility',
            'signals' => array('evidence_export'),
            'enforcement_supported' => false,
            'enforcement_toggle_key' => '',
            'recommended_action' => __('Export evidence for audits when requested.', 'shadowscan-security-link'),
            'technical_mapping' => 'evidence.export',
            'owasp_ids' => array('A09'),
            'control_keys' => array(),
            'notes' => '',
            'status_callback' => 'shadowscan_protection_status_evidence',
        );

        return $entries;
    }

    private static function group_for_owasp(string $category_id): string {
        if (in_array($category_id, array('A01', 'A07'), true)) {
            return 'account_access';
        }
        if (in_array($category_id, array('A06', 'A08'), true)) {
            return 'updates_integrity';
        }
        if ($category_id === 'A09') {
            return 'visibility';
        }
        if ($category_id === 'A10') {
            return 'network_request';
        }
        return 'application_hardening';
    }
}
