<?php
if (!defined('ABSPATH')) {
    exit;
}

const SHADOWSCAN_OPTION_SYNC_STATE = 'shadowscan_sync_state';

function shadowscan_get_sync_state(): array {
    $state = is_multisite()
        ? get_site_option(SHADOWSCAN_OPTION_SYNC_STATE, array())
        : get_option(SHADOWSCAN_OPTION_SYNC_STATE, array());
    if (!is_array($state)) {
        $state = array();
    }
    if (!isset($state['command_acks']) || !is_array($state['command_acks'])) {
        $state['command_acks'] = array();
    }
    return $state;
}

function shadowscan_set_sync_state(array $state): void {
    if (is_multisite()) {
        update_site_option(SHADOWSCAN_OPTION_SYNC_STATE, $state);
    } else {
        update_option(SHADOWSCAN_OPTION_SYNC_STATE, $state);
    }
}

function shadowscan_sync_get_site_id(): string {
    $state = shadowscan_get_sync_state();
    return isset($state['site_id']) ? (string) $state['site_id'] : '';
}

function shadowscan_sync_get_identity(): array {
    $state = shadowscan_get_sync_state();
    $site_id = isset($state['site_id']) ? (string) $state['site_id'] : '';
    $domain_id = isset($state['domain_id']) ? (string) $state['domain_id'] : '';

    if ($site_id === '' && defined('SHADOWSCAN_OPTION_SITE_ID')) {
        $legacy_site_id = is_multisite()
            ? get_site_option(SHADOWSCAN_OPTION_SITE_ID, '')
            : get_option(SHADOWSCAN_OPTION_SITE_ID, '');
        if (is_string($legacy_site_id) && $legacy_site_id !== '') {
            $site_id = $legacy_site_id;
            $state['site_id'] = $site_id;
            shadowscan_set_sync_state($state);
        }
    }

    return array(
        'site_id' => $site_id,
        'domain_id' => $domain_id,
    );
}

function shadowscan_sync_set_site_id(string $site_id): void {
    $site_id = trim($site_id);
    if ($site_id === '') {
        return;
    }
    $state = shadowscan_get_sync_state();
    $state['site_id'] = $site_id;
    shadowscan_set_sync_state($state);
}

function shadowscan_sync_set_identity(string $site_id, string $domain_id = ''): void {
    $site_id = trim($site_id);
    $domain_id = trim($domain_id);
    $state = shadowscan_get_sync_state();
    if ($site_id !== '') {
        $state['site_id'] = $site_id;
    }
    if ($domain_id !== '') {
        $state['domain_id'] = $domain_id;
    }
    shadowscan_set_sync_state($state);
}

function shadowscan_sync_get_domain_id(): string {
    $state = shadowscan_get_sync_state();
    return isset($state['domain_id']) ? (string) $state['domain_id'] : '';
}

function shadowscan_sync_set_domain_id(string $domain_id): void {
    $domain_id = trim($domain_id);
    if ($domain_id === '') {
        return;
    }
    $state = shadowscan_get_sync_state();
    $state['domain_id'] = $domain_id;
    shadowscan_set_sync_state($state);
}

function shadowscan_sync_clear_identity(): void {
    $state = shadowscan_get_sync_state();
    unset(
        $state['site_id'],
        $state['domain_id'],
        $state['auth_invalid'],
        $state['auth_invalid_first_at'],
        $state['last_auth_http_status'],
        $state['last_recovery_attempt_at'],
        $state['recovery_attempt_count']
    );
    shadowscan_set_sync_state($state);
}

function shadowscan_sync_set_auth_invalid(bool $invalid, string $reason = '', ?int $status = null): void {
    $state = shadowscan_get_sync_state();
    if ($invalid) {
        $existing = isset($state['auth_invalid']) && is_array($state['auth_invalid']) ? $state['auth_invalid'] : array();
        $first_at = (int) ($state['auth_invalid_first_at'] ?? ($existing['first_at'] ?? ($existing['at'] ?? 0)));
        if ($first_at <= 0) {
            $first_at = time();
        }
        $state['auth_invalid'] = array(
            'at' => time(),
            'first_at' => $first_at,
            'reason' => $reason,
            'status' => $status,
        );
        $state['auth_invalid_first_at'] = $first_at;
        if ($status !== null) {
            $state['last_auth_http_status'] = (int) $status;
        }
    } else {
        unset(
            $state['auth_invalid'],
            $state['auth_invalid_first_at'],
            $state['last_auth_http_status'],
            $state['last_recovery_attempt_at'],
            $state['recovery_attempt_count']
        );
    }
    shadowscan_set_sync_state($state);
}

function shadowscan_sync_is_auth_invalid(): bool {
    $state = shadowscan_get_sync_state();
    return !empty($state['auth_invalid']);
}

function shadowscan_sync_get_auth_invalid_state(): array {
    $state = shadowscan_get_sync_state();
    $meta = isset($state['auth_invalid']) && is_array($state['auth_invalid']) ? $state['auth_invalid'] : array();
    $first_at = (int) ($state['auth_invalid_first_at'] ?? ($meta['first_at'] ?? ($meta['at'] ?? 0)));
    $last_status = (int) ($state['last_auth_http_status'] ?? ($meta['status'] ?? 0));
    return array(
        'auth_invalid' => !empty($meta),
        'auth_invalid_first_at' => $first_at,
        'last_auth_http_status' => $last_status,
        'last_recovery_attempt_at' => (int) ($state['last_recovery_attempt_at'] ?? 0),
        'recovery_attempt_count' => (int) ($state['recovery_attempt_count'] ?? 0),
    );
}

function shadowscan_sync_get_auth_invalid_since(): int {
    $meta = shadowscan_sync_get_auth_invalid_state();
    return (int) ($meta['auth_invalid_first_at'] ?? 0);
}

function shadowscan_sync_get_recovery_attempt_count(): int {
    $meta = shadowscan_sync_get_auth_invalid_state();
    return (int) ($meta['recovery_attempt_count'] ?? 0);
}

function shadowscan_sync_get_last_recovery_attempt_at(): int {
    $meta = shadowscan_sync_get_auth_invalid_state();
    return (int) ($meta['last_recovery_attempt_at'] ?? 0);
}

function shadowscan_sync_mark_recovery_attempt(): void {
    $state = shadowscan_get_sync_state();
    $state['last_recovery_attempt_at'] = time();
    $state['recovery_attempt_count'] = (int) ($state['recovery_attempt_count'] ?? 0) + 1;
    shadowscan_set_sync_state($state);
}

function shadowscan_set_last_apply_result(array $result): void {
    $state = shadowscan_get_sync_state();
    $status = isset($result['status']) ? sanitize_text_field((string) $result['status']) : '';
    if (!in_array($status, array('ok', 'error', 'skipped'), true)) {
        if ($status === 'success') {
            $status = 'ok';
        } elseif ($status === 'failed') {
            $status = 'error';
        } else {
            $status = 'error';
        }
    }
    $normalized = array(
        'policy_version' => isset($result['policy_version']) ? sanitize_text_field((string) $result['policy_version']) : null,
        'status' => $status,
        'error_code' => isset($result['error_code']) ? sanitize_text_field((string) $result['error_code']) : null,
        'error_message' => isset($result['error_message']) ? shadowscan_sanitize_error_message((string) $result['error_message']) : null,
        'reason' => isset($result['reason']) ? sanitize_text_field((string) $result['reason']) : null,
        'applied_at' => isset($result['applied_at']) ? (int) $result['applied_at'] : time(),
    );
    $state['last_apply_result'] = $normalized;
    $state['last_policy_apply_at'] = (int) $normalized['applied_at'];
    $state['last_policy_apply_result'] = (string) $normalized['status'];
    $state['last_policy_apply_reason'] = isset($normalized['reason']) ? (string) $normalized['reason'] : '';
    $state['last_policy_error_code'] = isset($normalized['error_code']) ? (string) $normalized['error_code'] : '';
    $state['last_policy_error_message'] = isset($normalized['error_message']) ? (string) $normalized['error_message'] : '';
    shadowscan_set_sync_state($state);
}

function shadowscan_set_policy_versions($last_seen, $last_applied = null): void {
    $state = shadowscan_get_sync_state();
    if ($last_seen !== null) {
        $state['last_seen_policy_version'] = $last_seen;
    }
    if ($last_applied !== null) {
        $state['last_applied_policy_version'] = $last_applied;
    }
    shadowscan_set_sync_state($state);
}

function shadowscan_ack_command(string $id, array $ack): void {
    $state = shadowscan_get_sync_state();
    $acks = $state['command_acks'] ?? array();
    if (!is_array($acks)) {
        $acks = array();
    }
    $acks[$id] = $ack;
    $state['command_acks'] = shadowscan_prune_command_acks($acks, 50);
    shadowscan_set_sync_state($state);
}

function shadowscan_prune_command_acks(array $acks, int $limit): array {
    uasort($acks, static function ($a, $b) {
        $a_time = isset($a['handled_at']) ? (int) $a['handled_at'] : 0;
        $b_time = isset($b['handled_at']) ? (int) $b['handled_at'] : 0;
        return $b_time <=> $a_time;
    });
    return array_slice($acks, 0, $limit, true);
}

function shadowscan_compare_policy_version($incoming, $applied): ?int {
    if ($incoming === null || $incoming === '') {
        return null;
    }
    if ($applied === null || $applied === '') {
        return 1;
    }
    $incoming_str = is_string($incoming) || is_numeric($incoming) ? (string) $incoming : '';
    $applied_str = is_string($applied) || is_numeric($applied) ? (string) $applied : '';
    if ($incoming_str === '' || $applied_str === '') {
        return null;
    }
    if (ctype_digit($incoming_str) && ctype_digit($applied_str)) {
        $incoming_int = (int) $incoming_str;
        $applied_int = (int) $applied_str;
        return $incoming_int <=> $applied_int;
    }
    $incoming_numeric = ctype_digit($incoming_str);
    $applied_numeric = ctype_digit($applied_str);
    if ($incoming_numeric && !$applied_numeric) {
        // Legacy/dev sentinel applied values (for example "dev-seeded") should
        // never block a real numeric policy version from applying.
        return 1;
    }
    if (!$incoming_numeric && $applied_numeric) {
        return -1;
    }
    if ($incoming_str === $applied_str) {
        return 0;
    }
    return strcmp($incoming_str, $applied_str) > 0 ? 1 : -1;
}

function shadowscan_policy_version_is_legacy_migration($incoming, $applied): bool {
    $incoming_str = is_string($incoming) || is_numeric($incoming) ? (string) $incoming : '';
    $applied_str = is_string($applied) || is_numeric($applied) ? (string) $applied : '';
    if ($incoming_str === '' || $applied_str === '') {
        return false;
    }
    return ctype_digit($incoming_str) && !ctype_digit($applied_str);
}

function shadowscan_mark_policy_version_migrated(string $from_version, string $to_version): void {
    $state = shadowscan_get_sync_state();
    $state['policy_version_migrated_from_legacy'] = true;
    $state['policy_version_migrated_at'] = time();
    $state['policy_version_migrated_from'] = sanitize_text_field($from_version);
    $state['policy_version_migrated_to'] = sanitize_text_field($to_version);
    shadowscan_set_sync_state($state);
}

function shadowscan_sanitize_error_message(string $message): string {
    $message = wp_strip_all_tags($message);
    return sanitize_text_field($message);
}
