<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Subscription_State {
    public const STATUS_TRIAL = 'trial';
    public const STATUS_ACTIVE = 'active';
    public const STATUS_PAST_DUE = 'past_due';
    public const STATUS_SUSPENDED = 'suspended';
    public const STATUS_CANCELLED = 'cancelled';

    public const FEATURE_HARDENING = 'hardening';
    public const FEATURE_HEADERS = 'security_headers';
    public const FEATURE_SAFE_MODE = 'safe_mode';
    public const FEATURE_HTACCESS = 'htaccess';
    public const FEATURE_REMOTE_DIAGNOSTICS = 'remote_diagnostics';
    public const FEATURE_EMERGENCY = 'emergency';
    public const FEATURE_GUARD_PROFILE = 'guard_profile';

    private const OPTION_STATE = 'shadowscan_subscription_state';
    private const OPTION_STATE_UPDATED = 'shadowscan_subscription_state_updated';
    private const OPTION_LKG = 'shadowscan_subscription_state_lkg';
    private const OPTION_LKG_AT = 'shadowscan_subscription_state_lkg_at';
    private const OPTION_PORTAL_UNREACHABLE = 'shadowscan_subscription_portal_unreachable';
    private const CACHE_TTL = 900;
    private const LKG_TTL = 86400;

    public static function get_current_state(): array {
        $now = time();
        $stale = false;
        $override = array();
        $override_set = false;

        if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env()) {
            $override = self::get_override_state();
            if (is_array($override) && !empty($override['status'])) {
                $override_set = true;
            }
        }

        $force_override = defined('SHADOWSCAN_FORCE_LOCAL_OVERRIDE') && SHADOWSCAN_FORCE_LOCAL_OVERRIDE;
        $portal_unreachable = self::is_portal_unreachable();

        if ($override_set && ($portal_unreachable || $force_override || (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env()))) {
            $state = self::normalize_state($override);
            $state['override'] = true;
            $state['override_set_by'] = isset($override['set_by_user_id']) ? (int) $override['set_by_user_id'] : 0;
            $state['updated_at'] = (int) ($override['updated_at'] ?? $now);
            $state['stale'] = false;
            $state['portal_unreachable'] = $portal_unreachable;
            return $state;
        }

        $state = get_option(self::OPTION_STATE, array());
        $updated_at = (int) get_option(self::OPTION_STATE_UPDATED, 0);

        if (!$updated_at || ($now - $updated_at) > self::CACHE_TTL) {
            $lkg = get_option(self::OPTION_LKG, array());
            $lkg_at = (int) get_option(self::OPTION_LKG_AT, 0);
            if (is_array($lkg) && !empty($lkg) && $lkg_at && ($now - $lkg_at) <= self::LKG_TTL) {
                $state = $lkg;
                $updated_at = $lkg_at;
                $stale = true;
            }
        }

        $state = self::normalize_state($state);
        if (!$updated_at && !empty($state['updated_at'])) {
            $updated_at = (int) $state['updated_at'];
        }
        $state['updated_at'] = $updated_at;
        $state['stale'] = $stale;
        $state['portal_unreachable'] = $portal_unreachable;

        return $state;
    }

    public static function ingest_from_response(array $response): bool {
        $payload = array();
        $signature = '';

        if (isset($response['subscription']) && is_array($response['subscription'])) {
            $payload = $response['subscription'];
            $signature = isset($response['subscription_signature']) ? (string) $response['subscription_signature'] : '';
            if (isset($payload['signature'])) {
                $signature = (string) $payload['signature'];
                unset($payload['signature']);
            }
        }

        if (empty($payload)) {
            return false;
        }

        return self::ingest_payload($payload, $signature);
    }

    public static function ingest_payload(array $payload, string $signature): bool {
        $payload = self::normalize_state($payload);
        if (!self::verify_signature($payload, $signature)) {
            return false;
        }

        $previous = get_option(self::OPTION_STATE, array());
        $previous_status = strtolower((string) ($previous['status'] ?? ''));
        $new_status = $payload['status'];

        update_option(self::OPTION_STATE, $payload, false);
        update_option(self::OPTION_STATE_UPDATED, time(), false);
        update_option(self::OPTION_LKG, $payload, false);
        update_option(self::OPTION_LKG_AT, time(), false);
        self::clear_portal_unreachable();

        if ($previous_status && $previous_status !== $new_status) {
            self::emit_event(
                'SUBSCRIPTION_STATUS_CHANGED',
                'info',
                'Subscription status changed',
                array(
                    'from' => $previous_status,
                    'to' => $new_status,
                    'updated_at' => (int) ($payload['updated_at'] ?? time()),
                )
            );
        }

        if (in_array($new_status, array(self::STATUS_SUSPENDED, self::STATUS_CANCELLED), true)) {
            delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS);
            delete_option(SHADOWSCAN_OPTION_REMOTE_DIAGNOSTICS_EXPIRES);
        }

        return true;
    }

    public static function mark_portal_unreachable(): void {
        update_option(self::OPTION_PORTAL_UNREACHABLE, time(), false);
    }

    public static function clear_portal_unreachable(): void {
        delete_option(self::OPTION_PORTAL_UNREACHABLE);
    }

    public static function is_portal_unreachable(): bool {
        $timestamp = (int) get_option(self::OPTION_PORTAL_UNREACHABLE, 0);
        return $timestamp && (time() - $timestamp) <= self::LKG_TTL;
    }

    public static function is_feature_allowed(string $feature, ?array $state = null): bool {
        $state = $state ?: self::get_current_state();
        $status = $state['status'] ?? self::STATUS_TRIAL;
        $allowed = self::features_for_status($status);
        if (!in_array($feature, $allowed, true)) {
            return false;
        }

        if ($status === self::STATUS_ACTIVE) {
            $features_override = $state['features'] ?? array();
            if (is_array($features_override) && !empty($features_override)) {
                return in_array($feature, $features_override, true);
            }
        }

        return true;
    }

    public static function get_protection_state(?array $state = null): string {
        $state = $state ?: self::get_current_state();
        $status = $state['status'] ?? self::STATUS_TRIAL;
        $now = time();

        if ($status === self::STATUS_ACTIVE) {
            return 'live';
        }
        if ($status === self::STATUS_PAST_DUE) {
            return 'live';
        }
        if ($status === self::STATUS_CANCELLED) {
            return 'paused';
        }
        if ($status !== self::STATUS_SUSPENDED) {
            return 'paused';
        }

        $suspended_since = (int) ($state['suspended_since'] ?? 0);
        if ($suspended_since <= 0) {
            $suspended_since = (int) ($state['updated_at'] ?? 0);
        }
        if ($suspended_since <= 0) {
            $suspended_since = $now;
        }

        $days = max(0, (int) floor(($now - $suspended_since) / DAY_IN_SECONDS));
        $stale_after = self::policy_days('SHADOWSCAN_STALE_DAYS_AFTER_SUSPEND', 0);
        $outdated_after = self::policy_days('SHADOWSCAN_OUTDATED_AFTER_DAYS', 30);
        $paused_after = self::policy_days('SHADOWSCAN_PAUSED_AFTER_DAYS', 60);

        if ($days >= $paused_after) {
            return 'paused';
        }
        if ($days >= $outdated_after) {
            return 'outdated';
        }
        if ($days >= $stale_after) {
            return 'stale';
        }

        return 'stale';
    }

    public static function emit_event(string $type, string $severity, string $summary, array $details = array()): void {
        if (class_exists('ShadowScan_Signal_Manager')) {
            ShadowScan_Signal_Manager::emit($type, $severity, $summary, $details);
        }
    }

    private static function normalize_state(array $state): array {
        $status = strtolower((string) ($state['status'] ?? ''));
        $allowed = array(
            self::STATUS_TRIAL,
            self::STATUS_ACTIVE,
            self::STATUS_PAST_DUE,
            self::STATUS_SUSPENDED,
            self::STATUS_CANCELLED,
        );
        if (!in_array($status, $allowed, true)) {
            $status = self::STATUS_TRIAL;
        }

        $features = $state['features'] ?? array();
        if (!is_array($features)) {
            $features = array();
        }

        return array(
            'status' => $status,
            'grace_end' => isset($state['grace_end']) ? self::parse_time($state['grace_end']) : 0,
            'suspended_since' => isset($state['suspended_since']) ? self::parse_time($state['suspended_since']) : 0,
            'plan' => isset($state['plan']) ? sanitize_text_field((string) $state['plan']) : '',
            'features' => $features,
            'updated_at' => isset($state['updated_at']) ? self::parse_time($state['updated_at']) : 0,
            'test_mode' => !empty($state['test_mode']),
            'emulated' => !empty($state['emulated']),
            'emulated_status' => isset($state['emulated_status']) ? sanitize_text_field((string) $state['emulated_status']) : '',
            'emulated_by_admin_id' => isset($state['emulated_by_admin_id']) ? sanitize_text_field((string) $state['emulated_by_admin_id']) : '',
        );
    }

    private static function verify_signature(array $payload, string $signature): bool {
        if ($signature === '') {
            return false;
        }
        $key = self::signing_key();
        if ($key === '') {
            return false;
        }
        $canonical = self::canonicalize($payload);
        $expected = hash_hmac('sha256', $canonical, $key, true);

        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
        $sig_bin = base64_decode($signature, true);
        if ($sig_bin === false) {
            $sig_bin = hex2bin($signature);
        }
        if ($sig_bin === false) {
            return false;
        }

        return hash_equals($expected, $sig_bin);
    }

    private static function signing_key(): string {
        if (defined('SHADOWSCAN_SUBSCRIPTION_SIGNING_KEY') && SHADOWSCAN_SUBSCRIPTION_SIGNING_KEY) {
            return SHADOWSCAN_SUBSCRIPTION_SIGNING_KEY;
        }
        $env = getenv('SHADOWSCAN_SUBSCRIPTION_SIGNING_KEY');
        if (is_string($env) && $env !== '') {
            return $env;
        }
        return shadowscan_get_site_token();
    }

    private static function canonicalize(array $payload): string {
        $payload = self::recursive_ksort($payload);
        return wp_json_encode($payload);
    }

    private static function recursive_ksort(array $payload): array {
        foreach ($payload as $key => $value) {
            if (is_array($value)) {
                $payload[$key] = self::recursive_ksort($value);
            }
        }
        ksort($payload);
        return $payload;
    }

    private static function features_for_status(string $status): array {
        $base = array(
            'overview',
            'self_check',
            'diagnostics',
            'support_bundle',
            'reconnect',
            'open_portal',
            'billing',
        );

        if ($status === self::STATUS_ACTIVE) {
            return array_merge(
                $base,
                array(
                    self::FEATURE_HARDENING,
                    self::FEATURE_HEADERS,
                    self::FEATURE_SAFE_MODE,
                    self::FEATURE_HTACCESS,
                    self::FEATURE_REMOTE_DIAGNOSTICS,
                    self::FEATURE_EMERGENCY,
                    self::FEATURE_GUARD_PROFILE,
                )
            );
        }

        return $base;
    }

    private static function parse_time($value): int {
        if (is_int($value) || is_float($value)) {
            return (int) $value;
        }
        if (is_string($value)) {
            $parsed = strtotime($value);
            if ($parsed !== false) {
                return $parsed;
            }
            if (ctype_digit($value)) {
                return (int) $value;
            }
        }
        return 0;
    }

    private static function get_override_state(): array {
        $key = defined('SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE')
            ? SHADOWSCAN_OPTION_SUBSCRIPTION_OVERRIDE
            : 'shadowscan_subscription_override';
        $override = get_option($key, array());
        if (!is_array($override)) {
            return array();
        }
        return $override;
    }

    private static function policy_days(string $constant, int $default): int {
        if (defined($constant)) {
            return max(0, (int) constant($constant));
        }
        return $default;
    }
}
