<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Event_Buffer {
    private const OPTION_QUEUE = 'event_queue';
    private const OPTION_BACKOFF = 'event_backoff_until';
    private const OPTION_BACKOFF_COUNT = 'event_backoff_count';
    private const OPTION_FLUSH_STATS = 'event_flush_stats';
    private const MAX_EVENTS = 500;

    public static function can_deliver(string $site_id): bool {
        if ($site_id === '') {
            return false;
        }
        if (function_exists('shadowscan_get_auth_token')) {
            if (shadowscan_get_auth_token() === '') {
                return false;
            }
        } elseif (!function_exists('shadowscan_get_site_token') || shadowscan_get_site_token() === '') {
            return false;
        }
        if (function_exists('shadowscan_sync_is_auth_invalid') && shadowscan_sync_is_auth_invalid()) {
            return false;
        }
        $backoff_until = (int) ShadowScan_Storage::get(self::OPTION_BACKOFF, 0);
        if ($backoff_until > 0 && time() < $backoff_until) {
            return false;
        }
        return true;
    }

    public static function enqueue(array $event): void {
        $queue = ShadowScan_Storage::get_json(self::OPTION_QUEUE, array());
        if (!is_array($queue)) {
            $queue = array();
        }

        $hash = hash('sha256', wp_json_encode($event));
        $event['dedupe_hash'] = $hash;
        foreach ($queue as $queued) {
            if (is_array($queued) && ($queued['dedupe_hash'] ?? '') === $hash) {
                return;
            }
        }

        $queue[] = $event;
        if (count($queue) > self::MAX_EVENTS) {
            $queue = self::trim_queue($queue);
        }

        ShadowScan_Storage::set_json(self::OPTION_QUEUE, $queue);
        if (!empty($event['event_id'])) {
            ShadowScan_Storage::set('last_event_id', (string) $event['event_id']);
        }
        self::log_debug('Event enqueued: id=' . (string) ($event['event_id'] ?? 'unknown') . ' severity=' . (string) ($event['severity'] ?? 'unknown') . ' type=' . (string) ($event['type'] ?? 'unknown'));
    }

    public static function flush(ShadowScan_Api_Endpoints $api, string $site_id): bool {
        if (!self::can_deliver($site_id)) {
            $auth_invalid = function_exists('shadowscan_sync_is_auth_invalid') ? shadowscan_sync_is_auth_invalid() : false;
            $token = function_exists('shadowscan_get_auth_token')
                ? shadowscan_get_auth_token()
                : (function_exists('shadowscan_get_site_token') ? shadowscan_get_site_token() : '');
            if ($site_id === '') {
                self::log_debug('Event queue flush aborted: missing site_id.');
            } elseif ($token === '') {
                self::log_debug('Event queue flush aborted: missing auth token.');
            } elseif ($auth_invalid) {
                self::log_debug('Event queue flush aborted: auth invalid.');
            } else {
                self::log_debug('Event queue flush aborted: backoff or delivery disabled.');
            }
            return false;
        }

        $queue = ShadowScan_Storage::get_json(self::OPTION_QUEUE, array());
        if (!is_array($queue) || empty($queue)) {
            return true;
        }
        $first_event_id = is_array($queue[0] ?? null) ? (string) ($queue[0]['event_id'] ?? '') : '';
        $last_event = end($queue);
        $last_event_id = is_array($last_event) ? (string) ($last_event['event_id'] ?? '') : '';
        self::log_debug('Event queue flush batch: count=' . count($queue) . ' first_event_id=' . ($first_event_id !== '' ? $first_event_id : 'unknown') . ' last_event_id=' . ($last_event_id !== '' ? $last_event_id : 'unknown'));
        if (is_array($last_event)) {
            ShadowScan_Storage::set('last_event_id', $last_event_id);
        }

        $payload = array(
            'site_id' => $site_id,
            'generated_at' => gmdate('c'),
            'events' => $queue,
            'idempotency_key' => hash('sha256', wp_json_encode($queue)),
        );
        if (function_exists('shadowscan_sync_get_domain_id')) {
            $domain_id = shadowscan_sync_get_domain_id();
            if ($domain_id !== '') {
                $payload['domain_id'] = $domain_id;
            }
        }

        $response = $api->events($payload);
        self::record_delivery_attempt($response);
        if (is_wp_error($response)) {
            self::log_debug('Event queue flush failed: ' . $response->get_error_message());
            self::set_backoff();
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        if ($code === 401 || $code === 403) {
            if (function_exists('shadowscan_sync_set_auth_invalid')) {
                shadowscan_sync_set_auth_invalid(true, 'auth_failed', $code);
            }
            self::log_debug('Event queue flush auth failed: HTTP ' . $code);
            return false;
        }
        if ($code === 429 || $code >= 500) {
            self::log_debug('Event queue flush transient error: HTTP ' . $code);
            self::set_backoff();
            return false;
        }
        if ($code < 200 || $code >= 300) {
            self::log_debug('Event queue flush failed: HTTP ' . $code);
            self::set_backoff();
            return false;
        }

        $decoded = null;
        if (is_string($body) && $body !== '') {
            $decoded = json_decode($body, true);
        }

        $accepted_flag = is_array($decoded) && array_key_exists('accepted', $decoded) ? (bool) $decoded['accepted'] : null;
        $ack_id = '';
        if (is_array($decoded)) {
            $ack_id = (string) ($decoded['ack_event_id'] ?? $decoded['event_id'] ?? $decoded['id'] ?? '');
        }
        $has_error = is_array($decoded) && (isset($decoded['error']) || isset($decoded['errors']) || $accepted_flag === false);

        $rejected_hashes = array();
        if (is_array($decoded) && isset($decoded['rejected']) && is_array($decoded['rejected'])) {
            foreach ($decoded['rejected'] as $rejected) {
                if (is_string($rejected)) {
                    $rejected_hashes[] = $rejected;
                } elseif (is_array($rejected)) {
                    if (!empty($rejected['dedupe_hash'])) {
                        $rejected_hashes[] = (string) $rejected['dedupe_hash'];
                    } elseif (!empty($rejected['event']['dedupe_hash'])) {
                        $rejected_hashes[] = (string) $rejected['event']['dedupe_hash'];
                    }
                }
            }
        }

        $accepted = $accepted_flag === true || $ack_id !== '' || (is_array($decoded) && $accepted_flag === null && !$has_error);
        if (!$accepted) {
            $snippet = is_string($body) ? substr($body, 0, 200) : '';
            self::log_debug('Event queue flush 2xx but not accepted: keep ' . count($queue) . ' events. body_snip=' . $snippet);
            return false;
        }

        if (!empty($rejected_hashes)) {
            $rejected_hashes = array_values(array_unique($rejected_hashes));
            $next_queue = array_values(array_filter($queue, static function ($event) use ($rejected_hashes) {
                if (!is_array($event)) {
                    return false;
                }
                $hash = isset($event['dedupe_hash']) ? (string) $event['dedupe_hash'] : '';
                return $hash !== '' && in_array($hash, $rejected_hashes, true);
            }));
            $removed = count($queue) - count($next_queue);
            ShadowScan_Storage::set_json(self::OPTION_QUEUE, $next_queue);
            self::record_successful_flush(count($queue), count($next_queue));
            self::log_debug('Event queue flush partial: retained ' . count($next_queue) . ' rejected, removed ' . $removed . '.');
        } else {
            ShadowScan_Storage::set_json(self::OPTION_QUEUE, array());
            self::record_successful_flush(count($queue), 0);
            self::log_debug('Event queue flush success: cleared ' . count($queue) . ' events.');
        }
        ShadowScan_Storage::set(self::OPTION_BACKOFF, 0);
        ShadowScan_Storage::set(self::OPTION_BACKOFF_COUNT, 0);
        if ($ack_id !== '') {
            ShadowScan_Storage::set('last_portal_ack_id', $ack_id);
        }
        return true;
    }

    private static function record_successful_flush(int $queue_before, int $queue_after): void {
        $stats = ShadowScan_Storage::get_json(self::OPTION_FLUSH_STATS, array());
        if (!is_array($stats)) {
            $stats = array();
        }
        $stats['last_success_at'] = time();
        $stats['last_batch_count'] = max(0, $queue_before);
        $stats['last_queue_after'] = max(0, $queue_after);
        ShadowScan_Storage::set_json(self::OPTION_FLUSH_STATS, $stats);
    }

    private static function record_delivery_attempt($response): void {
        $attempt = array(
            'attempted_at' => time(),
            'http_status' => null,
            'error' => null,
            'response_body' => null,
        );

        if (is_wp_error($response)) {
            $attempt['error'] = $response->get_error_message();
            if (function_exists('shadowscan_record_endpoint_health')) {
                shadowscan_record_endpoint_health('events', false, (string) $attempt['error']);
            }
        } else {
            $attempt['http_status'] = (int) wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            if (is_string($body) && $body !== '') {
                $attempt['response_body'] = substr($body, 0, 2000);
            }
            if ($attempt['http_status'] === 401 || $attempt['http_status'] === 403) {
                if (function_exists('shadowscan_sync_set_auth_invalid')) {
                    shadowscan_sync_set_auth_invalid(true, 'auth_failed', $attempt['http_status']);
                }
            }
            if (function_exists('shadowscan_record_endpoint_health')) {
                $success = $attempt['http_status'] >= 200 && $attempt['http_status'] < 300;
                $error_message = !$success ? ('HTTP ' . (string) $attempt['http_status']) : '';
                shadowscan_record_endpoint_health('events', $success, $error_message);
            }
        }

        ShadowScan_Storage::set_json('event_delivery_last_attempt', $attempt);
    }

    private static function set_backoff(): void {
        $count = (int) ShadowScan_Storage::get(self::OPTION_BACKOFF_COUNT, 0);
        $count = min($count + 1, 6);
        ShadowScan_Storage::set(self::OPTION_BACKOFF_COUNT, $count);
        $delay = min(3600, 60 * (2 ** $count));
        ShadowScan_Storage::set(self::OPTION_BACKOFF, time() + $delay);
    }

    private static function log_debug(string $message): void {
        if (function_exists('shadowscan_log_message')) {
            shadowscan_log_message('[ShadowScan] ' . $message);
        }
    }

    public static function count(): int {
        $queue = ShadowScan_Storage::get_json(self::OPTION_QUEUE, array());
        return is_array($queue) ? count($queue) : 0;
    }

    private static function trim_queue(array $queue): array {
        while (count($queue) > self::MAX_EVENTS) {
            $lowest_rank = null;
            foreach ($queue as $queued) {
                if (!is_array($queued)) {
                    continue;
                }
                $rank = self::severity_rank((string) ($queued['severity'] ?? ''));
                if ($lowest_rank === null || $rank < $lowest_rank) {
                    $lowest_rank = $rank;
                }
            }
            if ($lowest_rank === null) {
                break;
            }
            $removed = false;
            foreach ($queue as $index => $queued) {
                if (!is_array($queued)) {
                    unset($queue[$index]);
                    $removed = true;
                    break;
                }
                $rank = self::severity_rank((string) ($queued['severity'] ?? ''));
                if ($rank === $lowest_rank) {
                    if ($rank === 3) {
                        self::log_debug('SEV1 trimmed due to overload: id=' . (string) ($queued['event_id'] ?? 'unknown'));
                    }
                    unset($queue[$index]);
                    $removed = true;
                    break;
                }
            }
            if (!$removed) {
                break;
            }
            $queue = array_values($queue);
        }
        return $queue;
    }

    private static function severity_rank(string $severity): int {
        $value = strtolower(trim($severity));
        if (in_array($value, array('sev1', 'critical'), true)) {
            return 3;
        }
        if (in_array($value, array('sev2', 'error'), true)) {
            return 2;
        }
        if (in_array($value, array('sev3', 'warning', 'info'), true)) {
            return 1;
        }
        return 1;
    }
}
