<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Policy_State {
    private const OPTION_POLICY_CACHE = 'shadowscan_policy_cache';
    private const OPTION_POLICY_CACHE_AT = 'shadowscan_policy_cache_at';
    private const OPTION_SUPPORT_DEBUG = 'shadowscan_support_debug';
    private const CACHE_TTL = 600;

    public static function get_current_policy(): array {
        $policy = get_option(self::OPTION_POLICY_CACHE, array());
        if (!is_array($policy)) {
            $policy = array();
        }
        $updated_at = (int) get_option(self::OPTION_POLICY_CACHE_AT, 0);
        $stale = !$updated_at || (time() - $updated_at) > self::CACHE_TTL;

        $normalized = self::normalize_policy($policy);
        $normalized['updated_at'] = $updated_at;
        $normalized['stale'] = $stale;

        return $normalized;
    }

    public static function get_support_debug(): array {
        $data = get_option(self::OPTION_SUPPORT_DEBUG, array());
        if (!is_array($data)) {
            $data = array();
        }
        return array(
            'enabled' => !empty($data['enabled']),
            'enabledUntil' => isset($data['enabledUntil']) ? (string) $data['enabledUntil'] : '',
        );
    }

    public static function refresh_policy(bool $force = false): ?array {
        if (!shadowscan_is_connected()) {
            return null;
        }

        $site_id = shadowscan_get_site_id();
        if ($site_id === '') {
            return null;
        }

        $updated_at = (int) get_option(self::OPTION_POLICY_CACHE_AT, 0);
        if (!$force && $updated_at && (time() - $updated_at) < self::CACHE_TTL) {
            return self::get_current_policy();
        }

        $response = shadowscan_send_api_request('site-policy?site_id=' . rawurlencode($site_id), array(), 'GET', true);
        if (is_wp_error($response) || !is_array($response)) {
            $error_code = 'invalid_response';
            $error_message = 'Policy fetch failed.';
            if (is_wp_error($response)) {
                $error_code = sanitize_text_field((string) $response->get_error_code());
                $error_message = shadowscan_sanitize_error_message((string) $response->get_error_message());
            }
            self::record_fetch_result('error', $error_code, $error_message);
            return null;
        }
        self::record_fetch_result('ok');
        if (function_exists('shadowscan_persist_identity_from_response')) {
            shadowscan_persist_identity_from_response($response);
        }

        $signature = isset($response['signature']) ? (string) $response['signature'] : '';
        $commands = isset($response['commands']) && is_array($response['commands']) ? $response['commands'] : array();
        $payload = array(
            'site_id' => isset($response['site_id']) ? (string) $response['site_id'] : $site_id,
            'policy' => isset($response['policy']) && is_array($response['policy']) ? $response['policy'] : array(),
            'updated_at' => isset($response['updated_at']) ? $response['updated_at'] : null,
            'updated_by' => isset($response['updated_by']) ? $response['updated_by'] : null,
            'supportDebug' => isset($response['supportDebug']) && is_array($response['supportDebug']) ? $response['supportDebug'] : array(),
            'policy_version' => $response['policy_version'] ?? null,
        );

        if (!self::verify_signature($payload, $signature)) {
            self::record_fetch_result('error', 'signature_invalid', 'Policy signature verification failed.');
            return null;
        }

        $payload['commands'] = $commands;
        return self::ingest_policy_payload($payload);
    }

    public static function ingest_policy_payload(array $payload): ?array {
        $policy_version = $payload['policy_version'] ?? null;
        $commands = isset($payload['commands']) && is_array($payload['commands']) ? $payload['commands'] : array();
        $require_version = defined('SHADOWSCAN_REQUIRE_POLICY_VERSION') && SHADOWSCAN_REQUIRE_POLICY_VERSION;
        $has_policy_version = !($policy_version === null || $policy_version === '');

        if (!$has_policy_version) {
            if ($require_version) {
                shadowscan_set_last_apply_result(array(
                    'policy_version' => null,
                    'status' => 'error',
                    'error_code' => 'missing_policy_version',
                    'error_message' => 'Policy version missing.',
                    'applied_at' => time(),
                ));
                ShadowScan_Commands::handle($commands);
                return null;
            }
        }

        $state = shadowscan_get_sync_state();
        $last_applied = $state['last_applied_policy_version'] ?? null;
        $legacy_migration = function_exists('shadowscan_policy_version_is_legacy_migration')
            && shadowscan_policy_version_is_legacy_migration($policy_version, $last_applied);
        $cmp = $has_policy_version ? shadowscan_compare_policy_version($policy_version, $last_applied) : null;
        if ($has_policy_version) {
            shadowscan_set_policy_versions($policy_version, null);
        }

        if ($cmp !== null && $cmp <= 0) {
            $skip_reason = $cmp === 0 ? 'version_equal' : 'version_older';
            shadowscan_set_last_apply_result(array(
                'policy_version' => $policy_version,
                'status' => 'skipped',
                'error_code' => 'version_not_newer',
                'error_message' => null,
                'reason' => $skip_reason,
                'applied_at' => time(),
            ));
            ShadowScan_Commands::handle($commands);
            return self::get_current_policy();
        }

        $policy = isset($payload['policy']) && is_array($payload['policy']) ? $payload['policy'] : null;
        if ($policy === null) {
            shadowscan_set_last_apply_result(array(
                'policy_version' => $policy_version,
                'status' => 'error',
                'error_code' => 'invalid_payload',
                'error_message' => 'Policy payload missing.',
                'reason' => 'invalid_payload',
                'applied_at' => time(),
            ));
            ShadowScan_Commands::handle($commands);
            return null;
        }

        $previous = get_option(self::OPTION_POLICY_CACHE, array());
        if (!is_array($previous)) {
            $previous = array();
        }

        try {
            $normalized = self::normalize_policy($policy);
            update_option(self::OPTION_POLICY_CACHE, $normalized, false);
            update_option(self::OPTION_POLICY_CACHE_AT, time(), false);
            if (isset($payload['supportDebug'])) {
                update_option(self::OPTION_SUPPORT_DEBUG, $payload['supportDebug'], false);
            }

            self::apply_policy($normalized, $previous);
            self::apply_security_controls($normalized);

            if ($has_policy_version) {
                shadowscan_set_policy_versions($policy_version, $policy_version);
                if ($legacy_migration && function_exists('shadowscan_mark_policy_version_migrated')) {
                    shadowscan_mark_policy_version_migrated((string) $last_applied, (string) $policy_version);
                }
            }
            shadowscan_set_last_apply_result(array(
                'policy_version' => $has_policy_version ? $policy_version : null,
                'status' => 'ok',
                'error_code' => null,
                'error_message' => null,
                'reason' => $legacy_migration ? 'legacy_migrated' : 'applied',
                'applied_at' => time(),
            ));
        } catch (Throwable $e) {
            shadowscan_set_last_apply_result(array(
                'policy_version' => $has_policy_version ? $policy_version : null,
                'status' => 'error',
                'error_code' => 'exception',
                'error_message' => shadowscan_sanitize_error_message($e->getMessage()),
                'reason' => 'exception',
                'applied_at' => time(),
            ));
        }

        ShadowScan_Commands::handle($commands);
        return self::get_current_policy();
    }

    private static function record_fetch_result(string $result, string $error_code = '', string $error_message = ''): void {
        if (!function_exists('shadowscan_get_sync_state') || !function_exists('shadowscan_set_sync_state')) {
            return;
        }
        $state = shadowscan_get_sync_state();
        $state['last_policy_fetch_at'] = time();
        $state['last_policy_fetch_result'] = $result === 'ok' ? 'ok' : 'error';
        $state['last_policy_fetch_error_code'] = sanitize_text_field($error_code);
        $state['last_policy_fetch_error_message'] = shadowscan_sanitize_error_message($error_message);
        shadowscan_set_sync_state($state);
    }

    private static function apply_policy(array $policy, array $previous): void {
        $toggle_map = self::toggle_option_map();
        if (empty($toggle_map)) {
            return;
        }
        $subscription_state = class_exists('ShadowScan_Subscription_State')
            ? ShadowScan_Subscription_State::get_current_state()
            : array();
        $billing_active = isset($subscription_state['status'])
            && $subscription_state['status'] === ShadowScan_Subscription_State::STATUS_ACTIVE;
        $dev_env = function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env();

        $new_map = $policy['owasp_enforcement'] ?? array();
        if (!is_array($new_map)) {
            $new_map = array();
        }
        $prev_map = $previous['owasp_enforcement'] ?? array();
        if (!is_array($prev_map)) {
            $prev_map = array();
        }

        $all_keys = array_unique(array_merge(array_keys($toggle_map), array_keys($new_map), array_keys($prev_map)));
        $tier = isset($policy['tier']) ? (string) $policy['tier'] : 'monitor';

        foreach ($all_keys as $control_key) {
            if (!isset($toggle_map[$control_key])) {
                continue;
            }
            if (!$billing_active && !$dev_env) {
                continue;
            }
            $enabled = array_key_exists($control_key, $new_map) ? (bool) $new_map[$control_key] : false;
            $previous_enabled = array_key_exists($control_key, $prev_map) ? (bool) $prev_map[$control_key] : false;
            if ($enabled === $previous_enabled) {
                continue;
            }

            $flag = $toggle_map[$control_key];
            if ($flag === 'mfa_enforce_admins' && class_exists('ShadowScan_MFA')) {
                ShadowScan_MFA::set_enforcement_enabled($enabled);
            } elseif ($flag === 'autoupdate_plugins_enabled' && class_exists('ShadowScan_AutoUpdates')) {
                ShadowScan_AutoUpdates::set_enabled($enabled);
            } elseif ($flag === 'password_enforce_min_length' && class_exists('ShadowScan_Password_Policy')) {
                ShadowScan_Password_Policy::set_enforce_min_length($enabled);
            } elseif ($flag === 'password_enforce_breached' && class_exists('ShadowScan_Password_Policy')) {
                ShadowScan_Password_Policy::set_enforce_breached($enabled);
            } elseif ($flag === 'geo_block_mode' && class_exists('ShadowScan_Geo_Policy')) {
                ShadowScan_Geo_Policy::set_mode($enabled ? 'enforce_wp_admin' : 'off');
            } else {
                shadowscan_guard_manager()->set_flag($flag, $enabled);
            }

            $details = array(
                'control_key' => $control_key,
                'status' => 'ok',
                'enforced' => $enabled,
                'previous_enforced' => $previous_enabled,
                'policy_tier' => $tier,
            );
            if (in_array($control_key, array(
                'mfa_enforce_admins',
                'autoupdate_plugins_enforce',
                'password_policy_enforce_min_length',
                'password_policy_enforce_breached',
                'geo_policy_enforce_wp_admin',
            ), true)) {
                $details['category'] = 'security_control';
            } else {
                $category_id = self::category_from_control($control_key);
                $details['category'] = 'owasp';
                $details['owasp_id'] = $category_id;
            }

            ShadowScan_Signal_Manager::emit(
                'owasp_policy_applied',
                'info',
                'OWASP policy applied',
                $details
            );
        }
    }

    private static function apply_security_controls(array $policy): void {
        $controls = $policy['security_controls'] ?? array();
        if (!is_array($controls)) {
            return;
        }
        if (!empty($controls['admin_geo_guard']) && is_array($controls['admin_geo_guard']) && class_exists('ShadowScan_Admin_Geo_Guard')) {
            ShadowScan_Admin_Geo_Guard::set_policy($controls['admin_geo_guard']);
        }

        $subscription_state = class_exists('ShadowScan_Subscription_State')
            ? ShadowScan_Subscription_State::get_current_state()
            : array();
        $billing_active = isset($subscription_state['status'])
            && $subscription_state['status'] === ShadowScan_Subscription_State::STATUS_ACTIVE;
        $dev_env = function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env();
        if (!$billing_active && !$dev_env) {
            return;
        }

        if (!empty($controls['autoupdate']) && is_array($controls['autoupdate']) && class_exists('ShadowScan_AutoUpdates')) {
            $v2_raw = isset($controls['enforcement_v2']) && is_array($controls['enforcement_v2'])
                ? $controls['enforcement_v2']
                : array();
            $v2 = class_exists('ShadowScan_Policy_Resolver')
                ? ShadowScan_Policy_Resolver::normalize_v2($v2_raw)
                : array();
            $site_id = function_exists('shadowscan_get_site_id') ? (string) shadowscan_get_site_id() : '';
            $resolved = class_exists('ShadowScan_Policy_Resolver')
                ? ShadowScan_Policy_Resolver::resolve_core_minor(
                    $v2,
                    $site_id,
                    isset($policy['applied_at']) ? (string) $policy['applied_at'] : null
                )
                : array('should_enforce' => false, 'target' => null, 'reason' => 'resolver_unavailable');
            $plugin_resolved = class_exists('ShadowScan_Policy_Resolver')
                ? ShadowScan_Policy_Resolver::resolve_plugin_allowlist(
                    $v2,
                    $site_id,
                    isset($policy['applied_at']) ? (string) $policy['applied_at'] : null
                )
                : array('should_enforce' => false, 'target' => null, 'reason' => 'resolver_unavailable');

            if (function_exists('shadowscan_get_sync_state') && function_exists('shadowscan_set_sync_state')) {
                $sync_state = shadowscan_get_sync_state();
                $core_minor_control = isset($v2['controls']['core_minor_updates']) && is_array($v2['controls']['core_minor_updates'])
                    ? $v2['controls']['core_minor_updates']
                    : array();
                $plugin_allowlist_control = isset($v2['controls']['plugin_allowlist_updates']) && is_array($v2['controls']['plugin_allowlist_updates'])
                    ? $v2['controls']['plugin_allowlist_updates']
                    : array();
                $effective_gate = !empty($resolved['should_enforce']) ? 'enforced' : (
                    !empty($core_minor_control['global_pause']) ? 'blocked' : 'deferred'
                );
                $plugin_gate = !empty($plugin_resolved['should_enforce']) ? 'enforced' : (
                    !empty($plugin_allowlist_control['global_pause']) ? 'blocked' : 'deferred'
                );
                $sync_state['core_minor_effective'] = array(
                    'mode' => isset($core_minor_control['mode']) ? (string) $core_minor_control['mode'] : 'detect',
                    'enforcement_enabled' => !empty($core_minor_control['enforcement_enabled']),
                    'global_pause' => !empty($core_minor_control['global_pause']),
                    'approval_required' => isset($core_minor_control['approval_required']) ? (string) $core_minor_control['approval_required'] : 'none',
                    'delay_hours' => isset($core_minor_control['delay_hours']) ? (int) $core_minor_control['delay_hours'] : 72,
                    'rollout_percent' => isset($resolved['rollout_percent']) ? (int) $resolved['rollout_percent'] : 0,
                    'site_percent' => isset($resolved['site_percent']) ? (int) $resolved['site_percent'] : 0,
                    'should_enforce' => !empty($resolved['should_enforce']),
                    'target' => isset($resolved['target']) ? $resolved['target'] : null,
                    'gate_status' => $effective_gate,
                    'gate_reason' => isset($resolved['reason']) ? (string) $resolved['reason'] : 'unknown',
                    'evaluated_at' => time(),
                );
                $sync_state['plugin_allowlist_effective'] = array(
                    'mode' => isset($plugin_allowlist_control['mode']) ? (string) $plugin_allowlist_control['mode'] : 'opt_in',
                    'enforcement_enabled' => !empty($plugin_allowlist_control['enforcement_enabled']),
                    'global_pause' => !empty($plugin_allowlist_control['global_pause']),
                    'approval_required' => isset($plugin_allowlist_control['approval_required']) ? (string) $plugin_allowlist_control['approval_required'] : 'none',
                    'delay_hours' => isset($plugin_allowlist_control['delay_hours']) ? (int) $plugin_allowlist_control['delay_hours'] : 72,
                    'rollout_percent' => isset($plugin_resolved['rollout_percent']) ? (int) $plugin_resolved['rollout_percent'] : 0,
                    'site_percent' => isset($plugin_resolved['site_percent']) ? (int) $plugin_resolved['site_percent'] : 0,
                    'max_sites_per_day' => isset($plugin_resolved['max_sites_per_day']) ? (int) $plugin_resolved['max_sites_per_day'] : null,
                    'daily_slot' => isset($plugin_resolved['daily_slot']) ? (int) $plugin_resolved['daily_slot'] : null,
                    'should_enforce' => !empty($plugin_resolved['should_enforce']),
                    'target' => isset($plugin_resolved['target']) ? $plugin_resolved['target'] : null,
                    'gate_status' => $plugin_gate,
                    'gate_reason' => isset($plugin_resolved['reason']) ? (string) $plugin_resolved['reason'] : 'unknown',
                    'plugin_update_holds' => isset($plugin_allowlist_control['plugin_update_holds']) && is_array($plugin_allowlist_control['plugin_update_holds'])
                        ? $plugin_allowlist_control['plugin_update_holds']
                        : array(),
                    'evaluated_at' => time(),
                );
                shadowscan_set_sync_state($sync_state);
            }

            $autoupdate_payload = $controls['autoupdate'];
            $autoupdate_payload['apply_plugins'] = !empty($plugin_resolved['should_enforce']);
            $autoupdate_payload['apply_core'] = !empty($autoupdate_payload['core_enabled']) && !empty($resolved['should_enforce']);
            $autoupdate_payload['core_target'] = $autoupdate_payload['apply_core'] ? 'minor' : null;
            $autoupdate_payload['plugin_update_holds'] = isset($plugin_allowlist_control['plugin_update_holds']) && is_array($plugin_allowlist_control['plugin_update_holds'])
                ? $plugin_allowlist_control['plugin_update_holds']
                : array();
            $autoupdate_payload['reason'] = 'policy_v2_core_minor_rollout';
            ShadowScan_AutoUpdates::sync_policy($autoupdate_payload, 'policy_v2_core_minor_rollout');
        }

        if (!empty($controls['password_policy']) && is_array($controls['password_policy']) && class_exists('ShadowScan_Password_Policy')) {
            $pw = $controls['password_policy'];
            if (isset($pw['min_length'])) {
                ShadowScan_Password_Policy::set_min_length((int) $pw['min_length']);
            }
            if (isset($pw['enforce_min_length'])) {
                ShadowScan_Password_Policy::set_enforce_min_length((bool) $pw['enforce_min_length']);
            }
            if (isset($pw['enforce_breached'])) {
                ShadowScan_Password_Policy::set_enforce_breached((bool) $pw['enforce_breached']);
            }
            if (isset($pw['provider'])) {
                ShadowScan_Password_Policy::set_breach_provider((string) $pw['provider']);
            }
        }
    }

    private static function toggle_option_map(): array {
        if (!function_exists('shadowscan_owasp_get_registry')) {
            return array(
                'mfa_enforce_admins' => 'mfa_enforce_admins',
                'autoupdate_plugins_enforce' => 'autoupdate_plugins_enabled',
                'password_policy_enforce_min_length' => 'password_enforce_min_length',
                'password_policy_enforce_breached' => 'password_enforce_breached',
                'geo_policy_enforce_wp_admin' => 'geo_block_mode',
            );
        }
        $registry = shadowscan_owasp_get_registry();
        $map = array();
        foreach ($registry as $category) {
            $controls = $category['controls'] ?? array();
            foreach ($controls as $control) {
                if (!empty($control['control_key']) && !empty($control['toggle_option_key'])) {
                    $map[$control['control_key']] = $control['toggle_option_key'];
                }
            }
        }
        $map['mfa_enforce_admins'] = 'mfa_enforce_admins';
        $map['autoupdate_plugins_enforce'] = 'autoupdate_plugins_enabled';
        $map['password_policy_enforce_min_length'] = 'password_enforce_min_length';
        $map['password_policy_enforce_breached'] = 'password_enforce_breached';
        $map['geo_policy_enforce_wp_admin'] = 'geo_block_mode';
        return $map;
    }

    private static function normalize_policy(array $policy): array {
        $tier = isset($policy['tier']) ? sanitize_text_field((string) $policy['tier']) : 'monitor';
        $enforcement_enabled = !empty($policy['enforcement_enabled']);
        $applied_at = isset($policy['applied_at']) ? sanitize_text_field((string) $policy['applied_at']) : '';
        $map = $policy['owasp_enforcement'] ?? array();
        $security_controls = $policy['security_controls'] ?? array();
        if (!is_array($map)) {
            $map = array();
        }
        if (!is_array($security_controls)) {
            $security_controls = array();
        }
        if (isset($security_controls['geo'])) {
            unset($security_controls['geo']);
        }
        $clean = array();
        foreach ($map as $key => $value) {
            if (!is_string($key)) {
                continue;
            }
            $clean[$key] = (bool) $value;
        }
        return array(
            'tier' => $tier,
            'enforcement_enabled' => $enforcement_enabled,
            'owasp_enforcement' => $clean,
            'security_controls' => $security_controls,
            'applied_at' => $applied_at,
        );
    }

    private static function verify_signature(array $payload, string $signature): bool {
        if ($signature === '') {
            return false;
        }
        $key = shadowscan_get_site_token();
        if ($key === '') {
            return false;
        }
        $canonical = self::canonicalize(self::normalize_signature_payload($payload));
        $expected = hash_hmac('sha256', $canonical, $key, true);

        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
        $sig_bin = base64_decode($signature, true);
        if ($sig_bin === false) {
            $sig_bin = hex2bin($signature);
        }
        if ($sig_bin === false) {
            return false;
        }

        return hash_equals($expected, $sig_bin);
    }

    private static function canonicalize(array $payload): string {
        $payload = self::recursive_ksort($payload);
        $json = wp_json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        return is_string($json) ? $json : '';
    }

    private static function normalize_signature_payload(array $payload): array {
        if (isset($payload['policy']['owasp_enforcement']) && is_array($payload['policy']['owasp_enforcement'])) {
            if ($payload['policy']['owasp_enforcement'] === array()) {
                $payload['policy']['owasp_enforcement'] = (object) array();
            }
        }
        return $payload;
    }

    private static function recursive_ksort(array $payload): array {
        foreach ($payload as $key => $value) {
            if (is_array($value)) {
                $payload[$key] = self::recursive_ksort($value);
            }
        }
        if (array_keys($payload) !== range(0, count($payload) - 1)) {
            ksort($payload);
        }
        return $payload;
    }

    private static function category_from_control(string $control_key): string {
        if (preg_match('/^a(\d{2})_/i', $control_key, $matches)) {
            return 'A' . $matches[1];
        }
        return 'A01';
    }
}
