<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Policy_Resolver {
    public static function normalize_v2($raw): array {
        if (!is_array($raw)) {
            $raw = array();
        }
        $controls_raw = isset($raw['controls']) && is_array($raw['controls']) ? $raw['controls'] : array();
        $defaults_raw = isset($raw['defaults']) && is_array($raw['defaults']) ? $raw['defaults'] : array();
        $control_raw = isset($controls_raw['core_minor_updates']) && is_array($controls_raw['core_minor_updates'])
            ? $controls_raw['core_minor_updates']
            : array();
        $plugin_control_raw = isset($controls_raw['plugin_allowlist_updates']) && is_array($controls_raw['plugin_allowlist_updates'])
            ? $controls_raw['plugin_allowlist_updates']
            : array();

        return array(
            'version' => 2,
            'profile' => isset($raw['profile']) && in_array($raw['profile'], array('low', 'medium', 'strict'), true)
                ? (string) $raw['profile']
                : 'medium',
            'defaults' => array(
                'approval_required' => 'none',
                'authorization_mode' => (isset($defaults_raw['authorization_mode']) && $defaults_raw['authorization_mode'] === 'analyst')
                    ? 'analyst'
                    : 'portal',
            ),
            'controls' => array(
                'core_minor_updates' => self::normalize_control($control_raw, array(
                    'mode' => 'detect',
                    'rollout_type' => 'all',
                )),
                'plugin_allowlist_updates' => self::normalize_control($plugin_control_raw, array(
                    'mode' => 'opt_in',
                    'rollout_type' => 'throttled_daily',
                    'max_sites_per_day' => 100,
                )),
            ),
        );
    }

    public static function resolve_core_minor(array $v2, string $site_id, ?string $policy_updated_at, ?int $now_ts = null): array {
        $control = isset($v2['controls']['core_minor_updates']) && is_array($v2['controls']['core_minor_updates'])
            ? $v2['controls']['core_minor_updates']
            : array();
        return self::resolve_control($control, $site_id, $policy_updated_at, $now_ts, 'minor');
    }

    public static function resolve_plugin_allowlist(array $v2, string $site_id, ?string $policy_updated_at, ?int $now_ts = null): array {
        $control = isset($v2['controls']['plugin_allowlist_updates']) && is_array($v2['controls']['plugin_allowlist_updates'])
            ? $v2['controls']['plugin_allowlist_updates']
            : array();
        return self::resolve_control($control, $site_id, $policy_updated_at, $now_ts, 'allowlist');
    }

    private static function resolve_control(array $control, string $site_id, ?string $policy_updated_at, ?int $now_ts = null, string $target = 'apply'): array {
        $now = $now_ts ?? time();
        $rollout = isset($control['rollout_strategy']) && is_array($control['rollout_strategy'])
            ? $control['rollout_strategy']
            : array('type' => 'all');
        $updated_ts = $policy_updated_at ? strtotime($policy_updated_at) : false;
        if (!$updated_ts) {
            $updated_ts = $now;
        }
        $elapsed_hours = max(0, ($now - (int) $updated_ts) / 3600);
        $delay_hours = self::clamp_int($control['delay_hours'] ?? 72, 0, 720, 72);

        $effective_percent = 100;
        if (($rollout['type'] ?? 'all') === 'phased_percent') {
            $start = self::clamp_int($rollout['start_percent'] ?? 10, 0, 100, 10);
            $step = self::clamp_int($rollout['step_percent'] ?? 15, 0, 100, 15);
            $interval = self::clamp_int($rollout['interval_hours'] ?? 24, 1, 720, 24);
            $steps = (int) floor($elapsed_hours / $interval);
            $effective_percent = min(100, $start + ($steps * $step));
        }
        $day_key = gmdate('Y-m-d', $now);
        $daily_slot = self::stable_daily_slot($site_id, $day_key);
        $max_sites_per_day = ($rollout['type'] ?? 'all') === 'throttled_daily'
            ? self::clamp_int($rollout['max_sites_per_day'] ?? 100, 1, 10000, 100)
            : null;
        $daily_cap_ready = $max_sites_per_day === null ? true : ($daily_slot < $max_sites_per_day);

        $site_percent = self::stable_percent_for_site($site_id);
        $mode = (string) ($control['mode'] ?? 'detect');
        $global_pause = !empty($control['global_pause']);
        $should_enforce = in_array($mode, array('auto_enforce', 'opt_in'), true)
            && !$global_pause
            && !empty($control['enforcement_enabled'])
            && (($control['approval_required'] ?? 'none') === 'none')
            && $elapsed_hours >= $delay_hours
            && $site_percent < $effective_percent
            && $daily_cap_ready;
        $reason = 'eligible';
        if (!$should_enforce) {
            if (!in_array($mode, array('auto_enforce', 'opt_in'), true)) {
                $reason = 'mode_not_auto';
            } elseif ($global_pause) {
                $reason = 'global_paused';
            } elseif (empty($control['enforcement_enabled'])) {
                $reason = 'enforcement_disabled';
            } elseif (($control['approval_required'] ?? 'none') !== 'none') {
                $reason = 'approval_required';
            } elseif ($elapsed_hours < $delay_hours) {
                $reason = 'delay_not_elapsed';
            } elseif (!$daily_cap_ready) {
                $reason = 'daily_cap_reached';
            } else {
                $reason = 'rollout_not_eligible';
            }
        }

        return array(
            'should_enforce' => $should_enforce,
            'target' => $should_enforce ? $target : null,
            'reason' => $reason,
            'rollout_percent' => (int) $effective_percent,
            'site_percent' => $site_percent,
            'delay_hours' => $delay_hours,
            'global_pause' => $global_pause,
            'max_sites_per_day' => $max_sites_per_day,
            'daily_slot' => $daily_slot,
            'mode' => $mode,
            'approval_required' => (string) ($control['approval_required'] ?? 'none'),
        );
    }

    private static function normalize_control(array $raw, array $defaults): array {
        $mode_default = isset($defaults['mode']) ? (string) $defaults['mode'] : 'detect';
        $mode = isset($raw['mode']) && in_array($raw['mode'], array('detect', 'auto_enforce', 'opt_in'), true)
            ? (string) $raw['mode']
            : $mode_default;
        $requested_enabled = !empty($raw['enforcement_enabled']);
        $enforcement_enabled = $mode === 'detect' ? false : $requested_enabled;
        $approval_required = isset($raw['approval_required']) && in_array($raw['approval_required'], array('none', 'portal_confirm', 'analyst_required'), true)
            ? (string) $raw['approval_required']
            : 'none';
        if ($mode === 'detect') {
            $approval_required = 'none';
        }
        $break_glass_raw = isset($raw['break_glass']) && is_array($raw['break_glass']) ? $raw['break_glass'] : array();
        $rollout_raw = isset($raw['rollout_strategy']) && is_array($raw['rollout_strategy']) ? $raw['rollout_strategy'] : array();
        $rollout_type_default = isset($defaults['rollout_type']) ? (string) $defaults['rollout_type'] : 'all';
        $rollout_type = isset($rollout_raw['type']) && in_array($rollout_raw['type'], array('all', 'phased_percent', 'throttled_daily'), true)
            ? (string) $rollout_raw['type']
            : $rollout_type_default;
        $rollout = array('type' => $rollout_type);
        if ($rollout_type === 'phased_percent') {
            $rollout['start_percent'] = self::clamp_int($rollout_raw['start_percent'] ?? 10, 0, 100, 10);
            $rollout['step_percent'] = self::clamp_int($rollout_raw['step_percent'] ?? 15, 0, 100, 15);
            $rollout['interval_hours'] = self::clamp_int($rollout_raw['interval_hours'] ?? 24, 1, 720, 24);
        } elseif ($rollout_type === 'throttled_daily') {
            $rollout['max_sites_per_day'] = self::clamp_int(
                $rollout_raw['max_sites_per_day'] ?? ($defaults['max_sites_per_day'] ?? 100),
                1,
                10000,
                (int) ($defaults['max_sites_per_day'] ?? 100)
            );
        }

        return array(
            'mode' => $mode,
            'enforcement_enabled' => $enforcement_enabled,
            'global_pause' => !empty($raw['global_pause']),
            'delay_hours' => self::clamp_int($raw['delay_hours'] ?? 72, 0, 720, 72),
            'break_glass' => array(
                'ttl_minutes' => self::clamp_int($break_glass_raw['ttl_minutes'] ?? 15, 5, 60, 15),
                'single_use' => !isset($break_glass_raw['single_use']) || (bool) $break_glass_raw['single_use'],
            ),
            'rollout_strategy' => $rollout,
            'approval_required' => $approval_required,
            'notify' => !isset($raw['notify']) || (bool) $raw['notify'],
            'telemetry_level' => (isset($raw['telemetry_level']) && $raw['telemetry_level'] === 'verbose')
                ? 'verbose'
                : 'standard',
            'plugin_update_holds' => self::normalize_plugin_holds($raw['plugin_update_holds'] ?? array()),
        );
    }

    private static function normalize_plugin_holds($value): array {
        if (!is_array($value)) {
            return array();
        }
        $holds = array();
        foreach ($value as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $slug = isset($entry['slug']) ? strtolower(trim((string) $entry['slug'])) : '';
            $version = isset($entry['version']) ? trim((string) $entry['version']) : '';
            if ($slug === '' || !preg_match('/^[a-z0-9][a-z0-9\-_]*$/', $slug)) {
                continue;
            }
            if ($version === '' || strlen($version) > 64) {
                continue;
            }
            $holds[] = array('slug' => $slug, 'version' => $version);
        }
        return $holds;
    }

    private static function clamp_int($value, int $min, int $max, int $fallback): int {
        if (!is_numeric($value)) {
            return $fallback;
        }
        $int = (int) floor((float) $value);
        if ($int < $min) {
            return $min;
        }
        if ($int > $max) {
            return $max;
        }
        return $int;
    }

    private static function stable_percent_for_site(string $site_id): int {
        $hash = substr(hash('sha256', $site_id), 0, 8);
        $value = hexdec($hash);
        return (int) ($value % 100);
    }

    private static function stable_daily_slot(string $site_id, string $day_key): int {
        $hash = substr(hash('sha256', $day_key . ':' . $site_id), 0, 8);
        $value = hexdec($hash);
        return (int) ($value % 10000);
    }
}
