<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Manager {
    private const OPTION_PROFILE = 'guard_profile';
    private const OPTION_STATE = 'guard_state';
    private const OPTION_FLAGS = 'feature_flags';
    private const OPTION_SAFE_MODE_UNTIL = 'guard_safe_mode_until';

    private array $profiles = array(
        'baseline' => array(
            'auto_disable_xmlrpc_on_abuse' => true,
            'allow_rest_disable' => false,
            'allow_auth_lock' => false,
        ),
        'strict' => array(
            'auto_disable_xmlrpc_on_abuse' => true,
            'allow_rest_disable' => false,
            'allow_auth_lock' => true,
        ),
    );

    private ShadowScan_Guard_Access_Surface $access_guard;
    private ShadowScan_Guard_Auth $auth_guard;
    private ShadowScan_Guard_Bruteforce $bruteforce_guard;
    private ShadowScan_Guard_Mfa $mfa_guard;
    private ShadowScan_Guard_Admin_Integrity $admin_integrity_guard;
    private ShadowScan_Guard_Integrity $integrity_guard;
    private ShadowScan_Guard_Integrity_Targeted $targeted_integrity_guard;
    private ShadowScan_Guard_Config $config_guard;
    private ShadowScan_Guard_Emergency_Mode $emergency_guard;
    private ShadowScan_Guard_Htaccess $htaccess_guard;
    private ShadowScan_Guard_Enumeration $enumeration_guard;
    private ShadowScan_Guard_Headers $headers_guard;
    private ShadowScan_Guard_Access_Control $access_control_guard;

    public function __construct() {
        $this->access_guard = new ShadowScan_Guard_Access_Surface($this);
        $this->auth_guard = new ShadowScan_Guard_Auth($this);
        $this->bruteforce_guard = new ShadowScan_Guard_Bruteforce($this);
        $this->mfa_guard = new ShadowScan_Guard_Mfa($this);
        $this->admin_integrity_guard = new ShadowScan_Guard_Admin_Integrity($this);
        $this->integrity_guard = new ShadowScan_Guard_Integrity($this);
        $this->targeted_integrity_guard = new ShadowScan_Guard_Integrity_Targeted($this);
        $this->config_guard = new ShadowScan_Guard_Config($this);
        $this->emergency_guard = new ShadowScan_Guard_Emergency_Mode($this);
        $this->htaccess_guard = new ShadowScan_Guard_Htaccess($this);
        $this->enumeration_guard = new ShadowScan_Guard_Enumeration($this);
        $this->headers_guard = new ShadowScan_Guard_Headers($this);
        $this->access_control_guard = new ShadowScan_Guard_Access_Control($this);
    }

    public function init(): void {
        $this->access_guard->register();
        $this->auth_guard->register();
        $this->bruteforce_guard->register();
        $this->mfa_guard->register();
        $this->admin_integrity_guard->register();
        $this->integrity_guard->register();
        $this->targeted_integrity_guard->register();
        $this->config_guard->register();
        $this->emergency_guard->register();
        $this->htaccess_guard->register();
        $this->enumeration_guard->register();
        $this->headers_guard->register();
        $this->access_control_guard->register();
    }

    public function get_profile(): string {
        $profile = ShadowScan_Storage::get(self::OPTION_PROFILE, 'baseline');
        return is_string($profile) && isset($this->profiles[$profile]) ? $profile : 'baseline';
    }

    public function apply_profile(string $profile): void {
        $profile = isset($this->profiles[$profile]) ? $profile : 'baseline';
        ShadowScan_Storage::set(self::OPTION_PROFILE, $profile);
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['last_apply'] = time();
        $state['profile'] = $profile;
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
    }

    public function get_profile_config(): array {
        return $this->profiles[$this->get_profile()];
    }

    public function set_flag(string $key, bool $enabled): void {
        $flags = ShadowScan_Storage::get_json(self::OPTION_FLAGS, array());
        $flags[$key] = $enabled;
        ShadowScan_Storage::set_json(self::OPTION_FLAGS, $flags);
    }

    public function is_flag_enabled(string $key, bool $default = true): bool {
        $flags = ShadowScan_Storage::get_json(self::OPTION_FLAGS, array());
        if (!is_array($flags)) {
            return $default;
        }
        return array_key_exists($key, $flags) ? (bool) $flags[$key] : $default;
    }

    public function guard_actions_enabled(): bool {
        $until = (int) ShadowScan_Storage::get(self::OPTION_SAFE_MODE_UNTIL, 0);
        return $until <= 0 || time() >= $until;
    }

    public function set_safe_mode(int $seconds): void {
        ShadowScan_Storage::set(self::OPTION_SAFE_MODE_UNTIL, time() + $seconds);
    }

    public function get_guard_status(): array {
        $mfa_enforced = $this->is_flag_enabled('enforce_mfa', false);
        if (class_exists('ShadowScan_MFA')) {
            $mfa_enforced = ShadowScan_MFA::is_enforcement_enabled();
        }
        return array(
            'profile' => $this->get_profile(),
            'last_integrity_scan' => (int) ShadowScan_Storage::get('integrity_last_scan', 0),
            'last_auth_anomaly' => (int) ShadowScan_Storage::get('auth_last_anomaly', 0),
            'safe_mode_until' => (int) ShadowScan_Storage::get(self::OPTION_SAFE_MODE_UNTIL, 0),
            'enforce_mfa' => $mfa_enforced,
        );
    }

    public function get_bruteforce_stats(bool $include_user_locks = false): array {
        return $this->bruteforce_guard->get_stats($include_user_locks);
    }

    public function get_emergency_status(): array {
        return $this->emergency_guard->get_status();
    }

    public function enable_emergency_mode(string $reason = ''): void {
        $this->emergency_guard->enable($reason);
    }

    public function disable_emergency_mode(): void {
        $this->emergency_guard->disable();
    }

    public function apply_htaccess_baseline(): bool {
        return $this->htaccess_guard->apply_rules();
    }

    public function rollback_htaccess(): bool {
        return $this->htaccess_guard->rollback_rules();
    }

    public function get_htaccess_state(): array {
        return $this->htaccess_guard->get_state();
    }

    public function get_enumeration_state(): array {
        return $this->enumeration_guard->get_state();
    }

    public function get_headers_state(): array {
        return $this->headers_guard->get_state();
    }

    public function get_access_control_state(): array {
        return $this->access_control_guard->get_state();
    }

    public function enable_csp_report_only(): void {
        $this->headers_guard->enable_csp_report_only();
    }

    public function toggle_header_x_frame(bool $enabled): void {
        $this->headers_guard->toggle_x_frame_options($enabled);
    }
}
