<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Integrity {
    private const OPTION_BASELINE = 'integrity_baseline';
    private const OPTION_LAST_SCAN = 'integrity_last_scan';
    private const MAX_SCAN_SECONDS = 5;

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('shadowscan_integrity_scan', array($this, 'run_scan'));
        add_action('shadowscan_integrity_scan_now', function () {
            $this->run_scan(true);
        });
    }

    public function run_scan(bool $force = false): void {
        $last = (int) ShadowScan_Storage::get(self::OPTION_LAST_SCAN, 0);
        if (!$force && $last > 0 && (time() - $last) < 6 * HOUR_IN_SECONDS) {
            return;
        }

        if ($force && $last > 0 && (time() - $last) < HOUR_IN_SECONDS) {
            return;
        }

        $baseline = ShadowScan_Storage::get_json(self::OPTION_BASELINE, array());
        if (empty($baseline)) {
            $baseline = $this->build_baseline();
            ShadowScan_Storage::set_json(self::OPTION_BASELINE, $baseline);
        }

        $start = microtime(true);

        $this->scan_core_files($baseline, $start);
        $this->scan_theme_files($baseline, $start);
        $this->scan_plugin_files($baseline, $start);
        $this->scan_uploads($baseline, $start);

        ShadowScan_Storage::set(self::OPTION_LAST_SCAN, time());
    }

    private function build_baseline(): array {
        $baseline = array(
            'core' => $this->hash_files($this->core_files()),
            'theme' => $this->hash_files($this->theme_files()),
            'plugins' => $this->plugin_hashes(),
            'uploads_php' => $this->uploads_php_files(),
        );

        return $baseline;
    }

    private function scan_core_files(array $baseline, float $start): void {
        $baseline_core = $baseline['core'] ?? array();
        foreach ($this->core_files() as $file) {
            if ($this->scan_time_exceeded($start)) {
                return;
            }
            $hash = $this->hash_file($file);
            $relative = $this->relative_path($file);
            if ($hash && isset($baseline_core[$relative]) && $baseline_core[$relative] !== $hash) {
                ShadowScan_Signal_Integrity::core_file_changed($relative);
            }
        }
    }

    private function scan_theme_files(array $baseline, float $start): void {
        $baseline_theme = $baseline['theme'] ?? array();
        foreach ($this->theme_files() as $file) {
            if ($this->scan_time_exceeded($start)) {
                return;
            }
            $hash = $this->hash_file($file);
            $relative = $this->relative_path($file);
            if ($hash && isset($baseline_theme[$relative]) && $baseline_theme[$relative] !== $hash) {
                ShadowScan_Signal_Integrity::plugin_file_changed($relative);
            }
        }
    }

    private function scan_plugin_files(array $baseline, float $start): void {
        $baseline_plugins = $baseline['plugins'] ?? array();
        foreach ($this->plugin_main_files() as $file) {
            if ($this->scan_time_exceeded($start)) {
                return;
            }
            $hash = $this->hash_file($file);
            $relative = $this->relative_path($file);
            if ($hash && isset($baseline_plugins[$relative]) && $baseline_plugins[$relative] !== $hash) {
                ShadowScan_Signal_Integrity::plugin_file_changed($relative);
            }
        }
    }

    private function scan_uploads(array $baseline, float $start): void {
        $previous = $baseline['uploads_php'] ?? array();
        $current = $this->uploads_php_files($start);
        if (!is_array($current)) {
            return;
        }
        $new_files = array_diff($current, $previous);
        foreach ($new_files as $relative) {
            ShadowScan_Signal_Integrity::new_php_in_uploads($relative);
        }
    }

    private function core_files(): array {
        return array(
            ABSPATH . 'wp-config.php',
            ABSPATH . 'wp-settings.php',
            ABSPATH . 'wp-load.php',
            ABSPATH . 'wp-includes/version.php',
        );
    }

    private function theme_files(): array {
        $theme = wp_get_theme();
        $dir = $theme->get_stylesheet_directory();
        return array_filter(array(
            $dir . '/style.css',
            $dir . '/functions.php',
        ), 'file_exists');
    }

    private function plugin_main_files(): array {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        $paths = array();
        foreach ($plugins as $file => $data) {
            $paths[] = WP_PLUGIN_DIR . '/' . $file;
        }
        return array_filter($paths, 'file_exists');
    }

    private function plugin_hashes(): array {
        $hashes = array();
        foreach ($this->plugin_main_files() as $file) {
            $hash = $this->hash_file($file);
            if ($hash) {
                $hashes[$this->relative_path($file)] = $hash;
            }
        }
        return $hashes;
    }

    private function uploads_php_files(float $start = 0): array {
        $uploads = wp_get_upload_dir();
        if (empty($uploads['basedir'])) {
            return array();
        }
        $paths = array();
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($uploads['basedir'], FilesystemIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if ($start && $this->scan_time_exceeded($start)) {
                break;
            }
            if (!$file->isFile()) {
                continue;
            }
            if (strtolower($file->getExtension()) !== 'php') {
                continue;
            }
            $paths[] = $this->relative_path($file->getPathname());
        }
        return $paths;
    }

    private function hash_files(array $files): array {
        $hashes = array();
        foreach ($files as $file) {
            $hash = $this->hash_file($file);
            if ($hash) {
                $hashes[$this->relative_path($file)] = $hash;
            }
        }
        return $hashes;
    }

    private function hash_file(string $file): ?string {
        if (!file_exists($file)) {
            return null;
        }
        return hash_file('sha256', $file);
    }

    private function relative_path(string $path): string {
        $normalized = wp_normalize_path($path);
        $root = wp_normalize_path(ABSPATH);
        if (strpos($normalized, $root) === 0) {
            return ltrim(substr($normalized, strlen($root)), '/');
        }
        return $normalized;
    }

    private function scan_time_exceeded(float $start): bool {
        return (microtime(true) - $start) >= self::MAX_SCAN_SECONDS;
    }
}
