<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Integrity_Targeted {
    private const OPTION_BASELINE = 'integrity_targeted_baseline';
    private const OPTION_LAST_SCAN = 'integrity_targeted_last_scan';
    private const MAX_SCAN_SECONDS = 4;
    private const MAX_ENTRIES = 5000;

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('shadowscan_integrity_targeted_scan', array($this, 'run_scan'));
        add_action('shadowscan_integrity_targeted_scan_now', function () {
            $this->run_scan(true);
        });
    }

    public function run_scan(bool $force = false): void {
        $last = (int) ShadowScan_Storage::get(self::OPTION_LAST_SCAN, 0);
        if (!$force && $last > 0 && (time() - $last) < 6 * HOUR_IN_SECONDS) {
            return;
        }
        if ($force && $last > 0 && (time() - $last) < HOUR_IN_SECONDS) {
            return;
        }

        $baseline = ShadowScan_Storage::get_json(self::OPTION_BASELINE, array());
        if (empty($baseline)) {
            $baseline = $this->build_baseline();
            ShadowScan_Storage::set_json(self::OPTION_BASELINE, $baseline);
        }

        $start = microtime(true);

        $this->scan_wp_config($baseline);
        $this->scan_mu_plugins($baseline, $start);
        $this->scan_plugins_root($baseline, $start);
        $this->scan_uploads_executables($baseline, $start);

        ShadowScan_Storage::set(self::OPTION_LAST_SCAN, time());
    }

    private function build_baseline(): array {
        $baseline = array(
            'wp_config' => $this->hash_file(ABSPATH . 'wp-config.php'),
            'mu_plugins' => $this->snapshot_directory(WP_CONTENT_DIR . '/mu-plugins'),
            'plugins' => $this->snapshot_plugin_dirs(),
            'uploads_exec' => $this->snapshot_uploads_exec(),
        );
        return $baseline;
    }

    private function scan_wp_config(array $baseline): void {
        $current = $this->hash_file(ABSPATH . 'wp-config.php');
        if (!$current || empty($baseline['wp_config'])) {
            return;
        }
        if ($baseline['wp_config'] !== $current) {
            ShadowScan_Signal_Manager::emit(
                'INTEGRITY_WP_CONFIG_CHANGED',
                'high',
                'wp-config.php checksum changed',
                array('path' => 'wp-config.php')
            );
        }
    }

    private function scan_mu_plugins(array $baseline, float $start): void {
        $current = $this->snapshot_directory(WP_CONTENT_DIR . '/mu-plugins', $start);
        if ($this->scan_time_exceeded($start)) {
            $this->emit_scan_limit();
            return;
        }
        if (!is_array($current) || empty($baseline['mu_plugins'])) {
            return;
        }
        if ($current !== $baseline['mu_plugins']) {
            ShadowScan_Signal_Manager::emit(
                'INTEGRITY_MU_PLUGIN_CHANGED',
                'high',
                'mu-plugins directory changed',
                array()
            );
        }
    }

    private function scan_plugins_root(array $baseline, float $start): void {
        $current = $this->snapshot_plugin_dirs();
        if ($this->scan_time_exceeded($start)) {
            $this->emit_scan_limit();
            return;
        }
        $baseline_plugins = $baseline['plugins'] ?? array();
        if (!is_array($current) || !is_array($baseline_plugins)) {
            return;
        }
        $new_plugins = array_diff($current, $baseline_plugins);
        foreach ($new_plugins as $plugin_dir) {
            ShadowScan_Signal_Manager::emit(
                'INTEGRITY_NEW_PLUGIN_INSTALLED',
                'medium',
                'New plugin directory detected',
                array('path' => 'wp-content/plugins/' . $plugin_dir)
            );
        }
    }

    private function scan_uploads_executables(array $baseline, float $start): void {
        $current = $this->snapshot_uploads_exec($start);
        if ($this->scan_time_exceeded($start)) {
            $this->emit_scan_limit();
            return;
        }
        $baseline_exec = $baseline['uploads_exec'] ?? array();
        if (!is_array($current) || !is_array($baseline_exec)) {
            return;
        }
        $new_exec = array_diff(array_keys($current), array_keys($baseline_exec));
        foreach ($new_exec as $path) {
            $extension = pathinfo($path, PATHINFO_EXTENSION);
            ShadowScan_Signal_Manager::emit(
                'INTEGRITY_NEW_EXECUTABLE_IN_UPLOADS',
                'high',
                'Executable file detected in uploads',
                array(
                    'path' => $path,
                    'extension' => $extension,
                )
            );
        }
    }

    private function snapshot_directory(string $path, float $start = 0): array {
        if (!is_dir($path)) {
            return array();
        }
        $entries = array();
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if ($start && $this->scan_time_exceeded($start)) {
                $this->emit_scan_limit();
                break;
            }
            if (!$file->isFile()) {
                continue;
            }
            $relative = $this->relative_path($file->getPathname());
            $entries[$relative] = array(
                'hash' => $this->hash_file($file->getPathname()),
                'mtime' => $file->getMTime(),
                'size' => $file->getSize(),
            );
            if (count($entries) >= self::MAX_ENTRIES) {
                $this->emit_scan_limit();
                break;
            }
        }
        return $entries;
    }

    private function snapshot_plugin_dirs(): array {
        if (!is_dir(WP_PLUGIN_DIR)) {
            return array();
        }
        $entries = scandir(WP_PLUGIN_DIR) ?: array();
        $dirs = array();
        foreach ($entries as $entry) {
            if ($entry === '.' || $entry === '..') {
                continue;
            }
            if (is_dir(WP_PLUGIN_DIR . '/' . $entry)) {
                $dirs[] = $entry;
            }
        }
        sort($dirs);
        return $dirs;
    }

    private function snapshot_uploads_exec(float $start = 0): array {
        $uploads = wp_get_upload_dir();
        if (empty($uploads['basedir'])) {
            return array();
        }
        $entries = array();
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($uploads['basedir'], FilesystemIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if ($start && $this->scan_time_exceeded($start)) {
                $this->emit_scan_limit();
                break;
            }
            if (!$file->isFile()) {
                continue;
            }
            $ext = strtolower($file->getExtension());
            if (!in_array($ext, array('php', 'phtml', 'phar'), true)) {
                continue;
            }
            $relative = $this->relative_path($file->getPathname());
            $entries[$relative] = array(
                'hash' => $this->hash_file($file->getPathname()),
                'mtime' => $file->getMTime(),
                'size' => $file->getSize(),
            );
            if (count($entries) >= self::MAX_ENTRIES) {
                $this->emit_scan_limit();
                break;
            }
        }
        return $entries;
    }

    private function emit_scan_limit(): void {
        ShadowScan_Signal_Manager::emit(
            'INTEGRITY_SCAN_LIMIT_REACHED',
            'medium',
            'Integrity scan limit reached',
            array()
        );
    }

    private function hash_file(string $file): ?string {
        if (!file_exists($file)) {
            return null;
        }
        return hash_file('sha256', $file);
    }

    private function relative_path(string $path): string {
        $normalized = wp_normalize_path($path);
        $root = wp_normalize_path(ABSPATH);
        if (strpos($normalized, $root) === 0) {
            return ltrim(substr($normalized, strlen($root)), '/');
        }
        return $normalized;
    }

    private function scan_time_exceeded(float $start): bool {
        return (microtime(true) - $start) >= self::MAX_SCAN_SECONDS;
    }
}
