<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Htaccess {
    private const OPTION_STATE = 'htaccess_state';

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        $this->emit_recommended();
    }

    public function apply_rules(): bool {
        $path = ABSPATH . '.htaccess';
        if (!$this->is_supported($path)) {
            ShadowScan_Signal_Manager::emit('HTACCESS_NOT_SUPPORTED', 'low', 'htaccess not supported or writable', array());
            return false;
        }

        if (!function_exists('insert_with_markers')) {
            require_once ABSPATH . 'wp-admin/includes/misc.php';
        }

        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $backup = $this->read_file($path, 262144);
        if ($backup !== null) {
            $state['backup_hash'] = hash('sha256', $backup);
            $state['backup'] = strlen($backup) <= 262144 ? $backup : substr($backup, 0, 262144);
        }

        $rules = $this->build_rules();
        $result = insert_with_markers($path, 'ShadowScan', $rules);
        $state['last_apply'] = time();
        $state['last_result'] = $result ? 'applied' : 'failed';
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);

        if (!$result) {
            ShadowScan_Signal_Manager::emit('HTACCESS_WRITE_FAILED', 'medium', 'Failed to write htaccess rules', array());
            return false;
        }

        $verified = $this->self_check();
        if ($verified === false) {
            $this->rollback($backup);
            ShadowScan_Signal_Manager::emit('HTACCESS_ROLLBACK_PERFORMED', 'high', 'htaccess rollback performed', array());
            return false;
        }

        ShadowScan_Signal_Manager::emit('HTACCESS_RULES_APPLIED', 'medium', 'htaccess rules applied', array());
        return true;
    }

    public function rollback_rules(): bool {
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $backup = isset($state['backup']) ? (string) $state['backup'] : '';
        if ($backup === '') {
            return false;
        }
        $this->rollback($backup);
        ShadowScan_Signal_Manager::emit('HTACCESS_ROLLBACK_PERFORMED', 'high', 'htaccess rollback performed', array());
        return true;
    }

    public function get_state(): array {
        return ShadowScan_Storage::get_json(self::OPTION_STATE, array());
    }

    private function is_supported(string $path): bool {
        $server = isset($_SERVER['SERVER_SOFTWARE']) ? strtolower(sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE']))) : '';
        if ($server && strpos($server, 'nginx') !== false) {
            return false;
        }
        if (!function_exists('shadowscan_get_filesystem')) {
            return false;
        }
        $fs = shadowscan_get_filesystem();
        if (!$fs || !$fs->exists($path)) {
            return false;
        }
        return $fs->is_writable($path);
    }

    private function emit_recommended(): void {
        $path = ABSPATH . '.htaccess';
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $last_notice = isset($state['last_recommended_at']) ? (int) $state['last_recommended_at'] : 0;
        if ($last_notice > 0 && (time() - $last_notice) < DAY_IN_SECONDS) {
            return;
        }
        if ($this->is_supported($path)) {
            ShadowScan_Signal_Manager::emit('HTACCESS_RULES_RECOMMENDED', 'low', 'htaccess hardening available', array());
        } else {
            ShadowScan_Signal_Manager::emit('HTACCESS_NOT_SUPPORTED', 'low', 'htaccess not supported or writable', array());
        }
        $state['last_recommended_at'] = time();
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
    }

    private function build_rules(): array {
        return array(
            'Options -Indexes',
            '<IfModule mod_rewrite.c>',
            'RewriteEngine On',
            'RewriteRule ^\\.env$ - [F]',
            'RewriteRule \\.sql$ - [F]',
            'RewriteRule \\.zip$ - [F]',
            'RewriteRule \\.tar$ - [F]',
            'RewriteRule \\.gz$ - [F]',
            'RewriteRule \\.bak$ - [F]',
            'RewriteRule \\.old$ - [F]',
            'RewriteRule \\.log$ - [F]',
            'RewriteRule ^composer\\.(json|lock)$ - [F]',
            'RewriteRule ^\\.git/ - [F]',
            '</IfModule>',
            '<Files wp-config.php>',
            '<IfModule mod_authz_core.c>',
            'Require all denied',
            '</IfModule>',
            '<IfModule !mod_authz_core.c>',
            'Deny from all',
            '</IfModule>',
            '</Files>',
        );
    }

    private function read_file(string $path, int $cap): ?string {
        if (!function_exists('shadowscan_get_filesystem')) {
            return null;
        }
        $fs = shadowscan_get_filesystem();
        if (!$fs || !$fs->exists($path) || !$fs->is_readable($path)) {
            return null;
        }
        $contents = $fs->get_contents($path);
        if (!is_string($contents)) {
            return null;
        }
        if (strlen($contents) > $cap) {
            return substr($contents, 0, $cap);
        }
        return $contents;
    }

    private function rollback(string $backup): void {
        $path = ABSPATH . '.htaccess';
        if ($backup === '') {
            return;
        }
        if (!function_exists('shadowscan_get_filesystem')) {
            return;
        }
        $fs = shadowscan_get_filesystem();
        if (!$fs) {
            return;
        }
        $fs->put_contents($path, $backup, FS_CHMOD_FILE);
    }

    private function self_check(): ?bool {
        $home = home_url('/');
        $login = home_url('/wp-login.php');
        $home_response = wp_remote_get($home, array('timeout' => 5));
        $login_response = wp_remote_get($login, array('timeout' => 5));

        if (is_wp_error($home_response) || is_wp_error($login_response)) {
            return null;
        }

        $home_code = wp_remote_retrieve_response_code($home_response);
        $login_code = wp_remote_retrieve_response_code($login_response);
        if (in_array($home_code, array(500, 502, 503), true) || in_array($login_code, array(500, 502, 503), true)) {
            return false;
        }
        return true;
    }
}
