<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Headers {
    private const OPTION_STATE = 'headers_state';
    private const OPTION_DEDUPE = 'headers_dedupe';

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('send_headers', array($this, 'apply_headers'));
        remove_action('wp_head', 'wp_generator');
        if (function_exists('header_remove')) {
            @header_remove('X-Powered-By');
        }
    }

    public function apply_headers(): void {
        $headers = $this->get_safe_headers();
        $applied = array();
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['csp_report_only'] = !empty($state['csp_report_only']);
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
        foreach ($headers as $name => $value) {
            if ($this->header_exists($name)) {
                ShadowScan_Signal_Manager::emit(
                    'HEADERS_SERVER_LEAK_DETECTED',
                    'medium',
                    'Header already set upstream',
                    array('header' => $name)
                );
                continue;
            }
            header($name . ': ' . $value);
            $applied[$name] = $value;
        }

        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['enabled_headers'] = array_keys($applied);
        $state['last_apply'] = time();
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);

        $this->emit_deduped('HEADERS_APPLIED', 86400, 'low', 'Security headers applied', array('headers' => array_keys($applied)));
        $this->emit_deduped('HEADERS_GUARD_RECOMMENDED', 86400, 'low', 'Security headers guard active', array());

        if (!empty($state['csp_report_only'])) {
            $policy = "default-src 'self' 'unsafe-inline' 'unsafe-eval' data: https:;";
            header('Content-Security-Policy-Report-Only: ' . $policy);
            $this->emit_deduped('HEADERS_CSP_REPORT_ONLY_ENABLED', 86400, 'medium', 'CSP report-only enabled', array());
        }
    }

    public function enable_csp_report_only(string $endpoint = ''): void {
        $policy = "default-src 'self' 'unsafe-inline' 'unsafe-eval' data: https:;";
        if ($endpoint !== '') {
            $policy .= " report-uri " . $endpoint . ";";
        }
        header('Content-Security-Policy-Report-Only: ' . $policy);
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['csp_report_only'] = true;
        $state['last_apply'] = time();
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
        $this->emit_deduped('HEADERS_CSP_REPORT_ONLY_ENABLED', 86400, 'medium', 'CSP report-only enabled', array());
    }

    public function toggle_x_frame_options(bool $enabled): void {
        $flags = ShadowScan_Storage::get_json('guard_flags', array());
        $flags['header_x_frame'] = $enabled;
        ShadowScan_Storage::set_json('guard_flags', $flags);
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['x_frame_disabled'] = !$enabled;
        $state['last_apply'] = time();
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
    }

    public function get_state(): array {
        return ShadowScan_Storage::get_json(self::OPTION_STATE, array());
    }

    private function get_safe_headers(): array {
        $flags = ShadowScan_Storage::get_json('guard_flags', array());
        $allow_frame = is_array($flags) ? (bool) ($flags['header_x_frame'] ?? true) : true;
        $headers = array(
            'X-Content-Type-Options' => 'nosniff',
            'Referrer-Policy' => 'strict-origin-when-cross-origin',
            'Permissions-Policy' => 'interest-cohort=()',
        );
        if ($allow_frame) {
            $headers['X-Frame-Options'] = 'SAMEORIGIN';
        }
        return $headers;
    }

    private function header_exists(string $name): bool {
        foreach (headers_list() as $header) {
            if (stripos($header, $name . ':') === 0) {
                return true;
            }
        }
        return false;
    }

    private function emit_deduped(string $type, int $ttl, string $severity, string $summary, array $details): void {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $now = time();
        $last = isset($dedupe[$type]) ? (int) $dedupe[$type] : 0;
        if ($last > 0 && ($now - $last) < $ttl) {
            return;
        }
        $dedupe[$type] = $now;
        ShadowScan_Storage::set_json(self::OPTION_DEDUPE, $dedupe);
        ShadowScan_Signal_Manager::emit($type, $severity, $summary, $details);
    }
}
