<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Enumeration {
    private const OPTION_STATE = 'enumeration_state';
    private const OPTION_DEDUPE = 'enumeration_dedupe';

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_filter('login_errors', array($this, 'normalize_login_errors'));
        add_action('template_redirect', array($this, 'block_author_query'));
        add_action('pre_get_posts', array($this, 'maybe_disable_author_archives'));
        add_filter('rest_endpoints', array($this, 'restrict_rest_users'));
        add_action('init', array($this, 'detect_rest_exposure'), 30);
    }

    public function normalize_login_errors($error): string {
        $this->emit_deduped('ENUM_LOGIN_ERROR_NORMALIZED', 86400, 'low', 'Login errors normalized', array());
        return __('The login details you entered are incorrect.', 'shadowscan-security-link');
    }

    public function block_author_query(): void {
        if (is_admin() || is_user_logged_in()) {
            return;
        }
        if (!get_query_var('author')) {
            return;
        }
        if (!$this->flag_enabled('block_author_query', true)) {
            return;
        }
        $this->emit_deduped('ENUM_AUTHOR_QUERY_BLOCKED', 3600, 'medium', 'Author query blocked', array());
        global $wp_query;
        $wp_query->set_404();
        status_header(404);
        nocache_headers();
    }

    public function maybe_disable_author_archives($query): void {
        if (!$query->is_main_query()) {
            return;
        }
        if (!$query->is_author()) {
            return;
        }
        if (!$this->flag_enabled('disable_author_archives', false)) {
            return;
        }
        $this->emit_deduped('ENUM_AUTHOR_ARCHIVE_DISABLED', 3600, 'medium', 'Author archives disabled', array());
        $query->set_404();
        status_header(404);
    }

    public function detect_rest_exposure(): void {
        if (!$this->flag_enabled('detect_rest_exposure', true)) {
            return;
        }
        $last = (int) ShadowScan_Storage::get(self::OPTION_STATE . '_rest_check', 0);
        if ($last > 0 && (time() - $last) < DAY_IN_SECONDS) {
            return;
        }
        ShadowScan_Storage::set(self::OPTION_STATE . '_rest_check', time());

        $url = home_url('/wp-json/wp/v2/users?per_page=1');
        $response = wp_remote_get($url, array('timeout' => 5));
        if (is_wp_error($response)) {
            return;
        }
        $code = wp_remote_retrieve_response_code($response);
        if ($code >= 200 && $code < 300) {
            $body = wp_remote_retrieve_body($response);
            if (is_string($body) && strpos($body, 'id') !== false) {
                ShadowScan_Signal_Manager::emit(
                    'ENUM_REST_EXPOSURE_DETECTED',
                    'high',
                    'REST users endpoint exposed',
                    array('endpoint' => '/wp-json/wp/v2/users')
                );
                $this->emit_deduped('ENUM_GUARD_RECOMMENDED', 86400, 'low', 'Enumeration guard recommended', array());
            }
        }
    }

    public function restrict_rest_users(array $endpoints): array {
        if (!$this->flag_enabled('restrict_rest_users', false)) {
            return $endpoints;
        }
        foreach ($endpoints as $route => $handlers) {
            if (strpos($route, '/wp/v2/users') === 0) {
                foreach ($handlers as &$handler) {
                    if (!isset($handler['permission_callback'])) {
                        continue;
                    }
                    $handler['permission_callback'] = function () {
                        return current_user_can('list_users');
                    };
                }
            }
        }
        $this->emit_deduped('ENUM_REST_USER_ENDPOINT_RESTRICTED', 3600, 'medium', 'REST users endpoint restricted', array());
        return $endpoints;
    }

    public function get_state(): array {
        return array(
            'block_author_query' => $this->flag_enabled('block_author_query', true),
            'disable_author_archives' => $this->flag_enabled('disable_author_archives', false),
            'detect_rest_exposure' => $this->flag_enabled('detect_rest_exposure', true),
            'restrict_rest_users' => $this->flag_enabled('restrict_rest_users', false),
        );
    }

    private function flag_enabled(string $key, bool $default): bool {
        $flags = ShadowScan_Storage::get_json('guard_flags', array());
        if (!is_array($flags)) {
            return $default;
        }
        return array_key_exists($key, $flags) ? (bool) $flags[$key] : $default;
    }

    private function emit_deduped(string $type, int $ttl, string $severity, string $summary, array $details): void {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $now = time();
        $last = isset($dedupe[$type]) ? (int) $dedupe[$type] : 0;
        if ($last > 0 && ($now - $last) < $ttl) {
            return;
        }
        $dedupe[$type] = $now;
        ShadowScan_Storage::set_json(self::OPTION_DEDUPE, $dedupe);
        ShadowScan_Signal_Manager::emit($type, $severity, $summary, $details);
    }
}
