<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Bruteforce {
    private const OPTION_CONFIG = 'bf_config';
    private const OPTION_STATE = 'bf_state';
    private const OPTION_BLOCKLIST = 'bf_blocklist';

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('wp_login_failed', array($this, 'handle_login_failed'));
        add_action('wp_login', array($this, 'handle_login_success'), 10, 2);
        add_filter('authenticate', array($this, 'enforce_locks'), 50, 3);
    }

    public function handle_login_failed(string $username): void {
        $config = $this->get_config();
        $ip = $this->get_ip();
        $window = (int) $config['username_window_minutes'] * 60;
        $ip_window = (int) $config['ip_window_minutes'] * 60;
        $global_window = (int) $config['global_window_minutes'] * 60;

        $username_key = $this->bucket_key('user', $username);
        $ip_key = $this->bucket_key('ip', $ip ?: 'unknown');
        $global_key = 'global';

        $user_count = ShadowScan_Rate_Limiter::increment(self::OPTION_STATE, $username_key, $window);
        $ip_count = ShadowScan_Rate_Limiter::increment(self::OPTION_STATE, $ip_key, $ip_window);
        $global_count = ShadowScan_Rate_Limiter::increment(self::OPTION_STATE, $global_key, $global_window);

        if ($user_count >= (int) $config['username_threshold'] || $ip_count >= (int) $config['ip_threshold']) {
            ShadowScan_Signal_Manager::emit(
                'AUTH_BRUTE_FORCE_SUSPECTED',
                'medium',
                'Repeated failed logins detected',
                array(
                    'username_hash' => $this->hash_value($username),
                    'ip_present' => $ip !== '',
                    'user_failures' => $user_count,
                    'ip_failures' => $ip_count,
                )
            );
        }

        if ($global_count >= (int) $config['global_threshold']) {
            ShadowScan_Signal_Manager::emit(
                'AUTH_BRUTE_FORCE_SPIKE',
                'high',
                'Login failure spike detected',
                array('count' => $global_count)
            );
            do_action('shadowscan_integrity_targeted_scan_now');
        }

        if ($this->is_xmlrpc_request() && $global_count >= (int) $config['xmlrpc_threshold']) {
            ShadowScan_Signal_Manager::emit(
                'ACCESS_XMLRPC_ABUSE_DETECTED',
                'medium',
                'XML-RPC login abuse suspected',
                array('ip_present' => $ip !== '')
            );
        }

        if ($user_count >= (int) $config['username_threshold']) {
            $this->lock_user($username, $config);
        }

        if ($config['ip_block_enabled'] && $ip !== '' && $ip_count >= (int) $config['ip_threshold']) {
            $this->block_ip($ip, $config);
        }

        if ($this->is_xmlrpc_request() && $global_count >= (int) $config['xmlrpc_threshold']) {
            $this->manager->set_flag('xmlrpc_disabled_due_to_abuse', true);
            ShadowScan_Signal_Manager::emit(
                'ACCESS_XMLRPC_DISABLED_DUE_TO_ABUSE',
                'medium',
                'XML-RPC disabled due to login abuse',
                array()
            );
        }
    }

    public function handle_login_success(string $user_login, $user): void {
        $config = $this->get_config();
        ShadowScan_Rate_Limiter::reset(self::OPTION_STATE, $this->bucket_key('user', $user_login));
        if ($config['ip_reduce_on_success']) {
            $ip = $this->get_ip();
            if ($ip !== '') {
                ShadowScan_Rate_Limiter::reset(self::OPTION_STATE, $this->bucket_key('ip', $ip));
            }
        }
    }

    public function enforce_locks($user, string $username, string $password) {
        if (!$this->manager->guard_actions_enabled() || $this->is_safe_mode()) {
            return $user;
        }

        if ($user instanceof WP_User) {
            $locked_until = (int) get_user_meta($user->ID, 'shadowscan_locked_until', true);
            if ($locked_until > time()) {
                return new WP_Error('shadowscan_locked', __('This account is temporarily locked. Please try again later.', 'shadowscan-security-link'));
            }
        }

        $config = $this->get_config();
        if ($config['ip_block_enabled']) {
            $ip = $this->get_ip();
            if ($ip !== '' && $this->is_ip_blocked($ip)) {
                return new WP_Error('shadowscan_ip_blocked', __('Login temporarily restricted. Please try again later.', 'shadowscan-security-link'));
            }
        }

        return $user;
    }

    public function get_stats(bool $include_user_locks = false): array {
        $config = $this->get_config();
        $window = (int) $config['global_window_minutes'] * 60;
        $global_count = ShadowScan_Rate_Limiter::count(self::OPTION_STATE, 'global', $window);
        $blocklist = ShadowScan_Storage::get_json(self::OPTION_BLOCKLIST, array());
        $blocklist = $this->prune_blocklist($blocklist);
        $active_blocks = count($blocklist);
        $active_user_locks = $include_user_locks ? $this->count_active_user_locks() : null;

        return array(
            'failed_last_window' => $global_count,
            'active_ip_blocks' => $active_blocks,
            'active_user_locks' => $active_user_locks,
            'xmlrpc_disabled' => $this->manager->is_flag_enabled('xmlrpc_disabled_due_to_abuse', false),
            'safe_mode' => $this->is_safe_mode(),
        );
    }

    private function lock_user(string $username, array $config): void {
        if (!$this->manager->guard_actions_enabled() || $this->is_safe_mode()) {
            return;
        }

        $user = get_user_by('login', $username);
        if (!$user) {
            return;
        }

        if ($this->is_last_admin($user->ID)) {
            ShadowScan_Signal_Auth::lock_skipped_last_admin($user->ID);
            return;
        }

        $lock_count = (int) get_user_meta($user->ID, 'shadowscan_lock_count', true);
        $lock_count++;
        $duration = $this->lock_duration($lock_count, (int) $config['lock_minutes']);
        $locked_until = time() + $duration;

        update_user_meta($user->ID, 'shadowscan_locked_until', $locked_until);
        update_user_meta($user->ID, 'shadowscan_lock_reason', 'login_abuse');
        update_user_meta($user->ID, 'shadowscan_lock_count', $lock_count);

        ShadowScan_Signal_Manager::emit(
            'AUTH_USER_LOCKED',
            'medium',
            'User temporarily locked after repeated failures',
            array(
                'user_id' => $user->ID,
                'locked_minutes' => (int) floor($duration / 60),
            )
        );
    }

    private function block_ip(string $ip, array $config): void {
        if (!$this->manager->guard_actions_enabled() || $this->is_safe_mode()) {
            return;
        }

        $blocklist = ShadowScan_Storage::get_json(self::OPTION_BLOCKLIST, array());
        if (!is_array($blocklist)) {
            $blocklist = array();
        }

        $expires = time() + ((int) $config['ip_block_minutes'] * 60);
        $blocklist[$ip] = $expires;
        $blocklist = $this->cap_blocklist($blocklist, (int) $config['ip_block_cap']);
        ShadowScan_Storage::set_json(self::OPTION_BLOCKLIST, $blocklist);

        ShadowScan_Signal_Manager::emit(
            'AUTH_IP_TEMP_BLOCKED',
            'medium',
            'Temporary IP block applied after repeated failures',
            array(
                'ip_present' => true,
                'blocked_minutes' => (int) $config['ip_block_minutes'],
            )
        );
    }

    private function is_ip_blocked(string $ip): bool {
        $blocklist = ShadowScan_Storage::get_json(self::OPTION_BLOCKLIST, array());
        if (!is_array($blocklist) || empty($blocklist[$ip])) {
            return false;
        }
        if ((int) $blocklist[$ip] <= time()) {
            unset($blocklist[$ip]);
            ShadowScan_Storage::set_json(self::OPTION_BLOCKLIST, $blocklist);
            return false;
        }
        return true;
    }

    private function cap_blocklist(array $blocklist, int $cap): array {
        $blocklist = $this->prune_blocklist($blocklist);
        if ($cap <= 0 || count($blocklist) <= $cap) {
            return $blocklist;
        }
        uasort($blocklist, function ($a, $b) {
            return $a <=> $b;
        });
        return array_slice($blocklist, -$cap, null, true);
    }

    private function prune_blocklist(array $blocklist): array {
        $now = time();
        foreach ($blocklist as $ip => $expires) {
            if ((int) $expires <= $now) {
                unset($blocklist[$ip]);
            }
        }
        return $blocklist;
    }

    private function lock_duration(int $count, int $base_minutes): int {
        $minutes = $base_minutes;
        if ($count === 2) {
            $minutes = 30;
        } elseif ($count === 3) {
            $minutes = 60;
        } elseif ($count > 3) {
            $minutes = 60 * min(24, (int) round(pow(2, $count - 3)));
        }
        $minutes = min($minutes, 60 * 24);
        return (int) ($minutes * 60);
    }

    private function is_last_admin(int $user_id): bool {
        $admins = get_users(array('role' => 'administrator', 'fields' => array('ID')));
        if (!is_array($admins)) {
            return true;
        }
        if (count($admins) <= 1 && (int) $admins[0]->ID === $user_id) {
            return true;
        }
        return false;
    }

    private function count_active_user_locks(): int {
        $now = time();
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
        $users = get_users(array(
            'fields' => array('ID'),
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Filter is required for lock tracking.
            'meta_key' => 'shadowscan_locked_until',
        ));
        if (!is_array($users)) {
            return 0;
        }
        $count = 0;
        foreach ($users as $user) {
            $locked_until = (int) get_user_meta($user->ID, 'shadowscan_locked_until', true);
            if ($locked_until > $now) {
                $count++;
            }
        }
        return $count;
    }

    private function get_ip(): string {
        if (!empty($_SERVER['REMOTE_ADDR'])) {
            return sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
        }
        return '';
    }

    private function bucket_key(string $prefix, string $value): string {
        return $prefix . ':' . $this->hash_value($value);
    }

    private function hash_value(string $value): string {
        if ($value === '') {
            return 'unknown';
        }
        return substr(hash('sha256', strtolower($value)), 0, 12);
    }

    private function get_config(): array {
        $defaults = array(
            'username_threshold' => 8,
            'username_window_minutes' => 15,
            'lock_minutes' => 15,
            'ip_threshold' => 25,
            'ip_window_minutes' => 10,
            'ip_block_minutes' => 10,
            'ip_block_enabled' => false,
            'ip_block_cap' => 50,
            'global_threshold' => 30,
            'global_window_minutes' => 10,
            'ip_reduce_on_success' => true,
            'xmlrpc_threshold' => 20,
        );

        $stored = ShadowScan_Storage::get_json(self::OPTION_CONFIG, array());
        if (!is_array($stored)) {
            $stored = array();
        }

        return array_merge($defaults, $stored);
    }

    private function is_safe_mode(): bool {
        if (defined('SHADOWSCAN_SAFE_MODE') && SHADOWSCAN_SAFE_MODE) {
            return true;
        }
        return (bool) ShadowScan_Storage::get('bf_safe_mode', false);
    }

    private function is_xmlrpc_request(): bool {
        if (empty($_SERVER['REQUEST_URI'])) {
            return false;
        }
        $uri = sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI']));
        return stripos($uri, 'xmlrpc.php') !== false;
    }
}
