<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Auth {
    private const OPTION_STATS = 'auth_stats';
    private const WINDOW_MINUTES = 20;
    private const THRESHOLD = 10;

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('wp_login_failed', array($this, 'handle_login_failed'));
        add_action('wp_login', array($this, 'handle_login_success'), 10, 2);
        add_action('user_register', array($this, 'handle_user_register'));
        add_action('set_user_role', array($this, 'handle_set_user_role'), 10, 3);
        add_action('add_user_role', array($this, 'handle_add_user_role'), 10, 2);
        add_action('profile_update', array($this, 'handle_profile_update'), 10, 2);
        add_filter('authenticate', array($this, 'maybe_block_locked_user'), 30, 3);
    }

    public function handle_login_failed(string $username): void {
        $stats = $this->get_stats();
        $stats['failed'] = $this->prune_times($stats['failed']);
        $stats['failed'][] = time();
        ShadowScan_Storage::set_json(self::OPTION_STATS, $stats);

        if (count($stats['failed']) >= self::THRESHOLD) {
            ShadowScan_Signal_Auth::brute_force_suspected(count($stats['failed']), self::WINDOW_MINUTES);
            $profile = $this->manager->get_profile_config();
            if (!empty($profile['auto_disable_xmlrpc_on_abuse'])) {
                $this->manager->set_flag('xmlrpc_disabled_due_to_abuse', true);
            }
            if (!empty($profile['allow_auth_lock'])) {
                $this->maybe_lock_user($username);
            }
            do_action('shadowscan_integrity_scan_now');
        }
    }

    public function handle_login_success(string $user_login, $user): void {
        $stats = $this->get_stats();
        $stats['failed'] = $this->prune_times($stats['failed']);
        ShadowScan_Storage::set_json(self::OPTION_STATS, $stats);
    }

    public function handle_user_register(int $user_id): void {
        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }
        if (in_array('administrator', (array) $user->roles, true)) {
            ShadowScan_Signal_Auth::new_admin_created($user_id);
        }
    }

    public function handle_set_user_role(int $user_id, string $role, array $old_roles): void {
        $old = implode(',', $old_roles);
        ShadowScan_Signal_Auth::role_changed($user_id, $old, $role);
        if (ShadowScan_Guard_Access_Control::is_high_risk_role($role)) {
            $actor = get_current_user_id();
            ShadowScan_Guard_Access_Control::emit_access_control_event(
                'access_control_privilege_escalation',
                'privilege_escalation',
                'high',
                'High-risk role assigned',
                array(
                    'actor_user_id' => $actor,
                    'target_user_id' => $user_id,
                    'old_roles' => $old_roles,
                    'new_role' => $role,
                ),
                'critical'
            );
        }
    }

    public function handle_add_user_role(int $user_id, string $role): void {
        if (ShadowScan_Guard_Access_Control::is_high_risk_role($role)) {
            $actor = get_current_user_id();
            ShadowScan_Guard_Access_Control::emit_access_control_event(
                'access_control_privilege_escalation',
                'privilege_escalation',
                'high',
                'High-risk role added to user',
                array(
                    'actor_user_id' => $actor,
                    'target_user_id' => $user_id,
                    'new_role' => $role,
                ),
                'critical'
            );
        }
    }

    public function handle_profile_update(int $user_id, $old_user_data): void {
        $user = get_userdata($user_id);
        if (!$user || !$old_user_data) {
            return;
        }
        $old_roles = isset($old_user_data->roles) ? (array) $old_user_data->roles : array();
        $new_roles = (array) $user->roles;
        if ($old_roles !== $new_roles) {
            ShadowScan_Signal_Auth::role_changed($user_id, implode(',', $old_roles), implode(',', $new_roles));
            $has_high_risk = false;
            foreach ($new_roles as $role) {
                if (ShadowScan_Guard_Access_Control::is_high_risk_role($role)) {
                    $has_high_risk = true;
                    break;
                }
            }
            if ($has_high_risk) {
                $actor = get_current_user_id();
                ShadowScan_Guard_Access_Control::emit_access_control_event(
                    'access_control_privilege_escalation',
                    'privilege_escalation',
                    'high',
                    'High-risk role change detected',
                    array(
                        'actor_user_id' => $actor,
                        'target_user_id' => $user_id,
                        'old_roles' => $old_roles,
                        'new_roles' => $new_roles,
                    ),
                    'critical'
                );
            }
        }
    }

    public function maybe_block_locked_user($user, string $username, string $password) {
        if (!$user || is_wp_error($user)) {
            return $user;
        }

        if ($this->is_safe_mode()) {
            return $user;
        }

        $locked_until = (int) get_user_meta($user->ID, 'shadowscan_locked_until', true);
        if ($locked_until > time()) {
            return new WP_Error('shadowscan_locked', __('This account is temporarily locked. Please try again later.', 'shadowscan-security-link'));
        }

        return $user;
    }

    private function maybe_lock_user(string $username): void {
        if (!$this->manager->guard_actions_enabled()) {
            return;
        }

        if ($this->is_safe_mode()) {
            return;
        }

        $user = get_user_by('login', $username);
        if (!$user) {
            return;
        }

        if ($this->is_last_admin($user->ID)) {
            ShadowScan_Signal_Auth::lock_skipped_last_admin($user->ID);
            return;
        }

        $locked_until = time() + (self::WINDOW_MINUTES * 60);
        update_user_meta($user->ID, 'shadowscan_locked_until', $locked_until);
        ShadowScan_Signal_Auth::user_locked($user->ID, $locked_until);
    }

    private function is_last_admin(int $user_id): bool {
        $admins = get_users(array('role' => 'administrator', 'fields' => array('ID')));
        if (!is_array($admins)) {
            return true;
        }
        if (count($admins) <= 1 && (int) $admins[0]->ID === $user_id) {
            return true;
        }
        return false;
    }

    private function get_stats(): array {
        $stats = ShadowScan_Storage::get_json(self::OPTION_STATS, array('failed' => array()));
        if (!isset($stats['failed']) || !is_array($stats['failed'])) {
            $stats['failed'] = array();
        }
        return $stats;
    }

    private function prune_times(array $times): array {
        $cutoff = time() - (self::WINDOW_MINUTES * 60);
        return array_values(array_filter($times, function ($ts) use ($cutoff) {
            return (int) $ts >= $cutoff;
        }));
    }

    private function is_safe_mode(): bool {
        if (defined('SHADOWSCAN_SAFE_MODE') && SHADOWSCAN_SAFE_MODE) {
            return true;
        }
        return (bool) ShadowScan_Storage::get('bf_safe_mode', false);
    }
}
