<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Geo_Policy {
    private const OPTION_MODE = 'geo_block_mode';
    private const OPTION_COUNTRIES = 'geo_blocked_countries';
    private const OPTION_ALLOW_IPS = 'geo_allow_admin_ips';
    private const OPTION_BLOCK_DEDUPE = 'geo_block_dedupe';

    public static function register(): void {
        add_action('init', array(__CLASS__, 'maybe_enforce_admin'));
    }

    public static function get_status(): array {
        $mode = self::get_mode();
        $countries = self::get_countries();
        $checked_at = time();

        $status = $mode === 'off' ? 'warn' : 'ok';
        $recommended = $mode === 'off'
            ? 'Configure geo restrictions for admin endpoints.'
            : 'Maintain a minimal block list and verify edge enforcement.';

        ShadowScan_Security_Controls::emit_status(
            'admin_geo_guard',
            $status,
            $mode !== 'off',
            $recommended,
            array(
                'mode' => $mode,
                'blocked_countries' => $countries,
                'blocked_count' => count($countries),
            ),
            $checked_at
        );

        return array(
            'mode' => $mode,
            'countries' => $countries,
            'checked_at' => $checked_at,
            'status' => $status,
        );
    }

    public static function set_mode(string $mode): void {
        $allowed = array('off', 'monitor', 'enforce_wp_admin');
        if (!in_array($mode, $allowed, true)) {
            $mode = 'off';
        }
        ShadowScan_Storage::set(self::OPTION_MODE, $mode);
    }

    public static function get_mode(): string {
        $mode = (string) ShadowScan_Storage::get(self::OPTION_MODE, 'off');
        return in_array($mode, array('off', 'monitor', 'enforce_wp_admin'), true) ? $mode : 'off';
    }

    public static function set_countries(array $countries): void {
        $clean = array();
        foreach ($countries as $country) {
            $code = strtoupper(trim((string) $country));
            if ($code !== '' && preg_match('/^[A-Z]{2}$/', $code)) {
                $clean[] = $code;
            }
        }
        $clean = array_values(array_unique($clean));
        ShadowScan_Storage::set_json(self::OPTION_COUNTRIES, $clean);
    }

    public static function get_countries(): array {
        $countries = ShadowScan_Storage::get_json(self::OPTION_COUNTRIES, array());
        return is_array($countries) ? $countries : array();
    }

    public static function set_allow_ips(array $ips): void {
        $clean = array();
        foreach ($ips as $ip) {
            $ip = trim((string) $ip);
            if ($ip !== '' && filter_var($ip, FILTER_VALIDATE_IP)) {
                $clean[] = $ip;
            }
        }
        $clean = array_values(array_unique($clean));
        ShadowScan_Storage::set_json(self::OPTION_ALLOW_IPS, $clean);
    }

    public static function get_allow_ips(): array {
        $ips = ShadowScan_Storage::get_json(self::OPTION_ALLOW_IPS, array());
        return is_array($ips) ? $ips : array();
    }

    public static function maybe_enforce_admin(): void {
        $mode = self::get_mode();
        if ($mode !== 'enforce_wp_admin') {
            return;
        }
        if (!self::is_admin_route()) {
            return;
        }

        $country = self::detect_country();
        if ($country === null) {
            return;
        }

        $blocked = self::get_countries();
        if (empty($blocked) || !in_array($country, $blocked, true)) {
            return;
        }

        if (self::is_allowed_ip()) {
            return;
        }

        self::emit_block_event($country);
        status_header(403);
        wp_die(
            esc_html__('Access to wp-admin is restricted from your region.', 'shadowscan-security-link'),
            esc_html__('Restricted', 'shadowscan-security-link'),
            array('response' => 403)
        );
    }

    private static function is_admin_route(): bool {
        $uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $path = (string) wp_parse_url($uri, PHP_URL_PATH);
        if ($path === '' ) {
            return false;
        }
        if (stripos($path, '/wp-admin') === 0) {
            return true;
        }
        if (stripos($path, '/wp-login.php') !== false) {
            return true;
        }
        return false;
    }

    private static function detect_country(): ?string {
        if (!empty($_SERVER['HTTP_CF_IPCOUNTRY'])) {
            $country = strtoupper(trim(sanitize_text_field(wp_unslash($_SERVER['HTTP_CF_IPCOUNTRY']))));
            if ($country !== 'XX' && preg_match('/^[A-Z]{2}$/', $country)) {
                return $country;
            }
        }
        return null;
    }

    private static function is_allowed_ip(): bool {
        $allow = self::get_allow_ips();
        if (empty($allow)) {
            return false;
        }
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        return $ip !== '' && in_array($ip, $allow, true);
    }

    private static function emit_block_event(string $country): void {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_BLOCK_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $key = $country;
        $last = (int) ($dedupe[$key] ?? 0);
        if ($last > 0 && (time() - $last) < HOUR_IN_SECONDS) {
            return;
        }
        $dedupe[$key] = time();
        ShadowScan_Storage::set_json(self::OPTION_BLOCK_DEDUPE, $dedupe);

        ShadowScan_Signal_Manager::emit(
            'geo_block_triggered',
            'warning',
            'Geo restriction blocked admin access',
            array(
                'category' => 'security_control',
                'control_key' => 'geo_policy_enforce_wp_admin',
                'status' => 'warn',
                'enforced' => true,
                'recommended_action' => 'Review blocked countries and edge enforcement settings.',
                'evidence' => array(
                    'country' => $country,
                ),
                'last_checked_at' => gmdate('c'),
            )
        );
    }
}
