<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_AutoUpdates {
    private const OPTION_ENABLED = 'autoupdate_plugins_enabled';
    private const OPTION_MODE = 'autoupdate_plugins_mode';
    private const OPTION_ALLOWLIST = 'autoupdate_plugins_allowlist';
    private const SITE_OPTION_PLUGINS = 'auto_update_' . 'plugins';
    private const OPTION_ALLOWLIST_HASH = 'autoupdate_plugins_allowlist_hash';
    private const OPTION_CORE_ENABLED = 'autoupdate_core_enabled';
    private const SITE_OPTION_CORE = 'auto_update_' . 'core';

    public static function register(): void {
        // Intentionally left blank. ShadowScan does not modify core update behavior.
    }

    public static function get_status(): array {
        $desired = self::get_desired_settings();
        $applied = self::get_site_autoupdate_plugins();
        $applied_count = count($applied);
        $desired_enabled = (bool) $desired['enabled'];
        $drift = ($desired_enabled && $applied_count === 0)
            || (!$desired_enabled && $applied_count > 0);
        if ($drift && $desired_enabled && $applied_count === 0) {
            $drift_reason = 'desired_enabled_applied_empty';
        } elseif ($drift && !$desired_enabled && $applied_count > 0) {
            $drift_reason = 'desired_disabled_applied_nonempty';
        } else {
            $drift_reason = null;
        }
        $checked_at = time();
        $core = self::get_core_status_data();

        if ($desired_enabled && $applied_count > 0) {
            $status = 'ok';
        } elseif (!$desired_enabled && $applied_count === 0) {
            $status = 'ok';
        } elseif ($desired_enabled && $applied_count === 0) {
            $status = 'warn';
        } else {
            $status = 'warn';
        }

        $recommended = $desired_enabled
            ? 'Review allowlist regularly to limit compatibility risk.'
            : 'Auto-updates are disabled by ShadowScan.';

        ShadowScan_Security_Controls::emit_status(
            'autoupdate_plugins_detect',
            $status,
            $desired_enabled,
            $recommended,
            array(
                'enabled' => $desired_enabled,
                'mode' => $desired['mode'],
                'allowlist' => $desired['allowlist'],
                'plugins_desired_enabled' => $desired_enabled,
                'plugins_desired_mode' => $desired['mode'],
                'plugins_desired_allowlist' => $desired['allowlist'],
                'plugins_desired_allowlist_slugs' => self::allowlist_to_slugs($desired['allowlist']),
                'plugins_desired_allowlist_count' => count($desired['allowlist']),
                'plugins_applied_count' => $applied_count,
                'plugins_drift' => $drift,
                'plugins_drift_reason' => $drift_reason,
                'core_desired_enabled' => $core['desired_enabled'],
                'core_applied_state' => $core['applied_state'],
                'core_drift' => $core['drift'],
                'core_drift_reason' => $core['drift_reason'],
            ),
            $checked_at
        );

        return array(
            'enabled' => $desired_enabled,
            'mode' => $desired['mode'],
            'allowlist' => $desired['allowlist'],
            'desired_enabled' => $desired_enabled,
            'desired_mode' => $desired['mode'],
            'desired_allowlist' => $desired['allowlist'],
            'applied_count' => $applied_count,
            'drift' => $drift,
            'plugins_desired_enabled' => $desired_enabled,
            'plugins_desired_mode' => $desired['mode'],
            'plugins_desired_allowlist' => $desired['allowlist'],
            'plugins_desired_allowlist_slugs' => self::allowlist_to_slugs($desired['allowlist']),
            'plugins_desired_allowlist_count' => count($desired['allowlist']),
            'plugins_applied_count' => $applied_count,
            'plugins_drift' => $drift,
            'plugins_drift_reason' => $drift_reason,
            'core_desired_enabled' => $core['desired_enabled'],
            'core_applied_state' => $core['applied_state'],
            'core_drift' => $core['drift'],
            'core_drift_reason' => $core['drift_reason'],
            'status' => $status,
            'checked_at' => $checked_at,
        );
    }

    public static function get_core_status(): array {
        $core = self::get_core_status_data();

        ShadowScan_Security_Controls::emit_status(
            'wp_core_autoupdate_detect',
            $core['status'],
            $core['desired_enabled'],
            $core['recommended'],
            array(
                'wp_version' => get_bloginfo('version'),
                'desired_enabled' => $core['desired_enabled'],
                'applied_level' => $core['applied_level'],
                'applied_state' => $core['applied_state'],
                'applied_source' => $core['applied_source'],
                'configurable' => $core['configurable'],
                'drift' => $core['drift'],
            ),
            time()
        );

        return $core;
    }

    public static function set_enabled(bool $enabled): void {
        $current = self::get_current_settings();
        self::sync_settings($enabled, $current['mode'], $current['allowlist'], 'policy');
    }

    public static function is_enabled(): bool {
        return !empty(self::get_site_autoupdate_plugins());
    }

    public static function set_mode(string $mode): void {
        $current = self::get_current_settings();
        self::sync_settings($current['enabled'], $mode, $current['allowlist'], 'policy');
    }

    public static function get_mode(): string {
        $current = self::get_current_settings();
        return $current['mode'];
    }

    public static function set_allowlist(array $allowlist): void {
        $current = self::get_current_settings();
        self::sync_settings($current['enabled'], $current['mode'], $allowlist, 'policy');
    }

    public static function get_allowlist(): array {
        $desired = self::get_desired_settings();
        return $desired['allowlist'];
    }

    public static function set_core_enabled(bool $enabled): void {
        self::update_option(self::OPTION_CORE_ENABLED, $enabled);
        self::reconcile_core_with_desired('manual');
    }

    public static function sync_settings(bool $enabled, string $mode, array $allowlist, string $reason = 'manual', array $plugin_update_holds = array()): void {
        $mode = $mode === 'all' ? 'all' : 'allowlist';
        $clean = self::sanitize_allowlist($allowlist);

        self::update_option(self::OPTION_ENABLED, $enabled);
        self::update_option(self::OPTION_MODE, $mode);
        self::update_option(self::OPTION_ALLOWLIST, $clean);

        $previous_plugins = self::get_site_autoupdate_plugins();
        $next_plugins = self::build_plugin_list($enabled, $mode, $clean, $plugin_update_holds);

        self::update_site_option(self::SITE_OPTION_PLUGINS, $next_plugins);
        self::update_option(self::OPTION_ALLOWLIST_HASH, self::hash_list($next_plugins));

        self::log_policy_change($previous_plugins, $next_plugins, $reason, $mode, $enabled);
    }

    public static function sync_policy(array $payload, string $reason = 'policy'): void {
        $enabled = isset($payload['enabled']) ? (bool) $payload['enabled'] : false;
        $mode = isset($payload['mode']) ? (string) $payload['mode'] : 'allowlist';
        $allowlist = isset($payload['allowlist']) && is_array($payload['allowlist']) ? $payload['allowlist'] : array();
        $plugin_update_holds = self::normalize_plugin_holds($payload['plugin_update_holds'] ?? array());
        $apply_plugins = !array_key_exists('apply_plugins', $payload) || (bool) $payload['apply_plugins'];
        $apply_core = !array_key_exists('apply_core', $payload) || (bool) $payload['apply_core'];
        $core_target = isset($payload['core_target']) && is_string($payload['core_target'])
            ? (string) $payload['core_target']
            : '';
        if (isset($payload['reason'])) {
            $reason = (string) $payload['reason'];
        }
        if ($apply_plugins) {
            self::sync_settings($enabled, $mode, $allowlist, $reason, $plugin_update_holds);
        } else {
            $mode = $mode === 'all' ? 'all' : 'allowlist';
            self::update_option(self::OPTION_ENABLED, $enabled);
            self::update_option(self::OPTION_MODE, $mode);
            self::update_option(self::OPTION_ALLOWLIST, self::sanitize_allowlist($allowlist));
        }
        if (array_key_exists('core_enabled', $payload)) {
            self::update_option(self::OPTION_CORE_ENABLED, (bool) $payload['core_enabled']);
            if (!$apply_core) {
                return;
            }
            if ($core_target === 'minor' || $core_target === 'disabled') {
                $before = self::get_site_option(self::SITE_OPTION_CORE, null);
                if ($before !== $core_target) {
                    self::update_site_option(self::SITE_OPTION_CORE, $core_target);
                    self::log_core_policy_change($before, $core_target, $reason, (bool) $payload['core_enabled']);
                }
                return;
            }
            self::reconcile_core_with_desired($reason);
        }
    }

    public static function reconcile_applied_with_desired(string $reason = 'reconcile'): void {
        $desired = self::get_desired_settings();
        $current = self::get_site_autoupdate_plugins();
        $next = self::build_plugin_list($desired['enabled'], $desired['mode'], $desired['allowlist']);
        sort($current);
        $next_sorted = $next;
        sort($next_sorted);
        if ($current !== $next_sorted) {
            self::update_site_option(self::SITE_OPTION_PLUGINS, $next);
            self::update_option(self::OPTION_ALLOWLIST_HASH, self::hash_list($next));
            self::log_policy_change($current, $next, $reason, $desired['mode'], $desired['enabled']);
        }
        self::reconcile_core_with_desired($reason);
    }

    private static function get_current_settings(): array {
        $desired = self::get_desired_settings();
        return array(
            'enabled' => $desired['enabled'],
            'mode' => $desired['mode'],
            'allowlist' => $desired['allowlist'],
        );
    }

    private static function get_desired_settings(): array {
        $enabled = (bool) self::get_option(self::OPTION_ENABLED, false);
        $mode = (string) self::get_option(self::OPTION_MODE, 'allowlist');
        $mode = $mode === 'all' ? 'all' : 'allowlist';
        $allowlist = self::get_option(self::OPTION_ALLOWLIST, array());
        if (!is_array($allowlist)) {
            $allowlist = array();
        }
        $allowlist = self::sanitize_allowlist($allowlist);
        return array(
            'enabled' => $enabled,
            'mode' => $mode,
            'allowlist' => $allowlist,
        );
    }

    private static function get_site_autoupdate_plugins(): array {
        $value = self::get_site_option(self::SITE_OPTION_PLUGINS, array());
        return is_array($value) ? array_values(array_filter($value, 'is_string')) : array();
    }

    private static function update_site_option(string $key, array $value): void {
        if (function_exists('update_site_option')) {
            update_site_option($key, $value);
        } else {
            update_option($key, $value);
        }
    }

    private static function get_site_option(string $key, $default) {
        return function_exists('get_site_option') ? get_site_option($key, $default) : get_option($key, $default);
    }

    private static function update_option(string $key, $value): void {
        update_option($key, $value);
    }

    private static function get_option(string $key, $default) {
        return get_option($key, $default);
    }

    private static function get_all_plugins_list(): array {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = function_exists('get_plugins') ? get_plugins() : array();
        $files = array_keys($plugins);
        return array_values(array_filter($files, 'is_string'));
    }

    private static function get_active_plugins_list(): array {
        $active = get_option('active_plugins', array());
        if (!is_array($active)) {
            $active = array();
        }
        $active = array_values(array_filter($active, 'is_string'));
        $installed = self::get_all_plugins_list();
        return array_values(array_intersect($active, $installed));
    }

    private static function is_all_plugins(array $allowlist): bool {
        $all = self::get_all_plugins_list();
        sort($all);
        $list = $allowlist;
        sort($list);
        return $all === $list && !empty($list);
    }

    private static function sanitize_allowlist(array $allowlist): array {
        $clean = array_values(array_unique(array_filter($allowlist, 'is_string')));
        $clean = array_map('wp_normalize_path', $clean);
        $clean = array_map(static function ($entry) {
            return ltrim(trim($entry), '/');
        }, $clean);

        $installed = self::get_all_plugins_list();
        $installed_lookup = array_fill_keys($installed, true);
        $slug_map = self::build_slug_plugin_map($installed);
        $resolved = array();

        foreach ($clean as $entry) {
            if ($entry === '' || strpos($entry, '..') !== false || strpos($entry, 'wp-content') === 0) {
                continue;
            }

            // Legacy allowlist entries may already be plugin main file paths.
            if (substr($entry, -4) === '.php' && isset($installed_lookup[$entry])) {
                $resolved[] = $entry;
                continue;
            }

            // Slug form (for example "akismet") maps to installed plugin file.
            $slug = strtolower($entry);
            if (isset($slug_map[$slug])) {
                $resolved[] = $slug_map[$slug];
            }
        }

        $resolved = array_values(array_unique($resolved));
        return array_values(array_intersect($resolved, $installed));
    }

    private static function build_slug_plugin_map(array $installed): array {
        $map = array();
        foreach ($installed as $plugin_file) {
            if (!is_string($plugin_file) || $plugin_file === '') {
                continue;
            }
            $plugin_file = wp_normalize_path($plugin_file);
            $dirname = dirname($plugin_file);
            $slug = $dirname !== '.' && $dirname !== ''
                ? strtolower(trim($dirname, '/'))
                : strtolower(pathinfo($plugin_file, PATHINFO_FILENAME));
            if ($slug === '') {
                continue;
            }
            if (!isset($map[$slug])) {
                $map[$slug] = $plugin_file;
                continue;
            }
            // Prefer slug/slug.php when multiple main files exist.
            $preferred = $slug . '/' . $slug . '.php';
            if ($plugin_file === $preferred) {
                $map[$slug] = $plugin_file;
            }
        }
        return $map;
    }

    private static function allowlist_to_slugs(array $allowlist): array {
        $slugs = array();
        foreach ($allowlist as $entry) {
            if (!is_string($entry) || $entry === '') {
                continue;
            }
            $normalized = wp_normalize_path($entry);
            $dirname = dirname($normalized);
            if ($dirname !== '.' && $dirname !== '') {
                $slugs[] = strtolower(trim($dirname, '/'));
            } else {
                $slugs[] = strtolower(pathinfo($normalized, PATHINFO_FILENAME));
            }
        }
        return array_values(array_unique(array_filter($slugs, 'is_string')));
    }

    private static function build_plugin_list(bool $enabled, string $mode, array $allowlist, array $plugin_update_holds = array()): array {
        if (!$enabled) {
            return array();
        }
        $hold_map = self::build_plugin_hold_map($plugin_update_holds);
        if ($mode === 'all') {
            return self::filter_held_plugins(self::get_active_plugins_list(), $hold_map);
        }
        return self::filter_held_plugins($allowlist, $hold_map);
    }

    private static function normalize_plugin_holds($value): array {
        if (!is_array($value)) {
            return array();
        }
        $holds = array();
        foreach ($value as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $slug = isset($entry['slug']) ? strtolower(trim((string) $entry['slug'])) : '';
            $version = isset($entry['version']) ? trim((string) $entry['version']) : '';
            if ($slug === '' || !preg_match('/^[a-z0-9][a-z0-9\-_]*$/', $slug)) {
                continue;
            }
            if ($version === '' || strlen($version) > 64) {
                continue;
            }
            $holds[] = array(
                'slug' => $slug,
                'version' => $version,
            );
        }
        return $holds;
    }

    private static function build_plugin_hold_map(array $holds): array {
        $map = array();
        foreach ($holds as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $slug = isset($entry['slug']) ? strtolower(trim((string) $entry['slug'])) : '';
            $version = isset($entry['version']) ? trim((string) $entry['version']) : '';
            if ($slug === '' || $version === '') {
                continue;
            }
            if (!isset($map[$slug])) {
                $map[$slug] = array();
            }
            $map[$slug][$version] = true;
        }
        return $map;
    }

    private static function filter_held_plugins(array $plugin_files, array $hold_map): array {
        if (empty($hold_map)) {
            return $plugin_files;
        }
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = function_exists('get_plugins') ? get_plugins() : array();
        $allowed = array();
        foreach ($plugin_files as $plugin_file) {
            if (!is_string($plugin_file) || $plugin_file === '') {
                continue;
            }
            $normalized = wp_normalize_path($plugin_file);
            $slug = strtolower(trim(dirname($normalized), '/'));
            if ($slug === '' || $slug === '.') {
                $slug = strtolower(pathinfo($normalized, PATHINFO_FILENAME));
            }
            $version = '';
            if (isset($plugins[$plugin_file]) && is_array($plugins[$plugin_file])) {
                $version = isset($plugins[$plugin_file]['Version']) ? trim((string) $plugins[$plugin_file]['Version']) : '';
            } elseif (isset($plugins[$normalized]) && is_array($plugins[$normalized])) {
                $version = isset($plugins[$normalized]['Version']) ? trim((string) $plugins[$normalized]['Version']) : '';
            }
            if (isset($hold_map[$slug])) {
                if (isset($hold_map[$slug]['*'])) {
                    continue;
                }
                if ($version !== '' && isset($hold_map[$slug][$version])) {
                    continue;
                }
            }
            $allowed[] = $plugin_file;
        }
        return array_values(array_unique($allowed));
    }

    private static function hash_list(array $list): string {
        sort($list);
        return hash('sha256', implode("\n", $list));
    }

    private static function detect_core_applied(): array {
        if (defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED) {
            return array('level' => 'disabled', 'source' => 'constant');
        }
        if (defined('WP_AUTO_UPDATE_CORE')) {
            $value = WP_AUTO_UPDATE_CORE;
            if ($value === true || $value === 'major') {
                return array('level' => 'major', 'source' => 'constant');
            }
            if ($value === 'minor') {
                return array('level' => 'minor', 'source' => 'constant');
            }
            if ($value === false || $value === 'disabled') {
                return array('level' => 'disabled', 'source' => 'constant');
            }
        }
        $option = self::get_site_option(self::SITE_OPTION_CORE, null);
        if ($option !== null) {
            if ($option === false || $option === 'disabled') {
                return array('level' => 'disabled', 'source' => 'site_option');
            }
            if ($option === true || $option === 'major' || $option === 'all') {
                return array('level' => 'major', 'source' => 'site_option');
            }
            if ($option === 'minor') {
                return array('level' => 'minor', 'source' => 'site_option');
            }
            return array('level' => 'enabled', 'source' => 'site_option');
        }
        return array('level' => 'unknown', 'source' => 'unknown');
    }

    private static function is_core_configurable(): bool {
        if (defined('AUTOMATIC_UPDATER_DISABLED') || defined('WP_AUTO_UPDATE_CORE')) {
            return false;
        }
        return self::get_site_option(self::SITE_OPTION_CORE, null) !== null;
    }

    private static function get_core_status_data(): array {
        $desired_enabled = (bool) self::get_option(self::OPTION_CORE_ENABLED, false);
        $applied = self::detect_core_applied();
        $applied_level = $applied['level'];
        $applied_source = $applied['source'];
        $configurable = self::is_core_configurable();

        if ($applied_level === 'unknown' && !$configurable) {
            $applied_state = 'locked';
        } elseif ($applied_level === 'unknown') {
            $applied_state = 'unknown';
        } elseif ($applied_level === 'disabled') {
            $applied_state = 'disabled';
        } else {
            $applied_state = 'enabled';
        }

        $drift = null;
        $drift_reason = null;
        if ($applied_state === 'enabled' || $applied_state === 'disabled') {
            $applied_enabled = $applied_state === 'enabled';
            $drift = ($desired_enabled && !$applied_enabled) || (!$desired_enabled && $applied_enabled);
            if ($drift && $desired_enabled && !$applied_enabled) {
                $drift_reason = 'desired_enabled_applied_disabled';
            } elseif ($drift && !$desired_enabled && $applied_enabled) {
                $drift_reason = 'desired_disabled_applied_enabled';
            }
        }

        if ($desired_enabled && $applied_state === 'enabled') {
            $status = 'ok';
        } elseif (!$desired_enabled && $applied_state === 'disabled') {
            $status = 'ok';
        } elseif ($applied_state === 'unknown' || $applied_state === 'locked') {
            $status = $desired_enabled ? 'warn' : 'info';
        } else {
            $status = 'warn';
        }

        $recommended = $configurable
            ? 'Core updates can occasionally cause compatibility issues. If your host already manages core updates, leave this off.'
            : 'Core update settings are host-managed or locked by configuration.';

        return array(
            'desired_enabled' => $desired_enabled,
            'applied_level' => $applied_level,
            'applied_state' => $applied_state,
            'applied_source' => $applied_source,
            'configurable' => $configurable,
            'drift' => $drift,
            'drift_reason' => $drift_reason,
            'status' => $status,
            'recommended' => $recommended,
        );
    }

    private static function reconcile_core_with_desired(string $reason = 'reconcile'): void {
        $desired_enabled = (bool) self::get_option(self::OPTION_CORE_ENABLED, false);
        if (!self::is_core_configurable()) {
            return;
        }
        $before = self::get_site_option(self::SITE_OPTION_CORE, null);
        $after = $desired_enabled ? 'minor' : 'disabled';
        if ($before !== $after) {
            self::update_site_option(self::SITE_OPTION_CORE, $after);
            self::log_core_policy_change($before, $after, $reason, $desired_enabled);
        }
    }

    private static function log_policy_change(array $before_plugins, array $after_plugins, string $reason, string $mode, bool $desired_enabled): void {
        $added_plugins = array_values(array_diff($after_plugins, $before_plugins));
        $removed_plugins = array_values(array_diff($before_plugins, $after_plugins));

        if (empty($added_plugins) && empty($removed_plugins)) {
            return;
        }

        $sample_added = array_slice($added_plugins, 0, 10);
        $sample_removed = array_slice($removed_plugins, 0, 10);
        $hash = self::hash_list($after_plugins);
        $summary = sprintf(
            '[ShadowScan] Auto-update policy changed (reason=%s, mode=%s, desired_enabled=%s). plugins:+%d -%d applied=%d hash=%s added=%s removed=%s',
            $reason,
            $mode,
            $desired_enabled ? 'true' : 'false',
            count($added_plugins),
            count($removed_plugins),
            count($after_plugins),
            $hash,
            $sample_added ? implode(',', $sample_added) : '-',
            $sample_removed ? implode(',', $sample_removed) : '-'
        );

        if (function_exists('shadowscan_log_message')) {
            shadowscan_log_message($summary);
        }
    }

    private static function log_core_policy_change($before, $after, string $reason, bool $desired_enabled): void {
        $summary = sprintf(
            '[ShadowScan] Core auto-update policy changed (reason=%s, desired_enabled=%s). applied_before=%s applied_after=%s',
            $reason,
            $desired_enabled ? 'true' : 'false',
            is_scalar($before) ? (string) $before : 'unset',
            is_scalar($after) ? (string) $after : 'unset'
        );
        if (function_exists('shadowscan_log_message')) {
            shadowscan_log_message($summary);
        }
    }
}
