<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Admin_Geo_Guard {
    private const OPTION_DEDUPE = 'admin_geo_observe_dedupe';
    private const OPTION_SUMMARY = 'admin_geo_observe_summary';
    private const OPTION_STATS = 'admin_geo_observe_stats';
    private const OPTION_DECISION_DEDUPE = 'admin_geo_decision_dedupe';
    private const OPTION_POLICY = 'admin_geo_guard_policy';
    private const OPTION_BLOCK_WINDOW = 'admin_geo_block_window';
    private const OPTION_ANOMALY_DEDUPE = 'admin_geo_anomaly_dedupe';
    private const WINDOW_DAYS = 30;
    private const BLOCK_THRESHOLD = 10;

    public static function register(): void {
        add_action('login_init', array(__CLASS__, 'handle_bypass_request'), 1);
        add_action('login_init', array(__CLASS__, 'observe_login_page'), 5);
        add_action('wp_login_failed', array(__CLASS__, 'observe_login_failed'));
        add_action('wp_login', array(__CLASS__, 'observe_login_success'), 10, 2);
        add_action('admin_init', array(__CLASS__, 'observe_admin_access'));
        add_action('admin_init', array(__CLASS__, 'handle_bypass_request'), 1);
        add_action('init', array(__CLASS__, 'observe_xmlrpc'), 5);
        add_action('admin_init', array(__CLASS__, 'maybe_clear_bypass'), 2);
    }

    public static function get_status(): array {
        $summary = self::get_summary();
        $confidence = self::get_confidence();
        $has_data = !empty($summary['countries']);
        $status = $has_data ? 'ok' : 'warn';
        $message = $has_data
            ? __('Tracking admin access locations.', 'shadowscan-security-link')
            : __('No location data yet. Location headers were not detected.', 'shadowscan-security-link');

        $policy = self::get_policy();
        $mode = (string) ($policy['mode'] ?? 'observe');
        $global_enabled = !empty($policy['enforcement_enabled_globally']);
        if ($mode === 'enforce' && $global_enabled && ($confidence['level'] ?? '') === 'LOW') {
            self::emit_low_confidence_anomaly($confidence);
        }

        return array(
            'status' => $status,
            'checked_at' => time(),
            'message' => $message,
            'countries' => $summary['countries'],
            'last_seen_at' => $summary['last_seen_at'],
            'last_country' => $summary['last_country'],
            'confidence' => $confidence,
            'policy' => $policy,
            'geo_source' => self::get_geo_source(),
        );
    }

    private static function emit_low_confidence_anomaly(array $confidence): void {
        if (!self::anomaly_dedupe_ok('low_confidence_enforce')) {
            return;
        }
        if (!class_exists('ShadowScan_Signal_Manager')) {
            return;
        }
        ShadowScan_Signal_Manager::emit(
            'admin_geo_anomaly_detected',
            'warning',
            'Enforcement enabled with low confidence location data',
            array(
                'category' => 'security_control',
                'control_id' => 'admin_geo_guard',
                'control_key' => 'admin_geo_guard',
                'status' => 'warn',
                'enforced' => true,
                'event_type' => 'admin_geo_anomaly_detected',
                'reason' => 'low_confidence_enforce',
                'metadata' => array(
                    'level' => $confidence['level'] ?? 'LOW',
                    'known_ratio' => $confidence['known_ratio'] ?? 0,
                    'total' => $confidence['total'] ?? 0,
                ),
                'last_checked_at' => gmdate('c'),
            )
        );
    }

    public static function set_policy(array $policy): void {
        ShadowScan_Storage::set_json(self::OPTION_POLICY, $policy);
    }

    public static function get_policy(): array {
        $policy = ShadowScan_Storage::get_json(self::OPTION_POLICY, array());
        return is_array($policy) ? $policy : array();
    }

    public static function observe_login_page(): void {
        self::maybe_enforce('wp_login');
        self::maybe_observe('login_page', null);
    }

    public static function observe_login_failed(string $username): void {
        self::maybe_observe('login_failed', null);
    }

    public static function observe_login_success(string $user_login, WP_User $user): void {
        self::maybe_enforce('wp_login');
        self::maybe_observe('login_success', $user->ID);
    }

    public static function observe_admin_access(): void {
        if (!is_admin()) {
            return;
        }
        self::maybe_enforce('wp_admin');
        self::maybe_observe('admin_access', get_current_user_id() ?: null);
    }

    public static function observe_xmlrpc(): void {
        $uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($uri === '') {
            return;
        }
        if (stripos($uri, 'xmlrpc.php') === false) {
            return;
        }
        self::maybe_observe('xmlrpc', null);
    }

    public static function get_geo_source(): string {
        $cf_country = isset($_SERVER['HTTP_CF_IPCOUNTRY']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_CF_IPCOUNTRY'])) : '';
        if ($cf_country !== '') {
            return 'CF-IPCountry';
        }
        $shadow_country = isset($_SERVER['HTTP_X_SHADOWSCAN_COUNTRY']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_X_SHADOWSCAN_COUNTRY'])) : '';
        if ($shadow_country !== '') {
            return 'X-ShadowScan-Country';
        }
        return 'none';
    }

    public static function get_current_country(): ?string {
        return self::detect_country();
    }

    public static function simulate_decision(?string $country): array {
        $policy = self::get_policy();
        $mode = (string) ($policy['mode'] ?? 'observe');
        $global_enabled = !empty($policy['enforcement_enabled_globally']);
        $allow = isset($policy['allow_countries']) && is_array($policy['allow_countries'])
            ? $policy['allow_countries']
            : array();

        if ($mode !== 'enforce') {
            return array('decision' => 'allow', 'reason' => 'observe_mode');
        }
        if (!$global_enabled) {
            return array('decision' => 'allow', 'reason' => 'global_disabled');
        }
        if (empty($allow)) {
            return array('decision' => 'allow', 'reason' => 'allowlist_empty');
        }
        if ($country === null || $country === '') {
            return array('decision' => 'allow', 'reason' => 'geo_unknown_fail_open');
        }
        return in_array($country, $allow, true)
            ? array('decision' => 'allow', 'reason' => 'country_allowed')
            : array('decision' => 'block', 'reason' => 'country_not_allowed');
    }

    private static function maybe_enforce(string $surface): void {
        if (!function_exists('shadowscan_is_connected') || !shadowscan_is_connected()) {
            return;
        }

        if (self::bypass_cookie_active()) {
            self::emit_decision('allow', 'bypass_cookie', null);
            return;
        }

        $policy = self::get_policy();
        $mode = (string) ($policy['mode'] ?? 'observe');
        $global_enabled = !empty($policy['enforcement_enabled_globally']);
        if ($mode !== 'enforce' || !$global_enabled) {
            return;
        }

        $apply_to = isset($policy['apply_to']) && is_array($policy['apply_to'])
            ? $policy['apply_to']
            : array();
        if ($surface === 'wp_login' && array_key_exists('wp_login', $apply_to) && $apply_to['wp_login'] === false) {
            return;
        }
        if ($surface === 'wp_admin' && array_key_exists('wp_admin', $apply_to) && $apply_to['wp_admin'] === false) {
            return;
        }

        $allow = isset($policy['allow_countries']) && is_array($policy['allow_countries'])
            ? $policy['allow_countries']
            : array();
        if (empty($allow)) {
            return;
        }

        $country = self::detect_country();
        if ($country === null) {
            self::emit_decision('allow', 'geo_unknown_fail_open', null);
            return;
        }

        if (!in_array($country, $allow, true)) {
            self::emit_decision('block', 'country_not_allowed', $country);
            status_header(403);
            $message = wp_kses_post(
                __(
                    '<p>Admin access restricted by location rules.</p><p>If you have an emergency bypass link, open it now.</p><p>Otherwise, update allowed locations in the portal.</p>',
                    'shadowscan-security-link'
                )
            );
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            wp_die($message, esc_html__('Restricted', 'shadowscan-security-link'), array('response' => 403));
        }

        self::emit_decision('allow', 'country_allowed', $country);
    }

    private static function maybe_observe(string $attempt, ?int $user_id): void {
        if (wp_doing_cron()) {
            return;
        }
        if (!function_exists('shadowscan_is_connected') || !shadowscan_is_connected()) {
            return;
        }

        $country = self::detect_country();
        $country_key = $country ?: 'unknown';

        if (!self::rate_limit_ok($country_key)) {
            return;
        }

        $path = self::current_path();
        $request_id = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : '';
        $ip_hash = self::hash_ip();
        $reason = $country ? 'observed' : 'geo_unknown_fail_open';

        self::update_summary($country);
        self::update_stats($country, $reason);

        if (!class_exists('ShadowScan_Signal_Manager')) {
            return;
        }

        ShadowScan_Signal_Manager::emit(
            'admin_geo_observed',
            'info',
            'Admin access observed',
            array(
                'category' => 'security_control',
                'control_id' => 'admin_geo_guard',
                'control_key' => 'admin_geo_guard',
                'status' => $country ? 'ok' : 'warn',
                'enforced' => false,
                'event_type' => 'admin_geo_observed',
                'country' => $country,
                'source' => 'plugin',
                'path' => $path,
                'is_admin_surface' => true,
                'decision' => 'observe',
                'reason' => $reason,
                'ip_hash' => $ip_hash,
                'user_id' => $user_id ? (string) $user_id : null,
                'request_id' => $request_id,
                'metadata' => array(
                    'attempt' => $attempt,
                ),
                'last_checked_at' => gmdate('c'),
            )
        );
    }

    private static function detect_country(): ?string {
        if (!empty($_SERVER['HTTP_CF_IPCOUNTRY'])) {
            $country = strtoupper(trim(sanitize_text_field(wp_unslash($_SERVER['HTTP_CF_IPCOUNTRY']))));
            if ($country !== 'XX' && preg_match('/^[A-Z]{2}$/', $country)) {
                return $country;
            }
        }
        if (!empty($_SERVER['HTTP_X_SHADOWSCAN_COUNTRY'])) {
            $country = strtoupper(trim(sanitize_text_field(wp_unslash($_SERVER['HTTP_X_SHADOWSCAN_COUNTRY']))));
            if (preg_match('/^[A-Z]{2}$/', $country)) {
                return $country;
            }
        }
        return null;
    }

    private static function current_path(): string {
        $uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($uri === '') {
            return '';
        }
        $path = (string) wp_parse_url($uri, PHP_URL_PATH);
        return $path !== '' ? $path : '';
    }

    private static function hash_ip(): string {
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        if ($ip === '') {
            return '';
        }
        $salt = shadowscan_get_site_id();
        if ($salt === '') {
            $salt = (string) wp_salt('auth');
        }
        return hash('sha256', $ip . '|' . $salt);
    }

    private static function rate_limit_ok(string $country_key): bool {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $last = (int) ($dedupe[$country_key] ?? 0);
        if ($last > 0 && (time() - $last) < MINUTE_IN_SECONDS) {
            return false;
        }
        $dedupe[$country_key] = time();
        ShadowScan_Storage::set_json(self::OPTION_DEDUPE, $dedupe);
        return true;
    }

    private static function decision_rate_limit_ok(string $key): bool {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DECISION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $last = (int) ($dedupe[$key] ?? 0);
        if ($last > 0 && (time() - $last) < MINUTE_IN_SECONDS) {
            return false;
        }
        $dedupe[$key] = time();
        ShadowScan_Storage::set_json(self::OPTION_DECISION_DEDUPE, $dedupe);
        return true;
    }

    private static function emit_decision(string $decision, string $reason, ?string $country): void {
        if (!class_exists('ShadowScan_Signal_Manager')) {
            return;
        }
        $country_key = $country ?: 'unknown';
        if (!self::decision_rate_limit_ok($decision . ':' . $country_key)) {
            return;
        }
        $request_id = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : '';
        $path = self::current_path();

        ShadowScan_Signal_Manager::emit(
            $decision === 'block' ? 'admin_geo_blocked' : 'admin_geo_allowed',
            $decision === 'block' ? 'warning' : 'info',
            $decision === 'block' ? 'Admin access blocked by location rules' : 'Admin access allowed by location rules',
            array(
                'category' => 'security_control',
                'control_id' => 'admin_geo_guard',
                'control_key' => 'admin_geo_guard',
                'status' => $decision === 'block' ? 'warn' : 'ok',
                'enforced' => true,
                'event_type' => $decision === 'block' ? 'admin_geo_blocked' : 'admin_geo_allowed',
                'country' => $country,
                'source' => 'plugin',
                'path' => $path,
                'is_admin_surface' => true,
                'decision' => $decision,
                'reason' => $reason,
                'ip_hash' => self::hash_ip(),
                'request_id' => $request_id,
                'metadata' => array(
                    'decision' => $decision,
                    'reason' => $reason,
                ),
                'last_checked_at' => gmdate('c'),
            )
        );

        if ($decision === 'block') {
            self::record_block_anomaly();
        }
    }

    public static function handle_bypass_request(): void {
        if (!function_exists('shadowscan_is_connected') || !shadowscan_is_connected()) {
            return;
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $token = isset($_GET['ss_bypass']) ? sanitize_text_field(wp_unslash($_GET['ss_bypass'])) : '';
        if ($token === '') {
            return;
        }
        if (self::bypass_cookie_active()) {
            $clean = remove_query_arg('ss_bypass');
            if (is_string($clean) && $clean !== '' && !headers_sent()) {
                wp_safe_redirect($clean);
                exit;
            }
            return;
        }
        $ttl = self::verify_bypass_token($token);
        if ($ttl <= 0) {
            return;
        }
        self::set_bypass_cookie($ttl);
        self::emit_decision('allow', 'bypass_cookie', null);
        $clean = remove_query_arg('ss_bypass');
        if (is_string($clean) && $clean !== '' && !headers_sent()) {
            wp_safe_redirect($clean);
            exit;
        }
    }

    private static function record_block_anomaly(): void {
        $window = ShadowScan_Storage::get_json(self::OPTION_BLOCK_WINDOW, array());
        if (!is_array($window)) {
            $window = array();
        }
        $now = time();
        $cutoff = $now - 600;
        $window = array_values(array_filter($window, function ($ts) use ($cutoff) {
            return is_numeric($ts) && (int) $ts >= $cutoff;
        }));
        $window[] = $now;
        ShadowScan_Storage::set_json(self::OPTION_BLOCK_WINDOW, $window);

        if (count($window) <= self::BLOCK_THRESHOLD) {
            return;
        }

        if (!self::anomaly_dedupe_ok('block_spike')) {
            return;
        }

        if (!class_exists('ShadowScan_Signal_Manager')) {
            return;
        }

        ShadowScan_Signal_Manager::emit(
            'admin_geo_anomaly_detected',
            'warning',
            'Admin access blocks spiked in a short window',
            array(
                'category' => 'security_control',
                'control_id' => 'admin_geo_guard',
                'control_key' => 'admin_geo_guard',
                'status' => 'warn',
                'enforced' => true,
                'event_type' => 'admin_geo_anomaly_detected',
                'reason' => 'block_spike',
                'metadata' => array(
                    'window_seconds' => 600,
                    'count' => count($window),
                ),
                'last_checked_at' => gmdate('c'),
            )
        );
    }

    private static function anomaly_dedupe_ok(string $key): bool {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_ANOMALY_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $last = (int) ($dedupe[$key] ?? 0);
        if ($last > 0 && (time() - $last) < 600) {
            return false;
        }
        $dedupe[$key] = time();
        ShadowScan_Storage::set_json(self::OPTION_ANOMALY_DEDUPE, $dedupe);
        return true;
    }

    public static function bypass_cookie_active(): bool {
        if (empty($_COOKIE['ss_geo_bypass'])) {
            return false;
        }
        $raw = sanitize_text_field(wp_unslash($_COOKIE['ss_geo_bypass']));
        if (!is_numeric($raw)) {
            return $raw === '1';
        }
        $expires = (int) $raw;
        if ($expires <= 0) {
            return false;
        }
        if (time() >= $expires) {
            self::clear_bypass_cookie();
            return false;
        }
        return true;
    }

    private static function verify_bypass_token(string $token): int {
        $site_id = shadowscan_get_site_id();
        if ($site_id === '') {
            return 0;
        }
        $payload = array(
            'token' => $token,
            'site_id' => $site_id,
        );
        $response = shadowscan_send_api_request('admin-geo-bypass-verify', $payload, 'POST', true);
        if (is_wp_error($response)) {
            return 0;
        }
        if (empty($response['ok']) || empty($response['ttl_seconds'])) {
            return 0;
        }
        return max(60, (int) $response['ttl_seconds']);
    }

    private static function set_bypass_cookie(int $ttl): void {
        if (headers_sent()) {
            return;
        }
        $secure = is_ssl();
        $expires = time() + max(60, $ttl);
        $value = (string) $expires;
        setcookie('ss_geo_bypass', $value, array(
            'expires' => $expires,
            'path' => '/wp-admin',
            'secure' => $secure,
            'httponly' => true,
            'samesite' => 'Lax',
        ));
        setcookie('ss_geo_bypass', $value, array(
            'expires' => $expires,
            'path' => '/wp-login.php',
            'secure' => $secure,
            'httponly' => true,
            'samesite' => 'Lax',
        ));
    }

    private static function clear_bypass_cookie(): void {
        if (headers_sent()) {
            return;
        }
        $secure = is_ssl();
        setcookie('ss_geo_bypass', '', array(
            'expires' => time() - 3600,
            'path' => '/wp-admin',
            'secure' => $secure,
            'httponly' => true,
            'samesite' => 'Lax',
        ));
        setcookie('ss_geo_bypass', '', array(
            'expires' => time() - 3600,
            'path' => '/wp-login.php',
            'secure' => $secure,
            'httponly' => true,
            'samesite' => 'Lax',
        ));
    }

    public static function maybe_clear_bypass(): void {
        $clear = isset($_GET['ss_bypass_clear']) ? sanitize_text_field(wp_unslash($_GET['ss_bypass_clear'])) : '';
        if ($clear === '') {
            return;
        }
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'shadowscan_geo_bypass_clear')) {
            return;
        }
        self::clear_bypass_cookie();
    }

    private static function get_summary(): array {
        $summary = ShadowScan_Storage::get_json(self::OPTION_SUMMARY, array());
        if (!is_array($summary)) {
            $summary = array();
        }
        $countries = isset($summary['countries']) && is_array($summary['countries'])
            ? $summary['countries']
            : array();
        $last_seen = isset($summary['last_seen_at']) ? (int) $summary['last_seen_at'] : 0;
        $last_country = isset($summary['last_country']) ? (string) $summary['last_country'] : '';

        return array(
            'countries' => $countries,
            'last_seen_at' => $last_seen,
            'last_country' => $last_country,
        );
    }

    private static function update_summary(?string $country): void {
        $summary = self::get_summary();
        $countries = $summary['countries'];
        $now = time();
        $window = self::WINDOW_DAYS * DAY_IN_SECONDS;

        foreach ($countries as $code => $last_seen) {
            if (!is_numeric($last_seen) || ($now - (int) $last_seen) > $window) {
                unset($countries[$code]);
            }
        }

        if ($country) {
            $countries[$country] = $now;
        }

        ShadowScan_Storage::set_json(self::OPTION_SUMMARY, array(
            'countries' => $countries,
            'last_seen_at' => $now,
            'last_country' => $country,
        ));
    }

    private static function update_stats(?string $country, string $reason): void {
        $stats = ShadowScan_Storage::get_json(self::OPTION_STATS, array());
        if (!is_array($stats)) {
            $stats = array();
        }
        $today = gmdate('Y-m-d');
        if (empty($stats[$today]) || !is_array($stats[$today])) {
            $stats[$today] = array('known' => 0, 'unknown' => 0);
        }

        if ($country && $reason !== 'geo_unknown_fail_open') {
            $stats[$today]['known'] = (int) ($stats[$today]['known'] ?? 0) + 1;
        } else {
            $stats[$today]['unknown'] = (int) ($stats[$today]['unknown'] ?? 0) + 1;
        }

        $cutoff = time() - (self::WINDOW_DAYS * DAY_IN_SECONDS);
        foreach ($stats as $day => $counts) {
            $timestamp = strtotime($day . ' 00:00:00 UTC');
            if ($timestamp !== false && $timestamp < $cutoff) {
                unset($stats[$day]);
            }
        }

        ShadowScan_Storage::set_json(self::OPTION_STATS, $stats);
    }

    private static function get_confidence(): array {
        $stats = ShadowScan_Storage::get_json(self::OPTION_STATS, array());
        if (!is_array($stats)) {
            $stats = array();
        }
        $known = 0;
        $unknown = 0;
        foreach ($stats as $counts) {
            if (!is_array($counts)) {
                continue;
            }
            $known += (int) ($counts['known'] ?? 0);
            $unknown += (int) ($counts['unknown'] ?? 0);
        }
        $total = $known + $unknown;
        $ratio = $total > 0 ? ($known / $total) : 0;
        if ($total < 20) {
            return array(
                'level' => 'LOW',
                'known_ratio' => $ratio,
                'total' => $total,
                'reason' => 'insufficient_data',
            );
        }
        if ($ratio >= 0.9) {
            return array('level' => 'HIGH', 'known_ratio' => $ratio, 'total' => $total);
        }
        if ($ratio >= 0.5) {
            return array('level' => 'MEDIUM', 'known_ratio' => $ratio, 'total' => $total);
        }
        return array(
            'level' => 'LOW',
            'known_ratio' => $ratio,
            'total' => $total,
            'reason' => 'missing_headers',
        );
    }
}
