<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Commands {
    public static function handle(array $commands): void {
        if (empty($commands)) {
            return;
        }
        foreach ($commands as $command) {
            if (!is_array($command)) {
                continue;
            }
            $command_id = isset($command['command_id']) ? (string) $command['command_id'] : '';
            if ($command_id === '') {
                shadowscan_log_message('[ShadowScan] Command missing command_id.');
                continue;
            }
            $state = shadowscan_get_sync_state();
            $acks = $state['command_acks'] ?? array();
            if (is_array($acks) && isset($acks[$command_id])) {
                continue;
            }
            $expires_at = isset($command['expires_at']) ? (int) $command['expires_at'] : 0;
            if ($expires_at > 0 && time() > $expires_at) {
                shadowscan_ack_command($command_id, array(
                    'status' => 'failed',
                    'error_code' => 'expired',
                    'error_message' => 'Command expired.',
                    'handled_at' => time(),
                ));
                continue;
            }

            $type = isset($command['type']) ? (string) $command['type'] : '';
            try {
                if ($type === 'reconcile_now') {
                    if (class_exists('ShadowScan_AutoUpdates')) {
                        ShadowScan_AutoUpdates::reconcile_applied_with_desired('command');
                    }
                    shadowscan_ack_command($command_id, array(
                        'status' => 'success',
                        'error_code' => null,
                        'error_message' => null,
                        'handled_at' => time(),
                    ));
                } elseif ($type === 'refresh_status') {
                    shadowscan_run_security_control_checks();
                    shadowscan_ack_command($command_id, array(
                        'status' => 'success',
                        'error_code' => null,
                        'error_message' => null,
                        'handled_at' => time(),
                    ));
                } elseif ($type === 'rotate_token') {
                    shadowscan_ack_command($command_id, array(
                        'status' => 'failed',
                        'error_code' => 'unsupported',
                        'error_message' => 'Rotate token is not supported via commands.',
                        'handled_at' => time(),
                    ));
                } else {
                    shadowscan_ack_command($command_id, array(
                        'status' => 'failed',
                        'error_code' => 'unknown_type',
                        'error_message' => 'Unknown command type.',
                        'handled_at' => time(),
                    ));
                }
            } catch (Throwable $e) {
                shadowscan_ack_command($command_id, array(
                    'status' => 'failed',
                    'error_code' => 'exception',
                    'error_message' => shadowscan_sanitize_error_message($e->getMessage()),
                    'handled_at' => time(),
                ));
            }
        }
    }
}
