<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Api_Client {
    private string $base_url;
    private string $token;
    private string $api_key;
    private int $timeout;
    private int $connect_timeout;
    private string $user_agent;

    public function __construct(string $base_url, string $token, int $timeout = 15, int $connect_timeout = 5, string $user_agent = '') {
        $this->base_url = rtrim($base_url, '/');
        $this->token = trim($token);
        $this->api_key = (string) (defined('SHADOWSCAN_API_KEY') ? SHADOWSCAN_API_KEY : getenv('SHADOWSCAN_API_KEY'));
        $this->timeout = $timeout;
        $this->connect_timeout = $connect_timeout;
        $this->user_agent = $user_agent;
    }

    public function request(string $path, string $method, array $body = array(), bool $auth_required = true, bool $idempotent = false) {
        $url = $this->base_url . '/' . ltrim($path, '/');
        $method = strtoupper($method);
        $headers = array(
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        );
        if ($this->api_key !== '') {
            $headers['apikey'] = $this->api_key;
        }

        $args = array(
            'method' => $method,
            'timeout' => $this->timeout,
            'connect_timeout' => $this->connect_timeout,
            'headers' => $headers,
        );
        if ($this->user_agent !== '') {
            $args['user-agent'] = $this->user_agent;
        }

        $body_string = '';
        if ($method !== 'GET' && $method !== 'HEAD') {
            $body_string = wp_json_encode($body);
            $args['body'] = $body_string;
        }

        if ($auth_required && $this->token !== '') {
            $headers['X-SS-Auth'] = 'site';
            $timestamp = (string) time();
            try {
                $nonce = rtrim(strtr(base64_encode(random_bytes(16)), '+/', '-_'), '=');
            } catch (Exception $e) {
                $nonce = wp_generate_uuid4();
            }
            $signed_path = self::canonical_path_with_query($url);
            $body_hash = hash('sha256', $body_string);
            $canonical = $method . "\n" . $signed_path . "\n" . $timestamp . "\n" . $nonce . "\n" . $body_hash;
            $signature = self::base64url_encode(hash_hmac('sha256', $canonical, $this->token, true));

            $auth_mode = function_exists('shadowscan_get_auth_header_mode')
                ? shadowscan_get_auth_header_mode()
                : 'dual';
            $legacy_auth_value = sprintf(
                'Bearer %s, signature="%s", timestamp="%s", nonce="%s", body-sha256="%s", signed-path="%s"',
                $this->token,
                $signature,
                $timestamp,
                $nonce,
                $body_hash,
                $signed_path
            );
            if ($auth_mode === 'legacy_only') {
                $headers['Authorization'] = $legacy_auth_value;
                $headers['X-ShadowScan-Timestamp'] = $timestamp;
                $headers['X-ShadowScan-Nonce'] = $nonce;
                $headers['X-ShadowScan-Signature'] = $signature;
                $headers['X-ShadowScan-Body-Sha256'] = $body_hash;
                $headers['X-ShadowScan-Signed-Path'] = $signed_path;
            } else {
                if ($auth_mode === 'dual') {
                    $headers['Authorization'] = $legacy_auth_value;
                } else {
                    $headers['Authorization'] = 'Bearer ' . $this->token;
                }
                $headers['X-SS-Signature'] = $signature;
                $headers['X-SS-Timestamp'] = $timestamp;
                $headers['X-SS-Nonce'] = $nonce;
                $headers['X-SS-Body-Sha256'] = $body_hash;
                $headers['X-SS-Signed-Path'] = $signed_path;
                $headers['X-SS-Auth-Version'] = '2';
                if ($auth_mode === 'dual') {
                    $headers['X-ShadowScan-Timestamp'] = $timestamp;
                    $headers['X-ShadowScan-Nonce'] = $nonce;
                    $headers['X-ShadowScan-Signature'] = $signature;
                    $headers['X-ShadowScan-Body-Sha256'] = $body_hash;
                    $headers['X-ShadowScan-Signed-Path'] = $signed_path;
                }
            }
            $args['headers'] = $headers;
            $this->maybe_log_signing($method, $signed_path, $timestamp, $nonce, $body_hash);
        }

        $should_retry = $method === 'GET' || $idempotent || (!empty($body['idempotency_key']));
        $attempts = 0;
        $max_attempts = $should_retry ? 3 : 1;
        $last_error = null;
        while ($attempts < $max_attempts) {
            $response = wp_remote_request($url, $args);
            if (!is_wp_error($response)) {
                return $response;
            }
            $last_error = $response;
            $attempts++;
            if ($attempts < $max_attempts) {
                usleep(200000 * $attempts);
            }
        }

        return $last_error;
    }

    private static function canonical_path_with_query(string $url): string {
        $path = (string) wp_parse_url($url, PHP_URL_PATH);
        $query = (string) wp_parse_url($url, PHP_URL_QUERY);
        if ($query === '') {
            return $path;
        }
        $pairs = array();
        foreach (explode('&', $query) as $part) {
            if ($part === '') {
                continue;
            }
            $kv = explode('=', $part, 2);
            $key = urldecode($kv[0]);
            $value = urldecode($kv[1] ?? '');
            $pairs[] = array($key, $value);
        }
        usort($pairs, static function ($a, $b) {
            if ($a[0] === $b[0]) {
                return strcmp($a[1], $b[1]);
            }
            return strcmp($a[0], $b[0]);
        });
        $encoded = array();
        foreach ($pairs as $pair) {
            $encoded[] = rawurlencode($pair[0]) . '=' . rawurlencode($pair[1]);
        }
        return $path . '?' . implode('&', $encoded);
    }

    private static function base64url_encode(string $binary): string {
        return rtrim(strtr(base64_encode($binary), '+/', '-_'), '=');
    }

    private function maybe_log_signing(string $method, string $signed_path, string $timestamp, string $nonce, string $body_hash): void {
        if (!defined('SHADOWSCAN_DEBUG_SIGNING') && !defined('WP_DEBUG')) {
            return;
        }
        $debug_enabled = (defined('SHADOWSCAN_DEBUG_SIGNING') && SHADOWSCAN_DEBUG_SIGNING)
            || (defined('WP_DEBUG') && WP_DEBUG);
        if (!$debug_enabled) {
            return;
        }
        if (function_exists('shadowscan_log_message')) {
            shadowscan_log_message('[ShadowScan] signing debug: ' . wp_json_encode(array(
                'method' => $method,
                'signed_path' => $signed_path,
                'timestamp' => $timestamp,
                'nonce' => $nonce,
                'body_sha256' => $body_hash,
            )));
        }
    }
}
