<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Signal_Manager {
    public static function emit(string $type, string $severity, string $summary, array $details = array()): void {
        $details = self::normalize_owasp_details($details);
        $normalized_severity = self::normalize_severity($severity);
        if (strtolower($severity) !== strtolower($normalized_severity)) {
            $details['raw_severity'] = $severity;
        }
        $event = array(
            'event_id' => function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : uniqid('ss_evt_', true),
            'type' => $type,
            'severity' => $normalized_severity,
            'summary' => $summary,
            'details' => $details,
            'occurred_at' => gmdate('c'),
        );

        ShadowScan_Event_Buffer::enqueue($event);

        if ($type === 'AUTH_BRUTE_FORCE_SUSPECTED') {
            ShadowScan_Storage::set('auth_last_anomaly', time());
        }

        if (strpos($type, 'INTEGRITY_') === 0 && in_array($severity, array('high', 'warning'), true)) {
            ShadowScan_Storage::set('integrity_last_high_event', time());
            ShadowScan_Storage::set('integrity_last_high_type', $type);
            ShadowScan_Storage::set('integrity_last_high_summary', $summary);
        }
    }

    private static function normalize_owasp_details(array $details): array {
        $category_id = null;
        if (!empty($details['owasp']) && is_array($details['owasp']) && !empty($details['owasp']['id'])) {
            $category_id = (string) $details['owasp']['id'];
        }
        if ($category_id === null && !empty($details['owasp_id'])) {
            $category_id = (string) $details['owasp_id'];
        }
        if ($category_id === null && !empty($details['control_id'])) {
            $category_id = (string) $details['control_id'];
        }
        if ($category_id === null && !empty($details['category']) && preg_match('/^A\d{2}$/', (string) $details['category'])) {
            $category_id = (string) $details['category'];
        }

        if ($category_id !== null) {
            $category_id = strtoupper(str_replace('OWASP-', '', $category_id));
            if (preg_match('/^A\d{2}$/', $category_id)) {
                $details['category'] = 'owasp';
                $details['owasp_id'] = $category_id;
                if (empty($details['control_id'])) {
                    $details['control_id'] = 'OWASP-' . $category_id;
                }
                if (empty($details['control_key']) && !empty($details['subcontrol'])) {
                    $details['control_key'] = $details['subcontrol'];
                }
                if (empty($details['owasp']) || !is_array($details['owasp'])) {
                    $details['owasp'] = array(
                        'version' => '2025',
                        'id' => $category_id,
                        'name' => function_exists('shadowscan_owasp_category_name')
                            ? shadowscan_owasp_category_name($category_id)
                            : $category_id,
                    );
                }
            }
        }

        return $details;
    }

    private static function normalize_severity(string $severity): string {
        $value = strtolower(trim($severity));
        if ($value === 'sev1' || $value === '1' || $value === 'critical') {
            return 'SEV1';
        }
        if ($value === 'sev2' || $value === '2' || $value === 'error') {
            return 'SEV2';
        }
        if ($value === 'sev3' || $value === '3' || $value === 'warning' || $value === 'info') {
            return 'SEV3';
        }
        return 'SEV3';
    }
}
