<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Policy_State {
    private const OPTION_POLICY_CACHE = 'shadowscan_policy_cache';
    private const OPTION_POLICY_CACHE_AT = 'shadowscan_policy_cache_at';
    private const OPTION_SUPPORT_DEBUG = 'shadowscan_support_debug';
    private const CACHE_TTL = 600;

    public static function get_current_policy(): array {
        $policy = get_option(self::OPTION_POLICY_CACHE, array());
        if (!is_array($policy)) {
            $policy = array();
        }
        $updated_at = (int) get_option(self::OPTION_POLICY_CACHE_AT, 0);
        $stale = !$updated_at || (time() - $updated_at) > self::CACHE_TTL;

        $normalized = self::normalize_policy($policy);
        $normalized['updated_at'] = $updated_at;
        $normalized['stale'] = $stale;

        return $normalized;
    }

    public static function get_support_debug(): array {
        $data = get_option(self::OPTION_SUPPORT_DEBUG, array());
        if (!is_array($data)) {
            $data = array();
        }
        return array(
            'enabled' => !empty($data['enabled']),
            'enabledUntil' => isset($data['enabledUntil']) ? (string) $data['enabledUntil'] : '',
        );
    }

    public static function refresh_policy(bool $force = false): ?array {
        if (!shadowscan_is_connected()) {
            return null;
        }

        $site_id = shadowscan_get_site_id();
        if ($site_id === '') {
            return null;
        }

        $updated_at = (int) get_option(self::OPTION_POLICY_CACHE_AT, 0);
        if (!$force && $updated_at && (time() - $updated_at) < self::CACHE_TTL) {
            return self::get_current_policy();
        }

        $response = shadowscan_send_api_request('site-policy?site_id=' . rawurlencode($site_id), array(), 'GET', true);
        if (is_wp_error($response) || !is_array($response)) {
            return null;
        }
        if (function_exists('shadowscan_persist_identity_from_response')) {
            shadowscan_persist_identity_from_response($response);
        }

        $signature = isset($response['signature']) ? (string) $response['signature'] : '';
        $commands = isset($response['commands']) && is_array($response['commands']) ? $response['commands'] : array();
        $payload = array(
            'site_id' => isset($response['site_id']) ? (string) $response['site_id'] : $site_id,
            'policy' => isset($response['policy']) && is_array($response['policy']) ? $response['policy'] : array(),
            'updated_at' => isset($response['updated_at']) ? $response['updated_at'] : null,
            'updated_by' => isset($response['updated_by']) ? $response['updated_by'] : null,
            'supportDebug' => isset($response['supportDebug']) && is_array($response['supportDebug']) ? $response['supportDebug'] : array(),
            'policy_version' => $response['policy_version'] ?? null,
        );

        if (!self::verify_signature($payload, $signature)) {
            return null;
        }

        $payload['commands'] = $commands;
        return self::ingest_policy_payload($payload);
    }

    public static function ingest_policy_payload(array $payload): ?array {
        $policy_version = $payload['policy_version'] ?? null;
        $commands = isset($payload['commands']) && is_array($payload['commands']) ? $payload['commands'] : array();
        $require_version = defined('SHADOWSCAN_REQUIRE_POLICY_VERSION') && SHADOWSCAN_REQUIRE_POLICY_VERSION;

        if ($policy_version === null || $policy_version === '') {
            if ($require_version) {
                shadowscan_set_last_apply_result(array(
                    'policy_version' => null,
                    'status' => 'failed',
                    'error_code' => 'missing_policy_version',
                    'error_message' => 'Policy version missing.',
                    'applied_at' => time(),
                ));
                ShadowScan_Commands::handle($commands);
                return null;
            }
            $policy_version = 'unversioned';
        }

        $state = shadowscan_get_sync_state();
        $last_applied = $state['last_applied_policy_version'] ?? null;
        $cmp = shadowscan_compare_policy_version($policy_version, $last_applied);
        shadowscan_set_policy_versions($policy_version, null);

        if ($cmp !== null && $cmp <= 0) {
            ShadowScan_Commands::handle($commands);
            return self::get_current_policy();
        }

        $policy = isset($payload['policy']) && is_array($payload['policy']) ? $payload['policy'] : null;
        if ($policy === null) {
            shadowscan_set_last_apply_result(array(
                'policy_version' => $policy_version,
                'status' => 'failed',
                'error_code' => 'invalid_payload',
                'error_message' => 'Policy payload missing.',
                'applied_at' => time(),
            ));
            ShadowScan_Commands::handle($commands);
            return null;
        }

        $previous = get_option(self::OPTION_POLICY_CACHE, array());
        if (!is_array($previous)) {
            $previous = array();
        }

        try {
            $normalized = self::normalize_policy($policy);
            update_option(self::OPTION_POLICY_CACHE, $normalized, false);
            update_option(self::OPTION_POLICY_CACHE_AT, time(), false);
            if (isset($payload['supportDebug'])) {
                update_option(self::OPTION_SUPPORT_DEBUG, $payload['supportDebug'], false);
            }

            self::apply_policy($normalized, $previous);
            self::apply_security_controls($normalized);

            shadowscan_set_policy_versions($policy_version, $policy_version);
            shadowscan_set_last_apply_result(array(
                'policy_version' => $policy_version,
                'status' => 'success',
                'error_code' => null,
                'error_message' => null,
                'applied_at' => time(),
            ));
        } catch (Throwable $e) {
            shadowscan_set_last_apply_result(array(
                'policy_version' => $policy_version,
                'status' => 'failed',
                'error_code' => 'exception',
                'error_message' => shadowscan_sanitize_error_message($e->getMessage()),
                'applied_at' => time(),
            ));
        }

        ShadowScan_Commands::handle($commands);
        return self::get_current_policy();
    }

    private static function apply_policy(array $policy, array $previous): void {
        $toggle_map = self::toggle_option_map();
        if (empty($toggle_map)) {
            return;
        }
        $subscription_state = class_exists('ShadowScan_Subscription_State')
            ? ShadowScan_Subscription_State::get_current_state()
            : array();
        $billing_active = isset($subscription_state['status'])
            && $subscription_state['status'] === ShadowScan_Subscription_State::STATUS_ACTIVE;
        $dev_env = function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env();

        $new_map = $policy['owasp_enforcement'] ?? array();
        if (!is_array($new_map)) {
            $new_map = array();
        }
        $prev_map = $previous['owasp_enforcement'] ?? array();
        if (!is_array($prev_map)) {
            $prev_map = array();
        }

        $all_keys = array_unique(array_merge(array_keys($toggle_map), array_keys($new_map), array_keys($prev_map)));
        $tier = isset($policy['tier']) ? (string) $policy['tier'] : 'monitor';

        foreach ($all_keys as $control_key) {
            if (!isset($toggle_map[$control_key])) {
                continue;
            }
            if (!$billing_active && !$dev_env) {
                continue;
            }
            $enabled = array_key_exists($control_key, $new_map) ? (bool) $new_map[$control_key] : false;
            $previous_enabled = array_key_exists($control_key, $prev_map) ? (bool) $prev_map[$control_key] : false;
            if ($enabled === $previous_enabled) {
                continue;
            }

            $flag = $toggle_map[$control_key];
            if ($flag === 'mfa_enforce_admins' && class_exists('ShadowScan_MFA')) {
                ShadowScan_MFA::set_enforcement_enabled($enabled);
            } elseif ($flag === 'autoupdate_plugins_enabled' && class_exists('ShadowScan_AutoUpdates')) {
                ShadowScan_AutoUpdates::set_enabled($enabled);
            } elseif ($flag === 'password_enforce_min_length' && class_exists('ShadowScan_Password_Policy')) {
                ShadowScan_Password_Policy::set_enforce_min_length($enabled);
            } elseif ($flag === 'password_enforce_breached' && class_exists('ShadowScan_Password_Policy')) {
                ShadowScan_Password_Policy::set_enforce_breached($enabled);
            } elseif ($flag === 'geo_block_mode' && class_exists('ShadowScan_Geo_Policy')) {
                ShadowScan_Geo_Policy::set_mode($enabled ? 'enforce_wp_admin' : 'off');
            } else {
                shadowscan_guard_manager()->set_flag($flag, $enabled);
            }

            $details = array(
                'control_key' => $control_key,
                'status' => 'ok',
                'enforced' => $enabled,
                'previous_enforced' => $previous_enabled,
                'policy_tier' => $tier,
            );
            if (in_array($control_key, array(
                'mfa_enforce_admins',
                'autoupdate_plugins_enforce',
                'password_policy_enforce_min_length',
                'password_policy_enforce_breached',
                'geo_policy_enforce_wp_admin',
            ), true)) {
                $details['category'] = 'security_control';
            } else {
                $category_id = self::category_from_control($control_key);
                $details['category'] = 'owasp';
                $details['owasp_id'] = $category_id;
            }

            ShadowScan_Signal_Manager::emit(
                'owasp_policy_applied',
                'info',
                'OWASP policy applied',
                $details
            );
        }
    }

    private static function apply_security_controls(array $policy): void {
        $controls = $policy['security_controls'] ?? array();
        if (!is_array($controls)) {
            return;
        }
        if (!empty($controls['admin_geo_guard']) && is_array($controls['admin_geo_guard']) && class_exists('ShadowScan_Admin_Geo_Guard')) {
            ShadowScan_Admin_Geo_Guard::set_policy($controls['admin_geo_guard']);
        }

        $subscription_state = class_exists('ShadowScan_Subscription_State')
            ? ShadowScan_Subscription_State::get_current_state()
            : array();
        $billing_active = isset($subscription_state['status'])
            && $subscription_state['status'] === ShadowScan_Subscription_State::STATUS_ACTIVE;
        $dev_env = function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env();
        if (!$billing_active && !$dev_env) {
            return;
        }

        if (!empty($controls['autoupdate']) && is_array($controls['autoupdate']) && class_exists('ShadowScan_AutoUpdates')) {
            ShadowScan_AutoUpdates::sync_policy($controls['autoupdate'], 'policy');
        }

        if (!empty($controls['password_policy']) && is_array($controls['password_policy']) && class_exists('ShadowScan_Password_Policy')) {
            $pw = $controls['password_policy'];
            if (isset($pw['min_length'])) {
                ShadowScan_Password_Policy::set_min_length((int) $pw['min_length']);
            }
            if (isset($pw['enforce_min_length'])) {
                ShadowScan_Password_Policy::set_enforce_min_length((bool) $pw['enforce_min_length']);
            }
            if (isset($pw['enforce_breached'])) {
                ShadowScan_Password_Policy::set_enforce_breached((bool) $pw['enforce_breached']);
            }
            if (isset($pw['provider'])) {
                ShadowScan_Password_Policy::set_breach_provider((string) $pw['provider']);
            }
        }
    }

    private static function toggle_option_map(): array {
        if (!function_exists('shadowscan_owasp_get_registry')) {
            return array(
                'mfa_enforce_admins' => 'mfa_enforce_admins',
                'autoupdate_plugins_enforce' => 'autoupdate_plugins_enabled',
                'password_policy_enforce_min_length' => 'password_enforce_min_length',
                'password_policy_enforce_breached' => 'password_enforce_breached',
                'geo_policy_enforce_wp_admin' => 'geo_block_mode',
            );
        }
        $registry = shadowscan_owasp_get_registry();
        $map = array();
        foreach ($registry as $category) {
            $controls = $category['controls'] ?? array();
            foreach ($controls as $control) {
                if (!empty($control['control_key']) && !empty($control['toggle_option_key'])) {
                    $map[$control['control_key']] = $control['toggle_option_key'];
                }
            }
        }
        $map['mfa_enforce_admins'] = 'mfa_enforce_admins';
        $map['autoupdate_plugins_enforce'] = 'autoupdate_plugins_enabled';
        $map['password_policy_enforce_min_length'] = 'password_enforce_min_length';
        $map['password_policy_enforce_breached'] = 'password_enforce_breached';
        $map['geo_policy_enforce_wp_admin'] = 'geo_block_mode';
        return $map;
    }

    private static function normalize_policy(array $policy): array {
        $tier = isset($policy['tier']) ? sanitize_text_field((string) $policy['tier']) : 'monitor';
        $enforcement_enabled = !empty($policy['enforcement_enabled']);
        $map = $policy['owasp_enforcement'] ?? array();
        $security_controls = $policy['security_controls'] ?? array();
        if (!is_array($map)) {
            $map = array();
        }
        if (!is_array($security_controls)) {
            $security_controls = array();
        }
        if (isset($security_controls['geo'])) {
            unset($security_controls['geo']);
        }
        $clean = array();
        foreach ($map as $key => $value) {
            if (!is_string($key)) {
                continue;
            }
            $clean[$key] = (bool) $value;
        }
        return array(
            'tier' => $tier,
            'enforcement_enabled' => $enforcement_enabled,
            'owasp_enforcement' => $clean,
            'security_controls' => $security_controls,
        );
    }

    private static function verify_signature(array $payload, string $signature): bool {
        if ($signature === '') {
            return false;
        }
        $key = shadowscan_get_site_token();
        if ($key === '') {
            return false;
        }
        $canonical = self::canonicalize(self::normalize_signature_payload($payload));
        $expected = hash_hmac('sha256', $canonical, $key, true);

        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
        $sig_bin = base64_decode($signature, true);
        if ($sig_bin === false) {
            $sig_bin = hex2bin($signature);
        }
        if ($sig_bin === false) {
            return false;
        }

        return hash_equals($expected, $sig_bin);
    }

    private static function canonicalize(array $payload): string {
        $payload = self::recursive_ksort($payload);
        $json = wp_json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        return is_string($json) ? $json : '';
    }

    private static function normalize_signature_payload(array $payload): array {
        if (isset($payload['policy']['owasp_enforcement']) && is_array($payload['policy']['owasp_enforcement'])) {
            if ($payload['policy']['owasp_enforcement'] === array()) {
                $payload['policy']['owasp_enforcement'] = (object) array();
            }
        }
        return $payload;
    }

    private static function recursive_ksort(array $payload): array {
        foreach ($payload as $key => $value) {
            if (is_array($value)) {
                $payload[$key] = self::recursive_ksort($value);
            }
        }
        if (array_keys($payload) !== range(0, count($payload) - 1)) {
            ksort($payload);
        }
        return $payload;
    }

    private static function category_from_control(string $control_key): string {
        if (preg_match('/^a(\d{2})_/i', $control_key, $matches)) {
            return 'A' . $matches[1];
        }
        return 'A01';
    }
}
