<?php
if (!defined('ABSPATH')) {
    exit;
}

const SHADOWSCAN_OPTION_OWASP_STATUS = 'owasp_status_cache';
const SHADOWSCAN_OPTION_OWASP_STATUS_AT = 'owasp_status_updated_at';
const SHADOWSCAN_OPTION_OWASP_DEDUPE = 'owasp_status_dedupe';
const SHADOWSCAN_OPTION_ADMIN_BASELINE = 'admin_baseline';

function shadowscan_owasp_get_registry(): array {
    $registry = require __DIR__ . '/registry.php';
    return is_array($registry) ? $registry : array();
}

function shadowscan_owasp_get_status_cache(): array {
    $cache = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_OWASP_STATUS, array());
    return is_array($cache) ? $cache : array();
}

function shadowscan_owasp_refresh_status(bool $force = false): array {
    $updated_at = (int) ShadowScan_Storage::get(SHADOWSCAN_OPTION_OWASP_STATUS_AT, 0);
    if (!$force && $updated_at > 0 && (time() - $updated_at) < 300) {
        return shadowscan_owasp_get_status_cache();
    }

    $previous_cache = shadowscan_owasp_get_status_cache();
    $registry = shadowscan_owasp_get_registry();
    $results = array(
        'updated_at' => time(),
        'categories' => array(),
    );

    foreach ($registry as $category) {
        $category_id = (string) ($category['category_id'] ?? '');
        if ($category_id === '') {
            continue;
        }
        $controls = $category['controls'] ?? array();
        $category_status = shadowscan_owasp_run_checks($category_id, $controls, $previous_cache);
        $results['categories'][$category_id] = $category_status;
    }

    ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_OWASP_STATUS, $results);
    ShadowScan_Storage::set(SHADOWSCAN_OPTION_OWASP_STATUS_AT, time());
    return $results;
}

function shadowscan_owasp_run_all_checks(): array {
    return shadowscan_owasp_refresh_status(true);
}

function shadowscan_owasp_get_category_status(string $category_id): array {
    $cache = shadowscan_owasp_refresh_status();
    return $cache['categories'][$category_id] ?? array();
}

function shadowscan_owasp_get_control_status(string $control_key): array {
    $cache = shadowscan_owasp_refresh_status();
    foreach ($cache['categories'] ?? array() as $category) {
        if (!empty($category['controls'][$control_key])) {
            return $category['controls'][$control_key];
        }
    }
    return array();
}

function shadowscan_owasp_run_checks(string $category_id, array $controls = array(), array $previous_cache = array()): array {
    if (empty($controls)) {
        $registry = shadowscan_owasp_get_registry();
        foreach ($registry as $category) {
            if (($category['category_id'] ?? '') === $category_id) {
                $controls = $category['controls'] ?? array();
                break;
            }
        }
    }
    $category_status = array(
        'category_id' => $category_id,
        'status' => 'not_implemented',
        'controls' => array(),
        'implemented_count' => 0,
        'total_controls' => is_array($controls) ? count($controls) : 0,
        'last_checked_at' => 0,
    );

    $has_warning = false;
    $has_critical = false;
    if (is_array($controls) && !empty($controls)) {
        foreach ($controls as $control) {
            $control_key = (string) ($control['control_key'] ?? '');
            $callback = $control['detection_callback'] ?? null;
            if ($control_key === '' || !is_callable($callback)) {
                continue;
            }
            $status = call_user_func($callback, $control);
            if (!is_array($status)) {
                continue;
            }
            $status['control_key'] = $control_key;
            $status['name'] = (string) ($control['name'] ?? $control_key);
            $status['recommended_action'] = (string) ($status['recommended_action'] ?? shadowscan_owasp_recommended_action($control_key));
            $status['last_checked_at'] = (int) ($status['last_checked_at'] ?? time());
            $status['risk_default'] = (string) ($control['risk_default'] ?? 'low');
            $status['toggle_option_key'] = $control['toggle_option_key'] ?? null;
            $status['enforcement_callback'] = $control['enforcement_callback'] ?? null;
            $status['enforced'] = !empty($control['toggle_option_key'])
                ? (bool) shadowscan_guard_manager()->is_flag_enabled((string) $control['toggle_option_key'], false)
                : false;
            $status['evidence'] = shadowscan_owasp_redact_evidence($status['details'] ?? array());
            $status['fingerprint'] = shadowscan_owasp_fingerprint($category_id, $status);
            $category_status['controls'][$control_key] = $status;
            $category_status['implemented_count']++;

            if (!empty($status['last_checked_at'])) {
                $category_status['last_checked_at'] = max($category_status['last_checked_at'], (int) $status['last_checked_at']);
            }
            if ($status['status'] === 'warn') {
                $has_warning = true;
            }
            if ($status['status'] === 'critical') {
                $has_critical = true;
            }

            shadowscan_owasp_maybe_emit_event($category_id, $status);
            shadowscan_owasp_maybe_emit_status_change($category_id, $status, $previous_cache);
        }
    }

    if ($category_status['implemented_count'] > 0) {
        if ($has_critical) {
            $category_status['status'] = 'critical';
        } elseif ($has_warning) {
            $category_status['status'] = 'warn';
        } else {
            $category_status['status'] = 'ok';
        }
    }

    return $category_status;
}

function shadowscan_owasp_status_label(string $status): string {
    $map = array(
        'ok' => __('Healthy', 'shadowscan-security-link'),
        'warn' => __('Degraded', 'shadowscan-security-link'),
        'critical' => __('High Risk', 'shadowscan-security-link'),
        'not_implemented' => __('Monitoring planned', 'shadowscan-security-link'),
    );
    return $map[$status] ?? $map['not_implemented'];
}

function shadowscan_owasp_fingerprint(string $category_id, array $status): string {
    $control_key = (string) ($status['control_key'] ?? '');
    $payload = array(
        'category_id' => $category_id,
        'control_key' => $control_key,
        'status' => $status['status'] ?? '',
        'details' => $status['details'] ?? array(),
    );
    return hash('sha256', wp_json_encode($payload));
}

function shadowscan_owasp_redact_evidence(array $details): array {
    $redacted = array();
    foreach ($details as $key => $value) {
        if (preg_match('/email|token|secret|password|ip_address|usernames?/i', (string) $key)) {
            continue;
        }
        $redacted[$key] = $value;
    }
    return $redacted;
}

function shadowscan_owasp_category_name(string $category_id): string {
    $map = array(
        'A01' => 'Broken Access Control',
        'A02' => 'Cryptographic Failures',
        'A03' => 'Injection',
        'A04' => 'Insecure Design',
        'A05' => 'Security Misconfiguration',
        'A06' => 'Vulnerable and Outdated Components',
        'A07' => 'Identification and Authentication Failures',
        'A08' => 'Software and Data Integrity Failures',
        'A09' => 'Security Logging and Monitoring Failures',
        'A10' => 'Server-Side Request Forgery',
    );
    return $map[$category_id] ?? $category_id;
}

function shadowscan_owasp_maybe_emit_event(string $category_id, array $status): void {
    $level = $status['status'] ?? 'ok';
    if (!in_array($level, array('warn', 'critical'), true)) {
        return;
    }

    $control_key = (string) ($status['control_key'] ?? '');
    if ($control_key === '') {
        return;
    }
    if (function_exists('shadowscan_is_known_control_key') && !shadowscan_is_known_control_key($control_key)) {
        if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env() && function_exists('shadowscan_log_message')) {
            shadowscan_log_message('[ShadowScan] Unknown OWASP control_key: ' . $control_key);
        }
        return;
    }

    $details = array(
        'control_id' => 'OWASP-' . $category_id,
        'owasp_id' => $category_id,
        'category' => 'owasp',
        'control_key' => $control_key,
        'risk_level' => $status['risk_default'] ?? 'low',
        'recommended_action' => $status['recommended_action'] ?? '',
        'status' => $level,
        'message' => $status['message'] ?? '',
        'fingerprint' => $status['fingerprint'] ?? '',
        'last_checked_at' => (int) ($status['last_checked_at'] ?? time()),
        'owasp' => array(
            'version' => '2025',
            'id' => $category_id,
            'name' => shadowscan_owasp_category_name($category_id),
        ),
        'evidence' => $status['evidence'] ?? array(),
        'enforced' => $status['enforced'] ?? false,
    );

    $fingerprint = hash('sha256', $control_key . '|' . $level . '|' . wp_json_encode($details));
    $dedupe = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_OWASP_DEDUPE, array());
    if (!is_array($dedupe)) {
        $dedupe = array();
    }
    $last_status = $dedupe[$control_key]['status'] ?? '';
    $last_sent = (int) ($dedupe[$control_key]['last_sent'] ?? 0);
    $last_fingerprint = $dedupe[$control_key]['fingerprint'] ?? '';

    $should_send = $last_status !== $level || $last_fingerprint !== $fingerprint || (time() - $last_sent) > DAY_IN_SECONDS;
    if (!$should_send) {
        return;
    }

    $event_type = $level === 'critical' ? 'owasp_control_critical' : 'owasp_control_warn';
    $summary = (string) ($status['message'] ?? __('OWASP control issue detected', 'shadowscan-security-link'));

    if (class_exists('ShadowScan_Signal_Manager')) {
        ShadowScan_Signal_Manager::emit($event_type, $level === 'critical' ? 'critical' : 'warning', $summary, $details);
    }

    $dedupe[$control_key] = array(
        'status' => $level,
        'last_sent' => time(),
        'fingerprint' => $fingerprint,
    );
    ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_OWASP_DEDUPE, $dedupe);
}

function shadowscan_owasp_maybe_emit_status_change(string $category_id, array $status, array $previous_cache): void {
    $control_key = (string) ($status['control_key'] ?? '');
    if ($control_key === '') {
        return;
    }
    if (function_exists('shadowscan_is_known_control_key') && !shadowscan_is_known_control_key($control_key)) {
        if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env() && function_exists('shadowscan_log_message')) {
            shadowscan_log_message('[ShadowScan] Unknown OWASP control_key: ' . $control_key);
        }
        return;
    }

    $previous_status = $previous_cache['categories'][$category_id]['controls'][$control_key]['status'] ?? null;
    $current_status = $status['status'] ?? 'ok';

    if ($previous_status === null) {
        return;
    }

    $resolved_transition = in_array($previous_status, array('warn', 'critical'), true) && $current_status === 'ok';
    $status_changed = $previous_status !== $current_status;
    if (!$status_changed && !$resolved_transition) {
        return;
    }

    $dedupe = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_OWASP_DEDUPE, array());
    if (!is_array($dedupe)) {
        $dedupe = array();
    }
    $dedupe_key = 'status_change_' . $control_key;
    $last_sent = (int) ($dedupe[$dedupe_key]['last_sent'] ?? 0);
    if (!$status_changed && (time() - $last_sent) < DAY_IN_SECONDS) {
        return;
    }

    $status_to = $resolved_transition ? 'resolved' : $current_status;
    $details = array(
        'control_id' => 'OWASP-' . $category_id,
        'owasp_id' => $category_id,
        'category' => 'owasp',
        'control_key' => $control_key,
        'status_from' => $previous_status,
        'status_to' => $status_to,
        'status' => $status_to,
        'risk_level' => $status['risk_default'] ?? 'low',
        'recommended_action' => $status['recommended_action'] ?? '',
        'fingerprint' => $status['fingerprint'] ?? '',
        'last_checked_at' => (int) ($status['last_checked_at'] ?? time()),
        'owasp' => array(
            'version' => '2025',
            'id' => $category_id,
            'name' => shadowscan_owasp_category_name($category_id),
        ),
        'evidence' => $status['evidence'] ?? array(),
        'enforced' => $status['enforced'] ?? false,
    );

    if (class_exists('ShadowScan_Signal_Manager')) {
        ShadowScan_Signal_Manager::emit(
            'owasp_control_status_changed',
            $current_status === 'critical' ? 'critical' : ($current_status === 'warn' ? 'warning' : 'info'),
            'OWASP control status changed',
            $details
        );
    }

    $dedupe[$dedupe_key] = array(
        'last_sent' => time(),
        'status' => $current_status,
    );
    ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_OWASP_DEDUPE, $dedupe);
}

function shadowscan_owasp_recommended_action(string $control_key): string {
    $map = array(
        'a01_admin_capabilities' => __('Review administrator roles in WordPress and reduce admin count where possible.', 'shadowscan-security-link'),
        'a01_privilege_escalation' => __('Review recent role changes in WordPress and confirm legitimacy.', 'shadowscan-security-link'),
        'a01_rest_route_permissions' => __('Review custom REST routes and ensure permission callbacks are enforced.', 'shadowscan-security-link'),
        'a01_rest_sensitive_block' => __('In ShadowScan, enable REST restrictions for sensitive endpoints if safe.', 'shadowscan-security-link'),
        'a01_xmlrpc' => __('Disable XML-RPC in ShadowScan unless required by trusted integrations.', 'shadowscan-security-link'),
        'a01_file_mods' => __('Set DISALLOW_FILE_EDIT and DISALLOW_FILE_MODS in wp-config.php if supported by your host.', 'shadowscan-security-link'),
        'a02_https_not_enforced' => __('Ask your host to enforce HTTPS and update WordPress URLs.', 'shadowscan-security-link'),
        'a02_insecure_cookies' => __('Enable secure cookies by enforcing HTTPS site-wide.', 'shadowscan-security-link'),
        'a02_weak_hashing_usage' => __('Update or remove plugins using weak hashing.', 'shadowscan-security-link'),
        'a02_missing_security_headers' => __('Enable recommended security headers in ShadowScan or your host.', 'shadowscan-security-link'),
        'a02_force_ssl_admin' => __('Enable FORCE_SSL_ADMIN if supported by your host.', 'shadowscan-security-link'),
        'a03_db_error_leak' => __('Disable database error display and debug output.', 'shadowscan-security-link'),
        'a03_untrusted_input_patterns' => __('Review input validation and sanitize custom code.', 'shadowscan-security-link'),
        'a03_waf_block_signals' => __('Review WAF alerts and confirm blocking rules.', 'shadowscan-security-link'),
        'a03_disable_debug_display' => __('Disable WP_DEBUG display in production.', 'shadowscan-security-link'),
        'a04_missing_rate_limits' => __('Enable rate limiting in ShadowScan settings.', 'shadowscan-security-link'),
        'a04_sensitive_actions_require_nonce' => __('Ensure forms and endpoints use nonces.', 'shadowscan-security-link'),
        'a04_break_glass_controls' => __('Review emergency access controls and limit use.', 'shadowscan-security-link'),
        'a04_enforce_sensitive_action_policy' => __('Enable sensitive action policy in ShadowScan.', 'shadowscan-security-link'),
        'a05_file_editor_enabled' => __('Disable the theme/plugin editor in WordPress.', 'shadowscan-security-link'),
        'a05_directory_listing_possible' => __('Disable directory listing in your server configuration.', 'shadowscan-security-link'),
        'a05_debug_enabled' => __('Disable WP_DEBUG in production.', 'shadowscan-security-link'),
        'a05_exposed_version' => __('Hide the WordPress version output.', 'shadowscan-security-link'),
        'a05_open_rest_user_enum' => __('Restrict REST user endpoints or enable the ShadowScan enumeration guard.', 'shadowscan-security-link'),
        'a06_outdated_wp_core' => __('Update WordPress core to the latest security release.', 'shadowscan-security-link'),
        'a06_outdated_plugins' => __('Update outdated plugins.', 'shadowscan-security-link'),
        'a06_outdated_themes' => __('Update outdated themes.', 'shadowscan-security-link'),
        'a06_auto_update_security_patches' => __('Enable auto-updates for security patches in WordPress.', 'shadowscan-security-link'),
        'a07_no_mfa_detected' => __('Enable MFA for administrator accounts.', 'shadowscan-security-link'),
        'a07_weak_password_policy' => __('Enable password policy enforcement on change/reset.', 'shadowscan-security-link'),
        'a07_bruteforce_signals' => __('Review login abuse signals and enable containment.', 'shadowscan-security-link'),
        'a07_default_admin_username' => __('Rename or remove the default admin user.', 'shadowscan-security-link'),
        'a07_login_rate_limit' => __('Enable login rate limiting in ShadowScan.', 'shadowscan-security-link'),
        'a08_plugin_file_integrity_changed' => __('Review file integrity changes and restore from backup if unexpected.', 'shadowscan-security-link'),
        'a08_updates_from_untrusted_sources' => __('Verify plugins and themes are sourced from trusted providers.', 'shadowscan-security-link'),
        'a08_webhook_signature_missing' => __('Configure command signing in the ShadowScan portal.', 'shadowscan-security-link'),
        'a08_enforce_signature_validation' => __('Enable signature enforcement once signing is configured.', 'shadowscan-security-link'),
        'a09_audit_log_disabled' => __('Enable admin audit logging in ShadowScan.', 'shadowscan-security-link'),
        'a09_missing_admin_audit' => __('Generate an admin activity event to confirm logging.', 'shadowscan-security-link'),
        'a09_no_alert_channels' => __('Configure alert routing in the ShadowScan portal.', 'shadowscan-security-link'),
        'a09_fail_closed_on_logging' => __('Enable fail-closed logging if your host supports it.', 'shadowscan-security-link'),
        'a10_unrestricted_http_requests' => __('Configure outbound allowlist rules in ShadowScan to restrict external calls.', 'shadowscan-security-link'),
        'a10_open_redirect_proxy_endpoints' => __('Review endpoints for open redirect or proxy behavior.', 'shadowscan-security-link'),
        'a10_metadata_ip_access' => __('Block access to cloud metadata IPs at the host or network level.', 'shadowscan-security-link'),
        'a10_enforce_outbound_allowlist' => __('Enable outbound allowlist enforcement in ShadowScan.', 'shadowscan-security-link'),
    );
    return $map[$control_key] ?? __('Open ShadowScan settings and review the flagged control.', 'shadowscan-security-link');
}

function shadowscan_owasp_check_admin_capabilities(array $control): array {
    $admins = get_users(array('role' => 'administrator', 'fields' => array('ID', 'user_login')));
    $admin_count = is_array($admins) ? count($admins) : 0;
    $admin_usernames = array();
    if (is_array($admins)) {
        foreach ($admins as $admin) {
            $admin_usernames[] = (string) $admin->user_login;
        }
    }
    $baseline = ShadowScan_Storage::get_json(SHADOWSCAN_OPTION_ADMIN_BASELINE, array());
    $baseline_hash = (string) ($baseline['hash'] ?? '');
    $current_hash = hash('sha256', wp_json_encode($admin_usernames));
    $new_admins_detected = $baseline_hash !== '' && $baseline_hash !== $current_hash;

    if ($baseline_hash === '') {
        ShadowScan_Storage::set_json(SHADOWSCAN_OPTION_ADMIN_BASELINE, array(
            'hash' => $current_hash,
            'admins' => $admin_usernames,
            'updated_at' => time(),
        ));
    }

    $status = 'ok';
    $message = __('Admin roles look healthy.', 'shadowscan-security-link');
    if ($new_admins_detected) {
        $status = 'critical';
        $message = __('Administrator list changed since last baseline.', 'shadowscan-security-link');
    } elseif ($admin_count > 2) {
        $status = 'warn';
        $message = __('Administrator count exceeds the recommended threshold.', 'shadowscan-security-link');
    }

    return array(
        'status' => $status,
        'message' => $message,
        'details' => array(
            'admin_count' => $admin_count,
            'admin_usernames' => $admin_usernames,
            'baseline_changed' => $new_admins_detected,
        ),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_privilege_escalation(array $control): array {
    $state = shadowscan_guard_manager()->get_access_control_state();
    $events = $state['subcontrol_events']['privilege_escalation'] ?? array();
    $last_event = (int) ($events['last_event_at'] ?? 0);
    if ($last_event > 0 && (time() - $last_event) < 30 * DAY_IN_SECONDS) {
        return array(
            'status' => 'critical',
            'message' => __('Recent privilege escalation detected.', 'shadowscan-security-link'),
            'details' => array('last_event_at' => $last_event),
            'last_checked_at' => time(),
        );
    }
    return array(
        'status' => 'ok',
        'message' => __('No recent privilege escalation detected.', 'shadowscan-security-link'),
        'details' => array(),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_rest_routes(array $control): array {
    $state = shadowscan_guard_manager()->get_access_control_state();
    $rest_state = $state['rest_routes'] ?? array();
    $insecure_count = (int) ($rest_state['insecure_count'] ?? 0);
    $status = $insecure_count > 0 ? 'warn' : 'ok';
    $message = $insecure_count > 0
        ? __('REST routes missing permission checks detected.', 'shadowscan-security-link')
        : __('REST route permissions look healthy.', 'shadowscan-security-link');
    return array(
        'status' => $status,
        'message' => $message,
        'details' => array('insecure_count' => $insecure_count),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_rest_sensitive_block(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('rest_block_users_settings', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Sensitive REST endpoints are protected.', 'shadowscan-security-link')
            : __('Sensitive REST endpoints are not blocked for unauthenticated access.', 'shadowscan-security-link'),
        'details' => array('enforced' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_xmlrpc(array $control): array {
    $state = shadowscan_guard_manager()->get_access_control_state();
    $xmlrpc = $state['xmlrpc'] ?? array();
    $enabled = !empty($xmlrpc['enabled']);
    $status = $enabled ? 'warn' : 'ok';
    $message = $enabled
        ? __('XML-RPC is enabled.', 'shadowscan-security-link')
        : __('XML-RPC is disabled.', 'shadowscan-security-link');
    return array(
        'status' => $status,
        'message' => $message,
        'details' => array('enabled' => $enabled, 'multicall' => !empty($xmlrpc['multicall'])),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_file_mods(array $control): array {
    $state = shadowscan_guard_manager()->get_access_control_state();
    $file_mods = $state['file_mods'] ?? array();
    $file_edit_allowed = !empty($file_mods['file_edit_allowed']);
    $file_mods_allowed = !empty($file_mods['file_mods_allowed']);
    $status = ($file_edit_allowed || $file_mods_allowed) ? 'warn' : 'ok';
    $message = ($file_edit_allowed || $file_mods_allowed)
        ? __('File editing or installs are allowed.', 'shadowscan-security-link')
        : __('File editing is disabled.', 'shadowscan-security-link');
    return array(
        'status' => $status,
        'message' => $message,
        'details' => array(
            'file_edit_allowed' => $file_edit_allowed,
            'file_mods_allowed' => $file_mods_allowed,
        ),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_rest_sensitive_block(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('rest_block_users_settings', $enabled);
}

function shadowscan_owasp_toggle_xmlrpc_disable(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('xmlrpc_enforce_disable', $enabled);
}

function shadowscan_owasp_check_https_not_enforced(array $control): array {
    $home = home_url();
    $site = site_url();
    $is_https = strpos($home, 'https://') === 0 && strpos($site, 'https://') === 0;
    return array(
        'status' => $is_https ? 'ok' : 'warn',
        'message' => $is_https ? __('HTTPS is enforced for site URLs.', 'shadowscan-security-link') : __('Site URLs are not HTTPS.', 'shadowscan-security-link'),
        'details' => array('home_url' => $home, 'site_url' => $site),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_insecure_cookies(array $control): array {
    $home = home_url();
    $is_https = strpos($home, 'https://') === 0;
    return array(
        'status' => $is_https ? 'ok' : 'warn',
        'message' => $is_https
            ? __('HTTPS enabled; secure cookies can be enforced.', 'shadowscan-security-link')
            : __('HTTPS not enabled; auth cookies cannot be secure.', 'shadowscan-security-link'),
        'details' => array('home_url' => $home),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_weak_hashing(array $control): array {
    $patterns = array('md5(', 'sha1(');
    $hits = shadowscan_owasp_scan_plugin_patterns($patterns);
    $status = $hits > 0 ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $hits > 0
            ? __('Weak hashing calls detected in plugin code.', 'shadowscan-security-link')
            : __('No weak hashing calls detected in plugin code.', 'shadowscan-security-link'),
        'details' => array('matches' => $hits),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_security_headers(array $control): array {
    $response = wp_remote_head(home_url('/'), array('timeout' => 5));
    if (is_wp_error($response)) {
        return array(
            'status' => 'warn',
            'message' => __('Unable to verify security headers.', 'shadowscan-security-link'),
            'details' => array('error' => $response->get_error_message()),
            'last_checked_at' => time(),
        );
    }
    $headers = wp_remote_retrieve_headers($response);
    $required = array('strict-transport-security', 'content-security-policy', 'x-content-type-options');
    $missing = array();
    foreach ($required as $header) {
        if (empty($headers[$header])) {
            $missing[] = $header;
        }
    }
    $status = empty($missing) ? 'ok' : 'warn';
    return array(
        'status' => $status,
        'message' => empty($missing)
            ? __('Core security headers detected.', 'shadowscan-security-link')
            : __('Some security headers are missing.', 'shadowscan-security-link'),
        'details' => array('missing_headers' => $missing),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_force_ssl_admin(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('force_ssl_admin', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Force SSL admin is enabled.', 'shadowscan-security-link')
            : __('Force SSL admin is disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_force_ssl_admin(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('force_ssl_admin', $enabled);
}

function shadowscan_owasp_check_db_error_leak(array $control): array {
    $debug = defined('WP_DEBUG') && WP_DEBUG;
    $display = defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY;
    $status = ($debug && $display) ? 'critical' : ($debug ? 'warn' : 'ok');
    return array(
        'status' => $status,
        'message' => $status === 'critical'
            ? __('Debug display enabled; errors may leak.', 'shadowscan-security-link')
            : ($status === 'warn' ? __('Debug mode enabled.', 'shadowscan-security-link') : __('Debug display disabled.', 'shadowscan-security-link')),
        'details' => array('wp_debug' => $debug, 'wp_debug_display' => $display),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_untrusted_input_patterns(array $control): array {
    $hits = shadowscan_owasp_scan_plugin_lines('/\\$wpdb->query\\(/', '/prepare\\(/');
    $status = $hits > 0 ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $hits > 0
            ? __('Potential unprepared database queries detected.', 'shadowscan-security-link')
            : __('No unprepared database query patterns detected.', 'shadowscan-security-link'),
        'details' => array('matches' => $hits),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_waf_signals(array $control): array {
    $state = shadowscan_guard_manager()->get_access_control_state();
    $abuse = (int) (($state['admin_endpoint_abuse']['count'] ?? 0));
    $status = $abuse >= 10 ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $abuse >= 10
            ? __('Repeated admin endpoint abuse signals detected.', 'shadowscan-security-link')
            : __('No high-confidence injection signals detected.', 'shadowscan-security-link'),
        'details' => array('abuse_count' => $abuse),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_disable_debug_display(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('owasp_disable_debug_display', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Debug display suppression enabled.', 'shadowscan-security-link')
            : __('Debug display suppression disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_disable_debug_display(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('owasp_disable_debug_display', $enabled);
}

function shadowscan_owasp_check_rate_limits(array $control): array {
    $refresh = defined('SHADOWSCAN_REFRESH_COOLDOWN') ? SHADOWSCAN_REFRESH_COOLDOWN : 0;
    $self_check = defined('SHADOWSCAN_SELF_CHECK_COOLDOWN') ? SHADOWSCAN_SELF_CHECK_COOLDOWN : 0;
    $status = ($refresh >= 30 && $self_check >= 30) ? 'ok' : 'warn';
    return array(
        'status' => $status,
        'message' => $status === 'ok'
            ? __('Rate limit guardrails are configured.', 'shadowscan-security-link')
            : __('Rate limit guardrails are below recommended thresholds.', 'shadowscan-security-link'),
        'details' => array('refresh_cooldown' => $refresh, 'self_check_cooldown' => $self_check),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_nonce_requirements(array $control): array {
    $checks = shadowscan_owasp_scan_plugin_patterns(array("check_admin_referer('shadowscan-dashboard')", 'check_admin_referer("shadowscan-dashboard")'));
    $status = $checks > 0 ? 'ok' : 'warn';
    return array(
        'status' => $status,
        'message' => $status === 'ok'
            ? __('Admin actions include nonce checks.', 'shadowscan-security-link')
            : __('Nonce checks not detected for admin actions.', 'shadowscan-security-link'),
        'details' => array('matches' => $checks),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_break_glass(array $control): array {
    $safe_until = (int) (shadowscan_guard_manager()->get_guard_status()['safe_mode_until'] ?? 0);
    $duration = $safe_until > time() ? $safe_until - time() : 0;
    $status = $duration > 600 ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $status === 'ok'
            ? __('Break-glass controls are within policy.', 'shadowscan-security-link')
            : __('Break-glass duration exceeds recommended limit.', 'shadowscan-security-link'),
        'details' => array('safe_mode_seconds' => $duration),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_sensitive_action_policy(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('owasp_enforce_sensitive_actions', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Sensitive action policy enforcement enabled.', 'shadowscan-security-link')
            : __('Sensitive action policy enforcement disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_sensitive_actions(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('owasp_enforce_sensitive_actions', $enabled);
}

function shadowscan_owasp_check_file_editor(array $control): array {
    $file_edit_allowed = !defined('DISALLOW_FILE_EDIT') || !DISALLOW_FILE_EDIT;
    $status = $file_edit_allowed ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $file_edit_allowed
            ? __('File editor is enabled.', 'shadowscan-security-link')
            : __('File editor is disabled.', 'shadowscan-security-link'),
        'details' => array('file_edit_allowed' => $file_edit_allowed),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_directory_listing(array $control): array {
    $state = shadowscan_guard_manager()->get_htaccess_state();
    $applied = ($state['last_result'] ?? '') === 'applied';
    $status = $applied ? 'ok' : 'warn';
    return array(
        'status' => $status,
        'message' => $applied
            ? __('Directory listing baseline applied.', 'shadowscan-security-link')
            : __('Directory listing baseline not applied.', 'shadowscan-security-link'),
        'details' => array('htaccess_applied' => $applied),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_debug_enabled(array $control): array {
    $debug = defined('WP_DEBUG') && WP_DEBUG;
    $script_debug = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG;
    $status = ($debug || $script_debug) ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $status === 'ok'
            ? __('Debug flags are disabled.', 'shadowscan-security-link')
            : __('Debug flags are enabled.', 'shadowscan-security-link'),
        'details' => array('wp_debug' => $debug, 'script_debug' => $script_debug),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_exposed_version(array $control): array {
    $generator = get_the_generator('html');
    $exposed = is_string($generator) && strpos($generator, get_bloginfo('version')) !== false;
    return array(
        'status' => $exposed ? 'warn' : 'ok',
        'message' => $exposed
            ? __('WordPress version may be exposed.', 'shadowscan-security-link')
            : __('WordPress version exposure not detected.', 'shadowscan-security-link'),
        'details' => array('generator_present' => $exposed),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_rest_user_enum(array $control): array {
    $flag = shadowscan_guard_manager()->is_flag_enabled('restrict_rest_users', false)
        || shadowscan_guard_manager()->is_flag_enabled('rest_block_users_settings', false);
    return array(
        'status' => $flag ? 'ok' : 'warn',
        'message' => $flag
            ? __('REST user enumeration protections enabled.', 'shadowscan-security-link')
            : __('REST user enumeration protections disabled.', 'shadowscan-security-link'),
        'details' => array('protected' => $flag),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_outdated_core(array $control): array {
    $updates = get_site_transient('update_core');
    $needs = is_object($updates) && !empty($updates->updates) && isset($updates->updates[0]->response) && $updates->updates[0]->response === 'upgrade';
    return array(
        'status' => $needs ? 'warn' : 'ok',
        'message' => $needs
            ? __('WordPress core update available.', 'shadowscan-security-link')
            : __('WordPress core is up to date.', 'shadowscan-security-link'),
        'details' => array('update_available' => $needs),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_outdated_plugins(array $control): array {
    $updates = get_site_transient('update_plugins');
    $count = is_object($updates) && !empty($updates->response) ? count((array) $updates->response) : 0;
    return array(
        'status' => $count > 0 ? 'warn' : 'ok',
        'message' => $count > 0
            ? __('Plugin updates are available.', 'shadowscan-security-link')
            : __('Plugins are up to date.', 'shadowscan-security-link'),
        'details' => array('outdated_plugins' => $count),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_outdated_themes(array $control): array {
    $updates = get_site_transient('update_themes');
    $count = is_object($updates) && !empty($updates->response) ? count((array) $updates->response) : 0;
    return array(
        'status' => $count > 0 ? 'warn' : 'ok',
        'message' => $count > 0
            ? __('Theme updates are available.', 'shadowscan-security-link')
            : __('Themes are up to date.', 'shadowscan-security-link'),
        'details' => array('outdated_themes' => $count),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_auto_updates(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('owasp_auto_update_minor', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Auto-updates for minor releases enabled.', 'shadowscan-security-link')
            : __('Auto-updates for minor releases disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_auto_updates(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('owasp_auto_update_minor', $enabled);
}

function shadowscan_owasp_check_mfa_detected(array $control): array {
    $mfa_enabled = class_exists('ShadowScan_MFA')
        ? ShadowScan_MFA::is_enforcement_enabled()
        : shadowscan_guard_manager()->is_flag_enabled('enforce_mfa', false);
    $active = get_option('active_plugins', array());
    $detected = false;
    foreach ($active as $plugin) {
        if (stripos($plugin, 'two-factor') !== false || stripos($plugin, 'google-authenticator') !== false || stripos($plugin, 'wordfence') !== false) {
            $detected = true;
            break;
        }
    }
    $status = ($mfa_enabled || $detected) ? 'ok' : 'warn';
    return array(
        'status' => $status,
        'message' => $status === 'ok'
            ? __('MFA controls detected.', 'shadowscan-security-link')
            : __('No MFA controls detected.', 'shadowscan-security-link'),
        'details' => array('mfa_enforced' => $mfa_enabled, 'plugin_detected' => $detected),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_password_policy(array $control): array {
    $stats = shadowscan_guard_manager()->get_bruteforce_stats(true);
    $failed = (int) ($stats['failed_last_window'] ?? 0);
    $config = ShadowScan_Storage::get_json('bf_config', array());
    $ip_block = !empty($config['ip_block_enabled']);
    $status = ($failed >= 20 && !$ip_block) ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $status === 'warn'
            ? __('Repeated failures detected without IP blocking.', 'shadowscan-security-link')
            : __('Password policy signals are normal.', 'shadowscan-security-link'),
        'details' => array('failed_last_window' => $failed, 'ip_block_enabled' => $ip_block),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_bruteforce_signals(array $control): array {
    $stats = shadowscan_guard_manager()->get_bruteforce_stats(true);
    $locks = (int) ($stats['active_user_locks'] ?? 0);
    $blocks = (int) ($stats['active_ip_blocks'] ?? 0);
    $status = ($locks > 0 || $blocks > 0) ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $status === 'warn'
            ? __('Active login containment detected.', 'shadowscan-security-link')
            : __('No active login containment.', 'shadowscan-security-link'),
        'details' => array('active_user_locks' => $locks, 'active_ip_blocks' => $blocks),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_default_admin_username(array $control): array {
    $state = shadowscan_guard_manager()->get_access_control_state();
    $flag = !empty($state['admin_capabilities']['admin_username_flag']);
    return array(
        'status' => $flag ? 'warn' : 'ok',
        'message' => $flag
            ? __('Default admin username detected.', 'shadowscan-security-link')
            : __('Default admin username not detected.', 'shadowscan-security-link'),
        'details' => array('admin_username_flag' => $flag),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_login_rate_limit(array $control): array {
    $config = ShadowScan_Storage::get_json('bf_config', array());
    $enabled = !empty($config['ip_block_enabled']);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Login rate limiting enabled.', 'shadowscan-security-link')
            : __('Login rate limiting disabled.', 'shadowscan-security-link'),
        'details' => array('ip_block_enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_login_rate_limit(bool $enabled): void {
    $config = ShadowScan_Storage::get_json('bf_config', array());
    $config['ip_block_enabled'] = $enabled;
    ShadowScan_Storage::set_json('bf_config', $config);
}

function shadowscan_owasp_check_integrity_changes(array $control): array {
    $last = (int) ShadowScan_Storage::get('integrity_last_high_event', 0);
    $recent = $last > 0 && (time() - $last) < 30 * DAY_IN_SECONDS;
    return array(
        'status' => $recent ? 'warn' : 'ok',
        'message' => $recent
            ? __('Recent integrity changes detected.', 'shadowscan-security-link')
            : __('No recent integrity changes detected.', 'shadowscan-security-link'),
        'details' => array('last_event_at' => $last),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_untrusted_updates(array $control): array {
    $active = get_option('active_plugins', array());
    $untrusted = 0;
    foreach ($active as $plugin) {
        $data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin, false, false);
        $update_uri = isset($data['UpdateURI']) ? (string) $data['UpdateURI'] : '';
        if ($update_uri !== '' && stripos($update_uri, 'wordpress.org') === false && stripos($update_uri, 'wp.org') === false) {
            $untrusted++;
        }
    }
    return array(
        'status' => $untrusted > 0 ? 'warn' : 'ok',
        'message' => $untrusted > 0
            ? __('Plugins with non-WordPress update sources detected.', 'shadowscan-security-link')
            : __('Plugin update sources appear standard.', 'shadowscan-security-link'),
        'details' => array('untrusted_plugins' => $untrusted),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_signature_validation(array $control): array {
    $has_key = defined('SHADOWSCAN_COMMAND_SIGNING_KEY') && SHADOWSCAN_COMMAND_SIGNING_KEY;
    return array(
        'status' => $has_key ? 'ok' : 'warn',
        'message' => $has_key
            ? __('Command signing key configured.', 'shadowscan-security-link')
            : __('Command signing key not configured.', 'shadowscan-security-link'),
        'details' => array('signing_key_configured' => $has_key),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_signature_enforcement(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('owasp_enforce_command_signing', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Command signature enforcement enabled.', 'shadowscan-security-link')
            : __('Command signature enforcement disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_command_signing(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('owasp_enforce_command_signing', $enabled);
}

function shadowscan_owasp_check_audit_pipeline(array $control): array {
    $queue = ShadowScan_Event_Buffer::count();
    $last_ping = (int) get_option(SHADOWSCAN_OPTION_LAST_PING, 0);
    $stale = $last_ping > 0 && (time() - $last_ping) > 15 * MINUTE_IN_SECONDS;
    $status = ($queue > 0 && $stale) ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $status === 'warn'
            ? __('Event queue backlog detected.', 'shadowscan-security-link')
            : __('Event pipeline is healthy.', 'shadowscan-security-link'),
        'details' => array('queue_count' => $queue, 'last_ping' => $last_ping),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_admin_audit(array $control): array {
    $last = (int) ShadowScan_Storage::get('admin_integrity_last_event', 0);
    $stale = $last > 0 && (time() - $last) > 30 * DAY_IN_SECONDS;
    $status = ($last === 0 || $stale) ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $status === 'warn'
            ? __('Admin audit events not seen recently.', 'shadowscan-security-link')
            : __('Admin audit events are present.', 'shadowscan-security-link'),
        'details' => array('last_event_at' => $last),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_alert_channels(array $control): array {
    $portal_unreachable = (bool) ShadowScan_Storage::get(SHADOWSCAN_OPTION_SUBSCRIPTION_PORTAL_UNREACHABLE, false);
    $status = $portal_unreachable ? 'warn' : 'ok';
    return array(
        'status' => $status,
        'message' => $portal_unreachable
            ? __('Portal unreachable; alert routing may be degraded.', 'shadowscan-security-link')
            : __('Alert routing status normal.', 'shadowscan-security-link'),
        'details' => array('portal_unreachable' => $portal_unreachable),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_fail_closed_logging(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('owasp_fail_closed_logging', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Fail-closed logging policy enabled.', 'shadowscan-security-link')
            : __('Fail-closed logging policy disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_fail_closed_logging(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('owasp_fail_closed_logging', $enabled);
}

function shadowscan_owasp_check_outbound_allowlist(array $control): array {
    $allowed = shadowscan_is_outbound_allowlist_enabled();
    return array(
        'status' => $allowed ? 'ok' : 'warn',
        'message' => $allowed
            ? __('Outbound requests are allowlisted.', 'shadowscan-security-link')
            : __('Outbound allowlist enforcement is disabled.', 'shadowscan-security-link'),
        'details' => array('enforced' => $allowed),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_proxy_endpoints(array $control): array {
    $hits = shadowscan_owasp_scan_plugin_patterns(array('wp_remote_get($_', 'wp_remote_post($_'));
    $status = $hits > 0 ? 'critical' : 'ok';
    return array(
        'status' => $status,
        'message' => $hits > 0
            ? __('Potential proxy-style endpoints detected.', 'shadowscan-security-link')
            : __('No proxy-style endpoints detected.', 'shadowscan-security-link'),
        'details' => array('matches' => $hits),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_metadata_access(array $control): array {
    $hits = shadowscan_owasp_scan_plugin_patterns(array('169.254.169.254'));
    return array(
        'status' => $hits > 0 ? 'warn' : 'ok',
        'message' => $hits > 0
            ? __('Metadata IP access pattern detected.', 'shadowscan-security-link')
            : __('Metadata IP access not detected.', 'shadowscan-security-link'),
        'details' => array('matches' => $hits),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_check_outbound_enforcement(array $control): array {
    $enabled = shadowscan_guard_manager()->is_flag_enabled('owasp_outbound_allowlist', false);
    return array(
        'status' => $enabled ? 'ok' : 'warn',
        'message' => $enabled
            ? __('Outbound allowlist enforcement enabled.', 'shadowscan-security-link')
            : __('Outbound allowlist enforcement disabled.', 'shadowscan-security-link'),
        'details' => array('enabled' => $enabled),
        'last_checked_at' => time(),
    );
}

function shadowscan_owasp_toggle_outbound_allowlist(bool $enabled): void {
    shadowscan_guard_manager()->set_flag('owasp_outbound_allowlist', $enabled);
}

function shadowscan_is_outbound_allowlist_enabled(): bool {
    return shadowscan_guard_manager()->is_flag_enabled('owasp_outbound_allowlist', false);
}

function shadowscan_owasp_scan_plugin_patterns(array $patterns): int {
    $count = 0;
    $dir = plugin_dir_path(__DIR__ . '/status.php');
    $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
    $checked = 0;
    $filesystem = function_exists('shadowscan_get_filesystem') ? shadowscan_get_filesystem() : null;
    if (!$filesystem) {
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        $filesystem = $GLOBALS['wp_filesystem'] ?? null;
    }
    if (!$filesystem || !method_exists($filesystem, 'get_contents')) {
        return $count;
    }

    foreach ($iterator as $file) {
        if ($file->isDir()) {
            continue;
        }
        if (substr($file->getFilename(), -4) !== '.php') {
            continue;
        }
        if ($checked >= 200) {
            break;
        }
        $checked++;
        $contents = $filesystem->get_contents($file->getPathname());
        if ($contents === false) {
            continue;
        }
        foreach ($patterns as $pattern) {
            if (strpos($contents, $pattern) !== false) {
                $count++;
                break;
            }
        }
    }
    return $count;
}

function shadowscan_owasp_scan_plugin_lines(string $pattern, string $preparePattern): int {
    $count = 0;
    $dir = plugin_dir_path(__DIR__ . '/status.php');
    $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
    foreach ($iterator as $file) {
        if ($file->isDir()) {
            continue;
        }
        if (substr($file->getFilename(), -4) !== '.php') {
            continue;
        }
        $lines = file($file->getPathname());
        if (!is_array($lines)) {
            continue;
        }
        foreach ($lines as $line) {
            if (preg_match($pattern, $line) && !preg_match($preparePattern, $line)) {
                $count++;
            }
        }
    }
    return $count;
}
