<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_MFA {
    private const OPTION_NUDGE_ENABLED = 'mfa_nudge_enabled';
    private const OPTION_ENFORCE_ADMINS = 'mfa_enforce_admins';
    private const OPTION_ENFORCE_ROLES = 'mfa_enforce_roles';
    private const OPTION_GRACE_PERIOD_HOURS = 'mfa_grace_period_hours';

    private const USER_META_SECRET = 'shadowscan_mfa_secret';
    private const USER_META_ENABLED = 'shadowscan_mfa_enabled';

    public static function is_user_enrolled(int $user_id): bool {
        $enabled = (bool) get_user_meta($user_id, self::USER_META_ENABLED, true);
        $secret = (string) get_user_meta($user_id, self::USER_META_SECRET, true);
        return $enabled && $secret !== '';
    }

    public static function get_provider_name(): string {
        return 'Authenticator App';
    }

    public static function get_setup_url(): string {
        if (!self::is_library_available()) {
            return '';
        }
        return admin_url('profile.php#shadowscan-mfa');
    }

    public static function get_verify_url(): string {
        return self::get_setup_url();
    }

    public static function is_library_available(): bool {
        return class_exists('PragmaRX\\Google2FA\\Google2FA');
    }

    public static function is_nudge_enabled(): bool {
        return (bool) ShadowScan_Storage::get(self::OPTION_NUDGE_ENABLED, true);
    }

    public static function set_nudge_enabled(bool $enabled): void {
        ShadowScan_Storage::set(self::OPTION_NUDGE_ENABLED, $enabled);
    }

    public static function is_enforcement_enabled(): bool {
        return (bool) ShadowScan_Storage::get(self::OPTION_ENFORCE_ADMINS, false);
    }

    public static function set_enforcement_enabled(bool $enabled): void {
        ShadowScan_Storage::set(self::OPTION_ENFORCE_ADMINS, $enabled);
        if (function_exists('shadowscan_guard_manager')) {
            shadowscan_guard_manager()->set_flag('enforce_mfa', $enabled);
        }
    }

    public static function get_enforced_roles(): array {
        $roles = ShadowScan_Storage::get_json(self::OPTION_ENFORCE_ROLES, array());
        if (!is_array($roles) || empty($roles)) {
            return array('administrator');
        }
        return array_values(array_filter($roles, 'is_string'));
    }

    public static function set_enforced_roles(array $roles): void {
        $clean = array_values(array_filter($roles, 'is_string'));
        ShadowScan_Storage::set_json(self::OPTION_ENFORCE_ROLES, $clean);
    }

    public static function get_grace_period_hours(): int {
        return max(0, (int) ShadowScan_Storage::get(self::OPTION_GRACE_PERIOD_HOURS, 0));
    }

    public static function set_grace_period_hours(int $hours): void {
        ShadowScan_Storage::set(self::OPTION_GRACE_PERIOD_HOURS, max(0, $hours));
    }

    public static function user_in_scope(WP_User $user): bool {
        $roles = self::get_enforced_roles();
        if (!empty(array_intersect($roles, $user->roles))) {
            return true;
        }
        return user_can($user, 'manage_options');
    }

    public static function compute_admin_coverage(): array {
        $users = get_users(array('fields' => array('ID')));
        $total = 0;
        $enrolled = 0;

        foreach ($users as $user) {
            if (!($user instanceof WP_User)) {
                $user_id = 0;
                if (is_object($user) && isset($user->ID) && is_numeric($user->ID)) {
                    $user_id = (int) $user->ID;
                } elseif (is_numeric($user)) {
                    $user_id = (int) $user;
                }
                $user = $user_id > 0 ? get_user_by('id', $user_id) : null;
            }
            if (!$user instanceof WP_User) {
                continue;
            }
            if (!self::user_in_scope($user)) {
                continue;
            }
            $total++;
            if (self::is_user_enrolled($user->ID)) {
                $enrolled++;
            }
        }

        $percent = $total > 0 ? round(($enrolled / $total) * 100, 1) : 0.0;

        return array(
            'admin_total' => $total,
            'admin_enrolled' => $enrolled,
            'coverage_percent' => $percent,
            'provider' => self::get_provider_name(),
            'computed_at' => time(),
        );
    }
}
