<?php
if (!defined('ABSPATH')) {
    exit;
}

use PragmaRX\Google2FA\Google2FA;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;

class ShadowScan_Guard_Mfa {
    private const USER_META_SECRET = 'shadowscan_mfa_secret';
    private const USER_META_ENABLED = 'shadowscan_mfa_enabled';
    private const USER_META_PENDING = 'shadowscan_mfa_pending_secret';
    private const USER_META_BACKUP_CODES = 'shadowscan_mfa_backup_codes';
    private const BACKUP_CODES_NOTICE = 'shadowscan_mfa_backup_codes_notice_';
    private const CHALLENGE_PREFIX = 'shadowscan_mfa_challenge_';
    private const CHALLENGE_TTL = 600;
    private const USER_META_PROMPT_PENDING = 'shadowscan_mfa_prompt_pending';
    private const USER_META_PROMPT_DISMISSED_UNTIL = 'shadowscan_mfa_prompt_dismissed_until';
    private const USER_META_PROMPT_LOGGED_AT = 'shadowscan_mfa_prompt_logged_at';
    private const USER_META_GRACE_START = 'shadowscan_mfa_grace_start';
    private const USER_META_BLOCK_LOGGED_AT = 'shadowscan_mfa_block_logged_at';

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_filter('authenticate', array($this, 'maybe_require_mfa'), 60, 3);
        add_action('login_form_shadowscan_mfa_verify', array($this, 'render_verify_form'));
        add_action('login_form_shadowscan_mfa_setup', array($this, 'render_setup_form'));
        add_action('show_user_profile', array($this, 'render_profile_fields'));
        add_action('edit_user_profile', array($this, 'render_profile_fields'));
        add_action('personal_options_update', array($this, 'handle_profile_update'));
        add_action('edit_user_profile_update', array($this, 'handle_profile_update'));
        add_action('wp_login', array($this, 'flag_mfa_prompt'), 10, 2);
        add_action('admin_notices', array($this, 'render_mfa_notice'));
        add_action('admin_init', array($this, 'enforce_admin_mfa'));
        add_action('admin_post_shadowscan_mfa_dismiss', array($this, 'handle_mfa_dismiss'));
    }

    public function maybe_require_mfa($user, string $username, string $password) {
        if (!$user || is_wp_error($user)) {
            return $user;
        }

        if (!class_exists(Google2FA::class)) {
            shadowscan_log_message('[ShadowScan] MFA library missing. Skipping MFA enforcement.');
            return $user;
        }

        if ($this->is_login_action('shadowscan_mfa_verify') || $this->is_login_action('shadowscan_mfa_setup')) {
            return $user;
        }

        if ($this->is_non_interactive_login()) {
            return $user;
        }

        if (!$user instanceof WP_User) {
            return $user;
        }

        $has_mfa = $this->user_has_mfa($user->ID);
        $requires_setup = !$has_mfa && $this->is_enforced_for_user($user) && !$this->grace_allows_access($user);
        $requires_mfa = $has_mfa || $requires_setup;
        if (!$requires_mfa) {
            return $user;
        }

        $action = $has_mfa ? 'shadowscan_mfa_verify' : 'shadowscan_mfa_setup';
        $token = $this->create_challenge($user->ID, $action === 'shadowscan_mfa_setup');
        $redirect = $this->get_login_redirect($action, $token);
        wp_safe_redirect($redirect);
        exit;
    }

    public function render_verify_form(): void {
        $this->render_login_form(false);
    }

    public function render_setup_form(): void {
        $this->render_login_form(true);
    }

    public function render_profile_fields(WP_User $user): void {
        if (!current_user_can('edit_user', $user->ID)) {
            return;
        }

        if (!class_exists(Google2FA::class)) {
            echo '<h2>' . esc_html__('ShadowScan MFA', 'shadowscan-security-link') . '</h2>';
            echo '<p class="description">' . esc_html__('Multi-factor authentication requires the Google2FA library. Please reinstall the plugin with dependencies or contact support.', 'shadowscan-security-link') . '</p>';
            return;
        }

        $notice = $this->get_notice($user->ID);
        $enabled = $this->user_has_mfa($user->ID);
        $pending_secret = $this->get_user_secret($user->ID, self::USER_META_PENDING);
        $secret = $this->get_user_secret($user->ID, self::USER_META_SECRET);
        $backup_codes = $this->get_backup_codes($user->ID);
        $new_backup_codes = $this->get_backup_codes_notice($user->ID);

        $label = $this->get_label($user);
        $otpauth = $pending_secret ? $this->get_qr_text($label, $pending_secret) : '';
        $qr_data_uri = $pending_secret ? $this->get_qr_data_uri($label, $pending_secret) : '';

        echo '<h2 id="shadowscan-mfa">' . esc_html__('ShadowScan MFA', 'shadowscan-security-link') . '</h2>';
        echo '<table class="form-table" role="presentation">';
        echo '<tbody>';

        if ($notice) {
            echo '<tr><th></th><td><div class="' . esc_attr($notice['class']) . '"><p>' . esc_html($notice['message']) . '</p></div></td></tr>';
        }

        echo '<tr>';
        echo '<th>' . esc_html__('Status', 'shadowscan-security-link') . '</th>';
        echo '<td>' . esc_html($enabled ? __('Enabled', 'shadowscan-security-link') : __('Disabled', 'shadowscan-security-link')) . '</td>';
        echo '</tr>';

        if (!empty($backup_codes)) {
            echo '<tr>';
            echo '<th>' . esc_html__('Backup codes', 'shadowscan-security-link') . '</th>';
            /* translators: %d: backup code count. */
            echo '<td>' . esc_html(sprintf(__('Available: %d', 'shadowscan-security-link'), count($backup_codes))) . '</td>';
            echo '</tr>';
        }

        if ($enabled && $secret) {
            $masked = substr($secret, 0, 4) . '...' . substr($secret, -4);
            echo '<tr>';
            echo '<th>' . esc_html__('Current secret', 'shadowscan-security-link') . '</th>';
            echo '<td><code>' . esc_html($masked) . '</code></td>';
            echo '</tr>';
        }

        if (!$enabled && $pending_secret) {
            echo '<tr>';
            echo '<th>' . esc_html__('Setup key', 'shadowscan-security-link') . '</th>';
            echo '<td><code>' . esc_html($pending_secret) . '</code></td>';
            echo '</tr>';
            if ($qr_data_uri !== '') {
                echo '<tr>';
                echo '<th>' . esc_html__('QR code', 'shadowscan-security-link') . '</th>';
                echo '<td><img src="' . esc_url($qr_data_uri, array('data')) . '" alt="' . esc_attr__('MFA QR code', 'shadowscan-security-link') . '" width="200" height="200" /></td>';
                echo '</tr>';
            }
            echo '<tr>';
            echo '<th>' . esc_html__('OTPAuth URL', 'shadowscan-security-link') . '</th>';
            echo '<td><code style="word-break:break-all;">' . esc_html($otpauth) . '</code></td>';
            echo '</tr>';
            echo '<tr>';
            echo '<th>' . esc_html__('Verification code', 'shadowscan-security-link') . '</th>';
            echo '<td><input type="text" name="shadowscan_mfa_code" class="regular-text" autocomplete="one-time-code" /></td>';
            echo '</tr>';
        }

        if (!empty($new_backup_codes)) {
            echo '<tr>';
            echo '<th>' . esc_html__('New backup codes', 'shadowscan-security-link') . '</th>';
            echo '<td><p>' . esc_html__('Store these codes safely. Each code can be used once.', 'shadowscan-security-link') . '</p>';
            echo '<div style="display:flex;flex-wrap:wrap;gap:6px;">';
            foreach ($new_backup_codes as $code) {
                echo '<code style="padding:2px 6px;border:1px solid #ccd0d4;border-radius:4px;">' . esc_html($code) . '</code>';
            }
            echo '</div></td>';
            echo '</tr>';
        }

        echo '<tr>';
        echo '<th>' . esc_html__('Actions', 'shadowscan-security-link') . '</th>';
        echo '<td>';
        wp_nonce_field('shadowscan_mfa_profile', 'shadowscan_mfa_profile_nonce');
        if (!$enabled && !$pending_secret) {
            echo '<button class="button" name="shadowscan_mfa_profile_action" value="start">' . esc_html__('Generate setup', 'shadowscan-security-link') . '</button>';
        } elseif (!$enabled && $pending_secret) {
            echo '<button class="button button-primary" name="shadowscan_mfa_profile_action" value="verify">' . esc_html__('Verify & enable', 'shadowscan-security-link') . '</button>';
            echo ' <button class="button" name="shadowscan_mfa_profile_action" value="reset">' . esc_html__('Regenerate secret', 'shadowscan-security-link') . '</button>';
        } else {
            echo '<button class="button" name="shadowscan_mfa_profile_action" value="disable">' . esc_html__('Disable MFA', 'shadowscan-security-link') . '</button>';
            echo ' <button class="button" name="shadowscan_mfa_profile_action" value="reset">' . esc_html__('Reset MFA', 'shadowscan-security-link') . '</button>';
            echo ' <button class="button" name="shadowscan_mfa_profile_action" value="backup_regenerate">' . esc_html__('Regenerate backup codes', 'shadowscan-security-link') . '</button>';
        }
        if ($enabled && empty($backup_codes)) {
            echo ' <button class="button" name="shadowscan_mfa_profile_action" value="backup_generate">' . esc_html__('Generate backup codes', 'shadowscan-security-link') . '</button>';
        }
        echo '</td>';
        echo '</tr>';

        echo '</tbody>';
        echo '</table>';
    }

    public function handle_profile_update(int $user_id): void {
        if (!current_user_can('edit_user', $user_id)) {
            return;
        }

        $action = isset($_POST['shadowscan_mfa_profile_action'])
            ? sanitize_text_field(wp_unslash($_POST['shadowscan_mfa_profile_action']))
            : '';

        if ($action === '') {
            return;
        }

        if (!isset($_POST['shadowscan_mfa_profile_nonce'])) {
            $this->set_notice($user_id, 'error', __('MFA update rejected. Please try again.', 'shadowscan-security-link'));
            return;
        }
        $nonce = sanitize_text_field(wp_unslash($_POST['shadowscan_mfa_profile_nonce']));
        if (!wp_verify_nonce($nonce, 'shadowscan_mfa_profile')) {
            $this->set_notice($user_id, 'error', __('MFA update rejected. Please try again.', 'shadowscan-security-link'));
            return;
        }

        if (!class_exists(Google2FA::class)) {
            $this->set_notice($user_id, 'error', __('MFA library missing.', 'shadowscan-security-link'));
            return;
        }

        if ($action === 'start' || $action === 'reset') {
            $secret = $this->tfa()->generateSecretKey(16);
            $this->set_user_secret($user_id, self::USER_META_PENDING, $secret);
            delete_user_meta($user_id, self::USER_META_ENABLED);
            $this->set_notice($user_id, 'updated', __('MFA setup generated. Scan the key and verify.', 'shadowscan-security-link'));
            return;
        }

        if ($action === 'disable') {
            delete_user_meta($user_id, self::USER_META_SECRET);
            delete_user_meta($user_id, self::USER_META_ENABLED);
            delete_user_meta($user_id, self::USER_META_PENDING);
            delete_user_meta($user_id, self::USER_META_BACKUP_CODES);
            delete_user_meta($user_id, self::USER_META_PROMPT_PENDING);
            delete_user_meta($user_id, self::USER_META_PROMPT_DISMISSED_UNTIL);
            delete_user_meta($user_id, self::USER_META_GRACE_START);
            $this->set_notice($user_id, 'updated', __('MFA disabled.', 'shadowscan-security-link'));
            return;
        }

        if ($action === 'backup_generate' || $action === 'backup_regenerate') {
            if (!$this->user_has_mfa($user_id)) {
                $this->set_notice($user_id, 'error', __('Enable MFA first to generate backup codes.', 'shadowscan-security-link'));
                return;
            }
            $codes = $this->generate_backup_codes(10);
            $this->store_backup_codes($user_id, $codes);
            $this->set_backup_codes_notice($user_id, $codes);
            $this->set_notice($user_id, 'updated', __('Backup codes generated.', 'shadowscan-security-link'));
            return;
        }

        if ($action === 'verify') {
            $code = isset($_POST['shadowscan_mfa_code']) ? sanitize_text_field(wp_unslash($_POST['shadowscan_mfa_code'])) : '';
            $pending = $this->get_user_secret($user_id, self::USER_META_PENDING);
            if ($pending === '') {
                $this->set_notice($user_id, 'error', __('No pending MFA setup found.', 'shadowscan-security-link'));
                return;
            }
            if (!$this->tfa()->verifyKey($pending, $code, 1)) {
                $this->set_notice($user_id, 'error', __('Invalid verification code.', 'shadowscan-security-link'));
                return;
            }
            $this->set_user_secret($user_id, self::USER_META_SECRET, $pending);
            update_user_meta($user_id, self::USER_META_ENABLED, 1);
            delete_user_meta($user_id, self::USER_META_PENDING);
            delete_user_meta($user_id, self::USER_META_PROMPT_PENDING);
            delete_user_meta($user_id, self::USER_META_PROMPT_DISMISSED_UNTIL);
            delete_user_meta($user_id, self::USER_META_GRACE_START);
            $this->set_notice($user_id, 'updated', __('MFA enabled.', 'shadowscan-security-link'));
        }
    }

    public function flag_mfa_prompt(string $user_login, WP_User $user): void {
        if (!$user instanceof WP_User) {
            return;
        }
        if (!class_exists('ShadowScan_MFA')) {
            return;
        }
        if (!ShadowScan_MFA::is_nudge_enabled()) {
            return;
        }
        if (!ShadowScan_MFA::is_library_available()) {
            return;
        }
        if (!ShadowScan_MFA::user_in_scope($user)) {
            return;
        }
        if (ShadowScan_MFA::is_user_enrolled($user->ID)) {
            return;
        }
        update_user_meta($user->ID, self::USER_META_PROMPT_PENDING, time());

        $grace_hours = ShadowScan_MFA::get_grace_period_hours();
        if (ShadowScan_MFA::is_enforcement_enabled() && $grace_hours > 0) {
            $existing = (int) get_user_meta($user->ID, self::USER_META_GRACE_START, true);
            if ($existing <= 0) {
                update_user_meta($user->ID, self::USER_META_GRACE_START, time());
            }
        }
    }

    public function render_mfa_notice(): void {
        if (!is_admin()) {
            return;
        }
        if (!class_exists('ShadowScan_MFA')) {
            return;
        }
        $user = wp_get_current_user();
        if (!$user instanceof WP_User || !$user->exists()) {
            return;
        }
        if (!ShadowScan_MFA::user_in_scope($user)) {
            return;
        }
        if (ShadowScan_MFA::is_user_enrolled($user->ID)) {
            return;
        }

        $enforcement = ShadowScan_MFA::is_enforcement_enabled();
        $pending = (int) get_user_meta($user->ID, self::USER_META_PROMPT_PENDING, true);
        if (!$enforcement && $pending <= 0) {
            return;
        }

        $dismissed_until = (int) get_user_meta($user->ID, self::USER_META_PROMPT_DISMISSED_UNTIL, true);
        if (!$enforcement && $dismissed_until > time()) {
            return;
        }

        $setup_url = ShadowScan_MFA::get_setup_url();
        if ($setup_url === '') {
            echo '<div class="notice notice-error"><p>' . esc_html__('MFA enforcement cannot be enabled until the setup page is available.', 'shadowscan-security-link') . '</p></div>';
            return;
        }

        $grace_message = '';
        if ($enforcement) {
            $grace_hours = ShadowScan_MFA::get_grace_period_hours();
            if ($grace_hours > 0) {
                $grace_start = (int) get_user_meta($user->ID, self::USER_META_GRACE_START, true);
                if ($grace_start > 0) {
                    $remaining = max(0, ($grace_start + ($grace_hours * HOUR_IN_SECONDS)) - time());
                    if ($remaining > 0) {
                        $grace_message = sprintf(
                            /* translators: %d: hours remaining. */
                            esc_html__('MFA enforcement begins in %d hours.', 'shadowscan-security-link'),
                            (int) ceil($remaining / HOUR_IN_SECONDS)
                        );
                    }
                }
            }
        }

        $this->maybe_log_prompt($user);

        echo '<div class="notice notice-warning"><p><strong>' . esc_html__('Enable MFA to protect your admin account.', 'shadowscan-security-link') . '</strong></p>';
        echo '<p>' . esc_html__('Multi-factor authentication adds an extra step at sign-in to keep your account safer.', 'shadowscan-security-link') . '</p>';
        if ($grace_message !== '') {
            echo '<p>' . esc_html($grace_message) . '</p>';
        }
        echo '<p><a class="button button-primary" href="' . esc_url($setup_url) . '">' . esc_html__('Set up MFA', 'shadowscan-security-link') . '</a>';

        if (!$enforcement) {
            echo ' <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;margin-left:6px;">';
            wp_nonce_field('shadowscan_mfa_dismiss', 'shadowscan_mfa_dismiss_nonce');
            echo '<input type="hidden" name="action" value="shadowscan_mfa_dismiss" />';
            echo '<button class="button" type="submit">' . esc_html__('Remind me later', 'shadowscan-security-link') . '</button>';
            echo '</form>';
        }
        echo '</p></div>';
    }

    public function handle_mfa_dismiss(): void {
        $user = wp_get_current_user();
        if (!$user instanceof WP_User || !$user->exists()) {
            wp_safe_redirect(admin_url());
            exit;
        }
        if (!current_user_can('edit_user', $user->ID)) {
            wp_safe_redirect(admin_url());
            exit;
        }
        if (!isset($_POST['shadowscan_mfa_dismiss_nonce'])) {
            wp_safe_redirect(admin_url());
            exit;
        }
        $nonce = sanitize_text_field(wp_unslash($_POST['shadowscan_mfa_dismiss_nonce']));
        if (!wp_verify_nonce($nonce, 'shadowscan_mfa_dismiss')) {
            wp_safe_redirect(admin_url());
            exit;
        }
        update_user_meta($user->ID, self::USER_META_PROMPT_DISMISSED_UNTIL, time() + DAY_IN_SECONDS);
        wp_safe_redirect(admin_url());
        exit;
    }

    public function enforce_admin_mfa(): void {
        if (!is_admin()) {
            return;
        }
        if (!class_exists('ShadowScan_MFA')) {
            return;
        }
        $user = wp_get_current_user();
        if (!$user instanceof WP_User || !$user->exists()) {
            return;
        }
        if (!ShadowScan_MFA::is_enforcement_enabled()) {
            return;
        }
        if (!ShadowScan_MFA::user_in_scope($user)) {
            return;
        }
        if (ShadowScan_MFA::is_user_enrolled($user->ID)) {
            return;
        }

        $setup_url = ShadowScan_MFA::get_setup_url();
        if ($setup_url === '' || !ShadowScan_MFA::is_library_available()) {
            $this->emit_enforcement_blocked_missing_setup($user->ID);
            return;
        }

        $grace_hours = ShadowScan_MFA::get_grace_period_hours();
        if ($grace_hours > 0) {
            $grace_start = (int) get_user_meta($user->ID, self::USER_META_GRACE_START, true);
            if ($grace_start <= 0) {
                $grace_start = time();
                update_user_meta($user->ID, self::USER_META_GRACE_START, $grace_start);
            }
            $grace_expires = $grace_start + ($grace_hours * HOUR_IN_SECONDS);
            if (time() < $grace_expires) {
                return;
            }
        }

        if ($this->is_allowed_admin_path($user)) {
            return;
        }

        $this->maybe_log_block($user);
        wp_safe_redirect(add_query_arg('shadowscan_mfa_required', '1', $setup_url));
        exit;
    }

    private function maybe_log_prompt(WP_User $user): void {
        $last = (int) get_user_meta($user->ID, self::USER_META_PROMPT_LOGGED_AT, true);
        if ($last > 0 && (time() - $last) < WEEK_IN_SECONDS) {
            return;
        }
        $details = array(
            'category' => 'security_control',
            'control_key' => 'mfa_nudge',
            'status' => 'warn',
            'user_id' => $user->ID,
            'role' => $user->roles[0] ?? '',
            'provider' => ShadowScan_MFA::get_provider_name(),
            'last_checked_at' => gmdate('c'),
        );
        ShadowScan_Signal_Manager::emit(
            'mfa_setup_prompt_shown',
            'info',
            'MFA setup prompt shown',
            $details
        );
        update_user_meta($user->ID, self::USER_META_PROMPT_LOGGED_AT, time());
    }

    private function maybe_log_block(WP_User $user): void {
        $last = (int) get_user_meta($user->ID, self::USER_META_BLOCK_LOGGED_AT, true);
        if ($last > 0 && (time() - $last) < DAY_IN_SECONDS) {
            return;
        }
        $path = isset($_SERVER['REQUEST_URI']) ? (string) wp_parse_url(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])), PHP_URL_PATH) : '';
        $details = array(
            'category' => 'security_control',
            'control_key' => 'mfa_enforcement',
            'status' => 'critical',
            'user_id' => $user->ID,
            'role' => $user->roles[0] ?? '',
            'requested_uri' => $path,
            'provider' => ShadowScan_MFA::get_provider_name(),
            'last_checked_at' => gmdate('c'),
        );
        ShadowScan_Signal_Manager::emit(
            'mfa_access_blocked',
            'warning',
            'Admin access blocked until MFA is set up',
            $details
        );
        update_user_meta($user->ID, self::USER_META_BLOCK_LOGGED_AT, time());
    }

    private function emit_enforcement_blocked_missing_setup(int $user_id): void {
        $last = (int) ShadowScan_Storage::get('mfa_missing_setup_alert', 0);
        if ($last > 0 && (time() - $last) < DAY_IN_SECONDS) {
            return;
        }
        ShadowScan_Signal_Manager::emit(
            'mfa_enforcement_blocked_missing_setup_url',
            'high',
            'MFA enforcement blocked because setup page is unavailable',
            array(
                'category' => 'security_control',
                'control_key' => 'mfa_enforcement',
                'status' => 'critical',
                'user_id' => $user_id,
                'provider' => ShadowScan_MFA::get_provider_name(),
                'last_checked_at' => gmdate('c'),
            )
        );
        ShadowScan_Storage::set('mfa_missing_setup_alert', time());
    }

    private function is_allowed_admin_path(WP_User $user): bool {
        if (wp_doing_ajax()) {
            return true;
        }
        global $pagenow;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $action = isset($_REQUEST['action']) ? sanitize_text_field(wp_unslash($_REQUEST['action'])) : '';

        if ($pagenow === 'profile.php') {
            return true;
        }
        if ($pagenow === 'user-edit.php') {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $target = isset($_GET['user_id'])
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                ? absint($_GET['user_id'])
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                : absint(wp_unslash($_POST['user_id'] ?? 0));
            return $target === $user->ID;
        }
        if ($pagenow === 'admin-post.php' && $action === 'shadowscan_mfa_dismiss') {
            return true;
        }
        if ($pagenow === 'admin.php' && $page === 'shadowscan') {
            return true;
        }

        return false;
    }

    private function render_login_form(bool $setup): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $token = isset($_REQUEST['shadowscan_mfa_token'])
            ? sanitize_text_field(wp_unslash($_REQUEST['shadowscan_mfa_token']))
            : '';
        $errors = new WP_Error();

        if (!class_exists(Google2FA::class)) {
            $errors->add('missing_library', __('Multi-factor authentication is unavailable. Please contact support.', 'shadowscan-security-link'));
            login_header(__('Two-factor authentication', 'shadowscan-security-link'), '', $errors);
            echo '<p class="message">' . esc_html__('MFA requires the Google2FA library.', 'shadowscan-security-link') . '</p>';
            echo '<p><a href="' . esc_url(wp_login_url()) . '">' . esc_html__('Back to login', 'shadowscan-security-link') . '</a></p>';
            login_footer();
            exit;
        }
        $challenge = $this->get_challenge($token);
        $user = $challenge ? get_user_by('id', (int) $challenge['user_id']) : null;

        if (!$token || !$challenge || !$user) {
            $errors->add('invalid_token', __('MFA session expired. Please sign in again.', 'shadowscan-security-link'));
        }

        $request_method = isset($_SERVER['REQUEST_METHOD']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD'])) : '';
        if ($request_method === 'POST') {
            if (!isset($_POST['shadowscan_mfa_nonce'])) {
                $errors->add('invalid_nonce', __('MFA verification failed. Please try again.', 'shadowscan-security-link'));
            } else {
                $nonce = sanitize_text_field(wp_unslash($_POST['shadowscan_mfa_nonce']));
                if (!wp_verify_nonce($nonce, 'shadowscan_mfa')) {
                    $errors->add('invalid_nonce', __('MFA verification failed. Please try again.', 'shadowscan-security-link'));
                } elseif ($challenge && $user) {
                    $code = isset($_POST['shadowscan_mfa_code']) ? sanitize_text_field(wp_unslash($_POST['shadowscan_mfa_code'])) : '';
                    $secret = $setup ? ($challenge['secret'] ?? '') : $this->get_user_secret($user->ID, self::USER_META_SECRET);
                    if ($secret === '') {
                        $errors->add('missing_secret', __('MFA secret missing. Please sign in again.', 'shadowscan-security-link'));
                    } else {
                        if ($this->tfa()->verifyKey($secret, $code, 1) || $this->consume_backup_code($user->ID, $code)) {
                            if ($setup) {
                                $this->set_user_secret($user->ID, self::USER_META_SECRET, $secret);
                                update_user_meta($user->ID, self::USER_META_ENABLED, 1);
                                delete_user_meta($user->ID, self::USER_META_PROMPT_PENDING);
                                delete_user_meta($user->ID, self::USER_META_PROMPT_DISMISSED_UNTIL);
                                delete_user_meta($user->ID, self::USER_META_GRACE_START);
                            }
                            $this->complete_login($token, $user, (bool) ($challenge['remember'] ?? false), (string) ($challenge['redirect_to'] ?? ''));
                            return;
                        }
                        $errors->add('invalid_code', __('Invalid verification or backup code.', 'shadowscan-security-link'));
                    }
                }
            }
        }

        $title = $setup
            ? __('Set up multi-factor authentication', 'shadowscan-security-link')
            : __('Two-factor authentication', 'shadowscan-security-link');
        login_header($title, '', $errors);

        if ($setup && $challenge && $user && !empty($challenge['secret'])) {
            $label = $this->get_label($user);
            $otpauth = $this->get_qr_text($label, $challenge['secret']);
            $qr_data_uri = $this->get_qr_data_uri($label, $challenge['secret']);
            echo '<p class="message">' . esc_html__('Scan the key with your authenticator app and enter the code (or a backup code) to continue.', 'shadowscan-security-link') . '</p>';
            echo '<p><strong>' . esc_html__('Setup key:', 'shadowscan-security-link') . '</strong> <code>' . esc_html($challenge['secret']) . '</code></p>';
            if ($qr_data_uri !== '') {
                echo '<p><img src="' . esc_url($qr_data_uri, array('data')) . '" alt="' . esc_attr__('MFA QR code', 'shadowscan-security-link') . '" width="200" height="200" /></p>';
            }
            echo '<p><strong>' . esc_html__('OTPAuth URL:', 'shadowscan-security-link') . '</strong> <code style="word-break:break-all;">' . esc_html($otpauth) . '</code></p>';
        } else {
            echo '<p class="message">' . esc_html__('Enter the code from your authenticator app or a backup code to continue.', 'shadowscan-security-link') . '</p>';
        }

        echo '<form method="post" action="' . esc_url($this->get_login_action_url($setup, $token)) . '">';
        wp_nonce_field('shadowscan_mfa', 'shadowscan_mfa_nonce');
        echo '<p>';
        echo '<label for="shadowscan_mfa_code">' . esc_html__('Authentication code', 'shadowscan-security-link') . '</label>';
        echo '<input type="text" name="shadowscan_mfa_code" id="shadowscan_mfa_code" class="input" value="" size="20" autocomplete="one-time-code" />';
        echo '</p>';
        echo '<p class="submit">';
        echo '<input type="submit" class="button button-primary button-large" value="' . esc_attr__('Verify', 'shadowscan-security-link') . '" />';
        echo '</p>';
        echo '</form>';
        echo '<p><a href="' . esc_url(wp_login_url()) . '">' . esc_html__('Back to login', 'shadowscan-security-link') . '</a></p>';
        login_footer();
        exit;
    }

    private function complete_login(string $token, WP_User $user, bool $remember, string $redirect_to): void {
        delete_transient(self::CHALLENGE_PREFIX . $token);
        wp_set_current_user($user->ID, $user->user_login);
        wp_set_auth_cookie($user->ID, $remember);
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        do_action('wp_login', $user->user_login, $user);
        $redirect = $redirect_to !== '' ? $redirect_to : admin_url();
        wp_safe_redirect(wp_validate_redirect($redirect, admin_url()));
        exit;
    }

    private function create_challenge(int $user_id, bool $setup): string {
        $token = wp_generate_password(20, false, false);
        $data = array(
            'user_id' => $user_id,
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            'redirect_to' => isset($_REQUEST['redirect_to']) ? esc_url_raw(wp_unslash($_REQUEST['redirect_to'])) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'remember' => !empty($_POST['rememberme']),
        );
        if ($setup) {
            $data['secret'] = $this->tfa()->generateSecretKey(16);
        }
        set_transient(self::CHALLENGE_PREFIX . $token, $data, self::CHALLENGE_TTL);
        return $token;
    }

    private function get_challenge(string $token): ?array {
        if ($token === '') {
            return null;
        }
        $data = get_transient(self::CHALLENGE_PREFIX . $token);
        if (!is_array($data) || empty($data['user_id'])) {
            return null;
        }
        return $data;
    }

    private function get_login_redirect(string $action, string $token): string {
        return add_query_arg(
            array(
                'action' => $action,
                'shadowscan_mfa_token' => $token,
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                'redirect_to' => isset($_REQUEST['redirect_to']) ? esc_url_raw(wp_unslash($_REQUEST['redirect_to'])) : '',
            ),
            wp_login_url()
        );
    }

    private function get_login_action_url(bool $setup, string $token): string {
        return add_query_arg(
            array(
                'action' => $setup ? 'shadowscan_mfa_setup' : 'shadowscan_mfa_verify',
                'shadowscan_mfa_token' => $token,
            ),
            wp_login_url()
        );
    }

    private function user_has_mfa(int $user_id): bool {
        $enabled = (bool) get_user_meta($user_id, self::USER_META_ENABLED, true);
        $secret = $this->get_user_secret($user_id, self::USER_META_SECRET);
        return $enabled && $secret !== '';
    }

    private function is_enforced_for_user(WP_User $user): bool {
        if (class_exists('ShadowScan_MFA')) {
            return ShadowScan_MFA::is_enforcement_enabled() && ShadowScan_MFA::user_in_scope($user);
        }
        return $this->manager->is_flag_enabled('enforce_mfa', false) && user_can($user, 'manage_options');
    }

    private function grace_allows_access(WP_User $user): bool {
        if (!class_exists('ShadowScan_MFA')) {
            return false;
        }
        $hours = ShadowScan_MFA::get_grace_period_hours();
        if ($hours <= 0) {
            return false;
        }
        $start = (int) get_user_meta($user->ID, self::USER_META_GRACE_START, true);
        if ($start <= 0) {
            $start = time();
            update_user_meta($user->ID, self::USER_META_GRACE_START, $start);
        }
        return time() < ($start + ($hours * HOUR_IN_SECONDS));
    }

    private function is_login_action(string $action): bool {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        return isset($_REQUEST['action']) && sanitize_text_field(wp_unslash($_REQUEST['action'])) === $action;
    }

    private function is_non_interactive_login(): bool {
        if (defined('XMLRPC_REQUEST') && XMLRPC_REQUEST) {
            return true;
        }
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return true;
        }
        if (wp_doing_ajax()) {
            return true;
        }
        return false;
    }

    private function tfa(): Google2FA {
        static $tfa = null;
        if ($tfa === null) {
            $tfa = new Google2FA();
        }
        return $tfa;
    }

    private function is_qr_available(): bool {
        return class_exists(Writer::class)
            && class_exists(ImageRenderer::class)
            && class_exists(RendererStyle::class)
            && class_exists(SvgImageBackEnd::class);
    }

    private function get_qr_data_uri(string $label, string $secret): string {
        try {
            if (!$this->is_qr_available()) {
                return '';
            }
            $qr_text = $this->get_qr_text($label, $secret);
            if ($qr_text === '') {
                return '';
            }
            $renderer = new ImageRenderer(
                new RendererStyle(200),
                new SvgImageBackEnd()
            );
            $writer = new Writer($renderer);
            $svg = $writer->writeString($qr_text);
            // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
            return 'data:image/svg+xml;base64,' . base64_encode($svg);
        } catch (Throwable $error) {
            shadowscan_log_message('[ShadowScan] MFA QR generation failed: ' . $error->getMessage());
            return '';
        }
    }

    private function get_issuer(): string {
        $name = get_bloginfo('name');
        return $name ? $name : 'ShadowScan';
    }

    private function get_label(WP_User $user): string {
        $issuer = $this->get_issuer();
        $identifier = $user->user_email ? $user->user_email : $user->user_login;
        return $issuer . ':' . $identifier;
    }

    private function get_qr_text(string $label, string $secret): string {
        $issuer = $this->get_issuer();
        $identifier = $label;
        $separator = strpos($label, ':');
        if ($separator !== false) {
            $issuer = substr($label, 0, $separator);
            $identifier = substr($label, $separator + 1);
        }
        return $this->tfa()->getQRCodeUrl($issuer, $identifier, $secret);
    }

    private function set_notice(int $user_id, string $class, string $message): void {
        set_transient(
            'shadowscan_mfa_notice_' . $user_id,
            array('class' => $class, 'message' => $message),
            30
        );
    }

    private function get_notice(int $user_id): ?array {
        $notice = get_transient('shadowscan_mfa_notice_' . $user_id);
        if (!is_array($notice)) {
            return null;
        }
        delete_transient('shadowscan_mfa_notice_' . $user_id);
        return $notice;
    }

    private function set_backup_codes_notice(int $user_id, array $codes): void {
        set_transient(self::BACKUP_CODES_NOTICE . $user_id, $codes, 300);
    }

    private function get_backup_codes_notice(int $user_id): array {
        $codes = get_transient(self::BACKUP_CODES_NOTICE . $user_id);
        if (!is_array($codes)) {
            return array();
        }
        delete_transient(self::BACKUP_CODES_NOTICE . $user_id);
        return $codes;
    }

    private function get_backup_codes(int $user_id): array {
        $codes = get_user_meta($user_id, self::USER_META_BACKUP_CODES, true);
        return is_array($codes) ? $codes : array();
    }

    private function store_backup_codes(int $user_id, array $codes): void {
        $hashed = array();
        foreach ($codes as $code) {
            $hashed[] = $this->hash_backup_code($code);
        }
        update_user_meta($user_id, self::USER_META_BACKUP_CODES, $hashed);
    }

    private function consume_backup_code(int $user_id, string $code): bool {
        $codes = $this->get_backup_codes($user_id);
        if (empty($codes)) {
            return false;
        }
        $needle = $this->hash_backup_code($code);
        foreach ($codes as $index => $hash) {
            if (hash_equals($hash, $needle)) {
                unset($codes[$index]);
                update_user_meta($user_id, self::USER_META_BACKUP_CODES, array_values($codes));
                return true;
            }
        }
        return false;
    }

    private function generate_backup_codes(int $count): array {
        $codes = array();
        for ($i = 0; $i < $count; $i++) {
            $codes[] = strtoupper(wp_generate_password(10, false, false));
        }
        return $codes;
    }

    private function hash_backup_code(string $code): string {
        return hash_hmac('sha256', strtoupper($code), wp_salt('secure_auth'));
    }

    private function set_user_secret(int $user_id, string $key, string $secret): void {
        $value = $this->encrypt_secret($secret);
        update_user_meta($user_id, $key, $value);
    }

    private function get_user_secret(int $user_id, string $key): string {
        $value = (string) get_user_meta($user_id, $key, true);
        if ($value === '') {
            return '';
        }
        return $this->decrypt_secret($value);
    }

    private function encrypt_secret(string $secret): string {
        if ($secret === '') {
            return '';
        }
        if (!function_exists('openssl_encrypt')) {
            shadowscan_log_message('[ShadowScan] OpenSSL missing, storing MFA secret in plain text.');
            return $secret;
        }
        $key = hash('sha256', wp_salt('auth'), true);
        $iv = random_bytes(12);
        $tag = '';
        $cipher = openssl_encrypt($secret, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
        if ($cipher === false) {
            shadowscan_log_message('[ShadowScan] MFA secret encryption failed.');
            return $secret;
        }
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
        return 'enc:' . base64_encode($iv . $tag . $cipher);
    }

    private function decrypt_secret(string $value): string {
        if ($value === '') {
            return '';
        }
        if (strpos($value, 'enc:') !== 0) {
            return $value;
        }
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
        $payload = base64_decode(substr($value, 4), true);
        if ($payload === false || strlen($payload) < 28) {
            return '';
        }
        if (!function_exists('openssl_decrypt')) {
            return '';
        }
        $iv = substr($payload, 0, 12);
        $tag = substr($payload, 12, 16);
        $cipher = substr($payload, 28);
        $key = hash('sha256', wp_salt('auth'), true);
        $secret = openssl_decrypt($cipher, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
        return $secret === false ? '' : $secret;
    }
}
