<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Emergency_Mode {
    private const OPTION_ENABLED = 'emergency_mode_enabled';
    private const OPTION_ENABLED_AT = 'emergency_mode_enabled_at';
    private const OPTION_REASON = 'emergency_mode_reason';
    private const AUTO_EXPIRE = 86400;

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_filter('xmlrpc_enabled', array($this, 'disable_xmlrpc'));
        add_action('admin_init', array($this, 'block_file_editors'));
        add_action('init', array($this, 'check_expiry'));
        add_action('shadowscan_emergency_mode_enabled', array($this, 'invalidate_admin_sessions'));
    }

    public function is_enabled(): bool {
        return (bool) ShadowScan_Storage::get(self::OPTION_ENABLED, false);
    }

    public function enable(string $reason = ''): void {
        ShadowScan_Storage::set(self::OPTION_ENABLED, true);
        ShadowScan_Storage::set(self::OPTION_ENABLED_AT, time());
        ShadowScan_Storage::set(self::OPTION_REASON, $reason);

        ShadowScan_Signal_Manager::emit(
            'EMERGENCY_MODE_ENABLED',
            'high',
            'Emergency containment mode enabled',
            array('reason' => $reason)
        );
        ShadowScan_Signal_Manager::emit(
            'EMERGENCY_MODE_ACTION_APPLIED',
            'medium',
            'Emergency actions applied',
            array('actions' => array('disable_xmlrpc', 'block_file_editors', 'invalidate_admin_sessions'))
        );

        do_action('shadowscan_emergency_mode_enabled');
        $this->schedule_heartbeat_boost();
        do_action('shadowscan_integrity_targeted_scan_now');
    }

    public function disable(): void {
        ShadowScan_Storage::set(self::OPTION_ENABLED, false);
        ShadowScan_Storage::set(self::OPTION_ENABLED_AT, 0);
        ShadowScan_Storage::set(self::OPTION_REASON, '');

        ShadowScan_Signal_Manager::emit(
            'EMERGENCY_MODE_DISABLED',
            'medium',
            'Emergency containment mode disabled',
            array()
        );
        $this->unschedule_heartbeat_boost();
    }

    public function get_status(): array {
        return array(
            'enabled' => $this->is_enabled(),
            'enabled_at' => (int) ShadowScan_Storage::get(self::OPTION_ENABLED_AT, 0),
            'reason' => (string) ShadowScan_Storage::get(self::OPTION_REASON, ''),
        );
    }

    public function disable_xmlrpc($enabled) {
        if ($this->is_enabled()) {
            return false;
        }
        return $enabled;
    }

    public function block_file_editors(): void {
        if (!$this->is_enabled()) {
            return;
        }
        if (!is_admin()) {
            return;
        }

        $screen = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if (strpos($screen, 'theme-editor.php') !== false || strpos($screen, 'plugin-editor.php') !== false) {
            wp_die(esc_html__('File editors are disabled during emergency containment.', 'shadowscan-security-link'));
        }
    }

    public function invalidate_admin_sessions(): void {
        $admins = get_users(array('role' => 'administrator', 'fields' => array('ID')));
        if (!is_array($admins)) {
            return;
        }
        foreach ($admins as $admin) {
            if (function_exists('WP_Session_Tokens')) {
                $tokens = WP_Session_Tokens::get_instance($admin->ID);
                $tokens->destroy_all();
            }
        }
    }

    public function check_expiry(): void {
        if (!$this->is_enabled()) {
            return;
        }
        $enabled_at = (int) ShadowScan_Storage::get(self::OPTION_ENABLED_AT, 0);
        if ($enabled_at > 0 && (time() - $enabled_at) > self::AUTO_EXPIRE) {
            $this->disable();
            ShadowScan_Signal_Manager::emit(
                'EMERGENCY_MODE_AUTO_EXPIRED',
                'medium',
                'Emergency mode auto-expired',
                array()
            );
        }
    }

    private function schedule_heartbeat_boost(): void {
        if (!wp_next_scheduled('shadowscan_emergency_heartbeat')) {
            wp_schedule_event(time() + 60, 'shadowscan_five_minutes', 'shadowscan_emergency_heartbeat');
        }
    }

    private function unschedule_heartbeat_boost(): void {
        $next = wp_next_scheduled('shadowscan_emergency_heartbeat');
        if ($next) {
            wp_unschedule_event($next, 'shadowscan_emergency_heartbeat');
        }
    }
}
