<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Config {
    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('init', array($this, 'check_config'), 30);
    }

    public function check_config(): void {
        $last_check = (int) ShadowScan_Storage::get('config_last_check', 0);
        if ($last_check > 0 && (time() - $last_check) < 6 * HOUR_IN_SECONDS) {
            return;
        }
        ShadowScan_Storage::set('config_last_check', time());

        if (defined('WP_DEBUG') && WP_DEBUG) {
            ShadowScan_Signal_Config::debug_enabled();
        }

        $paths = array(
            ABSPATH . 'wp-config.php',
            ABSPATH . 'wp-content',
        );

        foreach ($paths as $path) {
            if (!file_exists($path)) {
                continue;
            }
            $perms = substr(sprintf('%o', fileperms($path)), -4);
            if ($perms && ($perms === '0777' || $perms === '0775')) {
                ShadowScan_Signal_Config::weak_file_perms($this->relative_path($path), $perms);
            }
        }

        $this->check_updates();
    }

    private function check_updates(): void {
        if (!function_exists('get_core_updates')) {
            require_once ABSPATH . 'wp-admin/includes/update.php';
        }

        $updates = get_core_updates();
        if (is_array($updates) && !empty($updates)) {
            $latest = $updates[0]->current ?? null;
            if ($latest && version_compare(get_bloginfo('version'), $latest, '<')) {
                ShadowScan_Signal_Config::outdated_core(get_bloginfo('version'), $latest);
            }
        }

        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugin_updates = get_site_transient('update_plugins');
        $outdated = array();
        $abandoned = array();
        if ($plugin_updates && isset($plugin_updates->response) && is_array($plugin_updates->response)) {
            foreach ($plugin_updates->response as $file => $data) {
                $outdated[] = $file;
                if (isset($data->last_updated)) {
                    $last = strtotime((string) $data->last_updated);
                    if ($last && (time() - $last) > 365 * DAY_IN_SECONDS * 2) {
                        $abandoned[] = $file;
                    }
                }
            }
        }
        if (!empty($outdated)) {
            ShadowScan_Signal_Config::outdated_plugins($outdated);
        }
        foreach ($abandoned as $plugin) {
            ShadowScan_Signal_Config::abandoned_plugin($plugin);
        }
    }

    private function relative_path(string $path): string {
        $normalized = wp_normalize_path($path);
        $root = wp_normalize_path(ABSPATH);
        if (strpos($normalized, $root) === 0) {
            return ltrim(substr($normalized, strlen($root)), '/');
        }
        return $normalized;
    }
}
