<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Admin_Integrity {
    private const OPTION_DEDUPE = 'admin_integrity_dedupe';
    private const DEDUPE_WINDOW = 300;

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('user_register', array($this, 'handle_user_register'));
        add_action('set_user_role', array($this, 'handle_set_user_role'), 10, 3);
        add_action('added_user_meta', array($this, 'handle_caps_meta'), 10, 4);
        add_action('updated_user_meta', array($this, 'handle_caps_meta'), 10, 4);
        add_action('profile_update', array($this, 'handle_profile_update'), 10, 2);
        add_action('after_password_reset', array($this, 'handle_password_reset'), 10, 2);
    }

    public function handle_user_register(int $user_id): void {
        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }

        $this->emit_event('ADMIN_NEW_USER_CREATED', 'medium', 'New user created', array(
            'user_id' => $user_id,
        ));

        if (in_array('administrator', (array) $user->roles, true)) {
            $this->emit_event('ADMIN_NEW_ADMIN_CREATED', 'high', 'New administrator created', array(
                'user_id' => $user_id,
            ));
        }
    }

    public function handle_set_user_role(int $user_id, string $role, array $old_roles): void {
        if ($role === 'administrator') {
            $this->emit_event('ADMIN_ROLE_CHANGED', 'high', 'User role changed to administrator', array(
                'user_id' => $user_id,
                'old_roles' => $old_roles,
                'new_role' => $role,
            ));
        }
    }

    public function handle_caps_meta(int $meta_id, int $user_id, string $meta_key, $meta_value): void {
        if (strpos($meta_key, 'capabilities') === false) {
            return;
        }
        if (!$this->caps_include_admin($meta_value)) {
            return;
        }
            $this->emit_event('ADMIN_CAPS_CHANGED', 'high', 'Administrator capabilities updated', array(
                'user_id' => $user_id,
                // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Metadata label only, not a query argument.
                'meta_key' => $meta_key,
            ));
        }

    public function handle_profile_update(int $user_id, $old_user_data): void {
        if (!$old_user_data) {
            return;
        }
        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }
        if ((string) $old_user_data->user_email !== (string) $user->user_email) {
            $this->emit_event('ADMIN_EMAIL_CHANGED', 'medium', 'Admin email changed', array(
                'user_id' => $user_id,
            ));
        }
    }

    public function handle_password_reset($user, string $new_pass): void {
        if ($user instanceof WP_User) {
            $this->emit_event('ADMIN_PASSWORD_RESET_TRIGGERED', 'medium', 'Admin password reset triggered', array(
                'user_id' => $user->ID,
            ));
        }
    }

    private function emit_event(string $type, string $severity, string $summary, array $details): void {
        if (!$this->should_emit($type, (int) ($details['user_id'] ?? 0))) {
            return;
        }
        ShadowScan_Storage::set('admin_integrity_last_event', time());
        ShadowScan_Signal_Manager::emit($type, $severity, $summary, $details);
    }

    private function should_emit(string $type, int $user_id): bool {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }

        $key = $type . ':' . $user_id;
        $now = time();
        $last = isset($dedupe[$key]) ? (int) $dedupe[$key] : 0;
        if ($last > 0 && ($now - $last) < self::DEDUPE_WINDOW) {
            return false;
        }
        $dedupe[$key] = $now;
        ShadowScan_Storage::set_json(self::OPTION_DEDUPE, $dedupe);
        return true;
    }

    private function caps_include_admin($meta_value): bool {
        if (is_string($meta_value)) {
            $decoded = maybe_unserialize($meta_value);
            $meta_value = $decoded;
        }
        if (!is_array($meta_value)) {
            return false;
        }
        return !empty($meta_value['administrator']) || !empty($meta_value['manage_options']);
    }
}
