<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Guard_Access_Surface {
    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        add_filter('xmlrpc_enabled', array($this, 'maybe_disable_xmlrpc'));
        add_action('init', array($this, 'check_access_surface'), 20);
    }

    public function maybe_disable_xmlrpc($enabled) {
        if (!$this->manager->guard_actions_enabled()) {
            return $enabled;
        }

        $flag = $this->manager->is_flag_enabled('guard_access_xmlrpc', true);
        if (!$flag) {
            return $enabled;
        }

        if ($this->manager->is_flag_enabled('xmlrpc_enforce_disable', false)) {
            if (!apply_filters('shadowscan_allow_xmlrpc', false) && !$this->is_jetpack_active()) {
                return false;
            }
        }

        $profile = $this->manager->get_profile_config();
        if (!empty($profile['auto_disable_xmlrpc_on_abuse']) && $this->manager->is_flag_enabled('xmlrpc_disabled_due_to_abuse', false)) {
            ShadowScan_Signal_Manager::emit('ACCESS_XMLRPC_DISABLED', 'info', 'XML-RPC disabled due to anomalies', array());
            return false;
        }

        return $enabled;
    }

    public function check_access_surface(): void {
        if (!did_action('init')) {
            return;
        }

        $last_check = (int) ShadowScan_Storage::get('access_last_check', 0);
        if ($last_check > 0 && (time() - $last_check) < 6 * HOUR_IN_SECONDS) {
            return;
        }
        ShadowScan_Storage::set('access_last_check', time());

        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        $xmlrpc_enabled = apply_filters('xmlrpc_enabled', true);
        if ($xmlrpc_enabled) {
            ShadowScan_Signal_Manager::emit('ACCESS_XMLRPC_ENABLED', 'info', 'XML-RPC is enabled', array());
        }

        $rest_url = rest_url();
        if ($rest_url) {
            $response = wp_remote_get($rest_url, array('timeout' => 5));
            if (!is_wp_error($response)) {
                $code = wp_remote_retrieve_response_code($response);
                if ($code >= 200 && $code < 400) {
                    ShadowScan_Signal_Manager::emit('ACCESS_REST_EXPOSED', 'info', 'REST API is publicly accessible', array());
                }
            }
        }

        if ($this->manager->is_flag_enabled('guard_access_enumeration', false)) {
            if ($this->is_author_archive_public()) {
                ShadowScan_Signal_Manager::emit('ACCESS_ENUMERATION_RISK', 'warning', 'Author archives are accessible', array());
            }
        }
    }

    private function is_author_archive_public(): bool {
        $sample = get_users(array('number' => 1, 'fields' => array('ID')));
        if (empty($sample)) {
            return false;
        }
        $author_url = get_author_posts_url($sample[0]->ID);
        if (!$author_url) {
            return false;
        }
        $response = wp_remote_get($author_url, array('timeout' => 5));
        if (is_wp_error($response)) {
            return false;
        }
        $code = wp_remote_retrieve_response_code($response);
        return $code >= 200 && $code < 400;
    }

    private function is_jetpack_active(): bool {
        if (class_exists('Jetpack')) {
            return true;
        }
        if (!function_exists('is_plugin_active')) {
            if (defined('ABSPATH')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
        }
        return function_exists('is_plugin_active') && is_plugin_active('jetpack/jetpack.php');
    }
}
