<?php
if (!defined('ABSPATH')) {
    exit;
}

// Scope: WordPress-level access control only (users, roles, REST, XML-RPC, admin endpoints, file editing).
// This does NOT cover hosting panels, SSH/FTP, CDN dashboards, or email accounts.
class ShadowScan_Guard_Access_Control {
    private const OPTION_STATE = 'access_control_state';
    private const OPTION_DEDUPE = 'access_control_dedupe';
    private const ADMIN_THRESHOLD = 2;
    private const ADMIN_ABUSE_THRESHOLD = 10;
    private const ADMIN_ABUSE_WINDOW = 600;

    private ShadowScan_Guard_Manager $manager;

    public function __construct(ShadowScan_Guard_Manager $manager) {
        $this->manager = $manager;
    }

    public function register(): void {
        add_action('init', array($this, 'scan_periodic'), 20);
        add_action('rest_api_init', array($this, 'scan_rest_routes'), 20);
        add_filter('rest_authentication_errors', array($this, 'maybe_block_rest_users_settings'), 20);
        add_action('init', array($this, 'detect_admin_endpoint_abuse'), 1);
        add_action('set_user_role', array($this, 'detect_role_change'), 10, 3);
        add_action('add_user_role', array($this, 'detect_role_added'), 10, 2);
        add_action('profile_update', array($this, 'detect_profile_update'), 10, 2);
        add_action('user_register', array($this, 'detect_user_register'), 10, 1);
    }

    public static function emit_access_control_event(string $event, string $subcontrol, string $risk_level, string $summary, array $details = array(), string $severity = 'warning'): void {
        $payload = array_merge(
            array(
                'control_id' => 'OWASP-A01',
                'owasp_id' => 'A01',
                'category' => 'owasp',
                'owasp' => array(
                    'version' => '2025',
                    'id' => 'A01',
                    'name' => 'Broken Access Control',
                ),
                'control_name' => 'Broken Access Control',
                'subcontrol' => $subcontrol,
                'control_key' => $subcontrol,
                'risk_level' => $risk_level,
                'recommended_action' => self::recommended_action($subcontrol),
            ),
            $details
        );

        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        if (!is_array($state)) {
            $state = array();
        }
        if (!isset($state['subcontrol_events']) || !is_array($state['subcontrol_events'])) {
            $state['subcontrol_events'] = array();
        }
        $state['subcontrol_events'][$subcontrol] = array(
            'last_event_at' => time(),
            'risk_level' => $risk_level,
            'event' => $event,
            'summary' => $summary,
        );
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);

        if (class_exists('ShadowScan_Signal_Manager')) {
            ShadowScan_Signal_Manager::emit($event, $severity, $summary, $payload);
        }
    }

    public static function is_high_risk_role($role): bool {
        $roles = wp_roles();
        if (!$roles || !is_string($role)) {
            return false;
        }
        $role_obj = $roles->get_role($role);
        if (!$role_obj) {
            return false;
        }
        $caps = $role_obj->capabilities;
        $high_caps = self::high_risk_caps();
        foreach ($high_caps as $cap) {
            if (!empty($caps[$cap])) {
                return true;
            }
        }
        return false;
    }

    public static function user_has_high_risk_caps(int $user_id): bool {
        $high_caps = self::high_risk_caps();
        foreach ($high_caps as $cap) {
            if (user_can($user_id, $cap)) {
                return true;
            }
        }
        return false;
    }

    public static function high_risk_caps(): array {
        return array('manage_options', 'edit_users', 'install_plugins', 'delete_users');
    }

    public function get_state(): array {
        return ShadowScan_Storage::get_json(self::OPTION_STATE, array());
    }

    public function scan_periodic(): void {
        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $last_scan = isset($state['last_scan']) ? (int) $state['last_scan'] : 0;
        if ($last_scan > 0 && (time() - $last_scan) < 6 * HOUR_IN_SECONDS) {
            return;
        }

        $state['last_scan'] = time();
        $state['admin_capabilities'] = $this->scan_admin_capabilities();
        $state['file_mods'] = $this->scan_file_mods();
        $state['xmlrpc'] = $this->scan_xmlrpc();
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
    }

    public function scan_rest_routes(): void {
        if (!function_exists('rest_get_server')) {
            return;
        }
        $server = rest_get_server();
        if (!$server) {
            return;
        }
        $routes = $server->get_routes();
        if (!is_array($routes)) {
            return;
        }
        $insecure = array();
        foreach ($routes as $route => $handlers) {
            if (!is_array($handlers)) {
                continue;
            }
            foreach ($handlers as $handler) {
                if (!is_array($handler)) {
                    continue;
                }
                $perm = $handler['permission_callback'] ?? null;
                if ($perm === null) {
                    $insecure[] = array('route' => $route, 'reason' => 'missing_permission_callback');
                    continue;
                }
                if ($perm === '__return_true') {
                    $insecure[] = array('route' => $route, 'reason' => 'permission_callback_allows');
                }
            }
        }

        if (!empty($insecure)) {
            $this->emit_deduped(
                'access_control_insecure_rest_route',
                'rest_api',
                'medium',
                'Insecure REST routes detected',
                array('routes' => $insecure),
                'warning',
                86400
            );
        }

        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['rest_routes'] = array(
            'insecure_count' => count($insecure),
            'last_scan' => time(),
        );
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
    }

    public function maybe_block_rest_users_settings($result) {
        if (!empty($result)) {
            return $result;
        }
        if (!function_exists('rest_get_server')) {
            return $result;
        }
        if ($this->manager->is_flag_enabled('rest_block_users_settings', false) !== true) {
            return $result;
        }
        if (apply_filters('shadowscan_allow_rest_users_settings', false)) {
            return $result;
        }
        if (is_user_logged_in()) {
            return $result;
        }
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($request_uri === '') {
            return $result;
        }
        $path = wp_parse_url($request_uri, PHP_URL_PATH);
        if (!is_string($path)) {
            return $result;
        }
        if (stripos($path, '/wp-json/wp/v2/users') !== false || stripos($path, '/wp-json/wp/v2/settings') !== false) {
            $this->emit_deduped(
                'access_control_insecure_rest_route',
                'rest_api',
                'medium',
                'Blocked unauthenticated REST access',
                array('path' => $path),
                'warning',
                3600
            );
            return new WP_Error('shadowscan_rest_forbidden', __('REST access is restricted.', 'shadowscan-security-link'), array('status' => 401));
        }
        return $result;
    }

    public function detect_admin_endpoint_abuse(): void {
        if (is_user_logged_in()) {
            return;
        }
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($request_uri === '') {
            return;
        }
        $path = wp_parse_url($request_uri, PHP_URL_PATH);
        if (!is_string($path)) {
            return;
        }
        $targets = array('/wp-login.php', '/wp-admin/', '/wp-admin/admin-ajax.php');
        $match = false;
        foreach ($targets as $target) {
            if (stripos($path, $target) !== false) {
                $match = $target;
                break;
            }
        }
        if (!$match) {
            return;
        }

        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : 'unknown';
        $bucket = 'shadowscan_admin_abuse_' . md5($ip . $match);
        $count = (int) get_transient($bucket);
        $count++;
        set_transient($bucket, $count, self::ADMIN_ABUSE_WINDOW);

        if ($count === self::ADMIN_ABUSE_THRESHOLD || $count === (self::ADMIN_ABUSE_THRESHOLD * 2)) {
            $risk = $count >= self::ADMIN_ABUSE_THRESHOLD * 2 ? 'high' : 'medium';
            $severity = $risk === 'high' ? 'critical' : 'warning';
            $this->emit_deduped(
                'access_control_admin_endpoint_abuse',
                'admin_endpoints',
                $risk,
                'Repeated admin endpoint access detected',
                array('path' => $match, 'count' => $count),
                $severity,
                600
            );
        }

        $state = ShadowScan_Storage::get_json(self::OPTION_STATE, array());
        $state['admin_endpoint_abuse'] = array(
            'last_seen' => time(),
            'count' => $count,
            'path' => $match,
        );
        ShadowScan_Storage::set_json(self::OPTION_STATE, $state);
    }

    private function scan_admin_capabilities(): array {
        $admins = get_users(array('role' => 'administrator', 'fields' => array('ID', 'user_login')));
        $admin_count = is_array($admins) ? count($admins) : 0;
        $admin_usernames = array();
        $admin_username_flag = false;
        if (is_array($admins)) {
            foreach ($admins as $admin) {
                if (!empty($admin->user_login)) {
                    $admin_usernames[] = (string) $admin->user_login;
                    if (strtolower((string) $admin->user_login) === 'admin') {
                        $admin_username_flag = true;
                    }
                }
            }
        }

        $roles = wp_roles();
        $high_roles = array();
        if ($roles) {
            foreach ($roles->roles as $role_key => $role) {
                if (self::is_high_risk_role($role_key)) {
                    $high_roles[] = $role_key;
                }
            }
        }

        $risk_level = 'low';
        $severity = 'warning';
        if ($admin_count > self::ADMIN_THRESHOLD) {
            $risk_level = $admin_count > 5 ? 'high' : 'medium';
            $severity = $admin_count > 5 ? 'critical' : 'warning';
            $this->emit_deduped(
                'access_control_admin_capability_detected',
                'admin_capabilities',
                $risk_level,
                'Administrator count exceeds recommended threshold',
                array(
                    'admin_count' => $admin_count,
                    'threshold' => self::ADMIN_THRESHOLD,
                    'admin_usernames' => $admin_usernames,
                    'high_risk_roles' => $high_roles,
                ),
                $severity,
                86400
            );
        }

        if (!empty($high_roles)) {
            $this->emit_deduped(
                'access_control_admin_capability_detected',
                'admin_capabilities',
                'medium',
                'High-risk custom roles detected',
                array('high_risk_roles' => $high_roles),
                'warning',
                86400
            );
        }

        if ($admin_username_flag) {
            $this->emit_deduped(
                'access_control_admin_capability_detected',
                'admin_capabilities',
                'medium',
                'Administrator username "admin" detected',
                array('admin_usernames' => $admin_usernames),
                'warning',
                86400
            );
        }

        return array(
            'admin_count' => $admin_count,
            'admin_usernames' => $admin_usernames,
            'admin_username_flag' => $admin_username_flag,
            'high_risk_roles' => $high_roles,
        );
    }

    private function scan_file_mods(): array {
        $file_edit_allowed = !defined('DISALLOW_FILE_EDIT') || !DISALLOW_FILE_EDIT;
        $file_mods_allowed = !defined('DISALLOW_FILE_MODS') || !DISALLOW_FILE_MODS;

        if ($file_edit_allowed || $file_mods_allowed) {
            $this->emit_deduped(
                'access_control_file_mods_allowed',
                'file_mods',
                $file_edit_allowed ? 'medium' : 'low',
                'File editing or installs are allowed',
                array(
                    'file_edit_allowed' => $file_edit_allowed,
                    'file_mods_allowed' => $file_mods_allowed,
                ),
                'warning',
                86400
            );
        }

        return array(
            'file_edit_allowed' => $file_edit_allowed,
            'file_mods_allowed' => $file_mods_allowed,
        );
    }

    private function scan_xmlrpc(): array {
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        $xmlrpc_enabled = apply_filters('xmlrpc_enabled', true);
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        $methods = apply_filters('xmlrpc_methods', array());
        $multicall = is_array($methods) && array_key_exists('system.multicall', $methods);

        if ($xmlrpc_enabled) {
            $risk_level = $multicall ? 'medium' : 'low';
            $severity = $multicall ? 'warning' : 'info';
            $this->emit_deduped(
                'access_control_xmlrpc_enabled',
                'xmlrpc',
                $risk_level,
                'XML-RPC is enabled',
                array('multicall' => $multicall),
                $severity,
                86400
            );
        }

        return array(
            'enabled' => (bool) $xmlrpc_enabled,
            'multicall' => $multicall,
        );
    }

    private function emit_deduped(string $event, string $subcontrol, string $risk_level, string $summary, array $details, string $severity, int $ttl): void {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $now = time();
        $last = isset($dedupe[$event]) ? (int) $dedupe[$event] : 0;
        if ($last > 0 && ($now - $last) < $ttl) {
            return;
        }
        $dedupe[$event] = $now;
        ShadowScan_Storage::set_json(self::OPTION_DEDUPE, $dedupe);
        self::emit_access_control_event($event, $subcontrol, $risk_level, $summary, $details, $severity);
    }

    private static function recommended_action(string $subcontrol): string {
        $map = array(
            'admin_capabilities' => 'Review administrator roles in WordPress and reduce admin count where possible.',
            'privilege_escalation' => 'Review recent role changes in WordPress and confirm legitimacy.',
            'rest_api' => 'Review REST routes and ensure permission callbacks are enforced.',
            'xmlrpc' => 'Disable XML-RPC in ShadowScan unless required by trusted integrations.',
            'admin_endpoints' => 'Review login protection settings in ShadowScan and monitor access patterns.',
            'file_mods' => 'Set DISALLOW_FILE_EDIT and DISALLOW_FILE_MODS in wp-config.php if supported by your host.',
        );
        return $map[$subcontrol] ?? 'Open ShadowScan settings and review access controls.';
    }

    public function detect_role_change(int $user_id, string $new_role, array $old_roles): void {
        $new_roles = array($new_role);
        $this->maybe_emit_privilege_escalation($user_id, $old_roles, $new_roles, 'set_user_role');
    }

    public function detect_role_added(int $user_id, string $role): void {
        $user = get_userdata($user_id);
        $current_roles = $user ? $user->roles : array();
        $old_roles = array_diff($current_roles, array($role));
        $this->maybe_emit_privilege_escalation($user_id, $old_roles, $current_roles, 'add_user_role');
    }

    public function detect_profile_update(int $user_id, $old_user_data): void {
        $old_roles = is_object($old_user_data) && isset($old_user_data->roles) ? (array) $old_user_data->roles : array();
        $user = get_userdata($user_id);
        $new_roles = $user ? (array) $user->roles : array();
        $this->maybe_emit_privilege_escalation($user_id, $old_roles, $new_roles, 'profile_update');
    }

    public function detect_user_register(int $user_id): void {
        $user = get_userdata($user_id);
        $roles = $user ? (array) $user->roles : array();
        $this->maybe_emit_privilege_escalation($user_id, array(), $roles, 'user_register');
    }

    private function maybe_emit_privilege_escalation(int $user_id, array $old_roles, array $new_roles, string $context): void {
        $old_high = $this->roles_have_high_risk($old_roles);
        $new_high = $this->roles_have_high_risk($new_roles) || self::user_has_high_risk_caps($user_id);

        if ($new_high && !$old_high) {
            self::emit_access_control_event(
                'access_control_privilege_escalation',
                'privilege_escalation',
                'high',
                'Privilege escalation detected',
                array(
                    'target_user_id' => $user_id,
                    'old_roles' => array_values($old_roles),
                    'new_roles' => array_values($new_roles),
                    'context' => $context,
                ),
                'critical'
            );
        }
    }

    private function roles_have_high_risk(array $roles): bool {
        foreach ($roles as $role) {
            if (self::is_high_risk_role($role)) {
                return true;
            }
        }
        return false;
    }
}
