<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Security_Controls {
    private const OPTION_DEDUPE = 'security_control_dedupe';
    private const OPTION_UNKNOWN_KEYS = 'unknown_control_keys';

    public static function emit_status(
        string $control_key,
        string $status,
        bool $enforced,
        string $recommended_action,
        array $evidence,
        int $checked_at
    ): void {
        if (function_exists('shadowscan_normalize_control_key')) {
            $control_key = shadowscan_normalize_control_key($control_key);
        }
        if ($control_key === '') {
            return;
        }
        if (function_exists('shadowscan_is_known_control_key') && !shadowscan_is_known_control_key($control_key)) {
            if (function_exists('shadowscan_is_dev_env') && shadowscan_is_dev_env() && function_exists('shadowscan_log_message')) {
                self::record_unknown_control_key($control_key);
                shadowscan_log_message('[ShadowScan] Unknown control_key in emit_status: ' . $control_key);
            }
            return;
        }

        if (!class_exists('ShadowScan_Signal_Manager')) {
            return;
        }

        $status = strtolower($status);
        if (!in_array($status, array('ok', 'warn', 'critical', 'resolved'), true)) {
            $status = 'ok';
        }

        $fingerprint = hash(
            'sha256',
            wp_json_encode(array(
                'control_key' => $control_key,
                'status' => $status,
                'enforced' => $enforced,
                'evidence' => $evidence,
            ))
        );

        $dedupe = ShadowScan_Storage::get_json(self::OPTION_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }

        $last_status = $dedupe[$control_key]['status'] ?? '';
        $last_fingerprint = $dedupe[$control_key]['fingerprint'] ?? '';
        $last_sent = (int) ($dedupe[$control_key]['last_sent'] ?? 0);
        $should_send = $last_status !== $status
            || $last_fingerprint !== $fingerprint
            || (time() - $last_sent) > DAY_IN_SECONDS;

        if (!$should_send) {
            return;
        }

        $severity = 'info';
        if ($status === 'critical') {
            $severity = 'high';
        } elseif ($status === 'warn') {
            $severity = 'warning';
        }

        ShadowScan_Signal_Manager::emit(
            'security_control_status_changed',
            $severity,
            'Security control status updated',
            array(
                'category' => 'security_control',
                'control_key' => $control_key,
                'status' => $status,
                'enforced' => $enforced,
                'fingerprint' => $fingerprint,
                'recommended_action' => $recommended_action,
                'evidence' => $evidence,
                'last_checked_at' => gmdate('c', $checked_at),
            )
        );

        $dedupe[$control_key] = array(
            'status' => $status,
            'fingerprint' => $fingerprint,
            'last_sent' => time(),
        );
        ShadowScan_Storage::set_json(self::OPTION_DEDUPE, $dedupe);
    }

    private static function record_unknown_control_key(string $control_key): void {
        if (!class_exists('ShadowScan_Storage')) {
            return;
        }
        $unknown = ShadowScan_Storage::get_json(self::OPTION_UNKNOWN_KEYS, array());
        if (!is_array($unknown)) {
            $unknown = array();
        }
        $entry = $unknown[$control_key] ?? array('count' => 0, 'last_seen' => 0);
        $entry['count'] = (int) ($entry['count'] ?? 0) + 1;
        $entry['last_seen'] = time();
        $unknown[$control_key] = $entry;
        if (count($unknown) > 25) {
            uasort($unknown, static function ($a, $b) {
                return (int) ($b['last_seen'] ?? 0) <=> (int) ($a['last_seen'] ?? 0);
            });
            $unknown = array_slice($unknown, 0, 25, true);
        }
        ShadowScan_Storage::set_json(self::OPTION_UNKNOWN_KEYS, $unknown);
    }
}
