<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_Password_Policy {
    private const OPTION_MIN_LENGTH = 'password_min_length';
    private const OPTION_ENFORCE_MIN = 'password_enforce_min_length';
    private const OPTION_ENFORCE_BREACH = 'password_enforce_breached';
    private const OPTION_ENFORCE_ROLES = 'password_enforce_roles';
    private const OPTION_BREACH_PROVIDER = 'password_breach_provider';
    private const OPTION_BREACH_LAST_CHECK = 'password_breach_last_check';
    private const OPTION_REJECT_DEDUPE = 'password_reject_dedupe';
    private const OPTION_POLICY_ENABLED_AT = 'password_policy_enabled_at';
    private const OPTION_POLICY_NOTICE_SENT_AT = 'password_policy_notice_sent_at';
    private const OPTION_POLICY_EMAIL_ENABLED = 'password_policy_email_enabled';
    private const OPTION_LAST_CHANGE_AT = 'password_last_change_at';
    private const OPTION_FAILED_ATTEMPTS = 'password_failed_attempts';
    private const USER_META_LAST_CHANGE = '_shadowscan_password_changed_at';
    private const USER_META_SNOOZE_UNTIL = '_shadowscan_password_grace_snooze_until';
    private const FAILED_ATTEMPTS_CAP = 100;
    private const RESET_GRACE_DAYS = 14;
    private const SNOOZE_DAYS = 7;

    public static function register(): void {
        add_action('user_profile_update_errors', array(__CLASS__, 'validate_profile_update'), 10, 3);
        add_filter('validate_password_reset', array(__CLASS__, 'validate_password_reset'), 10, 2);
        add_action('after_password_reset', array(__CLASS__, 'record_password_reset'), 10, 2);
        add_action('profile_update', array(__CLASS__, 'record_profile_update'), 10, 2);
        add_filter('wp_authenticate_user', array(__CLASS__, 'enforce_password_reset_on_login'), 99, 2);
    }

    public static function get_status(): array {
        $checked_at = time();
        $min_length = self::get_min_length();
        $enforce_min = self::is_min_length_enforced();
        $enforce_breach = self::is_breach_enforced();
        $roles = self::get_enforced_roles();
        $provider = self::get_breach_provider();
        $last_change_at = self::get_last_password_change_at();
        $failed_attempts = self::get_failed_attempts();
        $policy_enabled_at = self::get_policy_enabled_at();
        $email_enabled = self::is_policy_email_enabled();

        $status = ($enforce_min || $enforce_breach) ? 'ok' : 'warn';
        $recommended = 'Password policy enforcement on change/reset with login enforcement after grace.';

        ShadowScan_Security_Controls::emit_status(
            'password_policy_enforced',
            $status,
            $enforce_min || $enforce_breach,
            $recommended,
            array(
                'min_length' => $min_length,
                'enforce_min_length' => $enforce_min,
                'enforce_breached' => $enforce_breach,
                'roles' => $roles,
                'breach_provider' => $provider,
                'enforced_on_change' => true,
                'enforced_on_login_after_grace' => $enforce_min || $enforce_breach,
                'minimum_length_enforced' => $enforce_min,
                'breached_password_check' => $enforce_breach ? 'enabled' : 'disabled',
                'last_password_change_at' => $last_change_at ?: null,
                'policy_enabled_at' => $policy_enabled_at ?: null,
                'reset_grace_days' => self::RESET_GRACE_DAYS,
                'email_notice_enabled' => $email_enabled,
                'failed_policy_attempts' => $failed_attempts,
            ),
            $checked_at
        );

        return array(
            'min_length' => $min_length,
            'enforce_min_length' => $enforce_min,
            'enforce_breached' => $enforce_breach,
            'roles' => $roles,
            'breach_provider' => $provider,
            'enforced_on_change' => true,
            'enforced_on_login_after_grace' => $enforce_min || $enforce_breach,
            'minimum_length_enforced' => $enforce_min,
            'breached_password_check' => $enforce_breach ? 'enabled' : 'disabled',
            'last_password_change_at' => $last_change_at ?: null,
            'policy_enabled_at' => $policy_enabled_at ?: null,
            'reset_grace_days' => self::RESET_GRACE_DAYS,
            'email_notice_enabled' => $email_enabled,
            'failed_policy_attempts' => $failed_attempts,
            'status' => $status,
            'checked_at' => $checked_at,
        );
    }

    public static function get_min_length(): int {
        $value = (int) ShadowScan_Storage::get(self::OPTION_MIN_LENGTH, 12);
        return max(8, $value);
    }

    public static function set_min_length(int $length): void {
        ShadowScan_Storage::set(self::OPTION_MIN_LENGTH, max(8, $length));
    }

    public static function is_min_length_enforced(): bool {
        return (bool) ShadowScan_Storage::get(self::OPTION_ENFORCE_MIN, true);
    }

    public static function set_enforce_min_length(bool $enabled): void {
        ShadowScan_Storage::set(self::OPTION_ENFORCE_MIN, $enabled);
    }

    public static function is_breach_enforced(): bool {
        return (bool) ShadowScan_Storage::get(self::OPTION_ENFORCE_BREACH, false);
    }

    public static function set_enforce_breached(bool $enabled): void {
        ShadowScan_Storage::set(self::OPTION_ENFORCE_BREACH, $enabled);
    }

    public static function get_enforced_roles(): array {
        $roles = ShadowScan_Storage::get_json(self::OPTION_ENFORCE_ROLES, array('administrator'));
        if (!is_array($roles) || empty($roles)) {
            return array('administrator');
        }
        return array_values(array_filter($roles, 'is_string'));
    }

    public static function set_enforced_roles(array $roles): void {
        $clean = array_values(array_filter($roles, 'is_string'));
        ShadowScan_Storage::set_json(self::OPTION_ENFORCE_ROLES, $clean);
    }

    public static function get_breach_provider(): string {
        $provider = (string) ShadowScan_Storage::get(self::OPTION_BREACH_PROVIDER, 'hibp');
        return $provider === 'hibp' ? 'hibp' : 'none';
    }

    public static function set_breach_provider(string $provider): void {
        $provider = $provider === 'hibp' ? 'hibp' : 'none';
        ShadowScan_Storage::set(self::OPTION_BREACH_PROVIDER, $provider);
    }

    public static function set_policy_enabled_at(int $timestamp): void {
        ShadowScan_Storage::set(self::OPTION_POLICY_ENABLED_AT, max(0, $timestamp));
    }

    public static function get_policy_enabled_at(): int {
        return (int) ShadowScan_Storage::get(self::OPTION_POLICY_ENABLED_AT, 0);
    }

    public static function set_policy_notice_sent_at(int $timestamp): void {
        ShadowScan_Storage::set(self::OPTION_POLICY_NOTICE_SENT_AT, max(0, $timestamp));
    }

    public static function get_policy_notice_sent_at(): int {
        return (int) ShadowScan_Storage::get(self::OPTION_POLICY_NOTICE_SENT_AT, 0);
    }

    public static function set_policy_email_enabled(bool $enabled): void {
        ShadowScan_Storage::set(self::OPTION_POLICY_EMAIL_ENABLED, $enabled);
    }

    public static function is_policy_email_enabled(): bool {
        return (bool) ShadowScan_Storage::get(self::OPTION_POLICY_EMAIL_ENABLED, true);
    }

    public static function get_policy_grace_deadline(): int {
        $enabled_at = self::get_policy_enabled_at();
        if ($enabled_at <= 0) {
            return 0;
        }
        return $enabled_at + (self::RESET_GRACE_DAYS * DAY_IN_SECONDS);
    }

    public static function is_within_grace(): bool {
        $deadline = self::get_policy_grace_deadline();
        if ($deadline <= 0) {
            return false;
        }
        return time() <= $deadline;
    }

    public static function user_needs_password_update(WP_User $user): bool {
        if (!self::is_policy_enforced()) {
            return false;
        }
        if (!self::is_user_in_scope($user)) {
            return false;
        }
        $enabled_at = self::get_policy_enabled_at();
        if ($enabled_at <= 0) {
            return false;
        }
        $last_change_at = self::get_user_last_password_change_at($user);
        if ($last_change_at <= 0) {
            return true;
        }
        return $last_change_at < $enabled_at;
    }

    public static function set_user_snooze_until(WP_User $user, int $timestamp): void {
        update_user_meta($user->ID, self::USER_META_SNOOZE_UNTIL, max(0, $timestamp));
    }

    public static function get_user_snooze_until(WP_User $user): int {
        return (int) get_user_meta($user->ID, self::USER_META_SNOOZE_UNTIL, true);
    }

    public static function is_user_snoozed(WP_User $user): bool {
        $until = self::get_user_snooze_until($user);
        return $until > time();
    }

    public static function get_snooze_days(): int {
        return self::SNOOZE_DAYS;
    }

    public static function validate_profile_update($errors, $update, $user): void {
        if (!$update || !($user instanceof WP_User)) {
            return;
        }
        if (empty($_POST['pass1'])) {
            return;
        }

        $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
        if ($nonce === '' || !wp_verify_nonce($nonce, 'update-user_' . $user->ID)) {
            return;
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be altered.
        $password = (string) wp_unslash($_POST['pass1']);
        self::validate_password($errors, $user, $password);
    }

    public static function validate_password_reset($errors, $user) {
        if (!($user instanceof WP_User)) {
            return $errors;
        }
        if (!isset($_POST['pass1'])) {
            return $errors;
        }

        $nonce = isset($_POST['wp-reset-password-nonce']) ? sanitize_text_field(wp_unslash($_POST['wp-reset-password-nonce'])) : '';
        if ($nonce === '' || !wp_verify_nonce($nonce, 'reset_password')) {
            return $errors;
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be altered.
        $password = (string) wp_unslash($_POST['pass1']);
        self::validate_password($errors, $user, $password);

        return $errors;
    }

    public static function record_password_reset(WP_User $user, string $new_pass): void {
        if (!self::is_user_in_scope($user)) {
            return;
        }
        self::set_user_last_password_change_at($user, time());
        self::set_last_password_change_at(time());
    }

    public static function record_profile_update(int $user_id, WP_User $old_user_data): void {
        $user = get_userdata($user_id);
        if (!($user instanceof WP_User)) {
            return;
        }
        if (!self::is_user_in_scope($user)) {
            return;
        }
        if ($old_user_data->user_pass !== $user->user_pass) {
            self::set_user_last_password_change_at($user, time());
            self::set_last_password_change_at(time());
        }
    }

    private static function validate_password(WP_Error $errors, WP_User $user, string $password): void {
        if (!self::is_user_in_scope($user)) {
            return;
        }

        $min_length = self::get_min_length();
        if (self::is_min_length_enforced() && strlen($password) < $min_length) {
            /* translators: %d: minimum password length. */
            $errors->add('shadowscan_password_length', sprintf(__('Password must be at least %d characters.', 'shadowscan-security-link'), $min_length));
            self::emit_rejection($user, 'min_length');
            return;
        }

        if (self::is_breach_enforced() && self::get_breach_provider() === 'hibp') {
            $breached = self::is_password_breached($password);
            if ($breached === null) {
                ShadowScan_Signal_Manager::emit(
                    'password_breach_check_failed',
                    'warning',
                    'Password breach check failed',
                    array(
                        'category' => 'security_control',
                        'control_key' => 'password_policy_enforced',
                        'status' => 'warn',
                        'recommended_action' => 'Retry breach checks when network is available.',
                        'last_checked_at' => gmdate('c'),
                    )
                );
            } elseif ($breached === true) {
                $errors->add('shadowscan_password_breached', __('Password does not meet policy requirements.', 'shadowscan-security-link'));
                self::emit_rejection($user, 'breached');
            }
        }
    }

    private static function is_user_in_scope(WP_User $user): bool {
        $roles = self::get_enforced_roles();
        return !empty(array_intersect($roles, $user->roles));
    }

    private static function set_user_last_password_change_at(WP_User $user, int $timestamp): void {
        update_user_meta($user->ID, self::USER_META_LAST_CHANGE, max(0, $timestamp));
    }

    private static function get_user_last_password_change_at(WP_User $user): int {
        return (int) get_user_meta($user->ID, self::USER_META_LAST_CHANGE, true);
    }

    private static function is_policy_enforced(): bool {
        return self::is_min_length_enforced() || self::is_breach_enforced();
    }

    private static function requires_reset_after_grace(WP_User $user): bool {
        if (!self::is_policy_enforced()) {
            return false;
        }
        if (!self::is_user_in_scope($user)) {
            return false;
        }
        $enabled_at = self::get_policy_enabled_at();
        if ($enabled_at <= 0) {
            return false;
        }
        $grace_seconds = self::RESET_GRACE_DAYS * DAY_IN_SECONDS;
        if (time() <= ($enabled_at + $grace_seconds)) {
            return false;
        }
        $last_change_at = self::get_user_last_password_change_at($user);
        if ($last_change_at <= 0) {
            return true;
        }
        return $last_change_at < $enabled_at;
    }

    public static function enforce_password_reset_on_login($user, $password) {
        if (!($user instanceof WP_User)) {
            return $user;
        }
        if (!self::requires_reset_after_grace($user)) {
            return $user;
        }
        $reset_url = wp_lostpassword_url();
        $message = sprintf(
            /* translators: %s: reset password URL. */
            __('Password update required. Please reset your password to continue: %s', 'shadowscan-security-link'),
            esc_url($reset_url)
        );
        return new WP_Error('shadowscan_password_reset_required', $message);
    }

    private static function is_password_breached(string $password): ?bool {
        $last_check = (int) ShadowScan_Storage::get(self::OPTION_BREACH_LAST_CHECK, 0);
        if ($last_check > 0 && (time() - $last_check) < 5) {
            return null;
        }
        ShadowScan_Storage::set(self::OPTION_BREACH_LAST_CHECK, time());

        $hash = strtoupper(sha1($password));
        $prefix = substr($hash, 0, 5);
        $suffix = substr($hash, 5);

        $cache_key = 'shadowscan_pwned_' . $prefix;
        $cached = get_transient($cache_key);
        if ($cached === false) {
            $response = wp_remote_get('https://api.pwnedpasswords.com/range/' . $prefix, array(
                'timeout' => 5,
                'headers' => array(
                    'User-Agent' => 'ShadowScan Password Policy',
                ),
            ));
            if (is_wp_error($response)) {
                return null;
            }
            $body = wp_remote_retrieve_body($response);
            if (!is_string($body) || $body === '') {
                return null;
            }
            $cached = $body;
            set_transient($cache_key, $cached, DAY_IN_SECONDS);
        }

        $lines = explode("\n", $cached);
        foreach ($lines as $line) {
            $parts = array_map('trim', explode(':', $line));
            if (count($parts) < 2) {
                continue;
            }
            if (strtoupper($parts[0]) === $suffix) {
                return true;
            }
        }
        return false;
    }

    private static function get_last_password_change_at(): int {
        return (int) ShadowScan_Storage::get(self::OPTION_LAST_CHANGE_AT, 0);
    }

    private static function set_last_password_change_at(int $timestamp): void {
        ShadowScan_Storage::set(self::OPTION_LAST_CHANGE_AT, max(0, $timestamp));
    }

    private static function get_failed_attempts(): int {
        return (int) ShadowScan_Storage::get(self::OPTION_FAILED_ATTEMPTS, 0);
    }

    private static function emit_rejection(WP_User $user, string $reason): void {
        $dedupe = ShadowScan_Storage::get_json(self::OPTION_REJECT_DEDUPE, array());
        if (!is_array($dedupe)) {
            $dedupe = array();
        }
        $key = $user->ID . ':' . $reason;
        $last = (int) ($dedupe[$key] ?? 0);
        if ($last > 0 && (time() - $last) < HOUR_IN_SECONDS) {
            return;
        }
        $dedupe[$key] = time();
        ShadowScan_Storage::set_json(self::OPTION_REJECT_DEDUPE, $dedupe);
        $failed_attempts = self::get_failed_attempts();
        if ($failed_attempts < self::FAILED_ATTEMPTS_CAP) {
            ShadowScan_Storage::set(self::OPTION_FAILED_ATTEMPTS, $failed_attempts + 1);
        }

        ShadowScan_Signal_Manager::emit(
            'password_policy_rejected',
            'warning',
            'Password rejected by policy',
            array(
                'category' => 'security_control',
                'control_key' => 'password_policy_enforced',
                'status' => 'warn',
                'recommended_action' => 'Use a 12+ character passphrase.',
                'evidence' => array(
                    'reason' => $reason,
                    'user_id' => $user->ID,
                    'role' => $user->roles[0] ?? '',
                ),
                'last_checked_at' => gmdate('c'),
            )
        );
    }
}
