<?php
if (!defined('ABSPATH')) {
    exit;
}

class ShadowScan_AutoUpdates {
    private const OPTION_ENABLED = 'autoupdate_plugins_enabled';
    private const OPTION_MODE = 'autoupdate_plugins_mode';
    private const OPTION_ALLOWLIST = 'autoupdate_plugins_allowlist';
    private const SITE_OPTION_PLUGINS = 'auto_update_' . 'plugins';
    private const OPTION_ALLOWLIST_HASH = 'autoupdate_plugins_allowlist_hash';
    private const OPTION_CORE_ENABLED = 'autoupdate_core_enabled';
    private const SITE_OPTION_CORE = 'auto_update_' . 'core';

    public static function register(): void {
        // Intentionally left blank. ShadowScan does not modify core update behavior.
    }

    public static function get_status(): array {
        $desired = self::get_desired_settings();
        $applied = self::get_site_autoupdate_plugins();
        $applied_count = count($applied);
        $desired_enabled = (bool) $desired['enabled'];
        $drift = ($desired_enabled && $applied_count === 0)
            || (!$desired_enabled && $applied_count > 0);
        if ($drift && $desired_enabled && $applied_count === 0) {
            $drift_reason = 'desired_enabled_applied_empty';
        } elseif ($drift && !$desired_enabled && $applied_count > 0) {
            $drift_reason = 'desired_disabled_applied_nonempty';
        } else {
            $drift_reason = null;
        }
        $checked_at = time();
        $core = self::get_core_status_data();

        if ($desired_enabled && $applied_count > 0) {
            $status = 'ok';
        } elseif (!$desired_enabled && $applied_count === 0) {
            $status = 'ok';
        } elseif ($desired_enabled && $applied_count === 0) {
            $status = 'warn';
        } else {
            $status = 'warn';
        }

        $recommended = $desired_enabled
            ? 'Review allowlist regularly to limit compatibility risk.'
            : 'Auto-updates are disabled by ShadowScan.';

        ShadowScan_Security_Controls::emit_status(
            'autoupdate_plugins_detect',
            $status,
            $desired_enabled,
            $recommended,
            array(
                'plugins_desired_enabled' => $desired_enabled,
                'plugins_desired_mode' => $desired['mode'],
                'plugins_desired_allowlist_count' => count($desired['allowlist']),
                'plugins_applied_count' => $applied_count,
                'plugins_drift' => $drift,
                'plugins_drift_reason' => $drift_reason,
                'core_desired_enabled' => $core['desired_enabled'],
                'core_applied_state' => $core['applied_state'],
                'core_drift' => $core['drift'],
                'core_drift_reason' => $core['drift_reason'],
            ),
            $checked_at
        );

        return array(
            'enabled' => $desired_enabled,
            'mode' => $desired['mode'],
            'allowlist' => $desired['allowlist'],
            'desired_enabled' => $desired_enabled,
            'desired_mode' => $desired['mode'],
            'desired_allowlist' => $desired['allowlist'],
            'applied_count' => $applied_count,
            'drift' => $drift,
            'plugins_desired_enabled' => $desired_enabled,
            'plugins_desired_mode' => $desired['mode'],
            'plugins_desired_allowlist_count' => count($desired['allowlist']),
            'plugins_applied_count' => $applied_count,
            'plugins_drift' => $drift,
            'plugins_drift_reason' => $drift_reason,
            'core_desired_enabled' => $core['desired_enabled'],
            'core_applied_state' => $core['applied_state'],
            'core_drift' => $core['drift'],
            'core_drift_reason' => $core['drift_reason'],
            'status' => $status,
            'checked_at' => $checked_at,
        );
    }

    public static function get_core_status(): array {
        $core = self::get_core_status_data();

        ShadowScan_Security_Controls::emit_status(
            'wp_core_autoupdate_detect',
            $core['status'],
            $core['desired_enabled'],
            $core['recommended'],
            array(
                'wp_version' => get_bloginfo('version'),
                'desired_enabled' => $core['desired_enabled'],
                'applied_level' => $core['applied_level'],
                'applied_state' => $core['applied_state'],
                'applied_source' => $core['applied_source'],
                'configurable' => $core['configurable'],
                'drift' => $core['drift'],
            ),
            time()
        );

        return $core;
    }

    public static function set_enabled(bool $enabled): void {
        $current = self::get_current_settings();
        self::sync_settings($enabled, $current['mode'], $current['allowlist'], 'policy');
    }

    public static function is_enabled(): bool {
        return !empty(self::get_site_autoupdate_plugins());
    }

    public static function set_mode(string $mode): void {
        $current = self::get_current_settings();
        self::sync_settings($current['enabled'], $mode, $current['allowlist'], 'policy');
    }

    public static function get_mode(): string {
        $current = self::get_current_settings();
        return $current['mode'];
    }

    public static function set_allowlist(array $allowlist): void {
        $current = self::get_current_settings();
        self::sync_settings($current['enabled'], $current['mode'], $allowlist, 'policy');
    }

    public static function get_allowlist(): array {
        $desired = self::get_desired_settings();
        return $desired['allowlist'];
    }

    public static function set_core_enabled(bool $enabled): void {
        self::update_option(self::OPTION_CORE_ENABLED, $enabled);
        self::reconcile_core_with_desired('manual');
    }

    public static function sync_settings(bool $enabled, string $mode, array $allowlist, string $reason = 'manual'): void {
        $mode = $mode === 'all' ? 'all' : 'allowlist';
        $clean = self::sanitize_allowlist($allowlist);

        self::update_option(self::OPTION_ENABLED, $enabled);
        self::update_option(self::OPTION_MODE, $mode);
        self::update_option(self::OPTION_ALLOWLIST, $clean);

        $previous_plugins = self::get_site_autoupdate_plugins();
        $next_plugins = self::build_plugin_list($enabled, $mode, $clean);

        self::update_site_option(self::SITE_OPTION_PLUGINS, $next_plugins);
        self::update_option(self::OPTION_ALLOWLIST_HASH, self::hash_list($next_plugins));

        self::log_policy_change($previous_plugins, $next_plugins, $reason, $mode, $enabled);
    }

    public static function sync_policy(array $payload, string $reason = 'policy'): void {
        $enabled = isset($payload['enabled']) ? (bool) $payload['enabled'] : false;
        $mode = isset($payload['mode']) ? (string) $payload['mode'] : 'allowlist';
        $allowlist = isset($payload['allowlist']) && is_array($payload['allowlist']) ? $payload['allowlist'] : array();
        if (isset($payload['reason'])) {
            $reason = (string) $payload['reason'];
        }
        self::sync_settings($enabled, $mode, $allowlist, $reason);
    }

    public static function reconcile_applied_with_desired(string $reason = 'reconcile'): void {
        $desired = self::get_desired_settings();
        $current = self::get_site_autoupdate_plugins();
        $next = self::build_plugin_list($desired['enabled'], $desired['mode'], $desired['allowlist']);
        sort($current);
        $next_sorted = $next;
        sort($next_sorted);
        if ($current !== $next_sorted) {
            self::update_site_option(self::SITE_OPTION_PLUGINS, $next);
            self::update_option(self::OPTION_ALLOWLIST_HASH, self::hash_list($next));
            self::log_policy_change($current, $next, $reason, $desired['mode'], $desired['enabled']);
        }
        self::reconcile_core_with_desired($reason);
    }

    private static function get_current_settings(): array {
        $desired = self::get_desired_settings();
        return array(
            'enabled' => $desired['enabled'],
            'mode' => $desired['mode'],
            'allowlist' => $desired['allowlist'],
        );
    }

    private static function get_desired_settings(): array {
        $enabled = (bool) self::get_option(self::OPTION_ENABLED, false);
        $mode = (string) self::get_option(self::OPTION_MODE, 'allowlist');
        $mode = $mode === 'all' ? 'all' : 'allowlist';
        $allowlist = self::get_option(self::OPTION_ALLOWLIST, array());
        if (!is_array($allowlist)) {
            $allowlist = array();
        }
        $allowlist = self::sanitize_allowlist($allowlist);
        return array(
            'enabled' => $enabled,
            'mode' => $mode,
            'allowlist' => $allowlist,
        );
    }

    private static function get_site_autoupdate_plugins(): array {
        $value = self::get_site_option(self::SITE_OPTION_PLUGINS, array());
        return is_array($value) ? array_values(array_filter($value, 'is_string')) : array();
    }

    private static function update_site_option(string $key, array $value): void {
        if (function_exists('update_site_option')) {
            update_site_option($key, $value);
        } else {
            update_option($key, $value);
        }
    }

    private static function get_site_option(string $key, $default) {
        return function_exists('get_site_option') ? get_site_option($key, $default) : get_option($key, $default);
    }

    private static function update_option(string $key, $value): void {
        update_option($key, $value);
    }

    private static function get_option(string $key, $default) {
        return get_option($key, $default);
    }

    private static function get_all_plugins_list(): array {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = function_exists('get_plugins') ? get_plugins() : array();
        $files = array_keys($plugins);
        return array_values(array_filter($files, 'is_string'));
    }

    private static function get_active_plugins_list(): array {
        $active = get_option('active_plugins', array());
        if (!is_array($active)) {
            $active = array();
        }
        $active = array_values(array_filter($active, 'is_string'));
        $installed = self::get_all_plugins_list();
        return array_values(array_intersect($active, $installed));
    }

    private static function is_all_plugins(array $allowlist): bool {
        $all = self::get_all_plugins_list();
        sort($all);
        $list = $allowlist;
        sort($list);
        return $all === $list && !empty($list);
    }

    private static function sanitize_allowlist(array $allowlist): array {
        $clean = array_values(array_unique(array_filter($allowlist, 'is_string')));
        $clean = array_map('wp_normalize_path', $clean);
        $clean = array_map(static function ($entry) {
            $entry = ltrim($entry, '/');
            return $entry;
        }, $clean);
        $clean = array_filter($clean, static function ($entry) {
            if ($entry === '') {
                return false;
            }
            if (strpos($entry, '..') !== false) {
                return false;
            }
            if (strpos($entry, 'wp-content') === 0) {
                return false;
            }
            return substr($entry, -4) === '.php';
        });
        $installed = self::get_all_plugins_list();
        return array_values(array_intersect($clean, $installed));
    }

    private static function build_plugin_list(bool $enabled, string $mode, array $allowlist): array {
        if (!$enabled) {
            return array();
        }
        if ($mode === 'all') {
            return self::get_active_plugins_list();
        }
        return $allowlist;
    }

    private static function hash_list(array $list): string {
        sort($list);
        return hash('sha256', implode("\n", $list));
    }

    private static function detect_core_applied(): array {
        if (defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED) {
            return array('level' => 'disabled', 'source' => 'constant');
        }
        if (defined('WP_AUTO_UPDATE_CORE')) {
            $value = WP_AUTO_UPDATE_CORE;
            if ($value === true || $value === 'major') {
                return array('level' => 'major', 'source' => 'constant');
            }
            if ($value === 'minor') {
                return array('level' => 'minor', 'source' => 'constant');
            }
            if ($value === false || $value === 'disabled') {
                return array('level' => 'disabled', 'source' => 'constant');
            }
        }
        $option = self::get_site_option(self::SITE_OPTION_CORE, null);
        if ($option !== null) {
            if ($option === false || $option === 'disabled') {
                return array('level' => 'disabled', 'source' => 'site_option');
            }
            if ($option === true || $option === 'major' || $option === 'all') {
                return array('level' => 'major', 'source' => 'site_option');
            }
            if ($option === 'minor') {
                return array('level' => 'minor', 'source' => 'site_option');
            }
            return array('level' => 'enabled', 'source' => 'site_option');
        }
        return array('level' => 'unknown', 'source' => 'unknown');
    }

    private static function is_core_configurable(): bool {
        if (defined('AUTOMATIC_UPDATER_DISABLED') || defined('WP_AUTO_UPDATE_CORE')) {
            return false;
        }
        return self::get_site_option(self::SITE_OPTION_CORE, null) !== null;
    }

    private static function get_core_status_data(): array {
        $desired_enabled = (bool) self::get_option(self::OPTION_CORE_ENABLED, false);
        $applied = self::detect_core_applied();
        $applied_level = $applied['level'];
        $applied_source = $applied['source'];
        $configurable = self::is_core_configurable();

        if ($applied_level === 'unknown' && !$configurable) {
            $applied_state = 'locked';
        } elseif ($applied_level === 'unknown') {
            $applied_state = 'unknown';
        } elseif ($applied_level === 'disabled') {
            $applied_state = 'disabled';
        } else {
            $applied_state = 'enabled';
        }

        $drift = null;
        $drift_reason = null;
        if ($applied_state === 'enabled' || $applied_state === 'disabled') {
            $applied_enabled = $applied_state === 'enabled';
            $drift = ($desired_enabled && !$applied_enabled) || (!$desired_enabled && $applied_enabled);
            if ($drift && $desired_enabled && !$applied_enabled) {
                $drift_reason = 'desired_enabled_applied_disabled';
            } elseif ($drift && !$desired_enabled && $applied_enabled) {
                $drift_reason = 'desired_disabled_applied_enabled';
            }
        }

        if ($desired_enabled && $applied_state === 'enabled') {
            $status = 'ok';
        } elseif (!$desired_enabled && $applied_state === 'disabled') {
            $status = 'ok';
        } elseif ($applied_state === 'unknown' || $applied_state === 'locked') {
            $status = $desired_enabled ? 'warn' : 'info';
        } else {
            $status = 'warn';
        }

        $recommended = $configurable
            ? 'Core updates can occasionally cause compatibility issues. If your host already manages core updates, leave this off.'
            : 'Core update settings are host-managed or locked by configuration.';

        return array(
            'desired_enabled' => $desired_enabled,
            'applied_level' => $applied_level,
            'applied_state' => $applied_state,
            'applied_source' => $applied_source,
            'configurable' => $configurable,
            'drift' => $drift,
            'drift_reason' => $drift_reason,
            'status' => $status,
            'recommended' => $recommended,
        );
    }

    private static function reconcile_core_with_desired(string $reason = 'reconcile'): void {
        $desired_enabled = (bool) self::get_option(self::OPTION_CORE_ENABLED, false);
        if (!self::is_core_configurable()) {
            return;
        }
        $before = self::get_site_option(self::SITE_OPTION_CORE, null);
        $after = $desired_enabled ? 'minor' : 'disabled';
        if ($before !== $after) {
            self::update_site_option(self::SITE_OPTION_CORE, $after);
            self::log_core_policy_change($before, $after, $reason, $desired_enabled);
        }
    }

    private static function log_policy_change(array $before_plugins, array $after_plugins, string $reason, string $mode, bool $desired_enabled): void {
        $added_plugins = array_values(array_diff($after_plugins, $before_plugins));
        $removed_plugins = array_values(array_diff($before_plugins, $after_plugins));

        if (empty($added_plugins) && empty($removed_plugins)) {
            return;
        }

        $sample_added = array_slice($added_plugins, 0, 10);
        $sample_removed = array_slice($removed_plugins, 0, 10);
        $hash = self::hash_list($after_plugins);
        $summary = sprintf(
            '[ShadowScan] Auto-update policy changed (reason=%s, mode=%s, desired_enabled=%s). plugins:+%d -%d applied=%d hash=%s added=%s removed=%s',
            $reason,
            $mode,
            $desired_enabled ? 'true' : 'false',
            count($added_plugins),
            count($removed_plugins),
            count($after_plugins),
            $hash,
            $sample_added ? implode(',', $sample_added) : '-',
            $sample_removed ? implode(',', $sample_removed) : '-'
        );

        if (function_exists('shadowscan_log_message')) {
            shadowscan_log_message($summary);
        }
    }

    private static function log_core_policy_change($before, $after, string $reason, bool $desired_enabled): void {
        $summary = sprintf(
            '[ShadowScan] Core auto-update policy changed (reason=%s, desired_enabled=%s). applied_before=%s applied_after=%s',
            $reason,
            $desired_enabled ? 'true' : 'false',
            is_scalar($before) ? (string) $before : 'unset',
            is_scalar($after) ? (string) $after : 'unset'
        );
        if (function_exists('shadowscan_log_message')) {
            shadowscan_log_message($summary);
        }
    }
}
