<?php
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WP_CLI')) {
    return;
}

class ShadowScan_CLI extends WP_CLI_Command {
    public static function register(): void {
        WP_CLI::add_command('shadowscan verify', array(__CLASS__, 'verify'));
        WP_CLI::add_command('shadowscan policy-refresh', array(__CLASS__, 'policy_refresh'));
    }

    public static function verify(array $args, array $assoc_args): void {
        $domain_id = isset($assoc_args['domain-id']) ? sanitize_text_field((string) $assoc_args['domain-id']) : '';
        $site_id_arg = isset($assoc_args['site-id']) ? sanitize_text_field((string) $assoc_args['site-id']) : '';
        $portal_jwt = isset($assoc_args['portal-jwt']) ? (string) $assoc_args['portal-jwt'] : '';
        if ($portal_jwt === '') {
            $portal_jwt = (string) getenv('SHADOWSCAN_PORTAL_JWT');
        }
        $run_policy_post = isset($assoc_args['write-policy']) && $assoc_args['write-policy'] !== false;
        $run_security_posture = isset($assoc_args['security-posture']) && $assoc_args['security-posture'] !== false;
        $run_mirror = isset($assoc_args['mirror-policy']) && $assoc_args['mirror-policy'] !== false;
        $dump_policy_diff = isset($assoc_args['dump-policy-diff']) && $assoc_args['dump-policy-diff'] !== false;

        $results = array();
        $site_id = $site_id_arg !== '' ? $site_id_arg : shadowscan_get_site_id();
        $token = shadowscan_get_site_token();

        $results[] = self::check('paired', shadowscan_is_paired(), 'Site is paired', 'Site is not paired');
        $results[] = self::check('site_id', $site_id !== '', 'site_id available', 'Missing site_id');
        $results[] = self::check('site_token', $token !== '', 'site token available', 'Missing site token');

        if ($site_id === '' || $token === '') {
            self::report($results);
            WP_CLI::error('Missing required pairing data.');
        }

        $policy_response = shadowscan_send_api_request('site-policy?site_id=' . rawurlencode($site_id), array(), 'GET', true);
        if (is_wp_error($policy_response)) {
            $results[] = self::check('site_policy_get', false, '', $policy_response->get_error_message());
        } else {
            $results[] = self::check('site_policy_get', true, 'site-policy GET ok', '');
            $signature = isset($policy_response['signature']) ? (string) $policy_response['signature'] : '';
            if ($signature !== '') {
                $payload = array(
                    'site_id' => isset($policy_response['site_id']) ? (string) $policy_response['site_id'] : $site_id,
                    'policy' => isset($policy_response['policy']) && is_array($policy_response['policy']) ? $policy_response['policy'] : array(),
                    'updated_at' => isset($policy_response['updated_at']) ? $policy_response['updated_at'] : null,
                    'updated_by' => isset($policy_response['updated_by']) ? $policy_response['updated_by'] : null,
                    'supportDebug' => isset($policy_response['supportDebug']) && is_array($policy_response['supportDebug']) ? $policy_response['supportDebug'] : array(),
                );
                $expected = self::compute_signature($payload, shadowscan_get_site_token());
                $results[] = self::check('policy_signature', hash_equals($expected, $signature), 'Signature match', 'Signature mismatch');
            }
        }

        if (class_exists('ShadowScan_Policy_State')) {
            $refreshed = ShadowScan_Policy_State::refresh_policy(true);
            $results[] = self::check('policy_refresh', is_array($refreshed), 'Policy refreshed', 'Policy refresh failed');
        }

        $heartbeat_result = shadowscan_send_heartbeat(true);
        $results[] = self::check('heartbeat', !is_wp_error($heartbeat_result), 'Heartbeat sent', is_wp_error($heartbeat_result) ? $heartbeat_result->get_error_message() : '');

        if ($policy_response && !is_wp_error($policy_response) && class_exists('ShadowScan_Policy_State')) {
            $current = ShadowScan_Policy_State::get_current_policy();
            $server_policy = is_array($policy_response['policy'] ?? null) ? $policy_response['policy'] : array();
            $local_normalized = self::normalize_policy_for_compare($current, $server_policy);
            $server_normalized = self::normalize_policy_for_compare($server_policy, $server_policy);
            $matches = self::canonicalize($local_normalized) === self::canonicalize($server_normalized);
            $results[] = self::check('policy_match', $matches, 'Cached policy matches server', 'Cached policy differs from server');
            if ($dump_policy_diff && !$matches) {
                $diffs = self::diff_policy($local_normalized, $server_normalized);
                if (empty($diffs)) {
                    WP_CLI::log('Policy diff: no field-level differences found (possible ordering/type mismatch).');
                } else {
                    WP_CLI::log('Policy diff (showing up to 50 fields):');
                    $count = 0;
                    foreach ($diffs as $line) {
                        WP_CLI::log(' - ' . $line);
                        $count++;
                        if ($count >= 50) {
                            WP_CLI::log(' - ...truncated');
                            break;
                        }
                    }
                }
            }
        }

        if ($portal_jwt !== '') {
            $portal_get = self::portal_request('site-policy?site_id=' . rawurlencode($site_id), 'GET', array(), $portal_jwt);
            if (is_wp_error($portal_get)) {
                $results[] = self::check('portal_policy_get', false, '', $portal_get->get_error_message());
            } else {
                $results[] = self::check('portal_policy_get', true, 'Portal policy GET ok', '');
            }

            if ($run_policy_post) {
                $payload = array(
                    'site_id' => $site_id,
                    'policy' => array(
                        'tier' => 'monitor',
                    ),
                );
                $portal_post = self::portal_request('site-policy', 'POST', $payload, $portal_jwt);
                if (is_wp_error($portal_post)) {
                    $results[] = self::check('portal_policy_post', false, '', $portal_post->get_error_message());
                } else {
                    $results[] = self::check('portal_policy_post', true, 'Portal policy POST ok', '');
                }
            }

            if ($run_security_posture) {
                $payload = array('domain_id' => $domain_id);
                if ($domain_id === '') {
                    $payload = array('site_id' => $site_id);
                }
                $posture = self::portal_request('security-posture', 'POST', $payload, $portal_jwt);
                if (is_wp_error($posture)) {
                    $results[] = self::check('portal_security_posture', false, '', $posture->get_error_message());
                } else {
                    $results[] = self::check('portal_security_posture', true, 'Security posture ok', '');
                }
            }

            if ($run_mirror && !is_wp_error($portal_get)) {
                $original = is_array($portal_get['policy'] ?? null) ? $portal_get['policy'] : array();
                $updated = $original;
                if (!isset($updated['security_controls']) || !is_array($updated['security_controls'])) {
                    $updated['security_controls'] = array();
                }
                if (!isset($updated['security_controls']['admin_geo_guard']) || !is_array($updated['security_controls']['admin_geo_guard'])) {
                    $updated['security_controls']['admin_geo_guard'] = array();
                }
                $current_window = isset($updated['security_controls']['admin_geo_guard']['learning_window_days'])
                    ? (int) $updated['security_controls']['admin_geo_guard']['learning_window_days']
                    : 30;
                $new_window = $current_window >= 31 ? 30 : 31;
                $updated['security_controls']['admin_geo_guard']['learning_window_days'] = $new_window;

                $mirror_payload = array(
                    'site_id' => $site_id,
                    'policy' => $updated,
                );
                $mirror_post = self::portal_request('site-policy', 'POST', $mirror_payload, $portal_jwt);
                if (is_wp_error($mirror_post)) {
                    $results[] = self::check('portal_policy_mirror_apply', false, '', $mirror_post->get_error_message());
                } else {
                    $results[] = self::check('portal_policy_mirror_apply', true, 'Mirror policy applied', '');
                }

                $after_apply = self::portal_request('site-policy?site_id=' . rawurlencode($site_id), 'GET', array(), $portal_jwt);
                if (is_wp_error($after_apply)) {
                    $results[] = self::check('portal_policy_mirror_read', false, '', $after_apply->get_error_message());
                } else {
                    $matches = self::canonicalize($after_apply['policy'] ?? array()) === self::canonicalize($updated);
                    $results[] = self::check('portal_policy_mirror_read', $matches, 'Mirror readback matches', 'Mirror readback mismatch');
                }

                $restore_payload = array(
                    'site_id' => $site_id,
                    'policy' => $original,
                );
                $restore_post = self::portal_request('site-policy', 'POST', $restore_payload, $portal_jwt);
                if (is_wp_error($restore_post)) {
                    $results[] = self::check('portal_policy_mirror_restore', false, '', $restore_post->get_error_message());
                } else {
                    $results[] = self::check('portal_policy_mirror_restore', true, 'Mirror policy restored', '');
                }
            }
        } elseif ($run_policy_post || $run_security_posture) {
            $results[] = self::check('portal_jwt', false, '', 'Missing SHADOWSCAN_PORTAL_JWT for portal checks');
        }

        if ($domain_id !== '') {
            $results[] = self::check('domain_id', true, 'Domain ID provided', '');
        }

        self::report($results);
    }

    private static function check(string $key, bool $ok, string $ok_message, string $fail_message): array {
        return array(
            'key' => $key,
            'ok' => $ok,
            'message' => $ok ? $ok_message : $fail_message,
        );
    }

    private static function report(array $results): void {
        foreach ($results as $result) {
            if ($result['ok']) {
                WP_CLI::success($result['message']);
            } else {
                WP_CLI::warning($result['message']);
            }
        }
    }

    private static function portal_request(string $path, string $method, array $body, string $jwt) {
        $url = rtrim(SHADOWSCAN_API_BASE, '/') . '/' . ltrim($path, '/');
        $headers = array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $jwt,
        );
        $args = array(
            'headers' => $headers,
            'timeout' => 20,
            'method' => $method,
        );
        if (strtoupper($method) !== 'GET') {
            $args['body'] = wp_json_encode($body);
        }
        $response = wp_remote_request($url, $args);
        if (is_wp_error($response)) {
            return $response;
        }
        $code = wp_remote_retrieve_response_code($response);
        $raw = wp_remote_retrieve_body($response);
        $decoded = json_decode($raw, true);
        if ($code < 200 || $code >= 300) {
            return new WP_Error('portal_error', 'Portal request failed', array('status' => $code, 'body' => $raw));
        }
        return $decoded ?? array();
    }

    private static function canonicalize($value): string {
        $sorted = self::sort_object($value);
        $json = wp_json_encode($sorted, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        return is_string($json) ? $json : '';
    }

    private static function sort_object($value) {
        if (is_array($value)) {
            $sorted = array();
            foreach ($value as $key => $item) {
                $sorted[$key] = self::sort_object($item);
            }
            if (self::is_assoc($sorted)) {
                ksort($sorted);
            }
            return $sorted;
        }
        return $value;
    }

    private static function is_assoc(array $array): bool {
        return array_keys($array) !== range(0, count($array) - 1);
    }

    private static function compute_signature(array $payload, string $key): string {
        if ($key === '') {
            return '';
        }
        $canonical = self::canonicalize(self::normalize_signature_payload($payload));
        $raw = hash_hmac('sha256', $canonical, $key, true);
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
        return base64_encode($raw);
    }

    private static function normalize_signature_payload(array $payload): array {
        if (isset($payload['policy']['owasp_enforcement']) && is_array($payload['policy']['owasp_enforcement'])) {
            if ($payload['policy']['owasp_enforcement'] === array()) {
                $payload['policy']['owasp_enforcement'] = (object) array();
            }
        }
        return $payload;
    }

    private static function normalize_policy_for_compare($local, array $server_shape) {
        if (!is_array($local)) {
            return $local;
        }
        $normalized = array();
        foreach ($server_shape as $key => $server_value) {
            if (array_key_exists($key, $local)) {
                if (is_array($server_value)) {
                    $normalized[$key] = self::normalize_policy_for_compare($local[$key], $server_value);
                } else {
                    $normalized[$key] = $local[$key];
                }
            } else {
                $normalized[$key] = $server_value;
            }
        }
        return $normalized;
    }

    private static function diff_policy($local, $server, string $prefix = ''): array {
        $diffs = array();
        $local_is_array = is_array($local);
        $server_is_array = is_array($server);

        if ($local_is_array && $server_is_array) {
            $local_assoc = self::is_assoc($local);
            $server_assoc = self::is_assoc($server);
            if ($local_assoc !== $server_assoc) {
                $diffs[] = sprintf('%s: type mismatch (local %s, server %s)', $prefix ?: '(root)', $local_assoc ? 'object' : 'list', $server_assoc ? 'object' : 'list');
                return $diffs;
            }
            if (!$local_assoc && !$server_assoc) {
                if (self::canonicalize($local) !== self::canonicalize($server)) {
                    $diffs[] = sprintf('%s: list differs (local %d items, server %d items)', $prefix ?: '(root)', count($local), count($server));
                }
                return $diffs;
            }
            $keys = array_unique(array_merge(array_keys($local), array_keys($server)));
            foreach ($keys as $key) {
                $path = $prefix === '' ? (string) $key : $prefix . '.' . $key;
                if (!array_key_exists($key, $local)) {
                    $diffs[] = sprintf('%s: missing locally (server=%s)', $path, self::stringify_value($server[$key] ?? null));
                    continue;
                }
                if (!array_key_exists($key, $server)) {
                    $diffs[] = sprintf('%s: missing on server (local=%s)', $path, self::stringify_value($local[$key] ?? null));
                    continue;
                }
                $diffs = array_merge($diffs, self::diff_policy($local[$key], $server[$key], $path));
            }
            return $diffs;
        }

        if ($local !== $server) {
            $diffs[] = sprintf('%s: local=%s server=%s', $prefix ?: '(root)', self::stringify_value($local), self::stringify_value($server));
        }
        return $diffs;
    }

    private static function stringify_value($value): string {
        if (is_bool($value)) {
            return $value ? 'true' : 'false';
        }
        if ($value === null) {
            return 'null';
        }
        if (is_scalar($value)) {
            return (string) $value;
        }
        $json = wp_json_encode($value, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        return is_string($json) ? $json : '[unserializable]';
    }

    public static function policy_refresh(array $args, array $assoc_args): void {
        $site_id_arg = isset($assoc_args['site-id']) ? sanitize_text_field((string) $assoc_args['site-id']) : '';
        $dump = isset($assoc_args['dump-signature']) && $assoc_args['dump-signature'] !== false;
        $site_id = $site_id_arg !== '' ? $site_id_arg : shadowscan_get_site_id();
        if ($site_id === '') {
            WP_CLI::error('Missing site_id.');
        }
        if (!shadowscan_is_paired()) {
            WP_CLI::error('Site is not paired.');
        }

        $response = shadowscan_send_api_request('site-policy?site_id=' . rawurlencode($site_id), array(), 'GET', true);
        if (is_wp_error($response)) {
            WP_CLI::warning('site-policy GET failed: ' . $response->get_error_message());
        } else {
            $signature = isset($response['signature']) ? (string) $response['signature'] : '';
            WP_CLI::success('site-policy GET ok');
            WP_CLI::log('Signature present: ' . ($signature !== '' ? 'yes' : 'no'));
            WP_CLI::log('Policy keys: ' . implode(', ', array_keys((array) ($response['policy'] ?? array()))));
            if ($signature !== '') {
                $payload = array(
                    'site_id' => isset($response['site_id']) ? (string) $response['site_id'] : $site_id,
                    'policy' => isset($response['policy']) && is_array($response['policy']) ? $response['policy'] : array(),
                    'updated_at' => isset($response['updated_at']) ? $response['updated_at'] : null,
                    'updated_by' => isset($response['updated_by']) ? $response['updated_by'] : null,
                    'supportDebug' => isset($response['supportDebug']) && is_array($response['supportDebug']) ? $response['supportDebug'] : array(),
                );
                $expected = self::compute_signature($payload, shadowscan_get_site_token());
                WP_CLI::log('Signature match: ' . (hash_equals($expected, $signature) ? 'yes' : 'no'));
                if ($dump) {
                    $canonical = self::canonicalize(self::normalize_signature_payload($payload));
                    WP_CLI::log('Canonical length: ' . strlen($canonical));
                    WP_CLI::log('Canonical sha256: ' . hash('sha256', $canonical));
                    WP_CLI::log('Signature (server): ' . $signature);
                    WP_CLI::log('Signature (local): ' . $expected);
                }
            }
        }

        if (class_exists('ShadowScan_Policy_State')) {
            $refreshed = ShadowScan_Policy_State::refresh_policy(true);
            if (is_array($refreshed)) {
                WP_CLI::success('Policy refreshed and cached.');
            } else {
                WP_CLI::warning('Policy refresh failed.');
            }
        }

        $last_error = get_option('shadowscan_last_error', array());
        if (is_array($last_error) && !empty($last_error)) {
            WP_CLI::log('Last error context: ' . (string) ($last_error['context'] ?? 'unknown'));
            WP_CLI::log('Last error message: ' . (string) ($last_error['message'] ?? 'unknown'));
            WP_CLI::log('Last error status: ' . (string) ($last_error['status'] ?? ''));
        }
    }
}

ShadowScan_CLI::register();
