/**
 * ShadowScan Admin Dashboard JavaScript
 *
 * Extracted from inline <script> blocks for better cacheability and maintainability.
 * Depends on jQuery (WordPress bundled) and shadowscanAdmin localized data.
 */

/* global jQuery, shadowscanAdmin */

// --- Block A: jQuery filter pills, accordion, keyboard nav ---
jQuery(function($) {
  'use strict';

  initFilterPills();
  initAccordions();
  initKeyboardNavigation();

  function initFilterPills() {
    $('.shadowscan-filter').on('click', function() {
      $(this).toggleClass('is-active');
      var filterType = $(this).data('filter');
      console.log('Filter toggled:', filterType);
    });
  }

  function initAccordions() {
    $('.shadowscan-accordion summary').on('click', function() {
      var $accordion = $(this).closest('.shadowscan-accordion');
      var isCurrentlyOpen = $accordion.prop('open');

      setTimeout(function() {
        if (!isCurrentlyOpen) {
          var rect = $accordion[0].getBoundingClientRect();
          var viewportHeight = $(window).height();

          if (rect.bottom > viewportHeight - 100) {
            $('html, body').animate({
              scrollTop: $accordion.offset().top - 20
            }, 300);
          }
        }
      }, 50);
    });
  }

  function initKeyboardNavigation() {
    $(document).on('keydown', '.shadowscan-filter', function(e) {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        $(this).trigger('click');
      }
    });

    $('.shadowscan-filter').attr('tabindex', '0');
  }

  $('.shadowscan-filter').attr({
    'role': 'button',
    'aria-pressed': 'false'
  });

  $('.shadowscan-filter').on('click', function() {
    var isActive = $(this).hasClass('is-active');
    $(this).attr('aria-pressed', isActive ? 'true' : 'false');
  });
});

// --- Block C: Coverage tabs, filters, search ---
(function() {
  var l10n = window.shadowscanAdmin || {};
  var focusAnchor = (l10n.coverageFocusAnchor || '').replace(/^#/, '');
  if (focusAnchor) {
    var focusTarget = document.getElementById(focusAnchor);
    if (focusTarget) {
      focusTarget.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  }

  var showAllBtn = document.querySelector('[data-shadowscan-show-coverage]');
  var fullCoverage = document.getElementById('shadowscan-coverage-all');
  if (showAllBtn && fullCoverage) {
    showAllBtn.addEventListener('click', function() {
      fullCoverage.scrollIntoView({ behavior: 'smooth', block: 'start' });
    });
  }
  function initCoverageTabs() {
    var tabButtons = document.querySelectorAll('.shadowscan-subtab');
    var filterButtons = document.querySelectorAll('.shadowscan-filter');
    var groups = document.querySelectorAll('.shadowscan-coverage-group');
    var cards = document.querySelectorAll('.shadowscan-coverage-card');
    var searchInput = document.getElementById('shadowscan-coverage-search');
    var searchWrap = document.getElementById('shadowscan-coverage-search-wrap');
    var clearBtn = searchWrap ? searchWrap.querySelector('.shadowscan-search-clear') : null;
    var countEl = document.getElementById('shadowscan-coverage-count');
    var toggleActionsBtn = document.querySelector('[data-shadowscan-toggle-actions]');
    var activeGroup = tabButtons.length ? tabButtons[0].getAttribute('data-group') : '';
    var activeStatus = 'all';
    var activeQuery = '';

    function applyFilters() {
      var visibleCount = 0;
      cards.forEach(function(card) {
        var group = card.closest('.shadowscan-coverage-group');
        var inGroup = !group || group.getAttribute('data-group') === activeGroup;
        if (!inGroup) {
          card.style.display = 'none';
          return;
        }
        var statusOk = activeStatus === 'all' || card.getAttribute('data-status') === activeStatus;
        var haystack = (card.getAttribute('data-search') || card.textContent || '').toLowerCase();
        var searchOk = activeQuery === '' || haystack.indexOf(activeQuery) !== -1;
        if (statusOk && searchOk) {
          card.style.display = '';
          visibleCount++;
        } else {
          card.style.display = 'none';
        }
      });
      if (countEl) {
        countEl.textContent = activeQuery ? (visibleCount + ' results') : '';
      }
      if (searchWrap) {
        searchWrap.classList.toggle('has-value', activeQuery !== '');
      }
    }

    function setActiveTab(groupId) {
      activeGroup = groupId;
      tabButtons.forEach(function(btn) {
        btn.classList.toggle('is-active', btn.getAttribute('data-group') === groupId);
      });
      groups.forEach(function(group) {
        group.style.display = group.getAttribute('data-group') === groupId ? '' : 'none';
      });
      applyFilters();
    }

    function setActiveFilter(status) {
      activeStatus = status;
      filterButtons.forEach(function(btn) {
        btn.classList.toggle('is-active', btn.getAttribute('data-status') === status);
      });
      applyFilters();
    }

    function setSearch(query) {
      activeQuery = (query || '').toLowerCase().trim();
      applyFilters();
    }

    function debounce(fn, wait) {
      var timeout;
      return function() {
        var args = arguments;
        if (timeout) {
          clearTimeout(timeout);
        }
        timeout = setTimeout(function() {
          fn.apply(null, args);
        }, wait);
      };
    }

    tabButtons.forEach(function(btn) {
      btn.addEventListener('click', function(evt) {
        evt.preventDefault();
        var groupId = btn.getAttribute('data-group') || '';
        if (groupId) setActiveTab(groupId);
      });
    });
    filterButtons.forEach(function(btn) {
      btn.addEventListener('click', function(evt) {
        evt.preventDefault();
        var status = btn.getAttribute('data-status') || 'all';
        setActiveFilter(status);
      });
    });
    if (searchInput) {
      searchInput.addEventListener('input', debounce(function() {
        setSearch(searchInput.value);
      }, 250));
      searchInput.addEventListener('keydown', function(evt) {
        if (evt.key === 'Escape') {
          searchInput.value = '';
          setSearch('');
        }
      });
    }
    if (clearBtn && searchInput) {
      clearBtn.addEventListener('click', function(evt) {
        evt.preventDefault();
        searchInput.value = '';
        setSearch('');
        searchInput.focus();
      });
    }
    if (toggleActionsBtn) {
      toggleActionsBtn.addEventListener('click', function(evt) {
        evt.preventDefault();
        var collapsed = toggleActionsBtn.getAttribute('data-expanded') !== '1';
        var hidden = document.querySelectorAll('.shadowscan-priority-action[data-hidden="1"]');
        hidden.forEach(function(item) {
          item.style.display = collapsed ? '' : 'none';
        });
        toggleActionsBtn.setAttribute('data-expanded', collapsed ? '1' : '0');
        var expandLabel = toggleActionsBtn.getAttribute('data-expand-label') || '';
        var collapseLabel = toggleActionsBtn.getAttribute('data-collapse-label') || '';
        toggleActionsBtn.textContent = collapsed ? collapseLabel : expandLabel;
      });
    }
    if (activeGroup) setActiveTab(activeGroup);
    setActiveFilter(activeStatus);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initCoverageTabs);
  } else {
    initCoverageTabs();
  }
})();

// --- Block D: Copy, modal, advanced toggle, collapse, geo chips ---
(function() {
  var l10n = window.shadowscanAdmin || {};

  var copyButtons = document.querySelectorAll('.shadowscan-copy-report');
  copyButtons.forEach(function(copyBtn) {
    copyBtn.addEventListener('click', function() {
      var target = copyBtn.getAttribute('data-copy-target');
      var textarea = target ? document.querySelector(target) : document.getElementById('shadowscan-status-report') || document.querySelector('.shadowscan-report');
      if (!textarea) return;
      textarea.select();
      textarea.setSelectionRange(0, textarea.value.length);
      var defaultLabel = copyBtn.getAttribute('data-copy-default') || copyBtn.textContent || '';
      var copiedLabel = copyBtn.getAttribute('data-copy-copied') || l10n.copiedLabel || 'Copied!';
      try {
        document.execCommand('copy');
        copyBtn.textContent = copiedLabel;
        setTimeout(function() {
          copyBtn.textContent = defaultLabel;
        }, 2000);
      } catch (err) {
        window.prompt(defaultLabel, textarea.value);
      }
    });
  });

  var modal = document.getElementById('shadowscan-advanced-modal');
  var openButtons = document.querySelectorAll('[data-shadowscan-open-advanced]');
  var openDetailButtons = document.querySelectorAll('[data-shadowscan-open-details]');
  var ackBox = document.getElementById('shadowscan-advanced-ack');
  var submitBtn = document.getElementById('shadowscan-advanced-submit');
  var focusInput = document.getElementById('shadowscan-coverage-target');
  var coverageContainer = document.querySelector('.shadowscan-coverage-container');
  var warnMissingAnchors = !!(l10n.isDevEnv);
  if (warnMissingAnchors) {
    console.log('ShadowScan admin CSS loaded');
  }
  function revealAdvancedControls() {
    var container = document.querySelector('.shadowscan-coverage-container');
    if (!container || !container.classList.contains('shadowscan-hide-advanced')) {
      return;
    }
    container.classList.remove('shadowscan-hide-advanced');
    var toggle = document.querySelector('[data-shadowscan-advanced-toggle]');
    if (toggle) {
      toggle.setAttribute('aria-pressed', 'true');
      var labelHide = toggle.getAttribute('data-label-hide') || '';
      if (labelHide) {
        toggle.textContent = labelHide;
      }
    }
    try {
      window.localStorage.setItem('shadowscan_show_advanced_controls', '1');
    } catch (e) {
      // Ignore storage errors.
    }
  }
  function isAdvancedVisible() {
    return !coverageContainer || !coverageContainer.classList.contains('shadowscan-hide-advanced');
  }
  if (openButtons.length) {
    openButtons.forEach(function(openBtn) {
      openBtn.addEventListener('click', function(event) {
        var targetId = openBtn.getAttribute('data-shadowscan-target') || '';
        var fallback = openBtn.getAttribute('data-shadowscan-fallback') || '';
        var target = targetId ? document.getElementById(targetId) : null;
        if (target) {
          event.preventDefault();
          if (target.getAttribute('data-shadowscan-advanced') === '1') {
            revealAdvancedControls();
          }
          var details = target.closest('details');
          if (details && !details.open) {
            details.open = true;
          }
          target.scrollIntoView({ behavior: 'smooth', block: 'start' });
          return;
        }
        if (warnMissingAnchors && targetId) {
          console.warn('ShadowScan: missing configuration anchor: ' + targetId);
        }
        if (isAdvancedVisible()) {
          if (fallback) {
            window.open(fallback, '_blank', 'noopener');
          }
          return;
        }
        if (!modal) {
          if (fallback) {
            window.open(fallback, '_blank', 'noopener');
          }
          return;
        }
        event.preventDefault();
        if (focusInput) {
          focusInput.value = targetId;
        }
        modal.classList.add('is-open');
        modal.setAttribute('aria-hidden', 'false');
      });
    });
  }
  if (openDetailButtons.length) {
    openDetailButtons.forEach(function(detailBtn) {
      detailBtn.addEventListener('click', function(event) {
        var detailId = detailBtn.getAttribute('data-shadowscan-details') || '';
        var details = detailId ? document.getElementById(detailId) : null;
        if (!details) {
          if (warnMissingAnchors && detailId) {
            console.warn('ShadowScan: missing details anchor: ' + detailId);
          }
          return;
        }
        event.preventDefault();
        var targetDetails = details.tagName === 'DETAILS' ? details : details.closest('details');
        if (targetDetails && !targetDetails.open) {
          targetDetails.open = true;
        }
        (targetDetails || details).scrollIntoView({ behavior: 'smooth', block: 'start' });
      });
    });
  }
  if (ackBox && submitBtn) {
    ackBox.addEventListener('change', function() {
      submitBtn.disabled = !ackBox.checked;
    });
  }
  var advancedToggle = document.querySelector('[data-shadowscan-advanced-toggle]');
  var advancedStorageKey = 'shadowscan_show_advanced_controls';
  function setAdvancedVisibility(show) {
    if (!advancedToggle) return;
    if (coverageContainer) {
      coverageContainer.classList.toggle('shadowscan-hide-advanced', !show);
    }
    advancedToggle.setAttribute('aria-pressed', show ? 'true' : 'false');
    var labelShow = advancedToggle.getAttribute('data-label-show') || '';
    var labelHide = advancedToggle.getAttribute('data-label-hide') || '';
    advancedToggle.textContent = show ? labelHide : labelShow;
    try {
      window.localStorage.setItem(advancedStorageKey, show ? '1' : '0');
    } catch (e) {
      // Ignore storage errors (private mode).
    }
  }
  if (advancedToggle) {
    var stored = null;
    try {
      stored = window.localStorage.getItem(advancedStorageKey);
    } catch (e) {
      stored = null;
    }
    if (stored === '1' || stored === '0') {
      setAdvancedVisibility(stored === '1');
    } else {
      var defaultVisible = advancedToggle.getAttribute('data-default-visible') === '1';
      var isVisible = coverageContainer
        ? !coverageContainer.classList.contains('shadowscan-hide-advanced')
        : defaultVisible;
      setAdvancedVisibility(isVisible);
    }
    advancedToggle.addEventListener('click', function() {
      var hidden = coverageContainer ? coverageContainer.classList.contains('shadowscan-hide-advanced') : advancedToggle.getAttribute('aria-pressed') === 'true';
      setAdvancedVisibility(hidden);
    });
  }

  if (warnMissingAnchors) {
    var hasCoverage = !!coverageContainer;
    var hasCoverageHeader = coverageContainer ? !!coverageContainer.querySelector('.shadowscan-coverage-header') : false;
    var hasCoverageAll = !!document.getElementById('shadowscan-coverage-all');

    if (!hasCoverage || !hasCoverageHeader) {
      console.warn('Coverage UI DOM missing — possible CSS regression. Found:', {
        coverageContainer: hasCoverage,
        coverageHeader: hasCoverageHeader,
        coverageAll: hasCoverageAll
      });
    } else {
      console.log('Coverage UI loaded successfully');
    }
  }

  (function() {
    var collapseTargets = document.querySelectorAll('[data-shadowscan-collapse="1"]');
    if (!collapseTargets.length) return;
    var setCollapse = function() {
      var shouldCollapse = window.innerWidth < 1100;
      collapseTargets.forEach(function(node) {
        if (!node) return;
        if (shouldCollapse) {
          node.removeAttribute('open');
        } else {
          node.setAttribute('open', 'open');
        }
      });
    };
    setCollapse();
    window.addEventListener('resize', function() {
      setCollapse();
    });
  })();

  (function() {
    var form = document.querySelector('[data-geo-config-form]');
    if (!form) return;
    var chipTargets = {
      countries: document.querySelector('[data-geo-chips-target="countries"]'),
      ips: document.querySelector('[data-geo-chips-target="ips"]')
    };
    var inputs = {
      countries: document.querySelector('[data-geo-chips="countries"]'),
      ips: document.querySelector('[data-geo-chips="ips"]')
    };
    var saveStatus = document.querySelector('[data-geo-save-status]');
    var saveButton = document.querySelector('[data-geo-save-button]');
    var saveBar = document.querySelector('[data-geo-save-bar]');
    var cidrSupported = inputs.ips ? inputs.ips.getAttribute('data-cidr-supported') === '1' : false;
    var initial = form ? new FormData(form) : null;
    var lastSavedKey = 'shadowscan_geo_saved_at';

    function parseValues(value) {
      return (value || '')
        .split(/[\s,]+/)
        .map(function(item) { return item.trim(); })
        .filter(Boolean);
    }

    function isValidCountry(code) {
      return /^[A-Za-z]{2}$/.test(code);
    }

    function isValidIPv4(value) {
      var parts = value.split('.');
      if (parts.length !== 4) return false;
      return parts.every(function(part) {
        if (!/^\d+$/.test(part)) return false;
        var num = parseInt(part, 10);
        return num >= 0 && num <= 255;
      });
    }

    function isValidIPv6(value) {
      return /^[0-9a-fA-F:]+$/.test(value) && value.indexOf(':') !== -1;
    }

    function isValidIpOrCidr(value) {
      if (value.indexOf('/') !== -1) {
        if (!cidrSupported) return false;
        var parts = value.split('/');
        if (parts.length !== 2) return false;
        var ip = parts[0];
        var prefix = parts[1];
        if (!/^\d+$/.test(prefix)) return false;
        var prefixNum = parseInt(prefix, 10);
        if (isValidIPv4(ip)) return prefixNum >= 0 && prefixNum <= 32;
        if (isValidIPv6(ip)) return prefixNum >= 0 && prefixNum <= 128;
        return false;
      }
      return isValidIPv4(value) || isValidIPv6(value);
    }

    function renderChips(kind) {
      var target = chipTargets[kind];
      var input = inputs[kind];
      if (!target || !input) return;
      target.innerHTML = '';
      var values = parseValues(input.value);
      values.forEach(function(value) {
        var chip = document.createElement('span');
        chip.className = 'shadowscan-chip';
        var valid = true;
        if (kind === 'countries') {
          valid = isValidCountry(value);
        } else if (kind === 'ips') {
          valid = isValidIpOrCidr(value);
        }
        if (!valid) chip.classList.add('is-invalid');
        chip.textContent = value.toUpperCase();
        target.appendChild(chip);
      });
    }

    function updateDirtyState() {
      if (!initial) return;
      var current = new FormData(form);
      var dirty = false;
      current.forEach(function(value, key) {
        var init = initial.getAll(key).join(',');
        var now = current.getAll(key).join(',');
        if (init !== now) dirty = true;
      });
      if (saveStatus) {
        if (dirty) {
          saveStatus.textContent = 'Unsaved changes';
        } else {
          var storedVal = null;
          try { storedVal = window.localStorage.getItem(lastSavedKey); } catch (e) { storedVal = null; }
          if (storedVal) {
            var date = new Date(parseInt(storedVal, 10));
            if (!Number.isNaN(date.getTime())) {
              saveStatus.textContent = 'Saved ' + date.toLocaleString();
              return;
            }
          }
          saveStatus.textContent = 'No changes yet.';
        }
      }
    }

    if (inputs.countries) {
      inputs.countries.addEventListener('input', function() {
        renderChips('countries');
        updateDirtyState();
      });
    }
    if (inputs.ips) {
      inputs.ips.addEventListener('input', function() {
        renderChips('ips');
        updateDirtyState();
      });
    }
    form.addEventListener('change', updateDirtyState);
    form.addEventListener('submit', function() {
      if (saveButton) saveButton.disabled = true;
      try {
        window.localStorage.setItem(lastSavedKey, String(Date.now()));
      } catch (e) {
        // ignore
      }
    });
    renderChips('countries');
    renderChips('ips');
    updateDirtyState();
    if (saveBar) {
      saveBar.style.position = 'sticky';
    }
  })();
  if (modal) {
    modal.querySelectorAll('[data-shadowscan-modal-close]').forEach(function(closeBtn) {
      closeBtn.addEventListener('click', function() {
        modal.classList.remove('is-open');

        if (document.activeElement && modal.contains(document.activeElement)) {
          var safeElement = document.querySelector('body') || document.documentElement;
          safeElement.focus();
          document.activeElement.blur();
        }

        setTimeout(function() {
          modal.setAttribute('aria-hidden', 'true');
        }, 10);
        if (ackBox) {
          ackBox.checked = false;
        }
        if (submitBtn) {
          submitBtn.disabled = true;
        }
      });
    });
  }
})();
