<?php
/**
 * Plugin Name: Shabbat Blocker
 * Plugin URI: https://autopeak.cloud/shabbat-blocker/
 * Author URI: https://autopeak.cloud
 * Description: Blocks the site on Shabbat and Jewish holidays based on the selected city, with local times display.
 * Version: 1.8.0
 * Author: AUTOPEAK
 * License: GPL2
 * Text Domain: shabbat-blocker
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

if ( ! defined( 'SHABBAT_BLOCKER_FREE_LOADED' ) ) {
    define( 'SHABBAT_BLOCKER_FREE_LOADED', true );
}

/* ------------------------------------------------
 * Freemius bootstrap – SAFE (single source of truth)
 * ----------------------------------------------*/
if ( function_exists( 'sb_fs' ) ) {
    $___fs = sb_fs();
    if ( is_object( $___fs ) ) {
        $___fs->set_basename( false, __FILE__ );
    }
    unset( $___fs );
} else {
    function sb_fs() {
        static $sb_fs = null;
        if ( isset( $sb_fs ) && $sb_fs ) return $sb_fs;

        $base = dirname( __FILE__ );
        $candidates = array(
            $base . '/freemius/start.php',
            $base . '/vendor/freemius/start.php',
        );
        $sdk = null;
        foreach ( $candidates as $p ) { if ( is_readable( $p ) ) { $sdk = $p; break; } }
        if ( ! $sdk ) return null;

        require_once $sdk;

        $sb_fs = fs_dynamic_init( array(
            'id'              => 21425,
            'slug'            => 'shabbat-blocker',
            'type'            => 'plugin',
            'public_key'      => 'pk_d5aad3fe62f139f8672c9925ff7cb',
            'is_premium'      => false,
            'is_premium_only' => false,
            'has_paid_plans'  => true,
            'menu'            => array(
                'slug'   => 'shabbat-blocker-settings',
                'parent' => array( 'slug' => 'options-general.php' ),
            ),
            'parallel_activation' => array(
    'enabled' => true,
    'premium_version_basename' => 'premium-slug/file-name.php',
),
        ) );

        return $sb_fs;
    }

    $___fs2 = sb_fs();
    if ( is_object( $___fs2 ) ) {
        $___fs2->set_basename( false, __FILE__ );
        do_action( 'sb_fs_loaded' );
    }
    unset( $___fs2 );
}

/** =========================
 *  Premium flag (enabled by PRO via filter)
 * ========================== */
/**
 * החינמי תמיד מחזיר false; ה-PRO משנה ל-true ע"י:
 * add_filter('sbk/is_premium','__return_true')
 */
function sbk_is_premium() {
    // בדיקה לפי Freemius
    $can_pro = (
        function_exists( 'sb_fs' )
        && sb_fs()
        && method_exists( sb_fs(), 'can_use_premium_code__premium_only' )
    )
        ? sb_fs()->can_use_premium_code__premium_only()
        : false;

    // עדיין מאפשרת לאדון/קוד אחר להתערב אם תרצי
    return (bool) apply_filters( 'sbk/is_premium', $can_pro );
}

/** =========================
 *  HELPERS
 * ========================== */

/**
 * שולח כותרות 503 תקינות + Retry-After עד לזמן הפתיחה הבא.
 * @param DateTime $reopen_local
 */
function sbk_send_maintenance_headers( DateTime $reopen_local ) {
    nocache_headers();

    if ( function_exists( 'http_response_code' ) ) {
        $cur = http_response_code();
        if ( $cur === 200 || $cur === null ) {
            http_response_code( 503 );
        }
    }

    $reopen_utc = ( clone $reopen_local )->setTimezone( new DateTimeZone( 'UTC' ) );
    header( 'Retry-After: ' . $reopen_utc->format( 'D, d M Y H:i:s' ) . ' GMT' );

    header( 'Cache-Control: no-store, no-cache, must-revalidate, max-age=0, private' );
    header( 'Pragma: no-cache' );
    header( 'Expires: 0' );
}

function shabbat_blocker_fetch_json( $url ) {
    $resp = wp_remote_get( $url, array( 'timeout' => 10 ) );
    if ( is_wp_error( $resp ) ) return null;
    $code = (int) wp_remote_retrieve_response_code( $resp );
    if ( 200 !== $code ) return null;
    $body = wp_remote_retrieve_body( $resp );
    $data = json_decode( $body, true );
    return is_array( $data ) ? $data : null;
}

function shabbat_blocker_valid_date( $s ) {
    $d = DateTime::createFromFormat( 'Y-m-d', $s );
    return $d && $d->format( 'Y-m-d' ) === $s;
}
function shabbat_blocker_valid_time( $s ) {
    return (bool) preg_match( '/^(2[0-3]|[01]\d):[0-5]\d$/', $s );
}
function shabbat_blocker_get_param( $key ) {
    return isset( $_GET[ $key ] ) ? sanitize_text_field( wp_unslash( $_GET[ $key ] ) ) : null;
}

/** =========================
 *  SAFE HTML
 * ========================== */
function shabbat_blocker_allowed_html() {
    $common = array( 'class' => true, 'id' => true, 'style' => true );
    return array(
        'html'  => array('lang' => true, 'dir' => true),
        'head'  => array(),
        'meta'  => array('charset' => true, 'name' => true, 'content' => true),
        'title' => array(),
        'link'  => array('href'=>true,'rel'=>true,'as'=>true,'type'=>true,'media'=>true,'crossorigin'=>true,'integrity'=>true),
        'body'  => array('class'=>true,'id'=>true,'style'=>true,'dir'=>true),
        'div'   => $common + array('dir'=>true,'lang'=>true),
        'section'=>$common,'header'=>$common,'footer'=>$common,
        'h1'=>$common,'h2'=>$common,'h3'=>$common,'h4'=>$common,'h5'=>$common,'h6'=>$common,
        'p'=>$common,'span'=>$common,'strong'=>$common,'em'=>$common,
        'ul'=>$common,'ol'=>$common,'li'=>$common,
        'br'=>array(),'hr'=>array('class'=>true,'style'=>true),
        'a'=>$common+array('href'=>true,'target'=>true,'rel'=>true),
        'img'=>$common+array('src'=>true,'alt'=>true,'width'=>true,'height'=>true,'loading'=>true,'decoding'=>true),
        'svg'=>$common+array('xmlns'=>true,'viewBox'=>true,'fill'=>true,'stroke'=>true,'width'=>true,'height'=>true,'role'=>true,'aria-hidden'=>true,'focusable'=>true),
        'path'=>array('d'=>true,'fill'=>true,'stroke'=>true,'stroke-width'=>true,'fill-rule'=>true,'clip-rule'=>true,'class'=>true,'style'=>true),
        'g'=>$common,'defs'=>$common,'linearGradient'=>array('id'=>true,'x1'=>true,'y1'=>true,'x2'=>true,'y2'=>true,'gradientUnits'=>true),
        'stop'=>array('offset'=>true,'stop-color'=>true,'stop-opacity'=>true),
        'style'=>array('type'=>true,'media'=>true),
        'script'=>array('type'=>true,'id'=>true),
    );
}

/** =========================
 *  BLOCK PAGE RENDER
 * ========================== */
function shabbat_blocker_render_page( $geonameid, $tzid, $candleTime, $havdalahTime ) {
    if ( $havdalahTime instanceof DateTime ) {
        sbk_send_maintenance_headers( $havdalahTime );
    } else {
        $fallback_open = ( new DateTime( 'now', new DateTimeZone( $tzid ?: 'Asia/Jerusalem' ) ) )->modify( '+6 hours' );
        sbk_send_maintenance_headers( $fallback_open );
    }

    $templatePath = plugin_dir_path( __FILE__ ) . 'shabbat-blocker-template.html';
    $html         = file_exists( $templatePath ) ? file_get_contents( $templatePath ) : null;

    $effectiveTz = $tzid ?: 'Asia/Jerusalem';

    $debugDateRaw = shabbat_blocker_get_param( 'debug_date' );
    $debugTimeRaw = shabbat_blocker_get_param( 'debug_time' );

    if ( $debugDateRaw && shabbat_blocker_valid_date( $debugDateRaw ) ) {
        $debugTime = ( $debugTimeRaw && shabbat_blocker_valid_time( $debugTimeRaw ) ) ? $debugTimeRaw : '12:00';
        $now       = new DateTime( "{$debugDateRaw} {$debugTime}", new DateTimeZone( $effectiveTz ) );
    } else {
        $now = new DateTime( 'now', new DateTimeZone( $effectiveTz ) );
    }

    $gy = (int) $now->format( 'Y' );
    $gm = (int) $now->format( 'n' );
    $gd = (int) $now->format( 'j' );

    $cities = [
        ['id'=>3435910,'name'=>'בואנוס איירס','name_en'=>'Buenos Aires','tz'=>'America/Argentina/Buenos_Aires'],
        ['id'=>2950159,'name'=>'ברלין','name_en'=>'Berlin','tz'=>'Europe/Berlin'],
        ['id'=>292223,'name'=>'דובאי','name_en'=>'Dubai','tz'=>'Asia/Dubai'],
        ['id'=>1850147,'name'=>'טוקיו','name_en'=>'Tokyo','tz'=>'Asia/Tokyo'],
        ['id'=>281184,'name'=>'ירושלים','name_en'=>'Jerusalem','tz'=>'Asia/Jerusalem'],
        ['id'=>2643743,'name'=>'לונדון','name_en'=>'London','tz'=>'Europe/London'],
        ['id'=>5368361,'name'=>'לוס אנג׳לס','name_en'=>'Los Angeles','tz'=>'America/Los_Angeles'],
        ['id'=>5101798,'name'=>'לייקווד','name_en'=>'Lakewood','tz'=>'America/New_York'],
        ['id'=>5125771,'name'=>'מונסי','name_en'=>'Monsey','tz'=>'America/New_York'],
        ['id'=>5128581,'name'=>'ניו יורק','name_en'=>'New York','tz'=>'America/New_York'],
        ['id'=>2147714,'name'=>'סידני','name_en'=>'Sydney','tz'=>'Australia/Sydney'],
        ['id'=>4887398,'name'=>'שיקגו','name_en'=>'Chicago','tz'=>'America/Chicago'],
    ];

   $hasGeo = !empty($geonameid);

// ברירת מחדל: קובעים שפה לפי ה-TZ האפקטיבי
$isIsrael  = ($effectiveTz === 'Asia/Jerusalem');
    $cityLabel = 'עיר לא ידועה';
if ( $hasGeo ) {
    foreach ( $cities as $city ) {
        if ( (string) $city['id'] === (string) $geonameid ) {
            $isIsrael  = ( $city['tz'] === 'Asia/Jerusalem' );
            $cityLabel = $isIsrael ? $city['name'] : $city['name_en'];
            break;
        }
    }
}



    if ( $isIsrael ) {
        $cityTitle     = ( 'עיר לא ידועה' === $cityLabel ) ? 'זמני שבת/חג' : 'זמני שבת/חג – ' . $cityLabel;
        $candleLabel   = 'כניסת שבת/חג';
        $havdalahLabel = 'צאת שבת/חג';
        $mainTitle     = 'האתר אינו זמין בשבת/חג';
        $subtitle      = 'נשמח לראותך בצאת השבת/חג';
        $sectionTitle  = 'זמני שבת/חג בערים נוספות';
    } else {
        $cityTitle     = ( 'עיר לא ידועה' === $cityLabel ) ? 'Shabbat/Holiday Times' : 'Shabbat/Holiday Times – ' . $cityLabel;
        $candleLabel   = 'Candle Lighting';
        $havdalahLabel = 'Havdalah';
        $mainTitle     = 'Site is unavailable on Shabbat or Jewish Holidays';
        $subtitle      = 'We’ll be happy to see you after Shabbat or the holiday';
        $sectionTitle  = 'Times in Other Cities';
    }

    $candleFormatted   = ( clone $candleTime )->setTimezone( new DateTimeZone( $effectiveTz ) )->format( 'H:i' );
    $havdalahFormatted = ( clone $havdalahTime )->setTimezone( new DateTimeZone( $effectiveTz ) )->format( 'H:i' );

    $otherCitiesHtml = '';
    foreach ( $cities as $city ) {
        if ( (string) $city['id'] === (string) $geonameid ) { continue; }

        $url  = "https://www.hebcal.com/shabbat?cfg=json&geonameid={$city['id']}&m=50&gy={$gy}&gm={$gm}&gd={$gd}";
        $data = shabbat_blocker_fetch_json( $url );

        $candle = '-';
        $havdalah = '-';
        if ( $data && isset( $data['items'] ) ) {
            foreach ( $data['items'] as $item ) {
                if ( ( $item['category'] ?? '' ) === 'candles'  && '-' === $candle )  { $candle   = $item['date']; }
                if ( ( $item['category'] ?? '' ) === 'havdalah' && '-' === $havdalah ){ $havdalah = $item['date']; }
            }
        }

        $tz = new DateTimeZone( $city['tz'] );
        $candleTimeStr   = ( '-' !== $candle   ) ? ( new DateTime( $candle   ) )->setTimezone( $tz )->format( 'H:i' ) : '-';
        $havdalahTimeStr = ( '-' !== $havdalah ) ? ( new DateTime( $havdalah ) )->setTimezone( $tz )->format( 'H:i' ) : '-';

        $cityNameSafe     = esc_html( ( $isIsrael ? $city['name'] : $city['name_en'] ) );
        $otherCitiesHtml .= '<li><strong>' . $cityNameSafe . '</strong><br>'
            . esc_html( $candleLabel ) . ': <strong>' . esc_html( $candleTimeStr ) . '</strong><br>'
            . esc_html( $havdalahLabel ) . ': <strong>' . esc_html( $havdalahTimeStr ) . '</strong></li>';
    }

    if ( $html ) {
        $base = plugin_dir_url( __FILE__ );
        $html = preg_replace( '#\b(src|href)=["\'](?!https?://|data:|/)([^"\']+)#i', '$1="' . esc_url( $base ) . '$2', $html );

        $replacements = [
            '{{MAIN_TITLE}}'         => esc_html( $mainTitle ),
            '{{SUBTITLE}}'           => esc_html( $subtitle ),
            '{{CANDLE_LABEL}}'       => esc_html( $candleLabel ),
            '{{HAVDALAH_LABEL}}'     => esc_html( $havdalahLabel ),
            '{{OTHER_CITIES_TITLE}}' => esc_html( $sectionTitle ),
            '{{CANDLE_TIME}}'        => esc_html( $candleFormatted ),
            '{{HAVDALAH_TIME}}'      => esc_html( $havdalahFormatted ),
            '{{CITY_TITLE}}'         => esc_html( $cityTitle ),
            '{{OTHER_CITIES_HTML}}'  => $otherCitiesHtml,
        ];
        foreach ( $replacements as $k => $v ) { $html = str_replace( $k, $v, $html ); }

        $favicon_links = '';
        if ( function_exists( 'get_site_icon_url' ) && get_site_icon_url() ) {
            $favicon_links .= '<link rel="icon" href="' . esc_url( get_site_icon_url(32) ) . '" sizes="32x32" />' . "\n";
            $favicon_links .= '<link rel="icon" href="' . esc_url( get_site_icon_url(192) ) . '" sizes="192x192" />' . "\n";
            $favicon_links .= '<link rel="apple-touch-icon" href="' . esc_url( get_site_icon_url(180) ) . '">' . "\n";
            $favicon_links .= '<meta name="msapplication-TileImage" content="' . esc_url( get_site_icon_url(270) ) . '">' . "\n";
        } else {
            $favicon_links .= '<link rel="icon" href="' . esc_url( home_url( '/favicon.ico' ) ) . '">' . "\n";
        }
        $html = preg_replace( '/<head(\s*)>/', '<head$1>' . $favicon_links, $html, 1 );

        echo wp_kses( $html, shabbat_blocker_allowed_html() );
    } else {
        echo '<h1 style="text-align:center">' . esc_html( $mainTitle ) . '</h1>';
        printf(
            "<p style='text-align:center'>%s: <strong>%s</strong> | %s: <strong>%s</strong></p>",
            esc_html( $candleLabel ),
            esc_html( $candleFormatted ),
            esc_html( $havdalahLabel ),
            esc_html( $havdalahFormatted )
        );
    }
}

/** =========================
 *  MAIN INTERCEPTOR (FREE core + extension point for PRO)
 * ========================== */
add_action( 'template_redirect', 'shabbat_blocker_intercept' );
function shabbat_blocker_intercept() {
    $dbgDateRaw = shabbat_blocker_get_param( 'debug_date' );
    $dbgTimeRaw = shabbat_blocker_get_param( 'debug_time' );
    $dbgDate    = ( $dbgDateRaw && shabbat_blocker_valid_date( $dbgDateRaw ) ) ? $dbgDateRaw : null;
    $dbgTime    = ( $dbgTimeRaw && shabbat_blocker_valid_time( $dbgTimeRaw ) ) ? $dbgTimeRaw : '12:00';

    $cityTzMap = [
        '3435910'=>'America/Argentina/Buenos_Aires',
        '2950159'=>'Europe/Berlin',
        '292223' =>'Asia/Dubai',
        '1850147'=>'Asia/Tokyo',
        '281184' =>'Asia/Jerusalem',
        '2643743'=>'Europe/London',
        '5368361'=>'America/Los_Angeles',
        '5101798'=>'America/New_York',
        '5125771'=>'America/New_York',
        '5128581'=>'America/New_York',
        '2147714'=>'Australia/Sydney',
        '4887398'=>'America/Chicago',
    ];

    $geonameid = get_option( 'shabbat_blocker_geonameid', '281184' );
    $tzid      = $cityTzMap[ (string) $geonameid ] ?? 'Asia/Jerusalem';

    $tzid = $tzid ?: 'Asia/Jerusalem';
    if ( $dbgDate ) {
        $now = new DateTime( "{$dbgDate} {$dbgTime}", new DateTimeZone( $tzid ) );
    } else {
        $now = new DateTime( 'now', new DateTimeZone( $tzid ) );
    }

    $today     = clone $now;
    $yesterday = ( clone $now )->modify( '-1 day' );

    $gy  = (int) $today->format( 'Y' );
    $gm  = (int) $today->format( 'n' );
    $gd  = (int) $today->format( 'j' );

    $gyY = (int) $yesterday->format( 'Y' );
    $gmY = (int) $yesterday->format( 'n' );
    $gdY = (int) $yesterday->format( 'j' );

    $geoVar      = $geonameid ?: get_option( 'shabbat_blocker_geonameid', '281184' );
    $apiUrlToday = "https://www.hebcal.com/shabbat?cfg=json&geonameid={$geoVar}&m=50&gy={$gy}&gm={$gm}&gd={$gd}";
    $apiUrlYest  = "https://www.hebcal.com/shabbat?cfg=json&geonameid={$geoVar}&m=50&gy={$gyY}&gm={$gmY}&gd={$gdY}";

    $dataToday = shabbat_blocker_fetch_json( $apiUrlToday );
    $dataYest  = shabbat_blocker_fetch_json( $apiUrlYest );
    if ( ! $dataToday && ! $dataYest ) {
        header( 'X-Shabbat-Decision: open' );
        header( 'X-Shabbat-Now: ' . $now->format( 'Y-m-d\TH:i:s T' ) );
        header( 'X-Shabbat-TZ: ' . $tzid );
        return;
    }

    $apiTz = null;
    if ( ! empty( $dataToday['location']['tzid'] ) ) {
        $apiTz = $dataToday['location']['tzid'];
    } elseif ( ! empty( $dataYest['location']['tzid'] ) ) {
        $apiTz = $dataYest['location']['tzid'];
    }
    if ( $apiTz ) {
        $tzid = $apiTz;
        $now->setTimezone( new DateTimeZone( $tzid ) );
    }

    $items = [];
    if ( $dataYest  && ! empty( $dataYest['items'] ) )  { $items = array_merge( $items, $dataYest['items'] ); }
    if ( $dataToday && ! empty( $dataToday['items'] ) ) { $items = array_merge( $items, $dataToday['items'] ); }
    if ( ! $items ) {
        header( 'X-Shabbat-Decision: open' );
        header( 'X-Shabbat-Now: ' . $now->format( 'Y-m-d\TH:i:s T' ) );
        header( 'X-Shabbat-TZ: ' . $tzid );
        return;
    }

    usort( $items, function( $a, $b ) { return strcmp( $a['date'], $b['date'] ); } );

    $candleTime = null;
    foreach ( $items as $item ) {
        if ( empty( $item['category'] ) || 'candles' !== $item['category'] || empty( $item['date'] ) ) { continue; }
        $t = new DateTime( $item['date'] );
        $t->setTimezone( new DateTimeZone( $tzid ) );
        if ( $t <= $now && ( ! $candleTime || $t > $candleTime ) ) {
            $candleTime = clone $t;
        }
    }

    $havdalahTime = null;
    if ( $candleTime ) {
        foreach ( $items as $item ) {
            if ( empty( $item['category'] ) || 'havdalah' !== $item['category'] || empty( $item['date'] ) ) { continue; }
            $t = new DateTime( $item['date'] );
            $t->setTimezone( new DateTimeZone( $tzid ) );
            if ( $t > $candleTime && ( ! $havdalahTime || $t < $havdalahTime ) ) {
                $havdalahTime = clone $t;
            }
        }
        if ( ! $havdalahTime ) {
            foreach ( $items as $item ) {
                if ( empty( $item['category'] ) || 'havdalah' !== $item['category'] || empty( $item['date'] ) ) { continue; }
                $t = new DateTime( $item['date'] );
                $t->setTimezone( new DateTimeZone( $tzid ) );
                if ( $t >= $now ) { $havdalahTime = clone $t; break; }
            }
        }
    }

    // --- Extension point to PRO ---
    $context = [
        'now'          => $now,
        'tzid'         => $tzid,
        'geonameid'    => $geonameid,
        'api_today'    => $apiUrlToday,
        'api_yesterday'=> $apiUrlYest,
        'items'        => $items,
        'candleTime'   => $candleTime,
        'havdalahTime' => $havdalahTime,
        'decision'     => null, // 'block' | 'open'
    ];
    $context = apply_filters( 'sbk/intercept_context', $context );

    // adopt any changes from PRO:
    $now          = $context['now']          ?? $now;
    $tzid         = $context['tzid']         ?? $tzid;
    $candleTime   = $context['candleTime']   ?? $candleTime;
    $havdalahTime = $context['havdalahTime'] ?? $havdalahTime;

$geoForRender =  $context['tzid']?$context['geonameid']    : ($geonameid ?: ( $geoVar ?? get_option( 'shabbat_blocker_geonameid', '281184' )) );

    if ( isset( $context['decision'] ) && in_array( $context['decision'], ['block','open'], true ) ) {
        $inBlock = ( $context['decision'] === 'block' );
    } else {
        $inBlock = ( $candleTime && $havdalahTime && $now >= $candleTime && $now <= $havdalahTime );
    }

    header( 'X-Shabbat-Decision: ' . ( $inBlock ? 'block' : 'open' ) );
    header( 'X-Shabbat-Now: ' . $now->format( 'Y-m-d\TH:i:s T' ) );
    header( 'X-Shabbat-TZ: ' . $tzid );

    if ( $inBlock ) {
        shabbat_blocker_render_page( $geoForRender, $tzid, $candleTime, $havdalahTime );
        exit;
    }
}

/** =========================
 *  SETTINGS PAGE (Single page for FREE/PRO)
 * ========================== */
add_action( 'admin_menu', 'shabbat_blocker_admin_menu' );
function shabbat_blocker_admin_menu() {
    add_options_page(
        'shabbat-blocker-settings',
        esc_html__( 'shabbat blocker', 'shabbat-blocker' ),
        'manage_options',
        'shabbat-blocker-settings',
        'shabbat_blocker_settings_page'
    );
}

function shabbat_blocker_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    // האם PRO פעיל (דרך Freemius)
    $can_pro = sbk_is_premium();
    // נתונים קבועים
    $cities = [
        '281184'  => __('Jerusalem', 'shabbat-blocker'),
        '5128581' => __('New York', 'shabbat-blocker'),
        '4887398' => __('Chicago', 'shabbat-blocker'),
        '5368361' => __('Los Angeles', 'shabbat-blocker'),
        '2643743' => __('London', 'shabbat-blocker'),
        '2950159' => __('Berlin', 'shabbat-blocker'),
        '3435910' => __('Buenos Aires', 'shabbat-blocker'),
        '1850147' => __('Tokyo', 'shabbat-blocker'),
        '2147714' => __('Sydney', 'shabbat-blocker'),
        '292223'  => __('Dubai', 'shabbat-blocker'),
        '5125771' => __('Monsey', 'shabbat-blocker'),
        '5101798' => __('Lakewood', 'shabbat-blocker'),
    ];

    $allowed_block_modes = ['manual', 'ip'];
    $allowed_logo_pos    = ['bottom','floating_br','floating_bl','floating_tr','floating_tl'];

    // ----- שמירה -----
    if ( isset( $_POST['submit'] ) ) {
        check_admin_referer( 'shabbat_blocker_save', 'shabbat_blocker_nonce' );

        // תמיד נשמר עיר קבועה (חינמי)
        $raw_geo  = isset($_POST['geonameid']) ? wp_unslash($_POST['geonameid']) : '';
        $geo      = sanitize_text_field($raw_geo);
        if ( ! array_key_exists($geo, $cities) ) {
            $geo = '281184';
        }
        update_option('shabbat_blocker_geonameid', $geo);

        // תמיד נשמר opt-in לקרדיט (חינמי)
        $show_credit = ! empty( $_POST['shabbat_blocker_show_credit'] ) ? 'yes' : 'no';
        update_option('shabbat_blocker_show_credit', $show_credit);

        // את שדות הפרו נשמור רק אם Pro פעיל
        if ( $can_pro ) {
            // block_mode
            $raw_block_mode = isset($_POST['block_mode']) ? wp_unslash($_POST['block_mode']) : '';
            $block_mode     = sanitize_text_field($raw_block_mode);
            if ( ! in_array($block_mode, $allowed_block_modes, true) ) {
                $block_mode = 'manual';
            }
            update_option('shabbat_blocker_block_mode', $block_mode);

            // fallback_geonameid
            $raw_fallback = isset($_POST['fallback_geonameid']) ? wp_unslash($_POST['fallback_geonameid']) : '';
            $fallback     = sanitize_text_field($raw_fallback);
            if ( ! array_key_exists($fallback, $cities) ) {
                $fallback = '281184';
            }
            update_option('shabbat_blocker_fallback_geonameid', $fallback);

            // logo_position
            $raw_logo = isset($_POST['logo_position']) ? wp_unslash($_POST['logo_position']) : '';
            $logo_pos = sanitize_text_field($raw_logo);
            if ( ! in_array($logo_pos, $allowed_logo_pos, true) ) {
                $logo_pos = 'bottom';
            }
            update_option('shabbat_blocker_logo_position', $logo_pos);
        }

        add_settings_error(
            'shabbat_blocker_messages',
            'shabbat_blocker_message',
            esc_html__( 'Settings saved.', 'shabbat-blocker' ),
            'updated'
        );
    }

    // ----- קריאה -----
    $geonameid        = get_option('shabbat_blocker_geonameid', '281184');
    $show_credit      = get_option('shabbat_blocker_show_credit', 'no');

    // ברירות מחדל לערכי פרו – גם אם Pro לא פעיל, כדי להציג UI עקבי
    $block_mode       = get_option('shabbat_blocker_block_mode', 'manual');
    $fallback_geo     = get_option('shabbat_blocker_fallback_geonameid', '281184');
    $logo_pos_current = get_option('shabbat_blocker_logo_position', 'bottom');

    $upg_url = ( function_exists('sb_fs') && sb_fs() && method_exists(sb_fs(), 'get_upgrade_url') )
        ? sb_fs()->get_upgrade_url()
        : 'https://autopeak.cloud/shabbat-blocker/';

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'Shabbat Blocker Settings', 'shabbat-blocker' ) . '</h1>';
    settings_errors( 'shabbat_blocker_messages' );

    echo '<form method="post">';
    wp_nonce_field( 'shabbat_blocker_save', 'shabbat_blocker_nonce' );

    echo '<table class="form-table" role="presentation">';

    // 1) Blocking mode (כמו בפרו)
    echo '<tr><th scope="row">' . esc_html__('Blocking mode', 'shabbat-blocker') . '</th><td>';
    echo '<select name="block_mode" ' . ( $can_pro ? '' : 'disabled' ) . '>';
    echo '<option value="manual" ' . selected($block_mode, 'manual', false) . '>' .
            esc_html__('Block by selected city', 'shabbat-blocker') . '</option>';
    $label_ip = esc_html__('Blocking by user location (IP)', 'shabbat-blocker');
    if ( $can_pro ) {
        echo '<option value="ip" ' . selected($block_mode, 'ip', false) . '>' . $label_ip . '</option>';
    } else {
        echo '<option value="ip" disabled>' . $label_ip . ' — ' . esc_html__('Pro', 'shabbat-blocker') . '</option>';
    }
    echo '</select>';
    if ( ! $can_pro ) {
        echo ' <a class="button button-primary" style="margin-inline-start:8px" href="' . esc_url($upg_url) . '" target="_blank" rel="noopener">' .
             esc_html__('Upgrade to Pro', 'shabbat-blocker') . '</a>';
    }
    echo '</td></tr>';

    // 2) Fixed city (כמו בפרו)
    echo '<tr><th scope="row">' . esc_html__('Fixed city (when blocking by city is selected)', 'shabbat-blocker') . '</th><td>';
    echo '<select name="geonameid">';
    foreach ( $cities as $id => $name ) {
        echo '<option value="' . esc_attr($id) . '" ' . selected($geonameid, $id, false) . '>' . esc_html($name) . '</option>';
    }
    echo '</select></td></tr>';

    // 3) Fallback city (כמו בפרו) – פעיל רק בפרו
    echo '<tr><th scope="row">' . esc_html__('Fallback city if location cannot be identified', 'shabbat-blocker') . '</th><td>';
    echo '<select name="fallback_geonameid" ' . ( $can_pro ? '' : 'disabled' ) . '>';
    foreach ( $cities as $id => $name ) {
        echo '<option value="' . esc_attr($id) . '" ' . selected($fallback_geo, $id, false) . '>' . esc_html($name) . '</option>';
    }
    echo '</select>';
    if ( ! $can_pro ) {
        echo ' <span class="sb-pro-badge" style="margin-inline-start:8px">PRO</span>';
    }
    echo '</td></tr>';

    // 4) Logo Position (כמו בפרו) – רק "bottom" חינמי, הצפים בפרו
    echo '<tr><th scope="row">' . esc_html__('Logo Position', 'shabbat-blocker') . '</th><td>';
    $logo_labels = [
        'bottom'      => esc_html__('At the bottom of the site (footer)', 'shabbat-blocker'),
        'floating_br' => esc_html__('Float – Bottom Right', 'shabbat-blocker'),
        'floating_bl' => esc_html__('Float – Bottom Left', 'shabbat-blocker'),
        'floating_tr' => esc_html__('Floating – Top Right', 'shabbat-blocker'),
        'floating_tl' => esc_html__('Floating – Top Left', 'shabbat-blocker'),
    ];
    echo '<select name="logo_position">';
    // תמיד זמין:
    echo '<option value="bottom" ' . selected($logo_pos_current, 'bottom', false) . '>' . $logo_labels['bottom'] . '</option>';
    // הצפות – Disabled אם לא פרו
    $pro_positions = ['floating_br','floating_bl','floating_tr','floating_tl'];
    foreach ( $pro_positions as $p ) {
        $label = $logo_labels[$p] . ( $can_pro ? '' : ' — ' . esc_html__('Pro', 'shabbat-blocker') );
        $attrs = $can_pro
            ? 'value="' . esc_attr($p) . '" ' . selected($logo_pos_current, $p, false)
            : 'value="' . esc_attr($p) . '" disabled';
        echo '<option ' . $attrs . '>' . esc_html($label) . '</option>';
    }
    if ( ! $can_pro ) {
        echo ' <a class="button button-primary" style="margin-inline-start:8px" href="' . esc_url($upg_url) . '" target="_blank" rel="noopener">' .
             esc_html__('Upgrade to Pro', 'shabbat-blocker') . '</a>';
    }
    echo '</select>';

    echo '</td></tr>';

    // 5) Show credit badge (חינמי)
    echo '<tr><th scope="row">' . esc_html__( 'Show credit badge in footer', 'shabbat-blocker' ) . '</th><td>';
    echo '<label class="sb-inline-label">';
    echo '<input type="checkbox" name="shabbat_blocker_show_credit" value="1" ' . checked( $show_credit, 'yes', false ) . ' />';
    echo '<span>' . esc_html__( 'Opt-in: display the “Shabbat site” credit/badge on the front-end', 'shabbat-blocker' ) . '</span>';
    echo '</label>';
    echo '</td></tr>';

    echo '</table>';

    submit_button( esc_html__( 'Save Settings', 'shabbat-blocker' ) );

    echo '</form></div>';
}


add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function( $links ) {
    // אם PRO פעיל – אל תוסיפי את קישור השדרוג בכלל
    if ( function_exists( 'sbk_is_premium' ) && sbk_is_premium() ) {
        return $links;
    }

    $default_url = 'https://autopeak.cloud/shabbat-blocker/';
    $url = ( function_exists('sb_fs') && sb_fs() && method_exists( sb_fs(), 'checkout_url' ) )
        ? sb_fs()->checkout_url()
        : $default_url;

    $upgrade = '<a href="' . esc_url( $url ) . '" target="_blank" rel="noopener noreferrer"><strong style="color:#2271b1">' .
               esc_html__( 'Upgrade to Pro', 'shabbat-blocker' ) . '</strong></a>';

    array_unshift( $links, $upgrade );
    return $links;
} );


add_action( 'wp_footer', 'shabbat_blocker_add_footer_tag' );
function shabbat_blocker_add_footer_tag() {
    if ( 'yes' !== get_option( 'shabbat_blocker_show_credit', 'no' ) ) return;

    $footerPath = plugin_dir_path( __FILE__ ) . 'shabbat-blocker-footer.html';
    $html       = file_exists( $footerPath ) ? file_get_contents( $footerPath ) : '';

    $plugin_url = plugins_url( '/', __FILE__ );
    if ( $html ) {
        $html = str_replace( array( '{{PLUGIN_URL}}', '{{PLUGIN_BASE}}' ), esc_url( $plugin_url ), $html );
        $html = wp_kses( $html, shabbat_blocker_allowed_html() );
    } else {
        $html = '<div>' . esc_html( '🕯️ אתר שומר שבת – AutoPeak' ) . '</div>';
    }

    // תמיכה במיקום צף – רק כש-PRO פעיל
    $position = get_option( 'shabbat_blocker_logo_position', 'bottom' );
    if ( ! sbk_is_premium() || $position === 'bottom' ) {
        echo $html;
        return;
    }

    $offset   = '20px';
    switch ( $position ) {
        case 'floating_bl': $stylePos = 'bottom:'.$offset.';left:'.$offset.';'; break;
        case 'floating_tr': $stylePos = 'top:'.$offset.';right:'.$offset.';';   break;
        case 'floating_tl': $stylePos = 'top:'.$offset.';left:'.$offset.';';    break;
        case 'floating_br':
        default:            $stylePos = 'bottom:'.$offset.';right:'.$offset.';'; break;
    }

    echo '<div style="position:fixed;'.$stylePos.'z-index:9999;pointer-events:none;">' .
            '<div style="pointer-events:auto;">' . $html . '</div>' .
         '</div>';
}

/** =========================
 *  Admin CSS – רק בעמוד ההגדרות
 * ========================== */
add_action( 'admin_head', function () {
    $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
    if ( ! $screen || $screen->id !== 'settings_page_shabbat-blocker-settings' ) return; ?>
    <style>
        .sb-card{background:#fff;border:1px solid #e5e7eb;border-radius:10px;padding:16px 18px;margin:12px 0;box-shadow:0 1px 1px rgba(0,0,0,.03)}
        .sb-card h3{margin:0 0 10px;font-size:16px;line-height:1.4;display:flex;gap:8px;align-items:center}
        .sb-pro-badge{display:inline-block;background:#f0f6ff;color:#1d4ed8;border:1px solid #cfe0ff;padding:2px 8px;border-radius:999px;font-size:11px;font-weight:600;margin-inline-start:8px}
        .sb-card .desc{color:#5f6c7b;margin:.25rem 0 .75rem;font-size:13px}
        .sb-actions{display:flex;gap:8px;align-items:center;flex-wrap:wrap}
        .sb-actions .button-primary{height:34px;padding:0 14px}
        .sb-muted{color:#6b7280;font-size:12px}
        .sb-row th{width:260px;vertical-align:top;padding-top:16px}
        .sb-row td{padding-top:12px}
        .sb-inline-label{display:flex;align-items:center;gap:10px}
        .sb-inline-label input[type=checkbox]{transform:scale(1.05)}
        .sb-sep{height:1px;background:#eef1f4;margin:14px 0}
        .dashicons{color:#2271b1}
    </style>
<?php } );

/** =========================
 *  Translations quick map (optional)
 * ========================== */
add_filter( 'gettext', function( $translated, $text, $domain ) {
    if ( $domain !== 'shabbat-blocker' ) return $translated;
     static $map = [
        'Shabbat Blocker Settings' => 'הגדרות חסימת שבת',
        'Settings saved.'          => 'ההגדרות נשמרו.',
'shabbat blocker'=>'חסימת שבת',
        'Blocking mode'                                 => 'מצב חסימה',
'Detect the visitor location by IP and block according to their local Shabbat/holiday times.' => 'גלה את מיקום המשתמש לפי IP וחסום לפי זמני שבת/חג המקומיים שלו',
'Enable IP-based blocking' => 'הפעל חסימה מבוססת-IP',
'Choose where to show the floating logo/badge on your site.' => 'בחר היכן להציג את הסמל/התג הצף באתר',
'Unlock Pro' => 'שדרג ל-Pro',
'Upgrade to Pro' => 'שדרוג ל-Pro',
'Pro features' => 'תכונות Pro',
        'Fixed city (when blocking by city is selected)'=> 'עיר קבועה (כאשר נבחרת חסימה לפי עיר)',
        'Fallback city if location cannot be identified'=> 'עיר ברירת מחדל כאשר לא ניתן לזהות מיקום',
        'Logo Position'                                 => 'מיקום הסמל (Logo)',
        'Show credit badge in footer'                   => 'הצג סמל קרדיט בפוטר',
        'Opt-in: display the “Shabbat site” credit/badge on the front-end' => 'בחירה: הצגת סמל/קרדיט "אתר שומר שבת" בצד המשתמש',
        'Save Settings'                                 => 'שמירת הגדרות',
'Block by city'        => 'חסימה לפי עיר',
        'Block by selected city'        => 'חסימה לפי עיר נבחרת',
        'Blocking by user location (IP)'=> 'חסימה לפי מיקום משתמש (IP)',
        'At the bottom of the site (footer)' => 'בתחתית האתר (Footer)',
        'Float – Bottom Right'          => 'צף – ימין למטה',
        'Float – Bottom Left'           => 'צף – שמאל למטה',
        'Floating – Top Right'          => 'צף – ימין למעלה',
        'Floating – Top Left'           => 'צף – שמאל למעלה',

        'Jerusalem'=> 'ירושלים',
        'New York'=> 'ניו יורק',
        'Chicago'=> 'שיקאגו',
        'Los Angeles'=> 'לוס אנג׳לס',
        'London'=> 'לונדון',
        'Berlin'=> 'ברלין',
        'Buenos Aires'=> 'בואנוס איירס',
        'Tokyo'=> 'טוקיו',
        'Sydney'=> 'סידני',
        'Dubai'=> 'דובאי',
        'Monsey'=> 'מונסי',
        'Lakewood'=> 'לייקווד',
    ];
    return $map[ $text ] ?? $translated;
}, 10, 3 );
