<?php
/*
Plugin Name: SH Email Tester
Description: SH Email Tester lets you send a test email from your WordPress site to confirm that email delivery is working correctly.
Version: 1.0.0
Requires PHP: 7.0
Author: Sakib Hasan
Author URI: https://profiles.wordpress.org/sakibhasan/
Text Domain: sh-email-tester
Domain Path: /languages
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) {
    exit;
}

define('SHET_EMAIL_TESTER_VERSION', '1.0.0');

function shet_migrate_storage_keys() {
    $new_value = get_option('shet_settings', null);
    if (null === $new_value) {
        $old_value = get_option('shmet_email_tester_settings', null);
        if (null !== $old_value) {
            add_option('shet_settings', $old_value);
        }
    }
}
add_action('admin_init', 'shet_migrate_storage_keys');

function shet_email_tester_get_logs_table_name() {
    global $wpdb;
    return $wpdb->prefix . 'shet_email_tester_logs';
}

function shet_email_tester_logs_table_exists() {
    static $table_exists = null;

    if (null !== $table_exists) {
        return $table_exists;
    }

    $cache_key = 'shet_logs_table_exists';
    $cached = wp_cache_get($cache_key, 'shet_email_tester');
    if (false !== $cached) {
        $table_exists = (bool) $cached;
        return $table_exists;
    }

    global $wpdb;
    $table_name = shet_email_tester_get_logs_table_name();
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $table_exists = ($wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table_name)) === $table_name);
    wp_cache_set($cache_key, $table_exists, 'shet_email_tester');

    return $table_exists;
}

function shet_email_tester_create_table() {
    global $wpdb;
    $table_name = shet_email_tester_get_logs_table_name();
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        time datetime NOT NULL,
        to_email text NOT NULL,
        subject text NOT NULL,
        body longtext NOT NULL,
        status varchar(20) NOT NULL,
        PRIMARY KEY  (id),
        KEY time (time)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
    wp_cache_set('shet_logs_table_exists', true, 'shet_email_tester');
}

function shet_email_tester_schedule_cleanup() {
    if (!wp_next_scheduled('shet_clear_logs')) {
        wp_schedule_event(time(), 'daily', 'shet_clear_logs');
    }
}

function shet_email_tester_activate() {
    shet_email_tester_create_table();
    shet_email_tester_schedule_cleanup();
    if (null === get_option('shet_settings', null)) {
        add_option(
            'shet_settings',
            array(
                'retention_days' => 30,
            )
        );
    }
}
register_activation_hook(__FILE__, 'shet_email_tester_activate');

function shet_email_tester_deactivate() {
    wp_clear_scheduled_hook('shet_clear_logs');
}
register_deactivation_hook(__FILE__, 'shet_email_tester_deactivate');

function shet_email_tester_register_admin_pages() {
    add_submenu_page(
        'tools.php',
        __('Test Email', 'sh-email-tester'),
        __('Test Email', 'sh-email-tester'),
        'manage_options',
        'shet-email-tester',
        'shet_render_test_email_page'
    );

    add_submenu_page(
        'tools.php',
        __('Email Logs', 'sh-email-tester'),
        __('Email Logs', 'sh-email-tester'),
        'manage_options',
        'shet_email_tester_logs',
        'shet_email_tester_render_logs_page'
    );
}
add_action('admin_menu', 'shet_email_tester_register_admin_pages');

function shet_render_test_email_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    $notice = array();
    $to = '';
    $subject = '';
    $default_subject = __('Test Email', 'sh-email-tester');

    if (
        isset($_SERVER['REQUEST_METHOD'])
        && 'POST' === sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD']))
        && isset($_POST['shet_email_tester_submit'])
    ) {
        check_admin_referer('shet_email_tester_send_test_email');

        $to = isset($_POST['mail_to']) ? sanitize_email(wp_unslash($_POST['mail_to'])) : '';
        $subject = isset($_POST['mail_subject']) ? sanitize_text_field(wp_unslash($_POST['mail_subject'])) : '';

        if ('' === $subject) {
            $subject = $default_subject;
        }

        if (empty($to) || !is_email($to)) {
            $notice = array(
                'type' => 'error',
                'message' => __('Please enter a valid email address.', 'sh-email-tester'),
            );
        } else {
            $site_name = wp_specialchars_decode(get_bloginfo('name'), ENT_QUOTES);
            $body = sprintf(
                /* translators: %s: Site name. */
                __('This is a test email sent from %s.', 'sh-email-tester'),
                $site_name
            );
            $headers = array('Content-Type: text/plain; charset=UTF-8');
            $sent = wp_mail($to, $subject, $body, $headers);

            if ($sent) {
                $notice = array(
                    'type' => 'success',
                    'message' => __('Email has been sent.', 'sh-email-tester'),
                );
            } else {
                $notice = array(
                    'type' => 'error',
                    'message' => __('Email could not be sent.', 'sh-email-tester'),
                );
            }
        }
    }

    $subject_value = ('' !== $subject) ? $subject : $default_subject;
    ?>
    <div class="wrap shet-email-tester-test">
        <h1><?php echo esc_html__('Test Email', 'sh-email-tester'); ?></h1>
        <p class="shet-email-tester-hint"><?php echo esc_html__('Send a quick message to confirm your mail setup is working.', 'sh-email-tester'); ?></p>

        <?php if (!empty($notice)) { ?>
            <div class="notice notice-<?php echo esc_attr($notice['type']); ?> is-dismissible">
                <p><?php echo esc_html($notice['message']); ?></p>
            </div>
        <?php } ?>

        <div class="shet-email-tester-card">
        <form method="post">
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><?php echo esc_html__('To', 'sh-email-tester'); ?></th>
                    <td>
                        <input class="regular-text" type="email" name="mail_to" value="<?php echo esc_attr($to); ?>" required />
                        <p class="description"><?php echo esc_html__('Enter the recipient email address.', 'sh-email-tester'); ?></p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php echo esc_html__('Subject', 'sh-email-tester'); ?></th>
                    <td>
                        <input class="regular-text" type="text" name="mail_subject" value="<?php echo esc_attr($subject_value); ?>" />
                        <p class="description"><?php echo esc_html__('Optional subject line for the test email.', 'sh-email-tester'); ?></p>
                    </td>
                </tr>
            </table>
            <?php wp_nonce_field('shet_email_tester_send_test_email'); ?>
            <?php submit_button(__('Send Test Email', 'sh-email-tester'), 'primary', 'shet_email_tester_submit'); ?>
        </form>

        </div>
    </div>
    <?php
}

add_action('phpmailer_init', 'shet_email_tester_log_outgoing_email');
function shet_email_tester_log_outgoing_email($phpmailer) {
    global $wpdb;

    if (!shet_email_tester_logs_table_exists()) {
        return;
    }

    $to_addresses = $phpmailer->getToAddresses();
    if (empty($to_addresses)) {
        return;
    }

    $to_emails = array();
    foreach ($to_addresses as $address) {
        if (!empty($address[0])) {
            $to_emails[] = $address[0];
        }
    }

    if (empty($to_emails)) {
        return;
    }

    $table_name = shet_email_tester_get_logs_table_name();
    $to = implode(', ', array_unique($to_emails));
    $subject = isset($phpmailer->Subject) ? $phpmailer->Subject : '';
    $body = isset($phpmailer->Body) ? $phpmailer->Body : '';
    $status = 'Sent';

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
    $wpdb->insert(
        $table_name,
        array(
            'time' => current_time('mysql'),
            'to_email' => $to,
            'subject' => $subject,
            'body' => $body,
            'status' => $status,
        ),
        array('%s', '%s', '%s', '%s', '%s')
    );
}

function shet_email_tester_render_logs_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    if (!shet_email_tester_logs_table_exists()) {
        ?>
        <div class="notice notice-warning">
            <p><?php echo esc_html__('Email log table not found. Deactivate and reactivate the plugin to create it.', 'sh-email-tester'); ?></p>
        </div>
        <?php
        return;
    }

    global $wpdb;
    $table_name = shet_email_tester_get_logs_table_name();
    $items_per_page = 10;
    $allowed_sort_columns = array('time', 'to_email', 'subject', 'status');

    $nonce_action = 'shet_email_tester_logs_search';
    $nonce_name = 'shet_email_tester_logs_nonce';
    $nonce_value = wp_create_nonce($nonce_action);
    $has_nonce = isset($_GET[$nonce_name]) && wp_verify_nonce(
        sanitize_text_field(wp_unslash($_GET[$nonce_name])),
        $nonce_action
    );

    $current_page = 1;
    $sort_by = 'time';
    $order = 'DESC';
    $search_query = '';

    if ($has_nonce) {
        $current_page = isset($_GET['paged']) ? max(1, absint(wp_unslash($_GET['paged']))) : 1;
        $sort_by = isset($_GET['orderby']) ? sanitize_key(wp_unslash($_GET['orderby'])) : 'time';
        $order = isset($_GET['order']) ? strtoupper(sanitize_key(wp_unslash($_GET['order']))) : 'DESC';
        $search_query = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
    }

    if (!in_array($sort_by, $allowed_sort_columns, true)) {
        $sort_by = 'time';
    }
    if ('ASC' !== $order && 'DESC' !== $order) {
        $order = 'DESC';
    }

    $offset = ($current_page - 1) * $items_per_page;
    $order_sql = ('ASC' === $order) ? 'ASC' : 'DESC';
    $link_args = array(
        'page' => 'shet_email_tester_logs',
        $nonce_name => $nonce_value,
    );
    if ('' !== $search_query) {
        $link_args['s'] = $search_query;
    }

    if ('' !== $search_query) {
        $search_like = '%' . $wpdb->esc_like($search_query) . '%';

        if ('ASC' === $order_sql) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE to_email LIKE %s OR subject LIKE %s OR body LIKE %s ORDER BY %i ASC LIMIT %d, %d',
                    $table_name,
                    $search_like,
                    $search_like,
                    $search_like,
                    $sort_by,
                    $offset,
                    $items_per_page
                )
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE to_email LIKE %s OR subject LIKE %s OR body LIKE %s ORDER BY %i DESC LIMIT %d, %d',
                    $table_name,
                    $search_like,
                    $search_like,
                    $search_like,
                    $sort_by,
                    $offset,
                    $items_per_page
                )
            );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total_items = (int) $wpdb->get_var(
            $wpdb->prepare(
                'SELECT COUNT(*) FROM %i WHERE to_email LIKE %s OR subject LIKE %s OR body LIKE %s',
                $table_name,
                $search_like,
                $search_like,
                $search_like
            )
        );
    } else {
        if ('ASC' === $order_sql) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i ORDER BY %i ASC LIMIT %d, %d',
                    $table_name,
                    $sort_by,
                    $offset,
                    $items_per_page
                )
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i ORDER BY %i DESC LIMIT %d, %d',
                    $table_name,
                    $sort_by,
                    $offset,
                    $items_per_page
                )
            );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total_items = (int) $wpdb->get_var(
            $wpdb->prepare('SELECT COUNT(*) FROM %i', $table_name)
        );
    }

    $total_pages = (int) ceil($total_items / $items_per_page);
    ?>

    <div class="wrap">
        <h1><?php echo esc_html__('Email Logs', 'sh-email-tester'); ?></h1>

        <div class="search-wrapper">
            <form method="get">
                <input type="hidden" name="page" value="shet_email_tester_logs" />
                <?php wp_nonce_field($nonce_action, $nonce_name, false); ?>
                <input type="text" name="s" value="<?php echo esc_attr($search_query); ?>" placeholder="<?php echo esc_attr__('Search emails...', 'sh-email-tester'); ?>" />
                <input type="submit" class="button" value="<?php echo esc_attr__('Search', 'sh-email-tester'); ?>" />
            </form>
        </div>

        <table class="widefat fixed" cellspacing="0">
            <thead>
                <tr>
                    <th><a href="<?php echo esc_url(add_query_arg(array_merge($link_args, array('orderby' => 'time', 'order' => ('time' === $sort_by && 'ASC' === $order) ? 'DESC' : 'ASC', 'paged' => 1)))); ?>"><?php echo esc_html__('Time', 'sh-email-tester'); ?></a></th>
                    <th><a href="<?php echo esc_url(add_query_arg(array_merge($link_args, array('orderby' => 'to_email', 'order' => ('to_email' === $sort_by && 'ASC' === $order) ? 'DESC' : 'ASC', 'paged' => 1)))); ?>"><?php echo esc_html__('To', 'sh-email-tester'); ?></a></th>
                    <th><a href="<?php echo esc_url(add_query_arg(array_merge($link_args, array('orderby' => 'subject', 'order' => ('subject' === $sort_by && 'ASC' === $order) ? 'DESC' : 'ASC', 'paged' => 1)))); ?>"><?php echo esc_html__('Subject', 'sh-email-tester'); ?></a></th>
                    <th><?php echo esc_html__('Body', 'sh-email-tester'); ?></th>
                    <th><a href="<?php echo esc_url(add_query_arg(array_merge($link_args, array('orderby' => 'status', 'order' => ('status' === $sort_by && 'ASC' === $order) ? 'DESC' : 'ASC', 'paged' => 1)))); ?>"><?php echo esc_html__('Status', 'sh-email-tester'); ?></a></th>
                </tr>
            </thead>
            <tbody>
                <?php if ($results) { ?>
                    <?php foreach ($results as $log) {
                        $log_time = isset($log->time) ? (string) $log->time : '';
                        $log_to = isset($log->to_email) ? (string) $log->to_email : '';
                        $log_subject = isset($log->subject) ? (string) $log->subject : '';
                        $log_body = isset($log->body) ? (string) $log->body : '';
                        $log_status = isset($log->status) ? (string) $log->status : '';
                        ?>
                        <tr>
                            <td><?php echo esc_html($log_time); ?></td>
                            <td><?php echo esc_html($log_to); ?></td>
                            <td><?php echo esc_html($log_subject); ?></td>
                            <td><a href="#" class="view-email-body" data-body="<?php echo esc_attr($log_body); ?>"><?php echo esc_html__('View', 'sh-email-tester'); ?></a></td>
                            <td><?php echo esc_html($log_status); ?></td>
                        </tr>
                    <?php } ?>
                <?php } else { ?>
                    <tr>
                        <td colspan="5"><?php echo esc_html__('No logs found.', 'sh-email-tester'); ?></td>
                    </tr>
                <?php } ?>
            </tbody>
        </table>

        <div class="tablenav">
            <div class="tablenav-pages">
                <?php
                $pagination_links = paginate_links(
                    array(
                        'base' => add_query_arg(array_merge($link_args, array('paged' => '%#%'))),
                        'format' => '',
                        'prev_text' => __('&laquo;', 'sh-email-tester'),
                        'next_text' => __('&raquo;', 'sh-email-tester'),
                        'total' => $total_pages,
                        'current' => $current_page,
                    )
                );
                if ($pagination_links) {
                    echo wp_kses_post($pagination_links);
                }
                ?>
            </div>
        </div>

        <div id="email-popup" style="display:none;">
            <div class="popup-content">
                <div class="popup-header">
                    <h2><?php echo esc_html__('Email Body', 'sh-email-tester'); ?></h2>
                    <a href="#" id="close-popup" class="close-popup"><?php echo esc_html__('Close', 'sh-email-tester'); ?></a>
                </div>
                <div class="popup-body"></div>
            </div>
        </div>
    </div>
    <?php
}

function shet_email_tester_enqueue_admin_assets($hook) {
    $allowed_hooks = array('tools_page_shet-email-tester', 'tools_page_shet_email_tester_logs');
    if (!in_array($hook, $allowed_hooks, true)) {
        return;
    }

    wp_enqueue_script('jquery');

    wp_register_style('shet_email_tester_admin', false, array(), SHET_EMAIL_TESTER_VERSION);
    wp_enqueue_style('shet_email_tester_admin');
    wp_add_inline_style(
        'shet_email_tester_admin',
        '.shet-email-tester-test .shet-email-tester-card{background:#fff;border:1px solid #dcdcde;border-radius:12px;padding:24px;max-width:720px;box-shadow:0 10px 30px rgba(16,24,40,.08)}.shet-email-tester-test .form-table{margin-top:0}.shet-email-tester-test .form-table th{width:140px;padding-left:0}.shet-email-tester-test .form-table td{padding-right:0}.shet-email-tester-test input.regular-text{max-width:420px}.shet-email-tester-test .submit{margin-top:8px}.shet-email-tester-test .description{color:#6c7278}.shet-email-tester-test .notice{margin:16px 0}.shet-email-tester-test h1{margin-bottom:10px}.shet-email-tester-test .shet-email-tester-hint{margin:0 0 18px;color:#50575e}.shet-email-tester-logs .search-wrapper{margin:16px 0;display:flex;justify-content:flex-end}.shet-email-tester-logs .search-wrapper form{display:flex;gap:8px;align-items:center}.shet-email-tester-logs .search-wrapper input[type="text"]{min-width:240px;padding:6px 10px;border:1px solid #c3c4c7;border-radius:4px}.shet-email-tester-logs .search-wrapper .button{padding:6px 12px;border-radius:4px}.shet-email-tester-logs .widefat{border-radius:6px;overflow:hidden}.shet-email-tester-logs .widefat tbody tr:hover{background:#f6f7f7}.shet-email-tester-logs a.view-email-body{font-weight:600}.shet-email-tester-logs #email-popup{position:fixed;top:0;left:0;width:100%;height:100%;background-color:rgba(15,23,42,.6);z-index:10000;display:none;justify-content:center;align-items:center;backdrop-filter:saturate(140%) blur(2px)}.shet-email-tester-logs .popup-content{background:#fff;padding:20px;border-radius:12px;max-width:640px;width:92%;max-height:80%;overflow-y:auto;overflow-x:hidden;margin:40px auto;box-shadow:0 20px 60px rgba(15,23,42,.25)}.shet-email-tester-logs .popup-header{display:flex;justify-content:space-between;align-items:center;margin-bottom:12px}.shet-email-tester-logs .close-popup{cursor:pointer;color:#1d2327;text-decoration:none;font-weight:600}.shet-email-tester-logs .popup-body{word-wrap:break-word;white-space:pre-wrap;line-height:1.5}.shet-email-tester-logs .tablenav .tablenav-pages{display:flex;justify-content:flex-end;margin-top:16px}.shet-email-tester-logs .tablenav .tablenav-pages a,.shet-email-tester-logs .tablenav .tablenav-pages span{display:inline-block;padding:6px 12px;margin:0 2px;border:1px solid #dcdcde;border-radius:4px;color:#0073aa;text-decoration:none;font-size:13px}.shet-email-tester-logs .tablenav .tablenav-pages a:hover{background-color:#f6f7f7;border-color:#c3c4c7}.shet-email-tester-logs .tablenav .tablenav-pages .current{background-color:#0073aa;color:#fff;border-color:#0073aa;cursor:default}'
    );

    $script = "jQuery(function($){\n"
        . "    $('.view-email-body').on('click',function(e){\n"
        . "        e.preventDefault();\n"
        . "        var emailBody=$(this).data('body')||'';\n"
        . "        $('#email-popup .popup-body').text(emailBody);\n"
        . "        $('#email-popup').css('display','flex');\n"
        . "    });\n"
        . "    $('#close-popup').on('click',function(e){\n"
        . "        e.preventDefault();\n"
        . "        $('#email-popup').hide();\n"
        . "    });\n"
        . "    $(document).on('click',function(e){\n"
        . "        if($(e.target).is('#email-popup')){\n"
        . "            $('#email-popup').hide();\n"
        . "        }\n"
        . "    });\n"
        . "});\n";
    wp_add_inline_script('jquery', $script);
}
add_action('admin_enqueue_scripts', 'shet_email_tester_enqueue_admin_assets');

add_action('shet_clear_logs', 'shet_email_tester_clear_old_logs');
function shet_email_tester_clear_old_logs() {
    if (!shet_email_tester_logs_table_exists()) {
        return;
    }

    global $wpdb;
    $table_name = shet_email_tester_get_logs_table_name();
    $settings = get_option('shet_settings', array());
    $retention_days = isset($settings['retention_days']) ? absint($settings['retention_days']) : 30;
    $retention_days = max(1, $retention_days);
    $date_threshold = date_i18n(
        'Y-m-d H:i:s',
        current_time('timestamp') - $retention_days * DAY_IN_SECONDS
    );

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->query(
        $wpdb->prepare(
            'DELETE FROM %i WHERE time < %s',
            $table_name,
            $date_threshold
        )
    );
}
