<?php
/**
 * Uninstall SFR Directory Map
 * 
 * Removes all plugin data from the database when the plugin is deleted.
 * Only runs if user has enabled "Delete all plugin data when plugin is deleted" in settings.
 *
 * @package SFRDirectoryMap
 * @since 1.0.0
 */

// Exit if accessed directly or not via WordPress uninstall
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

// Check if user wants to delete data on uninstall
if (!get_option('sfrdm_delete_data_on_uninstall', 0)) {
    // User wants to preserve data - exit without deleting anything
    return;
}

global $wpdb;

// 1. Drop custom database tables
$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sfrdm_search_analytics"); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sfrdm_search_summary"); // phpcs:ignore WordPress.DB.DirectDatabaseQuery

// 2. Delete all plugin options
$options_to_delete = array(
    'sfrdm_default_lat',
    'sfrdm_default_lng',
    'sfrdm_default_zoom',
    'sfrdm_settings',
    'sfrdm_enable_analytics',
    'sfrdm_delete_data_on_uninstall',
    'sfrdm_enable_local_dev',
    'sfrdm_development_mode',
    'sfrdm_colour_templates',
    'sfrdm_button_primary_color',
    'sfrdm_button_secondary_color',
    'sfrdm_button_hover_color',
    'sfrdm_button_text_color',
    'sfrdm_category_colors',
);

foreach ($options_to_delete as $option) {
    delete_option($option);
}

// 3. Delete all transients
delete_transient('sfrdm_plugin_detection');
delete_transient('sfrdm_detected_plugin');

// 4. Delete all custom post type posts (sfrdm_map) and their meta
$map_posts = get_posts(array(
    'post_type' => 'sfrdm_map',
    'numberposts' => -1,
    'post_status' => 'any',
));

foreach ($map_posts as $post) {
    $wpdb->delete($wpdb->postmeta, array('post_id' => $post->ID)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    wp_delete_post($post->ID, true);
}

// 5. Clear scheduled cron jobs
$cron_hooks = array(
    'sfrdm_cleanup_old_analytics',
);

foreach ($cron_hooks as $hook) {
    $timestamp = wp_next_scheduled($hook);
    if ($timestamp) {
        wp_unschedule_event($timestamp, $hook);
    }
    wp_clear_scheduled_hook($hook);
}

// 6. Clean up any uploaded assets using WP_Filesystem
$upload_dir = wp_upload_dir();
$assets_dir = $upload_dir['basedir'] . '/sfr-directory-map';

if (file_exists($assets_dir)) {
    global $wp_filesystem;
    if (empty($wp_filesystem)) {
        require_once ABSPATH . '/wp-admin/includes/file.php';
        WP_Filesystem();
    }
    $wp_filesystem->rmdir($assets_dir, true);
}

// 7. Flush rewrite rules
flush_rewrite_rules();
