<?php
/**
 * Plugin Name: SFR Directory Map
 * Plugin URI: https://supportfromrichard.co.uk/plugins/sfr-directory-map/
 * Description: Display your Directorist listings on an interactive OpenStreetMap with marker clustering, unlimited pins, and rich popups. No restrictions, no limits!
 * Version: 1.6.0
 * Author: Support From Richard
 * Author URI: https://supportfromrichard.co.uk/
 * Author Email: plugins@supportfromrichard.co.uk
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: sfr-directory-map
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * 
 * @package SFRDirectoryMap
 * @copyright 2025 Richard Jemmett
 * @license GPL-2.0-or-later
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define Pro version constant
// Free version - no constant needed

// Check if Pro version is active
if (defined('SFRDM_PRO_VERSION')) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error">
            <p>
                <strong><?php esc_html_e('SFR Directory Map', 'sfr-directory-map'); ?></strong>: 
                <?php esc_html_e('Please deactivate the Pro version of SFR Directory Map before activating the free version.', 'sfr-directory-map'); ?>
            </p>
        </div>
        <?php
    });
    return; // Stop loading Free if Pro is active
}

class SFRDM_Main {
    
    private $version = '1.6.0';
    private $plugin_slug = 'sfr-directory-map';
    private $detected_plugin_cache = null;
    private $adapter = null;
    // private $licence_manager = null; // Removed in free version
    private $analytics_manager = null;
    
    public function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_shortcode('sfr_directory_map', array($this, 'sfrdm_render_map_shortcode'));
        
        // AJAX handlers
        add_action('wp_ajax_sfrdm_get_directory_listings', array($this, 'sfrdm_get_listings_ajax'));
        add_action('wp_ajax_nopriv_sfrdm_get_directory_listings', array($this, 'sfrdm_get_listings_ajax'));
        // add_action('wp_ajax_sfrdm_save_map', array($this, 'sfrdm_save_map_ajax'));
        // add_action('wp_ajax_sfrdm_get_map_data', array($this, 'sfrdm_get_map_data_ajax'));
        add_action('wp_ajax_sfrdm_save_colour_template', array($this, 'sfrdm_save_colour_template_ajax'));
        add_action('wp_ajax_sfrdm_load_colour_template', array($this, 'sfrdm_load_colour_template_ajax'));
        add_action('wp_ajax_sfrdm_delete_colour_template', array($this, 'sfrdm_delete_colour_template_ajax'));
        
        // Admin
        if (is_admin()) {
            add_action('admin_menu', array($this, 'add_admin_menu'));
            add_action('init', array($this, 'register_map_post_type'));
            add_action('add_meta_boxes', array($this, 'add_map_meta_boxes'));
            add_action('save_post', array($this, 'save_map_meta'));
        }
    }
    
    public function init() {
        // Text domain is automatically loaded by WordPress for plugins hosted on WordPress.org
        // No manual loading required
        
        // Include analytics manager
        require_once plugin_dir_path(__FILE__) . 'includes/analytics-manager.php';
        
        // Initialize analytics manager
        $this->init_analytics_manager();
        
        // Include admin files (only in admin)
        if (is_admin()) {
            require_once plugin_dir_path(__FILE__) . 'includes/admin-settings.php';
            
            // Initialize admin settings class to ensure settings are registered
            new SFRDM_Admin_Settings();
        }
    }
    
    /**
     * Initialize licence manager
     */
    
    /**
     * Initialize analytics manager
     */
    private function init_analytics_manager() {
        if (class_exists('SFRDM_Analytics_Manager')) {
            $this->analytics_manager = new SFRDM_Analytics_Manager();
        }
    }
    
    /**
     * Initialize update checker
     */
    private function init_update_checker() {
        // Include update checker
        if (!class_exists('SFRDMPRO_Update_Checker')) {
            require_once plugin_dir_path(__FILE__) . 'includes/update-checker.php';
        }
        
        // Initialize with licence manager
        if ($this->licence_manager) {
            new SFRDMPRO_Update_Checker(__FILE__, $this->version, $this->licence_manager);
        }
    }
    
    /**
     * Check if premium features are available
     */
    private function is_premium_feature_available($feature = 'all') {
        // Free version: always returns false
        return false;
    }
    
    /**
     * Check if this is a local development environment
     */
    private function is_local_development() {
        // First check if we're actually on a local domain
        $domain = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
        $is_local_domain = (
            strpos($domain, '.local') !== false ||
            strpos($domain, 'localhost') !== false ||
            strpos($domain, '127.0.0.1') !== false ||
            strpos($domain, '192.168.') !== false ||
            strpos($domain, '10.0.') !== false ||
            strpos($domain, '172.16.') !== false
        );
        
        // Only allow development mode if it's actually a local domain AND the option is enabled
        return $is_local_domain && get_option('sfrdm_enable_local_dev', 0);
    }
    
    /**
     * Check if user has premium licence
     */
    private function is_premium_user() {
        $licence_status = get_option('sfrdm_licence_status', 'inactive');
        return $licence_status === 'active';
    }
    
    /**
     * Register custom post type for maps
     */
    public function register_map_post_type() {
        $labels = array(
            'name'                  => esc_html__('Directory Maps', 'sfr-directory-map'),
            'singular_name'         => esc_html__('Directory Map', 'sfr-directory-map'),
            'menu_name'             => esc_html__('Directory Maps', 'sfr-directory-map'),
            'add_new'               => esc_html__('Add New Map', 'sfr-directory-map'),
            'add_new_item'          => esc_html__('Add New Directory Map', 'sfr-directory-map'),
            'edit_item'             => esc_html__('Edit Directory Map', 'sfr-directory-map'),
            'new_item'              => esc_html__('New Directory Map', 'sfr-directory-map'),
            'view_item'             => esc_html__('View Directory Map', 'sfr-directory-map'),
            'search_items'          => esc_html__('Search Maps', 'sfr-directory-map'),
            'not_found'             => esc_html__('No maps found', 'sfr-directory-map'),
            'not_found_in_trash'    => esc_html__('No maps found in trash', 'sfr-directory-map'),
        );

        $args = array(
            'labels'                => $labels,
            'public'                => false,
            'publicly_queryable'    => false,
            'show_ui'               => true,
            'show_in_menu'          => true,
            'menu_position'         => 25,
            'menu_icon'             => 'dashicons-location-alt',
            'query_var'             => true,
            'rewrite'               => false,
            'capability_type'       => 'post',
            'has_archive'           => false,
            'hierarchical'          => false,
            'supports'              => array('title', 'editor'),
            'show_in_rest'          => false,
        );

        register_post_type('sfrdm_map', $args);
    }
    
    /**
     * Add meta boxes for map configuration
     */
    public function add_map_meta_boxes() {
        add_meta_box(
            'sfrdm_map_settings',
            esc_html__('Map Settings', 'sfr-directory-map'),
            array($this, 'map_settings_meta_box'),
            'sfrdm_map',
            'normal',
            'high'
        );
        
        add_meta_box(
            'sfrdm_map_preview',
            esc_html__('Map Preview', 'sfr-directory-map'),
            array($this, 'map_preview_meta_box'),
            'sfrdm_map',
            'side',
            'high'
        );
        
        add_meta_box(
            'sfrdm_shortcode_info',
            esc_html__('Shortcode', 'sfr-directory-map'),
            array($this, 'shortcode_info_meta_box'),
            'sfrdm_map',
            'side',
            'default'
        );
    }
    
    public function enqueue_scripts() {
        global $post;
        
        // Check if shortcode is present on current page (support both old and new shortcode names)
        if (!is_a($post, 'WP_Post') || !has_shortcode($post->post_content, 'sfr_directory_map')) {
            return;
        }
        
        // Leaflet CSS and JS from local assets
        wp_enqueue_style(
            'leaflet-css', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet/leaflet.css', 
            array(), 
            $this->version
        );
        
        wp_enqueue_script(
            'leaflet-js', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet/leaflet.js', 
            array(), 
            $this->version, 
            true
        );
        
        // Marker clustering plugin
        wp_enqueue_style(
            'leaflet-markercluster-css', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet-markercluster/MarkerCluster.css', 
            array('leaflet-css'), 
            $this->version
        );
        
        wp_enqueue_style(
            'leaflet-markercluster-default-css', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet-markercluster/MarkerCluster.Default.css', 
            array('leaflet-markercluster-css'), 
            $this->version
        );
        
        wp_enqueue_script(
            'leaflet-markercluster-js', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet-markercluster/leaflet.markercluster.js', 
            array('leaflet-js'), 
            $this->version, 
            true
        );
        
        // Our custom map script
        wp_enqueue_script(
            'sfr-directory-map', 
            plugin_dir_url(__FILE__) . 'assets/map.js', 
            array('leaflet-js', 'leaflet-markercluster-js', 'jquery'), 
            $this->version, 
            true
        );
        
        // Font Awesome for category icons
        wp_enqueue_style(
            'udm-font-awesome',
            plugin_dir_url(__FILE__) . 'assets/font-awesome.min.css',
            array(),
            '6.0.0'
        );
        
        // Custom styles
        wp_enqueue_style(
            'sfr-directory-map-style',
            plugin_dir_url(__FILE__) . 'assets/style.css',
            array('leaflet-css', 'udm-font-awesome'),
            $this->version
        );
        
        // Localize script for AJAX and settings
        wp_localize_script('sfr-directory-map', 'sfrdmSettings', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sfrdm_nonce'),
            'default_lat' => (float) get_option('sfrdm_default_lat', 54.5),
            'default_lng' => (float) get_option('sfrdm_default_lng', -2.0),
            'default_zoom' => (int) get_option('sfrdm_default_zoom', 6),
            'button_colors' => array(
                'primary' => get_option('sfrdm_button_primary_color', '#0073aa'),
                'secondary' => get_option('sfrdm_button_secondary_color', '#005a87'),
                'hover' => get_option('sfrdm_button_hover_color', '#004466'),
                'text' => get_option('sfrdm_button_text_color', '#ffffff'),
            ),
            'defaultColors' => get_option('sfrdm_category_colors', array()),
            'strings' => array(
                'loading' => __('Loading map...', 'sfr-directory-map'),
                'no_listings' => __('No listings found with coordinates.', 'sfr-directory-map'),
                'error_loading' => __('Failed to load directory listings.', 'sfr-directory-map'),
                'address' => __('Address:', 'sfr-directory-map'),
                'phone' => __('Phone:', 'sfr-directory-map'),
                'categories' => __('Categories:', 'sfr-directory-map'),
            ),
            'analytics_nonce' => wp_create_nonce('sfrdm_analytics_nonce')
        ));
    }
    
    /**
     * Enqueue scripts for admin pages
     */
    public function enqueue_admin_scripts($hook) {
        // Load on map edit pages
        if ($hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }
        
        global $post;
        if (!$post || $post->post_type !== 'sfrdm_map') {
            return;
        }
        
        // Enqueue Leaflet CSS
        wp_enqueue_style(
            'leaflet-css', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet/leaflet.css', 
            array(), 
            $this->version
        );
        
        // Enqueue Leaflet JS
        wp_enqueue_script(
            'leaflet-js', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet/leaflet.js', 
            array(), 
            $this->version
        );
        
        // Enqueue MarkerCluster CSS
        wp_enqueue_style(
            'leaflet-markercluster-css', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet-markercluster/MarkerCluster.css', 
            array('leaflet-css'), 
            $this->version
        );
        
        wp_enqueue_style(
            'leaflet-markercluster-default-css', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet-markercluster/MarkerCluster.Default.css', 
            array('leaflet-markercluster-css'), 
            $this->version
        );
        
        // Enqueue MarkerCluster JS
        wp_enqueue_script(
            'leaflet-markercluster-js', 
            plugin_dir_url(__FILE__) . 'assets/vendor/leaflet-markercluster/leaflet.markercluster.js', 
            array('leaflet-js'), 
            $this->version
        );
        
        // Enqueue our map script
        wp_enqueue_script(
            'sfr-directory-map', 
            plugin_dir_url(__FILE__) . 'assets/map.js', 
            array('leaflet-js', 'leaflet-markercluster-js', 'jquery'),
            $this->version,
            true
        );
        
        // Localize script with settings
        // Enqueue metabox interaction script
        wp_enqueue_script(
            'sfrdm-admin-metabox',
            plugin_dir_url(__FILE__) . 'assets/admin-metabox.js',
            array('jquery'),
            $this->version,
            true
        );
        
        wp_localize_script('sfr-directory-map', 'sfrdmSettings', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sfrdm_nonce'),
            'default_lat' => (float) get_option('sfrdm_default_lat', 54.5),
            'default_lng' => (float) get_option('sfrdm_default_lng', -2.0),
            'default_zoom' => (int) get_option('sfrdm_default_zoom', 6),
            'button_colors' => array(
                'primary' => get_option('sfrdm_button_primary_color', '#0073aa'),
                'secondary' => get_option('sfrdm_button_secondary_color', '#005a87'),
                'hover' => get_option('sfrdm_button_hover_color', '#004466'),
                'text' => get_option('sfrdm_button_text_color', '#ffffff'),
            ),
            'defaultColors' => get_option('sfrdm_category_colors', array()),
            'strings' => array(
                'loading' => __('Loading map...', 'sfr-directory-map'),
                'no_listings' => __('No listings found with coordinates.', 'sfr-directory-map'),
                'error_loading' => __('Failed to load directory listings.', 'sfr-directory-map'),
                'address' => __('Address:', 'sfr-directory-map'),
                'phone' => __('Phone:', 'sfr-directory-map'),
                'categories' => __('Categories:', 'sfr-directory-map'),
            ),
            'analytics_nonce' => wp_create_nonce('sfrdm_analytics_nonce')
        ));
    }
    
    public function sfrdm_render_map_shortcode($atts) {
        // Sanitize and validate attributes
        $atts = shortcode_atts(array(
            'id' => '',
            'height' => '',
            'show_search' => '',
            'show_filters' => ''
        ), $atts, 'sfr_directory_map');
        
        // If no ID provided, fall back to legacy mode
        if (empty($atts['id'])) {
            return $this->render_legacy_map_shortcode($atts);
        }
        
        // Get map configuration
        $map_id = intval($atts['id']);
        $map_post = get_post($map_id);
        
        if (!$map_post || $map_post->post_type !== 'sfrdm_map') {
            return '<div class="udm-error">' . esc_html__('Map not found.', 'sfr-directory-map') . '</div>';
        }
        
        // Get map settings
        $map_settings = get_post_meta($map_id, '_sfrdm_map_settings', true);
        if (empty($map_settings)) {
            $map_settings = $this->get_default_map_settings();
        }
        
        // Override settings with shortcode attributes
        if (!empty($atts['height'])) {
            $map_settings['height'] = sanitize_text_field($atts['height']);
        }
        if (!empty($atts['show_search'])) {
            $map_settings['show_search'] = filter_var($atts['show_search'], FILTER_VALIDATE_BOOLEAN);
        }
        if (!empty($atts['show_filters'])) {
            $map_settings['show_filters'] = filter_var($atts['show_filters'], FILTER_VALIDATE_BOOLEAN);
        }
        
        // Validate height
        if (!preg_match('/^\d+(px|em|rem|%|vh)$/', $map_settings['height'])) {
            $map_settings['height'] = '500px';
        }
        
        $unique_map_id = 'udm-map-' . wp_generate_uuid4();
        
        ob_start();
        ?>
        <?php 
        $frame_class = 'udm-map-container';
        $frame_style = '';
        
        if (!empty($map_settings['enhanced_frame'])) {
            $frame_class .= ' udm-map-frame-premium';
            
            // Build custom CSS variables for frame styling
            $frame_style = ' style="';
            $frame_color = $map_settings['frame_color'] ?? '#0073aa';
            $frame_style .= '--udm-frame-color: ' . esc_attr($frame_color) . '; ';
            $frame_style .= '--udm-frame-width: ' . esc_attr($map_settings['frame_width'] ?? '3') . 'px; ';
            $frame_style .= '--udm-frame-radius: ' . esc_attr($map_settings['frame_radius'] ?? '12') . 'px; ';
            $frame_style .= '--udm-frame-background: ' . esc_attr($map_settings['frame_background'] ?? '#ffffff') . '; ';
            $frame_style .= '--udm-frame-gradient: ' . esc_attr($map_settings['frame_background'] ?? '#f8f9fa') . '; ';
            $shadow_intensity = ($map_settings['shadow_intensity'] ?? 20) / 100;
            $shadow_blur = $map_settings['shadow_blur'] ?? 20;
            $frame_color_rgb = sscanf($frame_color, "#%02x%02x%02x");
            $frame_style .= '--udm-frame-shadow: 0 4px ' . $shadow_blur . 'px rgba(' . $frame_color_rgb[0] . ',' . $frame_color_rgb[1] . ',' . $frame_color_rgb[2] . ',' . $shadow_intensity . '); ';
            $frame_style .= '--udm-frame-hover-shadow: 0 6px ' . ($shadow_blur + 5) . 'px rgba(' . $frame_color_rgb[0] . ',' . $frame_color_rgb[1] . ',' . $frame_color_rgb[2] . ',' . min(1, $shadow_intensity + 0.1) . '); ';
            // Add button colors to match frame color
            $frame_style .= '--udm-primary-color: ' . esc_attr($frame_color) . '; ';
            // Create a darker shade for secondary color (darken by 15%)
            $darker_color = sprintf("#%02x%02x%02x", max(0, $frame_color_rgb[0] - 30), max(0, $frame_color_rgb[1] - 30), max(0, $frame_color_rgb[2] - 30));
            $frame_style .= '--udm-secondary-color: ' . esc_attr($darker_color) . '; ';
            $frame_style .= '"';
        }
        ?>
        <?php if (defined('WP_DEBUG') && WP_DEBUG): ?>
        <!-- Debug: Frame Class: <?php echo esc_attr($frame_class); ?> -->
        <!-- Debug: Frame Style: <?php echo esc_attr($frame_style); ?> -->
        <?php endif; ?>
        <div class="<?php echo esc_attr($frame_class); ?>" data-map-config="<?php echo esc_attr(wp_json_encode($map_settings)); ?>"<?php echo wp_kses_post($frame_style); ?>>
            <?php if (!empty($map_post->post_title)): ?>
                <div class="udm-map-header">
                    <h3 class="udm-map-title"><?php echo esc_html($map_post->post_title); ?></h3>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($map_post->post_content)): ?>
                <div class="udm-map-description">
                    <?php echo wp_kses_post($map_post->post_content); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($map_settings['show_search']): ?>
                <div class="udm-search-container">
                    <div class="udm-search-input-wrapper">
                        <input type="text" id="udm-search-<?php echo esc_attr($unique_map_id); ?>" 
                               placeholder="<?php esc_attr_e('Search listings...', 'sfr-directory-map'); ?>" 
                               class="udm-search-input">
                        <button type="button" class="udm-clear-search-button" 
                                title="<?php esc_attr_e('Clear Search', 'sfr-directory-map'); ?>">
                            <span class="udm-clear-icon">✕</span>
                        </button>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($map_settings['show_filters']): ?>
                <div class="udm-filters-container">
                    <select id="udm-category-filter-<?php echo esc_attr($unique_map_id); ?>" class="udm-category-filter">
                        <option value=""><?php esc_html_e('All Categories', 'sfr-directory-map'); ?></option>
                        <?php
                        $categories = $this->get_available_categories($map_settings['post_type']);
                        foreach ($categories as $category) {
                            echo '<option value="' . esc_attr($category->slug) . '">' . esc_html($category->name) . '</option>';
                        }
                        ?>
                    </select>
                </div>
            <?php endif; ?>
            
            <div id="<?php echo esc_attr($unique_map_id); ?>" 
                 class="udm-map"
                 style="height: <?php echo esc_attr($map_settings['height']); ?>;" 
                 data-map-id="<?php echo esc_attr($map_id); ?>"
                 data-unique-id="<?php echo esc_attr($unique_map_id); ?>"
                 data-show-phone="<?php echo (!empty($map_settings['show_phone']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-show-email="<?php echo (!empty($map_settings['show_email']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-show-website="<?php echo (!empty($map_settings['show_website']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-show-tags="<?php echo (!empty($map_settings['show_tags']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-frame-color="<?php echo !empty($map_settings['enhanced_frame']) ? esc_attr($map_settings['frame_color'] ?? '#0073aa') : ''; ?>">
                <div class="udm-loading">
                    <p><?php esc_html_e('Loading map...', 'sfr-directory-map'); ?></p>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Legacy shortcode support for backward compatibility
     */
    private function render_legacy_map_shortcode($atts) {
        // Sanitize and validate attributes
        $atts = shortcode_atts(array(
            'height' => '500px',
            'category' => '',
            'post_type' => 'auto',
            'cluster' => 'true'
        ), $atts, 'sfr_directory_map');
        
        // Validate height
        if (!preg_match('/^\d+(px|em|rem|%|vh)$/', $atts['height'])) {
            $atts['height'] = '500px';
        }
        
        // Sanitize inputs
        $category = sanitize_text_field($atts['category']);
        $post_type = sanitize_text_field($atts['post_type']);
        $cluster = filter_var($atts['cluster'], FILTER_VALIDATE_BOOLEAN);
        $height = esc_attr($atts['height']);
        
        $unique_map_id = 'udm-map-' . wp_generate_uuid4();
        
        ob_start();
        ?>
        <?php 
        $frame_class = 'udm-map-container';
        $frame_style = '';
        
        // Apply enhanced frame styling if enabled in global settings
        $global_settings = get_option('sfrdm_settings', array());
        if (!empty($global_settings['enhanced_frame'])) {
            $frame_class .= ' udm-map-frame-premium';
            
            // Build custom CSS variables for frame styling
            $frame_style = ' style="';
            $frame_style .= '--udm-frame-color: ' . esc_attr($global_settings['frame_color'] ?? '#0073aa') . '; ';
            $frame_style .= '--udm-frame-width: ' . esc_attr($global_settings['frame_width'] ?? '3') . 'px; ';
            $frame_style .= '--udm-frame-radius: ' . esc_attr($global_settings['frame_radius'] ?? '12') . 'px; ';
            $frame_style .= '--udm-frame-background: ' . esc_attr($global_settings['frame_background'] ?? '#ffffff') . '; ';
            $frame_style .= '--udm-frame-gradient: ' . esc_attr($global_settings['frame_background'] ?? '#f8f9fa') . '; ';
            $shadow_intensity = ($global_settings['shadow_intensity'] ?? 20) / 100;
            $shadow_blur = $global_settings['shadow_blur'] ?? 20;
            $frame_color_rgb = sscanf($global_settings['frame_color'] ?? '#0073aa', "#%02x%02x%02x");
            $frame_style .= '--udm-frame-shadow: 0 4px ' . $shadow_blur . 'px rgba(' . $frame_color_rgb[0] . ',' . $frame_color_rgb[1] . ',' . $frame_color_rgb[2] . ',' . $shadow_intensity . '); ';
            $frame_style .= '--udm-frame-hover-shadow: 0 6px ' . ($shadow_blur + 5) . 'px rgba(' . $frame_color_rgb[0] . ',' . $frame_color_rgb[1] . ',' . $frame_color_rgb[2] . ',' . min(1, $shadow_intensity + 0.1) . ');';
            $frame_style .= '"';
        }
        ?>
        <div class="<?php echo esc_attr($frame_class); ?>"<?php echo wp_kses_post($frame_style); ?>>
            <div id="<?php echo esc_attr($map_id); ?>" 
                 class="udm-map"
                 style="height: <?php echo esc_attr($height); ?>;" 
                 data-category="<?php echo esc_attr($category); ?>" 
                 data-post-type="<?php echo esc_attr($post_type); ?>"
                 data-cluster="<?php echo $cluster ? '1' : '0'; ?>"
                 data-map-id="<?php echo esc_attr($map_id); ?>"
                 data-show-phone="<?php echo (!empty($global_settings['show_phone']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-show-email="<?php echo (!empty($global_settings['show_email']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-show-website="<?php echo (!empty($global_settings['show_website']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>"
                 data-show-tags="<?php echo (!empty($global_settings['show_tags']) && $this->is_premium_feature_available()) ? '1' : '0'; ?>">
                <div class="udm-loading">
                    <p><?php esc_html_e('Loading map...', 'sfr-directory-map'); ?></p>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    public function sfrdm_get_listings_ajax() {
        // Check if request method is POST
        if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') {
            wp_send_json_error(array(
                'message' => __('Invalid request method.', 'sfr-directory-map')
            ));
        }
        
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'sfrdm_nonce')) {
            wp_send_json_error(array(
                'message' => __('Security check failed.', 'sfr-directory-map')
            ));
        }
        
        // Sanitize inputs
        $category = isset($_POST['category']) ? sanitize_text_field(wp_unslash($_POST['category'])) : '';
        $post_type = isset($_POST['post_type']) ? sanitize_text_field(wp_unslash($_POST['post_type'])) : 'auto';
        $map_id = isset($_POST['map_id']) ? intval(wp_unslash($_POST['map_id'])) : 0;
        
        try {
            // If map_id is provided, get settings from map configuration
            if ($map_id > 0) {
                $map_settings = get_post_meta($map_id, '_sfrdm_map_settings', true);
                if (!empty($map_settings)) {
                    $post_type = $map_settings['post_type'];
                    // Auto-detect directory plugin if post_type is 'auto'
                    if ($post_type === 'auto') {
                        $post_type = $this->detect_directory_plugin();
                    }
                    // Use map's category filter if no specific category requested
                    if (empty($category) && !empty($map_settings['categories'])) {
                        // For now, we'll use the first category or implement multi-category support later
                        $category = $map_settings['categories'][0] ?? '';
                    }
                }
            }
            
            $listings = $this->get_directory_listings($category, $post_type);
            
            if ($listings === false) {
                wp_send_json_error(array(
                    'message' => __('No compatible directory plugin detected.', 'sfr-directory-map'),
                    'code' => 'no_plugin'
                ));
            }
            
            if (empty($listings)) {
                wp_send_json_error(array(
                    'message' => __('No listings found with coordinates.', 'sfr-directory-map'),
                    'code' => 'no_listings'
                ));
            }
            
            // Get map settings for pin colours and tags
            $map_settings = array();
            if ($map_id > 0) {
                $map_settings = get_post_meta($map_id, '_sfrdm_map_settings', true);
            }
            
            // Add pin colours, icons, and tags to listings
            $enhanced_listings = array();
            foreach ($listings as $listing) {
                $listing['pin_color'] = $this->get_pin_color_for_category(
                    $listing['category_slug'] ?? 'default',
                    $map_settings['pin_colors'] ?? array()
                );
                
                
                // Add tags if available
                if (!empty($map_settings['show_tags'])) {
                    $listing['tags'] = $this->get_listing_tags($listing['id'], $post_type);
                }
                
                $enhanced_listings[] = $listing;
            }
            
            wp_send_json_success(array(
                'listings' => $enhanced_listings,
                'count' => count($enhanced_listings),
                'pin_colors' => $map_settings['pin_colors'] ?? array(),
                'default_colors' => $this->get_default_pin_colors()
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => __('An error occurred while loading listings.', 'sfr-directory-map'),
                'code' => 'general_error'
            ));
        }
    }
    
    private function get_directory_listings($category = '', $post_type = 'auto') {
        // Auto-detect directory plugin if not specified
        if ($post_type === 'auto') {
            $post_type = $this->detect_directory_plugin();
        }
        
        if (!$post_type) {
            return false;
        }
        
        // Verify post type exists
        if (!post_type_exists($post_type)) {
            return false;
        }
        
        $args = array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => -1, // Unlimited listings
            'meta_query' => array(
                'relation' => 'AND'
            )
        );
        
        // Add coordinate filters and category based on plugin type
        switch ($post_type) {
            case 'at_biz_dir': // Directorist
                $args['meta_query'][] = array(
                    'key' => '_manual_lat',
                    'value' => '',
                    'compare' => '!='
                );
                $args['meta_query'][] = array(
                    'key' => '_manual_lng',
                    'value' => '',
                    'compare' => '!='
                );
                
                if (!empty($category)) {
                    $args['tax_query'] = array(
                        array(
                            'taxonomy' => 'at_biz_dir-category',
                            'field' => 'slug',
                            'terms' => $category
                        )
                    );
                }
                break;
                
            case 'gd_place': // GeoDirectory
                $args['meta_query'][] = array(
                    'key' => 'post_latitude',
                    'value' => '',
                    'compare' => '!='
                );
                $args['meta_query'][] = array(
                    'key' => 'post_longitude',
                    'value' => '',
                    'compare' => '!='
                );
                break;
                
            default:
                // Generic latitude/longitude fields
                $args['meta_query'][] = array(
                    'key' => 'latitude',
                    'value' => '',
                    'compare' => '!='
                );
                $args['meta_query'][] = array(
                    'key' => 'longitude',
                    'value' => '',
                    'compare' => '!='
                );
        }
        
        $posts = get_posts($args);
        $listings = array();
        
        foreach ($posts as $post) {
            $listing_data = $this->format_listing_data($post, $post_type);
            if ($listing_data && $this->validate_coordinates($listing_data['lat'], $listing_data['lng'])) {
                $listings[] = $listing_data;
            }
        }
        
        return $listings;
    }
    
    private function detect_directory_plugin() {
        // Use cached result if available
        if ($this->detected_plugin_cache !== null) {
            return $this->detected_plugin_cache;
        }
        
        // Check transient cache first
        $cached_plugin = get_transient('sfrdm_detected_plugin');
        if ($cached_plugin !== false) {
            $this->detected_plugin_cache = $cached_plugin;
            return $cached_plugin;
        }
        
        $detected_plugin = false;
        
        // Check for Directorist (updated class detection)
        if ((class_exists('Directorist_Base') || class_exists('Directorist\Directorist') || class_exists('ATBDP')) && post_type_exists('at_biz_dir')) {
            $detected_plugin = 'at_biz_dir';
        }
        
        // Cache the result
        $this->detected_plugin_cache = $detected_plugin;
        set_transient('sfrdm_detected_plugin', $detected_plugin, HOUR_IN_SECONDS);
        
        return $detected_plugin;
    }
    
    /**
     * Get default map settings
     */
    private function get_default_map_settings() {
        return array(
            'height' => '500px',
            'post_type' => 'auto',
            'map_style' => 'openstreetmap',
            'default_lat' => (float) get_option('sfrdm_default_lat', 54.5),
            'default_lng' => (float) get_option('sfrdm_default_lng', -2.0),
            'default_zoom' => (int) get_option('sfrdm_default_zoom', 6),
            'cluster_markers' => true,
            'auto_fit_bounds' => true, // Auto-fit to show all pins
            'show_search' => false,
            'show_filters' => false,
            'categories' => array(),
            'custom_marker' => '',
            'popup_style' => 'default',
            'max_listings' => 0, // 0 = unlimited
            'custom_pin_colors' => true,
            'pin_colors' => array(),
            'show_tags' => true,
            'show_phone' => true,
            'show_email' => true,
            'show_website' => true,
            'enhanced_frame' => false,
            'frame_color' => '#0073aa',
            'frame_width' => 3,
            'frame_radius' => 12,
            'frame_background' => '#ffffff',
            'shadow_intensity' => 20,
            'shadow_blur' => 20,
            'enable_analytics' => false,
        );
    }
    
    /**
     * Get default pin colours for categories
     */
    private function get_default_pin_colors() {
        return array(
            'default' => '#3388ff',
            'restaurant' => '#ff6b6b',
            'hotel' => '#4ecdc4',
            'shop' => '#45b7d1',
            'service' => '#96ceb4',
            'entertainment' => '#feca57',
            'health' => '#ff9ff3',
            'education' => '#54a0ff',
            'automotive' => '#5f27cd',
            'real-estate' => '#00d2d3'
        );
    }
    
    /**
     * Get pin colour for a category
     */
    private function get_pin_color_for_category($category_slug, $pin_colors = array()) {
        $default_colors = $this->get_default_pin_colors();
        
        // Check if custom colour is set
        if (isset($pin_colors[$category_slug])) {
            return $pin_colors[$category_slug];
        }
        
        // Check default colours
        if (isset($default_colors[$category_slug])) {
            return $default_colors[$category_slug];
        }
        
        // Return default blue
        return $default_colors['default'];
    }
    
    
    
    
    /**
     * Get tags for a listing
     */
    private function get_listing_tags($post_id, $post_type) {
        $tags = array();
        
        
        // Get tags based on post type
        switch ($post_type) {
            case 'at_biz_dir': // Directorist
                $directorist_tags = get_the_terms($post_id, 'at_biz_dir-tags');
                if (!is_wp_error($directorist_tags) && $directorist_tags) {
                    foreach ($directorist_tags as $tag) {
                        $tags[] = array(
                            'name' => $tag->name,
                            'slug' => $tag->slug,
                            'url' => get_term_link($tag)
                        );
                    }
                }
                break;
                
            case 'gd_place': // GeoDirectory
                $gd_tags = get_the_terms($post_id, 'gd_place_tags');
                if (!is_wp_error($gd_tags) && $gd_tags) {
                    foreach ($gd_tags as $tag) {
                        $tags[] = array(
                            'name' => $tag->name,
                            'slug' => $tag->slug,
                            'url' => get_term_link($tag)
                        );
                    }
                }
                break;
                
            default:
                // Generic WordPress tags
                $wp_tags = get_the_tags($post_id);
                if ($wp_tags) {
                    foreach ($wp_tags as $tag) {
                        $tags[] = array(
                            'name' => $tag->name,
                            'slug' => $tag->slug,
                            'url' => get_tag_link($tag->term_id)
                        );
                    }
                }
                break;
        }
        
        return $tags;
    }
    
    /**
     * Get available categories for a post type
     */
    private function get_available_categories($post_type) {
        if ($post_type === 'auto') {
            $post_type = $this->detect_directory_plugin();
        }
        
        $taxonomy = '';
        switch ($post_type) {
            case 'at_biz_dir':
                $taxonomy = 'at_biz_dir-category';
                break;
            case 'gd_place':
                $taxonomy = 'gd_placecategory';
                break;
            default:
                return array();
        }
        
        $categories = get_terms(array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
        ));
        
        // Organize categories hierarchically
        return $this->organize_categories_hierarchically($categories);
    }
    
    /**
     * Organize categories hierarchically
     */
    private function organize_categories_hierarchically($categories) {
        if (empty($categories) || is_wp_error($categories)) {
            return array();
        }
        
        $parent_categories = array();
        $child_categories = array();
        
        foreach ($categories as $category) {
            if ($category->parent == 0) {
                $parent_categories[] = $category;
            } else {
                $child_categories[] = $category;
            }
        }
        
        // Sort parent categories alphabetically
        usort($parent_categories, function($a, $b) {
            return strcasecmp($a->name, $b->name);
        });
        
        // Sort child categories alphabetically
        usort($child_categories, function($a, $b) {
            return strcasecmp($a->name, $b->name);
        });
        
        // Build hierarchical structure
        $hierarchical_categories = array();
        
        foreach ($parent_categories as $parent) {
            $hierarchical_categories[] = $parent;
            
            // Add children of this parent
            foreach ($child_categories as $child) {
                if ($child->parent == $parent->term_id) {
                    $hierarchical_categories[] = $child;
                }
            }
        }
        
        return $hierarchical_categories;
    }
    
    /**
     * Map settings meta box
     */
    public function map_settings_meta_box($post) {
        wp_nonce_field('sfrdm_map_meta', 'sfrdm_map_meta_nonce');
        
        $settings = get_post_meta($post->ID, '_sfrdm_map_settings', true);
        if (empty($settings)) {
            $settings = $this->get_default_map_settings();
        } else {
            // Merge with defaults to ensure all new keys exist
            $settings = array_merge($this->get_default_map_settings(), $settings);
        }
        
        $detected_plugin = $this->detect_directory_plugin();
        $available_categories = $this->get_available_categories($settings['post_type']);
        ?>
        <div class="udm-map-settings">
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="sfrdm_height"><?php esc_html_e('Map Height', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="sfrdm_height" name="sfrdm_height" 
                               value="<?php echo esc_attr($settings['height']); ?>" 
                               placeholder="500px" class="regular-text" />
                        <p class="description"><?php esc_html_e('Map height (e.g., 500px, 50vh)', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="sfrdm_post_type"><?php esc_html_e('Directory Plugin', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <select id="sfrdm_post_type" name="sfrdm_post_type">
                            <option value="auto" <?php selected($settings['post_type'], 'auto'); ?>>
                                <?php esc_html_e('Auto-detect', 'sfr-directory-map'); ?>
                                <?php if ($detected_plugin): ?>
                                    (<?php echo esc_html(ucwords(str_replace('_', ' ', $detected_plugin))); ?>)
                                <?php endif; ?>
                            </option>
                            <option value="at_biz_dir" <?php selected($settings['post_type'], 'at_biz_dir'); ?>>
                                <?php esc_html_e('Directorist', 'sfr-directory-map'); ?>
                            </option>
                            <option value="gd_place" <?php selected($settings['post_type'], 'gd_place'); ?>>
                                <?php esc_html_e('GeoDirectory', 'sfr-directory-map'); ?>
                            </option>
                        </select>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="sfrdm_map_style"><?php esc_html_e('Map Style', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <select id="sfrdm_map_style" name="sfrdm_map_style">
                            <!-- Free Map Styles -->
                            <option value="openstreetmap" <?php selected($settings['map_style'], 'openstreetmap'); ?>>
                                <?php esc_html_e('OpenStreetMap', 'sfr-directory-map'); ?>
                            </option>
                            <option value="cartodb_positron" <?php selected($settings['map_style'], 'cartodb_positron'); ?>>
                                <?php esc_html_e('CartoDB Positron', 'sfr-directory-map'); ?>
                            </option>
                            <option value="cartodb_dark" <?php selected($settings['map_style'], 'cartodb_dark'); ?>>
                                <?php esc_html_e('CartoDB Dark Matter', 'sfr-directory-map'); ?>
                            </option>
                        </select>
                        <p class="description">
                            <?php esc_html_e('Some map styles may appear blank in certain areas or have limited coverage. The system will automatically fall back to OpenStreetMap if tiles fail to load.', 'sfr-directory-map'); ?>
                            <br><strong>Debug:</strong> Check browser console (F12) to see tile loading information and any errors.
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="sfrdm_categories"><?php esc_html_e('Categories', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <div class="udm-categories-container" style="max-height: 200px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;">
                            <?php if (!empty($available_categories)): ?>
                                <?php foreach ($available_categories as $category): ?>
                                    <label style="display: block; margin-bottom: 5px; <?php echo ($category->parent > 0) ? 'margin-left: 20px;' : ''; ?>">
                                        <input type="checkbox" name="sfrdm_categories[]" value="<?php echo esc_attr($category->slug); ?>"
                                               <?php checked(in_array($category->slug, $settings['categories'])); ?> />
                                        <span style="<?php echo ($category->parent > 0) ? 'font-style: italic; color: #666;' : ''; ?>">
                                            <?php echo esc_html($category->name); ?>
                                        </span>
                                        <span style="color: #666;">(<?php echo esc_html($category->count); ?>)</span>
                                    </label>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <p><?php esc_html_e('No categories found. Make sure you have listings with categories.', 'sfr-directory-map'); ?></p>
                            <?php endif; ?>
                        </div>
                        <p class="description"><?php esc_html_e('Leave empty to show all categories', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><?php esc_html_e('Features', 'sfr-directory-map'); ?></th>
                    <td>
                        <label style="display: block; margin-bottom: 10px;">
                            <input type="checkbox" name="sfrdm_cluster_markers" value="1" 
                                   <?php checked(!empty($settings['cluster_markers'])); ?> />
                            <?php esc_html_e('Enable marker clustering', 'sfr-directory-map'); ?>
                        </label>
                        
                        <label style="display: block; margin-bottom: 10px;">
                            <input type="checkbox" name="sfrdm_show_search" value="1" 
                                   <?php checked(!empty($settings['show_search'])); ?> />
                            <?php esc_html_e('Show search box', 'sfr-directory-map'); ?>
                        </label>
                        
                        <label style="display: block; margin-bottom: 10px;">
                            <input type="checkbox" name="sfrdm_show_filters" value="1" 
                                   <?php checked(!empty($settings['show_filters'])); ?> />
                            <?php esc_html_e('Show category filters', 'sfr-directory-map'); ?>
                        </label>
                        
                        <label style="display: block; margin-bottom: 10px;">
                            <input type="checkbox" name="sfrdm_custom_pin_colors" value="1" 
                                   <?php checked(!empty($settings['custom_pin_colors'])); ?> />
                            <?php esc_html_e('Enable custom pin colours for categories', 'sfr-directory-map'); ?>
                        </label>
                        
                    </td>
                </tr>
                
                <?php if (!empty($settings['custom_pin_colors']) && !empty($available_categories)): ?>
                <tr>
                    <th scope="row"><?php esc_html_e('Pin Colours by Category', 'sfr-directory-map'); ?></th>
                    <td>
                        <div class="udm-pin-colors-container">
                            <p class="description"><?php esc_html_e('Set custom colours for each category. Leave empty to use default colours.', 'sfr-directory-map'); ?></p>
                            <div class="udm-pin-colors-grid">
                                <?php 
                                $default_colors = $this->get_default_pin_colors();
                                $custom_colors = !empty($settings['pin_colors']) ? $settings['pin_colors'] : array();
                                
                                foreach ($available_categories as $category): 
                                    $current_color = isset($custom_colors[$category->slug]) ? $custom_colors[$category->slug] : '';
                                    $default_color = isset($default_colors[$category->slug]) ? $default_colors[$category->slug] : $default_colors['default'];
                                    $display_color = !empty($current_color) ? $current_color : $default_color;
                                ?>
                                <div class="udm-pin-color-item">
                                    <label for="pin_color_<?php echo esc_attr($category->slug); ?>">
                                        <span class="udm-category-name"><?php echo esc_html($category->name); ?></span>
                                        <div class="udm-color-input-wrapper">
                                            <div class="udm-pin-preview" style="background-color: <?php echo esc_attr($display_color); ?>;"></div>
                                            <input type="color" 
                                                   id="pin_color_<?php echo esc_attr($category->slug); ?>" 
                                                   name="sfrdm_pin_colors[<?php echo esc_attr($category->slug); ?>]" 
                                                   value="<?php echo esc_attr($current_color); ?>"
                                                   data-default="<?php echo esc_attr($default_color); ?>"
                                                   class="udm-color-picker" />
                                            <button type="button" class="button udm-reset-color" 
                                                    data-target="pin_color_<?php echo esc_attr($category->slug); ?>">
                                                <?php esc_html_e('Reset', 'sfr-directory-map'); ?>
                                            </button>
                                        </div>
                                    </label>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </td>
                </tr>
                <?php endif; ?>
                
                <?php if ($this->is_premium_feature_available()): ?>
                <tr>
                    <th scope="row"><?php esc_html_e('Map Frame Styling', 'sfr-directory-map'); ?></th>
                    <td>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="sfrdm_frame_color"><?php esc_html_e('Frame Color', 'sfr-directory-map'); ?></label>
                                <input type="color" name="sfrdm_frame_color" value="<?php echo esc_attr($settings['frame_color'] ?? '#0073aa'); ?>" style="width: 100%; height: 40px; border: none; border-radius: 4px;" />
                            </div>
                            <div>
                                <label for="sfrdm_frame_width"><?php esc_html_e('Frame Width', 'sfr-directory-map'); ?></label>
                                <input type="range" name="sfrdm_frame_width" min="1" max="8" value="<?php echo esc_attr($settings['frame_width'] ?? '3'); ?>" style="width: 100%;" oninput="this.nextElementSibling.textContent = this.value + 'px'" />
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($settings['frame_width'] ?? '3'); ?>px</span>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                            <div>
                                <label for="sfrdm_frame_radius"><?php esc_html_e('Border Radius', 'sfr-directory-map'); ?></label>
                                <input type="range" name="sfrdm_frame_radius" min="0" max="20" value="<?php echo esc_attr($settings['frame_radius'] ?? '12'); ?>" style="width: 100%;" oninput="this.nextElementSibling.textContent = this.value + 'px'" />
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($settings['frame_radius'] ?? '12'); ?>px</span>
                            </div>
                            <div>
                                <label for="sfrdm_frame_background"><?php esc_html_e('Background Color', 'sfr-directory-map'); ?></label>
                                <input type="color" name="sfrdm_frame_background" value="<?php echo esc_attr($settings['frame_background'] ?? '#ffffff'); ?>" style="width: 100%; height: 40px; border: none; border-radius: 4px;" />
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-top: 15px;">
                            <div>
                                <label for="sfrdm_shadow_intensity"><?php esc_html_e('Shadow Intensity', 'sfr-directory-map'); ?></label>
                                <input type="range" name="sfrdm_shadow_intensity" min="0" max="50" value="<?php echo esc_attr($settings['shadow_intensity'] ?? '20'); ?>" style="width: 100%;" oninput="this.nextElementSibling.textContent = this.value + '%'" />
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($settings['shadow_intensity'] ?? '20'); ?>%</span>
                            </div>
                            <div>
                                <label for="sfrdm_shadow_blur"><?php esc_html_e('Shadow Blur', 'sfr-directory-map'); ?></label>
                                <input type="range" name="sfrdm_shadow_blur" min="0" max="30" value="<?php echo esc_attr($settings['shadow_blur'] ?? '20'); ?>" style="width: 100%;" oninput="this.nextElementSibling.textContent = this.value + 'px'" />
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($settings['shadow_blur'] ?? '20'); ?>px</span>
                            </div>
                        </div>
                        <p class="description"><?php esc_html_e('Customize the appearance of your map frame with colors, borders, and effects.', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                <?php endif; ?>
                
                
                <tr>
                    <th scope="row">
                        <label for="sfrdm_default_lat"><?php esc_html_e('Default Location', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <input type="number" step="any" id="sfrdm_default_lat" name="sfrdm_default_lat" 
                               value="<?php echo esc_attr($settings['default_lat']); ?>" 
                               placeholder="54.5" style="width: 120px;" /> 
                        <input type="number" step="any" id="sfrdm_default_lng" name="sfrdm_default_lng" 
                               value="<?php echo esc_attr($settings['default_lng']); ?>" 
                               placeholder="-2.0" style="width: 120px;" />
                        <p class="description"><?php esc_html_e('Latitude and Longitude for map center', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="sfrdm_default_zoom"><?php esc_html_e('Default Zoom', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <input type="number" min="1" max="18" id="sfrdm_default_zoom" name="sfrdm_default_zoom" 
                               value="<?php echo esc_attr($settings['default_zoom']); ?>" />
                        <p class="description"><?php esc_html_e('Initial zoom level (1-18)', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="sfrdm_auto_fit_bounds"><?php esc_html_e('Map Positioning', 'sfr-directory-map'); ?></label>
                    </th>
                    <td>
                        <label>
                            <input type="checkbox" id="sfrdm_auto_fit_bounds" name="sfrdm_auto_fit_bounds" value="1" 
                                   <?php checked(!empty($settings['auto_fit_bounds'])); ?> />
                            <?php esc_html_e('Auto-fit map to show all pins', 'sfr-directory-map'); ?>
                        </label>
                        <p class="description">
                            <?php esc_html_e('When enabled, the map will automatically zoom and center to display all pins. When disabled, the map will use the default location and zoom level specified above.', 'sfr-directory-map'); ?>
                        </p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }
    
    /**
     * Map preview meta box
     */
    public function map_preview_meta_box($post) {
        $map_settings = get_post_meta($post->ID, '_sfrdm_map_settings', true);
        if (empty($map_settings)) {
            $map_settings = $this->get_default_map_settings();
        }
        
        // Get sample listings for preview
        $sample_listings = $this->get_sample_listings_for_preview();
        
        ?>
        <div class="udm-map-preview">
            <div style="margin-bottom: 10px; padding: 8px; background: #f0f0f0; border-radius: 4px; font-size: 12px;">
                <strong><?php esc_html_e('Map Style:', 'sfr-directory-map'); ?></strong> 
                <span id="preview-map-style"><?php echo esc_html($map_settings['map_style'] ?? 'openstreetmap'); ?></span>
            </div>
            <div id="udm-admin-preview" style="height: 300px; border: 1px solid #ddd; background: #f9f9f9; position: relative;">
                <div id="udm-preview-loading" style="text-align: center; margin-top: 120px; color: #666;">
                    <?php esc_html_e('Loading map preview...', 'sfr-directory-map'); ?>
                </div>
            </div>
            <p class="description">
                <?php esc_html_e('Preview shows the selected map style with sample locations. Toggle the "Auto-fit map to show all pins" checkbox below to see how it affects the map view.', 'sfr-directory-map'); ?>
            </p>
            
            
        </div>
        <?php
    }
    
    private function get_sample_listings_for_preview() {
        // Get a few sample listings for preview
        $sample_listings = array();
        
        // Try to get real listings first
        $post_type = $this->get_detected_directory_post_type();
        if ($post_type) {
            $listings = get_posts(array(
                'post_type' => $post_type,
                'posts_per_page' => 5,
                'meta_query' => array(
                    array(
                        'key' => '_latitude',
                        'compare' => 'EXISTS'
                    ),
                    array(
                        'key' => '_longitude',
                        'compare' => 'EXISTS'
                    )
                )
            ));
            
            foreach ($listings as $listing) {
                $lat = get_post_meta($listing->ID, '_latitude', true);
                $lng = get_post_meta($listing->ID, '_longitude', true);
                if ($lat && $lng) {
                    $sample_listings[] = array(
                        'title' => $listing->post_title,
                        'lat' => floatval($lat),
                        'lng' => floatval($lng)
                    );
                }
            }
        }
        
        // If no real listings, use sample UK locations
        if (empty($sample_listings)) {
            $sample_listings = array(
                array('title' => 'London', 'lat' => 51.5074, 'lng' => -0.1278),
                array('title' => 'Manchester', 'lat' => 53.4808, 'lng' => -2.2426),
                array('title' => 'Birmingham', 'lat' => 52.4862, 'lng' => -1.8904),
                array('title' => 'Leeds', 'lat' => 53.8008, 'lng' => -1.5491),
                array('title' => 'Glasgow', 'lat' => 55.8642, 'lng' => -4.2518)
            );
        }
        
        return $sample_listings;
    }
    
    private function get_detected_directory_post_type() {
        // Check for Directorist
        if (post_type_exists('at_biz_dir')) {
            return 'at_biz_dir';
        }
        
        // Check for GeoDirectory
        if (post_type_exists('gd_place')) {
            return 'gd_place';
        }
        
        // Check for WP Job Manager
        if (post_type_exists('job_listing')) {
            return 'job_listing';
        }
        
        return false;
    }
    
    /**
     * Shortcode info meta box
     */
    public function shortcode_info_meta_box($post) {
        if ($post->ID) {
            ?>
            <div class="udm-shortcode-info">
                <p><strong><?php esc_html_e('Shortcode:', 'sfr-directory-map'); ?></strong></p>
                <input type="text" readonly value="[sfr_directory_map id=&quot;<?php echo esc_attr($post->ID); ?>&quot;]" 
                       style="width: 100%; font-family: monospace;" onclick="this.select();" />
                
                <p><strong><?php esc_html_e('With overrides:', 'sfr-directory-map'); ?></strong></p>
                <input type="text" readonly value="[sfr_directory_map id=&quot;<?php echo esc_attr($post->ID); ?>&quot; height=&quot;600px&quot; show_search=&quot;true&quot;]" 
                       style="width: 100%; font-family: monospace;" onclick="this.select();" />
                
                <p class="description">
                    <?php esc_html_e('Copy and paste this shortcode into any page or post to display your map.', 'sfr-directory-map'); ?>
                </p>
            </div>
            <?php
        } else {
            ?>
            <p><?php esc_html_e('Save the map to get the shortcode.', 'sfr-directory-map'); ?></p>
            <?php
        }
    }
    
    /**
     * Save map meta data
     */
    public function save_map_meta($post_id) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check if this is the correct post type
        if (get_post_type($post_id) !== 'sfrdm_map') {
            return;
        }
        
        // Check nonce
        if (!isset($_POST['sfrdm_map_meta_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sfrdm_map_meta_nonce'])), 'sfrdm_map_meta')) {
            return;
        }
        
        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save map settings
        $settings = array(
            'height' => isset($_POST['sfrdm_height']) ? sanitize_text_field(wp_unslash($_POST['sfrdm_height'])) : '500px',
            'post_type' => isset($_POST['sfrdm_post_type']) ? sanitize_text_field(wp_unslash($_POST['sfrdm_post_type'])) : 'auto',
            'map_style' => isset($_POST['sfrdm_map_style']) ? sanitize_text_field(wp_unslash($_POST['sfrdm_map_style'])) : 'openstreetmap',
            'default_lat' => isset($_POST['sfrdm_default_lat']) ? (float) sanitize_text_field(wp_unslash($_POST['sfrdm_default_lat'])) : 54.5,
            'default_lng' => isset($_POST['sfrdm_default_lng']) ? (float) sanitize_text_field(wp_unslash($_POST['sfrdm_default_lng'])) : -2.0,
            'default_zoom' => isset($_POST['sfrdm_default_zoom']) ? (int) sanitize_text_field(wp_unslash($_POST['sfrdm_default_zoom'])) : 6,
            'auto_fit_bounds' => isset($_POST['sfrdm_auto_fit_bounds']),
            'cluster_markers' => isset($_POST['sfrdm_cluster_markers']),
            'show_search' => isset($_POST['sfrdm_show_search']),
            'show_filters' => isset($_POST['sfrdm_show_filters']),
            'categories' => isset($_POST['sfrdm_categories']) ? array_map('sanitize_text_field', wp_unslash((array) $_POST['sfrdm_categories'])) : array(),
            'custom_pin_colors' => isset($_POST['sfrdm_custom_pin_colors']),
            'show_tags' => isset($_POST['sfrdm_show_tags']),
            'show_phone' => isset($_POST['sfrdm_show_phone']),
            'show_email' => isset($_POST['sfrdm_show_email']),
            'show_website' => isset($_POST['sfrdm_show_website']),
            'enhanced_frame' => isset($_POST['sfrdm_enhanced_frame']),
            'frame_color' => isset($_POST['sfrdm_frame_color']) ? sanitize_hex_color(wp_unslash($_POST['sfrdm_frame_color'])) : '#0073aa',
            'frame_width' => isset($_POST['sfrdm_frame_width']) ? max(1, min(8, intval(wp_unslash($_POST['sfrdm_frame_width'])))) : 3,
            'frame_radius' => isset($_POST['sfrdm_frame_radius']) ? max(0, min(20, intval(wp_unslash($_POST['sfrdm_frame_radius'])))) : 12,
            'frame_background' => isset($_POST['sfrdm_frame_background']) ? sanitize_hex_color(wp_unslash($_POST['sfrdm_frame_background'])) : '#ffffff',
            'shadow_intensity' => isset($_POST['sfrdm_shadow_intensity']) ? max(0, min(50, intval(wp_unslash($_POST['sfrdm_shadow_intensity'])))) : 20,
            'shadow_blur' => isset($_POST['sfrdm_shadow_blur']) ? max(0, min(30, intval(wp_unslash($_POST['sfrdm_shadow_blur'])))) : 20,
            'pin_colors' => isset($_POST['sfrdm_pin_colors']) ? array_map('sanitize_hex_color', wp_unslash((array) $_POST['sfrdm_pin_colors'])) : array(),
            'max_listings' => 0 // Unlimited
        );
        
        update_post_meta($post_id, '_sfrdm_map_settings', $settings);
    }
    
    private function format_listing_data($post, $post_type) {
        // Check if map is hidden for this listing (different plugins use different meta fields)
        $hide_map = false;
        switch ($post_type) {
            case 'at_biz_dir': // Directorist
                $hide_map = get_post_meta($post->ID, '_hide_map', true);
                break;
            case 'gd_place': // GeoDirectory
                $hide_map = get_post_meta($post->ID, '_hide_map', true) || get_post_meta($post->ID, 'hide_map', true);
                break;
            default:
                // Try common field names
                $hide_map = get_post_meta($post->ID, '_hide_map', true) || 
                           get_post_meta($post->ID, 'hide_map', true) || 
                           get_post_meta($post->ID, 'map_hidden', true);
        }
        
        if ($hide_map) {
            // Return false to exclude this listing from the map
            return false;
        }
        
        $data = array(
            'id' => $post->ID,
            'title' => get_the_title($post),
            'url' => get_permalink($post),
            'excerpt' => wp_trim_words(get_the_excerpt($post), 25, '...'),
            'image' => '', // Will be set based on plugin type
            'category_icon' => '', // Will be set based on plugin type
            'lat' => 0,
            'lng' => 0,
            'address' => '',
            'phone' => '',
            'email' => '', // Will be set based on plugin type
            'website' => '', // Will be set based on plugin type
            'categories' => array(),
            'tags' => $this->is_premium_feature_available() ? $this->get_listing_tags($post->ID, $post_type) : array() // Add tags (Premium Feature)
        );
        
        switch ($post_type) {
            case 'at_biz_dir': // Directorist
                $data['lat'] = (float) get_post_meta($post->ID, '_manual_lat', true);
                $data['lng'] = (float) get_post_meta($post->ID, '_manual_lng', true);
                $data['address'] = get_post_meta($post->ID, '_address', true);
                $data['phone'] = get_post_meta($post->ID, '_phone', true);
                $data['email'] = get_post_meta($post->ID, '_email', true);
                $data['website'] = get_post_meta($post->ID, '_website', true);
                $categories = wp_get_post_terms($post->ID, 'at_biz_dir-category', array('fields' => 'names'));
                $data['categories'] = is_array($categories) ? $categories : array();
                
                // Directorist image handling
                $data['image'] = $this->get_directorist_image($post->ID);
                
                // Directorist category icon handling
                $data['category_icon'] = $this->get_category_icon($post->ID, $post_type);
                
                // Add primary category name for display
                $primary_category = $this->get_primary_category($post->ID, $post_type);
                $data['category_name'] = $primary_category ? $primary_category->name : '';
                $data['category_slug'] = $primary_category ? $primary_category->slug : '';
                break;
                
            case 'gd_place': // GeoDirectory
                $data['lat'] = (float) get_post_meta($post->ID, 'post_latitude', true);
                $data['lng'] = (float) get_post_meta($post->ID, 'post_longitude', true);
                $data['address'] = get_post_meta($post->ID, 'post_address', true);
                $data['phone'] = get_post_meta($post->ID, 'geodir_contact', true);
                $data['email'] = get_post_meta($post->ID, 'geodir_email', true);
                $data['website'] = get_post_meta($post->ID, 'geodir_website', true);
                
                // GeoDirectory image handling
                $data['image'] = $this->get_geodirectory_image($post->ID);
                
                // GeoDirectory category icon handling
                $data['category_icon'] = $this->get_category_icon($post->ID, $post_type);
                
                // Add primary category name for display
                $primary_category = $this->get_primary_category($post->ID, $post_type);
                $data['category_name'] = $primary_category ? $primary_category->name : '';
                $data['category_slug'] = $primary_category ? $primary_category->slug : '';
                break;
                
            default:
                // Generic handling
                $data['lat'] = (float) get_post_meta($post->ID, 'latitude', true);
                $data['lng'] = (float) get_post_meta($post->ID, 'longitude', true);
                $data['address'] = get_post_meta($post->ID, 'address', true);
                $data['phone'] = get_post_meta($post->ID, 'phone', true);
                $data['email'] = get_post_meta($post->ID, 'email', true);
                $data['website'] = get_post_meta($post->ID, 'website', true);
                
                // Generic image handling
                $data['image'] = get_the_post_thumbnail_url($post, 'medium');
                
                // Generic category icon handling
                $data['category_icon'] = $this->get_category_icon($post->ID, $post_type);
                
                // Add primary category name for display
                $primary_category = $this->get_primary_category($post->ID, $post_type);
                $data['category_name'] = $primary_category ? $primary_category->name : '';
                $data['category_slug'] = $primary_category ? $primary_category->slug : '';
        }
        
        // Return false if no valid coordinates
        if (!$data['lat'] || !$data['lng']) {
            return false;
        }
        
        return $data;
    }
    
    /**
     * Get Directorist listing image
     */
    private function get_directorist_image($post_id) {
        // Try Directorist's preview image first (this is the main image)
        $preview_img = get_post_meta($post_id, '_listing_prv_img', true);
        if (!empty($preview_img)) {
            if (is_numeric($preview_img)) {
                $image_url = wp_get_attachment_image_url($preview_img, 'medium');
                if ($image_url) {
                    return $image_url;
                }
            } elseif (filter_var($preview_img, FILTER_VALIDATE_URL)) {
                return $preview_img;
            }
        }
        
        // Try Directorist's listing images array
        $listing_images = get_post_meta($post_id, '_listing_img', true);
        if (!empty($listing_images) && is_array($listing_images)) {
            $first_image = reset($listing_images);
            if (is_numeric($first_image)) {
                $image_url = wp_get_attachment_image_url($first_image, 'medium');
                if ($image_url) {
                    return $image_url;
                }
            } elseif (is_string($first_image) && filter_var($first_image, FILTER_VALIDATE_URL)) {
                return $first_image;
            }
        }
        
        // Try Directorist's listing image as single value
        if (!empty($listing_images) && !is_array($listing_images)) {
            if (is_numeric($listing_images)) {
                $image_url = wp_get_attachment_image_url($listing_images, 'medium');
                if ($image_url) {
                    return $image_url;
                }
            } elseif (filter_var($listing_images, FILTER_VALIDATE_URL)) {
                return $listing_images;
            }
        }
        
        // Try to get the featured image as fallback
        $image_url = get_the_post_thumbnail_url($post_id, 'medium');
        if ($image_url) {
            return $image_url;
        }
        
        // Try Directorist's gallery images (older method)
        $gallery_images = get_post_meta($post_id, '_listing_gallery', true);
        if (!empty($gallery_images) && is_array($gallery_images)) {
            $first_image = reset($gallery_images);
            if (is_numeric($first_image)) {
                $image_url = wp_get_attachment_image_url($first_image, 'medium');
                if ($image_url) {
                    return $image_url;
                }
            } elseif (is_string($first_image) && filter_var($first_image, FILTER_VALIDATE_URL)) {
                return $first_image;
            }
        }
        
        return '';
    }
    
    /**
     * Get GeoDirectory listing image
     */
    private function get_geodirectory_image($post_id) {
        // Try to get the featured image first
        $image_url = get_the_post_thumbnail_url($post_id, 'medium');
        if ($image_url) {
            return $image_url;
        }
        
        // Try GeoDirectory's post images
        $post_images = get_post_meta($post_id, 'post_images', true);
        if (!empty($post_images)) {
            if (is_array($post_images)) {
                $first_image = reset($post_images);
                if (is_numeric($first_image)) {
                    $image_url = wp_get_attachment_image_url($first_image, 'medium');
                    if ($image_url) {
                        return $image_url;
                    }
                } elseif (is_string($first_image) && filter_var($first_image, FILTER_VALIDATE_URL)) {
                    return $first_image;
                }
            } elseif (is_string($post_images) && filter_var($post_images, FILTER_VALIDATE_URL)) {
                return $post_images;
            }
        }
        
        // Try GeoDirectory's featured image meta
        $featured_image = get_post_meta($post_id, 'featured_image', true);
        if (!empty($featured_image)) {
            if (is_numeric($featured_image)) {
                $image_url = wp_get_attachment_image_url($featured_image, 'medium');
                if ($image_url) {
                    return $image_url;
                }
            } elseif (filter_var($featured_image, FILTER_VALIDATE_URL)) {
                return $featured_image;
            }
        }
        
        return '';
    }
    
    /**
     * Get post type for detected plugin
     */
    private function get_post_type_for_plugin($detected_plugin) {
        switch ($detected_plugin) {
            case 'directorist':
                return 'at_biz_dir';
            case 'geodirectory':
                return 'gd_place';
            case 'business_directory':
                return 'wpbdp_listing';
            default:
                return 'at_biz_dir'; // Default to Directorist
        }
    }
    
    /**
     * Analytics page handler
     */
    public function analytics_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'sfr-directory-map'));
        }
        
        // Include and display the free analytics page
        require_once plugin_dir_path(__FILE__) . 'includes/analytics-page-free.php';
    }
    
    /**
     * Display analytics upgrade notice
     */
    public function display_analytics_upgrade_notice() {
        ?>
        <div class="udm-premium-notice">
            <div class="udm-premium-badge"><?php esc_html_e('Premium Feature', 'sfr-directory-map'); ?></div>
            <h3><?php esc_html_e('Advanced Directory Analytics', 'sfr-directory-map'); ?></h3>
            <p><?php esc_html_e('Get detailed insights about your directory listings with advanced analytics:', 'sfr-directory-map'); ?></p>
            <ul>
                <li><?php esc_html_e('Listing completion statistics', 'sfr-directory-map'); ?></li>
                <li><?php esc_html_e('Category breakdown with image and location data', 'sfr-directory-map'); ?></li>
                <li><?php esc_html_e('Default location and image detection', 'sfr-directory-map'); ?></li>
                <li><?php esc_html_e('Hidden map statistics', 'sfr-directory-map'); ?></li>
            </ul>
            <p><a href="https://supportfromrichard.co.uk/plugins/sfr-directory-map/" target="_blank" class="button button-primary">
                <?php esc_html_e('Upgrade to Premium', 'sfr-directory-map'); ?>
            </a></p>
        </div>
        <?php
    }
    
    /**
     * Display analytics for directory listings
     */
    public function display_analytics() {
        $detected_plugin = $this->detect_directory_plugin();
        
        if (!$detected_plugin) {
            echo '<p style="color: #d63638;">' . esc_html__('No compatible directory plugin detected. Analytics unavailable.', 'sfr-directory-map') . '</p>';
            return;
        }
        
        $post_type = $this->get_post_type_for_plugin($detected_plugin);
        $analytics = $this->get_directory_analytics($post_type);
        
        if (empty($analytics)) {
            echo '<p>' . esc_html__('No directory listings found.', 'sfr-directory-map') . '</p>';
            return;
        }
        
        ?>
        <div class="udm-analytics-container">
            <div class="udm-analytics-summary">
                <h3><?php esc_html_e('Summary', 'sfr-directory-map'); ?></h3>
                <div class="udm-stats-grid">
                    <div class="udm-stat-box">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['total_listings']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('Total Listings', 'sfr-directory-map'); ?></div>
                    </div>
                    <div class="udm-stat-box">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['with_coordinates']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('With Location Data', 'sfr-directory-map'); ?></div>
                    </div>
                    <div class="udm-stat-box">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['with_images']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('With Images', 'sfr-directory-map'); ?></div>
                    </div>
                    <div class="udm-stat-box">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['categories_count']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('Categories', 'sfr-directory-map'); ?></div>
                    </div>
                </div>
                
                <?php if ($analytics['with_default_locations'] > 0 || $analytics['with_default_images'] > 0 || $analytics['with_hidden_maps'] > 0): ?>
                <div class="udm-stats-grid">
                    <h4><?php esc_html_e('Directorist Specific Analytics', 'sfr-directory-map'); ?></h4>
                    <?php if ($analytics['with_default_locations'] > 0): ?>
                    <div class="udm-stat-box udm-stat-warning">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['with_default_locations']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('Using Default Locations', 'sfr-directory-map'); ?></div>
                    </div>
                    <?php endif; ?>
                    <?php if ($analytics['with_default_images'] > 0): ?>
                    <div class="udm-stat-box udm-stat-warning">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['with_default_images']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('Using Default Images', 'sfr-directory-map'); ?></div>
                    </div>
                    <?php endif; ?>
                    <?php if ($analytics['with_hidden_maps'] > 0): ?>
                    <div class="udm-stat-box udm-stat-warning">
                        <div class="udm-stat-number"><?php echo esc_html($analytics['with_hidden_maps']); ?></div>
                        <div class="udm-stat-label"><?php esc_html_e('Maps Hidden', 'sfr-directory-map'); ?></div>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
            
            <div class="udm-analytics-categories">
                <h3><?php esc_html_e('Category Breakdown', 'sfr-directory-map'); ?></h3>
                <div class="udm-categories-table">
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Category', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('Total', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('With Location', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('With Images', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('Default Locations', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('Default Images', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('Hidden Maps', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('Map Ready', 'sfr-directory-map'); ?></th>
                                <th><?php esc_html_e('Completion %', 'sfr-directory-map'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($analytics['categories'] as $category): ?>
                            <tr>
                                <td><strong><?php echo esc_html($category['name']); ?></strong></td>
                                <td><?php echo esc_html($category['total']); ?></td>
                                <td>
                                    <span class="udm-status-badge <?php echo $category['with_coordinates'] > 0 ? 'udm-status-good' : 'udm-status-bad'; ?>">
                                        <?php echo esc_html($category['with_coordinates']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="udm-status-badge <?php echo $category['with_images'] > 0 ? 'udm-status-good' : 'udm-status-bad'; ?>">
                                        <?php echo esc_html($category['with_images']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="udm-status-badge <?php echo $category['with_default_locations'] > 0 ? 'udm-status-warning' : 'udm-status-neutral'; ?>">
                                        <?php echo esc_html($category['with_default_locations']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="udm-status-badge <?php echo $category['with_default_images'] > 0 ? 'udm-status-warning' : 'udm-status-neutral'; ?>">
                                        <?php echo esc_html($category['with_default_images']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="udm-status-badge <?php echo $category['with_hidden_maps'] > 0 ? 'udm-status-warning' : 'udm-status-neutral'; ?>">
                                        <?php echo esc_html($category['with_hidden_maps']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="udm-status-badge <?php echo $category['map_ready'] > 0 ? 'udm-status-excellent' : 'udm-status-bad'; ?>">
                                        <?php echo esc_html($category['map_ready']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="udm-completion-bar">
                                        <div class="udm-completion-fill" style="width: <?php echo esc_attr($category['completion_percentage']); ?>%"></div>
                                        <span class="udm-completion-text"><?php echo esc_html($category['completion_percentage']); ?>%</span>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        
        <?php
    }
    
    
    /**
     * Get category icon for a listing
     * Uses the same robust approach as Universal Category plugin
     */
    private function get_category_icon($listing_id, $post_type) {
        $icon = '';
        
        // Get the primary category for this listing
        $primary_category = $this->get_primary_category($listing_id, $post_type);
        
        if (!$primary_category) {
            return $icon;
        }
        
        
        // Get icon based on plugin type using the same logic as Universal Category plugin
        switch ($post_type) {
            case 'at_biz_dir':
                // Directorist - use the same approach as Universal Category plugin
                $icon_class = get_term_meta($primary_category->term_id, 'category_icon', true);
                
                // Debug: Check all meta fields for this category
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // error_log('UDM: category_icon meta value: ' . ($icon_class ?: 'EMPTY'));
                    
                    // Check all possible meta keys
                    $possible_keys = array('category_icon', 'icon', 'atbdp_category_icon', 'directorist_icon', 'term_icon', 'fa_icon');
                    foreach ($possible_keys as $key) {
                        $value = get_term_meta($primary_category->term_id, $key, true);
                        // error_log('UDM: Meta key "' . $key . '": ' . ($value ?: 'EMPTY'));
                    }
                }
                
                // Try alternative meta keys if category_icon is empty
                if (empty($icon_class)) {
                    $alternative_keys = array('icon', 'atbdp_category_icon', 'directorist_icon', 'term_icon');
                    foreach ($alternative_keys as $key) {
                        $icon_class = get_term_meta($primary_category->term_id, $key, true);
                        if (!empty($icon_class)) {
                            if (defined('WP_DEBUG') && WP_DEBUG) {
                                // error_log('UDM: Found icon in alternative key "' . $key . '": ' . $icon_class);
                            }
                            break;
                        }
                    }
                }
                
                if ($icon_class) {
                    // Debug logging
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        // error_log('UDM: Directorist icon_class found: ' . $icon_class . ' for category: ' . $primary_category->name);
                    }
                    
                    // For now, use the raw icon class directly instead of processing through directorist_icon()
                    // This avoids the issue where directorist_icon() returns generic fallbacks
                    $icon = $icon_class;
                    
                    // Debug logging
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        // error_log('UDM: Using raw icon class: ' . $icon);
                    }
                }
                break;
                
            case 'gd_place':
                // GeoDirectory - try image icons first, then font icons
                $icon_keys = ['ct_cat_icon', 'icon', 'category_icon', 'gd_category_icon', 'term_icon'];
                foreach ($icon_keys as $key) {
                    $icon_data = get_term_meta($primary_category->term_id, $key, true);
                    if (!empty($icon_data)) {
                        // Handle serialized array format (GeoDirectory style)
                        if (is_array($icon_data) && isset($icon_data['src'])) {
                            $icon_url = $icon_data['src'];
                            if (!empty($icon_url)) {
                                // Convert relative URL to absolute if needed
                                if (strpos($icon_url, 'http') !== 0) {
                                    if (strpos($icon_url, 'wp-content/uploads/') !== false) {
                                        $icon_url = home_url($icon_url);
                                    } else {
                                        $icon_url = home_url('/wp-content/uploads/' . $icon_url);
                                    }
                                }
                                $icon = $icon_url;
                                break;
                            }
                        } elseif (is_string($icon_data)) {
                            $icon = $icon_data;
                            break;
                        }
                    }
                }
                
                // If no image icon found, try font icon
                if (empty($icon)) {
                    $icon_font = get_term_meta($primary_category->term_id, 'ct_cat_font_icon', true);
                    if (!empty($icon_font)) {
                        $icon = $icon_font;
                    }
                }
                break;
                
            default:
                // For other plugins, try common meta fields
                $icon_keys = ['category_icon', 'icon', 'fa_icon', 'term_icon'];
                foreach ($icon_keys as $key) {
                    $icon = get_term_meta($primary_category->term_id, $key, true);
                    if (!empty($icon)) {
                        break;
                    }
                }
        }
        
        return $icon;
    }
    
    /**
     * Get user-friendly plugin name for display
     */
    private function get_plugin_display_name($plugin_slug) {
        $plugin_names = array(
            'at_biz_dir' => 'Directorist',
            'gd_place' => 'GeoDirectory',
            'wpbdp_listing' => 'Business Directory',
            'directorist' => 'Directorist',
            'geodirectory' => 'GeoDirectory',
            'business_directory' => 'Business Directory'
        );
        
        return isset($plugin_names[$plugin_slug]) ? $plugin_names[$plugin_slug] : ucwords(str_replace('_', ' ', $plugin_slug));
    }
    
    /**
     * Get the primary category for a listing
     */
    private function get_primary_category($listing_id, $post_type) {
        $taxonomy = '';
        
        // Determine taxonomy based on post type
        switch ($post_type) {
            case 'at_biz_dir':
                $taxonomy = 'at_biz_dir-category';
                break;
            case 'gd_place':
                $taxonomy = 'gd_placecategory';
                break;
            default:
                // Try common taxonomy patterns
                $taxonomy = $post_type . '_category';
        }
        
        if (empty($taxonomy)) {
            return null;
        }
        
        // Get categories for this listing
        $categories = get_the_terms($listing_id, $taxonomy);
        
        if (is_wp_error($categories) || empty($categories)) {
            return null;
        }
        
        // Smart category selection: prefer categories with more listings (more established)
        // This is better than just using the first category
        $selected_category = $categories[0];
        $max_count = $selected_category->count;
        
        foreach ($categories as $category) {
            if ($category->count > $max_count) {
                $selected_category = $category;
                $max_count = $category->count;
            }
        }
        
        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // error_log('UDM: Selected category: ' . $selected_category->name . ' (ID: ' . $selected_category->term_id . ') with ' . $selected_category->count . ' listings for listing ID: ' . $listing_id);
            // error_log('UDM: Total categories available: ' . count($categories));
        }
        
        return $selected_category;
    }
    
    /**
     * Display colour templates in admin
     */
    public function display_colour_templates() {
        $templates = get_option('sfrdm_colour_templates', array());
        
        if (empty($templates)) {
            echo '<p class="description">' . esc_html__('No templates saved yet.', 'sfr-directory-map') . '</p>';
            return;
        }
        
        echo '<div class="udm-templates-grid">';
        foreach ($templates as $id => $template) {
            echo '<div class="udm-template-item" data-template-id="' . esc_attr($id) . '">';
            echo '<div class="udm-template-colors">';
            echo '<div class="udm-color-preview" style="background: ' . esc_attr($template['primary']) . ';" title="Primary"></div>';
            echo '<div class="udm-color-preview" style="background: ' . esc_attr($template['secondary']) . ';" title="Secondary"></div>';
            echo '<div class="udm-color-preview" style="background: ' . esc_attr($template['hover']) . ';" title="Hover"></div>';
            echo '<div class="udm-color-preview" style="background: ' . esc_attr($template['text']) . '; border: 1px solid #ccc;" title="Text"></div>';
            echo '</div>';
            echo '<div class="udm-template-name">' . esc_html($template['name']) . '</div>';
            echo '<div class="udm-template-actions">';
            echo '<button type="button" class="button button-small load-template" data-template-id="' . esc_attr($id) . '">' . esc_html__('Load', 'sfr-directory-map') . '</button>';
            echo '<button type="button" class="button button-small delete-template" data-template-id="' . esc_attr($id) . '">' . esc_html__('Delete', 'sfr-directory-map') . '</button>';
            echo '</div>';
            echo '</div>';
        }
        echo '</div>';
    }
    
    /**
     * Save colour template via AJAX
     */
    public function sfrdm_save_colour_template_ajax() {
        check_ajax_referer('sfrdm_admin_nonce', 'nonce');
        
        // Check premium feature access
        if (!$this->is_premium_feature_available('color_templates')) {
            wp_send_json_error(array('message' => __('Colour templates are a premium feature.', 'sfr-directory-map')));
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $template_name = isset($_POST['template_name']) ? sanitize_text_field(wp_unslash($_POST['template_name'])) : '';
        $template_data = array(
            'name' => $template_name,
            'primary' => isset($_POST['primary_color']) ? sanitize_hex_color(wp_unslash($_POST['primary_color'])) : '#0073aa',
            'secondary' => isset($_POST['secondary_color']) ? sanitize_hex_color(wp_unslash($_POST['secondary_color'])) : '#005a87',
            'hover' => isset($_POST['hover_color']) ? sanitize_hex_color(wp_unslash($_POST['hover_color'])) : '#004466',
            'text' => isset($_POST['text_color']) ? sanitize_hex_color(wp_unslash($_POST['text_color'])) : '#ffffff',
            'created' => current_time('mysql')
        );
        
        $templates = get_option('sfrdm_colour_templates', array());
        $template_id = sanitize_title($template_name) . '_' . time();
        $templates[$template_id] = $template_data;
        
        update_option('sfrdm_colour_templates', $templates);
        
        wp_send_json_success(array(
            'message' => __('Template saved successfully!', 'sfr-directory-map'),
            'template_id' => $template_id
        ));
    }
    
    /**
     * Load colour template via AJAX
     */
    public function sfrdm_load_colour_template_ajax() {
        check_ajax_referer('sfrdm_admin_nonce', 'nonce');
        
        // Check premium feature access
        if (!$this->is_premium_feature_available('color_templates')) {
            wp_send_json_error(array('message' => __('Colour templates are a premium feature.', 'sfr-directory-map')));
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $template_id = isset($_POST['template_id']) ? sanitize_text_field(wp_unslash($_POST['template_id'])) : '';
        $templates = get_option('sfrdm_colour_templates', array());
        
        if (!isset($templates[$template_id])) {
            wp_send_json_error('Template not found');
        }
        
        $template = $templates[$template_id];
        
        // Update the current colour settings
        update_option('sfrdm_button_primary_color', $template['primary']);
        update_option('sfrdm_button_secondary_color', $template['secondary']);
        update_option('sfrdm_button_hover_color', $template['hover']);
        update_option('sfrdm_button_text_color', $template['text']);
        
        wp_send_json_success(array(
            'message' => __('Template loaded successfully!', 'sfr-directory-map'),
            'colors' => $template
        ));
    }
    
    /**
     * Delete colour template via AJAX
     */
    public function sfrdm_delete_colour_template_ajax() {
        check_ajax_referer('sfrdm_admin_nonce', 'nonce');
        
        // Check premium feature access
        if (!$this->is_premium_feature_available('color_templates')) {
            wp_send_json_error(array('message' => __('Colour templates are a premium feature.', 'sfr-directory-map')));
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $template_id = isset($_POST['template_id']) ? sanitize_text_field(wp_unslash($_POST['template_id'])) : '';
        $templates = get_option('sfrdm_colour_templates', array());
        
        if (!isset($templates[$template_id])) {
            wp_send_json_error('Template not found');
        }
        
        unset($templates[$template_id]);
        update_option('sfrdm_colour_templates', $templates);
        
        wp_send_json_success(array(
            'message' => __('Template deleted successfully!', 'sfr-directory-map')
        ));
    }
    
    /**
     * Debug function to check Directorist image data for a specific listing
     */
    public function debug_directorist_image($post_id) {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        echo '<h4>Debug Directorist Image Data for Post ID: ' . esc_html($post_id) . '</h4>';
        
        // Check all possible image meta fields
        $meta_fields = array(
            '_listing_prv_img',
            '_listing_img', 
            '_listing_gallery',
            '_thumbnail_id'
        );
        
        foreach ($meta_fields as $field) {
            $value = get_post_meta($post_id, $field, true);
            echo '<p><strong>' . esc_html($field) . ':</strong> ';
            if (is_array($value)) {
                echo 'Array: ' . esc_html(wp_json_encode($value));
            } else {
                echo esc_html($value);
            }
            echo '</p>';
        }
        
        // Check featured image
        $featured_image = get_the_post_thumbnail_url($post_id, 'medium');
        echo '<p><strong>Featured Image URL:</strong> ' . esc_html($featured_image) . '</p>';
        
        // Test our image function
        $our_image = $this->get_directorist_image($post_id);
        echo '<p><strong>Our Function Result:</strong> ' . esc_html($our_image) . '</p>';
    }
    
    private function validate_coordinates($lat, $lng) {
        return (
            is_numeric($lat) && 
            is_numeric($lng) && 
            $lat >= -90 && 
            $lat <= 90 && 
            $lng >= -180 && 
            $lng <= 180 &&
            ($lat != 0 || $lng != 0) // Exclude 0,0 coordinates
        );
    }
    
    public function add_admin_menu() {
        // Debug: Log admin menu registration
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // error_log('Universal Directory Map: Registering admin menu');
            // error_log('Universal Directory Map: Licence manager exists: ' . ($this->licence_manager ? 'Yes' : 'No'));
        }
        
        // Add settings page to the existing Directory Maps menu
        add_submenu_page(
            'edit.php?post_type=sfrdm_map',
            __('Settings', 'sfr-directory-map'),
            __('Settings', 'sfr-directory-map'),
            'manage_options',
            'sfrdm-settings',
            array($this, 'admin_page')
        );
        
        // Add analytics page to the existing Directory Maps menu
        add_submenu_page(
            'edit.php?post_type=sfrdm_map',
            __('Search Analytics', 'sfr-directory-map'),
            __('Analytics', 'sfr-directory-map'),
            'manage_options',
            'sfrdm-analytics',
            array($this, 'analytics_page')
        );
    }
    
    public function admin_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'sfr-directory-map'));
        }
        
        // Call the settings render function from the admin settings class
        if (class_exists('SFRDM_Admin_Settings')) {
            $settings_admin = new SFRDM_Admin_Settings();
            $settings_admin->render_settings_page();
        } else {
            echo '<div class="wrap"><h1>Settings</h1><p>Settings functionality not available.</p></div>';
        }
    }
    
    /**
     * Legacy admin page method (kept for backwards compatibility)
     */
    public function legacy_admin_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'sfr-directory-map'));
        }
        
        // Debug: Log that admin page is being called
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // error_log('Universal Directory Map: Admin page called');
        }
        
        
        
        // Handle form submission
        
        if (isset($_POST['submit']) && isset($_POST['sfrdm_settings_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sfrdm_settings_nonce'])), 'sfrdm_save_settings')) {
            $default_lat = isset($_POST['default_lat']) ? (float) sanitize_text_field(wp_unslash($_POST['default_lat'])) : 54.5;
            $default_lng = isset($_POST['default_lng']) ? (float) sanitize_text_field(wp_unslash($_POST['default_lng'])) : -2.0;
            $default_zoom = isset($_POST['default_zoom']) ? (int) sanitize_text_field(wp_unslash($_POST['default_zoom'])) : 6;
            
            // Handle button colours
            $button_primary_color = isset($_POST['button_primary_color']) ? sanitize_hex_color(wp_unslash($_POST['button_primary_color'])) : '#0073aa';
            $button_secondary_color = isset($_POST['button_secondary_color']) ? sanitize_hex_color(wp_unslash($_POST['button_secondary_color'])) : '#005a87';
            $button_hover_color = isset($_POST['button_hover_color']) ? sanitize_hex_color(wp_unslash($_POST['button_hover_color'])) : '#004466';
            $button_text_color = isset($_POST['button_text_color']) ? sanitize_hex_color(wp_unslash($_POST['button_text_color'])) : '#ffffff';
            
            // Handle local development setting
            $enable_local_dev = isset($_POST['sfrdm_enable_local_dev']) ? 1 : 0;
            update_option('sfrdm_enable_local_dev', $enable_local_dev);
            
            // Handle development mode (only on local environments)
            // Check if we're on a local domain first
            $domain = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
            $is_local_domain = (
                strpos($domain, '.local') !== false ||
                strpos($domain, 'localhost') !== false ||
                strpos($domain, '127.0.0.1') !== false ||
                strpos($domain, '192.168.') !== false ||
                strpos($domain, '10.0.') !== false ||
                strpos($domain, '172.16.') !== false
            );
            
            if ($is_local_domain) {
                $dev_mode_value = isset($_POST['sfrdm_development_mode']) ? 1 : 0;
                update_option('sfrdm_development_mode', $dev_mode_value);
            }
            
            // Validate coordinates
            if ($default_lat >= -90 && $default_lat <= 90) {
                update_option('sfrdm_default_lat', $default_lat);
            }
            if ($default_lng >= -180 && $default_lng <= 180) {
                update_option('sfrdm_default_lng', $default_lng);
            }
            if ($default_zoom >= 1 && $default_zoom <= 18) {
                update_option('sfrdm_default_zoom', $default_zoom);
            }
            
            // Save button colours
            update_option('sfrdm_button_primary_color', $button_primary_color);
            update_option('sfrdm_button_secondary_color', $button_secondary_color);
            update_option('sfrdm_button_hover_color', $button_hover_color);
            update_option('sfrdm_button_text_color', $button_text_color);
            
            // Handle category colors
            $category_colors = array();
            if (isset($_POST['category_colors']) && is_array($_POST['category_colors'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized in loop below
                foreach (wp_unslash((array) $_POST['category_colors']) as $category => $color) {
                    $sanitized_category = sanitize_text_field($category);
                    $sanitized_color = sanitize_hex_color($color);
                    if ($sanitized_category && $sanitized_color) {
                        $category_colors[$sanitized_category] = $sanitized_color;
                    }
                }
            }
            update_option('sfrdm_category_colors', $category_colors);
            
            echo '<div class="notice notice-success"><p>' . esc_html__('Settings saved!', 'sfr-directory-map') . '</p></div>';
        }
        
        $default_lat = get_option('sfrdm_default_lat', 54.5);
        $default_lng = get_option('sfrdm_default_lng', -2.0);
        $default_zoom = get_option('sfrdm_default_zoom', 6);
        $button_primary_color = get_option('sfrdm_button_primary_color', '#0073aa');
        $button_secondary_color = get_option('sfrdm_button_secondary_color', '#005a87');
        $button_hover_color = get_option('sfrdm_button_hover_color', '#004466');
        $button_text_color = get_option('sfrdm_button_text_color', '#ffffff');
        $category_colors = get_option('sfrdm_category_colors', array());
        $detected_plugin = $this->detect_directory_plugin();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Universal Directory Map Settings', 'sfr-directory-map'); ?></h1>
            
            <!-- Debug info -->
            <div class="notice notice-info">
                <p><strong>Debug:</strong> Admin page loaded successfully. Plugin version: <?php echo esc_html($this->version); ?></p>
            </div>
            
            <div class="udm-admin-section">
                <h2><?php esc_html_e('Plugin Status', 'sfr-directory-map'); ?></h2>
                <p>
                    <strong><?php esc_html_e('Detected Directory Plugin:', 'sfr-directory-map'); ?></strong> 
                    <?php if ($detected_plugin): ?>
                        <span style="color: green;">✓ <?php echo esc_html($this->get_plugin_display_name($detected_plugin)); ?></span>
                    <?php else: ?>
                        <span style="color: red;">✗ <?php esc_html_e('No compatible directory plugin found', 'sfr-directory-map'); ?></span>
                    <?php endif; ?>
                </p>
            </div>
            
            <form method="post">
                <?php wp_nonce_field('sfrdm_save_settings', 'sfrdm_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Default Latitude', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="number" step="any" name="default_lat" value="<?php echo esc_attr($default_lat); ?>" min="-90" max="90" />
                            <p class="description"><?php esc_html_e('Latitude for map center (-90 to 90)', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Default Longitude', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="number" step="any" name="default_lng" value="<?php echo esc_attr($default_lng); ?>" min="-180" max="180" />
                            <p class="description"><?php esc_html_e('Longitude for map center (-180 to 180)', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Default Zoom Level', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="number" min="1" max="18" name="default_zoom" value="<?php echo esc_attr($default_zoom); ?>" />
                            <p class="description"><?php esc_html_e('Initial zoom level (1-18, where 18 is closest)', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <h3><?php esc_html_e('Button Customization', 'sfr-directory-map'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Primary Button Colour', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="color" name="button_primary_color" value="<?php echo esc_attr($button_primary_color); ?>" />
                            <p class="description"><?php esc_html_e('Main colour for "View Listing" buttons', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Secondary Button Colour', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="color" name="button_secondary_color" value="<?php echo esc_attr($button_secondary_color); ?>" />
                            <p class="description"><?php esc_html_e('Secondary colour for button gradients', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Button Hover Colour', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="color" name="button_hover_color" value="<?php echo esc_attr($button_hover_color); ?>" />
                            <p class="description"><?php esc_html_e('Colour when hovering over buttons', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Button Text Colour', 'sfr-directory-map'); ?></th>
                        <td>
                            <input type="color" name="button_text_color" value="<?php echo esc_attr($button_text_color); ?>" />
                            <p class="description"><?php esc_html_e('Text colour for buttons', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <h3><?php esc_html_e('Map Pin Colors', 'sfr-directory-map'); ?></h3>
                <p class="description"><?php esc_html_e('Customise the colours of map pins based on listing categories. Leave blank to use default colours.', 'sfr-directory-map'); ?></p>
                <table class="form-table">
                    <?php
                    // Get all categories from the detected directory plugin
                    $categories = array();
                    if ($detected_plugin) {
                        switch ($detected_plugin) {
                            case 'at_biz_dir': // Directorist
                                $categories = get_terms(array(
                                    'taxonomy' => 'at_biz_dir-category',
                                    'hide_empty' => false,
                                ));
                                break;
                            case 'gd_place': // GeoDirectory
                                $categories = get_terms(array(
                                    'taxonomy' => 'gd_placecategory',
                                    'hide_empty' => false,
                                ));
                                break;
                            case 'wpbdp_listing': // Business Directory
                                $categories = get_terms(array(
                                    'taxonomy' => 'wpbdp-category',
                                    'hide_empty' => false,
                                ));
                                break;
                        }
                    }
                    
                    if (!empty($categories) && !is_wp_error($categories)):
                        foreach ($categories as $category):
                            $category_slug = $category->slug;
                            $category_name = $category->name;
                            $current_color = isset($category_colors[$category_slug]) ? $category_colors[$category_slug] : '';
                    ?>
                    <tr>
                        <th scope="row">
                            <label for="category_color_<?php echo esc_attr($category_slug); ?>">
                                <?php echo esc_html($category_name); ?>
                            </label>
                        </th>
                        <td>
                            <input type="color" 
                                   name="category_colors[<?php echo esc_attr($category_slug); ?>]" 
                                   id="category_color_<?php echo esc_attr($category_slug); ?>" 
                                   value="<?php echo esc_attr($current_color); ?>" />
                            <span class="description">
                                <?php if (empty($current_color)): ?>
                                    <?php esc_html_e('Default colour will be used', 'sfr-directory-map'); ?>
                                <?php endif; ?>
                            </span>
                        </td>
                    </tr>
                    <?php 
                        endforeach;
                    else:
                    ?>
                    <tr>
                        <td colspan="2">
                            <p class="description">
                                <?php if (!$detected_plugin): ?>
                                    <?php esc_html_e('No directory plugin detected. Category colours will be available once a compatible directory plugin is installed.', 'sfr-directory-map'); ?>
                                <?php else: ?>
                                    <?php esc_html_e('No categories found in your directory plugin.', 'sfr-directory-map'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>
                    <?php endif; ?>
                </table>
                
                <h3><?php esc_html_e('Colour Templates', 'sfr-directory-map'); ?></h3>
                <div class="udm-templates-section">
                    <?php if ($this->is_premium_feature_available('color_templates')): ?>
                    <p class="description"><?php esc_html_e('Save your colour schemes as templates for easy reuse across different maps.', 'sfr-directory-map'); ?></p>
                    
                    <div class="udm-template-actions">
                        <button type="button" id="save-template-btn" class="button button-secondary">
                            <?php esc_html_e('Save Current as Template', 'sfr-directory-map'); ?>
                        </button>
                        <button type="button" id="load-template-btn" class="button button-secondary">
                            <?php esc_html_e('Load Template', 'sfr-directory-map'); ?>
                        </button>
                    </div>
                    
                    <div id="template-name-input" style="display: none; margin-top: 10px;">
                        <input type="text" id="template-name" placeholder="<?php esc_attr_e('Enter template name...', 'sfr-directory-map'); ?>" style="width: 200px;" />
                        <button type="button" id="confirm-save-template" class="button button-primary">
                            <?php esc_html_e('Save', 'sfr-directory-map'); ?>
                        </button>
                        <button type="button" id="cancel-save-template" class="button">
                            <?php esc_html_e('Cancel', 'sfr-directory-map'); ?>
                        </button>
                    </div>
                    
                    <div id="template-list" style="margin-top: 15px;">
                        <?php $this->display_colour_templates(); ?>
                    </div>
                    <?php else: ?>
                    <div class="udm-premium-notice">
                        <div class="udm-premium-badge"><?php esc_html_e('Premium Feature', 'sfr-directory-map'); ?></div>
                        <p><?php esc_html_e('Colour templates are available in the premium version. Save and reuse your colour schemes across multiple maps.', 'sfr-directory-map'); ?></p>
                        <p><a href="https://supportfromrichard.co.uk/plugins/sfr-directory-map/" target="_blank" class="button button-primary">
                            <?php esc_html_e('Upgrade to Premium', 'sfr-directory-map'); ?>
                        </a></p>
                    </div>
                    <?php endif; ?>
                </div>
                
                <?php if ($this->is_local_development()): ?>
                <tr>
                    <th scope="row"><?php esc_html_e('Development Mode', 'sfr-directory-map'); ?></th>
                    <td>
                        <input type="checkbox" name="sfrdm_development_mode" value="1" <?php checked(get_option('sfrdm_development_mode', 0)); ?> />
                        <p class="description"><?php esc_html_e('Enable all premium features for testing (bypasses licence check) - Local development only', 'sfr-directory-map'); ?></p>
                        <p class="description" style="color: #0073aa; font-weight: bold;"><?php esc_html_e('✓ Local development environment detected - Development mode available', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                <?php else: ?>
                <tr style="display: none;">
                    <th scope="row"><?php esc_html_e('Development Mode', 'sfr-directory-map'); ?></th>
                    <td>
                        <p class="description" style="color: #d63638;"><?php esc_html_e('Development mode is only available on local development environments (.local, localhost, etc.)', 'sfr-directory-map'); ?></p>
                    </td>
                </tr>
                <?php endif; ?>
                
                <?php submit_button(); ?>
            </form>
            
            
            <?php if ($this->is_local_development()): ?>
            <div class="udm-admin-section">
                <div class="udm-dev-notice">
                    <strong>🔧 Development Mode Active</strong> - All premium features are enabled for local development.
                </div>
                <h2><?php esc_html_e('Debug Directorist Images', 'sfr-directory-map'); ?></h2>
                <p><?php esc_html_e('Enter a Directorist post ID to debug image data:', 'sfr-directory-map'); ?></p>
                <form method="post">
                    <?php wp_nonce_field('sfrdm_debug_image', 'sfrdm_debug_nonce'); ?>
                    <input type="number" name="debug_post_id" placeholder="Post ID" value="<?php echo isset($_POST['debug_post_id']) ? esc_attr(absint(wp_unslash($_POST['debug_post_id']))) : ''; ?>" />
                    <input type="submit" name="debug_image" value="<?php esc_attr_e('Debug Image Data', 'sfr-directory-map'); ?>" class="button" />
                </form>
                
                <?php 
                if (isset($_POST['debug_image']) && isset($_POST['sfrdm_debug_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sfrdm_debug_nonce'])), 'sfrdm_debug_image')) {
                    $debug_post_id = isset($_POST['debug_post_id']) ? intval(wp_unslash($_POST['debug_post_id'])) : 0; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    if ($debug_post_id > 0) {
                        $this->debug_directorist_image($debug_post_id);
                    }
                }
                ?>
            </div>
            <?php endif; ?>
            
            <div class="udm-admin-section">
                <h2><?php esc_html_e('Usage', 'sfr-directory-map'); ?></h2>
                <p><?php esc_html_e('Use the shortcode:', 'sfr-directory-map'); ?> <code>[sfr_directory_map]</code></p>
                
                <h3><?php esc_html_e('Available Parameters:', 'sfr-directory-map'); ?></h3>
                <ul>
                    <li><code>height</code> - <?php esc_html_e('Map height (default: 500px)', 'sfr-directory-map'); ?></li>
                    <li><code>category</code> - <?php esc_html_e('Filter by category slug', 'sfr-directory-map'); ?></li>
                    <li><code>post_type</code> - <?php esc_html_e('Specify post type or "auto" to detect', 'sfr-directory-map'); ?></li>
                    <li><code>cluster</code> - <?php esc_html_e('Enable marker clustering (true/false)', 'sfr-directory-map'); ?></li>
                </ul>
                
                <h3><?php esc_html_e('Examples:', 'sfr-directory-map'); ?></h3>
                <code>[sfr_directory_map height="600px" category="accommodation" cluster="true"]</code>
            </div>
        </div>
        
        
        
        
        <?php
    }
    
    /**
     * Get analytics data for directory listings
     */
    private function get_directory_analytics($post_type) {
        global $wpdb;
        
        $analytics = array(
            'total_listings' => 0,
            'with_coordinates' => 0,
            'with_images' => 0,
            'with_default_locations' => 0,
            'with_default_images' => 0,
            'with_hidden_maps' => 0,
            'categories_count' => 0,
            'categories' => array()
        );
        
        // Get all listings
        $listings = get_posts(array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ));
        
        $analytics['total_listings'] = count($listings);
        
        if (empty($listings)) {
            return $analytics;
        }
        
        // Get categories based on plugin type
        $categories = array();
        switch ($post_type) {
            case 'at_biz_dir':
                $categories = get_terms(array(
                    'taxonomy' => 'at_biz_dir-category',
                    'hide_empty' => false
                ));
                break;
            case 'gd_place':
                $categories = get_terms(array(
                    'taxonomy' => 'gd_placecategory',
                    'hide_empty' => false
                ));
                break;
            default:
                // Generic categories
                $categories = get_terms(array(
                    'taxonomy' => 'category',
                    'hide_empty' => false
                ));
        }
        
        $analytics['categories_count'] = count($categories);
        
        // Process each category
        foreach ($categories as $category) {
            $category_analytics = array(
                'name' => $category->name,
                'total' => 0,
                'with_coordinates' => 0,
                'with_images' => 0,
                'with_default_locations' => 0,
                'with_default_images' => 0,
                'with_hidden_maps' => 0,
                'map_ready' => 0,
                'completion_percentage' => 0
            );
            
            // Get listings in this category
            $category_listings = get_posts(array(
                'post_type' => $post_type,
                'post_status' => 'publish',
                'numberposts' => -1,
                'fields' => 'ids',
                'tax_query' => array(
                    array(
                        'taxonomy' => $category->taxonomy,
                        'field' => 'term_id',
                        'terms' => $category->term_id
                    )
                )
            ));
            
            $category_analytics['total'] = count($category_listings);
            
            // Check each listing for coordinates and images
            foreach ($category_listings as $listing_id) {
                $has_coordinates = false;
                $has_image = false;
                $has_default_location = false;
                $has_default_image = false;
                $has_hidden_map = false;
                
                // Check coordinates based on plugin type
                switch ($post_type) {
                    case 'at_biz_dir':
                        $lat = get_post_meta($listing_id, '_manual_lat', true);
                        $lng = get_post_meta($listing_id, '_manual_lng', true);
                        $hide_map = get_post_meta($listing_id, '_hide_map', true);
                        
                        // Check if map is hidden
                        if ($hide_map) {
                            $has_hidden_map = true;
                            $category_analytics['with_hidden_maps']++;
                        }
                        
                        // Check for default location (empty manual coordinates means using default)
                        if (empty($lat) && empty($lng)) {
                            $has_default_location = true;
                            $category_analytics['with_default_locations']++;
                        }
                        break;
                    case 'gd_place':
                        $lat = get_post_meta($listing_id, 'post_latitude', true);
                        $lng = get_post_meta($listing_id, 'post_longitude', true);
                        break;
                    default:
                        $lat = get_post_meta($listing_id, 'latitude', true);
                        $lng = get_post_meta($listing_id, 'longitude', true);
                }
                
                if ($lat && $lng && $lat != 0 && $lng != 0) {
                    $has_coordinates = true;
                    $category_analytics['with_coordinates']++;
                }
                
                // Check for images
                $image = $this->get_listing_image_for_analytics($listing_id, $post_type);
                if (!empty($image)) {
                    $has_image = true;
                    $category_analytics['with_images']++;
                    
                    // Check if it's a default/placeholder image
                    if ($this->is_default_image($image, $post_type)) {
                        $has_default_image = true;
                        $category_analytics['with_default_images']++;
                    }
                }
                
                // Count map-ready listings (have both coordinates and images, and map not hidden)
                if ($has_coordinates && $has_image && !$has_hidden_map) {
                    $category_analytics['map_ready']++;
                }
            }
            
            // Calculate completion percentage
            if ($category_analytics['total'] > 0) {
                $category_analytics['completion_percentage'] = round(($category_analytics['map_ready'] / $category_analytics['total']) * 100);
            }
            
            $analytics['categories'][] = $category_analytics;
        }
        
        // Calculate global totals from category data
        $analytics['with_coordinates'] = 0;
        $analytics['with_images'] = 0;
        $analytics['with_default_locations'] = 0;
        $analytics['with_default_images'] = 0;
        $analytics['with_hidden_maps'] = 0;
        foreach ($analytics['categories'] as $category) {
            $analytics['with_coordinates'] += $category['with_coordinates'];
            $analytics['with_images'] += $category['with_images'];
            $analytics['with_default_locations'] += $category['with_default_locations'];
            $analytics['with_default_images'] += $category['with_default_images'];
            $analytics['with_hidden_maps'] += $category['with_hidden_maps'];
        }
        
        return $analytics;
    }
    
    /**
     * Get listing image for analytics (simplified version)
     */
    private function get_listing_image_for_analytics($post_id, $post_type) {
        // Try featured image first
        $image_url = get_the_post_thumbnail_url($post_id, 'thumbnail');
        if ($image_url) {
            return $image_url;
        }
        
        // Try plugin-specific image methods
        switch ($post_type) {
            case 'at_biz_dir':
                // Try Directorist's preview image first
                $preview_img = get_post_meta($post_id, '_listing_prv_img', true);
                if (!empty($preview_img) && is_numeric($preview_img)) {
                    return wp_get_attachment_image_url($preview_img, 'thumbnail');
                }
                
                // Try Directorist's listing images array
                $listing_images = get_post_meta($post_id, '_listing_img', true);
                if (!empty($listing_images) && is_array($listing_images)) {
                    $first_image = reset($listing_images);
                    if (is_numeric($first_image)) {
                        return wp_get_attachment_image_url($first_image, 'thumbnail');
                    }
                } elseif (!empty($listing_images) && is_numeric($listing_images)) {
                    return wp_get_attachment_image_url($listing_images, 'thumbnail');
                }
                break;
            case 'gd_place':
                // Try GeoDirectory's featured image
                $featured_image = get_post_meta($post_id, 'featured_image', true);
                if (!empty($featured_image)) {
                    return $featured_image;
                }
                break;
        }
        
        return false;
    }
    
    /**
     * Check if image is a default/placeholder image
     */
    private function is_default_image($image_url, $post_type) {
        if (empty($image_url)) {
            return false;
        }
        
        // Check for common placeholder patterns
        $placeholder_patterns = array(
            'placeholder',
            'default',
            'no-image',
            'noimage',
            'coming-soon',
            'coming_soon'
        );
        
        $image_url_lower = strtolower($image_url);
        foreach ($placeholder_patterns as $pattern) {
            if (strpos($image_url_lower, $pattern) !== false) {
                return true;
            }
        }
        
        // Check for specific plugin default images
        switch ($post_type) {
            case 'at_biz_dir':
                // Directorist default image patterns
                if (strpos($image_url_lower, 'directorist') !== false && strpos($image_url_lower, 'default') !== false) {
                    return true;
                }
                break;
            case 'gd_place':
                // GeoDirectory default image patterns
                if (strpos($image_url_lower, 'geodirectory') !== false && strpos($image_url_lower, 'default') !== false) {
                    return true;
                }
                break;
        }
        
        return false;
    }
    
    public function activate() {
        // Set default options
        add_option('sfrdm_default_lat', 54.5); // UK center
        add_option('sfrdm_default_lng', -2.0);
        add_option('sfrdm_default_zoom', 6);
        
        // Set default analytics to enabled
        if (get_option('sfrdm_enable_analytics') === false) {
            update_option('sfrdm_enable_analytics', 1);
        }
        
        // Create assets directory if it doesn't exist
        $upload_dir = wp_upload_dir();
        $assets_dir = $upload_dir['basedir'] . '/sfr-directory-map';
        
        if (!file_exists($assets_dir)) {
            wp_mkdir_p($assets_dir);
        }
        
        // Clear any cached plugin detection
        $this->detected_plugin_cache = null;
        
        // Create analytics table directly (don't depend on Analytics_Manager)
        global $wpdb;
        $table_name = $wpdb->prefix . 'sfrdm_search_analytics';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            map_id varchar(50) NOT NULL,
            search_term varchar(255) NOT NULL,
            results_count int(11) NOT NULL DEFAULT 0,
            timestamp datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            user_session varchar(64) DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            user_agent text DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY map_id (map_id),
            KEY search_term (search_term),
            KEY timestamp (timestamp),
            KEY user_session (user_session)
        ) $charset_collate;";
        
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
        
        // Register post type and flush rewrite rules
        $this->register_map_post_type();
        flush_rewrite_rules();
    }
    
    public function deactivate() {
        // Clean up transients and caches
        delete_transient('sfrdm_plugin_detection');
        
        // Note: We don't delete options on deactivation
        // They should persist in case plugin is reactivated
    }
    
}

// Initialize the plugin
new SFRDM_Main();
?>