<?php
/**
 * SFR Directory Map - Analytics Manager
 * Handles search analytics tracking and reporting for premium users
 */

if (!defined('ABSPATH')) {
    exit;
}

class SFRDM_Analytics_Manager {
    
    private $table_name;
    private $summary_table_name;
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'sfrdm_search_analytics';
        $this->summary_table_name = $wpdb->prefix . 'sfrdm_search_summary';
        
        add_action('wp_ajax_sfrdm_track_search', array($this, 'track_search'));
        add_action('wp_ajax_nopriv_sfrdm_track_search', array($this, 'track_search'));
        add_action('wp_ajax_sfrdm_get_analytics', array($this, 'get_analytics_data'));
        add_action('wp_ajax_sfrdm_export_analytics', array($this, 'export_analytics_data'));
        add_action('wp_ajax_sfrdm_get_analytics_nonce', array($this, 'get_analytics_nonce'));
        add_action('wp_ajax_nopriv_sfrdm_get_analytics_nonce', array($this, 'get_analytics_nonce'));
        
        // Create tables on plugin activation - this will be handled by the main plugin class
    }
    
    /**
     * Create database tables for analytics
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Main analytics table
        $sql_analytics = "CREATE TABLE {$this->table_name} (
            id int(11) NOT NULL AUTO_INCREMENT,
            map_id varchar(255) NOT NULL,
            search_term varchar(255) NOT NULL,
            results_count int(11) NOT NULL DEFAULT 0,
            timestamp datetime NOT NULL,
            user_session varchar(255) DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            user_agent text DEFAULT NULL,
            PRIMARY KEY (id),
            KEY map_id (map_id),
            KEY search_term (search_term),
            KEY timestamp (timestamp),
            KEY user_session (user_session)
        ) $charset_collate;";
        
        // Summary table for performance
        $sql_summary = "CREATE TABLE {$this->summary_table_name} (
            id int(11) NOT NULL AUTO_INCREMENT,
            map_id varchar(255) NOT NULL,
            search_term varchar(255) NOT NULL,
            search_count int(11) NOT NULL DEFAULT 1,
            last_searched datetime NOT NULL,
            total_results int(11) NOT NULL DEFAULT 0,
            zero_results_count int(11) NOT NULL DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY unique_map_term (map_id, search_term),
            KEY map_id (map_id),
            KEY search_term (search_term),
            KEY search_count (search_count),
            KEY last_searched (last_searched)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_analytics);
        dbDelta($sql_summary);
    }
    
    /**
     * Track a search event
     */
    public function track_search() {
        // Debug logging
        // error_log('UDM: Analytics track_search called with POST data: ' . // print_r($_POST, true));
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'sfrdm_analytics_nonce')) {
            // error_log('UDM: Analytics nonce verification failed');
            wp_die('Security check failed');
        }
        
        // Check if analytics is enabled
        $analytics_enabled = $this->is_analytics_enabled();
        // error_log('UDM: Analytics enabled check: ' . ($analytics_enabled ? 'true' : 'false'));
        
        if (!$analytics_enabled) {
            // error_log('UDM: Analytics not enabled - licence status: ' . get_option('sfrdm_licence_status', 'inactive') . ', local dev: ' . get_option('sfrdm_enable_local_dev', 0) . ', settings: ' . // print_r(get_option('sfrdm_settings', array()), true));
            wp_send_json_error('Analytics not enabled');
        }
        
        // Sanitize input data
        $map_id = isset($_POST['map_id']) ? sanitize_text_field(wp_unslash($_POST['map_id'])) : '';
        $search_term = isset($_POST['search_term']) ? sanitize_text_field(wp_unslash($_POST['search_term'])) : '';
        $results_count = isset($_POST['results_count']) ? intval(wp_unslash($_POST['results_count'])) : 0;
        
        // Debug: Log the received data
        
        // Validate data
        if (empty($map_id) || empty($search_term)) {
            wp_send_json_error('Invalid data');
        }
        
        // Get user session info (anonymized)
        $user_session = $this->get_user_session();
        $ip_address = $this->anonymize_ip(isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '');
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        
        // Insert into main analytics table
        global $wpdb;
        $result = $wpdb->insert(
            $wpdb->prefix . 'sfrdm_search_analytics',
            array(
                'map_id' => $map_id,
                'search_term' => $search_term,
                'results_count' => $results_count,
                'timestamp' => current_time('mysql'),
                'user_session' => $user_session,
                'ip_address' => $ip_address,
                'user_agent' => $user_agent
            ),
            array('%s', '%s', '%d', '%s', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            wp_send_json_error('Failed to record search');
        }
        
        // Update summary table
        $this->update_search_summary($map_id, $search_term, $results_count);
        
        // error_log('UDM: Search tracked successfully - Map: ' . $map_id . ', Term: ' . $search_term . ', Results: ' . $results_count);
        
        wp_send_json_success('Search tracked successfully');
    }
    
    /**
     * Get analytics nonce for testing
     */
    public function get_analytics_nonce() {
        wp_send_json_success(array(
            'nonce' => wp_create_nonce('sfrdm_analytics_nonce')
        ));
    }
    
    /**
     * Update search summary table
     */
    private function update_search_summary($map_id, $search_term, $results_count) {
        global $wpdb;
        
        // Check if summary exists
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM " . $wpdb->prefix . "sfrdm_search_summary WHERE (map_id = %s OR map_id = %d) AND search_term = %s",
            $map_id, intval($map_id), $search_term
        ));
        
        if ($existing) {
            // Update existing summary
            $wpdb->update(
                $wpdb->prefix . 'sfrdm_search_summary',
                array(
                    'search_count' => $existing->search_count + 1,
                    'last_searched' => current_time('mysql'),
                    'total_results' => $existing->total_results + $results_count,
                    'zero_results_count' => $results_count === 0 ? $existing->zero_results_count + 1 : $existing->zero_results_count
                ),
                array('id' => $existing->id),
                array('%d', '%s', '%d', '%d'),
                array('%d')
            );
        } else {
            // Insert new summary
            $wpdb->insert(
                $wpdb->prefix . 'sfrdm_search_summary',
                array(
                    'map_id' => $map_id,
                    'search_term' => $search_term,
                    'search_count' => 1,
                    'last_searched' => current_time('mysql'),
                    'total_results' => $results_count,
                    'zero_results_count' => $results_count === 0 ? 1 : 0
                ),
                array('%s', '%s', '%d', '%s', '%d', '%d')
            );
        }
    }
    
    /**
     * Get analytics data for admin dashboard
     */
    public function get_analytics_data() {
        // Debug logging
        // error_log('UDM: get_analytics_data called with GET data: ' . // print_r($_GET, true));
        
        // Check user permissions
        if (!current_user_can('manage_options')) {
            // error_log('UDM: get_analytics_data - insufficient permissions');
            wp_die('Insufficient permissions');
        }
        
        // Verify nonce
        if (!isset($_GET['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'sfrdm_analytics_admin_nonce')) {
            // error_log('UDM: get_analytics_data - nonce verification failed');
            wp_die('Security check failed');
        }
        
        $period = isset($_GET['period']) ? sanitize_text_field(wp_unslash($_GET['period'])) : '7';
        $map_id = isset($_GET['map_id']) ? sanitize_text_field(wp_unslash($_GET['map_id'])) : '';
        
        // Debug: Log the parameters being used
        // error_log('UDM: get_analytics_data - period: ' . $period . ', map_id: ' . $map_id);
        
        // Analytics is always available - the toggle is just for user preference
        
        // Make all data available for everyone - no premium checks
        $data = array(
            'live_stats' => $this->get_live_stats($map_id, $period),
            'popular_terms' => $this->get_popular_terms($map_id, $period),
            'search_trends' => $this->get_search_trends($map_id, $period),
            'recent_searches' => $this->get_recent_searches($map_id, 10)
        );
        
        // error_log('UDM: get_analytics_data - returning data: ' . print_r($data, true));
        
        wp_send_json_success($data);
    }
    
    /**
     * Get live statistics
     */
    private function get_live_stats($map_id = '', $period = '7') {
        global $wpdb;
        
        // error_log('UDM: get_live_stats called with map_id: ' . $map_id . ', period: ' . $period);
        
        $where_clause = $map_id ? $wpdb->prepare("WHERE (map_id = %s OR map_id = %d)", $map_id, intval($map_id)) : '';
        
        // Today's stats
        $table_name = $wpdb->prefix . 'sfrdm_search_analytics';
        
        // Debug: Check what map_ids exist in the database
        // $existing_map_ids = $wpdb->get_col("SELECT DISTINCT map_id FROM " . $table_name . " WHERE map_id IS NOT NULL AND map_id != ''");
        // $all_map_ids = $wpdb->get_col("SELECT DISTINCT map_id FROM " . $table_name);
        // error_log('UDM: All map_ids in database (including empty): ' . print_r($all_map_ids, true));
        // error_log('UDM: Non-empty map_ids in database: ' . print_r($existing_map_ids, true));
        
        if ($map_id) {
            // Try both string and integer versions of the map_id
            $today_stats = $wpdb->get_row($wpdb->prepare(
                "SELECT COUNT(*) as total_searches, COUNT(DISTINCT search_term) as unique_terms FROM {$table_name} WHERE DATE(timestamp) = DATE(NOW()) AND (map_id = %s OR map_id = %d)",
                $map_id, intval($map_id)
            ));
        } else {
            $today_stats = $wpdb->get_row(
                "SELECT COUNT(*) as total_searches, COUNT(DISTINCT search_term) as unique_terms FROM {$table_name} WHERE DATE(timestamp) = DATE(NOW())"
            );
            // error_log('UDM: Today stats for all maps: ' . print_r($today_stats, true));
        }
        
        // error_log('UDM: get_live_stats today_stats: ' . // print_r($today_stats, true));
        
        // This week's stats
        if ($map_id) {
            $week_stats = $wpdb->get_row($wpdb->prepare(
                "SELECT COUNT(*) as total_searches, COUNT(DISTINCT search_term) as unique_terms FROM {$table_name} WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY) AND (map_id = %s OR map_id = %d)",
                intval($period), $map_id, intval($map_id)
            ));
        } else {
            $week_stats = $wpdb->get_row($wpdb->prepare(
                "SELECT COUNT(*) as total_searches, COUNT(DISTINCT search_term) as unique_terms FROM {$table_name} WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)",
                intval($period)
            ));
        }
        
        // error_log('UDM: get_live_stats week_stats: ' . // print_r($week_stats, true));
        
        $result = array(
            'today' => $today_stats,
            'this_week' => $week_stats
        );
        
        // error_log('UDM: get_live_stats returning: ' . print_r($result, true));
        
        return $result;
    }
    
    
    /**
     * Get popular search terms
     */
    private function get_popular_terms($map_id = '', $period = '7', $limit = 20) {
        global $wpdb;
        
        // error_log('UDM: get_popular_terms called with map_id: ' . $map_id . ', period: ' . $period . ', limit: ' . $limit);
        
        if ($map_id) {
            $sql = "SELECT 
                    search_term,
                    search_count,
                    last_searched,
                    total_results,
                    zero_results_count
                FROM {$wpdb->prefix}sfrdm_search_summary 
                WHERE (map_id = %s OR map_id = %d)
                AND last_searched >= DATE_SUB(NOW(), INTERVAL %d DAY)
                ORDER BY search_count DESC 
                LIMIT %d";
            return $wpdb->get_results($wpdb->prepare($sql, $map_id, intval($map_id), intval($period), intval($limit))); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        } else {
            $sql = "SELECT 
                    search_term,
                    search_count,
                    last_searched,
                    total_results,
                    zero_results_count
                FROM {$wpdb->prefix}sfrdm_search_summary 
                WHERE last_searched >= DATE_SUB(NOW(), INTERVAL %d DAY)
                ORDER BY search_count DESC 
                LIMIT %d";
            return $wpdb->get_results($wpdb->prepare($sql, intval($period), intval($limit))); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        }
    }
    
    /**
     * Get search trends (daily counts)
     */
    private function get_search_trends($map_id = '', $period = '7') {
        global $wpdb;
        
        // error_log('UDM: get_search_trends called with map_id: ' . $map_id . ', period: ' . $period);
        
        $table_name = $wpdb->prefix . 'sfrdm_search_analytics';
        if ($map_id) {
            $results = $wpdb->get_results($wpdb->prepare(
                "SELECT DATE(timestamp) as date, COUNT(*) as search_count, COUNT(DISTINCT search_term) as unique_terms FROM {$table_name} WHERE (map_id = %s OR map_id = %d) AND timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY) GROUP BY DATE(timestamp) ORDER BY date ASC",
                $map_id, intval($map_id), intval($period)
            ));
        } else {
            $results = $wpdb->get_results($wpdb->prepare(
                "SELECT DATE(timestamp) as date, COUNT(*) as search_count, COUNT(DISTINCT search_term) as unique_terms FROM {$table_name} WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY) GROUP BY DATE(timestamp) ORDER BY date ASC",
                intval($period)
            ));
        }
        
        // error_log('UDM: get_search_trends results: ' . print_r($results, true));
        
        return $results;
    }
    
    /**
     * Get zero result searches
     */
    private function get_zero_results($map_id = '', $period = '7') {
        global $wpdb;
        
        if ($map_id) {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT 
                    search_term,
                    COUNT(*) as zero_count,
                    MAX(timestamp) as last_searched
                FROM " . $wpdb->prefix . "sfrdm_search_analytics 
                WHERE (map_id = %s OR map_id = %d)
                AND results_count = 0
                AND timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)
                GROUP BY search_term
                ORDER BY zero_count DESC 
                LIMIT 10",
                $map_id, intval($map_id), intval($period)
            ));
        } else {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT 
                    search_term,
                    COUNT(*) as zero_count,
                    MAX(timestamp) as last_searched
                FROM " . $wpdb->prefix . "sfrdm_search_analytics 
                WHERE results_count = 0
                AND timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)
                GROUP BY search_term
                ORDER BY zero_count DESC 
                LIMIT 10",
                intval($period)
            ));
        }
    }
    
    /**
     * Get recent searches
     */
    private function get_recent_searches($map_id = '', $limit = 10) {
        global $wpdb;
        
        if ($map_id) {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT 
                    search_term,
                    results_count,
                    timestamp
                FROM " . $wpdb->prefix . "sfrdm_search_analytics 
                WHERE (map_id = %s OR map_id = %d)
                ORDER BY timestamp DESC 
                LIMIT %d",
                $map_id, intval($map_id), intval($limit)
            ));
        } else {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT 
                    search_term,
                    results_count,
                    timestamp
                FROM " . $wpdb->prefix . "sfrdm_search_analytics 
                ORDER BY timestamp DESC 
                LIMIT %d",
                intval($limit)
            ));
        }
    }
    
    /**
     * Export analytics data
     */
    public function export_analytics_data() {
        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $format = isset($_GET['format']) ? sanitize_text_field(wp_unslash($_GET['format'])) : 'csv';
        $period = isset($_GET['period']) ? sanitize_text_field(wp_unslash($_GET['period'])) : '30';
        $map_id = isset($_GET['map_id']) ? sanitize_text_field(wp_unslash($_GET['map_id'])) : '';
        
        if ($format === 'csv') {
            $this->export_csv($map_id, $period);
        } else {
            wp_send_json_error('Unsupported format');
        }
    }
    
    /**
     * Export data as CSV
     */
    private function export_csv($map_id = '', $period = '30') {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'sfrdm_search_analytics';
        if ($map_id) {
            $data = $wpdb->get_results($wpdb->prepare(
                "SELECT map_id, search_term, results_count, timestamp, user_session FROM {$table_name} WHERE (map_id = %s OR map_id = %d) AND timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY) ORDER BY timestamp DESC",
                $map_id, intval($map_id), intval($period)
            ));
        } else {
            $data = $wpdb->get_results($wpdb->prepare(
                "SELECT map_id, search_term, results_count, timestamp, user_session FROM {$table_name} WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY) ORDER BY timestamp DESC",
                intval($period)
            ));
        }
        
        // Set headers for CSV download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="udm-search-analytics-' . gmdate('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // CSV headers
        fputcsv($output, array('Map ID', 'Search Term', 'Results Count', 'Timestamp', 'User Session'));
        
        // CSV data
        foreach ($data as $row) {
            fputcsv($output, array(
                $row->map_id,
                $row->search_term,
                $row->results_count,
                $row->timestamp,
                $row->user_session
            ));
        }
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Using php://output stream, not file system
        fclose($output);
        exit;
    }
    
    /**
     * Check if analytics is enabled
     */
    private function is_analytics_enabled() {
        // Analytics is available for all users (free and premium)
        // Check if analytics is enabled in settings
        // Free version uses separate option
        return (bool) get_option('sfrdm_enable_analytics', 1);
    }
    
    /**
     * Get anonymized user session
     */
    private function get_user_session() {
        // Create a session ID based on IP and user agent (anonymized)
        $ip = $this->anonymize_ip(isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '');
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        
        return hash('sha256', $ip . $user_agent . gmdate('Y-m-d'));
    }
    
    /**
     * Anonymize IP address for privacy
     */
    private function anonymize_ip($ip) {
        if (empty($ip)) {
            return '';
        }
        
        // For IPv4, remove last octet
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return substr($ip, 0, strrpos($ip, '.')) . '.0';
        }
        
        // For IPv6, remove last 4 groups
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $parts = explode(':', $ip);
            if (count($parts) >= 4) {
                return implode(':', array_slice($parts, 0, -4)) . ':0000:0000:0000:0000';
            }
        }
        
        return $ip;
    }
    
    /**
     * Clean up old analytics data
     */
    public function cleanup_old_data() {
        global $wpdb;
        
        // Keep data for 1 year
        $wpdb->query($wpdb->prepare(
            "DELETE FROM " . $wpdb->prefix . "sfrdm_search_analytics WHERE timestamp < DATE_SUB(NOW(), INTERVAL %d DAY)",
            365
        ));
        
        // Clean up orphaned summary entries
        $wpdb->query("
            DELETE s FROM " . $wpdb->prefix . "sfrdm_search_summary s 
            LEFT JOIN " . $wpdb->prefix . "sfrdm_search_analytics a ON s.map_id = a.map_id AND s.search_term = a.search_term 
            WHERE a.id IS NULL
        ");
    }
}

// Initialize the analytics manager
new SFRDM_Analytics_Manager();
