<?php
/**
 * SFR Directory Map - Admin Settings Page (Free Version)
 */

if (!defined('ABSPATH')) {
    exit;
}

class SFRDM_Admin_Settings {
    
    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    public function register_settings() {
        register_setting('sfrdm_settings', 'sfrdm_default_lat', array('sanitize_callback' => 'floatval'));
        register_setting('sfrdm_settings', 'sfrdm_default_lng', array('sanitize_callback' => 'floatval'));
        register_setting('sfrdm_settings', 'sfrdm_default_zoom', array('sanitize_callback' => 'absint'));
        register_setting('sfrdm_settings', 'sfrdm_delete_data_on_uninstall', array('sanitize_callback' => 'absint'));
    }
    
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'sfr-directory-map'));
        }
        
        $sfrdm_suite = $this->sfrdm_get_suite_status();
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('SFR Directory Map Settings', 'sfr-directory-map'); ?></h1>
            
            <?php 
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            if (isset($_GET['settings-updated']) && sanitize_text_field(wp_unslash($_GET['settings-updated'])) === 'true'): 
            ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e('Settings saved successfully!', 'sfr-directory-map'); ?></p>
                </div>
            <?php endif; ?>
            
            <form method="post" action="options.php">
                <?php settings_fields('sfrdm_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Default Latitude', 'sfr-directory-map'); ?></th>
                        <td>
                            <?php $value = get_option('sfrdm_default_lat', '54.5'); ?>
                            <input type="number" step="0.000001" name="sfrdm_default_lat" value="<?php echo esc_attr($value); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Default latitude for new maps (e.g., 54.5 for UK)', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Default Longitude', 'sfr-directory-map'); ?></th>
                        <td>
                            <?php $value = get_option('sfrdm_default_lng', '-2.0'); ?>
                            <input type="number" step="0.000001" name="sfrdm_default_lng" value="<?php echo esc_attr($value); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Default longitude for new maps (e.g., -2.0 for UK)', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Default Zoom Level', 'sfr-directory-map'); ?></th>
                        <td>
                            <?php $value = get_option('sfrdm_default_zoom', '6'); ?>
                            <input type="number" step="1" min="1" max="18" name="sfrdm_default_zoom" value="<?php echo esc_attr($value); ?>" class="small-text" />
                            <p class="description"><?php esc_html_e('Default zoom level (1-18)', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <h2><?php esc_html_e('Data Management', 'sfr-directory-map'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Plugin Deletion', 'sfr-directory-map'); ?></th>
                        <td>
                            <?php $value = get_option('sfrdm_delete_data_on_uninstall', 0); ?>
                            <label>
                                <input type="checkbox" name="sfrdm_delete_data_on_uninstall" value="1" <?php checked($value, 1); ?> />
                                <?php esc_html_e('Delete all plugin data when plugin is deleted', 'sfr-directory-map'); ?>
                            </label>
                            <p class="description"><?php esc_html_e('If enabled, all maps, settings, and analytics data will be permanently deleted when you delete this plugin. Leave unchecked to preserve your data.', 'sfr-directory-map'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
            
            <?php if ( ! empty( $sfrdm_suite ) ) : ?>
            <div style="margin-top:30px;padding:0;border:none;background:linear-gradient(135deg,#0A9396,#005F73 50%,#076668);color:#fff;box-shadow:0 25px 50px rgba(0,0,0,0.2);border-radius:18px;">
                <div style="padding:32px;">
                    <h2 style="margin-top:0;font-size:1.6em;color:#fff;"><?php esc_html_e( 'Support From Richard Toolkit', 'sfr-directory-map' ); ?></h2>
                    <p style="color:rgba(255,255,255,0.85);font-size:1.05em;line-height:1.7;"><?php esc_html_e( 'Mix and match the plugins that power your directory stack. Launch dashboards, compare free vs Pro, or install missing components in one place.', 'sfr-directory-map' ); ?></p>
                    <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(260px,1fr));gap:20px;margin-top:20px;">
                        <?php foreach ( $sfrdm_suite as $plugin ) : ?>
                            <div style="border-radius:14px;padding:22px;background:rgba(255,255,255,0.97);box-shadow:0 10px 30px rgba(0,0,0,0.12);color:#0f172a;">
                                <div style="display:flex;justify-content:space-between;align-items:flex-start;gap:10px;">
                                    <div>
                                        <h3 style="margin:0 0 6px 0;font-size:1.1em;"><?php echo esc_html( $plugin['title'] ); ?></h3>
                                        <p style="margin:0;color:#6b7280;font-size:0.9em;line-height:1.5;"><?php echo esc_html( $plugin['description'] ); ?></p>
                                    </div>
                                    <?php
                                    $sfrdm_status_label  = __( 'Not active', 'sfr-directory-map' );
                                    $sfrdm_status_colour = '#94a3b8';
                                    if ( ! empty( $plugin['active_variant'] ) ) {
                                        $sfrdm_status_label  = ( 'pro' === $plugin['active_variant'] ) ? __( 'Pro Active', 'sfr-directory-map' ) : __( 'Free Active', 'sfr-directory-map' );
                                        $sfrdm_status_colour = ( 'pro' === $plugin['active_variant'] ) ? '#F9B702' : '#10b981';
                                    }
                                    ?>
                                    <span style="padding:4px 12px;border-radius:999px;font-size:0.75em;font-weight:700;color:#0f172a;background:<?php echo esc_attr( $sfrdm_status_colour ); ?>;">
                                        <?php echo esc_html( $sfrdm_status_label ); ?>
                                    </span>
                                </div>

                                <?php if ( ! empty( $plugin['notes'] ) ) : ?>
                                    <p style="margin:10px 0 0 0;color:#b45309;font-size:0.85em;"><?php echo esc_html( $plugin['notes'] ); ?></p>
                                <?php endif; ?>

                                <div style="display:flex;flex-wrap:wrap;gap:10px;margin-top:14px;">
                                    <?php if ( ! empty( $plugin['analytics_url'] ) ) : ?>
                                        <a class="button button-primary" style="background:#0A9396;border:none;" href="<?php echo esc_url( $plugin['analytics_url'] ); ?>">
                                            <?php esc_html_e( 'Open stats dashboard', 'sfr-directory-map' ); ?>
                                        </a>
                                    <?php elseif ( ! empty( $plugin['active'] ) ) : ?>
                                        <span style="font-size:0.85em;color:#6b7280;"><?php esc_html_e( 'Active – open from its own menu.', 'sfr-directory-map' ); ?></span>
                                    <?php else : ?>
                                        <span style="font-size:0.85em;color:#6b7280;"><?php esc_html_e( 'Install or activate to start tracking.', 'sfr-directory-map' ); ?></span>
                                    <?php endif; ?>
                                    <?php if ( ! empty( $plugin['pro_teaser'] ) && $plugin['show_pro_link'] ) : ?>
                                        <p style="margin:0;font-size:0.85em;color:#334155;flex:1 0 100%;padding-top:4px;">
                                            <?php echo esc_html( $plugin['pro_teaser'] ); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>

                                <div style="display:flex;flex-wrap:wrap;gap:10px;margin-top:15px;">
                                    <a class="button button-secondary" style="border:none;background:rgba(10,147,150,0.12);color:#0A9396;" href="<?php echo esc_url( $plugin['free_url'] ); ?>" target="_blank" rel="noopener">
                                        <?php esc_html_e( 'View free version', 'sfr-directory-map' ); ?>
                                    </a>
                                    <?php if ( $plugin['show_pro_link'] ) : ?>
                                        <a class="button" style="background:#F9B702;color:#0f172a;border:none;" href="<?php echo esc_url( $plugin['pro_url'] ); ?>" target="_blank" rel="noopener">
                                            <?php esc_html_e( 'Explore Pro version', 'sfr-directory-map' ); ?>
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

            <div style="margin-top: 20px;background:linear-gradient(135deg,#076668,#005F73);color:#fff;padding:24px;border-radius:14px;display:flex;align-items:center;justify-content:space-between;flex-wrap:wrap;gap:15px;box-shadow:0 20px 40px rgba(0,0,0,0.25);">
                <div>
                    <h3 style="margin:0 0 5px 0;color:#fff;"><?php esc_html_e( 'Need more free website insight tools?', 'sfr-directory-map' ); ?></h3>
                    <p style="margin:0;color:#fff;"><?php esc_html_e( 'Audit performance, scan SEO, generate sitemaps, and more inside the Support From Richard tools directory.', 'sfr-directory-map' ); ?></p>
                </div>
                <a class="button button-primary" style="background:#F9B702;color:#0f172a;border:none;" href="https://supportfromrichard.co.uk/tools-directory/" target="_blank" rel="noopener">
                    <?php esc_html_e( 'Free website analysis insight tools', 'sfr-directory-map' ); ?>
                </a>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Build Support From Richard suite status array.
     *
     * @return array
     */
    private function sfrdm_get_suite_status() {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins          = get_plugins();
        $has_directorist  = function_exists( 'is_plugin_active' ) && is_plugin_active( 'directorist/directorist-base.php' );
        $is_plugin_active = function_exists( 'is_plugin_active' ) ? 'is_plugin_active' : null;

        $config = array(
            'category_display' => array(
                'title'            => __( 'Directory Category Display', 'sfr-directory-map' ),
                'description'      => __( 'Display engaging category layouts with imagery and SEO tips.', 'sfr-directory-map' ),
                'free_slug'        => 'sfr-directory-category-display/sfr-directory-category-display.php',
                'pro_slug'         => 'sfr-directory-category-display-pro/sfr-directory-category-display.php',
                'analytics_page'   => 'admin.php?page=udcd-analytics',
                'free_url'         => 'https://wordpress.org/plugins/sfr-directory-category-display/',
                'pro_url'          => 'https://supportfromrichard.co.uk/plugins/sfr-directory-category-display/',
                'free_only_note'   => __( 'Unlock icon packs, layouts, and animations with the Pro upgrade.', 'sfr-directory-map' ),
                'pro_teaser'       => __( 'Pro adds animation packs, custom icon sets, and more layout controls for agencies.', 'sfr-directory-map' ),
            ),
            'directory_map' => array(
                'title'            => __( 'Directory Map', 'sfr-directory-map' ),
                'description'      => __( 'Interactive listing maps with filters, clustering, and analytics.', 'sfr-directory-map' ),
                'free_slug'        => 'sfr-directory-map/sfr-directory-map.php',
                'pro_slug'         => 'sfr-directory-map-pro/sfr-directory-map.php',
                'analytics_page'   => 'admin.php?page=sfrdm-analytics',
                'free_url'         => 'https://wordpress.org/plugins/sfr-directory-map/',
                'pro_url'          => 'https://supportfromrichard.co.uk/sfr-directory-map/',
                'notes'            => $has_directorist ? '' : __( 'Pro mapping currently requires Directorist listings.', 'sfr-directory-map' ),
                'free_only_note'   => __( 'Heatmaps, custom themes, and CSV export land in the Pro edition.', 'sfr-directory-map' ),
                'pro_teaser'       => __( 'Unlock custom pin colours, reusable colour templates, tag badges, and premium search analytics built for large directories.', 'sfr-directory-map' ),
                'requires_directorist' => true,
                'always_show_pro_link' => true,
            ),
            'directory_analytics' => array(
                'title'            => __( 'Directory Analytics', 'sfr-directory-map' ),
                'description'      => __( 'Track listing and category engagement across your directory.', 'sfr-directory-map' ),
                'free_slug'        => 'sfr-directory-analytics/sfr-directory-analytics.php',
                'pro_slug'         => 'sfr-directory-analytics-pro/sfr-directory-analytics.php',
                'analytics_page'   => 'admin.php?page=sfrda-analytics',
                'free_url'         => 'https://wordpress.org/plugins/sfr-directory-analytics/',
                'pro_url'          => 'https://supportfromrichard.co.uk/sfr-directory-analytics/',
                'free_only_note'   => __( 'Upgrade to Pro for HTML exports, pagination, and scheduled email reports.', 'sfr-directory-map' ),
                'pro_teaser'       => __( 'Professional analytics add CSV exports, scheduled email summaries, and sortable pagination controls.', 'sfr-directory-map' ),
            ),
        );

        $results = array();

        foreach ( $config as $plugin_config ) {
            $free_slug = $plugin_config['free_slug'];
            $pro_slug  = $plugin_config['pro_slug'];

            $installed_free = isset( $plugins[ $free_slug ] );
            $installed_pro  = isset( $plugins[ $pro_slug ] );

            $is_active_free = $is_plugin_active ? $is_plugin_active( $free_slug ) : in_array( $free_slug, (array) get_option( 'active_plugins', array() ), true );
            $is_active_pro  = $is_plugin_active ? $is_plugin_active( $pro_slug ) : in_array( $pro_slug, (array) get_option( 'active_plugins', array() ), true );

            $active_variant = '';
            if ( $is_active_pro ) {
                $active_variant = 'pro';
            } elseif ( $is_active_free ) {
                $active_variant = 'free';
            }

            $analytics_url = '';
            if ( ! empty( $plugin_config['analytics_page'] ) && ( $is_active_pro || $is_active_free ) ) {
                $analytics_url = admin_url( $plugin_config['analytics_page'] );
            }

            $requires_directorist = ! empty( $plugin_config['requires_directorist'] );
            $can_suggest_pro      = ! $is_active_pro;
            if ( $requires_directorist && ! $has_directorist ) {
                $can_suggest_pro = false;
            }

            $always_show_pro_link = ! empty( $plugin_config['always_show_pro_link'] );

            $results[] = array(
                'title'          => $plugin_config['title'],
                'description'    => $plugin_config['description'],
                'active'         => ( $is_active_free || $is_active_pro ),
                'active_variant' => $active_variant,
                'analytics_url'  => $analytics_url,
                'free_url'       => $plugin_config['free_url'],
                'pro_url'        => $plugin_config['pro_url'],
                'show_pro_link'  => ( $can_suggest_pro || $always_show_pro_link ),
                'free_only_note' => ( $is_active_free && ! $is_active_pro && ! empty( $plugin_config['free_only_note'] ) ) ? $plugin_config['free_only_note'] : '',
                'pro_teaser'     => ! empty( $plugin_config['pro_teaser'] ) ? $plugin_config['pro_teaser'] : '',
                'notes'          => ( $requires_directorist && ! $has_directorist ) ? ( $plugin_config['notes'] ?? '' ) : '',
            );
        }

        return $results;
    }
}
