/**
 * Universal Directory Map JavaScript
 * Handles map initialization and listing display
 */

(function($) {
    'use strict';
    
    var udmMaps = {};
    var udmMarkers = {};
    var udmAllMarkers = {}; // Store all markers separately for accurate counting
    
    /**
     * Improved mobile device detection
     */
    function isMobileDevice() {
        // More aggressive mobile detection
        var hasTouch = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
        var isSmallScreen = window.innerWidth <= 768;
        var isIOS = /iPad|iPhone|iPod/.test(navigator.userAgent);
        var isAndroid = /Android/.test(navigator.userAgent);
        var isChromeDevTools = window.outerHeight - window.innerHeight > 100; // DevTools open
        
        // Consider it mobile if it has touch OR small screen OR mobile OS OR DevTools mobile simulation
        var isMobile = hasTouch || isSmallScreen || isIOS || isAndroid || isChromeDevTools;
        
        return isMobile;
    }
    
    /**
     * Initialize all maps on page load
     */
    $(document).ready(function() {
        $('.udm-map').each(function() {
            var mapId = $(this).attr('id');
            if (mapId) {
                initUniversalMap(mapId);
            }
        });
        
        // Initialize category filters
        $('.udm-category-filter').on('change', function() {
            var mapContainer = $(this).closest('.udm-map-container');
            var mapElement = mapContainer.find('.udm-map');
            var mapId = mapElement.attr('id');
            var selectedCategory = $(this).val();
            
            if (mapId && udmMaps[mapId]) {
                filterMapByCategory(mapId, selectedCategory);
            }
        });
        
        // Initialize search functionality
        $('.udm-search-input').on('input', function() {
            var $input = $(this);
            var mapContainer = $input.closest('.udm-map-container');
            var mapElement = mapContainer.find('.udm-map');
            var mapId = mapElement.attr('id');
            var searchTerm = $input.val();
            
            // Show searching state for live search
            if (searchTerm.trim() !== '') {
                $input.addClass('udm-search-typing');
            } else {
                $input.removeClass('udm-search-typing');
            }
            
            if (mapId && udmMaps[mapId]) {
                searchMapListings(mapId, searchTerm);
            }
        });
        
        // Handle Enter key in search input (no longer needed since we have live search)
        $('.udm-search-input').on('keypress', function(e) {
            if (e.which === 13) { // Enter key
                e.preventDefault();
                // Just focus the input - live search handles the rest
                $(this).focus();
            }
        });
        
        // Initialize clear search button functionality
        $('.udm-clear-search-button').on('click', function() {
            var mapContainer = $(this).closest('.udm-map-container');
            var searchInput = mapContainer.find('.udm-search-input');
            var mapElement = mapContainer.find('.udm-map');
            var mapId = mapElement.attr('id');
            
            // Clear the search input
            searchInput.val('');
            
            // Clear search results
            if (mapId && udmMaps[mapId]) {
                searchMapListings(mapId, '');
            }
            
            // Focus back to input
            searchInput.focus();
        });
    });
    
    /**
     * Initialize a single map instance
     */
    function initUniversalMap(mapId) {
        var mapElement = document.getElementById(mapId);
        if (!mapElement) {
            return;
        }
        
        var $mapElement = $(mapElement);
        var $mapContainer = $mapElement.closest('.udm-map-container');
        
        // Get configuration from map settings or fallback to legacy
        var config = {};
        
        // Check if this is a new map with configuration
        if ($mapContainer.data('map-config')) {
            try {
                config = $mapContainer.data('map-config');
                config.mapId = $mapElement.data('map-id');
                config.uniqueId = mapId;
            } catch (e) {
                config = getLegacyConfig($mapElement, mapId);
            }
        } else {
            // Legacy configuration
            config = getLegacyConfig($mapElement, mapId);
        }
        
        // Validate coordinates
        if (isNaN(config.default_lat || config.lat) || isNaN(config.default_lng || config.lng)) {
            showMapError(mapElement, sfrdmSettings.strings.error_loading);
            return;
        }
        
        try {
            // Apply custom button colours
            if (sfrdmSettings.button_colors) {
                mapElement.style.setProperty('--sfrdm-primary-color', sfrdmSettings.button_colors.primary);
                mapElement.style.setProperty('--sfrdm-secondary-color', sfrdmSettings.button_colors.secondary);
                mapElement.style.setProperty('--sfrdm-hover-color', sfrdmSettings.button_colors.hover);
                mapElement.style.setProperty('--sfrdm-text-color', sfrdmSettings.button_colors.text);
            }
            
            // Initialize map
            var map = L.map(mapId, {
                zoomControl: true,
                attributionControl: true
            }).setView([config.default_lat || config.lat, config.default_lng || config.lng], config.default_zoom || config.zoom);
            
            // Add map tiles based on style
            var tileLayer = getTileLayer(config.map_style || 'openstreetmap');
            
            // Add error handling for tile loading
            tileLayer.on('tileerror', function(error) {
                
                // Remove the failed tile layer
                map.removeLayer(tileLayer);
                
                // Add OpenStreetMap as fallback
                var fallbackLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                    maxZoom: 18,
                    subdomains: ['a', 'b', 'c']
                });
                fallbackLayer.addTo(map);
            });
            
            // Add loading success handler
            tileLayer.on('tileload', function() {
            });
            
            // Add loading start handler
            tileLayer.on('loading', function() {
                var mapContainer = document.querySelector('#' + mapId);
                if (mapContainer) {
                    mapContainer.style.opacity = '0.7';
                }
            });
            
            // Add loading complete handler
            tileLayer.on('load', function() {
                var mapContainer = document.querySelector('#' + mapId);
                if (mapContainer) {
                    mapContainer.style.opacity = '1';
                }
            });
            
            // Add timeout check for slow-loading tiles
            var tileLoadTimeout = setTimeout(function() {
                
                // Check if any tiles have loaded
                var mapContainer = document.querySelector('#' + mapId);
                var tiles = mapContainer.querySelectorAll('.leaflet-tile');
                
                if (tiles.length === 0) {
                    
                    // Remove the slow tile layer
                    map.removeLayer(tileLayer);
                    
                    // Add OpenStreetMap as fallback
                    var fallbackLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                        attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                        maxZoom: 18,
                        subdomains: ['a', 'b', 'c']
                    });
                    fallbackLayer.addTo(map);
                }
            }, 10000); // 10 second timeout
            
            // Clear timeout when tiles load successfully
            tileLayer.on('tileload', function() {
                clearTimeout(tileLoadTimeout);
            });
            
            tileLayer.addTo(map);
            
            // Initialize marker cluster group if enabled
            var markers = null;
            if (config.cluster_markers || config.cluster) {
                markers = L.markerClusterGroup({
                    chunkedLoading: true,
                    maxClusterRadius: 50,
                    spiderfyOnMaxZoom: true,
                    showCoverageOnHover: false,
                    zoomToBoundsOnClick: true
                });
            }
            
            // Store map reference
            udmMaps[mapId] = map;
            udmMarkers[mapId] = markers;
            udmAllMarkers[mapId] = []; // Initialize array to store all markers
            
            // Load listings
            loadListings(config, map, markers);
            
        } catch (error) {
            showMapError(mapElement, sfrdmSettings.strings.error_loading);
        }
    }
    
    /**
     * Load listings via AJAX
     */
    function loadListings(config, map, markers) {
        showMapLoading(config.uniqueId || config.mapId, true);
        
        $.ajax({
            url: sfrdmSettings.ajax_url,
            type: 'POST',
            data: {
                action: 'sfrdm_get_directory_listings',
                nonce: sfrdmSettings.nonce,
                category: config.category,
                post_type: config.postType || config.post_type,
                map_id: config.mapId || 0
            },
            timeout: 30000,
            success: function(response) {
                showMapLoading(config.uniqueId || config.mapId, false);
                
                if (response.success && response.data && response.data.listings) {
                    displayListings(response.data.listings, map, markers, config);
                } else {
                    var message = response.data && response.data.message 
                        ? response.data.message 
                        : sfrdmSettings.strings.no_listings;
                    showMapMessage(config.mapId, message);
                }
            },
            error: function(xhr, status, error) {
                showMapLoading(config.uniqueId || config.mapId, false);
                
                var message = status === 'timeout' 
                    ? 'Request timed out. Please try again.'
                    : sfrdmSettings.strings.error_loading;
                    
                showMapError(document.getElementById(config.uniqueId || config.mapId), message);
            }
        });
    }
    
    /**
     * Display listings on map
     */
    function displayListings(listings, map, markers, config) {
        if (!Array.isArray(listings) || listings.length === 0) {
            showMapMessage(config.uniqueId || config.mapId, sfrdmSettings.strings.no_listings);
            return;
        }
        
        var bounds = [];
        var validListings = 0;
        
        listings.forEach(function(listing) {
            if (!isValidListing(listing)) {
                return;
            }
            
            try {
                var mapElement = document.getElementById(config.uniqueId || config.mapId);
                var popupContent = createPopupContent(listing, mapElement);
                var marker = createMarker(listing, popupContent);
                
                
                if ((config.cluster_markers || config.cluster) && markers) {
                    markers.addLayer(marker);
                } else {
                    marker.addTo(map);
                }
                
                // Store marker in all markers array for accurate counting
                udmAllMarkers[config.uniqueId || config.mapId].push(marker);
                
                
                bounds.push([listing.lat, listing.lng]);
                validListings++;
                
            } catch (error) {
            }
        });
        
        // Add cluster layer to map if using clustering
        if ((config.cluster_markers || config.cluster) && markers && validListings > 0) {
            map.addLayer(markers);
        }
        
        // Fit map to show all markers (only if auto_fit_bounds is enabled)
        if (bounds.length > 0 && config.auto_fit_bounds !== false) {
            try {
                map.fitBounds(bounds, {
                    padding: [20, 20],
                    maxZoom: 15
                });
            } catch (error) {
                // Silently handle error
            }
        }
        
        // Show summary
        showMapMessage(config.uniqueId || config.mapId, validListings + ' listings displayed', 'success');
    }
    
    /**
     * Validate listing data
     */
    function isValidListing(listing) {
        return listing && 
               typeof listing.lat === 'number' && 
               typeof listing.lng === 'number' &&
               listing.lat >= -90 && listing.lat <= 90 &&
               listing.lng >= -180 && listing.lng <= 180 &&
               (listing.lat !== 0 || listing.lng !== 0) &&
               listing.title && listing.title.length > 0;
    }
    
    /**
     * Create marker for listing
     */
    function createMarker(listing, popupContent) {
        var markerOptions = {};
        
        // Debug logging for marker creation
        
        // Create colored pin based on category
        
        try {
            // Use the pin_color from the listing data (calculated by PHP)
            var color = listing.pin_color || getCategoryColor(listing.category_name, listing.category_slug, sfrdmSettings ? sfrdmSettings.defaultColors : {});
            
            // Modern SVG pin design with shadow and gradient effects
            var svgIcon = L.icon({
                iconUrl: 'data:image/svg+xml;base64,' + btoa(
                    '<svg width="32" height="40" viewBox="0 0 32 40" xmlns="http://www.w3.org/2000/svg">' +
                    // Drop shadow for depth
                    '<path d="M16 2C8.268 2 2 8.268 2 16c0 10 14 22 14 22s14-12 14-22c0-7.732-6.268-14-14-14z" fill="rgba(0,0,0,0.3)" transform="translate(1,1)"/>' +
                    // Main pin body with white border
                    '<path d="M16 2C8.268 2 2 8.268 2 16c0 10 14 22 14 22s14-12 14-22c0-7.732-6.268-14-14-14z" fill="' + color + '" stroke="white" stroke-width="2"/>' +
                    // Inner highlight circle
                    '<circle cx="16" cy="16" r="7" fill="white" opacity="0.8"/>' +
                    // Center dot
                    '<circle cx="16" cy="16" r="4" fill="' + color + '"/>' +
                    '</svg>'
                ),
                iconSize: [32, 40],
                iconAnchor: [16, 40],
                popupAnchor: [0, -40]
            });
            
            /* 
             * ALTERNATIVE PIN DESIGNS - Uncomment any of these to try different styles:
             * 
             * 1. MINIMAL CIRCLE PIN:
             * iconUrl: 'data:image/svg+xml;base64,' + btoa(
             *     '<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">' +
             *     '<circle cx="12" cy="12" r="10" fill="' + color + '" stroke="white" stroke-width="3"/>' +
             *     '</svg>'
             * )
             * 
             * 2. SQUARE MODERN PIN:
             * iconUrl: 'data:image/svg+xml;base64,' + btoa(
             *     '<svg width="24" height="30" viewBox="0 0 24 30" xmlns="http://www.w3.org/2000/svg">' +
             *     '<rect x="2" y="2" width="20" height="20" rx="4" fill="' + color + '" stroke="white" stroke-width="2"/>' +
             *     '<polygon points="12,22 8,30 16,30" fill="' + color + '"/>' +
             *     '<circle cx="12" cy="12" r="4" fill="white"/>' +
             *     '</svg>'
             * )
             * 
             * 3. DIAMOND PIN:
             * iconUrl: 'data:image/svg+xml;base64,' + btoa(
             *     '<svg width="24" height="30" viewBox="0 0 24 30" xmlns="http://www.w3.org/2000/svg">' +
             *     '<path d="M12 2l8 8v8l-8 8-8-8v-8z" fill="' + color + '" stroke="white" stroke-width="2"/>' +
             *     '<circle cx="12" cy="12" r="3" fill="white"/>' +
             *     '</svg>'
             * )
             */
            
            markerOptions.icon = svgIcon;
            
        } catch (error) {
            
            // Fallback to divIcon
            try {
                var color = listing.pin_color || getCategoryColor(listing.category_name, listing.category_slug, sfrdmSettings ? sfrdmSettings.defaultColors : {});
                markerOptions.icon = L.divIcon({
                    className: 'udm-simple-pin',
                    html: '<div style="background-color: ' + color + '; width: 20px; height: 20px; border-radius: 50%; border: 2px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>',
                    iconSize: [24, 24],
                    iconAnchor: [12, 12]
                });
            } catch (divError) {
            }
        }
        
        var marker = L.marker([listing.lat, listing.lng], markerOptions);
        
        // Attach listing data to marker for search functionality
        marker.listingData = listing;
        
        // Debug: Check if marker was created
        
        if (popupContent) {
            marker.bindPopup(popupContent, {
                maxWidth: 320,
                minWidth: 280,
                className: 'udm-popup udm-popup-enhanced'
            });
        }
        
        return marker;
    }
    
    /**
     * Create custom icon with colour and optional emoji
     */
    function createCustomIcon(color, icon) {
        var html = '';
        
        if (icon && icon.trim()) {
            // Use emoji/icon with coloured background
            html = '<div style="background-color: ' + color + '; width: 24px; height: 24px; border-radius: 50%; border: 2px solid #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.3); display: flex; align-items: center; justify-content: center; font-size: 12px;">' + icon + '</div>';
        } else {
            // Use coloured circle
            html = '<div style="background-color: ' + color + '; width: 20px; height: 20px; border-radius: 50%; border: 2px solid #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>';
        }
        
        return L.divIcon({
            className: 'udm-custom-marker',
            html: html,
            iconSize: icon ? [24, 24] : [20, 20],
            iconAnchor: icon ? [12, 12] : [10, 10]
        });
    }
    
    /**
     * Get category color for pin
     */
    function getCategoryColor(categoryName, categorySlug, defaultColors) {
        
        // Check if custom colors are set for this category
        if (defaultColors && defaultColors[categorySlug]) {
            return defaultColors[categorySlug];
        }
        
        // Default color mapping for common categories
        var categoryColors = {
            'accommodation': '#0073aa',
            'cottages-and-apartments': '#28a745',
            'glamping-camping': '#ffc107',
            'hotels': '#dc3545',
            'restaurants': '#fd7e14',
            'shopping': '#6f42c1',
            'services': '#20c997',
            'healthcare': '#e83e8c',
            'education': '#17a2b8'
        };
        
        // Try to match by slug first, then by name
        var color = categoryColors[categorySlug];
        
        if (!color) {
            // Try to match by partial name
            var categoryLower = categoryName.toLowerCase();
            
            if (categoryLower.includes('accommodation') || categoryLower.includes('hotel')) {
                color = '#0073aa';
            } else if (categoryLower.includes('cottage') || categoryLower.includes('apartment')) {
                color = '#28a745';
            } else if (categoryLower.includes('camping') || categoryLower.includes('glamping')) {
                color = '#ffc107';
            } else if (categoryLower.includes('restaurant') || categoryLower.includes('food')) {
                color = '#fd7e14';
            } else {
                color = '#6c757d'; // Default grey
            }
        }
        
        return color;
    }

    /**
     * Create colored pin marker based on category
     */
    function createColoredPin(categoryName, categorySlug, defaultColors) {
        var color = getCategoryColor(categoryName, categorySlug, defaultColors);
        
        // Debug logging
        
        var html = '<div style="background-color: ' + color + '; width: 24px; height: 24px; border-radius: 50%; border: 3px solid #fff; box-shadow: 0 3px 8px rgba(0,0,0,0.3); position: relative; display: block; visibility: visible; opacity: 1;">' +
                   '<div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); width: 8px; height: 8px; background-color: #fff; border-radius: 50%;"></div>' +
                   '</div>';
        
        
        var icon = L.divIcon({
            className: 'udm-colored-pin',
            html: html,
            iconSize: [30, 30],
            iconAnchor: [15, 15]
        });
        
        return icon;
    }
    
    /**
     * Create popup content for listing
     */
    function createPopupContent(listing, mapElement) {
        // Get display settings from map's data attributes (premium gating)
        var showPhone = mapElement ? (mapElement.getAttribute('data-show-phone') === '1') : false;
        var showEmail = mapElement ? (mapElement.getAttribute('data-show-email') === '1') : false;
        var showWebsite = mapElement ? (mapElement.getAttribute('data-show-website') === '1') : false;
        var showTags = mapElement ? (mapElement.getAttribute('data-show-tags') === '1') : false;
        var frameColor = mapElement ? mapElement.getAttribute('data-frame-color') : '';
        
        
        var content = '<div class="udm-popup-content">';
        
        // Main image - prominent display (clickable if URL exists)
        if (listing.image && listing.image.length > 0) {
            content += '<div class="udm-popup-image">';
            if (listing.url && listing.url.length > 0) {
                content += '<a href="' + escapeHtml(listing.url) + '" target="_blank" rel="noopener" style="display: block; width: 100%; height: 100%;">';
            }
            content += '<img src="' + escapeHtml(listing.image) + '" ';
            content += 'alt="' + escapeHtml(listing.title) + '" ';
            content += 'class="udm-listing-image" loading="lazy" ';
            content += 'style="cursor: ' + (listing.url ? 'pointer' : 'default') + ';" ';
            if (listing.url && listing.url.length > 0) {
                content += '</a>';
            }
            content += '</div>';
        } else {
            // Show placeholder when no image
            content += '<div class="udm-popup-image">';
            content += '<div style="display: flex; align-items: center; justify-content: center; height: 100%; color: #ccc; font-size: 48px;">📷</div>';
            content += '</div>';
        }
        
        // Title with link - clean and prominent (decode HTML entities)
        var decodedTitle = listing.title.replace(/&#(\d+);/g, function(match, dec) {
            return String.fromCharCode(dec);
        });
        
        content += '<div class="udm-popup-header">';
        content += '<h3 class="udm-popup-title">';
        if (listing.url && listing.url.length > 0) {
            content += '<a href="' + escapeHtml(listing.url) + '" target="_blank" rel="noopener">';
            content += escapeHtml(decodedTitle);
            content += '</a>';
        } else {
            content += escapeHtml(decodedTitle);
        }
        content += '</h3>';
        content += '</div>';
        
        // Essential info only - clean and minimal
        content += '<div class="udm-popup-body">';
        
        // Categories - as badges
        if (listing.categories && Array.isArray(listing.categories) && listing.categories.length > 0) {
            content += '<div class="udm-popup-categories">';
            listing.categories.forEach(function(category) {
                content += '<span class="udm-category-badge">' + escapeHtml(category) + '</span>';
            });
            content += '</div>';
        }
        
        // Smart address display - prefer postcode/zip, fallback to full address
        var displayAddress = '';
        if (listing.address && listing.address.length > 0) {
            // Try to extract postcode/zip code from address
            var postcodeMatch = listing.address.match(/\b[A-Z]{1,2}[0-9R][0-9A-Z]? [0-9][A-Z]{2}\b|\b[0-9]{5}(-[0-9]{4})?\b|\b[A-Z][0-9][A-Z] [0-9][A-Z][0-9]\b/gi);
            
            if (postcodeMatch && postcodeMatch.length > 0) {
                // Found postcode/zip - show it with city
                var addressParts = listing.address.split(',');
                var cityPart = '';
                var postcodePart = postcodeMatch[0];
                
                // Find city (usually the part before postcode)
                for (var i = 0; i < addressParts.length; i++) {
                    if (addressParts[i].trim().toLowerCase().includes(postcodePart.toLowerCase())) {
                        if (i > 0) {
                            cityPart = addressParts[i-1].trim();
                        }
                        break;
                    }
                }
                
                displayAddress = cityPart + ', ' + postcodePart;
            } else {
                // No postcode found - show first two parts of address (usually street + city)
                var addressParts = listing.address.split(',').map(function(part) { return part.trim(); });
                displayAddress = addressParts.slice(0, 2).join(', ');
            }
            
            content += '<div class="udm-popup-field udm-popup-address">';
            content += '<span class="udm-field-icon">📍</span>';
            content += '<span class="udm-field-text">' + escapeHtml(displayAddress) + '</span>';
            content += '</div>';
        }
        
        // Contact buttons section
        var hasContactInfo = (showPhone && listing.phone && listing.phone.length > 0) ||
                           (showEmail && listing.email && listing.email.length > 0) ||
                           (showWebsite && listing.website && listing.website.length > 0);
        
        if (hasContactInfo) {
            content += '<div class="udm-popup-contact-buttons">';
            
            // Phone button
            if (showPhone && listing.phone && listing.phone.length > 0) {
                content += '<a href="tel:' + escapeHtml(listing.phone) + '" class="udm-contact-btn udm-phone-btn">';
                content += '<span class="udm-btn-icon">📞</span>';
                content += '<span class="udm-btn-text">Call</span>';
                content += '</a>';
            }
            
            // Email button
            if (showEmail && listing.email && listing.email.length > 0) {
                content += '<a href="mailto:' + escapeHtml(listing.email) + '" class="udm-contact-btn udm-email-btn">';
                content += '<span class="udm-btn-icon">✉️</span>';
                content += '<span class="udm-btn-text">Email</span>';
                content += '</a>';
            }
            
            // Website button
            if (showWebsite && listing.website && listing.website.length > 0) {
                content += '<a href="' + escapeHtml(listing.website) + '" target="_blank" rel="noopener" class="udm-contact-btn udm-website-btn">';
                content += '<span class="udm-btn-icon">🌐</span>';
                content += '<span class="udm-btn-text">Website</span>';
                content += '</a>';
            }
            
            content += '</div>';
        }
        
        // Tags - display as small badges (premium feature)
        if (showTags && listing.tags && Array.isArray(listing.tags) && listing.tags.length > 0) {
            content += '<div class="udm-popup-tags">';
            listing.tags.forEach(function(tag) {
                content += '<span class="udm-tag">' + escapeHtml(tag.name) + '</span>';
            });
            content += '</div>';
        }
        
        // Action button
        if (listing.url && listing.url.length > 0) {
            content += '<div class="udm-popup-actions">';
            var buttonStyle = '';
            if (frameColor) {
                // Create gradient with frame color
                var rgb = parseInt(frameColor.slice(1), 16);
                var r = (rgb >> 16) & 255;
                var g = (rgb >> 8) & 255;
                var b = rgb & 255;
                var darkerColor = 'rgb(' + Math.max(0, r - 30) + ',' + Math.max(0, g - 30) + ',' + Math.max(0, b - 30) + ')';
                buttonStyle = ' style="background: linear-gradient(135deg, ' + frameColor + ' 0%, ' + darkerColor + ' 100%);"';
            }
            content += '<a href="' + escapeHtml(listing.url) + '" target="_blank" rel="noopener" class="udm-view-listing-btn"' + buttonStyle + '>';
            content += 'View Listing';
            content += '</a>';
            content += '</div>';
        }
        
        content += '</div>'; // Close popup-body
        content += '</div>'; // Close popup-content
        
        return content;
    }
    
    /**
     * Show loading state
     */
    function showMapLoading(mapId, show) {
        var $map = $('#' + mapId);
        var $loading = $map.find('.udm-loading');
        
        if (show) {
            if ($loading.length === 0) {
                $loading = $('<div class="udm-loading"><p>' + sfrdmSettings.strings.loading + '</p></div>');
                $map.append($loading);
            }
            $loading.show();
        } else {
            $loading.hide();
        }
    }
    
    /**
     * Show message on map
     */
    function showMapMessage(mapId, message, type) {
        type = type || 'info';
        var $map = $('#' + mapId);
        var $existing = $map.find('.udm-message');
        
        // Remove existing messages
        $existing.remove();
        
        if (message && message.length > 0) {
            var $message = $('<div class="udm-message udm-message-' + type + '"><p>' + escapeHtml(message) + '</p></div>');
            $map.append($message);
            
            // Auto-hide success messages after 3 seconds
            if (type === 'success') {
                setTimeout(function() {
                    $message.fadeOut(500);
                }, 3000);
            }
        }
    }
    
    /**
     * Show error message
     */
    function showMapError(mapElement, message) {
        var $map = $(mapElement);
        var $error = $('<div class="udm-error"><p>' + escapeHtml(message) + '</p></div>');
        $map.html($error);
    }
    
    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        if (typeof text !== 'string') {
            return '';
        }
        
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        
        return text.replace(/[&<>"']/g, function(m) {
            return map[m];
        });
    }
    
    /**
     * Get legacy configuration for backward compatibility
     */
    function getLegacyConfig($mapElement, mapId) {
        return {
            lat: parseFloat(sfrdmSettings.default_lat),
            lng: parseFloat(sfrdmSettings.default_lng),
            zoom: parseInt(sfrdmSettings.default_zoom),
            category: $mapElement.data('category') || '',
            postType: $mapElement.data('post-type') || 'auto',
            cluster: $mapElement.data('cluster') === 1,
            mapId: mapId,
            uniqueId: mapId
        };
    }
    
    /**
     * Get tile layer based on map style
     */
    function getTileLayer(style) {
        var tileLayers = {
            // Free Map Styles
            'openstreetmap': L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                maxZoom: 18,
                subdomains: ['a', 'b', 'c']
            }),
            'cartodb_positron': L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors &copy; <a href="https://carto.com/attributions">CARTO</a>',
                subdomains: 'abcd',
                maxZoom: 20
            }),
            'cartodb_dark': L.tileLayer('https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors &copy; <a href="https://carto.com/attributions">CARTO</a>',
                subdomains: 'abcd',
                maxZoom: 20
            }),
            
            // Premium Map Styles - Using reliable tile services
            'stamen_terrain': L.tileLayer('https://stamen-tiles-{s}.a.ssl.fastly.net/terrain/{z}/{x}/{y}.jpg', {
                attribution: 'Map tiles by <a href="http://stamen.com">Stamen Design</a>, <a href="http://creativecommons.org/licenses/by/3.0">CC BY 3.0</a> &mdash; Map data &copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                subdomains: 'abcd',
                maxZoom: 18,
                detectRetina: false
            }),
            'stamen_toner': L.tileLayer('https://stamen-tiles-{s}.a.ssl.fastly.net/toner/{z}/{x}/{y}.png', {
                attribution: 'Map tiles by <a href="http://stamen.com">Stamen Design</a>, <a href="http://creativecommons.org/licenses/by/3.0">CC BY 3.0</a> &mdash; Map data &copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                subdomains: 'abcd',
                maxZoom: 20,
                detectRetina: false
            }),
            'stamen_watercolor': L.tileLayer('https://stamen-tiles-{s}.a.ssl.fastly.net/watercolor/{z}/{x}/{y}.jpg', {
                attribution: 'Map tiles by <a href="http://stamen.com">Stamen Design</a>, <a href="http://creativecommons.org/licenses/by/3.0">CC BY 3.0</a> &mdash; Map data &copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                subdomains: 'abcd',
                maxZoom: 16,
                detectRetina: false
            }),
            'cartodb_voyager': L.tileLayer('https://{s}.basemaps.cartocdn.com/rastertiles/voyager/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors &copy; <a href="https://carto.com/attributions">CARTO</a>',
                subdomains: 'abcd',
                maxZoom: 20
            }),
            
            // Historical Map Styles
            'stamen_toner_lite': L.tileLayer('https://stamen-tiles-{s}.a.ssl.fastly.net/toner-lite/{z}/{x}/{y}.png', {
                attribution: 'Map tiles by <a href="http://stamen.com">Stamen Design</a>, <a href="http://creativecommons.org/licenses/by/3.0">CC BY 3.0</a> &mdash; Map data &copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                subdomains: 'abcd',
                maxZoom: 20,
                detectRetina: false
            }),
            'cartodb_vintage': L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors &copy; <a href="https://carto.com/attributions">CARTO</a>',
                subdomains: 'abcd',
                maxZoom: 20,
                className: 'cartodb-vintage-style' // Custom CSS class for vintage styling
            }),
            
            // Additional Reliable Styles
            'esri_world_imagery': L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}', {
                attribution: 'Tiles &copy; Esri &mdash; Source: Esri, i-cubed, USDA, USGS, AEX, GeoEye, Getmapping, Aerogrid, IGN, IGP, UPR-EGP, and the GIS User Community',
                maxZoom: 18
            }),
            'esri_world_street': L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}', {
                attribution: 'Tiles &copy; Esri &mdash; Source: Esri, DeLorme, NAVTEQ, USGS, Intermap, iPC, NRCAN, Esri Japan, METI, Esri China (Hong Kong), Esri (Thailand), TomTom, 2012',
                maxZoom: 18
            })
        };
        
        return tileLayers[style] || tileLayers['openstreetmap'];
    }
    
    /**
     * Filter map by category
     */
    function filterMapByCategory(mapId, category) {
        var map = udmMaps[mapId];
        var markers = udmMarkers[mapId];
        
        if (!map || !markers) return;
        
        // Clear existing markers
        if (markers.clearLayers) {
            markers.clearLayers();
        } else {
            map.eachLayer(function(layer) {
                if (layer instanceof L.Marker) {
                    map.removeLayer(layer);
                }
            });
        }
        
        // Reload listings with category filter
        var mapElement = document.getElementById(mapId);
        var $mapElement = $(mapElement);
        var $mapContainer = $mapElement.closest('.udm-map-container');
        var config = $mapContainer.data('map-config') || getLegacyConfig($mapElement, mapId);
        
        // Update config with selected category
        config.category = category;
        
        // Reload listings
        loadListings(config, map, markers);
    }
    
    /**
     * Search map listings
     */
    function searchMapListings(mapId, searchTerm) {
        
        var map = udmMaps[mapId];
        if (!map) {
            return;
        }
        
        var visibleCount = 0;
        var totalCount = 0;
        var visibleMarkers = [];
        
        // Get all markers from our stored array for accurate counting
        var allMarkers = udmAllMarkers[mapId] || [];
        var markersToSearch = allMarkers.slice(); // Create a copy for searching
        
        
        markersToSearch.forEach(function(marker) {
            totalCount++;
            
            // Get the listing data from the marker
            var listingData = marker.listingData;
            var isVisible = false;
                
                
                if (searchTerm === '') {
                    // Show all when search is empty
                    isVisible = true;
                    // For clustered markers, ensure they're all back in the cluster group
                    var markers = udmMarkers[mapId];
                    if (markers && markers.getLayers) {
                        if (!markers.hasLayer(marker)) {
                            markers.addLayer(marker);
                        }
                    }
                } else if (listingData) {
                    // Search in listing data (case-insensitive)
                    var searchLower = searchTerm.toLowerCase();
                    var title = (listingData.title || '').toLowerCase();
                    var address = (listingData.address || '').toLowerCase();
                    var categories = Array.isArray(listingData.categories) ? 
                        listingData.categories.join(' ').toLowerCase() : 
                        (listingData.categories || '').toLowerCase();
                    var phone = (listingData.phone || '').toLowerCase();
                    
                    if (listingData) {
                        // Debug logging removed for production
                        
                        // Check if any field contains the search term
                        var titleContains = (listingData.title || '').toLowerCase().includes(searchLower);
                        var addressContains = (listingData.address || '').toLowerCase().includes(searchLower);
                        var categoriesContain = false;
                        var tagsContain = false;
                        
                        if (listingData.categories) {
                            if (Array.isArray(listingData.categories)) {
                                categoriesContain = listingData.categories.some(function(cat) {
                                    return cat.toLowerCase().includes(searchLower);
                                });
                            } else {
                                categoriesContain = listingData.categories.toLowerCase().includes(searchLower);
                            }
                        }
                        
                        if (listingData.tags && Array.isArray(listingData.tags)) {
                            tagsContain = listingData.tags.some(function(tag) {
                                var tagName = typeof tag === 'string' ? tag : (tag.name || '');
                                return tagName.toLowerCase().includes(searchLower);
                            });
                        }
                        
                    }
                    
                    // Search in tags
                    var tags = '';
                    if (listingData.tags && Array.isArray(listingData.tags)) {
                        tags = listingData.tags.map(function(tag) {
                            return tag.name || tag;
                        }).join(' ').toLowerCase();
                    }
                    
                    // Enhanced search with multiple strategies
                    var searchWords = searchLower.split(/\s+/).filter(function(word) {
                        return word.length > 0;
                    });
                    
                    // Add location synonyms and variations
                    var locationSynonyms = {
                        'york': ['yorkshire', 'yorks'],
                        'yorkshire': ['york', 'yorks'],
                        'north': ['northern'],
                        'south': ['southern'],
                        'east': ['eastern'],
                        'west': ['western'],
                        'moors': ['moor'],
                        'dales': ['dale'],
                        'district': ['area', 'region'],
                        'national park': ['np', 'nationalpark'],
                        'cornwall': ['cornish'],
                        'devon': ['devonshire'],
                        'scotland': ['scottish', 'scots'],
                        'wales': ['welsh'],
                        'england': ['english'],
                        'ireland': ['irish'],
                        'london': ['londoner'],
                        'manchester': ['manchestrian'],
                        'birmingham': ['brummie'],
                        'liverpool': ['scouser']
                    };
                    
                    // Expand search words with synonyms
                    var expandedSearchWords = [];
                    searchWords.forEach(function(word) {
                        expandedSearchWords.push(word);
                        if (locationSynonyms[word]) {
                            expandedSearchWords = expandedSearchWords.concat(locationSynonyms[word]);
                        }
                    });
                    
                    // Strategy 1: Exact phrase match (highest priority)
                    var exactMatch = title.includes(searchLower) || 
                                   address.includes(searchLower) || 
                                   categories.includes(searchLower) || 
                                   tags.includes(searchLower);
                    
                    // Strategy 2: Simple contains match (most flexible - this should catch "corn" in "Cornwall")
                    var simpleContainsMatch = title.includes(searchLower) || 
                                            address.includes(searchLower) || 
                                            categories.includes(searchLower) || 
                                            tags.includes(searchLower) ||
                                            phone.includes(searchLower);
                    
                    // Strategy 3: Word matching (flexible for single or multiple words)
                    var titleMatch = false;
                    var addressMatch = false;
                    var categoryMatch = false;
                    var tagMatch = false;
                    
                    if (searchWords.length > 0) {
                        // For single word searches, use simple contains check
                        if (searchWords.length === 1) {
                            var word = searchWords[0];
                            titleMatch = title.includes(word) || expandedSearchWords.some(function(expanded) {
                                return title.includes(expanded);
                            });
                            
                            addressMatch = address.includes(word) || expandedSearchWords.some(function(expanded) {
                                return address.includes(expanded);
                            });
                            
                            categoryMatch = categories.includes(word) || expandedSearchWords.some(function(expanded) {
                                return categories.includes(expanded);
                            });
                            
                            tagMatch = tags.includes(word) || expandedSearchWords.some(function(expanded) {
                                return tags.includes(expanded);
                            });
                        } else {
                            // For multiple words, require all words in same field
                            titleMatch = searchWords.every(function(word) {
                                return title.includes(word) || expandedSearchWords.some(function(expanded) {
                                    return title.includes(expanded);
                                });
                            });
                            
                            addressMatch = searchWords.every(function(word) {
                                return address.includes(word) || expandedSearchWords.some(function(expanded) {
                                    return address.includes(expanded);
                                });
                            });
                            
                            categoryMatch = searchWords.every(function(word) {
                                return categories.includes(word) || expandedSearchWords.some(function(expanded) {
                                    return categories.includes(expanded);
                                });
                            });
                            
                            tagMatch = searchWords.every(function(word) {
                                return tags.includes(word) || expandedSearchWords.some(function(expanded) {
                                    return tags.includes(expanded);
                                });
                            });
                        }
                    }
                    
                    // Strategy 4: Partial word matching (for partial location names)
                    var partialMatch = false;
                    if (searchWords.length > 0) {
                        var totalWords = searchWords.length;
                        var matchedWords = 0;
                        
                        // Count how many words match anywhere (including synonyms)
                        searchWords.forEach(function(word) {
                            var wordMatches = title.includes(word) || address.includes(word) || 
                                            categories.includes(word) || tags.includes(word);
                            
                            // Also check synonyms
                            if (!wordMatches && locationSynonyms[word]) {
                                wordMatches = locationSynonyms[word].some(function(synonym) {
                                    return title.includes(synonym) || address.includes(synonym) || 
                                           categories.includes(synonym) || tags.includes(synonym);
                                });
                            }
                            
                            if (wordMatches) {
                                matchedWords++;
                            }
                        });
                        
                        // If more than 50% of words match, consider it a match
                        partialMatch = (matchedWords / totalWords) >= 0.5;
                    }
                    
                    // Strategy 5: Single word matches (for simple searches)
                    var singleWordMatch = false;
                    if (searchWords.length === 1) {
                        singleWordMatch = title.includes(searchWords[0]) || 
                                        address.includes(searchWords[0]) || 
                                        categories.includes(searchWords[0]) || 
                                        tags.includes(searchWords[0]) ||
                                        phone.includes(searchWords[0]);
                    }
                    
                    // Combine all strategies - prioritize simple contains match
                    isVisible = simpleContainsMatch || 
                               exactMatch || 
                               titleMatch || 
                               addressMatch || 
                               categoryMatch || 
                               tagMatch || 
                               partialMatch || 
                               singleWordMatch;
                    
                    // Debug: If no match found, try a simple fallback
                    if (!isVisible && listingData) {
                        var simpleMatch = false;
                        var allText = [
                            listingData.title || '',
                            listingData.address || '',
                            Array.isArray(listingData.categories) ? listingData.categories.join(' ') : (listingData.categories || ''),
                            Array.isArray(listingData.tags) ? listingData.tags.map(function(t) { return typeof t === 'string' ? t : (t.name || ''); }).join(' ') : ''
                        ].join(' ').toLowerCase();
                        
                        simpleMatch = allText.includes(searchLower);
                        if (simpleMatch) {
                            isVisible = true;
                        }
                    }
                    
                } else {
                    // Fallback: search in popup content
                    var popupContent = marker.getPopup() ? marker.getPopup().getContent() : '';
                    isVisible = popupContent.toLowerCase().indexOf(searchLower) !== -1;
                }
                
                if (isVisible) {
                    // For clustered markers, ensure they're in the cluster group
                    var markers = udmMarkers[mapId];
                    if (markers && markers.getLayers) {
                        // Check if marker is already in cluster group
                        if (!markers.hasLayer(marker)) {
                            markers.addLayer(marker);
                        }
                    } else {
                        // Direct markers - use opacity
                        marker.setOpacity(1);
                    }
                    visibleCount++;
                    visibleMarkers.push(marker);
                } else {
                    // For clustered markers, remove from cluster group to hide them
                    var markers = udmMarkers[mapId];
                    if (markers && markers.getLayers) {
                        // Remove marker from cluster group to hide it
                        if (markers.hasLayer(marker)) {
                            markers.removeLayer(marker);
                        }
                    } else {
                        // Direct markers - use opacity
                        marker.setOpacity(0.3);
                    }
                }
        });
        
        
        // Count markers with and without listing data
        var markersWithData = 0;
        var markersWithoutData = 0;
        allMarkers.forEach(function(marker) {
            if (marker.listingData) {
                markersWithData++;
            } else {
                markersWithoutData++;
            }
        });
        
        // Track search for analytics (only if search term is not empty)
        console.log('SFRDM: About to track search - searchTerm:', searchTerm, 'visible:', visibleCount);
        if (searchTerm && searchTerm.trim() !== '') {
            trackSearchForAnalytics(mapId, searchTerm, visibleCount);
        } else {
            console.log('SFRDM: Search term empty, not tracking');
        }
        
        // Show search results message
        showSearchResults(mapId, searchTerm, visibleCount, totalCount, visibleMarkers);
    }
    
    /**
     * Show search results message
     */
    function showSearchResults(mapId, searchTerm, visibleCount, totalCount, visibleMarkers) {
        var mapContainer = document.querySelector('#' + mapId).closest('.udm-map-container');
        var existingMessage = mapContainer.querySelector('.udm-search-results');
        
        // Remove existing message
        if (existingMessage) {
            existingMessage.remove();
        }
        
        if (searchTerm && searchTerm.trim() !== '') {
            var message = document.createElement('div');
            message.className = 'udm-search-results';
            
            var resultsHtml = '<div class="udm-search-results-info">';
            resultsHtml += '<strong>Search Results:</strong> ' + visibleCount + ' of ' + totalCount + ' listings match "' + escapeHtml(searchTerm) + '"';
            resultsHtml += '</div>';
            
            // Add center on results button if there are visible markers
            if (visibleCount > 0 && visibleMarkers && visibleMarkers.length > 0) {
                resultsHtml += '<div class="udm-search-results-actions">';
                resultsHtml += '<button type="button" class="udm-center-results-btn" data-map-id="' + mapId + '">';
                resultsHtml += '<span class="udm-center-icon">🎯</span> Center on Results';
                resultsHtml += '</button>';
                resultsHtml += '</div>';
            }
            
            message.innerHTML = resultsHtml;
            
            var searchContainer = mapContainer.querySelector('.udm-search-container');
            if (searchContainer) {
                searchContainer.appendChild(message);
                
                // Add click handler for center button
                var centerBtn = message.querySelector('.udm-center-results-btn');
                if (centerBtn) {
                    centerBtn.addEventListener('click', function() {
                        centerMapOnResults(mapId, visibleMarkers);
                    });
                }
            }
        }
    }
    
    /**
     * Center map on search results
     */
    function centerMapOnResults(mapId, visibleMarkers) {
        var map = udmMaps[mapId];
        if (!map || !visibleMarkers || visibleMarkers.length === 0) {
            return;
        }
        
        
        if (visibleMarkers.length === 1) {
            // Single marker - center on it with reasonable zoom
            var marker = visibleMarkers[0];
            var latlng = marker.getLatLng();
            map.setView(latlng, 13, { animate: true, duration: 1 });
        } else {
            // Multiple markers - fit bounds to show all
            var group = new L.featureGroup(visibleMarkers);
            var bounds = group.getBounds();
            
            // Add some padding around the bounds
            map.fitBounds(bounds, { 
                padding: [20, 20],
                animate: true,
                duration: 1
            });
        }
        
        // Add visual feedback
        var centerBtn = document.querySelector('[data-map-id="' + mapId + '"]');
        if (centerBtn) {
            centerBtn.classList.add('udm-center-loading');
            centerBtn.innerHTML = '<span class="udm-center-icon">⏳</span> Centering...';
            
            setTimeout(function() {
                centerBtn.classList.remove('udm-center-loading');
                centerBtn.innerHTML = '<span class="udm-center-icon">🎯</span> Center on Results';
            }, 1000);
        }
    }
    
    /**
     * Public API for external access
     */
    window.SFRDM = {
        maps: udmMaps,
        markers: udmMarkers,
        allMarkers: udmAllMarkers,
        init: initUniversalMap,
        reload: function(mapId) {
            if (udmMaps[mapId]) {
                var mapElement = document.getElementById(mapId);
                if (mapElement) {
                    // Clear existing map
                    udmMaps[mapId].remove();
                    delete udmMaps[mapId];
                    delete udmMarkers[mapId];
                    delete udmAllMarkers[mapId];
                    
                    // Reinitialize
                    initUniversalMap(mapId);
                }
            }
        },
        filterByCategory: filterMapByCategory,
        search: searchMapListings
    };
    
    // Initialize maps when DOM is ready
    document.addEventListener('DOMContentLoaded', function() {
        
        // Initialize colored pins system
        
        
        // Find all map containers
        var mapContainers = document.querySelectorAll('.udm-map-container');
        
        mapContainers.forEach(function(container) {
            var mapId = container.getAttribute('data-map-id');
            if (mapId) {
                initUniversalMap(mapId);
            }
        });
    });
    
    /**
     * Track search for analytics
     */
    function trackSearchForAnalytics(mapId, searchTerm, resultsCount) {
        console.log('SFRDM: trackSearchForAnalytics called - mapId:', mapId, 'searchTerm:', searchTerm, 'results:', resultsCount);
        
        // Only track if analytics is enabled and we have the necessary data
        if (typeof sfrdmSettings === 'undefined' || !sfrdmSettings.ajax_url) {
            console.log('SFRDM: Tracking aborted - sfrdmSettings not available');
            return;
        }
        
        // Only track searches with at least 3 characters to avoid noise from single letters
        if (!searchTerm || searchTerm.trim().length < 3) {
            console.log('SFRDM: Tracking aborted - search term too short:', searchTerm);
            return;
        }
        
        // Debug logging removed for production
        
        // Debounce tracking to avoid too many requests
        if (window.udmAnalyticsTimeout) {
            clearTimeout(window.udmAnalyticsTimeout);
        }
        
        window.udmAnalyticsTimeout = setTimeout(function() {
            // Get the actual post ID from the map container
            var mapElement = document.getElementById(mapId);
            if (!mapElement) {
                return;
            }
            
            var actualMapId = mapElement.getAttribute('data-map-id');
            if (!actualMapId) {
                return;
            }
            
            jQuery.ajax({
                url: sfrdmSettings.ajax_url,
                type: 'POST',
                data: {
                    action: 'sfrdm_track_search',
                    nonce: sfrdmSettings.analytics_nonce || '',
                    map_id: actualMapId,
                    search_term: searchTerm,
                    results_count: resultsCount
                },
                timeout: 5000,
                success: function(response) {
                    if (response.success) {
                    } else {
                    }
                },
                error: function(xhr, status, error) {
                }
            });
        }, 1000); // 1 second debounce
    }
    
})(jQuery);
            